/*
 *  Boa, an http server, basic_auth.c implements authentication and
 *  authorization
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <grp.h>
#include <crypt.h>
#include <sys/stat.h>
#include <unistd.h>
#include <limits.h>
#include "basic_auth.h"
#include "base64.h"
#ifdef GROUP_POLICY
#include "group_policy.h"
#endif
#include "users.h"
#include "realms.h"
#include "prot_paths.h"
#include "boa.h"

#define MAX_PATH_LEN 2048

/* Supported HTTP authentication method: Basic. */
static const char basic_str[] = "Basic ";
static const int basic_start = sizeof(basic_str) - 1;

/* Error messages. */
static char out_of_memory[] = "Out of memory.";

void define_users(void);
enum auth_response basic_auth_real(request *req);
static enum auth_response basic_auth(char *path, request *req);
static int is_valid_basic(const char *basic_info, char **user, char **passwd);
static struct user *is_valid_user(const char *user_name, const char *passwd);
static int is_member_of_groups(struct user *user_item,
                               struct realm_group *groups);

/* Define the users from the UNIX files. */
void
define_users(void)
{
#ifndef DISABLE_DEBUG
     DEBUG(DEBUG_BASIC_AUTH) {
	  dump_paths();
     }
#endif
     init_users();
     parse_passwd(passwd_file);
     parse_group(group_file);
#ifdef GROUP_POLICY
     set_user_policy();
#endif

#ifndef DISABLE_DEBUG
     DEBUG(DEBUG_BASIC_AUTH) {
	  dump_users();
	  dump_realms();
     }
#endif
}

/*
 * Return request_authorized_auth_response when the request is authorized,
 * authentication_required_auth_response when authentication is
 * required, file_not_found_auth_response when the file is not found,
 * file_not_readable_auth_response when the file cannot be read,
 * redirect_http_auth_response when redirecting from https to http,
 * authentication_required_auth_response when http must be https.
 */
enum auth_response
basic_auth_real(request *req)
{
     char absolute_path[MAX_PATH_LEN];
     struct stat sb;

     /* The request was checked previously and can be returned immediately. */
     if (req->auth_result != uninitialized_auth_response) {
	  DEBUG(DEBUG_BASIC_AUTH) {
	       fprintf(stderr, "%s %s return\n", __FUNCTION__, req->pathname);
	  }
	  goto basic_auth_real_return;
     }

     /* Handle uninitialized request. */
     if (req->pathname == NULL) {
	  /* 404 Not Found. */
	  req->auth_result = file_not_found_auth_response;
	  goto basic_auth_real_return;
     }

     DEBUG(DEBUG_BASIC_AUTH) {
	  fprintf(stderr, "%s %s %s\n", __FUNCTION__, req->pathname,
		  req->request_uri);
     }
     /* This is due to boa parsing. Different states. */
     if (strlen(req->pathname) + strlen(req->request_uri) < MAX_PATH_LEN - 1) {
	  strcpy(absolute_path, req->pathname);
/*	  // marked by Kate
	  if (strstr(absolute_path, req->request_uri) == NULL) {
	       if (absolute_path[strlen(absolute_path) - 1] != '/') {
		    strcat(absolute_path, "/");
	       }
	       strcat(absolute_path, req->request_uri);
	  }
*/
     } else {
	  /* 404 Not Found. */
	  req->auth_result = file_not_found_auth_response;
	  goto basic_auth_real_return;
     }

#ifdef TRANSFER
     /* Only if the absolute_path is not a transfer, it will be a reference to an existing file. */
    if (!(is_transfer_mime(req)
        || is_transfer_uri(req)
        || is_transfer_post_uri(req))) 
#endif // TRANSFER
	{
		 if (stat(absolute_path, &sb) < 0) {
		  /* 404 Not Found. */
		  req->auth_result = file_not_found_auth_response;
		  goto basic_auth_real_return;
		 }

		 /* Must be readable by process. */
		 if (access(absolute_path, R_OK) < 0) {
		  /* 403 Forbidden. */
		  req->auth_result = file_not_readable_auth_response;
		  goto basic_auth_real_return;
		 }
	}

     /* Ordinary authentication and authorization. */
     req->auth_result = basic_auth(absolute_path, req);

basic_auth_real_return:
     return req->auth_result;
}

/*
 * Return request_authorized_auth_response when the request is authorized,
 * authentication_required_auth_response when authentication is
 * required, redirect_http_auth_response when redirecting from https to http,
 * authentication_required_auth_response when http must be https.
 */
static enum auth_response
basic_auth(char *path, request *req)
{
     struct realm *realm;

     req->auth_result = request_authorized_auth_response;

     /* The path may be protected within a realm. */
     realm = is_prot_path(path);
     if (realm != NULL) {
	  char *user_name = NULL;
	  char *passwd = NULL;
	  struct user *user_item = NULL;

	  req->realm = strdup(realm->name);
	  req->protspace = strdup(realm->space);
	  if (req->realm == NULL || req->protspace == NULL) {
	       perror(out_of_memory);
	       WARN(out_of_memory);
	       _exit(EXIT_FAILURE);
	  }

	  /* Basic validation only. */
	  if (is_valid_basic(req->authent, &user_name, &passwd) < 0) {
	       req->auth_result = authentication_required_auth_response;
	       goto basic_auth_return;
	  }
	  /* Validate user and password. */
	  user_item = is_valid_user(user_name, passwd);
	  if (user_item == NULL) {
	       req->auth_result = authentication_required_auth_response;
	       goto basic_auth_return;
	  }
	  /* Validate group membership of realm. */
          if (is_member_of_groups(user_item, realm->groups) < 0) {
	       req->auth_result = authentication_required_auth_response;
	       goto basic_auth_return;
	  }

#ifdef GROUP_POLICY
	  if (is_https_redirect(req->remote_ip_addr, user_item->policy)) {
	       req->auth_result = authentication_required_auth_response;
	       goto basic_auth_return;
	  }
	  if (is_http_redirect(req->remote_ip_addr, user_item->policy)) {
	       req->auth_result = redirect_http_auth_response;
	       goto basic_auth_return;
	  }
#endif

     basic_auth_return:
	  /* Store the user name in the request. */
	  if (user_name) {
	       req->user = strdup(user_name);
	       free(user_name);
	       if (req->user == NULL) {
		    perror(out_of_memory);
		    WARN(out_of_memory);
		    _exit(EXIT_FAILURE);
	       }
	  }

	  if (passwd) {
	       free(passwd);
	  }

	  /* Store the list of realms the user has permission for. */
	  if (user_item) {
	       req->user_realms = strdup(user_item->realm_list);
	       if (req->user_realms == NULL) {
		    perror(out_of_memory);
		    WARN(out_of_memory);
		    _exit(EXIT_FAILURE);
	       }
	  }
     }

     return req->auth_result;
}

/* Return 0 when basic_info contains a user and password or -1. */
static int
is_valid_basic(const char *basic_info, char **user, char **passwd)
{
     int len;
     char authent[512];
     char *passwd_ptr;

     /* Only basic authentication supported. */
     if (basic_info != NULL && STR_PREFIX(basic_str, basic_info)) {
	  /* Decode and parse the encoded "user:password". */
	  len = base64_decode((unsigned char *)authent, sizeof(authent) - 1, 
			      (unsigned char *)&basic_info[basic_start],
			      strlen(&basic_info[basic_start]));
	  if (len < 0) {
	       return -1;
	  }
	  authent[len] = '\0';
	  passwd_ptr = strchr(authent, ':');
	  if (passwd_ptr != NULL) {
	       *passwd_ptr++ = '\0';
	       *user = strdup(authent);
	       if (user == NULL) {
		    return -1;
	       }
	       *passwd = strdup(passwd_ptr);
	       if (passwd == NULL) {
		    free(user);
		    return -1;
	       }
	       return 0;
	  }
     }
     return -1;
}

/* Return a struct user* to a valid user_name and passwd pair or NULL. */
static struct user*
is_valid_user(const char *user_name, const char *passwd)
{
     struct user *user_item;
     char *gen_pass;

     /* Find user in the authorized user list. */
     user_item = find_user(user_name);
     if (user_item == NULL) {
	  return NULL;
     }
     /* Found, check the cached encrypted password. */
     gen_pass = crypt(passwd, user_item->passwd);
     if (STR_EQ(gen_pass, user_item->passwd)) {
	  return user_item;
     }
     /* Not valid passwd. */
     return NULL;
}

/* Return 0 when struct user* user_item is a member of groups or -1. */
static int
is_member_of_groups(struct user *user_item, struct realm_group *groups)
{
     struct realm_group *group_item;
     int i;

     group_item = groups;
     while (group_item != NULL) {
	  for (i = 0; i < user_item->no_groups; i++) {
	       if (user_item->groups[i] == group_item->id) {
		    return 0;
	       }
	  }
	  group_item = group_item->next;
     }
     return -1;
}
