#!/bin/bash

RELEASECFG=$1
PKGDIR=/opt/mobilygen;
CURRENT_DIR=$(pwd);

if [ -z "${RELEASECFG}" ] || [ ! -r "${RELEASECFG}" ]; then
	echo "ERROR: You need to specify a valid release config file."
	exit 1;
fi

PACKAGES_LIST=;
RELEASEDIR=;
RELNOTE_TEMPLATE=;
RELEASEFILE=release.run;
RELEASE_TITLE=;
RELEASE_INSTALLSCRIPT=;
source ${RELEASECFG};

if [ -z "${RELEASEDIR}" ]; then
	echo "ERROR: ${RELEASECFG} is incorrect: RELEASEDIR undefined.";
	exit 5;
fi

# Make sure makeself.sh is in the path
which makeself.sh &> /dev/null
if [ $? -ne 0 ]; then
	echo "ERROR: Failed to find makeself.sh, it must be in your path.";
	echo "makeself.sh is part of the utils release in /opt/mobilygen/utils.";
	exit 10;
fi

if [ -z "${RELEASE_INSTALLSCRIPT}" ]; then
	RELEASE_INSTALLSCRIPT="echo done";
fi

echo -ne "Creating release directory ... "
if [ ! -d ${RELEASEDIR} ]; then
	mkdir -p ${RELEASEDIR};
	if [ $? -ne 0 ]; then
		echo "ERROR: Failed to create ${RELEASEDIR}.";
		exit 15;
	fi
else
	echo "ERROR: ${RELEASEDIR} already exist, if you want to";
	echo "re-generate the release, remove the directory first.";
	exit 16;
fi
cd ${RELEASEDIR}
if [ $? -ne 0 ]; then
	echo -e "ERROR:\Failed to cd to ${RELEASEDIR}.";
	exit 17;
fi
echo "done.";

# For each package we install them
for package in ${PACKAGES_LIST}; do
	name=$(echo ${package} | sed -e 's|\([^=]*\).*|\1|');
	version=$(echo ${package} | sed -e 's|[^=]*=\([^/]*\).*|\1|');
	config=$(echo ${package} | sed -e 's|[^=]*=[^/]*/\(.*\)|\1|');
	echo -ne "Installing ${name}-${version} configuration ${config} ... "
	case ${name} in
		"dist")
			if [ ! -d ${PKGDIR}/distributions/${config}-${version} ]; then
				echo -e "ERROR:\nCould not find ${name} revision ${config}-${version}."
				exit 20;
			fi
			cp ${PKGDIR}/distributions/${config}-${version}/${name}.${config}-${version}.tar.bz2 .;
			if [ $? -ne 0 ]; then
				echo -e "ERROR:\nFailed to copy ${name} release.";
				exit 25;
			fi
			cp ${PKGDIR}/distributions/${config}-${version}/${name}.${config}-${version}.src.tar.bz2 .;
			if [ $? -ne 0 ]; then
				echo -e "ERROR:\nFailed to copy ${name} sources release.";
				exit 26;
			fi
			;;
		"toolchain")
			if [ ! -d ${PKGDIR}/tools/${config}-${version} ]; then
				echo -e "ERROR:\nCould not find ${name} revision ${config}-${version}."
				exit 30;
			fi
			cp ${PKGDIR}/tools/${config}-${version}/${name}.${config}-${version}.*.tar.bz2 .;
			if [ $? -ne 0 ]; then
				echo -e "ERROR:\nFailed to copy ${name} release.";
				exit 35;
			fi
			;;
		"merlinsw_host")
			if [ ! -d ${PKGDIR}/${name}/${config}-${version} ]; then
				echo -e "ERROR:\nCould not find ${name} revision ${config}-${version}."
				exit 40;
			fi
			cp ${PKGDIR}/${name}/${config}-${version}/${name}.${config}-${version}.tar.bz2 .;
			if [ $? -ne 0 ]; then
				echo -e "ERROR:\nFailed to copy ${name} release.";
				exit 45;
			fi
			cp ${PKGDIR}/${name}/${config}-${version}/${name}.src-${version}.tar.bz2 .;
			if [ $? -ne 0 ]; then
				echo -e "ERROR:\nFailed to copy ${name} source release.";
				exit 46;
			fi
			;;
		*)
			if [ ! -d ${PKGDIR}/${name}/${config}-${version} ]; then
				echo -e "ERROR:\nCould not find ${name} revision ${config}-${version}."
				exit 50;
			fi
			cp ${PKGDIR}/${name}/${config}-${version}/${name}.${config}-${version}.tar.bz2 .;
			if [ $? -ne 0 ]; then
				echo -e "ERROR:\nFailed to copy ${name} release.";
				exit 55;
			fi
			;;
	esac
	echo "done.";
done

# Generate the release file out of the template
if [ -n "${RELNOTE_TEMPLATE}" ]; then
	echo -ne "Creating release notes ... "
	if [ ! -r ${RELNOTE_TEMPLATE} ]; then
		echo -e "ERROR:\nCould not find release notes template.";
		exit 100;
	fi
	echo > mkrelnotes.sh
	echo 'cat << STARTSTOP' >> mkrelnotes.sh
	cat ${RELNOTE_TEMPLATE} >> mkrelnotes.sh
	echo 'STARTSTOP' >> mkrelnotes.sh
	source mkrelnotes.sh > RELEASE.txt
	if [ $? -ne 0 ]; then
		echo "ERROR: Failed to compile release notes.";
		exit 105;
	fi
	rm mkrelnotes.sh
	if [ $? -ne 0 ]; then
		echo "ERROR: Failed to clean intermediate release note file.";
		exit 106;
	fi
	echo "done.";
fi

# If the release has any extra thing to do, do them now
customize
if [ $? -ne 0 ]; then
	echo "ERROR: Failed to run release customization method.";
	exit 110;
fi

# Last, use makeself.sh to create the release self extracting file
echo -ne "Generating release self-extractable file ... "
cd ${CURRENT_DIR};
if [ $? -ne 0 ]; then
	echo -e "ERROR:\nFailed to change current directory to ${CURRENT_DIR}.";
	exit 120;
fi
# If install script is an absolute path we copy it into the release directory
if [ "X${RELEASE_INSTALLSCRIPT:0:1}" == "X/" ]; then
	cp ${RELEASE_INSTALLSCRIPT} ${RELEASEDIR}/;
	if [ $? -ne 0 ]; then
		echo -e "ERROR:\nFailed to copy install script into release directory.";
		exit 125;
	fi
	RELEASE_INSTALLSCRIPT=./$(basename ${RELEASE_INSTALLSCRIPT});
fi
makeself.sh --notemp --gzip ${RELEASEDIR} ${RELEASE_FILE} "${RELEASE_TITLE}" ${RELEASE_INSTALLSCRIPT};
if [ $? -ne 0 ]; then
	echo -e "ERROR:\nFailed to run makeself to generate release run file.";
	exit 130;
fi
echo "done."

echo "Release done, the file is ${RELEASE_FILE}.";

exit 0;
