#!/bin/sh
# mkdist.sh
# This script take a distribution tar file and output
# a ready to use solution. The solution will include
# a bootloader and a rootfs image (that may be combined).
# It will go through the following steps:
# -Extract the config input file from the distribution
# -Unless the user has elected not to, start menu config tool.
# -Generate root filesystem image.
# -If selected merge bootstrap and root filesystem.

# NOTE: The distribution should have the following format:
# -Be an archive compatible with tar and compressed with BZIP2 algorithm
# -Have the following tree:
#   ./rootfs: root file system structure, pre-stripped and cleaned.
#   ./images: bootstrap and other images.
#   ./config: configuration files and tools for this distribution:
#	   config.default: default configuration (optional).
#	   config.in: configuration options (optional).
#	   runconfig.sh: script that analyze the config file
#		   and generate the images.
#   ./extra: files that are part of the distribution 
#	   but do not go in any of the other directories

help()
{   echo "$(basename $0): convert a distribution release into a useable image"
	echo "This script take a distribution tar file and output"
	echo "a ready to use solution. The solution will include"
	echo "a bootloader and a rootfs image (that may be combined)."
	echo "It will go through the following steps:"
	echo "-Extract the config input file from the distribution."
	echo "-Unless the user has elected not to, start menu config tool."
	echo "-Generate root filesystem image."
	echo "-If selected merge bootstrap and root filesystem."
	echo "The following arguments are expected (in the order they are described):"
	echo "<dist_release>: path to the distribution release tar file."
	echo "The following options are supported:"
	echo "--skipconfig: do not run the config utility."
	echo "--config: use another config file than .config."
	echo "--install: Install one or more extra packages on the root filesystem."
	echo "--tmp: temporary directory used by this script."
	echo "--keeptmp: do not clean the temp directory when the script exit."
	echo "-- <exttra_args ...>: arguments to transfer directly to runconfig.sh"
	echo
	return 0;
}

clean_and_exit()
{	local RET_CODE=$1
	if [ -n "${TMP_DIR}" ]; then
		if [ -d ${TMP_DIR} ]; then
			if [ ${KEEPTMP} -eq 0 ]; then
				echo -en "Cleaning temporary directory ... ";
				rm -rf ${TMP_DIR};
				if [ $? -ne 0 ]; then 
					echo -e "failed.";
					echo "WARNING: could not remove temp directory ${TMP_DIR}, please do it manually.";
				else
					echo -e "done.";
				fi
			else
				echo "Temporary files kept in ${TMP_DIR}";
			fi
		fi
	fi
	echo
	exit ${RET_CODE}
}

signal_handler()
{	echo -ne "\nError: script interrupted by a signal.\n";
	clean_and_exit 255;
}

init()
{   # Analyze arguments and detect any error
	DIST_TAR=;
	START_MENUCONFIG=1;
	CONFIG_FILE=".config";
	TMP_DIR=/tmp/$(basename $0).$$;
	EXTRA_PACKAGES=;
	RUNCONFIG_ARGS=;
	KEEPTMP=0;
	while [ ! -z "$1" ]; do
		case $1 in
			"--help") 
				help;
				return 255;
				;;
			"--skipconfig") START_MENUCONFIG=0; shift 1;;
			"--config") CONFIG_FILE=$2; shift 2;;
			"--tmp") TMP_DIR=$2; shift 2;;
			"--path") PATH=$PATH:$2 ; shift 2;;
			"--keeptmp") KEEPTMP=1; RUNCONFIG_ARGS="${RUNCONFIG_ARGS} --keeptmp"; shift 1;;
			"--install") RUNCONFIG_ARGS="${RUNCONFIG_ARGS} --postpkg $2"; 
						 EXTRA_PACKAGES="${EXTRA_PACKAGES} $2";
						 if [ "X$2" == "X" ] || [ ! -e $2 ]
						 then
							 echo "ERROR: extra install package does not exist $2"
							 return 1 
						 fi
						 shift 2;;
			"--") shift 1; RUNCONFIG_ARGS="${RUNCONFIG_ARGS} $*"; break;;
			*) 
				if [ -z "${DIST_TAR}" ]; then
					DIST_TAR=$1;
				else
					echo "ERROR: too many arguments: $1";
					echo "try --help";
					return 1;
				fi
				shift 1;;
		esac
	done

	if [ -z "${DIST_TAR}" ]; then
		echo "ERROR: Missing an argument, try --help";
		return 2;
	fi
	if [ ! -r ${DIST_TAR} ]; then
		echo "ERROR: Can't read distribution release file ${DIST_TAR}";
		return 3;
	fi
	for pkg in ${EXTRA_PACKAGES}; do
		if [ ! -f ${pkg} ]; then
			echo "ERROR: ${pkg} not found.";
			return 4;
		fi
	done
	if [ -e "${TMP_DIR}" ]; then
		echo "ERROR: Temp directory ${TMP_DIR} exists, please remove.";
		return 5;
	fi	
	if [ ! -z ${DISTINSTALL_KEEPTMP} ]
	then 
		echo "NOTICE:  distinstall.sh will keep tmp dir: ${TMP_DIR}"
		KEEPTMP=1; RUNCONFIG_ARGS="${RUNCONFIG_ARGS} --keeptmp"
	fi
	return 0;
}

# Setup a trap in case CtrlC is pressed
trap signal_handler SIGKILL SIGINT SIGHUP SIGQUIT SIGTERM

init $@;
if [ $? -ne 0 ]; then
	clean_and_exit 1;
fi

mkdir -p ${TMP_DIR};
if [ $? -ne 0 ]; then
	echo "ERROR: Failed to create temporary directory ${TMP_DIR}";
	clean_and_exit 2;
fi

echo -ne "Extracting distribution ... "
echo ${DIST_TAR} | grep "\.bz2"
if [ $? -ne 0 ]; then
	tar xf ${DIST_TAR} -C ${TMP_DIR} .
else
	tar jxf ${DIST_TAR} -C ${TMP_DIR} .
fi
if [ $? -ne 0 ]; then
	echo -ne "ERROR\nDistribution file is corrupted !\n\n";
	clean_and_exit 3;
fi
echo -ne "done\n";

CONFIG_DIR=${TMP_DIR}/config
TOOLS_DIR=${TMP_DIR}/$(echo $(uname -m)-$(uname -s) | awk '{ print tolower($1) }')/bin
if [ ! -r ${CONFIG_FILE} ]; then
	echo -ne "Config file not found, use default one from distribution.\n";
	if [ -f "${CONFIG_DIR}/config.default" ]; then
		cp ${CONFIG_DIR}/config.default ${CONFIG_FILE}
		if [ $? -ne 0 ]; then
			echo -ne "ERROR\nCan't write config file ${CONFIG_FILE}\n\n";
			clean_and_exit 6;
		fi
	else
		echo "" > ${CONFIG_FILE}
	fi
else
	echo -ne "Using defaults in ${CONFIG_FILE}.\n";
	# Save it under the default name
	if [ "X${CONFIG_FILE}" != "X.config" ]; then
		cp ${CONFIG_FILE} .config;
		if [ $? -ne 0 ]; then
			echo "ERROR: could not use ${CONFIG_FILE}, abort."
			clean_and_exit 7;
		fi
	fi
fi

# If no config file we skip the config utility
if [ ! -r "${CONFIG_DIR}/config.in" ]; then
	START_MENUCONFIG=0;
fi
if [ ${START_MENUCONFIG} -ne 0 ]; then
	echo -ne "Running distribution configuration utility ...\n";
	CURDIR=$(pwd)
	cp ${CONFIG_FILE} ${CONFIG_DIR}/.config;
	cd  ${CONFIG_DIR}
	if [ -x ${TOOLS_DIR}/mconf ]; then
		${TOOLS_DIR}/mconf ${CONFIG_DIR}/config.in;
		ERR=$?
	else
		mconf ${CONFIG_DIR}/config.in;
		ERR=$?
	fi
	if [ $ERR -ne 0 ]; then
		echo "ERROR: configuration utility returned an error, abort."
		clean_and_exit 7;
	fi
	cp .config  ${CURDIR}/.config;
	# Update the config file
	if [ "X${CONFIG_FILE}" != "X.config" ]; then
		cp .config  ${CURDIR}/${CONFIG_FILE};
		if [ $ERR -ne 0 ]; then
			echo "ERROR: could not save ${CONFIG_FILE}, abort."
			clean_and_exit 7;
		fi
	fi
	cd ${CURDIR}
fi







echo -ne "Preparing to run the configuration script ... ";
if [ ! -x "${CONFIG_DIR}/runconfig.sh" ]
then
	echo -ne "\nERROR: distribution is corrupted, can't find runconfig.sh.\n\n";
	clean_and_exit 8;
fi
echo -ne "done\nStarting the script: ";
RUNCONFIG_ARGS="${RUNCONFIG_ARGS} --tmp ${TMP_DIR}/runconfig";
${CONFIG_DIR}/runconfig.sh ${CONFIG_FILE} ${TMP_DIR} ${RUNCONFIG_ARGS};
ERR=$?
echo -ne "Back, check script results ... ";
if [ $ERR -ne 0 ]; then
	echo -ne "\nERROR: Failed to apply configuration.\n\n";
	clean_and_exit 9;
fi
if [ ! -f "${CONFIG_DIR}/config.result" ]; then
	echo -ne "ERROR: configuration utility has a bug, please report\n\n";
	clean_and_exit 10;
fi
echo -ne "done\n";

echo "Distribution installed. Here is your configuration:"
cat ${CONFIG_DIR}/config.result;

clean_and_exit 0;
