/*******************************************************************************
*
* The content of this file or document is CONFIDENTIAL and PROPRIETARY
* to Mobilygen Corporation.  It is subject to the terms of a
* License Agreement between Licensee and Mobilygen Corporation.
* restricting among other things, the use, reproduction, distribution
* and transfer.  Each of the embodiments, including this information and
* any derivative work shall retain this copyright notice.
*
* Copyright 2010 Mobilygen Corporation.
* All rights reserved.
*
* QuArc is a registered trademark of Mobilygen Corporation.
*
* Version: SDK7r26554
*
*******************************************************************************/


#ifndef __QMED_HH
#define __QMED_HH

/* QMED declarations */

#define QMED_VERSION_NUM 0

//////////////////////////////////////////////////////////////////////////////
// version: 0
//
// SDL
// aligned(8) class QMed extends FullBox('qmed', version = 0, boxflags) {
//     unsigned short major_media_type;
//     unsigned short minor_media_type;
//     unsigned char media_data[];
// }
//
// equivalent to 
// typedef struct {
//     unsigned long box_size;
//     unsigned long box_type; // "qmed"
//     unsigned long box_flags; // (version << 24 | boxflags)
//     unsigned short major_media_type;
//     unsigned short minor_media_type;
//     unsigned char media_data[];
// } QMed;
//
// version 0 does not use large box
//
// box_flags
// 31 - 24         23 - 0
// version         boxflags
//
// major_media_type:
// 0x01 AAC audio. media_data contains audio stream information
// 0x02 H.264 video. media_data contains audio frame information
// 0x05 H.264 video. sample_data contains video slice or configuration info.
// 0x06 MP1 audio. media_data contain contains stream information
// 0x09 G.711 audio. sample_data contains one audio frame
//
// minor_media_type:
//
//////////////////////////////////////////////////////////////////////////////

#define QMED_BOX_TYPE 0x716d6564
#define QMED_MAJOR_MEDIA_TYPE_AAC 0x1
#define QMED_MAJOR_MEDIA_TYPE_H264 0x2
#define QMED_MAJOR_MEDIA_TYPE_PCM 0x3
#define QMED_MAJOR_MEDIA_TYPE_MP2 0x6
#define QMED_MAJOR_MEDIA_TYPE_JPEG 0x7
#define QMED_MAJOR_MEDIA_TYPE_Q711 0x9
#define QMED_MAJOR_MEDIA_TYPE_Q728 0xa
#define QMED_MAJOR_MEDIA_TYPE_Q722 0xb
#define QMED_MAJOR_MEDIA_TYPE_Q726 0xc
#define QMED_MAJOR_MEDIA_TYPE_MAX 0xd

#define QMED_MINOR_MEDIA_TYPE_Q711_ALAW 0x0
#define QMED_MINOR_MEDIA_TYPE_Q711_ULAW 0x1
#define QMED_MINOR_MEDIA_TYPE_Q726_ITU_BYTE_ORDER 0x0
#define QMED_MINOR_MEDIA_TYPE_Q726_IETF_BYTE_ORDER 0x1

#define QMED_VERSION (A) ((A >> 24) && 0xff)

#define QMED_SHA_SIZE            8 

typedef struct
{
    unsigned long v : 8;
    unsigned long f : 24;
} QMedFlags;

typedef struct
{
    unsigned long boxSize;
    unsigned long boxType;
    union {
        unsigned long value;
        QMedFlags     field;
    } boxFlags;
    unsigned long majorMediaType;
    unsigned long minorMediaType;
} QMedStruct;

typedef struct
{
    unsigned long hashSize; 
    unsigned long hashPayload[QMED_SHA_SIZE];
} QMedVer1InfoStruct;

typedef struct
{
    unsigned long version;
    unsigned long width;
    unsigned long height;
    unsigned long sampleTicks;
    unsigned long motionCounter;
    unsigned long motionBitmapSize;
} QMedH264Struct;

typedef struct
{
    unsigned long version;
    unsigned int samplingFrequency;
    unsigned int accessUnits;
    unsigned int accessUnitSize;
    unsigned int channels;
} QMedPCMStruct;

typedef struct
{
    unsigned long version;
    unsigned int samplingFrequency;
    unsigned int channels;
    unsigned int sampleSize;
    unsigned int audioSpecificConfigSize;
    // Remainder of the qmed box is the variable length audioSpecificConfig
} QMedAACStruct;

typedef struct
{
    unsigned long version;
    unsigned int samplingFrequency;
    unsigned int channels;
} QMedMP2Struct;

typedef struct
{
    unsigned long version;
    unsigned long width;
    unsigned long height;
    unsigned long frameTicks;
} QMedJPEGStruct;

typedef struct
{
    // Always: 8KHz sample rate, 64Kbps
    // Minor type in header denotes A-law or U-law.
    unsigned long version;
    unsigned int accessUnits;   // Total number of samples in box
    unsigned int sampleSize;
    //unsigned int channels;    // Always 1 channel?
} QMed711Struct;

typedef struct
{
    // Always: 1 channel, 16KHz sample rate 
    // No minor type; AMR-WB will have its own QMED type.
    unsigned long version;
    unsigned int bitrate;   // 64000, 56000, or 48000 bps for decoder; enc always 64Kbps
    unsigned int accessUnits;   // Total number of samples
    unsigned int sampleSize;
    //unsigned int channels;    // Always 1 channel?
} QMed722Struct;

typedef struct
{
    // Always: 8KHz sample rate
    unsigned long version;
    unsigned int bitrate;   // Only: 16000, 24000, 32000, or 40000
    unsigned int accessUnits;   // Total number of samples in box
    unsigned int sampleSize;
    //unsigned int channels;    // Always 1 channel?
} QMed726Struct;

typedef struct
{
    // Always: 8KHz sample rate, 16Kbps 
    unsigned long version;
    unsigned int accessUnits;   // Total number of acess units in box. Each access unit is 5 bytes long and stores 4 codewords (=20 samples).
    //unsigned int channels;    // Always 1 channel?
} QMed728Struct;

#ifdef __cplusplus

class QMedBase
{
    public:
        void Alloc(int version, int extHdrSize);
        void Free();
        unsigned char * Addr();
        unsigned long BoxSize();
        unsigned long BoxType();
        unsigned long BoxFlags();
        unsigned char Version();
        unsigned long Flags();
        unsigned long MajorMediaType();
        unsigned long MinorMediaType();
        unsigned long HashSize();
        unsigned long *HashPayload();
        void Addr(unsigned char * addr);
        void BoxSize(unsigned long size);
        void BoxType(unsigned long type);
        void BoxFlags(unsigned long flags);
        void Version(unsigned char v);
        void Flags(unsigned long f);
        void MajorMediaType(unsigned long majorMediaType);
        void MinorMediaType(unsigned long minorMediaType);
        void HashSize(unsigned long hashSize);
        void HashPayload(unsigned long idx, unsigned long hashPayload);
    protected:
        QMedStruct          *mpHdr;
        QMedVer1InfoStruct  *mpVer1Info;
};

class QMedH264
{
    public:
        void Alloc(int baseVersion, int h264Version);
        void Free();
        unsigned char * Addr();
        unsigned long HeaderSize(int baseVersion, int h264Version);
        unsigned long H264Version();
        unsigned long Width();
        unsigned long Height();
        unsigned long SampleTicks();
        unsigned long MotionCounter();
        unsigned long MotionBitmapSize();
        void Addr(unsigned char * addr);
        void H264Version(unsigned long version);
        void Width(unsigned long width);
        void Height(unsigned long height);
        void SampleTicks(unsigned long sampleTicks);
        void MotionCounter(unsigned long motionCounter);
        void MotionBitmapSize(unsigned long motionBitmapSize);
        
        QMedBase         mQMedHdr;
    protected:
        QMedH264Struct  *mpH264Hdr;
};

class QMedPCM
{
    public:
        void Alloc(int baseVersion, int pcmVersion);
        void Free();
        unsigned char * Addr();
        unsigned long HeaderSize(int baseVersion, int pcmVersion);
        unsigned long PCMVersion();
        unsigned int  SampleFrequency();
        unsigned int  AccessUnits();
        unsigned int  AccessUnitSize();
        unsigned int  Channels();
        void Addr(unsigned char * addr);
        void PCMVersion(unsigned long version);
        void SampleFrequency(unsigned int sampleFrequency);
        void AccessUnits(unsigned int accessUnits);
        void AccessUnitSize(unsigned int accessUnitSize);
        void Channels(unsigned int channels);
        
        QMedBase         mQMedHdr;
    protected:
        QMedPCMStruct   *mpPCMHdr;
};

class QMedAAC
{
    public:
        void Alloc(int baseVersion, int aacVersion);
        void Free();
        unsigned char * Addr();
        unsigned long HeaderSize(int baseVersion, int aacVersion);
        unsigned long AACVersion();
        unsigned int  SampleFrequency();
        unsigned int  Channels();
        unsigned int  SampleSize();
        unsigned int  AudioSpecificConfigSize();
        void Addr(unsigned char * addr);
        void AACVersion(unsigned long version);
        void SampleFrequency(unsigned int sampleFrequency);
        void Channels(unsigned int channels);
        void SampleSize(unsigned int sampleSize);
        void AudioSpecificConfigSize(unsigned int audioSpecificConfigSize);
        
        QMedBase         mQMedHdr;
    protected:
        QMedAACStruct   *mpAACHdr;
};

class QMedMP2
{
    public:
        void Alloc(int baseVersion, int mp2Version);
        void Free();
        unsigned char * Addr();
        unsigned long HeaderSize(int baseVersion, int mp2Version);
        unsigned long MP2Version();
        unsigned int  SampleFrequency();
        unsigned int  Channels();
        void Addr(unsigned char * addr);
        void MP2Version(unsigned long version);
        void SampleFrequency(unsigned int sampleFrequency);
        void Channels(unsigned int channels);
        
        QMedBase         mQMedHdr;
    protected:
        QMedMP2Struct   *mpMP2Hdr;
};

class QMedJPEG
{
    public:
        void Alloc(int baseVersion, int jpegVersion);
        void Free();
        unsigned char * Addr();
        unsigned long HeaderSize(int baseVersion, int jpegVersion);
        unsigned long JPEGVersion();
        unsigned long Width();
        unsigned long Height();
        unsigned long FrameTicks();
        void Addr(unsigned char * addr);
        void JPEGVersion(unsigned long version);
        void Width(unsigned long width);
        void Height(unsigned long height);
        void FrameTicks(unsigned long sampleTicks);
        
        QMedBase         mQMedHdr;
    protected:
        QMedJPEGStruct  *mpJPEGHdr;
};

class QMed711
{
    public:
        void Alloc(int baseVersion, int g711Version);
        void Free();
        unsigned char * Addr();
        unsigned long HeaderSize(int baseVersion, int g711Version);
        unsigned long G711Version();
        unsigned int  AccessUnits();
        unsigned int  SampleSize();
        void Addr(unsigned char * addr);
        void G711Version(unsigned long version);
        void AccessUnits(unsigned int accessUnits);
        void SampleSize(unsigned int sampleSize);
        
        QMedBase         mQMedHdr;
    protected:
        QMed711Struct   *mp711Hdr;
};

class QMed722
{
    public:
        void Alloc(int baseVersion, int g722Version);
        void Free();
        unsigned char * Addr();
        unsigned long HeaderSize(int baseVersion, int g722Version);
        unsigned long G722Version();
        unsigned int  Bitrate();
        unsigned int  AccessUnits();
        unsigned int  SampleSize();
        void Addr(unsigned char * addr);
        void G722Version(unsigned long version);
        void Bitrate(unsigned long bitrate);
        void AccessUnits(unsigned int accessUnits);
        void SampleSize(unsigned int sampleSize);
        
        QMedBase         mQMedHdr;
    protected:
        QMed722Struct   *mp722Hdr;
};

class QMed726
{
    public:
        void Alloc(int baseVersion, int g726Version);
        void Free();
        unsigned char * Addr();
        unsigned long HeaderSize(int baseVersion, int g726Version);
        unsigned long G726Version();
        unsigned int  Bitrate();
        unsigned int  AccessUnits();
        unsigned int  SampleSize();
        void Addr(unsigned char * addr);
        void G726Version(unsigned long version);
        void Bitrate(unsigned long bitrate);
        void AccessUnits(unsigned int accessUnits);
        void SampleSize(unsigned int sampleSize);
        
        QMedBase         mQMedHdr;
    protected:
        QMed726Struct   *mp726Hdr;
};

class QMed728
{
    public:
        void Alloc(int baseVersion, int g728Version);
        void Free();
        unsigned char * Addr();
        unsigned long HeaderSize(int baseVersion, int g728Version);
        unsigned long G728Version();
        unsigned int  AccessUnits();
        void Addr(unsigned char * addr);
        void G728Version(unsigned long version);
        void AccessUnits(unsigned int accessUnits);
        
        QMedBase         mQMedHdr;
    protected:
        QMed728Struct   *mp728Hdr;
};

#ifdef _WIN32
#define LITTLE_ENDIAN 0
#define BIG_ENDIAN    1
#define __BYTE_ORDER LITTLE_ENDIAN
#else
#include <endian.h>
#endif

#define BE8(a) (a)
#if __BYTE_ORDER == BIG_ENDIAN
#define BE16(a) (a)
#define BE24(a) (a)
#define BE32(a) (a)
#define BE64(a) (a)
#else
#define BE16(a)                                                             \
    ((((a)>>8)&0xFF) |                                                      \
    (((a)<<8)&0xFF00))
#define BE24(a)                                                             \
    ((((a)>>16)&0xFF) |                                                     \
    ((a)&0xFF00) |                                                          \
    (((a)<<16)&0xFF0000))
#define BE32(a)                                                             \
    ((((a)>>24)&0xFF) |                                                     \
    (((a)>>8)&0xFF00) |                                                     \
    (((a)<<8)&0xFF0000) |                                                   \
    ((((a)<<24))&0xFF000000))
#define BE64(a)                                                             \
    (BE32(((a)>>32)&0xFFFFFFFF) |                                           \
    ((BE32((a)&0xFFFFFFFF)<<32)&0xFFFFFFFF00000000))
#endif

#ifdef __QMM__
#define QMED_MALLOC(x) QMM_MALLOC(x)
#define QMED_FREE(x)   QMM_FREE(x)
#else
#define QMED_MALLOC(x) malloc(x)
#define QMED_FREE(x)   free(x)
#endif

/*****************************************************************************
                                QMedBase Class
 *****************************************************************************/

inline void QMedBase::Alloc(int version, int extHdrSize)
{
    unsigned long boxSize;
    
    if (version == 0)
    {
        boxSize = sizeof(QMedStruct) + extHdrSize;
        mpHdr = (QMedStruct *) QMED_MALLOC(boxSize);
        mpVer1Info = NULL;
    }
    else
    {
        boxSize = sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct) + extHdrSize;
        mpHdr = (QMedStruct *) QMED_MALLOC(boxSize);
        mpVer1Info = (QMedVer1InfoStruct *) ((unsigned long) mpHdr + sizeof(QMedStruct));
    }
    
    Version(version);
    BoxSize(boxSize);
    BoxType(QMED_BOX_TYPE);
}

inline void QMedBase::Free()
{
    QMED_FREE(mpHdr);
}

inline unsigned char * QMedBase::Addr()
{
    return ((unsigned char *) mpHdr);
}

inline unsigned long QMedBase::BoxSize()
{
    return BE32(mpHdr->boxSize);
}

inline unsigned long QMedBase::BoxType()
{
    return BE32(mpHdr->boxType);
}

inline unsigned long QMedBase::BoxFlags()
{
    return BE32(mpHdr->boxFlags.value);
}

inline unsigned char QMedBase::Version()
{
    return BE8(mpHdr->boxFlags.field.v);
}

inline unsigned long QMedBase::Flags()
{
    return BE24(mpHdr->boxFlags.field.f);
}

inline unsigned long QMedBase::MajorMediaType()
{
    return BE32(mpHdr->majorMediaType);
}

inline unsigned long QMedBase::MinorMediaType()
{
    return BE32(mpHdr->minorMediaType);
}

inline unsigned long QMedBase::HashSize()
{
    if (Version() == 0)
        return 0;
    else
        return BE32(mpVer1Info->hashSize);
}

inline unsigned long * QMedBase::HashPayload()
{
    if (Version() == 0)
        return NULL;
    else
        return mpVer1Info->hashPayload;
}

inline void QMedBase::Addr(unsigned char * addr)
{
    mpHdr = (QMedStruct *) addr;
    
    if (Version() == 0)
        mpVer1Info = NULL;
    else
        mpVer1Info = (QMedVer1InfoStruct *) ((unsigned long) mpHdr + sizeof(QMedStruct));
}

inline void QMedBase::BoxSize(unsigned long size)
{
    mpHdr->boxSize = BE32(size);
}

inline void QMedBase::BoxType(unsigned long type)
{
    mpHdr->boxType = BE32(type);
}

inline void QMedBase::BoxFlags(unsigned long flags)
{
    mpHdr->boxFlags.value = BE32(flags);
}

inline void QMedBase::Version(unsigned char v)
{
    mpHdr->boxFlags.field.v = BE8(v);
}

inline void QMedBase::Flags(unsigned long f)
{
    mpHdr->boxFlags.field.f = BE8(f);
}

inline void QMedBase::MajorMediaType(unsigned long majorMediaType)
{
    mpHdr->majorMediaType = BE32(majorMediaType);
}

inline void QMedBase::MinorMediaType(unsigned long minorMediaType)
{
    mpHdr->minorMediaType = BE32(minorMediaType);
}

inline void QMedBase::HashSize(unsigned long hashSize)
{
    if (Version() == 1)
        mpVer1Info->hashSize = BE32(hashSize);
}

inline void QMedBase::HashPayload(unsigned long idx, unsigned long hashPayload)
{
    if (Version() == 1)
        mpVer1Info->hashPayload[idx] = BE32(hashPayload);
}

/*****************************************************************************
                                QMedH264 Class
 *****************************************************************************/

inline void QMedH264::Alloc(int baseVersion, int h264Version)
{
    mQMedHdr.Alloc(baseVersion, sizeof(QMedH264Struct));
    
    mpH264Hdr = (QMedH264Struct *) (mQMedHdr.Addr() + mQMedHdr.BoxSize() - sizeof(QMedH264Struct));
    
    H264Version(h264Version);
}

inline void QMedH264::Free()
{
    mQMedHdr.Free();
}

inline unsigned char * QMedH264::Addr()
{
    return mQMedHdr.Addr();
}

inline unsigned long QMedH264::HeaderSize(int baseVersion, int h264Version)
{
    if (baseVersion == 0)
        return (sizeof(QMedStruct) + sizeof(QMedH264Struct));
    else
        return (sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct) + sizeof(QMedH264Struct));
}

inline unsigned long QMedH264::H264Version()
{
    return BE32(mpH264Hdr->version);
}

inline unsigned long QMedH264::Width()
{
    return BE32(mpH264Hdr->width);
}

inline unsigned long QMedH264::Height()
{
    return BE32(mpH264Hdr->height);
}

inline unsigned long QMedH264::SampleTicks()
{
    return BE32(mpH264Hdr->sampleTicks);
}

inline unsigned long QMedH264::MotionCounter()
{
    return BE32(mpH264Hdr->motionCounter);
}

inline unsigned long QMedH264::MotionBitmapSize()
{
    return mpH264Hdr->motionBitmapSize;
}

inline void QMedH264::Addr(unsigned char * addr)
{
    mQMedHdr.Addr(addr);
    
    if (mQMedHdr.Version() == 0) 
        mpH264Hdr = (QMedH264Struct *) (addr + sizeof(QMedStruct));
    else
        mpH264Hdr = (QMedH264Struct *) (addr + sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct));
}

inline void QMedH264::H264Version(unsigned long version)
{
    mpH264Hdr->version = BE32(version);
}

inline void QMedH264::Width(unsigned long width)
{
    mpH264Hdr->width = BE32(width);
}

inline void QMedH264::Height(unsigned long height)
{
    mpH264Hdr->height = BE32(height);
}

inline void QMedH264::SampleTicks(unsigned long sampleTicks)
{
    mpH264Hdr->sampleTicks = BE32(sampleTicks);
}

inline void QMedH264::MotionCounter(unsigned long motionCounter)
{
    mpH264Hdr->motionCounter = BE32(motionCounter);
}

inline void QMedH264::MotionBitmapSize(unsigned long motionBitmapSize)
{
    mpH264Hdr->motionBitmapSize = BE32(motionBitmapSize);
}

/*****************************************************************************
                                QMedPCM Class
 *****************************************************************************/

inline void QMedPCM::Alloc(int baseVersion, int pcmVersion)
{
    mQMedHdr.Alloc(baseVersion, sizeof(QMedPCMStruct));
    
    mpPCMHdr = (QMedPCMStruct *) (mQMedHdr.Addr() + mQMedHdr.BoxSize() - sizeof(QMedPCMStruct));
    
    PCMVersion(pcmVersion);
}

inline void QMedPCM::Free()
{
    mQMedHdr.Free();
}

inline unsigned char * QMedPCM::Addr()
{
    return mQMedHdr.Addr();
}

inline unsigned long QMedPCM::HeaderSize(int baseVersion, int pcmVersion)
{
    if (baseVersion == 0)
        return (sizeof(QMedStruct) + sizeof(QMedPCMStruct));
    else
        return (sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct) + sizeof(QMedPCMStruct));
}

inline unsigned long QMedPCM::PCMVersion()
{
    return BE32(mpPCMHdr->version);
}

inline unsigned int QMedPCM::SampleFrequency()
{
    return BE32(mpPCMHdr->samplingFrequency);
}

inline unsigned int QMedPCM::AccessUnits()
{
    return BE32(mpPCMHdr->accessUnits);
}

inline unsigned int QMedPCM::AccessUnitSize()
{
    return BE32(mpPCMHdr->accessUnitSize);
}

inline unsigned int QMedPCM::Channels()
{
    return BE32(mpPCMHdr->channels);
}

inline void QMedPCM::Addr(unsigned char * addr)
{
    mQMedHdr.Addr(addr);
    
    if (mQMedHdr.Version() == 0) 
        mpPCMHdr = (QMedPCMStruct *) (addr + sizeof(QMedStruct));
    else
        mpPCMHdr = (QMedPCMStruct *) (addr + sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct));
}

inline void QMedPCM::PCMVersion(unsigned long version)
{
    mpPCMHdr->version = BE32(version);
}

inline void QMedPCM::SampleFrequency(unsigned int sampleFrequency)
{
    mpPCMHdr->samplingFrequency = BE32(sampleFrequency);
}

inline void QMedPCM::AccessUnits(unsigned int accessUnits)
{
    mpPCMHdr->accessUnits = BE32(accessUnits);
}

inline void QMedPCM::AccessUnitSize(unsigned int accessUnitSize)
{
    mpPCMHdr->accessUnitSize = BE32(accessUnitSize);
}

inline void QMedPCM::Channels(unsigned int channels)
{
    mpPCMHdr->channels = BE32(channels);
}

/*****************************************************************************
                                QMedAAC Class
 *****************************************************************************/

inline void QMedAAC::Alloc(int baseVersion, int aacVersion)
{
    mQMedHdr.Alloc(baseVersion, sizeof(QMedAACStruct));
    
    mpAACHdr = (QMedAACStruct *) (mQMedHdr.Addr() + mQMedHdr.BoxSize() - sizeof(QMedAACStruct));
    
    AACVersion(aacVersion);
}

inline void QMedAAC::Free()
{
    mQMedHdr.Free();
}

inline unsigned char * QMedAAC::Addr()
{
    return mQMedHdr.Addr();
}

inline unsigned long QMedAAC::HeaderSize(int baseVersion, int aacVersion)
{
    if (baseVersion == 0)
        return (sizeof(QMedStruct) + sizeof(QMedAACStruct));
    else
        return (sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct) + sizeof(QMedAACStruct));
}

inline unsigned long QMedAAC::AACVersion()
{
    return BE32(mpAACHdr->version);
}

inline unsigned int QMedAAC::SampleFrequency()
{
    return BE32(mpAACHdr->samplingFrequency);
}

inline unsigned int QMedAAC::Channels()
{
    return BE32(mpAACHdr->channels);
}

inline unsigned int QMedAAC::SampleSize()
{
    return BE32(mpAACHdr->sampleSize);
}

inline unsigned int QMedAAC::AudioSpecificConfigSize()
{
    return BE32(mpAACHdr->audioSpecificConfigSize);
}

inline void QMedAAC::Addr(unsigned char * addr)
{
    mQMedHdr.Addr(addr);
    
    if (mQMedHdr.Version() == 0) 
        mpAACHdr = (QMedAACStruct *) (addr + sizeof(QMedStruct));
    else
        mpAACHdr = (QMedAACStruct *) (addr + sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct));
}

inline void QMedAAC::AACVersion(unsigned long version)
{
    mpAACHdr->version = BE32(version);
}

inline void QMedAAC::SampleFrequency(unsigned int sampleFrequency)
{
    mpAACHdr->samplingFrequency = BE32(sampleFrequency);
}

inline void QMedAAC::Channels(unsigned int channels)
{
    mpAACHdr->channels = BE32(channels);
}

inline void QMedAAC::SampleSize(unsigned int sampleSize)
{
    mpAACHdr->sampleSize = BE32(sampleSize);
}

inline void QMedAAC::AudioSpecificConfigSize(unsigned int audioSpecificConfigSize)
{
    mpAACHdr->audioSpecificConfigSize = BE32(audioSpecificConfigSize);
}

/*****************************************************************************
                                QMedMP2 Class
 *****************************************************************************/

inline void QMedMP2::Alloc(int baseVersion, int mp2Version)
{
    mQMedHdr.Alloc(baseVersion, sizeof(QMedMP2Struct));
    
    mpMP2Hdr = (QMedMP2Struct *) (mQMedHdr.Addr() + mQMedHdr.BoxSize() - sizeof(QMedMP2Struct));
    
    MP2Version(mp2Version);
}

inline void QMedMP2::Free()
{
    mQMedHdr.Free();
}

inline unsigned char * QMedMP2::Addr()
{
    return mQMedHdr.Addr();
}

inline unsigned long QMedMP2::HeaderSize(int baseVersion, int mp2Version)
{
    if (baseVersion == 0)
        return (sizeof(QMedStruct) + sizeof(QMedMP2Struct));
    else
        return (sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct) + sizeof(QMedMP2Struct));
}

inline unsigned long QMedMP2::MP2Version()
{
    return BE32(mpMP2Hdr->version);
}

inline unsigned int QMedMP2::SampleFrequency()
{
    return BE32(mpMP2Hdr->samplingFrequency);
}

inline unsigned int QMedMP2::Channels()
{
    return BE32(mpMP2Hdr->channels);
}

inline void QMedMP2::Addr(unsigned char * addr)
{
    mQMedHdr.Addr(addr);
    
    if (mQMedHdr.Version() == 0) 
        mpMP2Hdr = (QMedMP2Struct *) (addr + sizeof(QMedStruct));
    else
        mpMP2Hdr = (QMedMP2Struct *) (addr + sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct));
}

inline void QMedMP2::MP2Version(unsigned long version)
{
    mpMP2Hdr->version = BE32(version);
}

inline void QMedMP2::SampleFrequency(unsigned int sampleFrequency)
{
    mpMP2Hdr->samplingFrequency = BE32(sampleFrequency);
}

inline void QMedMP2::Channels(unsigned int channels)
{
    mpMP2Hdr->channels = BE32(channels);
}

/*****************************************************************************
                                QMedJPEG Class
 *****************************************************************************/

inline void QMedJPEG::Alloc(int baseVersion, int jpegVersion)
{
    mQMedHdr.Alloc(baseVersion, sizeof(QMedJPEGStruct));
    
    mpJPEGHdr = (QMedJPEGStruct *) (mQMedHdr.Addr() + mQMedHdr.BoxSize() - sizeof(QMedJPEGStruct));
    
    JPEGVersion(jpegVersion);
}

inline void QMedJPEG::Free()
{
    mQMedHdr.Free();
}

inline unsigned char * QMedJPEG::Addr()
{
    return mQMedHdr.Addr();
}

inline unsigned long QMedJPEG::HeaderSize(int baseVersion, int jpegVersion)
{
    if (baseVersion == 0)
        return (sizeof(QMedStruct) + sizeof(QMedJPEGStruct));
    else
        return (sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct) + sizeof(QMedJPEGStruct));
}

inline unsigned long QMedJPEG::JPEGVersion()
{
    return BE32(mpJPEGHdr->version);
}

inline unsigned long QMedJPEG::Width()
{
    return BE32(mpJPEGHdr->width);
}

inline unsigned long QMedJPEG::Height()
{
    return BE32(mpJPEGHdr->height);
}

inline unsigned long QMedJPEG::FrameTicks()
{
    return BE32(mpJPEGHdr->frameTicks);
}

inline void QMedJPEG::Addr(unsigned char * addr)
{
    mQMedHdr.Addr(addr);
    
    if (mQMedHdr.Version() == 0) 
        mpJPEGHdr = (QMedJPEGStruct *) (addr + sizeof(QMedStruct));
    else
        mpJPEGHdr = (QMedJPEGStruct *) (addr + sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct));
}

inline void QMedJPEG::JPEGVersion(unsigned long version)
{
    mpJPEGHdr->version = BE32(version);
}

inline void QMedJPEG::Width(unsigned long width)
{
    mpJPEGHdr->width = BE32(width);
}

inline void QMedJPEG::Height(unsigned long height)
{
    mpJPEGHdr->height = BE32(height);
}

inline void QMedJPEG::FrameTicks(unsigned long sampleTicks)
{
    mpJPEGHdr->frameTicks = BE32(sampleTicks);
}

/*****************************************************************************
                                QMed711 Class
 *****************************************************************************/

inline void QMed711::Alloc(int baseVersion, int g711Version)
{
    mQMedHdr.Alloc(baseVersion, sizeof(QMed711Struct));
    
    mp711Hdr = (QMed711Struct *) (mQMedHdr.Addr() + mQMedHdr.BoxSize() - sizeof(QMed711Struct));
    
    G711Version(g711Version);
}

inline void QMed711::Free()
{
    mQMedHdr.Free();
}

inline unsigned char * QMed711::Addr()
{
    return mQMedHdr.Addr();
}

inline unsigned long QMed711::HeaderSize(int baseVersion, int g711Version)
{
    if (baseVersion == 0)
        return (sizeof(QMedStruct) + sizeof(QMed711Struct));
    else
        return (sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct) + sizeof(QMed711Struct));
}

inline unsigned long QMed711::G711Version()
{
    return BE32(mp711Hdr->version);
}

inline unsigned int QMed711::AccessUnits()
{
    return BE32(mp711Hdr->accessUnits);
}

inline unsigned int QMed711::SampleSize()
{
    return BE32(mp711Hdr->sampleSize);
}

inline void QMed711::Addr(unsigned char * addr)
{
    mQMedHdr.Addr(addr);
    
    if (mQMedHdr.Version() == 0) 
        mp711Hdr = (QMed711Struct *) (addr + sizeof(QMedStruct));
    else
        mp711Hdr = (QMed711Struct *) (addr + sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct));
}

inline void QMed711::G711Version(unsigned long version)
{
    mp711Hdr->version = BE32(version);
}

inline void QMed711::AccessUnits(unsigned int accessUnits)
{
    mp711Hdr->accessUnits = BE32(accessUnits);
}

inline void QMed711::SampleSize(unsigned int sampleSize)
{
    mp711Hdr->sampleSize = BE32(sampleSize);
}

/*****************************************************************************
                                QMed722 Class
 *****************************************************************************/

inline void QMed722::Alloc(int baseVersion, int g722Version)
{
    mQMedHdr.Alloc(baseVersion, sizeof(QMed722Struct));
    
    mp722Hdr = (QMed722Struct *) (mQMedHdr.Addr() + mQMedHdr.BoxSize() - sizeof(QMed722Struct));
    
    G722Version(g722Version);
}

inline void QMed722::Free()
{
    mQMedHdr.Free();
}

inline unsigned char * QMed722::Addr()
{
    return mQMedHdr.Addr();
}

inline unsigned long QMed722::HeaderSize(int baseVersion, int g722Version)
{
    if (baseVersion == 0)
        return (sizeof(QMedStruct) + sizeof(QMed722Struct));
    else
        return (sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct) + sizeof(QMed722Struct));
}

inline unsigned long QMed722::G722Version()
{
    return BE32(mp722Hdr->version);
}

inline unsigned int QMed722::Bitrate()
{
    return BE32(mp722Hdr->bitrate);
}

inline unsigned int QMed722::AccessUnits()
{
    return BE32(mp722Hdr->accessUnits);
}

inline unsigned int QMed722::SampleSize()
{
    return BE32(mp722Hdr->sampleSize);
}

inline void QMed722::Addr(unsigned char * addr)
{
    mQMedHdr.Addr(addr);
    
    if (mQMedHdr.Version() == 0) 
        mp722Hdr = (QMed722Struct *) (addr + sizeof(QMedStruct));
    else
        mp722Hdr = (QMed722Struct *) (addr + sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct));
}

inline void QMed722::G722Version(unsigned long version)
{
    mp722Hdr->version = BE32(version);
}

inline void QMed722::Bitrate(unsigned long bitrate)
{
    mp722Hdr->bitrate = BE32(bitrate);
}

inline void QMed722::AccessUnits(unsigned int accessUnits)
{
    mp722Hdr->accessUnits = BE32(accessUnits);
}

inline void QMed722::SampleSize(unsigned int sampleSize)
{
    mp722Hdr->sampleSize = BE32(sampleSize);
}

/*****************************************************************************
                                QMed726 Class
 *****************************************************************************/

inline void QMed726::Alloc(int baseVersion, int g726Version)
{
    mQMedHdr.Alloc(baseVersion, sizeof(QMed726Struct));
    
    mp726Hdr = (QMed726Struct *) (mQMedHdr.Addr() + mQMedHdr.BoxSize() - sizeof(QMed726Struct));
    
    G726Version(g726Version);
}

inline void QMed726::Free()
{
    mQMedHdr.Free();
}

inline unsigned char * QMed726::Addr()
{
    return mQMedHdr.Addr();
}

inline unsigned long QMed726::HeaderSize(int baseVersion, int g726Version)
{
    if (baseVersion == 0)
        return (sizeof(QMedStruct) + sizeof(QMed726Struct));
    else
        return (sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct) + sizeof(QMed726Struct));
}

inline unsigned long QMed726::G726Version()
{
    return BE32(mp726Hdr->version);
}

inline unsigned int QMed726::Bitrate()
{
    return BE32(mp726Hdr->bitrate);
}

inline unsigned int QMed726::AccessUnits()
{
    return BE32(mp726Hdr->accessUnits);
}

inline unsigned int QMed726::SampleSize()
{
    return BE32(mp726Hdr->sampleSize);
}

inline void QMed726::Addr(unsigned char * addr)
{
    mQMedHdr.Addr(addr);
    
    if (mQMedHdr.Version() == 0) 
        mp726Hdr = (QMed726Struct *) (addr + sizeof(QMedStruct));
    else
        mp726Hdr = (QMed726Struct *) (addr + sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct));
}

inline void QMed726::G726Version(unsigned long version)
{
    mp726Hdr->version = BE32(version);
}

inline void QMed726::Bitrate(unsigned long bitrate)
{
    mp726Hdr->bitrate = BE32(bitrate);
}

inline void QMed726::AccessUnits(unsigned int accessUnits)
{
    mp726Hdr->accessUnits = BE32(accessUnits);
}

inline void QMed726::SampleSize(unsigned int sampleSize)
{
    mp726Hdr->sampleSize = BE32(sampleSize);
}

/*****************************************************************************
                                QMed728 Class
 *****************************************************************************/

inline void QMed728::Alloc(int baseVersion, int g728Version)
{
    mQMedHdr.Alloc(baseVersion, sizeof(QMed728Struct));
    
    mp728Hdr = (QMed728Struct *) (mQMedHdr.Addr() + mQMedHdr.BoxSize() - sizeof(QMed728Struct));
    
    G728Version(g728Version);
}

inline void QMed728::Free()
{
    mQMedHdr.Free();
}

inline unsigned char * QMed728::Addr()
{
    return mQMedHdr.Addr();
}

inline unsigned long QMed728::HeaderSize(int baseVersion, int g728Version)
{
    if (baseVersion == 0)
        return (sizeof(QMedStruct) + sizeof(QMed728Struct));
    else
        return (sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct) + sizeof(QMed728Struct));
}

inline void QMed728::Addr(unsigned char * addr)
{
    mQMedHdr.Addr(addr);
    
    if (mQMedHdr.Version() == 0) 
        mp728Hdr = (QMed728Struct *) (addr + sizeof(QMedStruct));
    else
        mp728Hdr = (QMed728Struct *) (addr + sizeof(QMedStruct) + sizeof(QMedVer1InfoStruct));
}

inline unsigned long QMed728::G728Version()
{
    return BE32(mp728Hdr->version);
}

inline unsigned int QMed728::AccessUnits()
{
    return BE32(mp728Hdr->accessUnits);
}

inline void QMed728::G728Version(unsigned long version)
{
    mp728Hdr->version = BE32(version);
}

inline void QMed728::AccessUnits(unsigned int accessUnits)
{
    mp728Hdr->accessUnits = BE32(accessUnits);
}

#endif /* __cplusplus */


#endif  /* __QMED_HH */
/*******************************************************************************
* vi: set shiftwidth=4 tabstop=8 softtabstop=4 expandtab nosmarttab:
*******************************************************************************/
