/*******************************************************************************
*
* The content of this file or document is CONFIDENTIAL and PROPRIETARY
* to Mobilygen Corporation.  It is subject to the terms of a
* License Agreement between Licensee and Mobilygen Corporation.
* restricting among other things, the use, reproduction, distribution
* and transfer.  Each of the embodiments, including this information and
* any derivative work shall retain this copyright notice.
*
* Copyright 2010 Mobilygen Corporation.
* All rights reserved.
*
* QuArc is a registered trademark of Mobilygen Corporation.
*
* Version: SDK7r26554
*
*******************************************************************************/


/// \defgroup JPEGEncoderControlObject JPEG Encode
/// \{

/// \page jpegencoder JPEG Encoder
/// \latexonly
/// \chapter{JPEG Encoder}
/// \input{jpeg_encoder}
/// \endlatexonly
///
/// \section jpegencstatus Status Block
///
/// The JPEGEncoder status block contains information related to state and 
/// performance.
/// \latexonly
/// \input{structJPEGENC__STATUS}
/// \endlatexonly

#ifndef __QJPEGENC_H
#define __QJPEGENC_H

//////////////////////////////////////////////////////////////////////
//
// Commands
//

/// \brief This command binds a video input source to the encoder.  LVPP
///        and NVPP objects are supported as sources.  Incoming video from
///        the source will be encoded as it arrives.  Only one source can
///        be bound.
/// \param ctrlObjID Control object ID of the video source to be bound
/// \return Code:  1 = Success, 0 = Failure
#define Q_JPEGENC_CMD_BIND_VIDEO_INPUT      1

/// \brief This command unbinds the current video input source from the encoder.
///        Encoding of video from that source will stop.  This should only be used when
///        the encoder is in stop state (following a Stop or Flush command).
/// \param streamID 32 bit stream ID
/// \return Code:  1 = Success, 0 = Failure
#define Q_JPEGENC_CMD_UNBIND_VIDEO_INPUT    2

/// \brief Set the stream ID which will appear in QBox frames generated by this encoder, in
///        the sampleStreamID field.  This allows the host to examine a QBox stream containing
///        multiple encoded bitstreams, and identify which QBoxes would be generated by this
///        encoder.
/// \param streamID 16 bit unsigned stream ID
/// \return Code:  1 = Success, 0 = Failure
#define Q_JPEGENC_CMD_SET_STREAM_ID         3

/// \brief Begin encoding using the currently set and activated parameters.
#define Q_JPEGENC_CMD_RECORD                4

/// \brief Set the quality factor of the encoder.
/// \param qualityFactor The quality factor. Valid values from 0 to 100.
#define Q_JPEGENC_CMD_SET_QUALITY_FACTOR    5

/// \brief Signal the encoder that the host has finished reading a bitstream block, so
///        the encoder may reuse that space for future encoding.  The host will receive
///        bitstream block ready events specifying the location of the encoded bitstream
///        in memory.  The encoded bitstream is stored in a ring buffer, so the
///        host does not have to sent bitstream block done for every bitstream block
///        ready.  Instead, sending bitstream block done for a specific bitstream block
///        releases that block and ALL previously sent blocks.  For example, it is possible
///        to send bitstream block done only for every fifth block.
/// \param address Address field from the bitstream block ready event, specifying the block
///        (and all previous blocks) that the host is done processing
/// \param offset Offset field from the bitstream block ready event, specifying the block
///        (and all previous blocks) that the host is done processing
#define Q_JPEGENC_CMD_BITSTREAM_BLOCK_DONE  6

/// \brief Set a value in the main encoder configuration array.  Encoder configuration is
///        stored in double buffered arrays of 32 bit parameters, listed in the Q_JPEGENC_CMP_
///        defines.  This API sets entries in the inactive array.  They will have no effect
///        until the Q_JPEGENC_CMD_ACTIVATE_CFG command is sent.  This allows many
///        parameters to be set sequentially but take effect at the same time.
/// 
///        Parameters take effect at frame boundaries, never while a frame is being encoded.
///        Many parameters can be set on the fly, but most should be set before encoding begins.
/// \param param Parameter defined in Q_JPEGENC_CMP_ list.
/// \param value 32 bit value
#define Q_JPEGENC_CMD_SET_PARAM             7

/// \brief Activate any changed settings in the encoder configuration array.  Activate takes
///        effect between frames, and the pending configuration array (modified by any set param
///        calls since the last activate) is copied into the active configuration array.
#define Q_JPEGENC_CMD_ACTIVATE_CFG          8

/// \brief Signals the encoder to take a number of snapshots instantaneously.
/// \param shots Number of consecutive snapshots to be taken.
#define Q_JPEGENC_CMD_TAKE_SNAP_SHOTS       9

/// \brief Configures the bitrate control. Can turn on, turn off, or reset bitrate.
/// \param bitrate Target bitrate. Bits per second.
/// \param behavior The behavior of the bitrate control. Can be either conservative(0), aggressive(1),
///        or customized(2).
/// \param reset Reset the bitrate calculation. Instantly obtains new target bitrate for the
///        encoded bitstream rather than slowly averages to new bitrate over the whole bitstream.
/// \param disable Disables bitrate control.
#define Q_JPEGENC_CMD_BITRATE_CONTROL       10

/// \brief Set a value in the burnin configuration array.  The burnin configuration
///        array functions the same was as the main encoder configuration array.  It is a double
///        buffered array of 32 bit configuration parameters, and changes to the parameters take
///        effect at the next activate command.
/// \param param Parameter defined in Q_JPEGENC_BURNIN_CMP_ list.
/// \param value 32 bit value
/// \return Code:  1 = Success, 0 = Failure
#define Q_JPEGENC_CMD_SET_BURNIN_PARAM      11

/// \brief Set the minimum quality factor of the encoder. Only allows to use this command when
///        the behavior is set to 'CUSTOMIZED'
/// \param qualityFactor The min quality factor. Valid values from 0 to 100.
#define Q_JPEGENC_CMD_SET_MIN_QUALITY_FACTOR    12

/// \brief Set the maximum quality factor of the encoder. Only allows to use this command when
///        the behavior is set to 'CUSTOMIZED'
/// \param qualityFactor The max quality factor. Valid values from 0 to 100.
#define Q_JPEGENC_CMD_SET_MAX_QUALITY_FACTOR    13

/// \brief Stop encoding. Will flush the data in the input queue.
#define Q_JPEGENC_CMD_STOP                  14

/// \brief Set a value in the double-buffered quant matrix. Uses Q_JPEGENC_CMD_ACTIVATE_QUANT_MATRIX to
///        activate all changes at once.
/// \param type The type of quant matrix. JPEG encoder has two quant matries, one for luma and one for chroma.
///        Q_JPEGENC_CFP_QUANT_MATRIX_TYPE_LUMA Set a value in the luma matrix
///        Q_JPEGENC_CFP_QUANT_MATRIX_TYPE_CHROMA Set a value in the chroma matrix
/// \param index Index into the matrix in raster scanning order, ie,
///        [0, 1, 2, 3, 4, 5, 6, 7,
///         8, 9, 10, ...        
///         ...
///         ...                 63]
/// \param value integer. 0-255.
#define Q_JPEGENC_CMD_SET_QUANT_MATRIX      15

/// \brief Activate the modified quant matries
#define Q_JPEGENC_CMD_ACTIVATE_QUANT_MATRIX 16

/// \brief Set the number of blocks (QBoxes) reported in each bitstream block ready event.  The
///        bitstream block ready event will wait until that many QBoxes of data have been
///        encoded before sending them all in one event.
/// \param value blocks per event (1 to 5)
#define Q_JPEGENC_CMD_SET_TRANSFER_SIZE     17

#define Q_JPEGENC_MAX_CMD                   18


///////////////////////////////////////////////////////////////////////
//
// Command Payloads
//
typedef struct {
    int ctrlObjID;
} STRUCT_Q_JPEGENC_CMD_BIND_VIDEO_INPUT;

typedef struct {
    unsigned int streamID;
} STRUCT_Q_JPEGENC_CMD_SET_STREAM_ID;

typedef struct {
    unsigned int qualityFactor;
} STRUCT_Q_JPEGENC_CMD_SET_QUALITY_FACTOR;

typedef struct {
    unsigned int qualityFactor;
} STRUCT_Q_JPEGENC_CMD_SET_MIN_QUALITY_FACTOR;

typedef struct {
    unsigned int qualityFactor;
} STRUCT_Q_JPEGENC_CMD_SET_MAX_QUALITY_FACTOR;

typedef struct {
    unsigned int address;
    unsigned int size;
} STRUCT_Q_JPEGENC_CMD_BITSTREAM_BLOCK_DONE;

typedef struct {
    unsigned int param;
    int value;
} STRUCT_Q_JPEGENC_CMD_SET_PARAM;

typedef struct {
    int shots;
} STRUCT_Q_JPEGENC_CMD_TAKE_SNAP_SHOTS;

typedef struct {
    unsigned int bitrate;
    unsigned int fps;
    int behavior;
    int reset;
    int disable;
} STRUCT_Q_JPEGENC_CMD_BITRATE_CONTROL;

typedef struct {
    unsigned int idx;
    unsigned int param;
    int value;
} STRUCT_Q_JPEGENC_CMD_SET_INDEX_PARAM;

typedef struct {
    int type;
    int index;
    int value;
} STRUCT_Q_JPEGENC_CMD_SET_QUANT_MATRIX;

typedef struct {
    int value;
} STRUCT_Q_JPEGENC_CMD_SET_TRANSFER_SIZE;

//////////////////////////////////////////////////////////////////////
//
// Events
//
#define JPEGENC_EVENT(ID) ((JPEGENC_CTRLOBJ_TYPE << 16) | ID)

/// \brief The bitstream block ready event is used to notify the host that new encoded
///        bitstream data is available to be read.  Events are generated for every 1-5
///        encoded blocks (configurable using API function setAckInterval()).
///        Each block contains a single QBox which contains one JPEG encoded frame.
/// \param streamID Stream ID of the encoder which generated the events.
/// \param numFrames Number of frames (bitstream blocks) in this event
/// \param frameAddr Start address of encoded frame in codec memory
/// \param frameSize Size of encoded frame, in bytes
#define Q_JPEGENC_EV_BITSTREAM_BLOCK_READY  JPEGENC_EVENT(1)

/// \brief A Flush operation on the encoder has completed.  Flush is started by sending
///        the encoder flush command, or sending a Flush command to the LVPP or NVPP which
///        the encoder is bound to.  The event is sent once the encoder has finished
///        encoding any remaining frames, and bitstream block ready events have been sent
///        to the host for all of those frames.
/// \param streamID Stream ID of the encoder which is flushed.
#define Q_JPEGENC_EV_BITSTREAM_FLUSHED      JPEGENC_EVENT(2)

/// \brief This event is reserved.
#define Q_JPEGENC_EV_BURNIN_UPDATED         JPEGENC_EVENT(3)


typedef struct {
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        streamID;
    unsigned int        numFrames;
    unsigned int        frame0Addr;
    unsigned int        frame0Size;
    unsigned int        frame1Addr;
    unsigned int        frame1Size;
    unsigned int        frame2Addr;
    unsigned int        frame2Size;
    unsigned int        frame3Addr;
    unsigned int        frame3Size;
    unsigned int        frame4Addr;
    unsigned int        frame4Size;
} STRUCT_Q_JPEGENC_EV_BITSTREAM_BLOCK_READY;

typedef struct {
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        streamID;
} STRUCT_Q_JPEGENC_EV_BITSTREAM_FLUSHED;


typedef struct {
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        index;
} STRUCT_Q_JPEGENC_EV_BURNIN_UPDATED;

//////////////////////////////////////////////////////////////////////
//
// Single buffered Configuration parameters

/// \brief When set the encoder will acknowledge its own bitstream blocks
/// which removes the need for the host to send BITSTREAM_BLOCK_DONE 
/// commands.  Note that this means the encoder can overwrite un-read
/// data and the host can receive a corrupted bitstream
#define Q_JPEGENC_CFG_BBLOCK_SELF_ACK                  1


//////////////////////////////////////////////////////////////////////
//
// Double-buffer Configuration parameters
//
// Generated from mediaware.h JPEGENC_CFG s/JPEGENC_CFG/Q_JPEGENC_CMP/
/// \brief Encoding interval at 90khz ticks. If this parameter is larger than the
/// interval of video frames that are being input to the JPEG encoder, then the 
/// encoder will reduce the frame rate internally to the targeted rate.
/// \param value Frame interval in 90khz ticks. Use -1 to match incoming video.
#define Q_JPEGENC_CMP_CAPTURE_INTERVAL               0
/// \brief Controls when to start
/// encoding the incoming frames. In other words, it can be used to skip a number
/// of frames at the beginning of the incoming video stream. For example, if the
/// parameter is 10, the first 10 input frames will be skipped and the encoder will
/// start to encode (and continue to encode) after it sees the 11th frame.
#define Q_JPEGENC_CMP_CAPTURE_START_INDEX            1
/// \brief Max number of frames that the encoder will encode during its entire life of
/// operation. If the number is reached, the encoder will go into a dormant state.
/// \param value Number of frames. Use -1 to tell encoder to go on encoding forever.
#define Q_JPEGENC_CMP_CAPTURE_MAX_ENCODED_FRAMES     2
/// \brief Allow the bitstream output of the encoder to be disabled.  The encoder
///        will encode video as usual, but the output bitstream will be discarded
///        instead of being sent to the host.  Bitstream block ready events will
///        not be received.  This is useful for debugging, to determine if a problem
///        lies in capture/encode or in transfer of data to the host.
/// \param enable
#define Q_JPEGENC_CMP_BITSTREAM_BUFFER               3
/// \brief Set the size of guardband per macroblock. Unit in bytes.
#define Q_JPEGENC_CMP_AU_GUARD_BAND_PER_MB           4
/// \brief Enables UVC (USB Video Class) mode.
/// \param value 1 = enables UVC mode, 0 = disables
#define Q_JPEGENC_CMP_ENABLE_UVC                     5
/// \brief UVC frame interval
/// \param value Frame interval on a 6Mhz clock
#define Q_JPEGENC_CMP_UVC_FRAME_INTERVAL             6
/// \brief UVC packet size
/// \param value Size of each UVC packet. Max 1024 bytes and must be divisible by 8
#define Q_JPEGENC_CMP_UVC_PACKET_SIZE                7
/// \brief Word-align the encoded JPEG frames by padding them with zeroes.
/// \param value Number of words (4 bytes) of the alignment. 0 = no alignment. 1 = 1 word alignment.
#define Q_JPEGENC_CMP_WORD_ALIGN_FRAME_SIZE          8
/// \brief Enables QMED box.
/// \param value 1 = enables QMED box, 0 = disable
#define Q_JPEGENC_CMP_ENABLE_QMED                    9
/// \brief Controls how the PTS's of the qbox headers of encoded frames are generated. If passthrough
///        mode is enabled, the generated PTS's will be direct copies of incoming video source. Otherwise
///        the generated PTS's will be calculated by the encoder, starting from 0 and counting up by
///        encoding interval. The default option is enabled.
/// \param value 1 = enable passthrough mode. 0 = disable
#define Q_JPEGENC_CMP_PTS_PASSTHROUGH               10

// Generated from mediaware.h JPEGENC_BURNIN_CFG s/JPEGENC_BURNIN_CFG/Q_JPEGENC_CMP_BURNIN/
/// \brief This parameter enables text burn in.
/// \param value 0 = Disable, 1 = Enable.
///
#define Q_JPEGENC_CMP_BURNIN_ENABLE                  0
/// \brief This parameter sets the font used for burn in.
/// \param value Codec byte address of loaded font.
///
#define Q_JPEGENC_CMP_BURNIN_FONT_ADDR               1
/// \brief This parameter sets the font size for burn in.
/// \param value 8, 16 or 32.
///
#define Q_JPEGENC_CMP_BURNIN_FONT_SIZE               2
/// \brief This parameter sets the burn in update mode.
/// \param value JPEGENC_BURNIN_UPDATE_MODE_*
///
#define Q_JPEGENC_CMP_BURNIN_UPD_MODE                3
/// \brief This parameter sets the burn in x position.
/// \param value Positive integer.
///
#define Q_JPEGENC_CMP_BURNIN_X                       4
/// \brief This parameter sets the burn in y position.
/// \param value Positive integer.
///
#define Q_JPEGENC_CMP_BURNIN_Y                       5
/// \brief This parameter enables frame numbers to be inserted in the burn in text.
/// \param value 0 = Disable, 1 = Enable.
///
#define Q_JPEGENC_CMP_BURNIN_FRAME_NUM_ENABLE        6
/// \brief This parameter sets the position in the text that the frame number is inserted at.
/// \param value Positive integer.
///
#define Q_JPEGENC_CMP_BURNIN_FRAME_NUM_POS           7
/// \brief This parameter sets the frame number minimum value.
/// \param value Positive integer.
///
#define Q_JPEGENC_CMP_BURNIN_FRAME_NUM_L             8
/// \brief This parameter sets the frame number maximum value.
/// \param value Positive integer.
///
#define Q_JPEGENC_CMP_BURNIN_FRAME_NUM_H             9
/// \brief This parameter sets the length of the burn in text.
/// \param value Positive integer.
///
#define Q_JPEGENC_CMP_BURNIN_STR_LEN                10
/// \brief This parameter sets the characters 0 - 3 of the burn in text.
/// \param value Positive integer representing characters.
///
#define Q_JPEGENC_CMP_BURNIN_STR4_0                 11
/// \brief This parameter sets the characters 4 - 7 of the burn in text.
/// \param value Positive integer representing characters.
///
#define Q_JPEGENC_CMP_BURNIN_STR4_1                 12
/// \brief This parameter sets the characters 8 - 11 of the burn in text.
/// \param value Positive integer representing characters.
///
#define Q_JPEGENC_CMP_BURNIN_STR4_2                 13
/// \brief This parameter sets the characters 12 - 15 of the burn in text.
/// \param value Positive integer representing characters.
///
#define Q_JPEGENC_CMP_BURNIN_STR4_3                 14
/// \brief This parameter sets the characters 16 - 19 of the burn in text.
/// \param value Positive integer representing characters.
///
#define Q_JPEGENC_CMP_BURNIN_STR4_4                 15
/// \brief This parameter sets the characters 20 - 24 of the burn in text.
/// \param value Positive integer representing characters.
///
#define Q_JPEGENC_CMP_BURNIN_STR4_5                 16

// Generated from mediaware.h JPEGENC_BURNIN_UPDATE_MODE s/JPEGENC_BURNIN_UPDATE_MODE/Q_JPEGENC_CFP_BURNIN_UPDATE_MODE/
/// \brief update on next frame,
#define Q_JPEGENC_CFP_BURNIN_UPDATE_MODE_NOW         0
/// \brief update on frame number rollover.
#define Q_JPEGENC_CFP_BURNIN_UPDATE_MODE_ROLLOVER    1

// Generated from mediaware.h JPEGENC_QUANT_MATRIX_TYPE s/JPEGENC_QUANT_MATRIX_TYPE/Q_JPEGENC_CFP_QUANT_MATRIX_TYPE/
#define Q_JPEGENC_CFP_QUANT_MATRIX_TYPE_LUMA         0
#define Q_JPEGENC_CFP_QUANT_MATRIX_TYPE_CHROMA       1

/// \}
#endif
