/*******************************************************************************
*
* The content of this file or document is CONFIDENTIAL and PROPRIETARY
* to Mobilygen Corporation.  It is subject to the terms of a
* License Agreement between Licensee and Mobilygen Corporation.
* restricting among other things, the use, reproduction, distribution
* and transfer.  Each of the embodiments, including this information and
* any derivative work shall retain this copyright notice.
*
* Copyright 2010 Mobilygen Corporation.
* All rights reserved.
*
* QuArc is a registered trademark of Mobilygen Corporation.
*
* Version: SDK7r26554
*
*******************************************************************************/


/// \defgroup AVCEncoderControlObject AVC Encoder
/// \{

/// \page avcencoder AVC Encoder
/// \latexonly
/// \chapter{AVC Encoder}
/// \input{avc_encoder}
/// \endlatexonly
///
/// \section avcencstatus Status Block
///
/// The AVC Encoder status block contains information related to state and performance.
/// \latexonly
/// \input{structAVCENC__STATUS}
/// \endlatexonly
#ifndef __QAVCENC_HH
#define __QAVCENC_HH

#include "qglobal.h"

//////////////////////////////////////////////////////////////////////
//
// Commands
//

/// \brief This command binds a video input source to the encoder.  LVPP
///        and NVPP objects are supported as sources.  Incoming video from
///        the source will be encoded as it arrives.  Only one source can
///        be bound.
/// \param ctrlObjID Control object ID of the video source to be bound
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_BIND_VIDEO_INPUT                   1

/// \brief This command unbinds the current video input source from the encoder.
///        Encoding of video from that source will stop.  This should only be used when
///        the encoder is in stop state (following a Stop or Flush command).
/// \param streamID 32 bit stream ID
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_UNBIND_VIDEO_INPUT                 2

/// \brief Set the stream ID which will appear in QBox frames generated by this encoder, in
///        the sampleStreamID field.  This allows the host to examine a QBox stream containing
///        multiple encoded bitstreams, and identify which QBoxes would be generated by this
///        encoder.
/// \param streamID 16 bit unsigned stream ID
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_SET_STREAM_ID                      3

/// \brief Begin encoding using the currently set and activated parameters.
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_RECORD                             4

/// \brief Stop encoding gracefully.  The current sub-GOP (for B frame encoding) will
///        finish encoding, then the encoding will stop.  Once bitstream block ready
///        events have been sent to the host for all encoded frames, a bitstream
///        flushed event will be sent to the host.  The return code indicates that
///        the flush command has started, not that it has finished.
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_FLUSH                              5

/// \brief Stop functions in the same way as Flush except that encoding is terminated
///        immediately rather than at the end of the current sub-GOP.  Any currently
///        encoding frame will finish encoding, but any further B frames within the
///        current sub-GOP will not be encoded.  All encoded frames still result in
///        bitstream block ready events being sent to the host, followed by a bitstream
///        flushed event.
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_STOP                               6

/// \brief Signal the encoder that the host has finished reading a bitstream block, so
///        the encoder may reuse that space for future encoding.  The host will receive
///        bitstream block ready events specifying the location of the encoded bitstream
///        in memory.  The encoded bitstream is stored in a ring buffer, so the
///        host does not have to sent bitstream block done for every bitstream block
///        ready.  Instead, sending bitstream block done for a specific bitstream block
///        releases that block and ALL previously sent blocks.  For example, it is possible
///        to send bitstream block done only for every fifth block.
/// \param address Address field from the bitstream block ready event, specifying the block
///        (and all previous blocks) that the host is done processing
/// \param offset Offset field from the bitstream block ready event, specifying the block
///        (and all previous blocks) that the host is done processing
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_BITSTREAM_BLOCK_DONE               7

/// \brief Set a value in the main encoder configuration array.  Encoder configuration is
///        stored in double buffered arrays of 32 bit parameters, listed in the Q_AVCENC_CMP_
///        defines.  This API sets entries in the inactive array.  They will have no effect
///        until the Q_AVCENC_CMD_ACTIVATE_VIDEO_ENC_CFG command is sent.  This allows many
///        parameters to be set sequentially but take effect at the same time.
/// 
///        Parameters take effect at frame boundaries, never while a frame is being encoded.
///        Many parameters can be set on the fly, but most should be set before encoding begins.
/// \param param Parameter defined in Q_AVCENC_CMP_ list.
/// \param value 32 bit value
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_SET_VIDEO_ENC_PARAM                8

/// \brief This parameter is reserved.
/// \internal Set quantization scaling matrix entry.  This API functions the same as the main
///           Q_AVCENC_CMD_SET_VIDEO_ENC_PARAM API, but has an additional "table" field because
///           each part of the QS matrix set is stored in a different table.
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_SET_VIDEO_ENC_QS_PARAM             9

/// \brief This parameter is reserved.
/// \internal Set deadzone matrix entry.  This API functions the same as the main
///           Q_AVCENC_CMD_SET_VIDEO_ENC_PARAM API, but has an additional "table" field because
///           there is a seperate deadzone table for each region.
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_SET_VIDEO_ENC_DZ_PARAM             10

/// \brief This parameter is reserved.
/// \internal Set region parameter.  This API functions the same as the main
///           Q_AVCENC_CMD_SET_VIDEO_ENC_PARAM API, but has an additional "table" field because
///           there are eight regions.
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_SET_VIDEO_ENC_REGION_PARAM         11

/// \brief Set a value in the rate control configuration array.  The rate control array functions
///        the same way as the main encoder configuration array.  It is a double buffered array of
///        32 bit configuration parameters, and new parameters take effect at the next activate
///        command.
/// \param param Parameter defined in Q_AVCENC_RC_CMP_ list.
/// \param value 32 bit value
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_SET_VIDEO_ENC_RATE_CONTROL_PARAM   12

/// \brief Activate any changed settings in all encoder configuration arrays.  Activate takes
///        effect between frames, and the pending configuration array (modified by any set param
///        calls since the last activate) is copied into the active configuration array.
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_ACTIVATE_VIDEO_ENC_CFG             13

/// \brief Start a new GOP, with the IDR frame occurring at the next P frame that has yet to be
///        encoded.  This will shorten the current GOP.  In IP encoding, the new GOP will always
///        start on the next frame to be encoded (not the next frame to be captured).  With B
///        frames, if the next frame is a B frame then the new GOP will not start until the
///        next P frame encode.
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_FORCE_NEW_GOP                      14

/// \brief This command is reserved.
/// \internal Set a seed for the encoder's software random number generator.  Currently the
///           generator is not used.
#define Q_AVCENC_CMD_VIDEO_ENC_RAND_RESEED              15

/// \brief Set a parameter in a privacy region.  Privacy regions allow black rectangles to be drawn
///        over the image on macroblock boundaries.  Each region specifies the coordinates of a region,
///        in macrobocks, and whether the region is enabled.
/// \param idx Privacy region to modify parameters for
/// \param param Parameter to modify, values are Q_AVCENC_CMP_PREGION_*
/// \param value New parameter value
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_SET_VIDEO_ENC_PRIVACY_PARAM        16

/// \brief Set a value in the object tracking configuration array.  The object tracking configuration
///        array functions the same was as the main encoder configuration array.  It is a double
///        buffered array of 32 bit configuration parameters, and changes to the parameters take
///        effect at the next activate command.
/// \param param Parameter defined in Q_AVCENC_VA_OT_CMP_ list.
/// \param 32 bit value
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_SET_VIDEO_ENC_VA_PARAM             17

/// \brief Set the number of blocks (QBoxes) reported in each bitstream block ready event.  The
///        bitstream block ready event will wait until that many QBoxes of data have been
///        encoded before sending them all in one event.
/// \param value blocks per event (1 to 5)
#define Q_AVCENC_CMD_SET_VIDEO_ENC_TRANSFER_SIZE        18

/// \brief Set a value in the burnin configuration array.  The burnin configuration
///        array functions the same was as the main encoder configuration array.  It is a double
///        buffered array of 32 bit configuration parameters, and changes to the parameters take
///        effect at the next activate command.
/// \param param Parameter defined in Q_AVCENC_BURNIN_CMP_ list.
/// \param value 32 bit value
/// \return Code:  1 = Success, 0 = Failure
#define Q_AVCENC_CMD_SET_VIDEO_ENC_BURNIN_PARAM         19

/// \brief Add a rectangular area to the ROI of the specified region, which will trigger
///        motion detection.
/// \param index Region to add the ROI to
/// \param width Width of the ROI rectangle
/// \param height Height of the ROI rectangle
/// \param offsetX X offset of the ROI rectangle
/// \param offsetY Y offset of the ROI rectangle
#define Q_AVCENC_CMD_ADD_MD_ROI                         20

/// \brief Subtract a rectangular area to the ROI of the specified region, so that the area
///        will not trigger motion detection.
/// \param index Region to add the ROI to
/// \param width Width of the ROI rectangle
/// \param height Height of the ROI rectangle
/// \param offsetX X offset of the ROI rectangle
/// \param offsetY Y offset of the ROI rectangle
#define Q_AVCENC_CMD_SUB_MD_ROI                         21 

/// \brief Set a value in the motion detecting configuration array. 
// \param value  Parameter defined in Q_AVCENC_CMP_ROI_ 
#define Q_AVCENC_CMD_SET_VIDEO_ENC_ROI_PARAM            22 

#define Q_AVCENC_CMD_CHECK_PERFORMANCE                  23

/// \brief ADD a rectangular area to the Privacy Zone of the specified region, so that the area
///        will not trigger motion detection.
/// \param width MB Width of the Privacy Zone rectangle
/// \param height MB Height of the Privacy Zone rectangle
/// \param offsetX MB X offset of the Privacy Zone rectangle
/// \param offsetY MB Y offset of the Privacy Zone rectangle
#define Q_AVCENC_CMD_ADD_VA_PRIV_ZONE                   24 

/// \brief Subtract a rectangular area to the Privacy Zone of the specified region, so that the area
///        will not trigger motion detection.
/// \param width MB Width of the Privacy Zone rectangle
/// \param height MB Height of the Privacy Zone rectangle
/// \param offsetX MB X offset of the Privacy Zone rectangle
/// \param offsetY MB Y offset of the Privacy Zone rectangle
#define Q_AVCENC_CMD_SUB_VA_PRIV_ZONE                   25 

#define Q_AVCENC_MAX_CMD                                27


///////////////////////////////////////////////////////////////////////
//
// Command Payloads
//

typedef struct
{
    int ctrlObjID;
} STRUCT_Q_AVCENC_CMD_BIND_VIDEO_INPUT;

typedef struct
{
    unsigned int streamID;
} STRUCT_Q_AVCENC_CMD_SET_STREAM_ID;

typedef struct
{
    unsigned int address;
    unsigned int size;
} STRUCT_Q_AVCENC_CMD_BITSTREAM_BLOCK_DONE;

typedef struct
{
    unsigned int param;
    int value;
} STRUCT_Q_AVCENC_CMD_SET_VIDEO_ENC_PARAM;

typedef struct
{
    unsigned int table;
    unsigned int param;
    int value;
} STRUCT_Q_AVCENC_CMD_SET_VIDEO_ENC_QS_PARAM;

typedef struct
{
    unsigned int table;
    unsigned int param;
    int value;
} STRUCT_Q_AVCENC_CMD_SET_VIDEO_ENC_DZ_PARAM;

typedef struct
{
    unsigned int region;
    unsigned int param;
    int value;
} STRUCT_Q_AVCENC_CMD_SET_VIDEO_ENC_REGION_PARAM;

typedef struct
{
    unsigned int param;
    int value;
} STRUCT_Q_AVCENC_CMD_SET_VIDEO_ENC_RATE_CONTROL_PARAM;

typedef struct
{
    unsigned int idx;
    unsigned int param;
    int value;
} STRUCT_Q_AVCENC_CMD_SET_VIDEO_ENC_INDEX_PARAM;


typedef struct
{
    unsigned int param;
    int value;
} STRUCT_Q_AVCENC_CMD_SET_VA_PARAM;

typedef struct
{
    int value;
} STRUCT_Q_AVCENC_CMD_SET_VIDEO_ENC_TRANSFER_SIZE;

typedef struct
{
    unsigned int index;
    unsigned int width;
    unsigned int height;
    unsigned int offsetX;
    unsigned int offsetY;
} STRUCT_Q_AVCENC_CMD_ADD_MD_ROI;

typedef struct
{
    unsigned int index;
    unsigned int width;
    unsigned int height;
    unsigned int offsetX;
    unsigned int offsetY;
} STRUCT_Q_AVCENC_CMD_SUB_MD_ROI;

typedef struct
{
    unsigned int width;
    unsigned int height;
    unsigned int offsetX;
    unsigned int offsetY;
} STRUCT_Q_AVCENC_CMD_ADD_VA_PRIV_ZONE;

typedef struct
{
    unsigned int width;
    unsigned int height;
    unsigned int offsetX;
    unsigned int offsetY;
} STRUCT_Q_AVCENC_CMD_SUB_VA_PRIV_ZONE;


//////////////////////////////////////////////////////////////////////
//
// Command Parameter Values
//

#define Q_AVCENC_CFP_PORT_TYPE_VIDEO                1
#define Q_AVCENC_CFP_PORT_TYPE_STATS                2



//////////////////////////////////////////////////////////////////////
//
// Events
//
#define AVCENC_EVENT(ID) ((AVCENC_CTRLOBJ_TYPE << 16) | ID)

/// \brief The bitstream block ready event is used to notify the host that new encoded
///        bitstream data is available to be read.  Events are generated for every 1-5
///        encoded blocks (configurable using Q_AVCENC_CMD_SET_VIDEO_ENC_TRANSFER_SIZE).
///        Each block contains a single QBox which normally contains all NALUs for one
///        encoded video frame.
/// \param streamID Stream ID of the encoder which generated the events.
/// \param numFrames Number of frames (bitstream blocks) in this event
/// \param frameNAddr Start address of frame N in codec memory
/// \param frameNSize Size of frame N, in bytes
#define Q_AVCENC_EV_BITSTREAM_BLOCK_READY       AVCENC_EVENT(1)

/// \brief A Flush operation on the encoder has completed.  Flush is started by sending
///        the encoder flush command, or sending a Flush command to the LVPP or NVPP which
///        the encoder is bound to.  The event is sent once the encoder has finished
///        encoding any remaining frames, and bitstream block ready events have been sent
///        to the host for all of those frames.
/// \param streamID Stream ID of the encoder which is flushed.
#define Q_AVCENC_EV_BITSTREAM_FLUSHED           AVCENC_EVENT(3)

/// \brief The video encoder's bitstream buffer has filled up due to the host not removing
///        data fast enough.  In this case, it is likely that the encoder dropped video
///        input frames due to insufficient storage space for encoding.  The bitstream
///        is still decodable, but frames are dropped.
/// \param streamID Stream ID of the encoder which has a full buffer.
/// \param bufferSize Total size of the buffer, in bytes
/// \param bufferEmptiness Free space in the buffer, in bytes
/// \param guardBand Guard band of the buffer, in bytes.  The buffer is considered full
///        if the amount of free space is less than the guard band.
#define Q_AVCENC_EV_VIDEO_ENC_BUFFER_FULL       AVCENC_EVENT(4)

/// \brief A frame was dropped due to rate control.  Encoding the frame violated the target
///        bitrate, therefore the frame was dropped to reduce the bitrate.
/// \param streamID Stream ID of the encoder which dropped a frame.
/// \param PTS_MSB Most significant 32 bits of the dropped frame's 64 bit PTS
/// \param PTS_LSB Least significant 32 bits of the dropped frame's 64 bit PTS
#define Q_AVCENC_EV_VIDEO_ENC_RC_FRAME_DROP     AVCENC_EVENT(5)

/// \brief This event is reserved.
#define Q_AVCENC_EV_VIDEO_FRAME_ENCODED         AVCENC_EVENT(6)

/// \brief This event is reserved.
#define Q_AVCENC_EV_VA_OT_RECT_READY            AVCENC_EVENT(7)

/// \brief This event is reserved.
#define Q_AVCENC_EV_VA_GET_FRAME_INFO           AVCENC_EVENT(8)

/// \brief This event is reserved.
#define Q_AVCENC_EV_BURNIN_UPDATED              AVCENC_EVENT(9)

/// \brief Motion detection is found. 
/// \param Transition is ROIinMot. 
/// \param Average Motion is ration of the sum of motion MBs in the total MBs. 
/// \param MBinMot is MBs in motion. 
#define Q_AVCENC_EV_VIDEO_MD_ALERT              AVCENC_EVENT(10)

/// \brief A scene change has been detected
/// \param streamID Stream ID of the encoder which detected a scene change
/// \param PTS_MSB Most significant 32 bits of the scene change frame's 64 bit PTS
/// \param PTS_LSB Least significant 32 bits of the scene change frame's 64 bit PTS
#define Q_AVCENC_EV_VIDEO_SCENE_CHANGE          AVCENC_EVENT(11)

#define Q_AVCENC_EV_VA_CONTOUR_INFO             AVCENC_EVENT(12)

/// \brief This event is reserved
#define Q_AVCENC_EV_VA_CT_ALARM                 AVCENC_EVENT(13)


typedef struct
{
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        streamID;
    unsigned int        numFrames;
    unsigned int        frame0Addr;
    unsigned int        frame0Size;
    unsigned int        frame1Addr;
    unsigned int        frame1Size;
    unsigned int        frame2Addr;
    unsigned int        frame2Size;
    unsigned int        frame3Addr;
    unsigned int        frame3Size;
    unsigned int        frame4Addr;
    unsigned int        frame4Size;
} STRUCT_Q_AVCENC_EV_BITSTREAM_BLOCK_READY;

typedef struct
{
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        streamID;
} STRUCT_Q_AVCENC_EV_BITSTREAM_FLUSHED;

typedef struct
{
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        streamID;
    unsigned int        frameWidth;
    unsigned int        frameHeight;
    unsigned int 	frameID;
    unsigned int 	numBoundingBox;
    unsigned int 	sizeofFinalObject;
    unsigned int 	address;
    /* Add the members of Rect Structure */
} STRUCT_Q_AVCENC_EV_VA_OT_RECT_READY;

// NOTE: We are not going to enable camera tamper in AVC
typedef struct
{
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        streamID;
    unsigned int 	frameID;
    unsigned int 	tamperType;
    /* Add the members of Rect Structure */
} STRUCT_Q_AVCENC_EV_VA_CT_ALARM;

typedef struct
{
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        frameWidth;
    unsigned int        frameHeight;

} STRUCT_Q_AVCENC_EV_VA_FRAME_INFO;

typedef struct
{
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        index;
    unsigned int        streamID;
} STRUCT_Q_AVCENC_EV_BURNIN_UPDATED;

typedef struct
{
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        streamID;
    unsigned int        PTS_MSB;
    unsigned int        PTS_LSB;
    unsigned int        coding_QP;
    unsigned int        frameSize;
    unsigned int        distortion;
    unsigned int        latency;
    unsigned int        encodeTime;
    unsigned int        signature;
} STRUCT_Q_AVCENC_EV_VIDEO_FRAME_ENCODED;

typedef struct
{
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        streamID;
    unsigned int        bufferSize;
    unsigned int        bufferEmptiness;
    unsigned int        guardBand;
} STRUCT_Q_AVCENC_EV_VIDEO_ENC_BUFFER_FULL;

typedef struct
{
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        streamID;
    unsigned int        PTS_MSB;
    unsigned int        PTS_LSB;
} STRUCT_Q_AVCENC_EV_VIDEO_ENC_RC_FRAME_DROP;

typedef struct
{
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        streamID;
    unsigned int        index_transition;
    unsigned int        avgMot_MBInMot;
    unsigned int        PTS_MSB;
    unsigned int        PTS_LSB;
} STRUCT_Q_AVCENC_EV_VIDEO_MD_ALERT;

typedef struct
{
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        streamID;
    unsigned int        PTS_MSB;
    unsigned int        PTS_LSB;
} STRUCT_Q_AVCENC_EV_VIDEO_SCENE_CHANGE;


//////////////////////////////////////////////////////////////////////
//
// Single buffered Configuration parameters

/// \brief When set the encoder will acknowledge its own bitstream blocks
/// which removes the need for the host to send BITSTREAM_BLOCK_DONE 
/// commands.  Note that this means the encoder can overwrite un-read
/// data and the host can receive a corrupted bitstream
#define Q_AVCENC_CFG_BBLOCK_SELF_ACK                  1

//////////////////////////////////////////////////////////////////////
//
// Double buffered Configuration parameters
//
// Generated from mediaware.h VENC_CFG_2 s/VENC_CFG/Q_AVCENC_CMP/
/// \brief Allow the bitstream output of the encoder to be disabled.  The encoder
///        will encode video as usual, but the output bitstream will be discarded
///        instead of being sent to the host.  Bitstream block ready events will
///        not be received.  This is useful for debugging, to determine if a problem
///        lies in capture/encode or in transfer of data to the host.
/// \param enable
#define Q_AVCENC_CMP_BITSTREAM_OUTPUT                0
/// \brief Send the encoder's reconstructed frames to the display port.  If an avsn
///        is bound to the encoder, it will display the reconstructed frames instead
///        of a live preview.  The reconstructed frames look exactly like the decoded
///        bitstream would look.  This is useful for debugging, in order to check the
///        results of encoding without having to encode, store, and decode a file.
/// \param enable
#define Q_AVCENC_CMP_ENCODER_PREVIEW                 1
/// \brief This parameter sets the distance between I-frames, in frame units
/// \param value GOP size in frame units
#define Q_AVCENC_CMP_GOP_SIZE                        2
/// \brief This parameter controls the operation of the deblocking filter. 
/// \param value 0:Disable 1:Enable
#define Q_AVCENC_CMP_DEBLOCK_ENABLE                  3
/// \brief This parameter controls the alpha offset used by the deblocking 
///        filter to select the QP-based filter strength for Bs < 4. 
/// \param value Alpha offset in the range of -6 to 6, where -6 is weakest 
///        deblocking filter strength, +6 is strongest deblocking strength
#define Q_AVCENC_CMP_DEBLOCK_OFFSET_ALPHA            4
/// \brief This parameter controls the beta offset used by the deblocking filter 
///        to select the QP-based filter strength for Bs == 4. 
/// \param value Beta offset in the range of -6 to 6, where -6 is weakest 
///        deblocking filter strength, +6 is strongest deblocking strength
#define Q_AVCENC_CMP_DEBLOCK_OFFSET_BETA             5
/// \brief This parameter controls the Chroma QP delta for Cr. 
///        It sets the parameter chroma_qp_index_offset in the picture parameters set.
/// \param value Cr QP offset
#define Q_AVCENC_CMP_CHROMA_QP_OFFSET_CR             6
/// \brief This parameter controls the Chroma QP delta for Cb. 
///        It sets the parameter second_chroma_qp_index_offset in the 
///        picture parameters set. This parameter is only present in 
///        High Profile. In other profiles, chroma_qp_index_offset is 
///        used for both Cb and Cr.
/// \param value Cb QP offset
#define Q_AVCENC_CMP_CHROMA_QP_OFFSET_CB             7
/// \brief Enable the VUI VCL HRD parameters in the bitstream headers.  This is the
///        data used to check conformance with the specification's Hypothetical
///        Reference Decoder (HRD).  VCL HRD parameters will be stored in the VUI inside
///        the Sequence Parameter Set, and SEI Buffering Period messages will be sent in the
///        bitstream.  There is no real use for this except conformance testing.
/// \param enable
#define Q_AVCENC_CMP_VUI_VCL_HRD_ENABLE              8
/// \brief Bitrate field of the VCL HRD.  This value is directly inserted into the VUI.
/// \param bitrate
#define Q_AVCENC_CMP_VUI_VCL_HRD_BITRATE             9
/// \brief CPB size field of the VCL HRD.  This value is directly inserted into the VUI.
/// \param size
#define Q_AVCENC_CMP_VUI_VCL_HRD_CPB_SIZE           10
/// \brief Enable the VUI NAL HRD parameters in the bitstream headers.  This is the
///        data used to check conformance with the specification's Hypothetical
///        Reference Decoder (HRD).  NAL HRD parameters will be stored in the VUI inside
///        the Sequence Parameter Set, and SEI Buffering Period messages will be sent in the
///        bitstream.  There is no real use for this except conformance testing.
/// \param enable
#define Q_AVCENC_CMP_VUI_NAL_HRD_ENABLE             11
/// \brief Bitrate field of the NAL HRD.  This value is directly inserted into the VUI.
/// \param bitrate
#define Q_AVCENC_CMP_VUI_NAL_HRD_BITRATE            12
/// \brief CPB size field of the NAL HRD.  This value is directly inserted into the VUI.
/// \param size
#define Q_AVCENC_CMP_VUI_NAL_HRD_CPB_SIZE           13
/// \brief Enable SEI Picture Timing messages in the bitstream.  These messages carry timing
///        information as described in the AVC specification.  The timing information for a
///        frame is identical to the PTS of the QBox containing it.
///        SEI picture timing MUST NOT be used for recordings of greater than 24 hours.  The
///        binary format only supports 24 hours of time.  Longer recordings must use the QBox
///        PTS exclusively.
/// \param enable
#define Q_AVCENC_CMP_SEI_PICT_TIMING_ENABLE         14
/// \brief Enable SEI Buffering Period messages in the bitstream.  They will be enabled
///        automatically if the VUI VCL HRD or VUI NAL HRD are enabled.
#define Q_AVCENC_CMP_SEI_BUFFERING_PERIOD           15
/// \brief This parameter is reserved.
/// \internal obsolete seve1 feature
#define Q_AVCENC_CMP_SEI_RC_FRAME_STATS             16
/// \brief This parameter is reserved.
/// \internal obsolete seve1 feature
#define Q_AVCENC_CMP_SEI_RC_CALC_STATS              17
/// \brief This parameter is reserved.
/// \internal At the start of encoding, send a custom SEI message containing the encoder configuration
///           array.
#define Q_AVCENC_CMP_SEI_ENC_CFG                    18
/// \brief This parameter forces the coding of I-P field pair for intra field-coded frames. 
///        i.e., the top field is Intra coded and the bottom field is Inter codec (P-picture). 
///        When disabled, both fields in a picture are Intra coded 
/// \param value 0:Disable 1:Enable
#define Q_AVCENC_CMP_IP_FIELD_CODING                19
/// \brief Set the timing_info_present flag in the VUI and include timing information in the bitstream.
///        This must be set to use SEI picture timing messages.  It specifies a default framerate
///        for the stream, and is highly recommended to ensure the stream can be decoded at the
///        correct speed by all decoders.
/// \param enable
#define Q_AVCENC_CMP_VUI_TIMING_ENABLE              20
/// \brief This parameter enables scene change detection and processing
/// \param value 0:Disable 1:Enable
#define Q_AVCENC_CMP_SCENE_CHANGE_ENABLE            21
/// \brief This parameter enables Override pic coding type according to scene detection for P frames
///        The scene change detection can happen in the P or the same subGop B pictures
///        if it's frame coding, then the picture is set to I
///        if it's field coding, then only the top field set to I
/// \param value 0:Disable 1:Enable
#define Q_AVCENC_CMP_SCENE_CHANGE_I_SLICE           22
/// \brief This parameter enables new GOP after current subGop according to scene detection
/// \param value 0:Disable 1:Enable
#define Q_AVCENC_CMP_SCENE_CHANGE_NEW_GOP           23
/// \brief Not implemented
/// \param value 
#define Q_AVCENC_CMP_SCENE_CHANGE_PERIOD            24
/// \brief Set the AVC level for the sequence parameter set.  See Appendix A of the AVC specification.
///        Examples: 30 == level 3.0 (SD), 31 == level 3.1 (SD with MBAFF), 4.0 == 720p or 1080i
/// \param level AVC level without the decimal point.
#define Q_AVCENC_CMP_SPS_LEVEL                      25
/// \brief Fixed frame rate parameter from VUI information, as in the AVC specification.  Can be set if
///        the framerate is constant (no adaptive framerate, no inverse telecine, etc).
/// \param enable
#define Q_AVCENC_CMP_VUI_FIXED_FRAME_RATE           26
/// \brief The number of units in a tick from the VUI information, as in the AVC specification.
///        This should be set to the number of ticks per frame (example: 3003 for NTSC, 3600 for PAL).
/// \param PTS of one field in 180 KHz ticks (or one frame in 90 KHz ticks).
#define Q_AVCENC_CMP_VUI_NUM_UNITS_IN_TICK          27
/// \brief Minimum duration (PTS length) of an encoded frame.  This is used to have the encoder drop
///        frames in order to slow down the framerate.  Set to 0 (the default) to disable.
/// \param duration New frame duration in PTS (should be greater than or equal to the incoming frame
///        duration from video capture.
#define Q_AVCENC_CMP_MIN_FRAME_DURATION             28
/// \brief Not implemented
/// \param value 
#define Q_AVCENC_CMP_TIME_BASED_GOP_ENABLE          29
/// \brief Not implemented
/// \param value 
#define Q_AVCENC_CMP_TIME_BASED_GOP_TICKS           30
/// \brief Controls how much internal memory the encoder allocates for the encoded bitstream of one
///        frame.  If not enough memory is allocated, the output may be corrupted.  Allocation is
///        in number of bytes per macroblock, so it is automatically scaled to different resolutions.
///        The theoretical maximum is under 400 bytes per macroblock.  This parameter can be reduced
///        to save memory on low bitrate streams, although using less than 50 bytes per macroblock
///        is not recommended.
/// \param Bytes number of bytes per macroblock to reserve
#define Q_AVCENC_CMP_AU_GUARD_BAND_PER_MB           31
/// \brief This parameter specifies using encoding trial frames for rate control 
/// \param value 0: VENC_QP_GUESS_NONE
///        value 1: VENC_QP_GUESS_3FRAME
#define Q_AVCENC_CMP_QP_GUESS                       32
/// \brief This parameter is reserved.
/// \internal
/// Set a new 32-bit seed for the random number generator.  The random number generator is currently
/// unused.
/// \param value 32-bit random seed
#define Q_AVCENC_CMP_RAND_SEED                      33
/// \brief This parameter is reserved.
/// \internal deleted
#define Q_AVCENC_CMP_OBSOLETE_1                     34
/// \brief This parameter is reserved.
/// \internal deleted
#define Q_AVCENC_CMP_OBSOLETE_2                     35
/// \brief This parameter is reserved.
/// \internal obsolete due to segment encoding
#define Q_AVCENC_CMP_SLICES_PER_FRAME               36
/// \brief This parameter is reserved.
/// \internal obsolete due to segment encoding
#define Q_AVCENC_CMP_SLICES_PER_FIELD               37
/// \brief This parameter enables low latency mode in which each slice is sent out as an AU
///        after it is got encoded
/// \param value 0:Disable 1:Enable
#define Q_AVCENC_CMP_LOW_LATENCY_MODE               38
/// \brief Pad each access unit in the AVC bitstream to the specified alignment, by padding with
///        zero bytes.  This is alignment within the QBox, not the alignment of the QBox as a
///        whole.  Padding occurs at the end of each frame.
/// \param alignment alignment to pad the end of each frame to
#define Q_AVCENC_CMP_OUTPUT_AU_ALIGNMENT            39
/// \brief This parameter is reserved.
/// \internal obsolete param
#define Q_AVCENC_CMP_REPEATED_FRAME_PSKIP           40
/// \brief This parameter is reserved.
/// \internal obsolete param
#define Q_AVCENC_CMP_FRAMERATE_CHANGE_INTERVAL      41
/// \brief This parameter is reserved
#define Q_AVCENC_CMP_RESERVED0                      42
/// \brief Enable or disable the deblocking filter (defaults to enabled).  If disabled, all other
///        deblocking filter control parameters are ignored.
/// \param enable
#define Q_AVCENC_CMP_DEBLOCK_PRESENT                43
/// \brief Set to encode baseline profile.  Must not be set for main or high profiles.
/// \param enable
#define Q_AVCENC_CMP_PROFILE_BASELINE_COMPATIBLE    44
/// \brief Set to encode main profile.  Must be set for main profile.  Can be set for baseline
///        profile (a baseline profile image can also be compatible with main profile).  Setting
///        both baseline and main profile compatibility to disabled will result in a high profile
///        image.
/// \param enable
#define Q_AVCENC_CMP_PROFILE_MAIN_COMPATIBLE        45
/// \brief Not implemented
/// \param value
#define Q_AVCENC_CMP_PIC_ORDER_CNT_TYPE             46
/// \brief This parameter sets the GOP structure of the encoder.d 
/// \param value GOP structure 
/// \param 0: I-only
/// \param 1: IP
/// \param 2: IBP
/// \param 3: IBBP
/// \param 4: IBBrBP (Pyramidal GOP with one B-reference)
#define Q_AVCENC_CMP_GOP_STRUCTURE                  47
/// \brief This parameter enables Macroblock Adaptive Field/Frame (MBAFF) Selection
/// \param value 0:Disable 1:Enable
#define Q_AVCENC_CMP_ENABLE_MBAFF                   48
/// \brief This parameter is reserved.
/// \internal
/// This parameter determines the amount of motion in a macroblock to classify as field
/// coded in MBAFF when VPP cannot reliably make a decision. 
/// \param value Motion threshold in the range of 0-255, lower value will select field coding more often.
#define Q_AVCENC_CMP_MBAFF_FIELD_MOTION_THRESHOLD   49
/// \brief This parameter is reserved.
/// \internal
/// This parameter specifies the content of the 16x1 LUT which generates the field decision bit
/// from the region 7 similarity value in MBAFF. 
/// \param value 16-bit 16x1 LUT value. bit 0 = location 0 etc.
#define Q_AVCENC_CMP_MBAFF_REGION_LUT               50
/// \brief This parameter is reserved.
/// \internal
/// This parameter specifies the content of the 16x2 MBAFF decision LUT. 
/// Each 2-bit entry represents the decision, 00: MBAFFDefaultMode, 01: Field, 10: Frame, 11: Undefined
/// \param value 32-bit 16x2 MBAFF decision LUT value. 
///        bit  0-15 correspond to 2-bit values for Dec7-Dec0, respectively
///        bit 16-31 correspond to 2-bit values for Decf-Dec8, respectively 
#define Q_AVCENC_CMP_MBAFF_DECISION_LUT             51
/// \brief This parameter enables contrained intra prediction
/// \param value 0:Disable 1:Enable
#define Q_AVCENC_CMP_CONSTRAINED_INTRA_PRED         52
/// \brief This parameter enables the use of 8x8 transform in the picture parameter set.
///        The selection process for 4x4 versus 8x8 is controlled by VENC_REG_CFG_INTRAMODES_INTERTRANS
/// \param value 0:Disable 1:Enable
#define Q_AVCENC_CMP_ENABLE_8X8_TRANSFORM           53
/// \brief This parameter enables CABAC entropy coding. When disabled, CAVLC entropy coding is used.
/// \param value 0:CAVLC 1:CABAC
#define Q_AVCENC_CMP_CABAC                          54
/// \brief This parameter enables the use of the Pre-Motion Estimation (PME)
///        PME performs large search range in decimated picture.
/// \param value 0:Disable 1:Enable
#define Q_AVCENC_CMP_PME_ENABLE                     55
/// \brief This parameter is reserved.
/// \internal
/// This parameter enables the use of Region statistics in the VPP.
/// Region statistics are essential to the operation of the AVC encoder, 
/// they should only be disabled for debugging purposes.
/// \param value 0:Disable 1:Enable 
#define Q_AVCENC_CMP_VPP_REGION_STAT_ENABLE         56
/// \brief This parameter enables the use of Temporal Direct mode in B-slices.
///        When disabled, Spatial Direct mode is used.
/// \param value 0:Disable (Use Spatial Direct mode) 1:Enable (Use Temporal Direct mode) 
#define Q_AVCENC_CMP_TEMPORAL_DIRECT_MODE           57
/// \brief This parameter is reserved.
/// \internal
/// The MEMBPART set of parameters control Inter modes enabling using the results of the 
/// Sobel edge detector of VPP. See Mobilygen internal documentation on EPP for more details.
/// \param value Edge Threshold 0, range [0-255]
#define Q_AVCENC_CMP_MEMBPART_EDGETHOLD0            58
/// \brief This parameter is reserved.
/// \internal
/// The MEMBPART set of parameters control Inter modes enabling using the results of the 
/// Sobel edge detector of VPP. See Mobilygen internal documentation on EPP for more details.
/// \param value Edge Threshold 1, range [0-255]
#define Q_AVCENC_CMP_MEMBPART_EDGETHOLD1            59
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strength of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to P-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_0 for P-slice
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_0_PSLICE     60
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to P-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_1 for P-slice
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_1_PSLICE     61
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to P-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_2 for P-slice
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_2_PSLICE     62
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to P-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_3 for P-slice
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_3_PSLICE     63
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to P-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_4 for P-slice
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_4_PSLICE     64
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to P-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_5 for P-slice
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_5_PSLICE     65
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to P-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_6 for P-slice
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_6_PSLICE     66
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to P-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_7 for P-slice
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_7_PSLICE     67
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to P-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_8 for P-slice
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_8_PSLICE     68
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to P-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_9 for P-slice
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_9_PSLICE     69
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to P-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_10 for P-slice
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_10_PSLICE    70
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to P-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_11 for P-slice
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_11_PSLICE    71
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to B-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_0 for B-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_0_BSLICE     72
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to B-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_1 for B-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_1_BSLICE     73
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to B-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_2 for B-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_2_BSLICE     74
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to B-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_3 for B-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_3_BSLICE     75
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to B-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_4 for B-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_4_BSLICE     76
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to B-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_5 for B-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_5_BSLICE     77
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to B-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_6 for B-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_6_BSLICE     78
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to B-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_7 for B-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_7_BSLICE     79
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to B-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_8 for B-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_8_BSLICE     80
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to B-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_9 for B-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_9_BSLICE     81
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to B-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_10 for B-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_10_BSLICE    82
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to B-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_11 for B-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_11_BSLICE    83
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to Br-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_0 for Br-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_0_BRSLICE    84
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to Br-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_1 for Br-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_1_BRSLICE    85
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to Br-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_2 for Br-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_2_BRSLICE    86
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to Br-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_3 for Br-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_3_BRSLICE    87
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to Br-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_4 for Br-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_4_BRSLICE    88
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to Br-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_5 for Br-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_5_BRSLICE    89
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to Br-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_6 for Br-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_6_BRSLICE    90
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to Br-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_7 for Br-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_7_BRSLICE    91
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to Br-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_8 for Br-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_8_BRSLICE    92
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to Br-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_9 for Br-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_9_BRSLICE    93
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to Br-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_10 for Br-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_10_BRSLICE   94
/// \brief This parameter is reserved.
/// \internal
///        The MEMBPART set of parameters control Inter modes enabling using the results of the 
///        Sobel edge detector of VPP. Each of the 12 MBPARTEN_[0-11] parameters controls the 
///        inter mode enable based on edge direction and edge strenght of the macroblock. 
///        Each MBPARTEN is a 12-bit mask for enabling inter partitons, where bit 11 to 0 correspond
///        respectively to enabling modes 16x16, 16x8, 8x16, 8x8 (all), top-left 8x8, top-right 8x8, 
///        bottom-left 8x8, bottom-right 8x8, 8x4, 4x8, 4x4.
///        The _PSLICE set of parameters apply to Br-slices.
///        To bypass edge direction usage to enable partitions, set MBPARTEN_[0-11] to the same values. 
///        See Mobilygen internal documentation on EPP for more details.
/// \param value 12-bit MEMBPART_11 for Br-slice
///
#define Q_AVCENC_CMP_MEMBPART_MBPARTEN_11_BRSLICE   95
/// \brief This parameter is reserved.
/// \internal
///        This parameter controls the motion estimation rough (decimate) search mode
///        (NOTE: in MG3500, only 0:Full is supported)
/// \param value 0:Full, 1:Sub-sampled, 2:Horizontal Average, 3: Vertical Average
///
#define Q_AVCENC_CMP_MEGRPCTRL_SEARCHMODE           96
/// \brief This parameter is not supported in MG3500
///
#define Q_AVCENC_CMP_MEGRPCTRL_MVCORRTHOLD_PSLICE   97
/// \brief This parameter is not supported in MG3500
///
#define Q_AVCENC_CMP_MEGRPCTRL_MVCORRTHOLD_BSLICE   98
/// \brief This parameter selects the Pre-Motion estimation (PME) start vector 
///        for the 16x16 partition in the AVC motion estimation 
/// \param value PME MV Select 0:8x8, 1:8x4, 2:4x8, 3:4x4
///
#define Q_AVCENC_CMP_MEGRPCTRL_PMESEL16x16          99
/// \brief This parameter selects the Pre-Motion estimation (PME) start vector 
///        for the 16x8 partition in the AVC motion estimation 
/// \param value PME MV Select 0:8x8, 1:8x4, 2:4x8, 3:4x4
///
#define Q_AVCENC_CMP_MEGRPCTRL_PMESEL16x8          100
/// \brief This parameter selects the Pre-Motion estimation (PME) start vector 
///        for the 8x16 partition in the AVC motion estimation 
/// \param value PME MV Select 0:8x8, 1:8x4, 2:4x8, 3:4x4
///
#define Q_AVCENC_CMP_MEGRPCTRL_PMESEL8x16          101
/// \brief This parameter selects the Pre-Motion estimation (PME) start vector 
///        for the 8x8 partition in the AVC motion estimation 
/// \param value PME MV Select 0:8x8, 1:8x4, 2:4x8, 3:4x4
///
#define Q_AVCENC_CMP_MEGRPCTRL_PMESEL8x8           102
/// \brief P-slice Lambda constant used by rate distortion optimization
///        for Partition size decision in motion estimation:
///        Cost = Distortion + Lambda*Rate, 
///        Where Lambda = Lambda_constant*2^((QP-12)/6)
/// \param value Lambda_contant*100
///
#define Q_AVCENC_CMP_MEGRPCTRL_MDLAMBDA_PSLICE     103
/// \brief B-slice Lambda constant used by rate distortion optimization
///        for Partition size decision in motion estimation:
///        Cost = Distortion + Lambda*Rate, 
///        Where Lambda = Lambda_constant*2^((QP-12)/6)
/// \param value Lambda_contant*100
///
#define Q_AVCENC_CMP_MEGRPCTRL_MDLAMBDA_BSLICE     104
/// \brief Br-slice Lambda constant used by rate distortion optimization
///        for Partition size decision in motion estimation:
///        Cost = Distortion + Lambda*Rate, 
///        Where Lambda = Lambda_constant*2^((QP-12)/6)
/// \param value Lambda_contant*100
///
#define Q_AVCENC_CMP_MEGRPCTRL_MDLAMBDA_BRSLICE    105
/// \brief P-slice Lambda constant used by rate distortion optimization
///        for Motion Estimation during the search:
///        Cost = Distortion + Lambda*Rate, 
///        Where Lambda = Lambda_constant*2^((QP-12)/6)
/// \param value Lambda_contant*100
///
#define Q_AVCENC_CMP_MEGRPCTRL_MELAMBDA_PSLICE     106
/// \brief B-slice Lambda constant used by rate distortion optimization
///        for Motion Estimation during the search:
///        Cost = Distortion + Lambda*Rate, 
///        Where Lambda = Lambda_constant*2^((QP-12)/6)
/// \param value Lambda_contant*100
///
#define Q_AVCENC_CMP_MEGRPCTRL_MELAMBDA_BSLICE     107
/// \brief Br-slice Lambda constant used by rate distortion optimization
///        for Motion Estimation during the search:
///        Cost = Distortion + Lambda*Rate, 
///        Where Lambda = Lambda_constant*2^((QP-12)/6)
/// \param value Lambda_contant*100
///
#define Q_AVCENC_CMP_MEGRPCTRL_MELAMBDA_BRSLICE    108
/// \brief Threshold for culling references within a List based on Cost.
///        Value is in the range of [0-63], where 0 will only select the 
///        best reference, 63 will do no culling.
///        
/// \param value Reference Threshold
///
#define Q_AVCENC_CMP_MEGRPCTRL_REFTHOLD            109
/// \brief This parameter is reserved.
/// \internal
/// \brief Threshold for culling a List based on Cost for B-slices.
///        Value is in the range of [0-63], where 0 will only select the 
///        best list, 63 will do no culling.
///        
/// \param value List Threshold
///
#define Q_AVCENC_CMP_MEGRPCTRL_LISTTHOLD           110
/// \brief This parameter is reserved.
/// \internal
/// This parameter enables QP modulation based on macroblock DC value
/// \param value 0:Disable 1:Enable 
///
#define Q_AVCENC_CMP_QPMOD_DCEN                    111
/// \brief This parameter is reserved.
/// \internal
/// This parameter over-rides the DC average of the picture used in the QP Modulation.
///        A value of zero means that firmware sets this value to the average of the picture using VPP statistics.
/// \param value DC average
///
#define Q_AVCENC_CMP_QPMOD_DCAVG                   112
/// \brief This parameter is reserved.
/// \internal
/// This parameter clips the minimum MB DC average from the VPP picture statistics
/// \param value Minimum DC average
///
#define Q_AVCENC_CMP_QPMOD_DCAVGMIN                113
/// \brief This parameter is reserved.
/// \internal
/// This parameter clips the maximum MB DC average from the VPP picture statistics
/// \param value Maximum DC average
///
#define Q_AVCENC_CMP_QPMOD_DCAVGMAX                114
/// \brief This parameter is reserved.
/// \internal
/// This parameter sets the a_b ratio for DC
/// \param value a_b DC ratio
///
#define Q_AVCENC_CMP_QPMOD_DCABRATIO               115
/// \brief This parameter is reserved.
/// \internal
/// This parameter sets the b_a ratio for DC
/// \param value b_a DC ratio
///
#define Q_AVCENC_CMP_QPMOD_DCBARATIO               116
/// \brief This parameter is reserved.
/// \internal
/// This parameter enables QP modulation based on macroblock High Frequency value
/// \param value 0:Disable 1:Enable 
///
#define Q_AVCENC_CMP_QPMOD_HFEN                    117
/// \brief This parameter is reserved.
/// \internal
/// This parameter over-rides the HF average of the picture used in the QP Modulation.
///        A value of zero means that firmware sets this value to the average of the picture using VPP statistics.
/// \param value HF average
///
#define Q_AVCENC_CMP_QPMOD_HFAVG                   118
/// \brief This parameter is reserved.
/// \internal
/// This parameter clips the minimum MB HF average from the VPP picture statistics
/// \param value Minimum HF average
///
#define Q_AVCENC_CMP_QPMOD_HFAVGMIN                119
/// \brief This parameter is reserved.
/// \internal
/// This parameter clips the maximum MB HF average from the VPP picture statistics
/// \param value Maximum HF average
///
#define Q_AVCENC_CMP_QPMOD_HFAVGMAX                120
/// \brief This parameter is reserved.
/// \internal
/// This parameter sets the a_b ratio for HF
/// \param value a_b HF ratio
///
#define Q_AVCENC_CMP_QPMOD_HFABRATIO               121
/// \brief This parameter is reserved.
/// \internal
/// This parameter sets the b_a ratio for HF
/// \param value b_a HF ratio
///
#define Q_AVCENC_CMP_QPMOD_HFBARATIO               122
/// \brief This parameter is reserved.
/// \internal
/// This parameter enables QP modulation based on macroblock Motion value
///        
/// \param value 0:Disable 1:Enable 
///
#define Q_AVCENC_CMP_QPMOD_MOTEN                   123
/// \brief This parameter is reserved.
/// \internal
/// This parameter over-rides the Motion average of the picture used in the QP Modulation.
///        A value of zero means that firmware sets this value to the average of the picture using VPP statistics.
/// \param value Motion average
///
#define Q_AVCENC_CMP_QPMOD_MOTAVG                  124
/// \brief This parameter is reserved.
/// \internal
/// This parameter clips the minimum MB Motion average from the VPP picture statistics
/// \param value Minimum Motion average
///
#define Q_AVCENC_CMP_QPMOD_MOTAVGMIN               125
/// \brief This parameter is reserved.
/// \internal
/// This parameter clips the maximum MB Motion average from the VPP picture statistics
/// \param value Maximum Motion average
///
#define Q_AVCENC_CMP_QPMOD_MOTAVGMAX               126
/// \brief This parameter is reserved.
/// \internal
/// This parameter sets the a_b ratio for Motion
/// \param value a_b Motion ratio
///
#define Q_AVCENC_CMP_QPMOD_MOTABRATIO              127
/// \brief This parameter is reserved.
/// \internal
/// This parameter sets the b_a ratio for Motion
/// \param value b_a Motion ratio
///
#define Q_AVCENC_CMP_QPMOD_MOTBARATIO              128
/// \brief This parameter is reserved.
/// \internal
/// This parameter enables QP modulation based on macroblock Motion Edge value
///        
/// \param value 0:Disable 1:Enable 
///
#define Q_AVCENC_CMP_QPMOD_EDGEEN                  129
/// \brief This parameter is reserved.
/// \internal
/// This parameter over-rides the Motion Edge average of the picture used in the QP Modulation.
///        A value of zero means that firmware sets this value to the average of the picture using VPP statistics.
/// \param value Motion Edge average
///
#define Q_AVCENC_CMP_QPMOD_EDGEAVG                 130
/// \brief This parameter is reserved.
/// \internal
/// This parameter clips the minimum MB Motion Edge average from the VPP picture statistics
/// \param value Minimum Motion Edge average
///
#define Q_AVCENC_CMP_QPMOD_EDGEAVGMIN              131
/// \brief This parameter is reserved.
/// \internal
/// This parameter clips the maximum MB Motion Edge average from the VPP picture statistics
/// \param value Maximum Motion Edge average
///
#define Q_AVCENC_CMP_QPMOD_EDGEAVGMAX              132
/// \brief This parameter is reserved.
/// \internal
/// This parameter sets the a_b ratio for Motion Edge
/// \param value a_b Motion Edge ratio
///
#define Q_AVCENC_CMP_QPMOD_EDGEABRATIO             133
/// \brief This parameter is reserved.
/// \internal
/// This parameter sets the b_a ratio for Motion Edge
/// \param value b_a Motion Edge ratio
///
#define Q_AVCENC_CMP_QPMOD_EDGEBARATIO             134
/// \brief This parameter is reserved.
/// \internal
/// This parameter is to be documented.
#define Q_AVCENC_CMP_QPMOD_VPP_HF_MBGAIN           135
/// \brief Selects how the predicted motion vector is computed
///        during Pre-Motion Estimation (PME). This affects the search results by altering
///        the motion vector delta.
///
/// \param value 0:Use the zero vector 1:Use the left neighbour's vector
///
#define Q_AVCENC_CMP_PME_PRED_MV_MODE              136
/// \brief The horizontal search range used during Pre-Motion Estimation.
///
/// \param value number of full-pel pixels to search horizontally
///
#define Q_AVCENC_CMP_PME_SRCHRNG_X                 137
/// \brief The vertical search range used during Pre-Motion Estimation.
///
/// \param value number of full-pel pixels to search vertically
///
#define Q_AVCENC_CMP_PME_SRCHRNG_Y                 138
/// \brief P-slice Lambda constant used by PME.
///        Cost = Distortion + Lambda*MVDelta, 
///        Where Lambda = Lambda_constant*2^((QP-12)/6)
/// \param value Lambda_contant*100
///
#define Q_AVCENC_CMP_PME_LAMBDA_PSLICE             139
/// \brief B-slice Lambda constant used by PME.
///        Cost = Distortion + Lambda*MVDelta, 
///        Where Lambda = Lambda_constant*2^((QP-12)/6)
/// \param value Lambda_contant*100
///
#define Q_AVCENC_CMP_PME_LAMBDA_BSLICE             140
/// \brief Br-slice Lambda constant used by PME.
///        Cost = Distortion + Lambda*MVDelta, 
///        Where Lambda = Lambda_constant*2^((QP-12)/6)
/// \param value Lambda_contant*100
///
#define Q_AVCENC_CMP_PME_LAMBDA_BRSLICE            141
/// \brief Select the range of this ME search algorithm.  Rough search + refine (0) or
///        refine only (1).
///        This is part of ME search algorithm 0 control (there are 8 ME algorithms to
///        configure, which can then be assigned to regions), to set up its use on
///        P frames.
/// \param algtype enable/disable rough search
#define Q_AVCENC_CMP_MEALGDEF_ALGTYPE_ALG0         142
/// \brief Specify the vertical refinement range for algorithm 0 on P frames.
///        This is algorithm 0 of 8.  The algorithms are used by selecting one to be
///        used for each region.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG0_PSLICE 143
/// \brief Specify the horizontal refinement range for algorithm 0 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG0_PSLICE 144
/// \brief Specify the vertical refinement range for algorithm 0 on B frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG0_BSLICE 145
/// \brief Specify the horizontal refinement range for algorithm 0 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG0_BSLICE 146
/// \brief Specify the vertical search range for algorithm 0 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG0_PSLICE   147
/// \brief Specify the horizontal search range for algorithm 0 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG0_PSLICE   148
/// \brief Specify the vertical search range for algorithm 0 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG0_BSLICE   149
/// \brief Specify the horizontal search range for algorithm 0 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG0_BSLICE   150
/// \brief Maximum number of references to full-pel refine for algorithm 0 on P frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG0_PSLICE   151
/// \brief Maximum number of L0 references to full-pel refine for algorithm 0 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG0_BSLICE   152
/// \brief Maximum number of L1 references to full-pel refine for algorithm 0 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL1_ALG0_BSLICE   153
/// \brief Select the range of this ME search algorithm.  Rough search + refine (0) or
///        refine only (1).
///        This is part of ME search algorithm 1 control (there are 8 ME algorithms to
///        configure, which can then be assigned to regions), to set up its use on
///        P frames.
/// \param algtype enable/disable rough search
#define Q_AVCENC_CMP_MEALGDEF_ALGTYPE_ALG1         154
/// \brief Specify the vertical refinement range for algorithm 1 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG1_PSLICE 155
/// \brief Specify the horizontal refinement range for algorithm 1 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG1_PSLICE 156
/// \brief Specify the vertical refinement range for algorithm 1 on B frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG1_BSLICE 157
/// \brief Specify the horizontal refinement range for algorithm 1 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG1_BSLICE 158
/// \brief Specify the vertical search range for algorithm 1 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG1_PSLICE   159
/// \brief Specify the horizontal search range for algorithm 1 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG1_PSLICE   160
/// \brief Specify the vertical search range for algorithm 1 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG1_BSLICE   161
/// \brief Specify the horizontal search range for algorithm 1 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG1_BSLICE   162
/// \brief Maximum number of references to full-pel refine for algorithm 1 on P frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG1_PSLICE   163
/// \brief Maximum number of L0 references to full-pel refine for algorithm 1 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG1_BSLICE   164
/// \brief Maximum number of L1 references to full-pel refine for algorithm 1 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL1_ALG1_BSLICE   165
/// \brief Select the range of this ME search algorithm.  Rough search + refine (0) or
///        refine only (1).
///        This is part of ME search algorithm 2 control (there are 8 ME algorithms to
///        configure, which can then be assigned to regions), to set up its use on
///        P frames.
/// \param algtype enable/disable rough search
#define Q_AVCENC_CMP_MEALGDEF_ALGTYPE_ALG2         166
/// \brief Specify the vertical refinement range for algorithm 2 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG2_PSLICE 167
/// \brief Specify the horizontal refinement range for algorithm 2 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG2_PSLICE 168
/// \brief Specify the vertical refinement range for algorithm 2 on B frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG2_BSLICE 169
/// \brief Specify the horizontal refinement range for algorithm 2 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG2_BSLICE 170
/// \brief Specify the vertical search range for algorithm 2 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG2_PSLICE   171
/// \brief Specify the horizontal search range for algorithm 2 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG2_PSLICE   172
/// \brief Specify the vertical search range for algorithm 2 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG2_BSLICE   173
/// \brief Specify the horizontal search range for algorithm 2 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG2_BSLICE   174
/// \brief Maximum number of references to full-pel refine for algorithm 2 on P frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG2_PSLICE   175
/// \brief Maximum number of L0 references to full-pel refine for algorithm 2 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG2_BSLICE   176
/// \brief Maximum number of L1 references to full-pel refine for algorithm 2 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL1_ALG2_BSLICE   177
/// \brief Select the range of this ME search algorithm.  Rough search + refine (0) or
///        refine only (1).
///        This is part of ME search algorithm 3 control (there are 8 ME algorithms to
///        configure, which can then be assigned to regions), to set up its use on
///        P frames.
/// \param algtype enable/disable rough search
#define Q_AVCENC_CMP_MEALGDEF_ALGTYPE_ALG3         178
/// \brief Specify the vertical refinement range for algorithm 3 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG3_PSLICE 179
/// \brief Specify the horizontal refinement range for algorithm 3 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG3_PSLICE 180
/// \brief Specify the vertical refinement range for algorithm 3 on B frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG3_BSLICE 181
/// \brief Specify the horizontal refinement range for algorithm 3 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG3_BSLICE 182
/// \brief Specify the vertical search range for algorithm 3 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG3_PSLICE   183
/// \brief Specify the horizontal search range for algorithm 3 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG3_PSLICE   184
/// \brief Specify the vertical search range for algorithm 3 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG3_BSLICE   185
/// \brief Specify the horizontal search range for algorithm 3 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG3_BSLICE   186
/// \brief Maximum number of references to full-pel refine for algorithm 3 on P frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG3_PSLICE   187
/// \brief Maximum number of L0 references to full-pel refine for algorithm 3 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG3_BSLICE   188
/// \brief Maximum number of L1 references to full-pel refine for algorithm 3 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL1_ALG3_BSLICE   189
/// \brief Select the range of this ME search algorithm.  Rough search + refine (0) or
///        refine only (1).
///        This is part of ME search algorithm 4 control (there are 8 ME algorithms to
///        configure, which can then be assigned to regions), to set up its use on
///        P frames.
/// \param algtype enable/disable rough search
#define Q_AVCENC_CMP_MEALGDEF_ALGTYPE_ALG4         190
/// \brief Specify the vertical refinement range for algorithm 4 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG4_PSLICE 191
/// \brief Specify the horizontal refinement range for algorithm 4 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG4_PSLICE 192
/// \brief Specify the vertical refinement range for algorithm 4 on B frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG4_BSLICE 193
/// \brief Specify the horizontal refinement range for algorithm 4 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG4_BSLICE 194
/// \brief Specify the vertical search range for algorithm 4 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG4_PSLICE   195
/// \brief Specify the horizontal search range for algorithm 4 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG4_PSLICE   196
/// \brief Specify the vertical search range for algorithm 4 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG4_BSLICE   197
/// \brief Specify the horizontal search range for algorithm 4 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG4_BSLICE   198
/// \brief Maximum number of references to full-pel refine for algorithm 4 on P frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG4_PSLICE   199
/// \brief Maximum number of L0 references to full-pel refine for algorithm 4 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG4_BSLICE   200
/// \brief Maximum number of L1 references to full-pel refine for algorithm 4 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL1_ALG4_BSLICE   201
/// \brief Select the range of this ME search algorithm.  Rough search + refine (0) or
///        refine only (1).
///        This is part of ME search algorithm 5 control (there are 8 ME algorithms to
///        configure, which can then be assigned to regions), to set up its use on
///        P frames.
/// \param algtype enable/disable rough search
#define Q_AVCENC_CMP_MEALGDEF_ALGTYPE_ALG5         202
/// \brief Specify the vertical refinement range for algorithm 5 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG5_PSLICE 203
/// \brief Specify the horizontal refinement range for algorithm 5 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG5_PSLICE 204
/// \brief Specify the vertical refinement range for algorithm 5 on B frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG5_BSLICE 205
/// \brief Specify the horizontal refinement range for algorithm 5 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG5_BSLICE 206
/// \brief Specify the vertical search range for algorithm 5 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG5_PSLICE   207
/// \brief Specify the horizontal search range for algorithm 5 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG5_PSLICE   208
/// \brief Specify the vertical search range for algorithm 5 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG5_BSLICE   209
/// \brief Specify the horizontal search range for algorithm 5 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG5_BSLICE   210
/// \brief Maximum number of references to full-pel refine for algorithm 5 on P frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG5_PSLICE   211
/// \brief Maximum number of L0 references to full-pel refine for algorithm 5 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG5_BSLICE   212
/// \brief Maximum number of L1 references to full-pel refine for algorithm 5 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL1_ALG5_BSLICE   213
/// \brief Select the range of this ME search algorithm.  Rough search + refine (0) or
///        refine only (1).
///        This is part of ME search algorithm 6 control (there are 8 ME algorithms to
///        configure, which can then be assigned to regions), to set up its use on
///        P frames.
/// \param algtype enable/disable rough search
#define Q_AVCENC_CMP_MEALGDEF_ALGTYPE_ALG6         214
/// \brief Specify the vertical refinement range for algorithm 6 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG6_PSLICE 215
/// \brief Specify the horizontal refinement range for algorithm 6 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG6_PSLICE 216
/// \brief Specify the vertical refinement range for algorithm 6 on B frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG6_BSLICE 217
/// \brief Specify the horizontal refinement range for algorithm 6 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG6_BSLICE 218
/// \brief Specify the vertical search range for algorithm 6 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG6_PSLICE   219
/// \brief Specify the horizontal search range for algorithm 6 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG6_PSLICE   220
/// \brief Specify the vertical search range for algorithm 6 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG6_BSLICE   221
/// \brief Specify the horizontal search range for algorithm 6 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG6_BSLICE   222
/// \brief Maximum number of references to full-pel refine for algorithm 6 on P frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG6_PSLICE   223
/// \brief Maximum number of L0 references to full-pel refine for algorithm 6 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG6_BSLICE   224
/// \brief Maximum number of L1 references to full-pel refine for algorithm 6 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL1_ALG6_BSLICE   225
/// \brief Select the range of this ME search algorithm.  Rough search + refine (0) or
///        refine only (1).
///        This is part of ME search algorithm 7 control (there are 8 ME algorithms to
///        configure, which can then be assigned to regions), to set up its use on
///        P frames.
/// \param algtype enable/disable rough search
#define Q_AVCENC_CMP_MEALGDEF_ALGTYPE_ALG7         226
/// \brief Specify the vertical refinement range for algorithm 7 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG7_PSLICE 227
/// \brief Specify the horizontal refinement range for algorithm 7 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG7_PSLICE 228
/// \brief Specify the vertical refinement range for algorithm 7 on B frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEV_ALG7_BSLICE 229
/// \brief Specify the horizontal refinement range for algorithm 7 on P frames.
/// \param refrange 0=-4/+3, 1=-8/+7, 2=-12/+11, 3=-16/+15
#define Q_AVCENC_CMP_MEALGDEF_REFRANGEH_ALG7_BSLICE 230
/// \brief Specify the vertical search range for algorithm 7 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG7_PSLICE   231
/// \brief Specify the horizontal search range for algorithm 7 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG7_PSLICE   232
/// \brief Specify the vertical search range for algorithm 7 on P frames.
/// \param range vertical range divided by 2 (-2* VRange to + 2*VRange-1)
#define Q_AVCENC_CMP_MEALGDEF_VRANGE_ALG7_BSLICE   233
/// \brief Specify the horizontal search range for algorithm 7 on P frames.
/// \param range horizontal range divided by 4 (-4* HRange to + 4*HRange-1)
#define Q_AVCENC_CMP_MEALGDEF_HRANGE_ALG7_BSLICE   234
/// \brief Maximum number of references to full-pel refine for algorithm 7 on P frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG7_PSLICE   235
/// \brief Maximum number of L0 references to full-pel refine for algorithm 7 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL0_ALG7_BSLICE   236
/// \brief Maximum number of L1 references to full-pel refine for algorithm 7 on B frames,
///        minus 1.  Must be 0.
/// \param nrefL0 Set to 0.
#define Q_AVCENC_CMP_MEALGDEF_NREFL1_ALG7_BSLICE   237
/// \brief Control ME sub-pel refinement for 16x16 partitions:
///        Enable half-pel refinement (one of half-pel or quarter-pel must be enabled, but not both).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_HPEN_16x16           238
/// \brief Control ME sub-pel refinement for 16x16 partitions:
///        Enable quarter-pel refinement (one of half-pel or quarter-pel must be enabled, but not both).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_QPEN_16x16           239
/// \brief Control ME sub-pel refinement for 16x16 partitions:
///        Disable bi-directional prediction try (applies to B frames only).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_DISBIMODE_16x16      240
/// \brief Control ME sub-pel refinement for 16x16 partitions:
///        Set number of motion compensations on L0 for P slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML0_16x16_PSLICE 241
/// \brief Control ME sub-pel refinement for 16x16 partitions:
///        Set number of motion compensations on L0 for B slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML0_16x16_BSLICE 242
/// \brief Control ME sub-pel refinement for 16x16 partitions:
///        Set number of motion compensations on L1 for P slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML1_16x16_BSLICE 243
/// \brief Control ME sub-pel refinement for 16x8 partitions:
///        Enable half-pel refinement (one of half-pel or quarter-pel must be enabled, but not both).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_HPEN_16x8            244
/// \brief Control ME sub-pel refinement for 16x8 partitions:
///        Enable quarter-pel refinement (one of half-pel or quarter-pel must be enabled, but not both).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_QPEN_16x8            245
/// \brief Control ME sub-pel refinement for 16x16 partitions:
///        Disable bi-directional prediction try (applies to B frames only).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_DISBIMODE_16x8       246
/// \brief Control ME sub-pel refinement for 16x8 partitions:
///        Set number of motion compensations on L0 for P slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML0_16x8_PSLICE  247
/// \brief Control ME sub-pel refinement for 16x8 partitions:
///        Set number of motion compensations on L0 for B slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML0_16x8_BSLICE  248
/// \brief Control ME sub-pel refinement for 16x8 partitions:
///        Set number of motion compensations on L1 for P slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML1_16x8_BSLICE  249
/// \brief Control ME sub-pel refinement for 8x16 partitions:
///        Enable half-pel refinement (one of half-pel or quarter-pel must be enabled, but not both).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_HPEN_8x16            250
/// \brief Control ME sub-pel refinement for 8x16 partitions:
///        Enable quarter-pel refinement (one of half-pel or quarter-pel must be enabled, but not both).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_QPEN_8x16            251
/// \brief Control ME sub-pel refinement for 8x16 partitions:
///        Disable bi-directional prediction try (applies to B frames only).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_DISBIMODE_8x16       252
/// \brief Control ME sub-pel refinement for 8x16 partitions:
///        Set number of motion compensations on L0 for P slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML0_8x16_PSLICE  253
/// \brief Control ME sub-pel refinement for 8x16 partitions:
///        Set number of motion compensations on L0 for B slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML0_8x16_BSLICE  254
/// \brief Control ME sub-pel refinement for 8x16 partitions:
///        Set number of motion compensations on L1 for P slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML1_8x16_BSLICE  255
/// \brief Control ME sub-pel refinement for 8x8 partitions:
///        Enable half-pel refinement (one of half-pel or quarter-pel must be enabled, but not both).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_HPEN_8x8             256
/// \brief Control ME sub-pel refinement for 8x8 partitions:
///        Enable quarter-pel refinement (one of half-pel or quarter-pel must be enabled, but not both).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_QPEN_8x8             257
/// \brief Control ME sub-pel refinement for 8x8 partitions:
///        Disable bi-directional prediction try (applies to B frames only).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_DISBIMODE_8x8        258
/// \brief Control ME sub-pel refinement for 8x8 partitions:
///        Set number of motion compensations on L0 for P slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML0_8x8_PSLICE   259
/// \brief Control ME sub-pel refinement for 8x8 partitions:
///        Set number of motion compensations on L0 for B slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML0_8x8_BSLICE   260
/// \brief Control ME sub-pel refinement for 8x8 partitions:
///        Set number of motion compensations on L1 for P slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML1_8x8_BSLICE   261
/// \brief Control ME sub-pel refinement for 8x4 partitions:
///        Enable half-pel refinement (one of half-pel or quarter-pel must be enabled, but not both).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_HPEN_8x4             262
/// \brief Control ME sub-pel refinement for 8x4 partitions:
///        Enable quarter-pel refinement (one of half-pel or quarter-pel must be enabled, but not both).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_QPEN_8x4             263
/// \brief Control ME sub-pel refinement for 8x4 partitions:
///        Disable bi-directional prediction try (applies to B frames only).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_DISBIMODE_8x4        264
/// \brief Control ME sub-pel refinement for 8x4 partitions:
///        Set number of motion compensations on L0 for P slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML0_8x4_PSLICE   265
/// \brief Control ME sub-pel refinement for 8x4 partitions:
///        Set number of motion compensations on L0 for B slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML0_8x4_BSLICE   266
/// \brief Control ME sub-pel refinement for 8x4 partitions:
///        Set number of motion compensations on L1 for P slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML1_8x4_BSLICE   267
/// \brief Control ME sub-pel refinement for 4x8 partitions:
///        Enable half-pel refinement (one of half-pel or quarter-pel must be enabled, but not both).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_HPEN_4x8             268
/// \brief Control ME sub-pel refinement for 4x8 partitions:
///        Enable quarter-pel refinement (one of half-pel or quarter-pel must be enabled, but not both).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_QPEN_4x8             269
/// \brief Control ME sub-pel refinement for 4x8 partitions:
///        Disable bi-directional prediction try (applies to B frames only).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_DISBIMODE_4x8        270
/// \brief Control ME sub-pel refinement for 4x8 partitions:
///        Set number of motion compensations on L0 for P slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML0_4x8_PSLICE   271
/// \brief Control ME sub-pel refinement for 4x8 partitions:
///        Set number of motion compensations on L0 for B slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML0_4x8_BSLICE   272
/// \brief Control ME sub-pel refinement for 4x8 partitions:
///        Set number of motion compensations on L1 for P slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML1_4x8_BSLICE   273
/// \brief Control ME sub-pel refinement for 4x4 partitions:
///        Enable half-pel refinement (one of half-pel or quarter-pel must be enabled, but not both).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_HPEN_4x4             274
/// \brief Control ME sub-pel refinement for 4x4 partitions:
///        Enable quarter-pel refinement (one of half-pel or quarter-pel must be enabled, but not both).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_QPEN_4x4             275
/// \brief Control ME sub-pel refinement for 4x4 partitions:
///        Disable bi-directional prediction try (applies to B frames only).
/// \param enable
#define Q_AVCENC_CMP_MESUBPEL_DISBIMODE_4x4        276
/// \brief Control ME sub-pel refinement for 4x4 partitions:
///        Set number of motion compensations on L0 for P slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML0_4x4_PSLICE   277
/// \brief Control ME sub-pel refinement for 4x4 partitions:
///        Set number of motion compensations on L0 for B slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML0_4x4_BSLICE   278
/// \brief Control ME sub-pel refinement for 4x4 partitions:
///        Set number of motion compensations on L1 for P slices.
/// \param value Number of motion compensations minus 1.
#define Q_AVCENC_CMP_MESUBPEL_MCNUML1_4x4_BSLICE   279
/// \brief Number of ME tries for 16x16 partitions in P slices
/// \brief number number of tries minus 1
#define Q_AVCENC_CMP_METRY_NUMTRY16x16_PSLICE      280
/// \brief Number of ME tries for 16x8 partitions in P slices
/// \brief number number of tries minus 1
#define Q_AVCENC_CMP_METRY_NUMTRY16x8_PSLICE       281
/// \brief Number of ME tries for 8x16 partitions in P slices
/// \brief number number of tries minus 1
#define Q_AVCENC_CMP_METRY_NUMTRY8x16_PSLICE       282
/// \brief Number of ME tries for small (8x8,8x4,4x8,4x4) partitions in P slices
/// \brief number number of tries minus 1
#define Q_AVCENC_CMP_METRY_NUMTRYSMALL_PSLICE      283
/// \brief Number of ME tries for 16x16 partitions in B slices
/// \brief number number of tries minus 1
#define Q_AVCENC_CMP_METRY_NUMTRY16x16_BSLICE      284
/// \brief Number of ME tries for 16x8 partitions in B slices
/// \brief number number of tries minus 1
#define Q_AVCENC_CMP_METRY_NUMTRY16x8_BSLICE       285
/// \brief Number of ME tries for 8x16 partitions in B slices
/// \brief number number of tries minus 1
#define Q_AVCENC_CMP_METRY_NUMTRY8x16_BSLICE       286
/// \brief Number of ME tries for small (8x8,8x4,4x8,4x4) partitions in B slices
/// \brief number number of tries minus 1
#define Q_AVCENC_CMP_METRY_NUMTRYSMALL_BSLICE      287
/// \brief Number of ME tries for 16x16 partitions in B reference slices
/// \brief number number of tries minus 1
#define Q_AVCENC_CMP_METRY_NUMTRY16x16_BRSLICE     288
/// \brief Number of ME tries for 16x8 partitions in B reference slices
/// \brief number number of tries minus 1
#define Q_AVCENC_CMP_METRY_NUMTRY16x8_BRSLICE      289
/// \brief Number of ME tries for 8x16 partitions in B reference slices
/// \brief number number of tries minus 1
#define Q_AVCENC_CMP_METRY_NUMTRY8x16_BRSLICE      290
/// \brief Number of ME tries for small (8x8,8x4,4x8,4x4) partitions in B reference slices
/// \brief number number of tries minus 1
#define Q_AVCENC_CMP_METRY_NUMTRYSMALL_BRSLICE     291
/// \brief Set to have statistics register 2 gather statistics for region 7 (background) instead
///        of region 2.
/// \param enable
#define Q_AVCENC_CMP_REGION_BGNDSTAT               292
/// \brief Use a downloaded quantization scaling matrix for the sequence instead of the default
///        matrix.  Must be set to enable any of the specific downloaded matrices, and must only
///        be used in high profile.
/// \param enable
#define Q_AVCENC_CMP_SEQ_SCALE_MATRIX_PRESENT      293
/// \brief Use a downloaded matrix for sequence scaling list 0.
/// \param enable
#define Q_AVCENC_CMP_SEQ_SCALE_LIST_0              294
/// \brief Use a downloaded matrix for sequence scaling list 1.
/// \param enable
#define Q_AVCENC_CMP_SEQ_SCALE_LIST_1              295
/// \brief Use a downloaded matrix for sequence scaling list 2.
/// \param enable
#define Q_AVCENC_CMP_SEQ_SCALE_LIST_2              296
/// \brief Use a downloaded matrix for sequence scaling list 3.
/// \param enable
#define Q_AVCENC_CMP_SEQ_SCALE_LIST_3              297
/// \brief Use a downloaded matrix for sequence scaling list 4.
/// \param enable
#define Q_AVCENC_CMP_SEQ_SCALE_LIST_4              298
/// \brief Use a downloaded matrix for sequence scaling list 5.
/// \param enable
#define Q_AVCENC_CMP_SEQ_SCALE_LIST_5              299
/// \brief Use a downloaded matrix for sequence scaling list 6.
/// \param enable
#define Q_AVCENC_CMP_SEQ_SCALE_LIST_6              300
/// \brief Use a downloaded matrix for sequence scaling list 7.
/// \param enable
#define Q_AVCENC_CMP_SEQ_SCALE_LIST_7              301
/// \brief This parameter is reserved.
/// \internal obsolete
#define Q_AVCENC_CMP_QS_MODE                       302
/// \brief Type of field coding to use on interlaced material, including frame coding.  Selecting
///        frame coding will produce MBAFF frames if MBAFF is enabled, standard frames otherwise.
/// \param coding 2 == frame, 1 == field, 0 == PAFF (picture-adaptive frame/field coding)
#define Q_AVCENC_CMP_FIELD_CODING_TYPE             303
/// \brief Picture Adaptive Frame/Field Coding decison (PAFF)
/// \param value 0:Disable, 1:Enable
#define Q_AVCENC_CMP_PAFF_ENABLE                   304
/// \brief PAFF Low Threshold
/// \param value Low Threshold to determine frame coding range
#define Q_AVCENC_CMP_PAFF_LOW_THRESH               305
/// \brief PAFF High Threshold
/// \param value High Threshold to determine frame coding range
#define Q_AVCENC_CMP_PAFF_HIGH_THRESH              306
/// \brief PAFF Picture offset for MBAFF pictures
/// \param offset
#define Q_AVCENC_CMP_PAFF_OFFSET_MBAFF             307
/// \brief PAFF Picture offset for I-pictures
/// \param offset
#define Q_AVCENC_CMP_PAFF_OFFSET_I                 308
/// \brief PAFF Picture offset for P-pictures
/// \param offset
#define Q_AVCENC_CMP_PAFF_OFFSET_P                 309
/// \brief PAFF Picture offset for B-pictures
/// \param offset
#define Q_AVCENC_CMP_PAFF_OFFSET_B                 310
/// \brief PAFF Picture offset for Br-pictures
/// \param offset
#define Q_AVCENC_CMP_PAFF_OFFSET_BR                311
/// \brief This parameter controls closed GOP operation. When enabled, 
///        an IDR picture is used and no prediction from the IDR is enabled.
///        When disabledm open GOP is used with a non-IDR I-picture and
///        prediciton from the I-picture is allowed.
/// \param value 0:Disable, 1:Enable
#define Q_AVCENC_CMP_CLOSED_GOP                    312
/// \brief This parameters enable weighted prediciton in P-pictures
/// \param value 0:Disable, 1:Enable
#define Q_AVCENC_CMP_PWEIGHTED_PREDICTION          313
/// \brief This parameters enable weighted prediciton in B-pictures
/// \param value 0:Disable, 1:Enable
#define Q_AVCENC_CMP_PWEIGHTED_BIPRED_IDC          314
/// \brief This parameter controls the resolution of luma weight,
///        as per the H.264 specification.
/// \param value Luma Log Weight Denominator
#define Q_AVCENC_CMP_LUMA_LOG_WEIGHT_DENOM         315
/// \brief This parameter controls the resolution of chroma weight,
///        as per the H.264 specification.
/// \param value Chroma Log Weight Denominator
#define Q_AVCENC_CMP_CHROMA_LOG_WEIGHT_DENOM       316
/// \brief This parameter is reserved.
/// \internal QMED should always be enabled.
#define Q_AVCENC_CMP_QMED_ENABLE                   317
/// \brief This parameter enables the generation of a new SPS/PPS pair per
/// IDR picture.
/// \param value 0:Disable, 1:Enable
#define Q_AVCENC_CMP_NEW_SPS_EACH_IDR              318
/// \brief Enable access unit delimiter NALUs as defined in the AVC specification.  They have no
///        effect on the encoding but could be used by some applications.
#define Q_AVCENC_CMP_AU_DELIMITERS                 319
/// \brief Align the ends of all NALUs to the specified alignment, in bytes.  This includes slices,
///        headers, SPS/PPS, SEI message blocks, and so on.  Padding the end of one NALU means the
///        next NALU will start at the same alignment once the AVC elementary stream is demuxed.
///        It may not have that alignment in memory due to the size of the QBox headers.
/// \param NALU alignment in bytes
#define Q_AVCENC_CMP_NALU_BYTE_ALIGNMENT           320
/// \brief This parameter enables the inverse telecine detection and processing,
/// \param value 0:Disable, 1:Enable      
#define Q_AVCENC_CMP_INVERSE_TELECINE_ENABLE       321
/// \brief Inverse telecine field distance 1
///        Field distances from current repeat top field to previous repeat bottom field
/// \param value Number of fields
#define Q_AVCENC_CMP_INVERSE_TELECINE_CAD_DIS_1    322
/// \brief Inverse telecine field distance 2
///        Field distances from current repeat bottom field to previous repeat top field
/// \param value Number of fields
#define Q_AVCENC_CMP_INVERSE_TELECINE_CAD_DIS_2    323
/// \brief Inverse telecine lock in time
///        The number that continous cadences has to be bigger than 
///        before entering inverse telecine state
/// \param value Number of cadences
#define Q_AVCENC_CMP_INVERSE_TELECINE_LOCK_IN_TIME 324
/// \brief Inverse telecine lock out time
///        The number that continous broken cadences has to be bigger than 
///        before inverse telecine state is getting out
/// \param value Number of cadences
#define Q_AVCENC_CMP_INVERSE_TELECINE_LOCK_OUT_TIME 325
/// \brief This parameter is reserved.
/// \internal obsolete
#define Q_AVCENC_CMP_PRIVACY_REGION_ENABLE         326
/// \brief Controls optional intra refresh of P frames, useful for extremely long GOPs and
///        infinite GOPs.  Selects a length for a row of macroblocks that will be forced to
///        intra coding, moving through the image by the length of the row each P frame.
///        This means that a larger row size means a faster refresh.  For example, with a size
///        of 20, macroblocks 0-19 would be forced to intra coding in the first P frame, then
///        macroblocks 20-39 would be forced to intra coding in the next P frame.
/// \param rows Number of macroblocks per row, 0 to disable.
#define Q_AVCENC_CMP_ROLLING_REFRESH_MB_ROWS       327
/// \brief This parameter enables fixed length nalu
/// \param value 0:Disable, 1:Enable
#define Q_AVCENC_CMP_FIXED_LENGTH_NALU_ENABLE      328
/// \brief This parameter specifies maximum allowable NAL unit size in bytes
/// \param value Number of bytes 
#define Q_AVCENC_CMP_FIXED_NALU_LENGTH             329
/// \brief This parameter specifies the guard bytes of I picture,
///        The encoder would use this to offset size of fixed length nalu internally
/// \param value Number of bytes   
#define Q_AVCENC_CMP_FIXED_LENGTH_NALU_GUARD_I     330
/// \brief This parameter specifies the guard bytes of B/P picture,
///        The encoder would use this to offset size of fixed length nalu internally
/// \param value Number of bytes   
#define Q_AVCENC_CMP_FIXED_LENGTH_NALU_GUARD_NI    331
/// \brief This parameter helps the calculation of initial sending number of MBs of I picture.
///        The inital sending number of MBs equals the estimate MBs per NALU right shift 
///        with this value
/// \param value Positive interger
#define Q_AVCENC_CMP_FIXED_LENGTH_NALU_INIT_I      332
/// \brief This parameter helps the calculation of initial sending number of MBs of B/P picture.
///        The inital sending number of MBs equals the estimate MBs per NALU right shift 
///        with this value
/// \param value Positive interger
#define Q_AVCENC_CMP_FIXED_LENGTH_NALU_INIT_NI     333
/// \brief This parameter specifies the slice size of check point 1 for I picture.
///        The threshould that encoder uses to adjust the encoding pipe length before
///        current slice finishes
/// \param value Number of bytes
#define Q_AVCENC_CMP_FIXED_LENGTH_NALU_CKPT_I_1    334
/// \brief This parameter specifies the slice size of check point 2 for I picture.
///        The threshould that encoder uses to adjust the encoding pipe length before
///        current slice finishes. It should be bigger than VENC_CFG_FIXED_LENGTH_NALU_CKPT_I_1
/// \param value Number of bytes
#define Q_AVCENC_CMP_FIXED_LENGTH_NALU_CKPT_I_2    335
/// \brief This parameter specifies the slice size of check point 1 for B/P picture.
///        The threshould that encoder uses to adjust the encoding pipe length before
///        current slice finishes
/// \param value Number of bytes
#define Q_AVCENC_CMP_FIXED_LENGTH_NALU_CKPT_NI_1   336
/// \brief This parameter specifies the slice size of check point 2 for B/P picture.
///        The threshould that encoder uses to adjust the encoding pipe length before
///        current slice finishes, It should be bigger than VENC_CFG_FIXED_LENGTH_NALU_CKPT_NI_1
/// \param value Number of bytes
#define Q_AVCENC_CMP_FIXED_LENGTH_NALU_CKPT_NI_2   337
/// \brief This parameter specifies the number of MBs to send to the encoder pipe after
///        the slice size is bigger than check point 1 for I picture
/// \param value Number of MBs
#define Q_AVCENC_CMP_FIXED_LENGTH_NALU_QDEPTH_I_1  338
/// \brief This parameter specifies the number of MBs to send to the encoder pipe after
///        the slice size is bigger than check point 2 for I picture
/// \param value Number of MBs
#define Q_AVCENC_CMP_FIXED_LENGTH_NALU_QDEPTH_I_2  339
/// \brief This parameter specifies the number of MBs to send to the encoder pipe after
///        the slice size is bigger than check point 1 for P/B picture
/// \param value Number of MBs
#define Q_AVCENC_CMP_FIXED_LENGTH_NALU_QDEPTH_NI_1 340
/// \brief This parameter specifies the number of MBs to send to the encoder pipe after
///        the slice size is bigger than check point 2 for P/B picture
/// \param value Number of MBs
#define Q_AVCENC_CMP_FIXED_LENGTH_NALU_QDEPTH_NI_2 341
/// \brief Maximum number of reference frames used by the decoder.  Defaults to 4, which is
///        required for B frame support.  Can be reduced to 2 for baseline profile (frame
///        coding and no B frames), for compatibility with some decoders.
#define Q_AVCENC_CMP_MAX_REFERENCE_FRAMES          342
/// \brief This parameter enables segment based encoding
/// \param value 0:Disable, 1:Enable
#define Q_AVCENC_CMP_SEGMENT_ENCODE_ENABLE         343
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_OVERRIDE              344
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_1REFS_LUMA_PART0 345
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_1REFS_CHROMA_PART0 346
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_1REFS_LUMA_PART1 347
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_1REFS_CHROMA_PART1 348
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_1REFS_LUMA_PART2 349
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_1REFS_CHROMA_PART2 350
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_1REFS_LUMA_PART3 351
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_1REFS_CHROMA_PART3 352
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_2REFS_LUMA_PART0 353
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_2REFS_CHROMA_PART0 354
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_2REFS_LUMA_PART1 355
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_2REFS_CHROMA_PART1 356
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_2REFS_LUMA_PART2 357
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_2REFS_CHROMA_PART2 358
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_2REFS_LUMA_PART3 359
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_2REFS_CHROMA_PART3 360
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_3REFS_LUMA_PART0 361
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_3REFS_CHROMA_PART0 362
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_3REFS_LUMA_PART1 363
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_3REFS_CHROMA_PART1 364
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_3REFS_LUMA_PART2 365
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_3REFS_CHROMA_PART2 366
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_3REFS_LUMA_PART3 367
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_3REFS_CHROMA_PART3 368
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_4REFS_LUMA_PART0 369
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_4REFS_CHROMA_PART0 370
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_4REFS_LUMA_PART1 371
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_4REFS_CHROMA_PART1 372
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_4REFS_LUMA_PART2 373
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_4REFS_CHROMA_PART2 374
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_4REFS_LUMA_PART3 375
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_CMP_MECACHE_PARTSIZE_4REFS_CHROMA_PART3 376
/// \brief This parameter is reserved.
/// \internal
/// \brief Area Filter Enable
/// \param value 0:Disable, 1:Enable
#define Q_AVCENC_CMP_AF_ENABLE                     377
/// \brief This parameter is reserved.
/// \internal
/// Area Filter Region Combination control. Defines how the region selected to
/// control the Area Filter is derived from the four contributing region
/// similarity values in interlaced or MBAFF encoding modes
/// \param value 0: no combination
///              1: Frame
///              2: Field
///              3: MBAFF
#define Q_AVCENC_CMP_AF_REG_COMB                   378
/// \brief This parameter is reserved.
/// \internal
/// Area Filter Region Select
/// \param value Region from 0-4
#define Q_AVCENC_CMP_AF_REG_SEL                    379
/// \brief This parameter is reserved.
/// \internal
/// Area Filter Y0 Coefficient
/// \param value [-127,128]
#define Q_AVCENC_CMP_AF_COEFF_Y0                   380
/// \brief This parameter is reserved.
/// \internal
/// Area Filter C0 Coefficient
/// \param value [-127,128]
#define Q_AVCENC_CMP_AF_COEFF_C0                   381
/// \brief This parameter is reserved.
/// \internal
/// Area Filter Y1 Coefficient
/// \param value [-127,128]
#define Q_AVCENC_CMP_AF_COEFF_Y1                   382
/// \brief This parameter is reserved.
/// \internal
/// Area Filter C1 Coefficient
/// \param value [-127,128]
#define Q_AVCENC_CMP_AF_COEFF_C1                   383
/// \brief This parameter is reserved.
/// \internal
/// Area Filter Y2 Coefficient
/// \param value [-127,128]
#define Q_AVCENC_CMP_AF_COEFF_Y2                   384
/// \brief This parameter is reserved.
/// \internal
/// Area Filter C2  Coefficient
/// \param value [-127,128]
#define Q_AVCENC_CMP_AF_COEFF_C2                   385
/// \brief This parameter is reserved.
/// \internal
/// Area FilteR Y3 Coefficient
/// \param value [-127,128]
#define Q_AVCENC_CMP_AF_COEFF_Y3                   386
/// \brief This parameter is reserved.
/// \internal
/// Area FilteR C3 Coefficient
/// \param value [-127,128]
#define Q_AVCENC_CMP_AF_COEFF_C3                   387
/// \brief This parameter is reserved.
/// \internal
/// Target performance for I frame
/// \param value encoding time in microseconds
#define Q_AVCENC_CMP_TARGET_PERFORMANCE_I          388
/// \brief This parameter is reserved.
/// \internal
/// Target performance for P frame
/// \param value encoding time in microseconds
#define Q_AVCENC_CMP_TARGET_PERFORMANCE_P          389
/// \brief This parameter is reserved.
/// \internal
/// Target performance for P frame
/// \param value encoding time in microseconds
#define Q_AVCENC_CMP_TARGET_PERFORMANCE_B          390
/// \brief This parameter is reserved.
/// \internal
/// Allow output of encoder statistics for capture
/// \param enable
#define Q_AVCENC_CMP_STATS_CAPTURE                 391
/// \brief This parameter is reserved.
/// \internal
/// Restrict the encoding so that motion vectors do not point too far outside the picture
/// \param value motion vector restriction in macroblocks (0-2)
#define Q_AVCENC_CMP_RESTRICT_MV_RANGE             392
/// \brief This parameter is reserved.
/// \internal
/// Activate dumping of VCIF commands to memory
/// \param enable
#define Q_AVCENC_CMP_VCIF_CAPTURE                  393
/// \brief This parameter is reserved.
/// \internal Allow the encoder to block when the reconstructed frame pool is full, rather than
///           dropping reconstructed preview frames.  This is necessary for capturing the
///           reconstructed frames to a file.
/// \param enable
#define Q_AVCENC_CMP_RECON_PREVIEW_BLOCKING        394
/// \brief This parameter is reserved.
/// \internal obsolete
#define Q_AVCENC_CMP_QS_FOR_EVERY_PIC              395
/// \brief Use picture level quantization scaling matrices.  When enabled, a matrix will be
///        automatically selected on every P picture based on the current QP.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_MATRIX_PRESENT      396
/// \brief Use a downloaded matrix for picture scaling list 0.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_LIST_0              397
/// \brief Use a downloaded matrix for picture scaling list 1. Can be set to 0 to save header bits
///        if the list 1 coefficients are the same as the list 0 coefficients.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_LIST_1              398
/// \brief Use a downloaded matrix for picture scaling list 2.  Can be set to 0 to save header bits
///        if the list 2 coefficients are the same as the list 0 coefficients.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_LIST_2              399
/// \brief Use a downloaded matrix for picture scaling list 3.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_LIST_3              400
/// \brief Use a downloaded matrix for picture scaling list 4.  Can be set to 0 to save header bits
///        if the list 4 coefficients are the same as the list 3 coefficients.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_LIST_4              401
/// \brief Use a downloaded matrix for picture scaling list 5.  Can be set to 0 to save header bits
///        if the list 5 coefficients are the same as the list 3 coefficients.
#define Q_AVCENC_CMP_PIC_SCALE_LIST_5              402
/// \brief Use a downloaded matrix for picture scaling list 6.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_LIST_6              403
/// \brief Use a downloaded matrix for picture scaling list 7.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_LIST_7              404
/// \brief Send a downloaded matrix to the hardware for picture scaling list 0.  Must be 1 if
///        VENC_CFG_PIC_SCALE_LIST_0 is 1.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_USAGE_0             405
/// \brief Send a downloaded matrix to the hardware for picture scaling list 1.  Must be 1 if
///        VENC_CFG_PIC_SCALE_LIST_1 is 1.  Can be 1 if VENC_PIC_SCALE_LIST_0 is 1 and list 1
///        uses the same coefficients as list 0.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_USAGE_1             406
/// \brief Send a downloaded matrix to the hardware for picture scaling list 2.  Must be 1 if
///        VENC_CFG_PIC_SCALE_LIST_2 is 1.  Can be 1 if VENC_PIC_SCALE_LIST_0 is 1 and list 2
///        uses the same coefficients as list 0.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_USAGE_2             407
/// \brief Send a downloaded matrix to the hardware for picture scaling list 3.  Must be 1 if
///        VENC_CFG_PIC_SCALE_LIST_3 is 1.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_USAGE_3             408
/// \brief Send a downloaded matrix to the hardware for picture scaling list 4.  Must be 1 if
///        VENC_CFG_PIC_SCALE_LIST_4 is 1.  Can be 1 if VENC_PIC_SCALE_LIST_3 is 1 and list 4
///        uses the same coefficients as list 3.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_USAGE_4             409
/// \brief Send a downloaded matrix to the hardware for picture scaling list 5.  Must be 1 if
///        VENC_CFG_PIC_SCALE_LIST_5 is 1.  Can be 1 if VENC_PIC_SCALE_LIST_3 is 1 and list 5
///        uses the same coefficients as list 3.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_USAGE_5             410
/// \brief Send a downloaded matrix to the hardware for picture scaling list 6.  Must be 1 if
///        VENC_CFG_PIC_SCALE_LIST_6 is 1.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_USAGE_6             411
/// \brief Send a downloaded matrix to the hardware for picture scaling list 7.  Must be 1 if
///        VENC_CFG_PIC_SCALE_LIST_7 is 1.
/// \param enable
#define Q_AVCENC_CMP_PIC_SCALE_USAGE_7             412
/// \brief Send a parameter to enable motion detection. 
/// \param enable
#define Q_AVCENC_CMP_MOTION_DETECT_ENABLE          413
/// \brief this parameter is reserved
#define Q_AVCENC_CMP_RESERVED_PARAM_0              414
/// \brief this parameter is reserved
#define Q_AVCENC_CMP_RESERVED_PARAM_1              415
#define Q_AVCENC_CMP_IPX_BACKWARD                  416
#define Q_AVCENC_CMP_IPX_JOIN                      417
#define Q_AVCENC_CMP_TF_ENABLE                     418
#define Q_AVCENC_CMP_TF_OUT_TRAN_EN                419
#define Q_AVCENC_CMP_TF_SHIFT                      420
#define Q_AVCENC_CMP_TF_HF_GAIN                    421
#define Q_AVCENC_CMP_TF_MOT_THOLD0                 422
#define Q_AVCENC_CMP_TF_MOT_THOLD1                 423
#define Q_AVCENC_CMP_TF_MOT_THOLD2                 424
#define Q_AVCENC_CMP_TF_HF_THOLD0                  425
#define Q_AVCENC_CMP_TF_HF_THOLD1                  426
#define Q_AVCENC_CMP_TF_HF_THOLD2                  427
#define Q_AVCENC_CMP_TF_STAT0                      428
#define Q_AVCENC_CMP_TF_STAT1                      429
#define Q_AVCENC_CMP_TF_STAT2                      430
#define Q_AVCENC_CMP_TF_STAT3                      431
#define Q_AVCENC_CMP_TF_STAT4                      432
#define Q_AVCENC_CMP_TF_STAT5                      433
#define Q_AVCENC_CMP_TF_STAT6                      434
#define Q_AVCENC_CMP_TF_STAT7                      435
#define Q_AVCENC_CMP_TF_STAT8                      436
#define Q_AVCENC_CMP_TF_STAT9                      437
#define Q_AVCENC_CMP_TF_STATA                      438
#define Q_AVCENC_CMP_TF_STATB                      439
#define Q_AVCENC_CMP_TF_STATC                      440
#define Q_AVCENC_CMP_TF_STATD                      441
#define Q_AVCENC_CMP_TF_STATE                      442
#define Q_AVCENC_CMP_TF_STATF                      443
#define Q_AVCENC_CMP_TF_TRAN0                      444
#define Q_AVCENC_CMP_TF_TRAN1                      445
#define Q_AVCENC_CMP_TF_TRAN2                      446
#define Q_AVCENC_CMP_TF_TRAN3                      447
#define Q_AVCENC_CMP_TF_TRAN4                      448
#define Q_AVCENC_CMP_TF_TRAN5                      449
#define Q_AVCENC_CMP_TF_TRAN6                      450
#define Q_AVCENC_CMP_TF_TRAN7                      451
#define Q_AVCENC_CMP_TF_TRAN8                      452
#define Q_AVCENC_CMP_TF_TRAN9                      453
#define Q_AVCENC_CMP_TF_TRANA                      454
#define Q_AVCENC_CMP_TF_TRANB                      455
#define Q_AVCENC_CMP_TF_TRANC                      456
#define Q_AVCENC_CMP_TF_TRAND                      457
#define Q_AVCENC_CMP_TF_TRANE                      458
#define Q_AVCENC_CMP_TF_TRANF                      459
/// \brief If enabled encoder generates elementary stream instead of qbox. 
/// \param enable
#define Q_AVCENC_CMP_ELEMENTARY_STREAM             460
/// \brief The number of levels of hierarchy to use in the current GOP structure.
///        Each level of the hierarchy will reference only frames at the same or
///        higher levels. The level of frame X = Y*2^n is n, where 0 <= n <= LEVEL.
/// \param value The number of levels of hierarchy to use in the GOP structure
#define Q_AVCENC_CMP_GOP_HIERARCHY_LEVEL           461
/// \brief Each picture will have a PPS if this is enabled
/// \param enable
#define Q_AVCENC_CMP_PPS_PER_PICTURE               462
/// \brief SPS id
/// \param value [0,31]
#define Q_AVCENC_CMP_SPS_ID                        463
/// \brief Adaptive skip control
/// \param enable
#define Q_AVCENC_CMP_ADAPTIVE_SKIP_ENABLE          464
/// \brief This parameter is reserved.
/// \internal
/// VCIF dump should store the entire GOP (dangerous as there may not be enough memory)
/// \param enable
#define Q_AVCENC_CMP_VCIF_CAPTURE_GOP              465
/// \brief Number of TS packets will be packed together and padded before and after as a group
/// \param value [0,5] 0 means no padding between any TS packets
#define Q_AVCENC_CMP_TS_PACK_NUM                   466
/// \brief Adaptive skip fade threshold 
/// \param value
#define Q_AVCENC_CMP_ADAPTIVE_SKIP_FADE_THOLD      467
/// \brief Each frame will have a SPS if this is enabled
/// \param enable
#define Q_AVCENC_CMP_SPS_PER_FRAME                 468
/// \brief specify dts increase value per frame
/// \param value
#define Q_AVCENC_CMP_DTS_INTERVAL                  469
/// \brief specify dts initial value low
/// \param value
#define Q_AVCENC_CMP_START_DTS_LOW                 470
/// \brief Enable qpmod for 1264 compatible encoding
/// \param enable
#define Q_AVCENC_CMP_COMPATIBLE1264_QPMOD_ENABLE   471
/// \brief Enable motion detection bit map. 
/// \param enable
#define Q_AVCENC_CMP_MOTION_BITMAP_ENABLE          472
/// \brief specify the threshold of motion bit map. 
/// \param value 
#define Q_AVCENC_CMP_MOTION_BITMAP_THRESHOLD       473
/// \brief specify bytes will be inserted per frame
/// \param value
#define Q_AVCENC_CMP_BYTES_INSERT_PER_FRAME        474
/// \brief Enable motion counter. 
/// \param enable
#define Q_AVCENC_CMP_MOTION_COUNTER_ENABLE         475
/// \brief Transport stream instead of qbox stream enable
/// \param enable    
#define Q_AVCENC_CMP_TS_ENABLE                     476
/// \brief Transport stream ID
/// \param value
#define Q_AVCENC_CMP_TS_STREAM_ID                  477
/// \brief Enable extra au per gop
/// \param enable
#define Q_AVCENC_CMP_EXTRA_AU_PER_GOP              478
/// \brief timing master in transport stream
/// \param enable
#define Q_AVCENC_CMP_TS_PCR_MASTER                 479
/// \brief Enable the bitstream restriction information in the VUI.  This defaults to off.
/// \param enable
#define Q_AVCENC_CMP_VUI_BITSTREAM_RESTRICTION_FLAG 480
/// \brief Enables the encoder to work with input frames that have NTP timestamps
/// \param enable
#define Q_AVCENC_CMP_NTP_ENABLE                    481
/// \brief Start encoding on this signature, zero means encoding without waiting 
#define Q_AVCENC_CMP_START_SIGNATURE               482
/// \brief Enable the blueray compatible pic type 
#define Q_AVCENC_CMP_BLUERAY_COMPLIANCE            483
/// \brief This parameter enables motion detection to work at the same time with privacy region,
///        burn-in, etc even when their region of interests overlaps. The current frame processed
///        by the preproc object is delayed by one frame so that the capturing frame of LVPP is
///        not being compared to a pre-processed frame.
/// \param 1 to enable, 0 to disable
#define Q_AVCENC_CMP_DELAY_PREPROC                 484
/// \brief This parameter enables motion vector restriction when intra refresh is on
/// \param 1 to enable, 0 to disable
#define Q_AVCENC_CMP_INTRA_REFRESH_RESTRICT_MV     485
/// \brief Enable the ts adaptation field private data
/// \param is 1 to enable, 0 to disable
#define Q_AVCENC_CMP_TS_ADAPTATION_PRIV_DATA_ENABLE 486
/// \brief This parameter is reserved.
///        This parameter will always be the last parameter in the enum.
///        Never add new entry after this. Add new entries BEFORE.
///        This is done to prevent the corruption of the default config array caused by addition of
///        new enum entries to this enum but their default values are not added to the default config array.
#define Q_AVCENC_CMP_THIS_IS_THE_LAST_PARAMETER    487

// Generated from mediaware.h VENC_REG_CFG s/VENC_REG_CFG/Q_AVCENC_REG_CMP/
/// \brief Set this region to be a priority region.  When a macroblock has any match
///        to this region, it will use this region.
/// \param enable
#define Q_AVCENC_REG_CMP_PRIORITY                    0
/// \brief Set this region to be an exterior region.  Macroblocks will match this region
///        if they are outside the detected region area, instead of inside it.
#define Q_AVCENC_REG_CMP_EXTERIOR                    1
/// \brief Macroblock partitions using BIGALG algorithm parameter (the rest will use SMALLALG).
/// \param partitions 0 = 16x16, 1 = 16x16, 16x8, 8x16 2 = 16x16, 16x8, 8x16, 8x8
#define Q_AVCENC_REG_CMP_MEREGION_BIGPART            2
/// \brief Algorithm ID (from MEAlgDef) for big partitions
/// \brief id algorithm ID
#define Q_AVCENC_REG_CMP_MEREGION_BIGALG             3
/// \brief Algorithm ID (from MEAlgDef) for small partitions
/// \brief id algorithm ID
#define Q_AVCENC_REG_CMP_MEREGION_SMALLALG           4
/// \brief Initial ME search vector for 16x16 partitions
/// \param type 0 = rough MV, 1 = PME, 2 = MVPRED, 3 = zero MV
#define Q_AVCENC_REG_CMP_MEREGION_MVSEL16X16         5
/// \brief Initial ME search vector for 16x8 partitions
/// \param type 0 = rough MV, 1 = PME, 2 = MVPRED, 3 = zero MV
#define Q_AVCENC_REG_CMP_MEREGION_MVSEL16X8          6
/// \brief Initial ME search vector for 8x16 partitions
/// \param type 0 = rough MV, 1 = PME, 2 = MVPRED, 3 = zero MV
#define Q_AVCENC_REG_CMP_MEREGION_MVSEL8X16          7
/// \brief Initial ME search vector for 8x8 partitions
/// \param type 0 = rough MV, 1 = PME, 2 = MVPRED, 3 = zero MV
#define Q_AVCENC_REG_CMP_MEREGION_MVSEL8X8           8
/// \brief Inter mode rate offset for zero luma and nonzero chroma residue (in P slices)
/// \brief offset [0..31] offset
#define Q_AVCENC_REG_CMP_RATEOFFSET_ROINTERY_PSLICE   9
/// \brief Inter mode rate offset for nonzero luma and zero chroma residue (in P slices)
/// \brief offset [0..31] offset
#define Q_AVCENC_REG_CMP_RATEOFFSET_ROINTERC_PSLICE  10
/// \brief Inter mode rate offset for zero luma and zero chroma residue (in P slices)
/// \brief offset [0..31] offset
#define Q_AVCENC_REG_CMP_RATEOFFSET_ROINTERYC_PSLICE  11
/// \brief Inter mode rate offset for P skip (in P slices)
/// \brief offset [0..31] offset
#define Q_AVCENC_REG_CMP_RATEOFFSET_ROSKIP_PSLICE   12
/// \brief Inter mode rate offset for zero luma and nonzero chroma residue (in B slices)
/// \brief offset [0..31] offset
#define Q_AVCENC_REG_CMP_RATEOFFSET_ROINTERY_BSLICE  13
/// \brief Inter mode rate offset for nonzero luma and zero chroma residue (in B slices)
/// \brief offset [0..31] offset
#define Q_AVCENC_REG_CMP_RATEOFFSET_ROINTERC_BSLICE  14
/// \brief Inter mode rate offset for zero luma and zero chroma residue (in B slices)
/// \brief offset [0..31] offset
#define Q_AVCENC_REG_CMP_RATEOFFSET_ROINTERYC_BSLICE  15
/// \brief Inter mode rate offset for P skip (in B slices)
/// \brief offset [0..31] offset
#define Q_AVCENC_REG_CMP_RATEOFFSET_ROSKIP_BSLICE   16
/// \brief Inter mode rate offset for zero luma and nonzero chroma residue (in BR slices)
/// \brief offset [0..31] offset
#define Q_AVCENC_REG_CMP_RATEOFFSET_ROINTERY_BRSLICE  17
/// \brief Inter mode rate offset for nonzero luma and zero chroma residue (in BR slices)
/// \brief offset [0..31] offset
#define Q_AVCENC_REG_CMP_RATEOFFSET_ROINTERC_BRSLICE  18
/// \brief Inter mode rate offset for zero luma and zero chroma residue (in BR slices)
/// \brief offset [0..31] offset
#define Q_AVCENC_REG_CMP_RATEOFFSET_ROINTERYC_BRSLICE  19
/// \brief Inter mode rate offset for P skip (in BR slices)
/// \brief offset [0..31] offset
#define Q_AVCENC_REG_CMP_RATEOFFSET_ROSKIP_BRSLICE  20
/// \brief P-skip is disqualified if any of the luma 4x4 SSE is larger than or equal to this
///        threshold (<< 4).
/// \param threshold threshold/16 [0..255]
#define Q_AVCENC_REG_CMP_RATEOFFSET_MAXSSE4X4Y      21
/// \brief P-skip is disqualified if any of the chroma 4x4 SSE is larger than or equal to this
///        threshold (<< 4).
/// \param threshold threshold/16 [0..255]
#define Q_AVCENC_REG_CMP_RATEOFFSET_MAXSSE4X4C      22
/// \brief If nonzero, use a MaxSSE derived from the slice QP rather than the constant values
///        set by VENC_REG_CFG_RATEOFFSET_MAXSSE4X4Y and VENC_REG_CFG_RATEOFFSET_MAXSSE4X4C.
///        The value is based on the parameter using param * QP / 100 - 50 for luma, and half
///        this value for chroma (clipped to [1..255]).
/// \param [1..255] value, 0 to disable.
#define Q_AVCENC_REG_CMP_RATEOFFSET_MAX4x4SSE_SLICEQP_CONST  23
/// \brief Minimum QP delta for intra blocks (QP >= slice QP - QPDELTAINTRAMIN)
/// \brief mindelta [0..31 value]
#define Q_AVCENC_REG_CMP_MULTQP_QPDELTAINTRAMIN     24
/// \brief Maximum QP delta for intra blocks (QP <= slice QP + QPDELTAINTRAMAX)
/// \brief maxdelta [0..31] value
#define Q_AVCENC_REG_CMP_MULTQP_QPDELTAINTRAMAX     25
/// \brief Minimum QP delta for intra blocks (QP >= slice QP - QPDELTAINTRAMIN)
/// \brief mindelta [0..31 value]
#define Q_AVCENC_REG_CMP_MULTQP_QPDELTAINTRA        26
/// \brief Decrease the intra QP delta for regions that are common across the image.  This
///        avoids having large QP deltas applied across most of the image.
/// \param enable
#define Q_AVCENC_REG_CMP_MULTQP_QPDELTAINTRACOLOREN  27
/// \brief For P and B frames, limit the fixed QP delta to be less than the base QP
///        of the last I slice + QPDELTAINTRA.  So the QP delta does not cause the
///        final QP to increase above that of the last I slice.
/// \param enable
#define Q_AVCENC_REG_CMP_MULTQP_QPDELTAINTRAININTEREN  28
/// \brief Minimum QP delta for inter blocks (QP >= slice QP - QPDELTAINTERMIN)
/// \brief mindelta [0..31 value]
#define Q_AVCENC_REG_CMP_MULTQP_QPDELTAINTERMIN     29
/// \brief Maximum QP delta for inter blocks (QP <= slice QP + QPDELTAINTERMAX)
/// \brief maxdelta [0..31] value
#define Q_AVCENC_REG_CMP_MULTQP_QPDELTAINTERMAX     30
/// \brief Minimum QP delta for inter blocks (QP >= slice QP - QPDELTAINTERMIN)
/// \brief mindelta [0..31 value]
#define Q_AVCENC_REG_CMP_MULTQP_QPDELTAINTER        31
/// \brief Decrease the inter QP delta for regions that are common across the image.  This
///        avoids having large QP deltas applied across most of the image.
/// \param enable
#define Q_AVCENC_REG_CMP_MULTQP_QPDELTAINTERCOLOREN  32
/// \brief Enable alternate QP deltas for intra modes.  An alternate value will be selected
///        if it produces superior rate/distortion results.
/// \param mask (-6, -5, -4, -3, -2, -1, 1, 2)
#define Q_AVCENC_REG_CMP_MULTQP_QPDELTAINTRAMASK    33
/// \brief Enable alternate QP deltas for inter modes.  An alternate value will be selected
///        if it produces superior rate/distortion results.
/// \param mask (-6, -5, -4, -3, -2, -1, 1, 2)
#define Q_AVCENC_REG_CMP_MULTQP_QPDELTAINTERMASK    34
/// \brief Control which inter transforms can be selected in I slices.
/// \param transforms 0 = 4x4 only, 1 = 8x8 for 8x8 and large partitions, otherwise 4x4,
///                   2 = 4x4 always, plus 8x8 for 8x8 partitions, 3 = 4x4 always, and 8x8
///                   for 8x8 and larger partitions.
#define Q_AVCENC_REG_CMP_INTRAMODES_INTERTRANS_ISLICE  35
/// \brief Select intra chroma prediction modes in I slices.
/// \param modes 0 = use VPP statistics which may skip horizontal or vertical prediction,
///              1 = use all modes
#define Q_AVCENC_REG_CMP_INTRAMODES_I_CHROMA_ISLICE  36
/// \brief Select intra luma prediction modes for 16x16 blocks in I slices.
/// \param modes 0 = disable 16x16 modes
///              1 = use VPP statistics which may skip horizontal or vertical prediction, and skips planar
///              2 = use VPP statistics but allow planar prediction
///              3 = use all modes
#define Q_AVCENC_REG_CMP_INTRAMODES_I_16X16_ISLICE  37
/// \brief Select intra luma prediction modes for 8x8 blocks in I slices.
/// \param modes 0 = disable 8x8 modes
///              1 = use VPP statistics which may skip horizontal, vertical, or planar modes
///              2 = use VPP statistics with optional use of all modes
///              3 = use all modes
#define Q_AVCENC_REG_CMP_INTRAMODES_I_8X8_ISLICE    38
/// \brief Select intra luma prediction modes for 4x4 blocks in I slices.
/// \param modes 0 = disable 4x4 modes
///              1 = use VPP statistics which may skip horizontal, vertical, or planar modes
///              2 = use VPP statistics with optional use of all modes
///              3 = use all modes
#define Q_AVCENC_REG_CMP_INTRAMODES_I_4X4_ISLICE    39
/// \brief Control which inter transforms can be selected in P slices.
/// \param transforms 0 = 4x4 only, 1 = 8x8 for 8x8 and large partitions, otherwise 4x4,
///                   2 = 4x4 always, plus 8x8 for 8x8 partitions, 3 = 4x4 always, and 8x8
///                   for 8x8 and larger partitions.
#define Q_AVCENC_REG_CMP_INTRAMODES_INTERTRANS_PSLICE  40
/// \brief Select intra chroma prediction modes in P slices.
/// \param modes 0 = use VPP statistics which may skip horizontal or vertical prediction,
///              1 = use all modes
#define Q_AVCENC_REG_CMP_INTRAMODES_I_CHROMA_PSLICE  41
/// \brief Select intra luma prediction modes for 16x16 blocks in P slices.
/// \param modes 0 = disable 16x16 modes
///              1 = use VPP statistics which may skip horizontal or vertical prediction, and skips planar
///              2 = use VPP statistics but allow planar prediction
///              3 = use all modes
#define Q_AVCENC_REG_CMP_INTRAMODES_I_16X16_PSLICE  42
/// \brief Select intra luma prediction modes for 8x8 blocks in P slices.
/// \param modes 0 = disable 8x8 modes
///              1 = use VPP statistics which may skip horizontal, vertical, or planar modes
///              2 = use VPP statistics with optional use of all modes
///              3 = use all modes
#define Q_AVCENC_REG_CMP_INTRAMODES_I_8X8_PSLICE    43
/// \brief Select intra luma prediction modes for 4x4 blocks in P slices.
/// \param modes 0 = disable 4x4 modes
///              1 = use VPP statistics which may skip horizontal, vertical, or planar modes
///              2 = use VPP statistics with optional use of all modes
///              3 = use all modes
#define Q_AVCENC_REG_CMP_INTRAMODES_I_4X4_PSLICE    44
/// \brief Control which inter transforms can be selected in B slices.
/// \param transforms 0 = 4x4 only, 1 = 8x8 for 8x8 and large partitions, otherwise 4x4,
///                   2 = 4x4 always, plus 8x8 for 8x8 partitions, 3 = 4x4 always, and 8x8
///                   for 8x8 and larger partitions.
#define Q_AVCENC_REG_CMP_INTRAMODES_INTERTRANS_BSLICE  45
/// \brief Select intra chroma prediction modes in B slices.
/// \param modes 0 = use VPP statistics which may skip horizontal or vertical prediction,
///              1 = use all modes
#define Q_AVCENC_REG_CMP_INTRAMODES_I_CHROMA_BSLICE  46
/// \brief Select intra luma prediction modes for 16x16 blocks in B slices.
/// \param modes 0 = disable 16x16 modes
///              1 = use VPP statistics which may skip horizontal or vertical prediction, and skips planar
///              2 = use VPP statistics but allow planar prediction
///              3 = use all modes
#define Q_AVCENC_REG_CMP_INTRAMODES_I_16X16_BSLICE  47
/// \brief Select intra luma prediction modes for 8x8 blocks in B slices.
/// \param modes 0 = disable 8x8 modes
///              1 = use VPP statistics which may skip horizontal, vertical, or planar modes
///              2 = use VPP statistics with optional use of all modes
///              3 = use all modes
#define Q_AVCENC_REG_CMP_INTRAMODES_I_8X8_BSLICE    48
/// \brief Select intra luma prediction modes for 4x4 blocks in B slices.
/// \param modes 0 = disable 4x4 modes
///              1 = use VPP statistics which may skip horizontal, vertical, or planar modes
///              2 = use VPP statistics with optional use of all modes
///              3 = use all modes
#define Q_AVCENC_REG_CMP_INTRAMODES_I_4X4_BSLICE    49
/// \brief Enable half-pel motion vectors.  Either half-pel or quarter-pel motion vectors must
///        be enabled, but not both.
/// \param enable
#define Q_AVCENC_REG_CMP_INTERMODES_HALFPELEN       50
/// \brief Enable quarter-pel motion vectors.  Either half-pel or quarter-pel motion vectors must
///        be enabled, but not both.
/// \param enable
#define Q_AVCENC_REG_CMP_INTERMODES_QUARTERPELEN    51
/// \brief Enable P-skip in P slices.
/// \param enable
#define Q_AVCENC_REG_CMP_INTERMODES_SKIPEN          52
/// \brief Enable 16x16 inter prediction with 0 coded motion vectors (P_16x16 and B_Direct_16x16)
/// \param enable
#define Q_AVCENC_REG_CMP_INTERMODES_DIRECT16EN      53
/// \brief Enable B_Direct_8x8 inter prediction in B slices
/// \param enable
#define Q_AVCENC_REG_CMP_INTERMODES_DIRECT8EN       54
/// \brief Enable inter prediction.
/// \param enable
#define Q_AVCENC_REG_CMP_INTERMODES_INTEREN         55
/// \brief Control the lambda dependency on QP.
/// \param mode 1 = constant per region (based on slice QP plus region intra QP delta for I slices
///                 and region inter QP delta for P and B slices
///             2 = vary based on the QP of each individual macroblock
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDAUSAGE      56
/// \brief I slice lambda * 100
/// \brief lambda lambda/100
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_INTRA_ISLICE  57
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_COLINTERP_ISLICE  58
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_INTERP_ISLICE  59
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_INTERB_ISLICE  60
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_SKIP_ISLICE  61
/// \brief P slice lambda * 100
/// \brief lambda lambda/100
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_INTRA_PSLICE  62
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_COLINTERP_PSLICE  63
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_INTERP_PSLICE  64
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_INTERB_PSLICE  65
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_SKIP_PSLICE  66
/// \brief B slice lambda * 100
/// \brief lambda lambda/100
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_INTRA_BSLICE  67
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_COLINTERP_BSLICE  68
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_INTERP_BSLICE  69
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_INTERB_BSLICE  70
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_SKIP_BSLICE  71
/// \brief BR slice lambda * 100
/// \brief lambda lambda/100
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_INTRA_BRSLICE  72
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_COLINTERP_BRSLICE  73
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_INTERP_BRSLICE  74
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_INTERB_BRSLICE  75
/// \brief This parameter is reserved.
/// \internal unused.  what is this?
#define Q_AVCENC_REG_CMP_LAMBDALUT_LAMBDA_SKIP_BRSLICE  76
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_REG_CMP_DZMATRIX_TABLE_ISLICE      77
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_REG_CMP_DZMATRIX_TABLE_PSLICE      78
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_REG_CMP_DZMATRIX_TABLE_BSLICE      79
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_REG_CMP_DZMATRIX_TABLE_BRSLICE     80

// Generated from mediaware.h VENC_DZ_CFG s/VENC_DZ_CFG/Q_AVCENC_DZ_CMP/
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_CORRINTRAYDC                 0
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_CORRINTRAY                   1
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_CORRINTRACAC                 2
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_CORRINTERY                   3
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_CORRINTERCAC                 4
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_CORRINTRACDC                 5
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_CORRINTERCDC                 6
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRAYDCIDX0                 7
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRACDCIDX0                 8
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTERCDCIDX0                 9
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA0IDX0                  10
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA1IDX0                  11
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA2IDX0                  12
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA3IDX0                  13
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA4IDX0                  14
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA5IDX0                  15
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA6IDX0                  16
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA7IDX0                  17
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA8IDX0                  18
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA9IDX0                  19
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER0IDX0                  20
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER1IDX0                  21
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER2IDX0                  22
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER3IDX0                  23
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER4IDX0                  24
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER5IDX0                  25
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER6IDX0                  26
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER7IDX0                  27
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER8IDX0                  28
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER9IDX0                  29
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRAYDCIDX1                30
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRACDCIDX1                31
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTERCDCIDX1                32
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA0IDX1                  33
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA1IDX1                  34
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA2IDX1                  35
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA3IDX1                  36
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA4IDX1                  37
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA5IDX1                  38
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA6IDX1                  39
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA7IDX1                  40
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA8IDX1                  41
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTRA9IDX1                  42
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER0IDX1                  43
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER1IDX1                  44
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER2IDX1                  45
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER3IDX1                  46
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER4IDX1                  47
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER5IDX1                  48
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER6IDX1                  49
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER7IDX1                  50
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER8IDX1                  51
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CMP_INTER9IDX1                  52

// Generated from mediaware.h VENC_DZ_TABLE s/VENC_DZ_TABLE/Q_AVCENC_DZ_CFP_TABLE/
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_LBR                    0
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_MBR                    1
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_HBR                    2
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_QS                     3
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_0                      4
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_1                      5
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_2                      6
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_3                      7
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_4                      8
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_5                      9
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_6                     10
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_7                     11
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_8                     12
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_9                     13
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_10                    14
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_11                    15
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_12                    16
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_13                    17
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_14                    18
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_15                    19
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_16                    20
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_17                    21
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_18                    22
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_19                    23
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_20                    24
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_21                    25
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_22                    26
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_23                    27
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_24                    28
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_25                    29
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_26                    30
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_27                    31
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_28                    32
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_29                    33
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_30                    34
/// \brief This parameter is reserved.
/// \internal
#define Q_AVCENC_DZ_CFP_TABLE_31                    35

// Generated from mediaware.h VRC_CFG s/VRC_CFG/Q_AVCENC_RC_CMP/
/// \brief This parameter sets encoding bitrate.  Normally the bitrate will fall a bit below
///        the target bitrate (95-98%).  By default, the rate control works in a CBR mode.  The
///        difference between the peak bitrate and the average bitrate is controlled by
///        BUFFER_TRANSFER_RATE_BITS.  VBR encoding can be turned on by setting QP_RANGE_MIN and
///        QP_RANGE_MAX.
/// 
/// \param value Integer number 
///
#define Q_AVCENC_RC_CMP_BITRATE                      0
/// \brief This parameter sets the initial QP (compression level) of rate control for the first
///        frame.  The default is selected for medium bitrates.  In order to ensure that the
///        first frame is consistent with the bitrate of the rest of the stream, low bitrate
///        applications may set BASE_QP higher and high bitrate applications may set it lower.
/// 
///        If rate control is disabled, this can be used to set the QP for constant QP encoding.
/// \param value Integer number [0..51] (default 36)
///
#define Q_AVCENC_RC_CMP_BASE_QP                      1
/// \brief Reserved
/// \internal not implemented for 3500
///
#define Q_AVCENC_RC_CMP_TRIAL_FRAMES                 2
/// \brief This parameter enables the motion based QP delta for I picture, controlled by
///        I_DELTA_MOT_THRESHOLD_*.  When motion is low, I frames will be encoded at a lower
///        compression ratio to try and improve overall image quality.
/// \param value 0 = Disable, 1 = Enable (default 1).
///
#define Q_AVCENC_RC_CMP_I_DELTA_VARIABLE             3
/// \brief This parameter sets the minimum motion based QP delta for I picture, overriding
///        some results from the I_DELTA_MOT_THRESHOLD_* table.  This is used to give lower
///        compression to all I frames, regardless of QP.  This can increase quality at
///        medium to high bitrate, with moderate motion.  At low bitrates or with extremely
///        high motion, this does not improve quality.  Note that the parameter is negative
///        because it subtracts from the QP (compression ratio).
/// 
///        I_DELTA_MIN must be greater than or equal to the magnitude of I_DELTA_MAX (remember
///        they are negative numbers).
/// \param value Integer number [0..-10] (default 0)
///
#define Q_AVCENC_RC_CMP_I_DELTA_MIN                  4
/// \brief This parameter sets the maximum motion based QP delta for I picture, overriding
///        some results from the I_DELTA_MOT_THRESHOLD_* table.  This is used to turn off
///        the higher in the table, so that the compression level will not be reduced by
///        too much.  This is useful at low bitrates or with very difficult streams, when
///        it is too risky to encode I frames at very low compression ratios.
/// 
///        I_DELTA_MIN must be greater than or equal to the magnitude of I_DELTA_MAX (remember
///        they are negative numbers).
/// \param value Integer number [0..-10] (default -6)
///
#define Q_AVCENC_RC_CMP_I_DELTA_MAX                  5
/// \brief This parameter sets the QP delta for B picture.  B frame QP (compression level) is based on
///        most recent I/P frame QP plus this delta.  The value is positive, so that B frames are
///        given a higher compression level.  Changing this setting is not recommended.
/// \param value Integer number [0..4] (default 2)
///
#define Q_AVCENC_RC_CMP_B_DELTA_VARIABLE             6
/// \brief Reserved
/// \internal obsolete
///
#define Q_AVCENC_RC_CMP_SIZE_ENABLE                  7
/// \brief Reserved
/// \internal obsolete
///
#define Q_AVCENC_RC_CMP_SIZE_BIT_TOLERANCE           8
/// \brief This parameter enable the rate control buffer constraint model.  Disabling this will
///        shut off the rate control's bitrate changes, leaving only fixed QP changes based on
///        the different I/P/B frame types.
/// \param value 0 = Disable, 1 = Enable (default 1).
///
#define Q_AVCENC_RC_CMP_BUFFER_ENABLE                9
/// \brief This parameter sets maximum buffer size of HRD model.  Normally it should be set to the
///        maximum value allowed by the current level (10,000,000 for level 3, for example).  Use
///        the other rate control parameters to control the bitrate.
/// \param value Integer number
///
#define Q_AVCENC_RC_CMP_BUFFER_SIZE_BITS            10
/// \brief This parameter sets the maximum transfer bitrate of the HRD model.  It controls the peak
///        bitrate, the maximum the rate control can generate.  Peak bitrate should be lower than
///        the average bitrate.  Setting peak == average is strict CBR mode, which is not recommended
///        because it usually produces frame drops.  Normal setting is TRANSFER_RATE_BITS > 2 * BITRATE.
///        For storage applications, set the transfer rate to be high.  For streaming applications it
///        may need to be reduced due to low network bandwidth.  However, if the transfer rate is close
///        to the bitrate, frame drops may occur.
/// \param value Integer number
///
#define Q_AVCENC_RC_CMP_BUFFER_TRANSFER_RATE_BITS   11
/// \brief This parameter sets the initial delay of HRD model.  This controls both
///        the required latency of the decoder and the stability of the bitrate.  A low
///        initial delay will produce more bitrate variation and a higher chance of frame
///        drops.  An initial delay of under 3 frame times (3 * 3003 in NTSC) is not recommended.
///        An initial delay of over 2 seconds is not recommended because that much time is not useful.
/// \param value Integer number in ticks
///
#define Q_AVCENC_RC_CMP_BUFFER_INITIAL_DELAY        12
/// \brief This parameter enables the low delay mode of HRD model.  This influences how frame drops
///        are dealt with.  If there are any frame drops, low delay mode will typically result in higher
///        quality video and less frame drops.  If there are no frame drops, the setting has no effect.
///        Low delay mode is superior,  and should be used unless strict compliance with the AVC CBR
///        specification is required.
/// \param value 0 = Disable, 1 = Enable.
///
#define Q_AVCENC_RC_CMP_BUFFER_LOW_DELAY_MODE       13
/// \brief This parameter is used for strict CBR compliance.  Normally the rate control bitrate is
///        95-98% of the target bitrate.  This parameter increases the bitrate to 100% of the target
///        by padding the bitstream with zero bytes.  Strictly compliant CBR assumes that the BITRATE
///        and TRANSFER_RATE_BITS settings are identical.  Therefore, zero padding is used to match
///        the TRANSFER_RATE_BITS setting.  This should be used only when a standard or specification
///        requires a completely precise bitrate, because it adds padding to the stream and increases
///        the chance of frame drops.
/// 
///        When BUFFER_CBR_ZERO_PADDING is enabled, the cbr_flag in the AVC VUI information is also set.
/// \param value 0 = Disable, 1 = Enable (default 0).
///
#define Q_AVCENC_RC_CMP_BUFFER_CBR_ZERO_PADDING     14
/// \brief Not used.
/// \internal obsolete
///
#define Q_AVCENC_RC_CMP_BUFFER_MAX_SNR              15
/// \brief This parameter is used for VBR encoding.  Normally the rate control is free to choose any
///        compression level (QP) between 0 and 51, where 51 is the most compression.  The rate
///        control uses very little compression on easy content, in order to hit the target bitrate.
///        When a minimum compression level (QP) is set, the rate control will not use lower compression
///        on very easy content and the bitrate will drop (VBR).
/// 
///        QP_RANGE_MIN must be less than QP_RANGE_MAX.
/// 
///        To avoid high bitrates on very simple content (such as black screens), it is reasonable to
///        set QP_RANGE_MIN to about 13.  Setting it higher depends on the overall bitrate and quality.
/// \param value [0..51] (default 0).
///
#define Q_AVCENC_RC_CMP_QP_RANGE_MIN                16
/// \brief This parameter is used for VBR encoding.  Normally the rate control is free to choose any
///        compression level (QP) between 0 and 51, where 51 is the most compression.  The rate
///        control uses a lot of compression on difficult content, in order to hit the target bitrate.
///        When a maximum compression level (QP) is set, the rate control will not use higher compression
///        on very hard content and the bitrate will increase (VBR).  This can be useful to prevent short
///        periods of very low quality encoding.
/// 
///        WARNING: Setting QP_RANGE_MAX to a value other than the default (51) can cause the bitrate to
///        exceed the target bitrate for long periods.  It can cause the rate control to violate the AVC
///        HRD model, which does not matter for most decoders but will cause stream analyzers to detect
///        an invalid bitstream.
/// 
///        QP_RANGE_MAX must be greater than QP_RANGE_MIN.
/// 
///        To avoid high bitrates on very simple content (such as black screens), it is reasonable to
///        set QP_RANGE_MIN to about 13.  Setting it higher depends on the overall bitrate and quality.
/// \param value [0..51] (default 51)
///
#define Q_AVCENC_RC_CMP_QP_RANGE_MAX                17
/// \brief This parameter sets Motion to I frame delta LUT 1..10.  Recommended for advanced use only.
///        Requires I_DELTA_VARIABLE to be set.
/// 
///        Each video frame is assigned a motion value from 0 to 255.  The rate control can apply QP
///        deltas (increases in compression quality) from 0 to 10 to an I frame.  This is useful to
///        reduce the compression of I frames on low motion scenes, which can increase quality.  A
///        lookup table controls this.  For entry N, the I frame compression ratio will be reduced by
///        N if the motion is LESS THAN entry N.
/// \param value motion value required to produce a QP delta of N [0..255]
///
#define Q_AVCENC_RC_CMP_I_DELTA_MOT_THRESHOLD_1     18
#define Q_AVCENC_RC_CMP_I_DELTA_MOT_THRESHOLD_2     19
#define Q_AVCENC_RC_CMP_I_DELTA_MOT_THRESHOLD_3     20
#define Q_AVCENC_RC_CMP_I_DELTA_MOT_THRESHOLD_4     21
#define Q_AVCENC_RC_CMP_I_DELTA_MOT_THRESHOLD_5     22
#define Q_AVCENC_RC_CMP_I_DELTA_MOT_THRESHOLD_6     23
#define Q_AVCENC_RC_CMP_I_DELTA_MOT_THRESHOLD_7     24
#define Q_AVCENC_RC_CMP_I_DELTA_MOT_THRESHOLD_8     25
#define Q_AVCENC_RC_CMP_I_DELTA_MOT_THRESHOLD_9     26
#define Q_AVCENC_RC_CMP_I_DELTA_MOT_THRESHOLD_10    27
/// \brief This parameter enables I frame delta scaling based on base QP.  Enabling is recommended.
///        For low QP frames (very low compression), it reduces the compression changes that happen
///        when I_DELTA_VARIABLE is set.  This makes sure that changes between very low compression
///        ratios do not cause very large changes in the size of I frames.
/// \param value 0 = Disable, 1 = Enable
///
#define Q_AVCENC_RC_CMP_I_DELTA_SCALE_ENABLE        28
/// \brief This parameter enables adaptive framerate adjustment.  Both the framerate and QP
///        will be changed to hit the target bitrate.  AFR_MAX_QP specifies a maximum compression
///        level.  Once the maximum compression level is hit, bitrate will be maintained only by
///        lowering the framerate.  Framerate will only be lowered from the base framerate, never 
///        raised above it.
/// 
///        Adaprive framerate is NOT supported when encoding with B frames.
/// \param value 0 = Disable, 1 = Enable (default 0).
///
#define Q_AVCENC_RC_CMP_ADAPTIVE_FRAMERATE_ENABLE   29
/// \brief This parameter sets the minimum QP of adaptive framerate adjustment.  When the framerate
///        has been lowered, this controls when it should be raised again.  When framerate is reduced
///        and the QP drops below AFR_MAX_QP, the framerate can be raised but raising it too quickly
///        can cause unpredictable changes.  Framerate will be raised when the QP is at least AFR_MIN_QP.
///        AFR_MIN_QP must be less than AFR_MAX_QP.  One good choice is AFR_MIN_QP = AFR_MAX_QP - 4.
/// \param value Integer [0..51] (default 0)
///
#define Q_AVCENC_RC_CMP_AFR_MIN_QP                  30
///
#define Q_AVCENC_RC_CMP_AFR_MAX_QP                  31
/// \brief This parameter, along with SCALING_MIN_NUMERATOR, sets the range of framerates that can be
///        selected by the adaptive framerate algorithm.  For a denominator of N, the framerate can be
///        lowered in increments of 1/N to a minimum of 1/N.  Example: N = 30.  Framerate can be lowered
///        to half (15/30), then slightly more (14/30), all the way down to a 30x slowdown (1/30).
/// \param value Integer number [1..255]
///
#define Q_AVCENC_RC_CMP_AFR_SCALING_DENOMINATOR     32
/// \brief This parameter sets the minimum framerate produced by AFR.  Based on the scaling
///        denominator N, the framerate is multiplied by a fraction from N/N to (min numerator)/N.
///        Example: denominator = 30, min numerator = 5.  Minimum framerate is 5/30 of the base
///        framerate.
/// \param value Integer number [1..AFR_SCALING_DENOMINATOR]
///
#define Q_AVCENC_RC_CMP_AFR_SCALING_MIN_NUMERATOR   33
/// \brief This parameter enables the complexity based rate control.  This is an adjustment
///        intended to deal with very rapid changes in content, such as scene changes.  If the
///        image statistics of the next frame to encode are very different from those of the
///        current frame, the normal rate control decision is overidden to make a more rapid
///        change in the compression level.
/// \param value 0 = Disable, 1 = Enable (default 1).
///
#define Q_AVCENC_RC_CMP_COMPLEXITY_ENABLE           34
/// \brief This parameter enables special handling for static scenes.  This is an adjustment
///        intended to deal with motionless scenes, keeping the bitrate below target (VBR) on
///        those scenes in order to save bits and avoid a rapid bitrate change when there is motion again.
/// \param value 0 = Disable, 1 = Enable (default 1).
///
#define Q_AVCENC_RC_CMP_STATIC_DETECT_ENABLE        35
/// \brief This parameter sets QP delta variable applied during fades.
/// \param value Integer number (default 0)
///
#define Q_AVCENC_RC_CMP_FADE_DELTA_VARIABLE         36
/// \brief This parameter enables fade QP delta variable applied during fades and bypass rate control
/// \param value 0 = Disable, 1 = Enable (default 0).
///
#define Q_AVCENC_RC_CMP_FADE_DETECT_ENABLE          37
/// \brief This parameter enables edge based rate control
/// \param value 0 = Disable, 1 = Enable (default 0).
///
#define Q_AVCENC_RC_CMP_EDGERC_ENABLE               38

// Generated from mediaware.h VENC_PRIV_REGIONS_CFG s/VENC_PRIV_REGIONS_CFG/Q_AVCENC_CMP_PREGION/
/// \brief Enable/disable the privacy region to draw a black rectangle over the image using the rectangle
///        coordinates for that region.
/// \param value 0 = Disable, 1 = Enable
#define Q_AVCENC_CMP_PREGION_ENABLE                  0
/// \brief X offset of the rectangle, in macroblocks
/// \param X X offset
#define Q_AVCENC_CMP_PREGION_MB_XPOS                 1
/// \brief Y offset of the rectangle, in macroblocks
/// \param Y Y offset
#define Q_AVCENC_CMP_PREGION_MB_YPOS                 2
/// \brief Width of the rectangle, in macroblocks
/// \param value Width of the rectangle, in macroblocks
#define Q_AVCENC_CMP_PREGION_MB_WIDTH                3
/// \brief Height of the rectangle, in macroblocks
/// \param value Height of the rectangle, in macroblocks
#define Q_AVCENC_CMP_PREGION_MB_HEIGHT               4

/// \brief This parameter is reserved.
#define Q_AVCENC_QS_CFP_TABLE_INTRA_4X4_Y               0
/// \brief This parameter is reserved.
#define Q_AVCENC_QS_CFP_TABLE_INTRA_4X4_CB              1
/// \brief This parameter is reserved.
#define Q_AVCENC_QS_CFP_TABLE_INTRA_4X4_CR              2
/// \brief This parameter is reserved.
#define Q_AVCENC_QS_CFP_TABLE_INTER_4X4_Y               3
/// \brief This parameter is reserved.
#define Q_AVCENC_QS_CFP_TABLE_INTER_4X4_CB              4
/// \brief This parameter is reserved.
#define Q_AVCENC_QS_CFP_TABLE_INTER_4X4_CR              5
/// \brief This parameter is reserved.
#define Q_AVCENC_QS_CFP_TABLE_INTRA_8X8_Y               6
/// \brief This parameter is reserved.
#define Q_AVCENC_QS_CFP_TABLE_INTER_8X8_Y               7

// Generated from mediaware.h VENC_BURNIN_CFG s/VENC_BURNIN_CFG/Q_AVCENC_CMP_BURNIN/
/// \brief This parameter enables text burn in.
/// \param value 0 = Disable, 1 = Enable.
///
#define Q_AVCENC_CMP_BURNIN_ENABLE                   0
/// \brief This parameter sets the font used for burn in.
/// \param value Codec byte address of loaded font.
///
#define Q_AVCENC_CMP_BURNIN_FONT_ADDR                1
/// \brief This parameter sets the font size for burn in.
/// \param value 8, 16 or 32.
///
#define Q_AVCENC_CMP_BURNIN_FONT_SIZE                2
/// \brief This parameter sets the burn in update mode.
/// \param value VENC_BURNIN_UPDATE_MODE_*
///
#define Q_AVCENC_CMP_BURNIN_UPD_MODE                 3
/// \brief This parameter sets the burn in x position.
/// \param value Positive integer.
///
#define Q_AVCENC_CMP_BURNIN_X                        4
/// \brief This parameter sets the burn in y position.
/// \param value Positive integer.
///
#define Q_AVCENC_CMP_BURNIN_Y                        5
/// \brief This parameter enables frame numbers to be inserted in the burn in text.
/// \param value 0 = Disable, 1 = Enable.
///
#define Q_AVCENC_CMP_BURNIN_FRAME_NUM_ENABLE         6
/// \brief This parameter sets the position in the text that the frame number is inserted at.
/// \param value Positive integer.
///
#define Q_AVCENC_CMP_BURNIN_FRAME_NUM_POS            7
/// \brief This parameter sets the frame number minimum value.
/// \param value Positive integer.
///
#define Q_AVCENC_CMP_BURNIN_FRAME_NUM_L              8
/// \brief This parameter sets the frame number maximum value.
/// \param value Positive integer.
///
#define Q_AVCENC_CMP_BURNIN_FRAME_NUM_H              9
/// \brief This parameter sets the length of the burn in text.
/// \param value Positive integer.
///
#define Q_AVCENC_CMP_BURNIN_STR_LEN                 10
/// \brief This parameter sets the characters 0 - 3 of the burn in text.
/// \param value Positive integer representing characters.
///
#define Q_AVCENC_CMP_BURNIN_STR4_0                  11
/// \brief This parameter sets the characters 4 - 7 of the burn in text.
/// \param value Positive integer representing characters.
///
#define Q_AVCENC_CMP_BURNIN_STR4_1                  12
/// \brief This parameter sets the characters 8 - 11 of the burn in text.
/// \param value Positive integer representing characters.
///
#define Q_AVCENC_CMP_BURNIN_STR4_2                  13
/// \brief This parameter sets the characters 12 - 15 of the burn in text.
/// \param value Positive integer representing characters.
///
#define Q_AVCENC_CMP_BURNIN_STR4_3                  14
/// \brief This parameter sets the characters 16 - 19 of the burn in text.
/// \param value Positive integer representing characters.
///
#define Q_AVCENC_CMP_BURNIN_STR4_4                  15
/// \brief This parameter sets the characters 20 - 24 of the burn in text.
/// \param value Positive integer representing characters.
///
#define Q_AVCENC_CMP_BURNIN_STR4_5                  16

// Generated from mediaware.h VENC_ROI_CFG s/VENC_ROI_CFG/Q_AVCENC_CMP_ROI/
/// \brief This flag controls the enabling/disabling of the motion detection in the codec. 
/// \param value 0 = Disable, 1 = Enable.
#define Q_AVCENC_CMP_ROI_ENABLE                      0
/// \brief This parameter controls sets the threshold for motion detection for a macroblock. A macroblock is in motion if the average macroblock motion for the 
/// macroblock is greater than the threshold. The minimum and maximum values for this threshold is 0 and 255 respectively.
#define Q_AVCENC_CMP_ROI_MOTION_THRESHOLD            1
/// \brief This parameter controls ther percentage of macroblocks that must be in motion in a region of interest to trigger the alarm. The scale of the
/// value is from 0 to 10000 and hence the percentage value should be multiplied by 100 when configuring this parameter. eg. 58 percent should be written as 5800.
#define Q_AVCENC_CMP_ROI_PERCENT                     2

// Generated from mediaware.h VENC_CFG_VA s/VENC_CFG/Q_AVCENC_CMP/
/// \brief This parameter is used to enable Object Tracking.
/// \param value 0:Disable 1:Enable.
#define Q_AVCENC_CMP_VA_OT_ENABLE                    0
/// \brief This parameter controls the minimum luminance level difference between current frame and background 
///  frame required to classify an object. If absolute luminance difference for a macroblock is below this threshold, 
///  objects will not be detected in this area.
/// \param value is [0-255].
#define Q_AVCENC_CMP_VA_OT_BG_DIFF_THRESHOLD         1
/// \brief This parameter controls the minimum luminance level difference between current frame and background 
///  frame required to classify an object. If absolute luminance difference for a macroblock is below this threshold, 
///  objects will not be detected in this area.
/// \param value is [0-255].
#define Q_AVCENC_CMP_VA_OT_BG_DIFF_THRESHOLD_MIN     2
/// \brief This parameter controls the min allowable value for VENC_CFG_VA_OT_BG_DIFF_THRESHOLD
/// \param value is [0-255].
#define Q_AVCENC_CMP_VA_OT_BG_DIFF_THRESHOLD_MAX     3
/// \brief This parameter controls the max allowable value for VENC_CFG_VA_OT_BG_DIFF_THRESHOLD
/// \param value is [0-255].
#define Q_AVCENC_CMP_VA_OT_LUMAMOT_THRESHOLD         4
/// \brief This parameter controls the minimum horizontal size of the object that should be detected. 
/// \param value [0-width]
#define Q_AVCENC_CMP_VA_OT_MINISIZE_X                5
/// \brief This parameter controls the minimum vertical size of the object that should be detected. 
/// \param value [0-height]
#define Q_AVCENC_CMP_VA_OT_MINISIZE_Y                6
///  This parameter is applicable only if the motion vector processing is enabled for object tracking.
/// \param value 0:Disable 1:Enable.
#define Q_AVCENC_CMP_VA_OT_MEDIAN_FILTER_ENABLE      7
/// \brief This parameter enables the use of motion vectors to cluster objects together. If processing of
///  motion vectors is turned off, the motion vectors reported in the form of events will be invalid.
/// \param value 0:Disable 1:Enable.
#define Q_AVCENC_CMP_VA_OT_MOTION_VECTOR_ENABLE      8
/// \brief This parameter enables the computation of motion vectors on the downscaled image. For HD images
///  this value should be set to 1.
/// \param value 0:Disable 1:Enable.
#define Q_AVCENC_CMP_VA_OT_USE_DOWNSCALED_MOTION_VECTOR   9
/// \brief If the average motion of the macroblock is less than this threshold the macroblock is considered
///  to have no motion. Used to figure out number of MB's in a frame that are static. 
#define Q_AVCENC_CMP_VA_OT_STATIC_MB_THRESH         10
/// \brief If the number of consecutive static frames exceeds this threshold the background is forced to be the 
///  current frame or the MB.
#define Q_AVCENC_CMP_VA_OT_STATIC_NUMFRAMES_THRESH  11
/// \brief The time constant for blending current frame and the background frame. 
#define Q_AVCENC_CMP_VA_OT_BG_UPDATE_TIMECONSTANT   12
/// \brief The time constant for blending current frame and the background frame. 
#define Q_AVCENC_CMP_VA_OT_BG_UPDATE_TIME           13
/// \brief This parameter controls the minimum number of frames a MB should be in existence before it could be
///  classified as a part of an object.
#define Q_AVCENC_CMP_VA_OT_APPEARANCETIME_THRESH    14
/// \brief The generation of contour maps in the events will be enabled. 
#define Q_AVCENC_CMP_VA_OT_CONTOUR_GENERATION_ENABLE  15
/// \brief The tracking of blobs is controlled by this flag.
#define Q_AVCENC_CMP_VA_OT_BLOBTRACKER_ENABLE       16
/// \brief If this flag is set the first frame captured is forced to be the background frame.
#define Q_AVCENC_CMP_VA_OT_BG_FORCE_FIRST_FRAME     17
/// \brief This parameter is used to scale the threshBGDiff based on the luminance of the current MB. It is represented as a multiple of 256.
///  The minimum value is used to scale the threshold for a MB having luminance 0.
#define Q_AVCENC_CMP_VA_OT_BG_DIFF_THRESHOLD_MULTFACT_MIN  18
/// \brief This parameter is used to scale the threshBGDiff based on the luminance of the current MB. It is represented as a multiple of 256.
///  The minimum value is used to scale the threshold for a MB having luminance 255.
#define Q_AVCENC_CMP_VA_OT_BG_DIFF_THRESHOLD_MULTFACT_MAX  19
/// \brief Factor by which to run object tracking at a lower frame rate
/// than the video frame rate.    
/// \param value 1: run every frame (30fps object-tracking on 30fps video)
///              2: run every 2nd frame
///              3: run every 3rd frame  
///              4: run every 4th frame (7.5fps object-tracking on 30fps video)
///              values larger than 4 are possible but not recommended.
#define Q_AVCENC_CMP_VA_OT_DOWN_SAMPLE_FACTOR       20
/// \brief Allows control over which frame object tracking is computed
/// in the down-sampled case.  For example a down-sample-factor of 4
/// causes object tracking to run every 4 frames. Choosing an offset of
/// 0 will enable object tracking on frames 0, 4, 8 and so on; while an
/// offset of 3 will enable object tracking on frame 3, 7, 11 and so
/// on.  This is useful for load balancing in the multichannel case.
/// \param values [0, (VENC_CFG_VA_OT_DOWN_SAMPLE_FACTOR - 1)]
#define Q_AVCENC_CMP_VA_OT_DOWN_SAMPLE_OFFSET       21


//////////////////////////////////////////////////////////////////////
//
// Configuration parameter values
//
// Generated from mediaware.h VENC_GOP_STRUCTURE s/VENC_GOP/Q_AVCENC_CFP_GOP_STRUCTURE/
#define Q_AVCENC_CFP_GOP_STRUCTURE_I                 0
#define Q_AVCENC_CFP_GOP_STRUCTURE_IP                1
#define Q_AVCENC_CFP_GOP_STRUCTURE_IBP               2
#define Q_AVCENC_CFP_GOP_STRUCTURE_IBBP              3
#define Q_AVCENC_CFP_GOP_STRUCTURE_IBBRBP            4
#define Q_AVCENC_CFP_GOP_STRUCTURE_IPPPR             5
#define Q_AVCENC_CFP_GOP_STRUCTURE_IPX               6

// Generated from mediaware.h VENC_BURNIN_UPDATE_MODE s/VENC_BURNIN_UPDATE_MODE/Q_AVCENC_CFP_BURNIN_UPDATE_MODE/
/// \brief update on next frame,
#define Q_AVCENC_CFP_BURNIN_UPDATE_MODE_NOW          0
/// \brief update on frame number rollover.
#define Q_AVCENC_CFP_BURNIN_UPDATE_MODE_ROLLOVER     1

/// \}
#endif
