#!/bin/bash
# mkrootfs.sh
# Copyright 2006 Mobilygen Corp.

# This script can take a directory, a device list and a filesystem
# and generate an image out of those. It uses fakeroot that is assumed
# to be in the path.

PROG=$0;

function help ()
{	echo -e "$(basename ${PROG}): generate a binary image of a filesystem from a";
	echo -e "directory and a device nodes list.";
	echo -e "Syntax: $(basename ${PROG}) <rootfs_dir> <rootfs_bin> \\";
	echo -e "\t--fs | -t <fs_type> [ --devtable | -d <dev_file> ] \\";
	echo -e "\t[ --chmod <dir|file>=<user>.<group> [ --chmod ... ] ]\\";
	echo -e "\t[ --size | -s <size> ] [ --inplace ] [ --quiet ]";
	echo -e "\t[ --keeptmp ] -- <extra_args_mkfs>";
	echo;
	echo -e "<rootfs_dir>: directory that contains the file to copy into the image.";
	echo -e "<rootfs_bin>: name of the resulting image.";
	echo -e "--fs | -t <fs_type>: specify the file-system that should be used.";
	echo -e "--devtable | -d <dev_file>: this file should contain a list of devices node";
	echo -e "\tthat must be on the image, the format is the same as mkfs.jffs2.";
	echo -e "--chmod <dir|file>=<user>.<group>: tell the script to change <dir|file>";
	echo -e "\townership to <user>.<group>.";
	echo -e "Note that user must exist in <rootfs_dir>/etc/passwd and group must";
	echo -e "\texist in <rootfs_dir>/etc/group.";
	echo -e "Also note that if not forced with --chmod a file will be own by root";
	echo -e "\t(group: root).";
	echo -e "--size | -s <size>: maximum size of the image, it will be padded to that";
	echo -e "\tsize if possible but if the image is too big the script will return";
	echo -e "\tan error."
	echo -e "--fakeroot <fakeroot path> : override the use of the internal fakeroot and"
	echo -e "\tuse the one installed on the local machine instead"
	echo -e "\t<size> can be in kB if it end with k, MB if it ends with M, otherwise";
	echo -e "\tit is in B.";
	echo -e "--tmp <dir>: use <dir> as the temporary directory, otherwise use";
	echo -e "\t/tmp/$(basename ${PROG}).xxxx where xxxx is a random number.";
	echo -e "--quiet: do not display any message except errors.";
	echo -e "--inplace: do not copy <rootfs_dir>, but do modification directly in it.";
	echo -e "--keeptmp: do not remove the temp directory at the end.";
	echo -e "-- <extra_args_mkfs>: list of options to send directly to the mkfs.<fs>.";
	echo -e "\tThose are specific to each filesystem.";
	echo;
	return 0;
}	

function clean_and_exit()
{	local RET_CODE=$1
	if [ -n "${TMP_DIR}" ]; then
		if [ -d ${TMP_DIR} ] && [ $((KEEPTMP)) -ne 1 ]; then
			${PRINT} -n "Cleaning temporary directory ... ";
			rm -r ${TMP_DIR};
			if [ $? -ne 0 ]; then 
				${PRINT} "failed.";
				echo "WARNING: could not remove temp directory ${TMP_DIR}, please do it manually.";
			else
				${PRINT} "done.";
			fi
		fi
	fi
	if [ $((VERBOSE)) -ne 0 ]; then echo; fi
	exit ${RET_CODE}
}

ROOTFS_DIR=;
ROOTFS_IMAGE=;
FS_TYPE=;
DEV_FILE=;
CHMOD_LIST="";
FS_SIZE=0;
MKFS_EXTRAARGS="";
INPLACE=0;
FAKEROOT=fakeroot
TMP_DIR="/tmp/$(basename ${PROG}).$$";
PRINT="echo -e";
KEEPTMP=0;
while [ -n "$1" ]; do
	case "$1" in
		"--fs" | "-t") FS_TYPE=$2; shift 2;;
		"--devtable" | "-d") 
			if [ ! -r $2 ]; then
				echo "ERROR: device file $2 is not readable or do not exist.";
				clean_and_exit 1;
			fi
			DEV_FILE=$2; 
			shift 2;;
		"--chmod") CHMOD_LIST="${CHMOD_LIST} $2"; shift 2;;
		"--size" | "-s")
			case ${2:${#2}-1:1} in
				"k") FS_SIZE=$(( ${2:0:${#2}-1} * 1024 ));;
				"M") FS_SIZE=$(( ${2:0:${#2}-1} * 1024 * 1024 ));;
				*) FS_SIZE=$(( $2 ));;
			esac
			shift 2;;
		"--fakeroot") FAKEROOT=$2; shift 2;;
		"--tmp") TMP_DIR=$2; shift 2;;
		"--inplace") INPLACE=1; shift 1;;
		"--quiet") PRINT="false"; shift 1;;
		"--keeptmp") KEEPTMP=1; shift 1;;
		"--") shift 2; MKFS_EXTRAARGS=$*; break;;
		"--help" | "-h") help; clean_and_exit 0;;
		*)
			if [ -z "${ROOTFS_DIR}" ]; then
				if [ ! -d $1 ]; then
					echo "ERROR: $1 is not a valid directory for <rootfs_dir>.";
					clean_and_exit 1;
				fi
				ROOTFS_DIR=$1;
			elif [ -z "${ROOTFS_IMAGE}" ]; then
				ROOTFS_IMAGE=$1;
			else
				echo "ERROR: invalid argument $1, try --help.";
				clean_and_exit 1;
			fi
			shift 1;;
	esac
done

# Prepare the temp directory and make sure everything is properly setup
if [ -z "${ROOTFS_DIR}" ]; then
	echo "ERROR: missing argument, the image name <rootfs_bin> must be specified.";
	clean_and_exit 1;
elif [ -z "${ROOTFS_IMAGE}" ]; then
	echo "ERROR: missing argument, the root filesystem content <rootfs_dir> must be specified.";
	clean_and_exit 1;
elif [ -z "${FS_TYPE}" ]; then
	echo "ERROR: missing argument, the file system type <fs_type> must be specified.";
	clean_and_exit 1;
elif [ -z "${TMP_DIR}" ]; then
	echo "ERROR: wrong --tmp argument.";
	clean_and_exit 1;
fi

if [ -d "${TMP_DIR}" ]; then
	rm -r ${TMP_DIR};
	if [ $? -ne 0 ]; then
		echo "ERROR: Could not clean the temp directory, please specidy a new temp directory with --tmp.";
		clean_and_exit 10;
	fi
fi
mkdir -p ${TMP_DIR};
if [ $? -ne 0 ]; then
	echo "ERROR: Could not create the temp directory ${TMP_DIR},use --tmp to specify an alternate directory.";
	clean_and_exit 11;
fi
if [ $((INPLACE)) -ne 0 ]; then
	ROOTFS_WORKDIR=${ROOTFS_DIR}
else
	ROOTFS_WORKDIR=${TMP_DIR}/rootfs
fi
ROOTFS_SCRIPT=${TMP_DIR}/fakeroot.sh
mkdir -p ${ROOTFS_WORKDIR};
if [ $? -ne 0 ]; then
	echo "ERROR: Could not create the temp directory ${ROOTFS_WORKDIR},use --tmp to specify an alternate directory.";
	clean_and_exit 12;
fi
touch ${ROOTFS_SCRIPT};
if [ $? -ne 0 ]; then
	echo "ERROR: Could not create the temp script for fakeroot, use --tmp to specify an alternate directory.";
	clean_and_exit 13;
fi
chmod +x ${ROOTFS_SCRIPT};
if [ $? -ne 0 ]; then
	echo "ERROR: Could not set exec flag on the temp script for fakeroot.";
	clean_and_exit 13;
fi

# Check the directory with files for the root filesystem contains key elements
${PRINT} -n "Check filesystem type ... ";
# The image will be transfered to fakeroot.sh => we must use an absolute
# path because it will be outside of the fake root environment
if [ "X${ROOTFS_IMAGE:0:1}" != "X/" ]; then
	ROOTFS_IMAGE="$(pwd)/${ROOTFS_IMAGE}";
fi
case "${FS_TYPE}" in
	"jffs2")
		MKFS="mkfs.jffs2 --output=${ROOTFS_IMAGE} --root=${ROOTFS_WORKDIR} ${MKFS_EXTRAARGS}";
		if [ $((FS_SIZE)) -ne 0 ]; then
			MKFS="${MKFS} --pad=$((SIZE))";
		fi
		;;
	"cramfs")
		MKFS="mkfs.cramfs ${ROOTFS_WORKDIR} ${ROOTFS_IMAGE} ${MKFS_EXTRAARGS}";
		;;
	"ext2")
		MKFS="genext2fs --root ${ROOTFS_WORKDIR} ${ROOTFS_IMAGE} ${MKFS_EXTRAARGS}";
		if [ $((FS_SIZE)) -ne 0 ]; then
			# Leave 1MB for the inodes (not very accurate but will have to do for now)
			MKFS="${MKFS} --size-in-blocks $(($(($((FS_SIZE+1023))/1024)) - 1024))";
		fi
		;;
	"ramfs")
		# Warning: the following lines change the path in fakeroot !
		MKFS="cd ${ROOTFS_WORKDIR}; find . | cpio ${MKFS_EXTRAARGS} -o -H newc | gzip > ${ROOTFS_IMAGE}";
		;;
	"tar")
		# We do a bit of auto-detection to figure out which flavor is requested
		ROOTFS_IMAGE_EXT=$(basename ${ROOTFS_IMAGE} | sed -e 's/.*\.//g');
		case ${ROOTFS_IMAGE_EXT} in
			"bz2") MKFS="tar -jcf ${ROOTFS_IMAGE} -C ${ROOTFS_WORKDIR} ${MKFS_EXTRAARGS} .";;
			"gz" | "tgz") MKFS="tar -zcf ${ROOTFS_IMAGE} -C ${ROOTFS_WORKDIR} ${MKFS_EXTRAARGS} .";;
			*) MKFS="tar -cf ${ROOTFS_IMAGE} -C ${ROOTFS_WORKDIR} ${MKFS_EXTRAARGS} .";;
		esac
		;;
	*)
		${PRINT} "failed.";
		echo "ERROR: filesystem ${FS_TYPE} is not supported by this script.";
		clean_and_exit 15;
		;;
esac
${PRINT} "done.";

# Generate a script for fakeroot
${PRINT} -n "Parsing device file ... ";
echo '#!/bin/bash' >> ${ROOTFS_SCRIPT}
if [ -n "${DEV_FILE}" ]; then
	while read DEV_FILE DEV_TYPE DEV_MODE DEV_UID DEV_GID DEV_MAJOR DEV_MINOR DEV_START DEV_INC DEV_COUNT; do
		if [ -z "${DEV_FILE}" ] || [ "X${DEV_FILE:0:1}" == "X#" ]; then
			continue;
		elif [ -z "${DEV_COUNT}" ]; then
			${PRINT} "failed.";
			echo "ERROR: ${DEV_FILE} is not in the proper format, fields are missing.";
			clean_and_exit 20;
		fi
		# Remove any comment that may have been at the end
		DEV_COUNT=$(echo ${DEV_COUNT} | sed -e 's/[ ]*#.*//');
		if [ -e "${ROOTFS_DIR}/${DEV_FILE}" ]; then
			if ( ( [ "${DEV_TYPE}" == "f" ] && [ ! -f "${ROOTFS_DIR}/${DEV_FILE}" ] ) \
				|| ( [ "${DEV_TYPE}" == "d" ] && [ ! -d "${ROOTFS_DIR}/${DEV_FILE}" ] ) \
				|| ( [ "${DEV_TYPE}" == "b" ] && [ ! -b "${ROOTFS_DIR}/${DEV_FILE}" ] ) \
				|| ( [ "${DEV_TYPE}" == "c" ] && [ ! -c "${ROOTFS_DIR}/${DEV_FILE}" ] ) \
				|| ( [ "${DEV_TYPE}" == "p" ] && [ ! -p "${ROOTFS_DIR}/${DEV_FILE}" ] ) \
			); then
				${PRINT} "failed.";
				echo "ERROR: ${DEV_FILE} already exist in the root filesystem and is not of type ${DEV_TYPE}.";
				clean_and_exit 21;
			fi
		fi
		if [ "${DEV_TYPE}" != "f" ] && [ "${DEV_TYPE}" != "b" ] && [ "${DEV_TYPE}" != "c" ] && [ "${DEV_TYPE}" != "d" ] && [ "${DEV_TYPE}" != "p" ]; then
			${PRINT} "failed.";
			echo "ERROR: ${DEV_FILE} is of unknown type ${DEV_TYPE}.";
			clean_and_exit 22;
		fi
		if [ "X${DEV_COUNT}" == "X-" ]; then DEV_COUNT=0; fi
		if [ $((DEV_COUNT)) -ne 0 ]; then
			count=0;
			while [ $((count)) -lt $((DEV_COUNT)) ]; do
				ext=$(( ${DEV_START} + ${DEV_INC} * ${count} ));
				case "${DEV_TYPE}" in
					"c" | "b" | "p") echo "mknod -m ${DEV_MODE} ${ROOTFS_WORKDIR}/${DEV_FILE}${ext} ${DEV_TYPE} ${DEV_MAJOR} $(( ${DEV_MINOR} + ${count} ))" >> ${ROOTFS_SCRIPT};;
					"d") echo "mkdir -p ${ROOTFS_WORKDIR}/${DEV_FILE}${ext}" >> ${ROOTFS_SCRIPT};;
					"f") echo "touch ${ROOTFS_WORKDIR}/${DEV_FILE}${ext}" >> ${ROOTFS_SCRIPT};;
				esac
				echo 'if [ $? -ne 0 ]; then exit 1; fi' >> ${ROOTFS_SCRIPT};
				echo "chown --no-dereference ${DEV_UID}:${DEV_GID} ${ROOTFS_WORKDIR}/${DEV_FILE}${ext}" >> ${ROOTFS_SCRIPT};
				echo 'if [ $? -ne 0 ]; then exit 1; fi' >> ${ROOTFS_SCRIPT};
				count=$(( ${count} + 1 ));
			done
		else
			case "${DEV_TYPE}" in
				"c" | "b" | "p") echo "mknod -m ${DEV_MODE} ${ROOTFS_WORKDIR}/${DEV_FILE} ${DEV_TYPE} ${DEV_MAJOR} ${DEV_MINOR}" >> ${ROOTFS_SCRIPT};;
				"d") echo "mkdir -p ${ROOTFS_WORKDIR}/${DEV_FILE}" >> ${ROOTFS_SCRIPT};;
				"f") echo "touch ${ROOTFS_WORKDIR}/${DEV_FILE}" >> ${ROOTFS_SCRIPT};;
			esac
			echo 'if [ $? -ne 0 ]; then exit 1; fi' >> ${ROOTFS_SCRIPT};
			echo "chown --no-dereference ${DEV_UID}:${DEV_GID} ${ROOTFS_WORKDIR}/${DEV_FILE}" >> ${ROOTFS_SCRIPT};
			echo 'if [ $? -ne 0 ]; then exit 1; fi' >> ${ROOTFS_SCRIPT};
		fi
	done < ${DEV_FILE};
fi
${PRINT} "done.";

${PRINT} -n "Parsing chmod list ... ";
echo "chown -R --no-dereference root:root ${ROOTFS_WORKDIR}/*" >> ${ROOTFS_SCRIPT};
echo 'if [ $? -ne 0 ]; then exit 1; fi' >> ${ROOTFS_SCRIPT};	
# We set the root dir default permissions
echo "chmod 755 ${ROOTFS_WORKDIR}/" >> ${ROOTFS_SCRIPT};
echo 'if [ $? -ne 0 ]; then exit 1; fi' >> ${ROOTFS_SCRIPT};	
for modecfg in ${CHMOD_LIST}; do
	file=${modecfg/=*/};
	group=${modecfg/${file}=*./};
	user=${modecfg/${file}=/};
	user=${user/.${group}/};
	if [ ! -e ${ROOTFS_DIR}/${file} ]; then
		${PRINT} "failed.";
		echo "ERROR: ${file} in --chmod directive does not exist in root filesystem.";
		clean_and_exit 25;
	fi
	echo "chown -R --no-dereference ${user}:${group} ${ROOTFS_WORKDIR}/${file}" >> ${ROOTFS_SCRIPT};
	echo 'if [ $? -ne 0 ]; then exit 1; fi' >> ${ROOTFS_SCRIPT};	
done
${PRINT} "done.";

# Last addition to the script: generate image
echo "${MKFS}" >> ${ROOTFS_SCRIPT};
echo 'if [ $? -ne 0 ]; then exit 1; fi' >> ${ROOTFS_SCRIPT};	
echo 'exit 0;' >> ${ROOTFS_SCRIPT};

# Run the generated script under fakeroot
if [ "X${ROOTFS_DIR}" != "X${ROOTFS_WORKDIR}" ]; then
	${PRINT} -n "Importing root filesystem ... ";
	tar c -C ${ROOTFS_DIR} . | tar x --preserve -C ${ROOTFS_WORKDIR};
	if [ $? -ne 0 ]; then
		${PRINT} "failed.";
		echo "ERROR: Could not import root filesystem into the temp directory.";
		clean_and_exit 30;
	fi
	${PRINT} "done.";
fi

${PRINT} -n "Generating image ... ";
$FAKEROOT ${ROOTFS_SCRIPT};
if [ $? -ne 0 ]; then
	${PRINT} "failed.";
	echo "ERROR: fakeroot failed to generate the image ${ROOTFS_IMAGE}.";
	clean_and_exit 35;
fi
chmod 664 ${ROOTFS_IMAGE}
if [ $? -ne 0 ]; then
	${PRINT} "failed.";
	echo "ERROR: failed to change access rights of ${ROOTFS_IMAGE}.";
	clean_and_exit 36;
fi
${PRINT} "done.";

# Check size is less or equal to FS_SIZE
if [ $((FS_SIZE)) -ne 0 ]; then
	${PRINT} -n "Check image size ... ";
	case "${FS_TYPE}" in
		"tar")
			# Note that this is not accurate because the current filesystem blocks may be
			# very different from the destination filesystem blocks
			real_size=$(du -bs ${ROOTFS_DIR} | sed -e 's/\([0-9]*\).*/\1/');
			;;
		*)
			real_size=$(ls -s --block-size=1 ${ROOTFS_IMAGE} | sed -e 's/\([0-9]*\).*/\1/');
			;;
	esac
	if [ $((real_size)) -gt $((FS_SIZE)) ]; then
		${PRINT} "failed.";
		echo "ERROR: image is ${real_size} bytes, max set to ${FS_SIZE} B.";
		clean_and_exit 40;
	fi
	${PRINT} "ok.";
fi

${PRINT} "Created image ${ROOTFS_IMAGE} successfully".
clean_and_exit 0;
