#!/bin/bash
# mkinitrd.sh
# Copyright 2006 Mobilygen Corp.

# This script can take a rootfs directory, a device list and a list
# of files and modules to put in and generate an initrd image.

PROG=$0

function help ()
{	echo -e "$(basename ${PROG}): Compile the content of a disk to use as a boot disk for Linux from a list of files and their dependencies.";
	echo -e "Syntax: $(basename ${PROG}) <rootfs_dir> <initd_bin> \\"
	echo -e "\t--modules <mod_list> --files <files_list> --devtable <dev_file> [ --init <init_or_linuxrc_file> ]\\";
	echo -e "\t[ --rootdev <dev>,<fs> ] [ --fs <ramd_fs> ] [ --size <fs_size_B> ] [ --kernel <kver> ] [ --tmp <tmp> ]";
	echo -e "\t[ --quiet | -q ] | [ --help | -h ] [ --keeptmp ]";
	echo -e "--fakeroot <fakeroot path> : override the use of the internal fakeroot and"
	echo -e "\tuse the one installed on the local machine instead"
	echo -ne "This script will use <mod_list> and <files_list> to generate a list of files ";
	echo -ne "to copy on the Ram disk. To do so it will find all the modules in ";
	echo -ne "<rootfs_path>/lib/modules/<kver> (<kver> is automatically picked if only one ";
	echo -ne "modules directory is present, otherwise use --kernel), resolve their ";
	echo -ne "dependencies and put all required modules in a temporary directory (can be ";
	echo -ne "forced using --tmp). The <mod_list> is then going to be copied as is into ";
	echo -ne "/etc/modules of the RAM disk (which means you can transfer arguments to ";
	echo -ne "modules listed in that file).It will also take all files listed in <files_list> ";
	echo -ne "each line being:\n";
	echo -ne "\t<file_in_rdisk> [ <file_in_rootfs> ]\n";
	echo -ne "<file_in_rootfs> is relative to <rootfs_path> and only needed if different ";
	echo -ne "from <file_in_rdisk> unless if it starts with // which means the path is "
	echo -ne "on the host machine. You do not need to list libraries because the program is ";
	echo -ne "going to resolve libraries dependencies automatically. The script will also create";
	echo -ne "/dev entries listed in file <dev_file> that should match mkfs.jffs2 syntax.\n"
	echo -ne "By default this script will use the root fs /etc/fstab / entry as the mountpoint, "
	echo -ne "if you want it to use another device use --rootdev, where <dev> is the device and "
	echo -ne "<fs> its file system.\n"
	echo -ne "The generated file can be either a RAM disk or an initramfs. If you use --initramfs ";
	echo -ne "or '--fs ramfs' the script will create an initramfs image, otherwise it will be a ";
	echo -ne "RAM disk.\n";
	echo -ne "The kernel expect a file named /linuxrc, if the root filesystem does not contain such";
	echo -ne "file use --init to specify one.\n";
	echo -ne "--quiet: in quiet mode the program is very silent, otherwise it is quite verbose.\n";
	echo -ne "--keeptmp: do not remove the temp directory at the end.\n";
	echo
}

function clean_and_exit()
{	local RET_CODE=$1
	if [ -n "${TMP_DIR}" ]; then
		if [ -d ${TMP_DIR} ] && [ $((KEEPTMP)) -ne 1 ]; then
			if [ $((VERBOSE)) -ne 0 ]; then
				echo -ne "Cleaning temporary directory ... ";
			fi
			rm -rf ${TMP_DIR};
			if [ $? -ne 0 ]; then 
				if [ $((VERBOSE)) -ne 0 ]; then
					echo -e "failed.";
				fi
				echo "WARNING: could not remove temp directory ${TMP_DIR}, please do it manually.";
			elif [ $((VERBOSE)) -ne 0 ]; then
				echo -e "done.";
			fi
		fi
	fi
	if [ $((VERBOSE)) -ne 0 ]; then echo; fi
	exit ${RET_CODE}
}

function import_file()
{	local ORIGIN=$1;
	local DEST=$2;
	local file;
	local ERR=0;
	# If file already exist we assume it has already been analyzed
	if [ -e ${TMP_DIR}/initrd/${DEST} ]; then
		return 0;
	fi
	# Create parent directory if missing
	if [ ! -d $(dirname ${TMP_DIR}/initrd/${DEST}) ]; then
		mkdir -p $(dirname ${TMP_DIR}/initrd/${DEST});
		if [ $? -ne 0 ]; then return 1; fi
	fi
	if [ -d ${ROOTFS_DIR}/${ORIGIN} ]; then
		mkdir -p ${TMP_DIR}/initrd/${DEST};
		if [ $(ls -1 ${ROOTFS_DIR}/${ORIGIN}/ | wc -l) -eq 0 ]; then return 0; fi
		for file in ${ROOTFS_DIR}/${ORIGIN}/*; do
			import_file ${ORIGIN}/$(basename ${file}) ${DEST}/$(basename ${file});
			if [ $? -ne 0 ]; then return 2; fi
		done
		return 0;
	fi
	# Copy file 'as is'
	cp -a ${ROOTFS_DIR}/${ORIGIN} ${TMP_DIR}/initrd/${DEST};
	if [ $? -ne 0 ]; then return 2; fi
	# If file is a link, copy origin of the link
	if [ -h ${ROOTFS_DIR}/${ORIGIN} ]; then
		file=$(ls -l ${ROOTFS_DIR}/${ORIGIN} | sed -e 's/.*->[ ]*//');
		if [ "X${file:0:1}" != "X/" ]; then
			file=$(dirname ${ORIGIN})/${file};
		fi
		if [ -e ${ROOTFS_DIR}/${file} ]; then
			import_file ${file} ${file};
			ERR=$?;
			if [ $((ERR)) -ne 0 ]; then return $ERR; fi
		fi
	# Resolve dependencies
	elif [ -x ${ROOTFS_DIR}/${ORIGIN} ]; then
		# Check if it is an ELF file and retrieve the info
		local elf_info=$(readelf -d ${ROOTFS_DIR}/${ORIGIN} 2> /dev/null | grep '(NEEDED)' | sed -e 's/[^[]*\[\([^]]*\).*/\1/');
		for file in ${elf_info}; do
			if [ -r ${ROOTFS_DIR}/${file} ]; then
				import_file ${file} ${file};
				ERR=$?;
				if [ $((ERR)) -ne 0 ]; then return $ERR; fi
			elif [ -r ${ROOTFS_DIR}/lib/${file} ]; then
				import_file /lib/${file} /lib/${file};
				ERR=$?;
				if [ $((ERR)) -ne 0 ]; then return $ERR; fi
			elif [ -r ${ROOTFS_DIR}/usr/lib/${file} ]; then
				import_file /usr/lib/${file} /usr/lib/${file};
				ERR=$?;
				if [ $((ERR)) -ne 0 ]; then return $ERR; fi
			elif [ -r ${ROOTFS_DIR}/$(dirname ${ORIGIN})/${file} ]; then
				import_file $(dirname ${ORIGIN})/${file} $(dirname ${DEST})/${file};
				ERR=$?;
				if [ $((ERR)) -ne 0 ]; then return $ERR; fi
			else
				echo -e "WARNING: unresolved reference to ${file} in ${DEST}";
			fi
		done
	fi
	return 0;
}

function import_module()
{	
	local MODULE_NAME=$1;
	# First we must find that module
	local MODLINE=
	local MODPATH=
	local MODDEPS=

	if [ -e ${ROOTFS_DIR}/lib/modules/${KERNEL_VER}/extra/${MODULE_NAME}.ko ]
	then
		MODPATH=/lib/modules/${KERNEL_VER}/extra/${MODULE_NAME}.ko
	else
		MODLINE=$(grep -e "[/^]${MODULE_NAME}.ko:" ${ROOTFS_DIR}/lib/modules/${KERNEL_VER}/modules.dep);
		MODPATH=${MODLINE/:*/}
		MODDEPS=${MODLINE/*:/}
	fi
	if [ -z "${MODPATH}" ]
	then
		if [ ${BUILTIN} -eq 0 ]
		then
			if [ $((VERBOSE)) -ne 0 ]
			then
				echo -e "failed.";
			fi
			echo -e "ERROR: module ${MODULE_NAME} not found for linux ${KERNEL_VER}.";
			return 1;
		else
			# if builtin is set, the modules is assumed to be builtin and so we 
			# don't return an error
			return 0
		fi
	fi
	# If it is already installed, bypass the remaining as it has probably been done
	if [ -f ${TMP_DIR}/initrd/${MODPATH} ]; then
		return 0;
	fi
	# Copy the module into the RAM disk
	if [ ! -d $(dirname ${TMP_DIR}/initrd/${MODPATH}) ]; then
		mkdir -p $(dirname ${TMP_DIR}/initrd/${MODPATH});
		if [ $? -ne 0 ]; then
			if [ $((VERBOSE)) -ne 0 ]; then
				echo -e "failed.";
			fi
			echo -e "ERROR: failed to create directory $(dirname ${TMP_DIR}/initrd/${MODPATH}).";
			return 2;
		fi
	fi
	cp ${ROOTFS_DIR}/${MODPATH} ${TMP_DIR}/initrd/${MODPATH};
	if [ $? -ne 0 ]; then
		if [ $((VERBOSE)) -ne 0 ]; then
			echo -e "failed.";
		fi
		echo -e "ERROR: failed to install ${MODULE_NAME}.";
		return 3;
	fi
	# Then we find each modules it depends on and install them
	local module;
	for module in ${MODDEPS}; do
		import_module $(basename ${module/.ko/})
		if [ $? -ne 0 ]; then
			if [ $((VERBOSE)) -ne 0 ]; then
				echo -e "failed.";
			fi
			echo -e "ERROR: failed to install ${MODULE_NAME} dependency ${module}.";
			return 4;
		fi
	done
	# Update initrd modules.dep
	if [ "X${MODLINE}" != "X" ]
	then
		echo ${MODLINE} >> ${TMP_DIR}/initrd/lib/modules/${KERNEL_VER}/modules.dep;
		if [ $? -ne 0 ]; then
			if [ $((VERBOSE)) -ne 0 ]; then
				echo -e "failed.";
			fi
			echo -e "ERROR: failed to update module list with dependencies for ${MODULE_NAME}.";
			return 5;
		fi
	fi
	return 0;
}


MODULES_LIST=;
FILES_LIST=;
ROOT_DEVICE=;
KERNEL_VER=;
TMP_DIR=;
VERBOSE=1;
INITRD_IMAGE=;
ROOTFS_DIR=;
INIT_FILE=;
DEV_FILE=;
FS_TYPE=;
FS_SIZE=0;
INITRAMFS=0;
KEEPTMP=0;
FAKEROOT=
# set --builtin if modules may be builtin so we don't fail
BUILTIN=0
while [ -n "$1" ]; do
	case "$1" in
		"--modules")
			MODULES_LIST=$2;
			shift 2;
			;;
		"--builtin")
			BUILTIN=1
			shift
			;;
		"--files")
			FILES_LIST=$2;
			shift 2;
			;;
		"--rootdev")
			ROOT_DEVICE=${2/,*/};
			ROOT_DEVICE_FS=${2/*,/};
			shift 2;
			;;
		"--kernel")
			KERNEL_VER=$2;
			shift 2;
			;;
		"--fakeroot")
			FAKEROOT=$2;
			shift 2;
			;;
		"--tmp")
			TMP_DIR=$2;
			shift 2;
			;;
		"--quiet" | "-q")
			VERBOSE=0;
			shift 1;
			;;
		"--init" | "--linuxrc")
			INIT_FILE=$2;
			shift 2;
			;;
		"--fs")
			FS_TYPE=$2;
			shift 2;
			;;
		"--initramfs")
			INITRAMFS=1;
			shift 1;
			;;
		"--size")
			FS_SIZE=$2;
			shift 2;
			;;
		"--devtable")
			DEV_FILE=$2;
			shift 2;
			;;
		"--help" | "-h")
			help;
			clean_and_exit 0;
			;;
		"--keeptmp") 
			KEEPTMP=1; 
			shift 1;
			;;
		*)
			if [ -z "${ROOTFS_DIR}" ]; then
				ROOTFS_DIR=$1;
			elif [ -z "${INITRD_IMAGE}" ]; then
				INITRD_IMAGE=$1;
			else
				echo "ERROR: Invalid argument $1, try --help";
				clean_and_exit 1;
			fi
			shift 1;
			;;
	esac
done

# Check arguments and prepare environment
if [ -z "${INITRD_IMAGE}" ]; then
	echo -e "ERROR: You must give the name of the resulting image.";
	clean_and_exit 2;
elif [ -e ${INITRD_IMAGE} ]; then
	rm ${INITRD_IMAGE};
	if [ $? -ne 0 ]; then
		echo -e "ERROR: Could not erase previous image ${INITRD_IMAGE}.";
		clean_and_exit 3;
	fi
fi

if [ -z "${ROOTFS_DIR}" ]; then
	echo -e "ERROR: You must give the path to the root directory of the target (on the host).";
	clean_and_exit 4;
elif [ ! -d ${ROOTFS_DIR} ]; then
	echo -e "ERROR: ${ROOTFS_DIR} not found or not a directory.";
	clean_and_exit 5;
elif [ "X${ROOTFS_DIR:0:1}" != "X/" ]; then
	# import_file does not support a rootfs_dir that is not an absolute path, fix it
	ROOTFS_DIR="$(pwd)/${ROOTFS_DIR}";
fi

if [ -z "${MODULES_LIST}" ]; then
	if [ $((VERBOSE)) -ne 0 ]; then
		echo "WARNING: no modules specified, this RAM disk may not be very usefull !!";
	fi
elif [ ! -r "${MODULES_LIST}" ]; then
	echo "ERROR: modules list ${MODULES_LIST} is not accessible.";
	clean_and_exit 6;
else
	if [ -z "${KERNEL_VER}" ]; then
		KERNEL_VER=$(ls -1 ${ROOTFS_DIR}/lib/modules | tail -n1);
		if [ -z "${KERNEL_VER}" ]; then
			echo "ERROR: no module support found in root filesystem.";
			clean_and_exit 7;
		fi
	fi	
	if [ ! -d "${ROOTFS_DIR}/lib/modules/${KERNEL_VER}" ]; then
		echo "ERROR: your root filesystem has no modules support for linux ${KERNEL_VER}.";
		clean_and_exit 8;
	fi
fi

if [ -z "${FILES_LIST}" ]; then
	echo "ERROR: You must specify a list of files with at least /linuxrc or /init.";
elif [ ! -r "${FILES_LIST}" ]; then
	echo "ERROR: files list ${FILES_LIST} is not accessible.";
	clean_and_exit 9;
fi

if [ -z "${ROOT_DEVICE}" ] && [ ! -r ${ROOTFS_DIR}/etc/fstab ]; then
	echo "ERROR: Cannot find /etc/fstab in root filesystem, use --rootdev to select root device.";
	clean_and_exit 10;
elif [ -n "${ROOT_DEVICE}" ] && [ -z "${ROOT_DEVICE_FS}" ]; then
	echo "ERROR: Cannot find /etc/fstab in root filesystem, use --rootdev to select root device.";
	clean_and_exit 10;
fi	

if [ -z "${INIT_FILE}" ]; then
	if [ $((INITRAMFS)) -eq 0 ]; then
		if [ ! -e ${ROOTFS_DIR}/linuxrc ]; then
			echo "ERROR: Cannot find /linuxrc in root filesystem, use --init to select an alternate script.";
			clean_and_exit 11;
		fi
		INIT_FILE=${ROOTFS_DIR}/linuxrc;
	else
		if [ ! -e ${ROOTFS_DIR}/init ]; then
			echo "ERROR: Cannot find /init in root filesystem, use --init to select an alternate script.";
			clean_and_exit 11;
		fi
		INIT_FILE=${ROOTFS_DIR}/init;
	fi
elif [ ! -r ${INIT_FILE} ]; then
	echo "ERROR: Cannot find ${INIT_FILE}, invalid --init argument.";
	clean_and_exit 11;
fi	

if [ $((INITRAMFS)) -ne 0 ]; then
	if [ ! -z "${FS_TYPE}" ]; then
		echo "ERROR: you cannot specify the filesystem type with --initramfs.";
		clean_and_exit 12;
	fi
	FS_TYPE=ramfs;
fi

case "${FS_TYPE}" in
	"") FS_TYPE=ext2;;
	"cramfs") ;;
	"ext2") ;;
	"ramfs") ;;
	*)
		echo "ERROR: unsupported filesystem ${FS_TYPE} given as argument to --fs.";
		clean_and_exit 12;
		;;
esac

if [ -n "${DEV_FILE}" ]; then
	if [ ! -r ${DEV_FILE} ]; then
		echo "ERROR: device file ${DEV_FILE} is not accessible.";
		clean_and_exit 13;
	fi
fi

if [ -z "${TMP_DIR}" ]; then
	TMP_DIR="/tmp/$(basename ${PROG}).$$";
fi

if [ -d ${TMP_DIR} ]; then
	rm -r ${TMP_DIR};
	if [ $? -ne 0 ]; then
		echo -e "ERROR: Could not clean temporary directy ${TMP_DIR}.";
		clean_and_exit 14;
	fi
fi

if [ $((VERBOSE)) -ne 0 ]; then
	echo -e "The RAM disk image ${INITRD_IMAGE} will be generated using the following:";
	if [ -n "${MODULES_LIST}" ]; then
		echo -e "\tModules listed in ${MODULES_LIST} for linux ${KERNEL_VER}.";
	fi
	echo -e "\tFiles listed in ${FILES_LIST} from rootfs ${ROOTFS_DIR}.";
	if [ -n "${ROOT_DEVICE}" ]; then
		echo -e "\tfstab will be ignored and ${ROOT_DEVICE} will be mounted as the root filesystem.";
	else
		echo -e "\tfstab will be used to determine the final root filesystem.";
	fi
	echo -e "\tThe initrd entry point will be ${INIT_FILE}.";
	echo -e "\ttemp dir: ${TMP_DIR}";
fi

# Create temp dir
mkdir -p ${TMP_DIR}/initrd;
if [ $? -ne 0 ]; then
	echo -e "ERROR: could not create temporary directy ${TMP_DIR}.";
	clean_and_exit 15;
fi

if [ ! -d ${TMP_DIR}/initrd/etc ]; then
	mkdir -p ${TMP_DIR}/initrd/etc;
	if [ $? -ne 0 ]; then
		echo -e "ERROR: failed to create RAM disk /etc directory.";
		clean_and_exit 19;
	fi
fi

# Generate file list with files in FILES_LIST and
# eventual lib dependencies
if [ $((VERBOSE)) -ne 0 ]; then
	echo -ne "Importing requested files ... ";
fi
while read INITRD_FILE ROOTFS_FILE; do
	if [ -z "${INITRD_FILE}" ] || [ "X#" == "X${INITRD_FILE:0:1}" ]; then
		continue;
	fi
	if [ -z "${ROOTFS_FILE}" ]; then 
		ROOTFS_FILE=${INITRD_FILE};
	else
		ROOTFS_FILE=${ROOTFS_FILE/\#*/}
	fi
	import_file ${ROOTFS_FILE} ${INITRD_FILE}
	ERR=$?;
	if [ $((ERR)) -ne 0 ]; then
		if [ $((VERBOSE)) -ne 0 ]; then
			echo -e "failed.";
		fi
		echo -e "ERROR: failed to import ${INITRD_FILE} into RAM disk, error $ERR.";
		clean_and_exit 20;
	fi
done < ${FILES_LIST};

if [ -n "${MODULES_LIST}" ]; then
	# If applicable append modules in MODULES_LIST and eventual dependencies
	if [ $((VERBOSE)) -ne 0 ]; then
		echo -ne "done.\nInstalling linux ${KERNEL_VER} modules ... ";
	fi
	while read MODULE MODARGS; do
		if [ -z "${MODULE}" ] || [ "X#" == "X${MODULE:0:1}" ]; then
			continue;
		fi
		if [ -n "${MODARGS}" ]; then 
			MODARGS=${MODARGS/\#*/}
		fi
		import_module ${MODULE};
		ERR=$?;
		if [ $((ERR)) -ne 0 ]; then
			if [ $((VERBOSE)) -ne 0 ]; then
				echo -e "failed.";
			fi
			echo -e "ERROR: failed to import module ${MODULE} into RAM disk, error ${ERR}.";
			clean_and_exit 30;
		fi
	done < ${MODULES_LIST};

	# If applicable copy MODULES_LIST file into /etc/modules
	cp ${MODULES_LIST} ${TMP_DIR}/initrd/etc/modules;
	if [ $? -ne 0 ]; then
		if [ $((VERBOSE)) -ne 0 ]; then
			echo -e "failed.";
		fi
		echo -e "ERROR: failed to create RAM disk /etc/modules.";
		clean_and_exit 31;
	fi
fi

if [ $((VERBOSE)) -ne 0 ]; then
	echo -ne "done.\nCreating /etc/fstab ... ";
fi

# Generate /etc/fstab if not already installed
if [ ! -f ${TMP_DIR}/initrd/etc/fstab ]; then
	if [ -n "${ROOT_DEVICE}" ]; then
		echo "${ROOT_DEVICE} /rootfs ${ROOT_DEVICE_FS} defaults 1 1" > ${TMP_DIR}/initrd/etc/fstab
		if [ $? -ne 0 ]; then
			if [ $((VERBOSE)) -ne 0 ]; then
				echo -e "failed.";
			fi
			echo -e "ERROR: failed to create RAM disk /etc/fstab for ${ROOT_DEVICE}.";
			clean_and_exit 40;
		fi
	else
		grep -e '[^ ]*[ *]/[ *].*' ${ROOTFS_DIR}/etc/fstab | sed -e 's:[ \t]/[ \t]: /rootfs :' > ${TMP_DIR}/initrd/etc/fstab
		if [ $? -ne 0 ]; then
			if [ $((VERBOSE)) -ne 0 ]; then
				echo -e "failed.";
			fi
			echo -e "ERROR: failed to create RAM disk /etc/fstab from rootfs fstab.";
			clean_and_exit 41;
		fi
	fi
fi

if [ $((VERBOSE)) -ne 0 ]; then
	echo -ne "done.\nCreating RAM disk image ... ";
fi

# Create CRAMFS filesystem
MKROOTFS_ARGS="--fs ${FS_TYPE} --inplace --quiet --tmp ${TMP_DIR}/mkrootfs ${TMP_DIR}/initrd ${INITRD_IMAGE}"
if [ "X${FAKEROOT}" != "X" ] 
then 
	MKROOTFS_ARGS="${MKROOTFS_ARGS} --fakeroot ${FAKEROOT}" 
fi

if [ -n "${DEV_FILE}" ]; then
	MKROOTFS_ARGS="${MKROOTFS_ARGS} --devtable ${DEV_FILE}";
fi
if [ $((FS_SIZE)) -ne 0 ]; then
	MKROOTFS_ARGS="${MKROOTFS_ARGS} --size ${FS_SIZE}";
fi
if [ $((KEEPTMP)) -eq 1 ]; then
	MKROOTFS_ARGS="${MKROOTFS_ARGS} --keeptmp";
fi
mkrootfs.sh ${MKROOTFS_ARGS};
if [ $? -ne 0 ]; then
	if [ $((VERBOSE)) -ne 0 ]; then
		echo -e "failed.";
	fi
	echo -e "ERROR: failed to create image ${INITRD_IMAGE}.";
	clean_and_exit 50;
fi

# initrd image is now done
if [ $((VERBOSE)) -ne 0 ]; then
	echo "done, image is ${INITRD_IMAGE}.";
fi
clean_and_exit 0;
