/*
 *  This file Copyright (C) 2007 Mobilygen Corp.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE	LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef _GPIO_CORE_H_
#define _GPIO_CORE_H_

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/sysfs.h>
#include <linux/cdev.h>
#include <linux/ioctl.h>

#include <asm/atomic.h>
#include <asm-generic/gpio.h>
/*-------------------------------------------------------------------------*/

/*****************************************************
 * COMMON STUFF
 *****************************************************/
 
 /* Interrupts. */

 #ifndef NO_IRQ
 #define NO_IRQ 	(-1)
 #endif

typedef int (*gpio_interrupt_callback_t)(unsigned gpio,
	unsigned gpio_index, void *cbdata);

struct gpio_interrupt_config {
	/* 
	 * You can have GPIO_INT_TYPE_LEVEL | GPIO_INT_TYPE_EDGE 
	 * if controller supports it.
	 */
#define GPIO_INT_TYPE_NONE			0
#define GPIO_INT_TYPE_LEVEL			1
#define GPIO_INT_TYPE_EDGE			2
	char 					type;
	
	/* 
	 * You can have GPIO_INT_POLARITY_LOW_OR_FALLING | GPIO_INT_POLARITY_HIGH_OR_RISING 
	 * if controller supports it.
	 */
#define GPIO_INT_POLARITY_NONE			0
#define GPIO_INT_POLARITY_LOW_OR_FALLING	1
#define GPIO_INT_POLARITY_HIGH_OR_RISING	2
	char 					polarity;
	
	/* 
	 * This feature may not be supported by controller.
	 */
	char					debounce_on;
};

/* Userspace interface: IOCTL defines. */

#define GPIO_IOC_MAGIC				'g'
#define GPIO_READ				_IO (GPIO_IOC_MAGIC, 0)
#define GPIO_SET				_IO (GPIO_IOC_MAGIC, 1)
#define GPIO_CLEAR				_IO (GPIO_IOC_MAGIC, 2)
#define GPIO_INPUT				_IO (GPIO_IOC_MAGIC, 4)
#define GPIO_RWTIMING				_IOW(GPIO_IOC_MAGIC, 5, unsigned long)

#define GPIOINT_IOC_MAGIC			'i'
#define GPIOINT_ENABLE				_IO (GPIOINT_IOC_MAGIC, 0)
#define GPIOINT_DISABLE				_IO (GPIOINT_IOC_MAGIC, 1)
#define GPIOINT_IS_ACTIVE			_IO (GPIOINT_IOC_MAGIC, 2)
#define GPIOINT_CLEAR				_IO (GPIOINT_IOC_MAGIC, 3)
#define GPIOINT_TYPE				_IOW(GPIOINT_IOC_MAGIC, 4, unsigned long)
#define GPIOINT_POLARITY			_IOW(GPIOINT_IOC_MAGIC, 5, unsigned long)
#define GPIOINT_DEBOUNCE_ON			_IOW(GPIOINT_IOC_MAGIC, 6, unsigned long)

#define GPIOBANK_IOC_MAGIC			'G'
#define GPIOBANK_SETGRP				_IOW(GPIOBANK_IOC_MAGIC, 0, unsigned long)
#define GPIOBANK_GETGRP				_IOR(GPIOBANK_IOC_MAGIC, 1, unsigned long)
#define GPIOBANK_INPUT				_IO (GPIOBANK_IOC_MAGIC, 2)
#define GPIOBANK_WRITE				_IOW(GPIOBANK_IOC_MAGIC, 3, unsigned long)
#define GPIOBANK_READ				_IOR(GPIOBANK_IOC_MAGIC, 4, unsigned long)
#define GPIOBANK_WRISATOMIC			_IO (GPIOBANK_IOC_MAGIC, 5)
#define GPIOBANK_RWTIMING			_IOW(GPIOBANK_IOC_MAGIC, 6, unsigned long)

/*-------------------------------------------------------------------------*/

/*****************************************************
 * DRIVER INTERFACE
 *****************************************************/

/* Various defines */

#define GPIO_DRV_NAME_MAXLEN 			32
#define GPIO_ANYFILE_MAXLEN			32
#define GPIO_ROOT_CLASS_NAME			"gpio"
#define GPIO_BANK_FILE_FORMAT			"bank%u"
#define GPIO_STATUS_FILE_FORMAT			"status%u"
#define GPIO_FILE_FORMAT 			"gpio%03u"
#define GPIO_INT_FILE_FORMAT 			"gpio%03uint"
#define GPIO_INDEX_LINK_FORMAT 			"%02u"

/* /proc specific */
#ifdef CONFIG_PROC_FS
#define GPIO_ROOT_PROC_DIR        		"driver/gpio"
#define GPIO_PROC_FILE_FORMAT 			"%02u"
#endif

/* Structs & co */

struct gpio_interrupt_callback {
	struct list_head			list;
	gpio_interrupt_callback_t 		cb;
	void 					*cbdata;
};

struct gpio_pin_info {
	unsigned 				gpio;
	struct gpio_driver			*drv;
	unsigned 				cached_index;
	int					irq;
	struct mutex				used_lock;
	const char 				*label;
	struct list_head			callbacks;
	atomic_t				int_counter;
	atomic_t				int_enabled;
	atomic_t				fop_open_count;
	struct device				*class_dev, *class_dev_int;
	void 					*driver_data;
};

#define gpio_driver_set_pininfo(gpio_drv, gpio_index, gpio_num, gpio_irq) { \
	(gpio_drv)->bank[gpio_index].gpio = gpio_num; \
	(gpio_drv)->bank[gpio_index].drv = gpio_drv; \
	(gpio_drv)->bank[gpio_index].cached_index = gpio_index; \
	(gpio_drv)->bank[gpio_index].irq = gpio_irq; \
}

#define gpio_pin_get_drvdata(gpio_drv, gpio_index) \
	(gpio_drv)->bank[gpio_index].driver_data
#define gpio_pin_set_drvdata(gpio_drv, gpio_index, data) \
	(gpio_drv)->bank[gpio_index].driver_data = data

struct gpio_driver {
	/*-----------------------------------------------------------------*/
	
	/*
	 * Those members should not be touched by driver itself.
	 */
	 
	struct list_head 			list;
	atomic_t				refcount;
	struct mutex				bank_lock;
	
	unsigned				bank_id;
	
#ifdef CONFIG_PROC_FS
	struct proc_dir_entry			*pentry;
	struct proc_dir_entry			*bank_pentry, *status_pentry;
	struct proc_dir_entry			**pentries;
#endif
	
	struct cdev				chrdev;
	dev_t					devt;
	int					minors;
	struct class				*class;
	struct device				*class_dev_bank;
	int					status_offset;
	
	spinlock_t				irq_lock;
	unsigned long				spin_flags;
	atomic_t				pending_spin_unlock;
	
	wait_queue_head_t			fs_int_wait_queue;
	
	int					fs_rw_interval_us, fs_bank_rw_interval_us;
	int 					fs_bank_mask;
	struct fasync_struct 			*fs_async_queue;
	
	/*-----------------------------------------------------------------*/
	
	/*
	 * Those members configures the driver and must be set.
	 */
	
	/* Name of driver, cannot be empty. Must be unique! */
	char 					name[GPIO_DRV_NAME_MAXLEN];
	
	/* Name of group/alias to which the driver belongs. Should not be unique! */
	char 					group[GPIO_DRV_NAME_MAXLEN];
	
	/* 
	 * The parent device for the bank.
	 * Can be NULL, but /sys entries will not be able to be created.
	 */
	struct device 				*dev;
	
	/*
	 * Table with pins info, cannot be NULL.
	 * You can use the macro below to allocate it.
	 */
#define gpio_driver_alloc_bank(drv, size) \
	{ (drv)->bank = kzalloc(sizeof(struct gpio_pin_info) * size, GFP_KERNEL); }
	struct gpio_pin_info			*bank;
	
	/* The size of the table above, in GPIO pin count. */
	unsigned				bank_size;
	
	/*
	 * Set to 1 if you want to handle the locking yourself,
	 * or dont need any.
	 */
	char					dont_lock;
	
	/* Set to 1 if your bank operations modify all pins at the same time. */
	char					bank_is_atomic;
	
	/* Set to 1 to disable creation of entries in /dev. */
	char 					no_dev_entries; 
	
	struct gpio_chip	chip;
	/* 
	 * Use this to put your driver specific data and 
	 * retrieve it when passed a struct gpio_driver.
	 */
#define gpio_set_drvdata(drv, data) 		((drv)->driver_data) = data
#define gpio_get_drvdata(drv) 			((drv)->driver_data)
	void 					*driver_data;
	
	/*
	 * Exported functions. MANDATORY means that
	 * exporting the function is mandatory.
	 */
	
	int 	(*request) 			(struct gpio_driver *drv, unsigned gpio_index, const char *label); 
	
	void 	(*free) 			(struct gpio_driver *drv, unsigned gpio_index);
	
	int 	(*is_enabled) 			(struct gpio_driver *drv, unsigned gpio_index);
	
	int 	(*direction_input) 		(struct gpio_driver *drv, unsigned gpio_index);
	
	int 	(*direction_output) 		(struct gpio_driver *drv, unsigned gpio_index, int value); 
	
	int 	(*get_value) 			(struct gpio_driver *drv, unsigned gpio_index); /* MANDATORY */
	
	/* If possible, this should not change data direction: */
	int 	(*set_value) 			(struct gpio_driver *drv, unsigned gpio_index, int value); /* MANDATORY */
	
#define GPIO_DIRECTION_INPUT	0
#define GPIO_DIRECTION_OUTPUT	1
	int 	(*get_direction) 		(struct gpio_driver *drv, unsigned gpio_index);
	
	int 	(*read_bank)			(struct gpio_driver *drv, u32 *data, int dword_count); 
	
	/* Changes the data direction of concerned pins to OUTPUT */
	int 	(*write_bank)			(struct gpio_driver *drv, const u32 *data, const u32 *mask, int dword_count);
	
	int	(*interrupt_configure)		(struct gpio_driver *drv, unsigned gpio_index, const struct gpio_interrupt_config *config);
	int	(*interrupt_get_config)		(struct gpio_driver *drv, unsigned gpio_index, struct gpio_interrupt_config *config);
	int	(*interrupt_enable)		(struct gpio_driver *drv, unsigned gpio_index);
	int	(*interrupt_is_enabled)		(struct gpio_driver *drv, unsigned gpio_index);
	int	(*interrupt_is_active)		(struct gpio_driver *drv, unsigned gpio_index, int irq);
	int	(*interrupt_disable)		(struct gpio_driver *drv, unsigned gpio_index);
	int	(*interrupt_clear)		(struct gpio_driver *drv, unsigned gpio_index);
};

/* Methods */

/*
 * Driver (un)registration
 */
int gpio_driver_register(struct gpio_driver *drv);
int gpio_driver_unregister(struct gpio_driver *drv);

/*-------------------------------------------------------------------------*/

/*****************************************************
 * CLIENT INTERFACE
 *****************************************************/

/* A handle representing a GPIO bank driver. */
typedef struct gpio_driver *gpio_driver_handle;

/*
 * Find the driver for a given GPIO, into given group.
 * If group is NULL or empty, all groups will be looked for this GPIO.
 * If GPIO is not found, NULL is returned.
 * If the driver is found then gpio_index is assigned the actual GPIO pin index
 * within the driver bank (gpio_index can be NULL if you dont care).
 */
gpio_driver_handle gpio_driver_find(const char *group,
	unsigned gpio, unsigned *gpio_index);

/*
 * Get a driver matching name, returns NULL if not found.
 * Increments driver usage refcount if found.
 */
gpio_driver_handle gpio_driver_get(const char *name);

/*
 * You must call this consecutively to a gpio_driver_get to decrement refcount.
 */
void gpio_driver_put(gpio_driver_handle drv);

/*
 * Returns the device associated with driver, if any.
 * Otherwise returns NULL.
 */
struct device *gpio_get_device(gpio_driver_handle hdrv);

/*
 * If you want to add your own proc entry to control some GPIO pin,
 * use this function.
 */
#ifdef CONFIG_PROC_FS
int gpio_proc_setup_pentry(gpio_driver_handle hdrv, unsigned gpio_index, 
	struct proc_dir_entry *pentry);
#endif

/*
 * Use to know how large in pin count is the bank for this driver.
 */
unsigned gpio_driver_bank_size(gpio_driver_handle hdrv);

/*
 * Use to know how large in bytes/dwords (rounded up) is the bank 
 * for this driver, useful for gpio_read/write_bank.
 */
unsigned gpio_driver_bank_size_bytes(gpio_driver_handle hdrv);
unsigned gpio_driver_bank_size_dwords(gpio_driver_handle hdrv);

/*
 * Use this to convert back & forth from a GPIO 
 * id to the GPIO index for the given bank/driver
 */
int gpio_to_index(gpio_driver_handle hdrv, unsigned gpio);
int gpio_from_index(gpio_driver_handle hdrv, unsigned gpio_index);

/*
 * Map GPIO numbers to IRQ numbers.
 */
static inline int gpio_to_irq(unsigned int gpio)
{
	        return __gpio_to_irq(gpio);
}
int gpio_from_irq(gpio_driver_handle hdrv, unsigned irq);

#ifdef CONFIG_GPIOLIB
#undef ARCH_NR_GPIOS
#define ARCH_NR_GPIOS	(8+32+32)
#endif
/*
 * Public access functions, should be safe for access within atomic code.
 */
 
int 	arch_gpio_request			(gpio_driver_handle hdrv, unsigned gpio_index, const char *label);
int 	gpio_request_wait		(gpio_driver_handle hdrv, unsigned gpio_index, const char *label);
void 	arch_gpio_free			(gpio_driver_handle hdrv, unsigned gpio_index);
int 	gpio_is_enabled			(gpio_driver_handle hdrv, unsigned gpio_index);
int 	arch_gpio_direction_input 		(gpio_driver_handle hdrv, unsigned gpio_index);
int 	arch_gpio_direction_output 		(gpio_driver_handle hdrv, unsigned gpio_index, int value);
int 	arch_gpio_get_value			(gpio_driver_handle hdrv, unsigned gpio_index);
int 	arch_gpio_set_value	 		(gpio_driver_handle hdrv, unsigned gpio_index, int value);
int 	gpio_get_direction		(gpio_driver_handle hdrv, unsigned gpio_index);
int 	gpio_read_bank			(gpio_driver_handle hdrv, u32 *data, int dword_count);
int 	gpio_write_bank		 	(gpio_driver_handle hdrv, const u32 *data, const u32 *mask, int dword_count);
int	gpio_interrupt_register		(gpio_driver_handle hdrv, unsigned gpio_index, gpio_interrupt_callback_t cb, void *cbdata);
int	gpio_interrupt_registered	(gpio_driver_handle hdrv, unsigned gpio_index, gpio_interrupt_callback_t cb);
int	gpio_interrupt_configure	(gpio_driver_handle hdrv, unsigned gpio_index, const struct gpio_interrupt_config *config);
int	gpio_interrupt_get_config	(gpio_driver_handle hdrv, unsigned gpio_index, struct gpio_interrupt_config *config);
int	gpio_interrupt_enable		(gpio_driver_handle hdrv, unsigned gpio_index);
int	gpio_interrupt_is_enabled	(gpio_driver_handle hdrv, unsigned gpio_index);
int	gpio_interrupt_is_active	(gpio_driver_handle hdrv, unsigned gpio_index, int irq);
int	gpio_interrupt_disable		(gpio_driver_handle hdrv, unsigned gpio_index);
int	gpio_interrupt_clear		(gpio_driver_handle hdrv, unsigned gpio_index);
int	gpio_interrupt_unregister	(gpio_driver_handle hdrv, unsigned gpio_index, gpio_interrupt_callback_t cb);


/* Optional callback mode registration/unregistration. */

#define GPIO_CLIENT_PIN_LABEL_MAXLEN		32

/*
 * This struct is used to specify which pins you need
 * in a given group. 
 */
struct gpio_client_pin {
	/* The GPIO pin you need. */
	unsigned				gpio;
	
	/*
	 * Do you want the framework to request/free those GPIOs for you
	 * upon probing? If yes supply a non-empty string in label.
	 */
	char					label[GPIO_CLIENT_PIN_LABEL_MAXLEN];
	
	/*
	 * Below are the members set by the framework itself
	 * when your probe function is called.
	 */
	/* Found GPIO index. */
	unsigned				index;
	/* Found driver. */
	gpio_driver_handle			hdrv;
};

struct gpio_client {
	/* Do not use that. */
	struct list_head 			list;
	
	/*
	 * OPTION 1:
	 * Specify the exact driver name you want OR NULL if you
	 * want to match against group & pins, see below. 
	 * When the specified driver becomes available, you probe
	 * function is called.
	 */
	const char				*match_name;
	
	/*
	 * When your probe function is called, this is set by
	 * the framework with the driver handle: you do not have
	 * to gpio_driver_get()/put().
	 */
	gpio_driver_handle			gpio_hdrv;
	
	/*
	 * OPTION 2: 
	 * Match several pins in a group. When all the specified
	 * pins become available (all the required GPIO controller 
	 * drivers have been registered), your probe function is called.
	 */
	const char				*match_group;
	struct gpio_client_pin			*match_pins;
	unsigned				pin_count;
	
	/* Do what you please with those. */
	void					*parent_data;
	void					*client_data;
	
	/* Traditional probe/remove callbacks. */
	int 	(*probe) 			(struct gpio_client *client);
	int 	(*remove) 			(struct gpio_client *client);
};

/*
 * Registers a gpio_client. When the specified driver/pins 
 * become available, the probe function of you registered client is called.
 * See struct gpio_client declaration for more info.
 */
int gpio_client_register(struct gpio_client *client);

/*
 * Unregisters a client. This will remove() the client if
 * it has been probed. 
 */
int gpio_client_unregister(struct gpio_client *client);

/*-------------------------------------------------------------------------*/

#endif /* #ifndef _GPIO_CORE_H_ */
