#ifndef _ASM_ARCH_DMA_LIST_H
#define _ASM_ARCH_DMA_LIST_H

#include <mach/platform.h>
#include <mach/dw_dmac_regs.h>
#include <mach/mobi_dma.h>

typedef union {
	unsigned long r;
	struct {
		uint32_t sar	: 32 ;
	} b;
} dmac_sar_t;

typedef union {
	unsigned long r;
	struct {
		uint32_t dar	: 32 ;
	} b;
} dmac_dar_t;

typedef union {
	unsigned long r;
	struct {
		uint32_t lms	:  2 ;
		uint32_t loc	: 30 ;
	} b;
} dmac_llp_t;

typedef union {
	unsigned long long r;
	struct {
		uint32_t int_en	:  1 ;
		uint32_t dst_tr_width	:  3 ;
		uint32_t src_tr_width	:  3 ;
		uint32_t dinc	:  2 ;
		uint32_t sinc	:  2 ;
		uint32_t dst_msize	:  3 ;
		uint32_t src_msize	:  3 ;
		uint32_t src_gather_en	:  1 ;
		uint32_t dst_scatter_en	:  1 ;
		uint32_t reserved0	:  1 ;
		uint32_t tt_fc	:  3 ;
		uint32_t dms	:  2 ;
		uint32_t sms	:  2 ;
		uint32_t llp_dst_en	:  1 ;
		uint32_t llp_src_en	:  1 ;
		uint32_t reserved1	:  3 ;
		uint32_t block_ts	: 12 ;
		uint32_t done	:  1 ;
		uint32_t reserved2	:  1 ;
	} b;
} dmac_ctl_t;

typedef union {
	unsigned long long r;
	struct {
		uint32_t reserved0	:  5 ;
		uint32_t ch_prior	:  3 ;
		uint32_t ch_susp	:  1 ;
		uint32_t fifo_empty	:  1 ;
		uint32_t hs_sel_dst	:  1 ;
		uint32_t hs_sel_src	:  1 ;
		uint32_t lock_ch_l	:  2 ;
		uint32_t loc_b_l	:  2 ;
		uint32_t lock_ch	:  1 ;
		uint32_t lock_b	:  1 ;
		uint32_t dst_hs_pol	:  1 ;
		uint32_t src_hs_pol	:  1 ;
		uint32_t max_abrst	: 10 ;
		uint32_t reload_src	:  1 ;
		uint32_t reload_dst	:  1 ;
		uint32_t fcmode	:  1 ;
		uint32_t fifo_mode	:  1 ;
		uint32_t protctl	:  3 ;
		uint32_t ds_upd_en	:  1 ;
		uint32_t ss_upd_en	:  1 ;
		uint32_t src_per	:  4 ;
		uint32_t dest_per	:  4 ;
		uint32_t reserved1	: 17 ;
	} b;
} dmac_cfg_t;

typedef union {
	unsigned long r;
	struct {
		uint32_t sgi	: 20 ;
		uint32_t sgc	: 12 ;
	} b;
} dmac_sgr_t;

typedef union {
	unsigned long r;
	struct {
		uint32_t dsi	: 20 ;
		uint32_t dsc	: 12 ;
	} b;
} dmac_dsr_t;

typedef struct dmac_lli {
    dmac_sar_t sar;
    dmac_dar_t dar;
    dmac_llp_t llp;
    dmac_ctl_t ctl;
} dmac_lli_t;

struct dmac_channel {
    uint32_t sar;
    uint32_t reserved0;
    uint32_t dar;
    uint32_t reserved1;
    dmac_llp_t llp;
    uint32_t reserved2;
    dmac_ctl_t ctl;
    unsigned long long sstat;
    unsigned long long dstat;
    unsigned long long sstatar;
    unsigned long long dstatar;
    dmac_cfg_t cfg;
    dmac_sgr_t sgr;
    uint32_t reserved3;
    dmac_dsr_t dsr;
    uint32_t reserved4;
};

#define DMAC_AHB_MASTER_0 0
#define DMAC_AHB_MASTER_1 1
#define DMAC_AHB_MASTER_2 2

#define DMAC_TR_WIDTH_8_BIT 0
#define DMAC_TR_WIDTH_16_BIT 1
#define DMAC_TR_WIDTH_32_BIT 2

/* DMAC handhshake ports */
#define DMAC_HANDSHAKE_SDMMC 0
#define DMAC_HANDSHAKE_AES   1
#define DMAC_HANDSHAKE_SHA   2
#define DMAC_HANDSHAKE_CF_RD 3
#define DMAC_HANDSHAKE_CF_WR 4
#define DMAC_HANDSHAKE_MUX_0 5
#define DMAC_HANDSHAKE_MUX_1 6
#define DMAC_HANDSHAKE_MUX_2 7
#define DMAC_HANDSHAKE_MUX_3 8

/* DMAC Handshake types */
#define DMAC_HS_SEL_HW 0
#define DMAC_HS_SEL_SW 1

/* ttfc field in ctl reg */
#define DMAC_TTFC_M2M_DMAC   0x0
#define DMAC_TTFC_M2P_DMAC   0x1
#define DMAC_TTFC_P2M_DMAC   0x2
#define DMAC_TTFC_P2P_DMAC   0x3
#define DMAC_TTFC_P2M_PER    0x4
#define DMAC_TTFC_P2P_SRCPER 0x5
#define DMAC_TTFC_M2P_PER    0x6
#define DMAC_TTFC_P2P_DSTPER 0x7

/* sinc & dinc in ctl reg */
#define DMAC_ADDRADJ_INC  0x0
#define DMAC_ADDRADJ_DEC  0x1
#define DMAC_ADDRADJ_NONE 0x2

/* msize field in ctl register */
#define DMAC_BURST_SIZE_1   0x0
#define DMAC_BURST_SIZE_4   0x1
#define DMAC_BURST_SIZE_8   0x2
#define DMAC_BURST_SIZE_16  0x3
#define DMAC_BURST_SIZE_32  0x4
#define DMAC_BURST_SIZE_64  0x5
#define DMAC_BURST_SIZE_128 0x6
#define DMAC_BURST_SIZE_256 0x7

#define LLI_SIZE 4096

/* static state variables */
static mobi_dma_handle dmah;
static volatile void* dmac_base;
static void* lli_base;
static volatile struct dmac_channel* chan;
static unsigned cache_ptr = LLI_SIZE;

#define DMAC_READL(x) readl(dmac_base + (x))
#define DMAC_WRITEL(v,x) writel((v), dmac_base + (x))
#define DMAC_READD(x)  (*(volatile unsigned long long*)(dmac_base+(x)))
#define DMAC_WRITED(v,x)  (*(volatile unsigned long long*)(dmac_base+(x)) = (v))

#define LLP_VALUE(lli) ((DTCM_BASE + LLI_SIZE*dmah + ((unsigned)(lli) - (unsigned)lli_base)) | dmac_ahb_master(DTCM_BASE) )

static int dmac_alloc(const char* name) {
	if((dmah = mobi_dma_request(name, MOBI_DMA_O_NONE)) < 0)
		return dmah;

	dmac_base = ioremap(DMAC_BASE, DMAC_SIZE);
	if(NULL == dmac_base) {
		mobi_dma_free(dmah);
		return -ENOMEM;
	}
	chan = dmac_base + dmah*sizeof(struct dmac_channel);

	lli_base = ioremap(DTCM_BASE+LLI_SIZE*dmah, LLI_SIZE);
	if(NULL == lli_base) {
		mobi_dma_free(dmah);
		iounmap(dmac_base);
		return -ENOMEM;
	}

	/* clear all int */
	DMAC_WRITEL(1 << dmah, MOBI_DMAC_CLEAR_TFR_OFFSET);
	DMAC_WRITEL(1 << dmah, MOBI_DMAC_CLEAR_BLOCK_OFFSET);
	DMAC_WRITEL(1 << dmah, MOBI_DMAC_CLEAR_SRC_TRAN_OFFSET);
	DMAC_WRITEL(1 << dmah, MOBI_DMAC_CLEAR_DST_TRAN_OFFSET);
	DMAC_WRITEL(1 << dmah, MOBI_DMAC_CLEAR_ERR_OFFSET);

	/* unmask only tfr and err int */
	DMAC_WRITEL(0x101 << dmah, MOBI_DMAC_MASK_TFR_OFFSET);
	DMAC_WRITEL(0x100 << dmah, MOBI_DMAC_MASK_BLOCK_OFFSET);
	DMAC_WRITEL(0x100 << dmah, MOBI_DMAC_MASK_SRC_TRAN_OFFSET);
	DMAC_WRITEL(0x100 << dmah, MOBI_DMAC_MASK_DST_TRAN_OFFSET);
	DMAC_WRITEL(0x101 << dmah, MOBI_DMAC_MASK_ERR_OFFSET);

	/* setup dmac channel */
	chan->cfg.r = 0;
	chan->cfg.b.protctl = 1;
	chan->cfg.b.hs_sel_src = DMAC_HS_SEL_SW;
	chan->cfg.b.hs_sel_dst = DMAC_HS_SEL_SW;
	/* limit AMBA burst length to 8 */
	chan->cfg.b.max_abrst = 8;

	return 0;
}

static void dmac_free(void) {
	iounmap(lli_base);
	iounmap(dmac_base);
	mobi_dma_free(dmah);
}

static inline void dmac_set_handshake(unsigned hs) {
	chan->cfg.b.hs_sel_src = DMAC_HS_SEL_HW;
	chan->cfg.b.hs_sel_dst = DMAC_HS_SEL_HW;
	chan->cfg.b.src_per = hs;
	chan->cfg.b.dest_per = hs;
}

static inline void wait_dmac(void) {
	unsigned chan_val = 1 << dmah;
	DMAC_WRITEL((chan_val<<8)|chan_val, MOBI_DMAC_CHANNEL_OFFSET);
	while(DMAC_READL(MOBI_DMAC_CHANNEL_OFFSET) & chan_val);
}

static inline unsigned dmac_ahb_master(dma_addr_t addr) {
	if(addr < APB_BUS_BASE)
		return DMAC_AHB_MASTER_0;
	if(addr < AHB_DMA_BUS_BASE)
		return DMAC_AHB_MASTER_1;
	return DMAC_AHB_MASTER_2;
}

static inline void dmac_process_list(void)
{
	chan->llp.r = LLP_VALUE(lli_base);

	chan->ctl.r = 0;
	chan->ctl.b.llp_src_en = 1;
	chan->ctl.b.llp_dst_en = 1;

	/* start dma */
	wait_dmac();
}

static inline dmac_lli_t* dmac_lli_alloc(dmac_lli_t* lli)
{
	if(!lli)
		return lli_base;
	/* check for DTCM overflow */
	if( (unsigned)(lli+1) - (unsigned)lli_base >= (cache_ptr - sizeof(*lli) - 4) ) {
		dmac_process_list();
		return lli_base;
	}
	lli->ctl.b.llp_dst_en = 1;
	lli->ctl.b.llp_src_en = 1;
	lli->llp.r = LLP_VALUE(lli+1);

	return lli+1;
}

static inline dmac_lli_t* dmac_lli_node(dmac_lli_t* prev, dma_addr_t src, dma_addr_t dst, unsigned length)
{
	static const dmac_ctl_t ctl_tmpl = {
		.b = {
			.dst_tr_width = DMAC_TR_WIDTH_32_BIT,
			.src_tr_width = DMAC_TR_WIDTH_32_BIT,
			.dst_msize = DMAC_BURST_SIZE_8,
			.src_msize = DMAC_BURST_SIZE_8,
			.tt_fc = DMAC_TTFC_M2P_DMAC,
		}
	};

	dmac_lli_t* lli = dmac_lli_alloc(prev);

	lli->sar.r = src;
	lli->dar.r = dst;
	lli->ctl.r = ctl_tmpl.r;
	lli->ctl.b.block_ts = length/4;
	lli->ctl.b.sms = dmac_ahb_master(src);
	lli->ctl.b.dms = dmac_ahb_master(dst);

	return lli;
}

static inline void dmac_cache_reset(void)
{
	cache_ptr = LLI_SIZE;
}

/* don't call more than once per lli allocation */
static inline dma_addr_t dmac_cache_value(u32 value)
{
	cache_ptr -= sizeof(value);
	*(u32*)(lli_base+cache_ptr) = value;
	return DTCM_BASE + dmah*LLI_SIZE + cache_ptr;
}

#endif
