#!/bin/bash 
# Copyright 2007 Maxim Corp.
#

CONFIG_FILE=$1
#WORK_DIR=${QUARC_ROOT}/dist/${DIST_NAME}
WORK_DIR=$2
MTD_PARTS=$3
KERN_SIZE_KB=$4

# let runconfig pass in it's tmp dir, which is based on it's 
# pid so it will be unique, if not the env file will be in 
# the current dir
if [ $# -eq 5 ]
then
	TMP_DIR=$5
else
	TMP_DIR=.
fi

SVN_TAG=$(cat ${WORK_DIR}/svntag);

MBOOTCFG_OUT=${TMP_DIR}/mboot.cfg

echo "Generating mboot config file: ${MBOOTCFG_OUT}"

# default console settings
CONSOLE_BAUDRATE=115200
CONSOLE_PARITY=N
CONSOLE_NBITS=8
CONSOLE_BITSTOP=1

function writet
{
	echo -e $* >> $MBOOTCFG_OUT
}

source ${CONFIG_FILE};
if [ $? -ne 0 ]
then
	echo -ne "ERROR\nConfig file is corrupted.\n";
	exit
fi

boot_devices_found=0

function version
{
	writet "ENV_VERSION_SECTION"
	writet "version	${SVN_TAG}"
	writet "ENV_END_SECTION"
	writet ""
}

function ddr_csr
{
	if [ -n "${CONFIG_RAMINIT_REG_FILE}" ]
	then
		writet "# this reg file is used to initialize the DDR"
		writet "ENV_CSR_SECTION"
		writet "readfile ${CONFIG_RAMINIT_REG_FILE}"
		writet "ENV_END_SECTION"
		writet ""
	fi
}

# normally, the memory partitioning is in the reg file. however, if 
# we put the CSR section after the reg file csr section we can overwrite
# the values and get our own!  since this is based on the linux ram size
# config option we could always add this section without adding config
# option since we don't hurt anything by writing the same values
function partition_mem
{

	if [ -n "${CONFIG_RAMINIT_REG_FILE}" ]
	then
		ramsize=$((${CONFIG_LINUX_RAM_SIZE}*1024*1024))
		partsize=$((${ramsize}/${CONFIG_RAM_SEGMENT_SIZE}))
		partsize=`printf "0x%x" ${partsize}`
		partsize_plus1=$(((${ramsize}+${CONFIG_RAM_SEGMENT_SIZE})/${CONFIG_RAM_SEGMENT_SIZE}))
		partsize_plus1=`printf "0x%x" ${partsize_plus1}`
		writet "# is used to define the memory partitioning"
		writet "ENV_CSR_SECTION"

cat >> ${MBOOTCFG_OUT} <<ENDCMDS
write 12 2f8 2 ${partsize_plus1}
write 12 2fa 2 ${partsize_plus1}
write 12 2fc 2 ${partsize}
ENDCMDS

		writet "ENV_END_SECTION"
		writet ""
	fi
}

function pll0init
{

	if [ "X${CONFIG_CORE_CLOCK_648MHZ}" == "Xy" ]
	then
		writet "# Here is the PLL0 config for 648Mhz which yields 216Mhz ARM and 216Mhz AVC"
		writet "ENV_CLOCK_SECTION"

cat >> ${MBOOTCFG_OUT} <<ENDCMDS
comment bypass PLL
write      08 010 4 3cb31b0
comment set Scaler0
write      08 020 4 c006
write      08 020 4 d106
comment wait for lock bit to be set in PLL0
poll       08 010 4 0004 0004 100 1000
comment use PLL output
write      08 010 4 1cb31b0
ENDCMDS

	elif [ "X${CONFIG_CORE_CLOCK_600MHZ}" == "Xy" ]
	then
		writet "# Here is the PLL0 config for 600Mhz which yields 267Mhz ARM and 200Mhz AVC"
		writet "ENV_CLOCK_SECTION"

cat >> ${MBOOTCFG_OUT} <<ENDCMDS
comment bypass PLL
write 8 010 4 03CB3190
comment set Scaler0
write 8 020 4 00008006
write 8 020 4 00009106
comment wait for lock bit to be set in PLL0
poll  8 010 4 00000004 00000004
comment use PLL output
write 8 010 4 01CB3190
ENDCMDS

	else

		writet "# PLL0 config for 540Mhz which yields 240Mhz ARM and 180Mhz AVC"

		if [ "X${CONFIG_CLKSRC_USB}" == "Xy" ] || \
			[ "X${CONFIG_CLKSRC_DIRECT}" == "Xy" -a ${CONFIG_CLKSRC_DIRECT_RATE} -eq 24000000 ]
		then

			writet "# PLL0 input clock is 12MHz"
			writet "ENV_CLOCK_SECTION"

cat >> ${MBOOTCFG_OUT} <<ENDCMDS
comment bypass PLL
write 8 010 4 03CB72D0
comment set Scaler0
write 8 020 4 00008006
write 8 020 4 00009106
comment wait for lock bit to be set in PLL0
poll  8 010 4 00000004 00000004
comment use PLL output
write 8 010 4 01CB72D0
ENDCMDS

		elif [ "X${CONFIG_CLKSRC_DIRECT}" == "Xy" -a ${CONFIG_CLKSRC_DIRECT_RATE} -eq 27000000 ]
		then

			writet "# PLL0 input clock is 13.5MHz"
			writet "ENV_CLOCK_SECTION"

cat >> ${MBOOTCFG_OUT} <<ENDCMDS
comment bypass PLL
write 8 010 4 03CB3140
comment set Scaler0
write 8 020 4 00008006
write 8 020 4 00009106
comment wait for lock bit to be set in PLL0
poll  8 010 4 00000004 00000004
comment use PLL output
write 8 010 4 01CB3140
ENDCMDS

		else
			echo "ERROR:  Unsupported direct clock source rate: ${CONFIG_CLKSRC_DIRECT_RATE}"
			echo "Contact Maxim Support for details"
			exit
		fi
	fi

	writet "ENV_END_SECTION"
	writet ""
}

function init_bridges
{

	writet "# init the memory bridges"
	writet "ENV_ARM_SECTION"

cat >> ${MBOOTCFG_OUT} <<ENDCMDS
comment write the partition ID to the Lpartition on first bridge
write   0x0c000010 0x7f
comment init the first bridge
write   0x0c000014 0x510
write   0x0c00001c 0x1
comment write the partition ID to the Lpartition on second bridge
write   0xcc000010 0x7f
comment init the second bridge
write   0xcc000014 0x500
write   0xcc00001c 0x1
ENDCMDS

	writet "ENV_END_SECTION"
	writet ""
}

function linux
{

	# there is a predefined cmdline which is constructed here and then any usr
	# cmdline args are tacked on the end
	cmdline=""

	# console
	if [ "X${CONFIG_UART_DEVICE}" == "Xy" ] 
	then
		parity=`echo ${CONFIG_UART_PARITY} | tr '[:upper:]' '[:lower:]'`
		cmdline="${cmdline} console=ttyS0,${CONFIG_UART_BAUDRATE}${parity}${CONFIG_UART_NBITS}"
	fi
	# memory
	cmdline="${cmdline} mem=${CONFIG_SYSTEM_RAM_SIZE}M memmap=${CONFIG_LINUX_RAM_SIZE}M@${CONFIG_SYSTEM_RAM_SIZE}M"

	# direct clk source rate if selected; otherwise kernel default to 12MHz
	if [ "X${CONFIG_CLKSRC_DIRECT}" == "Xy" ] 
	then
		# the DIRECT_RATE is divided by 2 before entering the chip, clock driver will divide this in kernel
		cmdline="${cmdline} clk_in=${CONFIG_CLKSRC_DIRECT_RATE}"
	fi

	# user defined
	cmdline="${cmdline} ${CONFIG_MBOOT_LINUX_CMDLINE}"

	writet "ENV_LINUX_SECTION" 
	writet "kernel	${CONFIG_MBOOT_LINUX_KERNADDR}"	
	writet "initrd	${CONFIG_MBOOT_LINUX_INITRDADDR}"	
	writet "cmdline	${cmdline}"
	writet "ENV_END_SECTION"
	writet ""
}

# Calculate the size of mboot, kernel and initrd partitions
# from an MTD partition table. If the partition is not found
# it will be set to 0. This function expects the mboot partition
# to be named mboot1, the kernel partition to be kernel1 and
# the initrd partition to be initrd1. If there are multiple
# partitions for one or the other they will be skipped.
function calc_parts
{	local offset=0;

	if [ -z "${MTD_PARTS}" ]; then
		echo -ne "ERROR\nInvalid partition table for the boot device.\n";
		return 1;
	fi
	MBOOT_PART_SIZE_B=0;
	MBOOT_PART_OFFSET=0;
	KERNEL_PART_SIZE_B=0;
	KERNEL_PART_OFFSET=0;
	INITRD_PART_SIZE_B=0;
	INITRD_PART_OFFSET=0;
	OLD_IFS=${IFS};
	IFS=',';
	for partition in ${MTD_PARTS}; do
		name=$(echo ${partition} | sed -e 's/[^\(]*[\(]\([^\)]*\).*/\1/');
		size=$(echo ${partition} | sed -e 's/\([^\(]*\).*/\1/');
		case "${size:${#size}-1}" in
			"k")	size=$((${size:0:${#size}-1}*1024));;
			"M")	size=$((${size:0:${#size}-1}*1024*1024));;
			"G")	size=$((${size:0:${#size}-1}*1024*1024*1024));;
			[0-9])	;;
			"-")	size=0;;# this means all remaining space, so set to 0 to avoid offset 
						    # addition error below, offset doesn't matter at this point anyway
			*)
				IFS=${OLD_IFS};
				echo -ne "ERROR\nUnrecognized partition size ${size} for partition ${name}.\n";
				return 2;
				;;
		esac
		case ${name} in
			"mboot1")
				MBOOT_PART_SIZE_B=${size};
				MBOOT_PART_OFFSET=${offset};
				;;
			"kernel1")
				KERNEL_PART_SIZE_B=${size};
				KERNEL_PART_OFFSET=${offset};
				;;
			"initrd1")
				INITRD_PART_SIZE_B=${size};
				INITRD_PART_OFFSET=${offset};
				;;
		esac
		offset=$((offset+size));
	done
	IFS=${OLD_IFS};
	return 0;
}

function bootparams
{
	if [ "X${CONFIG_BOOT_DEVICE_RAM}" == "Xy" ] || \
		[ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ] || \
		[ "X${CONFIG_BOOT_DEVICE_NOR}" == "Xy" ]  || \
		[ "X${CONFIG_BOOT_DEVICE_JTAG}" == "Xy" ] 
	then
		calc_parts
		if [ $? -ne 0 ]; then
			return 1;
		fi

		if [ "X${CONFIG_BOOT_DEVICE_RAM}" == "Xy" ] 
		then
			let "boot_devices_found += 1"
			writet "ENV_BOOTPARAMS_SECTION" 
			writet "device	ddr"
			writet "kernel	`printf "0x%x" ${KERNEL_PART_OFFSET}` uncompressed 1 0"
			writet "initrd	`printf "0x%x" ${INITRD_PART_OFFSET}` uncompressed 1 0"
			writet "ENV_END_SECTION"
			writet ""
		fi

		if [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ] 
		then
			if [ ${CONFIG_NAND_PAGESIZE} -eq 2048 ]
			then
				let "nand_blk_size = ${CONFIG_NAND_PAGESIZE} * 64"
			else
				let "nand_blk_size = ${CONFIG_NAND_PAGESIZE} * 32"
			fi

			let "boot_devices_found += 1"
			writet "ENV_BOOTPARAMS_SECTION" 
			writet "device		nand"

			if [ ${CONFIG_MBOOT_KERN_COPIES} -gt 1 ]
			then
				# determine number of block needed for image, use as stride to skip if get bad ecc
				let "kernel_blk_cnt = ${KERNEL_PART_SIZE_B} % ${nand_blk_size}"
				if [ ${kernel_blk_cnt} != 0 ]
				then
					let "kernel_blk_cnt = ${KERNEL_PART_SIZE_B} / ${nand_blk_size}"
					let "kernel_blk_cnt += 1"
				else
					let "kernel_blk_cnt = ${KERNEL_PART_SIZE_B} / ${nand_blk_size}"
				fi

				writet "kernel `printf "0x%x" ${KERNEL_PART_OFFSET}` uncompressed ${CONFIG_MBOOT_KERN_COPIES} $kernel_blk_cnt"
			else
				writet "kernel `printf "0x%x" ${KERNEL_PART_OFFSET}` uncompressed 1 0"
			fi

			if [ ${CONFIG_MBOOT_INITRD_COPIES} -gt 1 ]
			then
				let "initrd_blk_cnt = ${INITRD_PART_SIZE_B} % ${nand_blk_size}"
				if [ ${initrd_blk_cnt} != 0 ]
				then
					let "initrd_blk_cnt = ${INITRD_PART_SIZE_B} / ${nand_blk_size}"
					let "initrd_blk_cnt += 1"
				else
					let "initrd_blk_cnt = ${INITRD_PART_SIZE_B} / ${nand_blk_size}"
				fi

				writet "initrd `printf "0x%x" ${INITRD_PART_OFFSET}` uncompressed ${CONFIG_MBOOT_INITRD_COPIES} $initrd_blk_cnt"
			else
				writet "initrd `printf "0x%x" ${INITRD_PART_OFFSET}` uncompressed 1 0"
			fi
			writet "ENV_END_SECTION"
			writet ""
		fi

		if [ "X${CONFIG_BOOT_DEVICE_NOR}" == "Xy" ] 
		then
			let "boot_devices_found += 1"
			writet "ENV_BOOTPARAMS_SECTION" 
			writet "device		nor"
			writet "kernel	`printf "0x%x" ${KERNEL_PART_OFFSET}` uncompressed 1 0"
			writet "initrd	`printf "0x%x" ${INITRD_PART_OFFSET}` uncompressed 1 0"
			writet "ENV_END_SECTION"
			writet ""
		fi

		if [ "X${CONFIG_BOOT_DEVICE_JTAG}" == "Xy" ] 
		then
			let "boot_devices_found += 1"
			writet "ENV_BOOTPARAMS_SECTION" 
			writet "device	jtag"
			writet "kernel	`printf "0x%x" ${KERNEL_PART_OFFSET}` uncompressed 1 0"
			writet "initrd	`printf "0x%x" ${INITRD_PART_OFFSET}` uncompressed 1 0"
			writet "ENV_END_SECTION"
			writet ""
		fi
	else

		if [ "X${CONFIG_BOOT_DEVICE_XMODEM}" == "Xy" ] 
		then
			let "boot_devices_found += 1"
			writet "ENV_BOOTPARAMS_SECTION" 
			writet "device	xmodem"
			writet "ENV_END_SECTION"
			writet ""
		fi

		if [ "X${CONFIG_BOOT_DEVICE_ENET_TFTP}" == "Xy" ] 
		then
			let "boot_devices_found += 1"
			writet "ENV_BOOTPARAMS_SECTION" 
			writet "device	tftp"
			writet "ENV_END_SECTION"
			writet ""
		fi

		if [ "X${CONFIG_BOOT_DEVICE_USB}" == "Xy" ] 
		then
			let "boot_devices_found += 1"
			writet "ENV_BOOTPARAMS_SECTION" 
			writet "device	usb"
			writet "ENV_END_SECTION"
			writet ""
		fi


		if [ "X${CONFIG_BOOT_DEVICE_MEMTEST}" == "Xy" ] || [ "X${CONFIG_MBOOT_MEMCHECK}" == "Xy" ]
		then
			writet "ENV_BOOTPARAMS_SECTION" 
			writet "device memtest"
			writet "ENV_END_SECTION"
			writet ""
		fi
		if [ "X${CONFIG_BOOT_DEVICE_DDRTEST}" == "Xy" ]
		then
			writet "ENV_BOOTPARAMS_SECTION" 
			writet "device ddrtest"
			writet "ENV_END_SECTION"
			writet ""
		fi
	fi
}

function driver
{

	if [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ] 
	then
		writet "ENV_DRIVER_SECTION" 
		writet "device		nand"
		writet "chipsel		${CONFIG_NAND_CHIPSELECT}"
		writet "buswidth	${CONFIG_NAND_BUSWIDTH}"
		writet "pagesize	${CONFIG_NAND_PAGESIZE}"
		writet "chipsize	${CONFIG_NAND_CHIPSIZE}"
		writet "ENV_END_SECTION"
		writet ""
	fi

	if [ "X${CONFIG_BOOT_DEVICE_NOR}" == "Xy" ] 
	then
		writet "ENV_DRIVER_SECTION" 
		writet "device		nor"
		writet "chipsel		${CONFIG_NOR_CHIPSELECT}"
		if [ "X${CONFIG_NOR_AD_22}" == "Xy" ]
		then
			writet "chipsel_a23	15"
		else
			writet "chipsel_a23	${CONFIG_NOR_CHIPSELECT_A23}"
		fi
		writet "chipsize	${CONFIG_NOR_CHIPSIZE}"
		writet "ENV_END_SECTION"
		writet ""
	fi

	if [ "X${CONFIG_BOOT_DEVICE_XMODEM}" == "Xy" ] 
	then
		writet "ENV_DRIVER_SECTION" 
		writet "device		xmodem"
		writet "baudrate	${CONFIG_UART_BAUDRATE}"
		writet "nbits		${CONFIG_UART_NBITS}"
		writet "parity		${CONFIG_UART_PARITY}"
		writet "stopbits	${CONFIG_UART_STOPBITS}"
		writet "ENV_END_SECTION"
		writet ""
	fi

	if [ "X${CONFIG_BOOT_DEVICE_ENET_TFTP}" == "Xy" ] 
	then
		writet "ENV_DRIVER_SECTION" 
		writet "device		tftp"

		if [ "X${CONFIG_NETWORK_MAC_INCONFIG}" == "Xy" ]
		then
			writet "mac			${CONFIG_NETWORK_MACADDR}" 
		else
			writet "mac			spi" 
		fi

		if [ "X${CONFIG_NETWORK_IPADDR_STATIC}" == "Xy" ]
		then
			writet "ip			${CONFIG_NETWORK_IPADDR}"
		elif [ "X${CONFIG_NETWORK_IPADDR_INSPI}" == "Xy" ]
		then
			writet "ip			spi"
		else
			writet "ip			dynamic"
		fi

		writet "serverip	${CONFIG_TFTP_SERVER_IP}"
		writet "gateway		${CONFIG_TFTP_GATEWAY_IP}"
		writet "netmask		${CONFIG_TFTP_NETMASK}"
		writet "kernel_filename		${CONFIG_TFTP_KERNEL_FILENAME}"
		writet "initrd_filename		${CONFIG_TFTP_INITRD_FILENAME}"
		writet "ENV_END_SECTION"
		writet ""
	fi

	if [ "X${CONFIG_BOOT_DEVICE_USB}" == "Xy" ] 
	then
		writet "ENV_DRIVER_SECTION" 
		writet "device		usb"
		writet "ENV_END_SECTION"
		writet ""
	fi


	if [ "X${CONFIG_BOOT_DEVICE_MEMTEST}" == "Xy" ] || [ "X${CONFIG_MBOOT_MEMCHECK}" == "Xy" ]
	then
		writet "ENV_DRIVER_SECTION" 
		writet "device memtest"
		writet "size $((CONFIG_LINUX_RAM_SIZE*1024*1024))"
		writet "ENV_END_SECTION"
		writet ""
	fi
	if [ "X${CONFIG_BOOT_DEVICE_DDRTEST}" == "Xy" ]
	then
		writet "ENV_DRIVER_SECTION" 
		writet "device ddrtest"
		writet "size $((CONFIG_LINUX_RAM_SIZE*1024*1024))"
		writet "ENV_END_SECTION"
		writet ""
	fi
}

function bootdevice
{
	writet "ENV_BOOTDEVICE_SECTION" 
	if [ ${boot_devices_found} -gt 1 ]
	then
		echo "ERROR: multiple boot devices not supported anymore.\n"
		exit 1;
	else
		if [ "X${CONFIG_BOOT_DEVICE_RAM}" == "Xy" ] 
		then
			writet "device	ddr"
		elif [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ] 
		then
			writet "device	nand"
		elif [ "X${CONFIG_BOOT_DEVICE_NOR}" == "Xy" ] 
		then
			writet "device	nor"
		elif [ "X${CONFIG_BOOT_DEVICE_XMODEM}" == "Xy" ] 
		then
			writet "device	xmodem"
		elif [ "X${CONFIG_BOOT_DEVICE_JTAG}" == "Xy" ] 
		then
			writet "device	jtag"
		elif [ "X${CONFIG_BOOT_DEVICE_ENET_TFTP}" == "Xy" ] 
		then
			writet "device	tftp"
		elif [ "X${CONFIG_BOOT_DEVICE_MEMTEST}" == "Xy" ] 
		then
			writet "device memtest"
		elif [ "X${CONFIG_BOOT_DEVICE_USB}" == "Xy" ] 
		then
			writet "device usb"
		elif [ "X${CONFIG_BOOT_DEVICE_DDRTEST}" == "Xy" ] 
		then
			writet "device ddrtest"
		else
			writet "device	query"
		fi
	fi
	writet "ENV_END_SECTION"
	writet ""
}

function system
{
	# direct clk source rate if selected and it is not 27Mhz, let the env know
	if [ "X${CONFIG_CLKSRC_DIRECT}" == "Xy" ] 
	then
		if [ ${CONFIG_CLKSRC_DIRECT_RATE} -ne 27000000 ]
		then
			writet "ENV_SYSTEM_SECTION" 
			writet "xin ${CONFIG_CLKSRC_DIRECT_RATE}"
			writet "ENV_END_SECTION"
			writet ""
		fi
	fi
}

function env_end
{
	writet "\nENV_END"
}

rm -f ${MBOOTCFG_OUT}

#function calls
version
# Cannot initialize clocks on FPGA
if [ "X${CONFIG_BOARD_TYPE_NAME}" != "Xmerlinproto" ]; then
	pll0init
fi
ddr_csr
partition_mem
init_bridges
linux
bootparams
driver
bootdevice
system
env_end