#include <stdio.h>
#include <string.h>
#include <i2c.h>
#include <stdlib.h>

void help(void)
{
	printf("ic2exp <options>  : i2c port expander utility\n");
	printf(" where options are:\n");
	printf("\t--bus <i2c_bus>         : Specify I2C bus to open(default /dev/i2c-0).\n");
	printf("\t--dev <0xNN>            : Specify device address on I2C bus");
	printf("\t--verbose               : don't display any messages\n");
	printf("\t--set <bit>             : Set <bit> in port expander\n");
	printf("\t--clear <bit>           : Clear <bit> in port expander\n");
	printf("\t--readbit <bit>         : print the value of a single bit\n");
	printf("\t--read                  : "
			"Read all bit in port expander(outputs 0x<7..0> 0x<15..8>)\n");
	printf("\t--raw-write 0x<7..0> 0x<15..8> : "
			"Overwrite all bits with new data.  Use with caution!\n");
}

int main(int argc,char *argv[])
{
	int fd;
	char *i2c_bus = NULL;
	int i2c_addr = -1;
	int i = 1, ret = 0;
	int write = 0, read = 0;
	int8_t set_bit = -1;
	int8_t clear_bit = -1;
	int8_t read_bit = -1;
	char verbose = 0;
	unsigned char buf[2];
	memset(buf, 0, 2);

	if (argc == 1) {
		help();
		return(0);
	}
	while (i < argc) {
		if ((strcmp("--help", argv[i]) == 0) || 
				(strcmp("-h", argv[i]) == 0)) {
			help();
			return(0);
		}
		else if(strcmp(argv[i],"--bus")==0)
			i2c_bus = argv[++i];
		else if(strcmp(argv[i],"--dev")==0) {
			i2c_addr = strtol(argv[++i], NULL, 16);
		}
		else if (strcmp("--set", argv[i]) == 0) {
			set_bit = atoi(argv[++i]);
			write = 1;
			if (set_bit < 0 || set_bit > 15) {
				fprintf(stderr, "Invalid bit, must be 0-15 inclusive: %d\n", 
						set_bit);
				return -1;
			}
		}
		else if (strcmp("--clear", argv[i]) == 0) {
			clear_bit = atoi(argv[++i]);
			write = 1;
			if (clear_bit < 0 || clear_bit > 15) {
				fprintf(stderr, "Invalid bit, must be 0-15 inclusive: %d\n", 
						clear_bit);
				return -1;
			}
		}
		else if (strcmp("--readbit", argv[i]) == 0) {
			read_bit = atoi(argv[++i]);
			read = 1;
			if (read_bit < 0 || read_bit > 15) {
				fprintf(stderr, "Invalid bit, must be 0-15 inclusive: %d\n", 
						read_bit);
				return -1;
			}
		}
		else if (strcmp("--read", argv[i]) == 0) {
			read = 1;
		}
		else if (strcmp("--raw-write", argv[i]) == 0) {
			if (i == argc - 1) {
				fprintf(stderr, "%s: please specify bytes to raw write (up to %d)\n", 
						argv[0], 2);
				return -1;
			}
			for (i++; i < argc; i++) {
				int x;
				if (sscanf(argv[i], "%i", &x) != 1 || (x & 0xff) != x) {
					fprintf(stderr, "%s: invalid byte value '%s', aborting\n", 
							argv[0], argv[i]);
					return -1;
				}
				buf[write++] = (x & 0xff);
			}
		}
		else if (strcmp("--verbose", argv[i]) == 0) {
			verbose = 1;
		}
		i++;
	}

	if (i2c_addr == -1) {
		fprintf(stderr, "Error, you must specify and i2c device(--dev)\n");
		return -1;
	}

	fd = i2c_open(i2c_bus, i2c_addr, I2CDEV_STD);
	if (fd < 0) {   
		fprintf(stderr, "%s: failed to open I2C device: error %d\n", argv[0], fd);
		return -1;
	}

	if (write < 2) {
		/* don't read if doing raw write */
		if (verbose)
			printf("Read  address 0x%x on %s: ", i2c_addr, i2c_bus);
		ret = i2c_read_block(fd, 0x0, buf, 2);
		if (ret < 0) {
			fprintf(stderr, "%s: read failed with error %d\n", argv[0], ret);
			i2c_close(fd);
			return -1;
		}
		if (verbose)
			printf("0x%02x 0x%02x\n", buf[0], buf[1]);
	}

	if (clear_bit != -1) 
		*(short*)buf &= ~(1 << clear_bit);
	else if (set_bit != -1) 
		*(short*)buf |= (1 << set_bit);

	if (write) {
		if (verbose)
			printf("Write address 0x%x on %s: 0x%02x 0x%02x\n",
					i2c_addr, i2c_bus, buf[0], buf[1]);
		ret = i2c_write_block(fd, buf[0], &buf[1], write);
		if (ret < 0) {
			fprintf(stderr, "%s: write failed with error %d\n", argv[0], ret);
			i2c_close(fd);
			return -1;
		}
	}
	else if (read) {
		if (read_bit != -1)
			printf("%d\n", 
					(*(short*)buf >> read_bit) & 0x1);
		else {
			if (verbose)
				printf("LSB(<7..0>) 0x%02x MSB(<15..8>) 0x%02x\n", 
						buf[0], buf[1]);
			else
				printf("0x%02x 0x%02x\n", buf[0], buf[1]);
		}
	}
	
	i2c_close(fd);
	return 0;
}
