/**i2c.c
 * I2C library based on Linux I2C kernel driver
 */

/*******************************************************************************
 * 
 * The content of this file or document is CONFIDENTIAL and PROPRIETARY
 * to Mobilygen Corporation.  It is subject to the terms of a
 * License Agreement between Licensee and Mobilygen Corporation.
 * restricting among other things, the use, reproduction, distribution
 * and transfer.  Each of the embodiments, including this information and
 * any derivative work shall retain this copyright notice.
 * 
 * Copyright 2004 Mobilygen Corporation.
 * All rights reserved.
 * 
 * QuArc is a registered trademark of Mobilygen Corporation.
 * 
 *******************************************************************************/


#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <linux/i2c.h>
#include <linux/i2c-dev.h>
#include "i2c.h"

#define I2CFLAGS_SCCB 0x1
#define NO_PAGE_INDEX -1

typedef struct i2c_info_struct {
	char i2c_bus[I2C_BUSDEV_MAXNAME];
	int dev_address;
	int flags;
	int subaddr_size;
	i2c_handle_t handle;
	struct i2c_info_struct *next;
} i2c_info_t;

static i2c_info_t *i2c_info_first=NULL;

#define GET_INFO_FROM_HANDLE(i2c_info_t_p,int_handle) \
	do \
{   for(i2c_info_t_p=i2c_info_first; \
		i2c_info_t_p!=NULL; \
		i2c_info_t_p=i2c_info_t_p->next) \
	if(i2c_info_t_p->handle==int_handle) break; \
	if(i2c_info_t_p==NULL) \
	return I2C_ERR_HANDLE; \
} while (0)


static int i2c_send(i2c_info_t *i2c_p, int addr, char read, 
		int nBytes, char *data, int page)
{   
	struct i2c_msg msg[3];
	int flags=0,fd;
	char addrt[2]={(addr>>8) & 0xFF, addr & 0xFF};

	do
	{   
		fd=open(i2c_p->i2c_bus,O_RDWR);
		usleep(10000);
	} while((fd<0) && (errno==EBUSY));

	if(fd<0)
		return I2C_ERR_DRV;

	/*
	fprintf(stderr, "\n[libi2c] %s\n", (read ? "read" : "write"));
	fprintf(stderr, "[libi2c] page is 0x%x\n", page);
	fprintf(stderr, "[libi2c] address is 0x%x\n", addr);
	fprintf(stderr, "[libi2c] nBytes is 0x%x\n", nBytes);
	if (!read)
		fprintf(stderr, "[libi2c] data[0] = 0x%x\n", (unsigned char *)data[0]);
	*/
	if (i2c_p->flags & I2CFLAGS_SCCB && read) { //SCCB protocol read access
		struct i2c_rdwr_ioctl_data msgtable={msg,1};
		if(i2c_p->dev_address>0xff)   /* 10bits address mode */
			flags=I2C_M_TEN;

		msg[0].addr=i2c_p->dev_address>>1;
		msg[0].flags=flags;
		msg[0].len=i2c_p->subaddr_size;
		msg[0].buf=(unsigned char *)(addrt+2-i2c_p->subaddr_size);
		if (ioctl(fd,I2C_RDWR,&msgtable)<0) {   
			close(fd);
			return I2C_ERR_TX;
		};
		msg[0].addr=i2c_p->dev_address>>1;
		msg[0].flags=flags | (read?I2C_M_RD:I2C_M_NOSTART);
		msg[0].len=nBytes;
		msg[0].buf=(unsigned char *)data;
		if (ioctl(fd,I2C_RDWR,&msgtable)<0) {   
			close(fd);
			return I2C_ERR_TX;
		};
		close(fd);
		return I2C_SUCCESS;
	}
	else { //standart i2c
		int i = 0;
		if (i2c_p->dev_address > 0xff)   /* 10bits address mode */
			flags = I2C_M_TEN;

		if (page != NO_PAGE_INDEX) {
			//fprintf(stderr, "[libi2c] got a valid page\n");
			if (read) {
				msg[i].addr = i2c_p->dev_address >> 1;
				msg[i].flags = flags;
				msg[i].len = 0;
				msg[i].buf = (unsigned char *)(addrt + 2 - i2c_p->subaddr_size);
				i++;
			}
			else {
				unsigned char pgindex[1]={page & 0xff};
				msg[i].addr = i2c_p->dev_address >> 1;
				msg[i].flags = flags;
				msg[i].len = i2c_p->subaddr_size;
				msg[i].buf = (unsigned char *)pgindex;
				i++;

				if (addr >= 0 && i2c_p->subaddr_size > 0) {
					//fprintf(stderr, "[libi2c] valid addr and subaddr_size\n");
					msg[i].addr = i2c_p->dev_address >> 1;
					msg[i].flags = flags;
					msg[i].len = i2c_p->subaddr_size;
					msg[i].buf = (unsigned char *)(addrt + 2 - i2c_p->subaddr_size);
					i++;
				}
			}
		}
		else {
			if (addr >= 0 && i2c_p->subaddr_size > 0) {
				//fprintf(stderr, "[libi2c] valid addr and subaddr_size\n");
				msg[i].addr = i2c_p->dev_address >> 1;
				msg[i].flags = flags;
				msg[i].len = i2c_p->subaddr_size;
				msg[i].buf = (unsigned char *)(addrt + 2 - i2c_p->subaddr_size);
				i++;
			}
		}

		msg[i].addr = i2c_p->dev_address >> 1;
		msg[i].flags = flags | (read ? I2C_M_RD : (i ? I2C_M_NOSTART : 0));
		msg[i].len = nBytes;
		msg[i].buf = (unsigned char *)data;
		i++;

		{
			struct i2c_rdwr_ioctl_data msgtable = {msg,i};
			if (ioctl(fd,I2C_RDWR,&msgtable) < 0) {
				close(fd);
				return I2C_ERR_TX;
			}
			close(fd);
		}
		return I2C_SUCCESS;
	}
}

i2c_handle_t i2c_open(char * busdev,int dev_address, i2c_dev_t type)
{   
	static volatile i2c_handle_t handle_index=1;
	i2c_info_t *i2c_p,*i2c_p_last;

	i2c_p=(i2c_info_t *)malloc(sizeof(i2c_info_t));
	if (i2c_p==NULL)
		return I2C_ERR_MEM; 
	i2c_p->dev_address=dev_address;
	if(busdev==NULL)
		strncpy(i2c_p->i2c_bus,I2C_DEFAULT_BUSDEV,I2C_BUSDEV_MAXNAME);
	else strncpy(i2c_p->i2c_bus,busdev,I2C_BUSDEV_MAXNAME);
	i2c_p->handle=handle_index++;
	i2c_p->next=NULL;
	i2c_p->flags=0;
	switch(type)
	{   
		case I2CDEV_STD:
			i2c_p->subaddr_size=1;
			break;
		case I2CDEV_ADDR16:
			i2c_p->subaddr_size=2;
			break;
		case I2CDEV_NOSUBADDR:
			i2c_p->subaddr_size=0;
			break;
		case I2CDEV_SCCB:
			i2c_p->flags|=I2CFLAGS_SCCB;
			i2c_p->subaddr_size=1;
			break;
		default:
			free(i2c_p);
			return I2C_ERR_BADPARAM;
	};
	if(i2c_info_first!=NULL) {   
		for(i2c_p_last=i2c_info_first;i2c_p_last->next!=NULL;i2c_p_last=i2c_p_last->next);
		i2c_p_last->next=i2c_p;
	} 
	else {   
		i2c_info_first=i2c_p;
	};
	return i2c_p->handle;
}

int i2c_read(i2c_handle_t i2c_h, int sub_address, unsigned char *datap)
{   
	i2c_info_t *i2c_p;
	GET_INFO_FROM_HANDLE(i2c_p,i2c_h);
	return i2c_send(i2c_p, sub_address, 1, 1, (char *)datap, NO_PAGE_INDEX);
}

int i2c_page_read(i2c_handle_t i2c_h, int page, 
		int sub_address, unsigned char *datap)
{   
	return(i2c_read_page_block(i2c_h, page, 
				sub_address, datap, 1));
}

int i2c_write(i2c_handle_t i2c_h, int sub_address, unsigned char data)
{   
	i2c_info_t *i2c_p;
	GET_INFO_FROM_HANDLE(i2c_p,i2c_h);
	return i2c_send(i2c_p, sub_address, 0, 1, (char *)&data, NO_PAGE_INDEX);
}

int i2c_page_write(i2c_handle_t i2c_h, int page, 
		int sub_address, unsigned char data)
{   
	i2c_info_t *i2c_p;
	GET_INFO_FROM_HANDLE(i2c_p,i2c_h);
	if (page < 0) {
		fprintf(stderr, "Invalid page: 0x%x\n", page);
		return I2C_ERR_DRV;
	}
	return i2c_send(i2c_p, sub_address, 0, 1, (char *)&data, page);
}

int i2c_read_block(i2c_handle_t i2c_h, int start_sub_address, 
		unsigned char *datap, int nBytes)
{   
	i2c_info_t *i2c_p;
	GET_INFO_FROM_HANDLE(i2c_p,i2c_h);
	return i2c_send(i2c_p,start_sub_address,1,nBytes,(char *)datap, NO_PAGE_INDEX);
}

int i2c_read_page_block(i2c_handle_t i2c_h, int page, int start_sub_address, 
		unsigned char *datap, int nBytes)
{   
	i2c_info_t *i2c_p;
	int ret = 0;
	char ac = (char) (start_sub_address & 0xff);
	GET_INFO_FROM_HANDLE(i2c_p,i2c_h);
	if (page < 0) {
		fprintf(stderr, "Invalid page: 0x%x\n", page);
		return I2C_ERR_DRV;
	}
	/* doing read on devices with pages is a bit tricky.  it works like this
	 * a 3 phase write <device> <page> <subaddress>
	 * so we just send the page as the subaddress and the subaddress as the
	 * first(and only) data byte.  i2c subaddress are just data anyway so
	 * don't do anything special except fix the order
	 */
	i2c_send(i2c_p, page, 0, 1, &ac, NO_PAGE_INDEX);

	/* to do the read, we just have to put the device address onto the 
	 * bus and then read data.  no subaddress.  so here we give a bogus
	 * subaddress which will be ignored when we pass in a page and a 
	 * read request to i2c_send
	 */
	ret = i2c_send(i2c_p, 0x0, 1, nBytes, (char *)datap, page);
	return ret;
}

int i2c_write_block(i2c_handle_t i2c_h, int start_sub_address, 
		unsigned char *datap, int nBytes)
{   
	i2c_info_t *i2c_p;
	GET_INFO_FROM_HANDLE(i2c_p,i2c_h);
	return i2c_send(i2c_p,start_sub_address,0,nBytes,(char *)datap, NO_PAGE_INDEX);
}

int i2c_write_page_block(i2c_handle_t i2c_h, int page, int start_sub_address, 
		unsigned char *datap, int nBytes)
{   
	i2c_info_t *i2c_p;
	GET_INFO_FROM_HANDLE(i2c_p,i2c_h);
	if (page < 0) {
		fprintf(stderr, "Invalid page: 0x%x\n", page);
		return I2C_ERR_DRV;
	}
	return i2c_send(i2c_p,start_sub_address,0,nBytes,(char *)datap, page);
}

int i2c_close(i2c_handle_t i2c_h)
{   
	i2c_info_t *i2c_p,*i2c_p_last;

	GET_INFO_FROM_HANDLE(i2c_p,i2c_h);
	if(i2c_p==i2c_info_first) 
		i2c_info_first=i2c_info_first->next;
	else 
	{   for(i2c_p_last=i2c_info_first;i2c_p_last->next!=i2c_p;i2c_p_last=i2c_p_last->next);
		i2c_p_last->next=i2c_p->next;
	};
	free(i2c_p);
	return I2C_SUCCESS;
}
