/*  ftpd.h: Prototypes for BetaFTPD
    Copyright (C) 1999-2000 Steinar H. Gunderson

    This program is is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 2 if the
    License as published by the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifndef _FTPD_H
#define _FTPD_H 1

/*
 * This is the port you want BetaFTPD to listen on. The standard
 * FTP port is 21 -- if you really want to use BetaFTPD as your
 * primary FTP server, change FTP_PORT.
 */
#if WANT_NONROOT
#define FTP_PORT 12121
#else
#define FTP_PORT 21
#endif

/*
 * This is the number of seconds an idle connection is allowed to
 * remain idle (`idle' is defined as `no activity on the data socket',
 * more or less) without getting shut down. This is not accurate,
 * as such delays are only checked for every 60 seconds.
 *
 * The default (15 minutes) should be OK for most people.
 */
#define TIMEOUT_SECS 900

/*
 * This is the maximum block size you think you need. (This will most
 * likely be the block size of your filesystem, and you're not likely
 * to need a bigger number than this, unless your TCP stack likes
 * big send()s better than small ones, and still manages to `interleave'
 * the framents.) If this value is too small, your performance would be
 * slightly worse, but it would still work. Try to keep it at a power of
 * two -- most (read: all) FS block sizes _are_ powers of two. If you
 * set it too high, it won't affect performance much -- you would just
 * use a bit more memory.
 */
#define MAX_BLOCK_SIZE 4096

#if HAVE_PWD_H
#include <pwd.h>
#endif

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#if HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#if HAVE_LINUX_SENDFILE && !HAVE_MMAP
#warning sendfile() without mmap() is not supported -- disabling sendfile()
#undef HAVE_LINUX_SENDFILE
#endif

#if WANT_DCACHE && !HAVE_MMAP
#warning directory cache requires use of mmap() -- disabling directory cache
#undef WANT_DCACHE
#endif

struct list_options {
	int recursive;
	int long_listing;
	int classify;
};

/*
 * General structure for the doubly linked lists (conn, ftran, dcache).
 * This is used only by the generic linked list code (which inserts and
 * removes elements from the lists).
 */
struct list_element {
	struct list_element *prev;
	struct list_element *next;

	/* structure specific data here */
};

/* doubly linked list of active connections */
struct conn {
	struct conn *prev_conn;
	struct conn *next_conn;

	int sock;
#if WANT_STAT
	struct sockaddr addr;
#endif
	char recv_buf[256];
#if WANT_FULLSCREEN
	char last_cmd[256];
#endif
	char rename_from[256];

	int buf_len;
	int auth;

	char username[17];

	uid_t uid;
	char root_dir[256];
	char curr_dir[256];

	struct ftran *transfer;

	int rest_pos;
#if WANT_ASCII
	int ascii_mode;
#endif

	time_t last_transfer;
};

/* doubly linked list of file transfers */
struct ftran {
	struct ftran *prev_ftran;
	struct ftran *next_ftran;
	struct conn *owner;

	int state;		/*
				 * 0 = none, 1 = got PASV addr,
				 * 2 = waiting on PASV socket,  
				 * 3 = got PORT addr, 4 = waiting for
				 *     PORT connect, 
				 * 5 = transferring file (or waiting 
				 *     for PORT connect)
				 */
	struct sockaddr_in sin;
	int sock;
	int dir_listing;
#if WANT_DCACHE
	struct dcache *dir_cache;
#endif
#if WANT_ASCII
	int ascii_mode;
#endif
	char filename[256];
	time_t tran_start;
	long int size;

	int local_file;
	int block_size;

#if HAVE_MMAP
	char *file_data;	/* mmap'ed */
#endif
	long int pos;

#if WANT_UPLOAD
	int upload;
	int append;
#endif
};

void add_to_linked_list(struct list_element * const first,
                        struct list_element * const elem);
void remove_from_linked_list(struct list_element * const elem);

struct conn *alloc_new_conn(const int sock);
struct ftran *alloc_new_ftran(const int sock, const struct conn * const c);

int add_fd(const int fd, const int events);
void del_fd(const int fd);

void destroy_conn(struct conn * const c);
void destroy_ftran(struct ftran * const f);

#if HAVE_POLL
int process_all_clients(const int num_ac);
int process_all_sendfiles(const int num_ac);
#else
int process_all_clients(const fd_set * const active_clients, const int num_ac);
int process_all_sendfiles(fd_set * const active_clients, const int num_ac);
#endif

int do_upload(struct ftran *f);
int do_download(struct ftran *f);
void write_xferlog(struct ftran *f);
int main(void);

RETSIGTYPE handle_alarm(int signum);

void accept_new_client(int * const server_sock);
void time_out_sockets();

void remove_bytes(struct conn * const c, const int i);
void numeric(struct conn * const c, const int numeric, const char * const format, ...);
void init_file_transfer(struct ftran * const f);
int create_server_socket();

#if !HAVE_POLL
void clear_bad_fds(int * const server_sock);
#endif

#if WANT_MESSAGE
void dump_file(struct conn * const c, const int num, const char * const filename);
void list_readmes(struct conn * const c);
#endif

#endif
