/*  dcache.c: BetaFTPD directory listing cache
    Copyright (C) 2000 Steinar H. Gunderson

    This program is is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 2 if the
    License as published by the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#if HAVE_CONFIG_H
#include <config.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif

#if HAVE_STRING_H
#include <string.h>
#endif

#if HAVE_STRINGS_H
#include <strings.h>
#endif

#if WANT_DCACHE
#include <ftpd.h>
#include <dcache.h>

extern struct dcache *first_dcache;

/*
 * alloc_new_dcache():
 *		Allocates a new directory cache entry (type `struct dcache'),
 *		and adds it to the linked list.
 */
struct dcache *alloc_new_dcache()
{
	struct dcache *d = (struct dcache *)(malloc(sizeof(struct dcache)));

	if (d == NULL) return d;

	d->use_count = 0;
	d->last_used = 0;
	strcpy(d->dir_name, "");
	d->dir_data = NULL;

	add_to_linked_list((struct list_element *)first_dcache,
			   (struct list_element *)d);

	return d;
}

/*
 * destroy_dcache():
 *		Destroy a directory listing cache entry, remove it from the
 *		linked list, and clean up after it.
 *
 *		If you free a cache entry that is in use (use_count > 0),
 *		BetaFTPD will most likely crash (later). The thing you're supposed
 *		to do when you're done with a dcache entry, is to decrement
 *		its use_count, and let the timeout functions do the destroying
 *		when it's time to do so.
 */
void destroy_dcache(struct dcache * const d)
{
	if (d == NULL) return;

	if (d->dir_data != NULL) free(d->dir_data);
	remove_from_linked_list((struct list_element *)d);
}

/*
 * time_out_dcache():
 *		Time out expired directory listing cache entries.
 *		Uses much of the same code as time_out_sockets().
 */
void time_out_dcache()
{
	struct dcache *d = NULL, *next = first_dcache->next_dcache;
	time_t now = time(NULL);

	/* run through the linked list */
	while (next != NULL) {
		d = next;
		next = d->next_dcache;

		if (d->use_count == 0 && (now - d->last_used > 900)) {
			destroy_dcache(d);
		}
	}
}

/*
 * populate_dcache():
 *		Add a new entry in the dcache, using information from
 *		the given file transfer object, and the current directory
 *		(cwd).
 */
void populate_dcache(struct ftran * const f, const char * const cwd,
		     const char * const pattern, const struct list_options * const lo)
{
	struct stat buf;
	struct dcache *d = alloc_new_dcache();
	if (d != NULL && stat(cwd, &buf) > -1) {
		d->use_count++;
		f->dir_cache = d;
		d->dir_data = f->file_data;
		d->dir_size = f->size;
		d->generated = buf.st_mtime;

		strcpy(d->dir_name, cwd);
		strncpy(d->pattern, pattern, 255);
		d->pattern[255] = 0;
		d->lo = *lo;
	}
}

/*
 * find_dcache():
 *		Check if there is an existing dcache object matching our
 *		criteria.
 */
struct dcache *find_dcache(const char * const cwd, const char * const pattern,
                           const struct list_options * const lo)
{
	struct dcache *d = NULL, *next = first_dcache->next_dcache;
	struct stat buf;

	if (stat(cwd, &buf) > -1) {
		/* run through the linked list */
		while (next != NULL) {
			d = next;
			next = d->next_dcache;

			if (buf.st_mtime <= d->generated &&
			    strcmp(d->dir_name, cwd) == 0 &&
			    strcmp(d->pattern, pattern) == 0 &&
			    memcmp(&(d->lo), lo, sizeof(struct list_options)) == 0) {
				return d;
			}
		}
	}
	return NULL;
}
#endif /* WANT_DCACHE */
