/*  cmds.c: BetaFTPD command handlers
    Copyright (C) 1999-2000 Steinar H. Gunderson

    This program is is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 2 if the
    License as published by the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#define _GNU_SOURCE

#if HAVE_CONFIG_H
#include <config.h>
#endif

#if HAVE_STROPTS_H
#include <stropts.h>
#endif

#if HAVE_SYS_CONF_H
#include <sys/conf.h>
#endif

#if HAVE_DIRENT_H
#include <dirent.h>
#endif

#if HAVE_CRYPT_H
#include <crypt.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif

#if HAVE_GLOB_H
#include <glob.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif

#if HAVE_STRING_H
#include <string.h>
#endif

#if HAVE_STRINGS_H
#include <strings.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_FCNTL_H
#include <fcntl.h>
#endif

#if HAVE_GRP_H
#include <grp.h>
#endif

#if HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif

#if HAVE_STROPTS_H
#include <stropts.h>
#endif

#if HAVE_SYS_CONF_H
#include <sys/conf.h>
#endif

#if HAVE_SHADOW_H
#include <shadow.h>
#endif

#if HAVE_SYS_FILIO_H
#include <sys/filio.h>
#endif

#if HAVE_SYS_POLL_H
#include <sys/poll.h>
#endif

#if WANT_NONROOT
#define NO_SETUID
#define DO_SETUID
#else
#define NO_SETUID ,0
#define DO_SETUID ,1
#endif

#ifndef NBBY
#define NBBY 8
#endif

#include <ftpd.h>
#include <cmds.h>
#include <nonroot.h>

#if WANT_DCACHE
#include <dcache.h>
#endif

#define lstat stat

extern struct conn *first_conn;
#if WANT_DCACHE
extern struct dcache *first_dcache;
#endif

#if HAVE_POLL
extern struct pollfd fds[];
#else
extern fd_set master_fds, master_send_fds;
#endif

struct handler {
	char cmd_name[6];
	char add_cmlen;		/* =1 if the command takes an argument */
	int (*callback)(struct conn * const);
	char min_auth;
#if !WANT_NONROOT
	char do_setuid;		/* =1 if root is not *really* needed */
#endif
};

static const struct handler handler_table[] = {
	{ "user ", 1, cmd_user, 0	NO_SETUID },
	{ "pass ", 1, cmd_pass, 1	NO_SETUID },
	{ "retr ", 1, cmd_retr, 3	DO_SETUID },
	{ "acct ", 1, cmd_acct, 0	NO_SETUID },
	{ "port ", 1, cmd_port, 3	DO_SETUID },
	{ "pasv" , 0, cmd_pasv, 3	DO_SETUID },
	{ "pwd"  , 0, cmd_pwd,  3	DO_SETUID },
	{ "cwd " , 1, cmd_cwd,  3	DO_SETUID },
	{ "cdup" , 0, cmd_cdup, 3	DO_SETUID },
	{ "rest ", 1, cmd_rest, 3	DO_SETUID },
	{ "list" , 0, cmd_list, 3	DO_SETUID },
	{ "nlst" , 0, cmd_nlst, 3	DO_SETUID },
	{ "type ", 1, cmd_type, 3	DO_SETUID },
	{ "mode ", 1, cmd_mode, 3	DO_SETUID },
	{ "stru ", 1, cmd_stru, 3	DO_SETUID },
	{ "size ", 1, cmd_size, 3	DO_SETUID },
	{ "mdtm ", 1, cmd_mdtm, 3	DO_SETUID },
	{ "abor" , 0, cmd_abor, 3	DO_SETUID },
	{ "dele ", 1, cmd_dele, 3	DO_SETUID },
	{ "rnfr ", 1, cmd_rnfr, 3	DO_SETUID },
	{ "rnto ", 1, cmd_rnto, 3	DO_SETUID },
	{ "mkd " , 1, cmd_mkd,  3	DO_SETUID },
	{ "rmd " , 1, cmd_rmd,  3	DO_SETUID },
	{ "allo ", 1, cmd_allo, 3	DO_SETUID },
	{ "stat" , 0, cmd_stat, 0	NO_SETUID },
	{ "noop" , 0, cmd_noop, 0	DO_SETUID },
	{ "syst" , 0, cmd_syst, 0	DO_SETUID },
	{ "help" , 0, cmd_help, 0	NO_SETUID },
	{ "quit" , 0, cmd_quit, 0	DO_SETUID },
	{ "rein" , 0, cmd_rein, 0	DO_SETUID },

	/* deprecated forms */
	{ "xcup" , 0, cmd_cdup, 3	DO_SETUID },
	{ "xcwd ", 1, cmd_cwd,  3	DO_SETUID },
	{ "xpwd" , 0, cmd_pwd,  3	DO_SETUID },
	{ "xmkd ", 1, cmd_mkd,  3	DO_SETUID },
	{ "xrmd ", 1, cmd_rmd,  3	DO_SETUID },
#if WANT_UPLOAD
	{ "stor ", 1, cmd_stor, 3	DO_SETUID },
	{ "appe ", 1, cmd_appe, 3	DO_SETUID },
#endif
#if DOING_PROFILING
#warning Use DOING_PROFILING with caution, and NEVER on a production server! :-)
	{ "exit",  0, cmd_exit, 0	NO_SETUID },
#endif
	{ ""    ,  0, NULL,	0	NO_SETUID }
};

/*
 * do_chdir():	Does a chdir() to newd on c, staying inside the
 *		limits of root_dir. Use this instead of a chdir() whenever
 * 		you can, and possibly even when you can't :-)
 *
 *		This command quirks around some problems in the rest of
 *		the code (namely translate_path()), so a blank newdir is
 *		interpreted as the root directory.
 */
int do_chdir(struct conn * const c, const char * const newd)
{
	char chd[512], temp[512];

	TRAP_ERROR(chdir(c->curr_dir) == -1, 550, return -1);

	/* handle `absolute' paths */
	if (newd[0] == '/' || newd[0] == '\0') {
		strcpy(temp, c->root_dir);

		/*
		 * is this the root directory? if not, remove the trailing `/'
		 * and concatenate the new directory on
		 */
		if (newd[1] != '\0' && newd[0] != '\0') {
			temp[strlen(temp) - 1] = 0;
			strcat(temp, newd);
		}
	} else {
		strcpy(temp, newd);
	}

#if WANT_NONROOT
	if (nr_check_permission(c->uid, temp, 1, 1, NULL) == -1) {
	        numeric(c, 550, "Permission denied");
	        return -1;
	}
#endif

	TRAP_ERROR(chdir(temp) == -1, 550, return -1);

	getcwd(chd, 254);
	if (chd[strlen(chd) - 1] != '/') {
		strcat(chd, "/");
	}

	if (strncmp(chd, c->root_dir, strlen(c->root_dir)) != 0) {
		numeric(c, 550, "No such file or directory.");
		return -1;
	}

	return 0;
}

/*
 * cmd_user():	Handles the USER command, and does most of the initial
 *		authentication work. User names are limited to 16
 *		characters, by force...
 */
int cmd_user(struct conn * const c)
{
	strncpy(c->username, c->recv_buf, 16);
	c->username[16] = 0;

	if (strcasecmp(c->username, "anonymous") == 0) {
		strcpy(c->username, "ftp");
	}
       	if (strcasecmp(c->username, "ftp") == 0) {
	       	numeric(c, 331, "Login OK, send password (your e-mail).");
		c->auth = 1;
	} else {
		numeric(c, 331, "Password required for %s.", c->username);
		c->auth = 2;
	}
	return 1;
}

/*
 * cmd_pass():	Handles the PASS command, and checks the password.
 *		This function is rather long and complicated, mostly
 *		because there are so many ways of doing users
 *		(including my nonroot system) out there... And we
 *		don't even support PAM or real shadow passwords (with
 *		expiry etc) yet...
 */
int cmd_pass(struct conn * const c)
{
#if WANT_NONROOT
	c->auth = nr_userinfo(c->username, &c->uid, c->curr_dir, c->root_dir,
		              c->recv_buf);
#else /* !WANT_NONROOT */
#if WANT_SHADOW && HAVE_SHADOW_H
	struct spwd *s;
#endif
	struct passwd *p;

       	p = getpwnam(c->username);
#if WANT_SHADOW && HAVE_SHADOW_H
	s = getspnam(c->username);
#endif
       	
	if (p == NULL) {
		c->auth = 0;
	} else {
		c->uid = p->pw_uid;
		//--> 20090402: Modify by Ken. Asking for testing.
		//strncpy(c->curr_dir, p->pw_dir, 254);
		//strncpy(c->curr_dir, "/upgrade", 254);
		/// Jim@2009_0904: Change to /tmp folder
		strncpy(c->curr_dir, "/tmp", 254);
		//<-- 20090402: Modify by Ken. Asking for testing.
		c->curr_dir[254] = 0;
	}

       	if (c->auth == 1) {
		if (c->curr_dir[strlen(c->curr_dir) - 1] != '/') {
			strcat(c->curr_dir, "/");
		}
		strcpy(c->root_dir, c->curr_dir);	
		c->auth = 3;
	} else if (c->auth != 0) {
		strcpy(c->root_dir, "/");
		if (strcmp(crypt(c->recv_buf, p->pw_passwd), p->pw_passwd) != 0
#if WANT_SHADOW && HAVE_SHADOW_H
		    && (s == NULL || strcmp(crypt(c->recv_buf, s->sp_pwdp), s->sp_pwdp) != 0)
#endif
		) {
			c->auth = 0;
		} else {
			c->auth = 3;
		}
	}
#endif /* !WANT_NONROOT */

	/* root should not be allowed to FTP */
	if (c->uid == 0) {
		c->auth = 0;
	}
	if (c->auth == 0) {
		numeric(c, 530, "Login incorrect.");
	} else {
#if WANT_MESSAGE
		chdir(c->curr_dir);
		dump_file(c, 230, "welcome.msg");
#endif
		numeric(c, 230, "User logged in.");
	}
	return 1;
}

/*
 * cmd_acct():	Handle (ignore) the ACCT command. I don't see how we
 *		could make use of this command... wu-ftpd doesn't, either.
 *		However, wu-ftpd (at least the version I have here) uses
 *		502, which isn't a legal error code according to RFC959.
 *		202, on the other hand, is, and seems to be applicable.
 *
 *		I'm unsure if this one should require setuid or not, but
 *		I feel that the RFC959 intention is having it _before_
 *		USER/PASS. Therefore, this one runs with root privilegies :-)
 */
int cmd_acct(struct conn * const c)
{
	numeric(c, 202, "ACCT ignored OK -- not applicable on this system.");
	return 1;
}

/*
 * cmd_port():	Handles the PORT command, and sets up the data socket.
 *		Making a brief uid=0 (root) appearance (to bind the socket) --
 * 		I feel it's safer that way (instead of running as root
 *		the whole way), in case there are some weird overflows
 * 		somewhere.
 */
int cmd_port(struct conn * const c)
{
	short int a0, a1, a2, a3, p0, p1;
	int i, sock, err;
	struct ftran *f;
	struct sockaddr_in sin;
    
	if ((c->transfer != NULL) && (c->transfer->state >= 4)) {
		numeric(c, 500, "Sorry, only one transfer at a time.");
		return 1;
	}

	sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	TRAP_ERROR(sock == -1, 500, return 1);

	destroy_ftran(c->transfer);
	c->transfer = f = alloc_new_ftran(sock, c);

	i = sscanf(c->recv_buf, "%3hu,%3hu,%3hu,%3hu,%3hu,%3hu", &a0, &a1, &a2, &a3, &p0, &p1);
	if (i < 6) {
		numeric(c, 501, "Parse error.");
	} else {
		const int one = 1;
		int tmp;

		/* bind to own address, port 20 (FTP data) */
		tmp = sizeof(sin);
		err = getsockname(c->sock, (struct sockaddr *)&sin, &tmp);
		TRAP_ERROR(err == -1, 500, return 1);
		sin.sin_port = FTP_PORT - 1;

		numeric(c, 200, "PORT command OK.");

		/* note that bind() might well fail, so we don't error check */
#if !WANT_NONROOT
		/* need root privilegies for a short while */
		seteuid(getuid());
#endif
		bind(sock, (struct sockaddr *)&sin, sizeof(sin));
#if !WANT_NONROOT
		seteuid(c->uid);
#endif

		f->sin.sin_family = AF_INET;
		f->sin.sin_addr.s_addr = htonl(
			((unsigned char)(a0) << 24) +
			((unsigned char)(a1) << 16) +
			((unsigned char)(a2) <<  8) +
			((unsigned char)(a3)      ));
		f->sin.sin_port = htons(
			((unsigned char)(p0) << 8) +
			((unsigned char)(p1)     ));
		f->sock = sock;
		f->state = 3;

		i = 1;		
		ioctl(f->sock, FIONBIO, &one);
	}
	return 1;
}

/*
 * cmd_pasv():	Handles the PASV command, and sets up the data socket.
 *		Uses port numbers > 1024, since it doesn't run as root.
 */
int cmd_pasv(struct conn * const c)
{
	struct ftran *f;
	int tmp, sock, err;
	unsigned int one = 1;
	struct sockaddr_in addr;

	if ((c->transfer != NULL) && (c->transfer->state >= 4)) {
		numeric(c, 503, "Sorry, only one transfer at once.");
		return 1;
	}
	destroy_ftran(c->transfer);

	sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	TRAP_ERROR(sock == -1, 500, return 1);
	err = add_fd(sock, POLLIN);
	TRAP_ERROR(err != 0, 501, return 1);

	c->transfer = f = alloc_new_ftran(sock, c);

	ioctl(sock, FIONBIO, &one);

	/* setup socket */
	tmp = sizeof(addr);
	err = getsockname(c->sock, (struct sockaddr *)&addr, &tmp);
	TRAP_ERROR(err == -1, 500, return 1);

	addr.sin_port = 0;	/* let the system choose */
	err = bind(sock, (struct sockaddr *)&addr, sizeof(struct sockaddr));
	TRAP_ERROR(err == -1, 500, return 1);

	tmp = sizeof(addr);
	err = getsockname(sock, (struct sockaddr *)&addr, &tmp);
	TRAP_ERROR(err == -1, 500, return 1);

	err = listen(f->sock, 1);
	TRAP_ERROR(err == -1, 500, return 1);
	f->state = 1;

	numeric(c, 227, "Entering passive mode (%u,%u,%u,%u,%u,%u)",
		(htonl(addr.sin_addr.s_addr) & 0xff000000) >> 24,
		(htonl(addr.sin_addr.s_addr) & 0x00ff0000) >> 16,
		(htonl(addr.sin_addr.s_addr) & 0x0000ff00) >>  8,
		(htonl(addr.sin_addr.s_addr) & 0x000000ff),
		(htons(addr.sin_port) & 0xff00) >> 8,
		(htons(addr.sin_port) & 0x00ff));
	return 1;
}

/*
 * cmd_pwd():	Handles PWD command (print working directory).
 *
 *		Note that if somebody contacts you with the message `the server
 *		says curr_dir() is outside root_dir()', you should fix your
 *		/betaftpd.users file, if you use nonroot. If not, it's a bug.
 *		Try to get it _reproducible_, and mail it to me.
 */
int cmd_pwd(struct conn * const c)
{
	char temp[512], *cdir = NULL;

	cdir = do_pwd(c, temp, c->curr_dir);
	if (cdir != NULL) {
		numeric(c, 257, "\"%s\" is current working directory.", cdir);
	}
	return 1;
}

/*
 * do_pwd():	Translates an absolute path to a path suitable for viewing
 *		to the user (ie. removes the root_dir, and removes a trailing
 *		slash if it exists). Note that the retbuf is only used as a
 *		storage place -- the pointer to the right place within retbuf
 *		is _returned_.
 */
char *do_pwd(struct conn * const c, char * const retbuf, const char * const dir) 
{
	char *cdir = NULL;

	strcpy(retbuf, dir);
	if (strncmp(retbuf, c->root_dir, strlen(c->root_dir)) != 0) {
		numeric(c, 550, "curr_dir is outside root_dir, please contact site administrator.");
		return NULL;
	}

	cdir = retbuf + strlen(c->root_dir) - 1;
	if (cdir[strlen(cdir) - 1] == '/' && strlen(cdir) > 1) {
		cdir[strlen(cdir) - 1] = 0;
	} else if (strlen(cdir) == 0) {
		strcpy(cdir, "/");
	}	

	return cdir;
}

/*
 * cmd_cwd():	Handles CWD command (change working directory). Uses
 *		cmd_cwd_internal() (see below).
 */
int cmd_cwd(struct conn * const c)
{
	cmd_cwd_internal(c, c->recv_buf);
	return 1;
}

/*
 * cmd_cdup():  Handles a CDUP command (identical to `CWD ..'). Note that
 *		RFC959 gives two different response codes (250 and 200) --
 *		250 is the same as CWD gives, which sounds logical to me.
 *		wu-ftpd uses it as well.
 *
 *		Note that using a CDUP to try to get outside root_dir returns
 *		an error, instead of just staying in the root directory (as
 *		the OS and thus wu-ftpd does).
 */
int cmd_cdup(struct conn * const c)
{
	cmd_cwd_internal(c, "..");
	return 1;
}

/*
 * cmd_cwd_internal():
 *		Does the work for CWD and CDUP (modularized to save some
 *		space and have clearer code). Mostly, it just uses do_chdir(),
 *		and sees where that takes us. It adds a trailing slash if needed.
 */
void cmd_cwd_internal(struct conn * const c, const char * const newd)
{
	if (do_chdir(c, newd) != -1) {
		int i;

		getcwd(c->curr_dir, 254);
		i = strlen(c->curr_dir);
		if (c->curr_dir[i - 1] != '/') {
			c->curr_dir[i++] = '/';
			c->curr_dir[i] = '\0';
		}

#if WANT_MESSAGE
		dump_file(c, 250, ".message");
		list_readmes(c);
#endif

		numeric(c, 250, "CWD successful.");
	}
}

/*
 * cmd_rest():	Handles the REST command. All it does is tell the file
 *		sending functions to start at the correct number. We should
 *		perhaps add some better error checking to this?
 */
int cmd_rest(struct conn * const c)
{
	c->rest_pos = abs(atoi(c->recv_buf));
	numeric(c, 350, "Setting resume at %u bytes.", c->rest_pos);
	return 1;
}

/*
 * cmd_retr():	Handles the RETR command. This command doesn't send the
 *		file, but it opens it and tells the socket handling code
 *		to check for activity on the data socket. When the
 *		connection occurs (or succeeds, if we're using PORT mode),
 *		the actual file transfer begins.
 */
int cmd_retr(struct conn * const c)
{
	struct ftran *f = c->transfer;

	if ((f == NULL) || ((f->state != 1) && (f->state != 3))) {
		numeric(c, 425, "No data connection set up; please use PASV or PORT.");
		return 1;
	}

#if WANT_ASCII
	if ((c->rest_pos > 0) && (c->ascii_mode == 1)) {
		numeric(c, 500, "Cannot resume while in ASCII mode.");
		return 1;
	}
#endif

	f->local_file = do_openfile(c, c->recv_buf, f->filename, O_RDONLY
#if WANT_NONROOT
		, 4
#endif
	);
	f->dir_listing = 0;

	if (f->local_file == -1) {
		numeric(f->owner, 550, strerror(errno));
		destroy_ftran(f);
	} else if (f->local_file == -2) {
		f->local_file = -1;
		destroy_ftran(f);
	} else {
#if WANT_UPLOAD
		f->upload = 0;
#endif
		prepare_for_transfer(f);
	}
	return 1;
}

#if WANT_UPLOAD
/*
 * cmd_stor():	Handles the STOR command (upload file). Pushes the
 *		work down to do_store(), below.
 */
int cmd_stor(struct conn * const c)
{
	do_store(c, 0);
	return 1;
}

/*
 * cmd_appe():	Handles the APPE command (append to file). Pushes
 *		the work down to do_store(), below.
 */
int cmd_appe(struct conn * const c)
{
	do_store(c, 1);
	return 1;
}

/*
 * do_store():	Initiate an upload. Most of the comments to do_retr()
 *		(above) apply to this one as well.
 */
void do_store(struct conn * const c, const int append)
{
	struct ftran *f = c->transfer;

	if ((f == NULL) || ((f->state != 1) && (f->state != 3))) {
		numeric(c, 425, "No data connection set up; please use PASV or PORT.");
		return;
	}

#if WANT_ASCII
	if ((c->rest_pos > 0) && (c->ascii_mode == 1)) {
		numeric(c, 500, "Cannot resume while in ASCII mode.");
		return;
	}
#endif

	f->local_file = do_openfile(c, c->recv_buf, f->filename, O_WRONLY |
		O_CREAT | ((append || c->rest_pos > 0) ? 0 : O_TRUNC)
#if WANT_NONROOT
	        , 2
#endif
	);
	f->dir_listing = 0;

	if (f->local_file == -1) {
	        numeric(f->owner, 550, strerror(errno));
	} else if (f->local_file == -2) {
		f->local_file = -1;
	} else {
	        f->upload = 1;
	        f->append = append;
#if WANT_ASCII
		f->ascii_mode = c->ascii_mode;
#endif
	        prepare_for_transfer(f);
	}
}
#endif /* WANT_UPLOAD */

/*
 * cmd_size():	Handle the SIZE command -- returns the size of a
 *		file. Note that this command is not part of RFC959,
 *		and thus there is no clear specification (except
 *		for some ftpext documents, which we try to follow
 *		as closely as we can). BetaFTPD deviates from wu-ftpd
 *		in that it lets you check the `size' of directories
 *		as well (instead of giving 550). This is _not_ the
 *		size of all the files in the directory, rather how
 *		much space the directory inode uses.
 */
int cmd_size(struct conn * const c)
{
#if WANT_ASCII
	if (c->ascii_mode) {
		numeric(c, 550, "SIZE not available in ASCII mode.");
		return 1;
	}
#endif
	{
		const char * const fname = translate_path(c, c->recv_buf);
		struct stat buf;
	
		TRAP_ERROR(fname == NULL || lstat(fname, &buf) == -1, 550, return 1);
	
		numeric(c, 213, "%lu", (unsigned long)(buf.st_size));
		return 1;
	}
}

/*
 * cmd_mdtm():	Handle the MDTM command -- returns the modification
 *		date/time of a file. See the comments on cmd_size(),
 *		above.
 */
int cmd_mdtm(struct conn * const c)
{
	const char * const fname = translate_path(c, c->recv_buf);
	struct stat buf;
	struct tm *m;

	TRAP_ERROR(fname == NULL || lstat(fname, &buf) == -1, 550, return 1);

	m = gmtime(&(buf.st_mtime));	/* at least wu-ftpd does it in GMT */
	numeric(c, 213, "%u%02u%02u%02u%02u%02u", m->tm_year + 1900,
		m->tm_mon + 1, m->tm_mday, m->tm_hour, m->tm_min, m->tm_sec);
	return 1;
}

/*
 * cmd_abor():	Handle the ABOR command (abort a file transfer). This should
 *		be clean enough, but isn't tested extensively.
 */
int cmd_abor(struct conn * const c)
{
	if (c->transfer != NULL) {
		numeric(c, 426, "File transfer aborted.");
		destroy_ftran(c->transfer);
	}
	numeric(c, 226, "ABOR command processed OK.");
	return 1;
}

/*
 * cmd_dele():	Handle the DELE command (delete a file).
 */
int cmd_dele(struct conn * const c)
{
	const char * const fname = translate_path(c, c->recv_buf);
	
	TRAP_ERROR(fname == NULL || unlink(fname) == -1, 550, return 1);
	numeric(c, 250, "File deleted OK.");
	return 1;
}

/*
 * cmd_rnfr():	Handle the RNFR command (take a filename to rename from).
 */
int cmd_rnfr(struct conn * const c)
{
	const char * const fname = translate_path(c, c->recv_buf);
	char cwd[256];
	struct stat buf;

	c->rename_from[0] = '\0';
	if (fname == NULL) return 1;
	
	getcwd(cwd, 256);
	snprintf(c->rename_from, 256, "%s/%s", cwd, fname);

	/* Just check that the file exists. */
	TRAP_ERROR(lstat(c->rename_from, &buf) == -1, 550, c->rename_from[0] = '\0'; return 1);

	numeric(c, 350, "File exists, send RNTO.");
	return 1;
}

/*
 * cmd_rnto():	Handle the RNTO command (do the actual renaming).
 */
int cmd_rnto(struct conn * const c)
{
	const char * const fname = translate_path(c, c->recv_buf);

	if (fname == NULL) return 1;
	if (c->rename_from[0] == '\0') {
		numeric(c, 503, "Please send RNFR first.");
		return 1;
	}

	TRAP_ERROR(rename(c->rename_from, fname) == -1, 550, c->rename_from[0] = '\0'; return 1);
	c->rename_from[0] = '\0';

	numeric(c, 250, "File renamed successfully.");
	return 1;
}

/*
 * cmd_mkd():	Handle the MKD/XMKD command (create a new directory).
 *		RFC959 is not clear on the error codes for this command --
 *		one place, 521 is cited as the correct error, but is
 *		mentioned nowhere else. Different FTP servers differ here
 *		as well. Thus, I've followed what appears to be the intention
 *		(having `analogous' errors with STOR), and use 550 instead.
 *
 *		Making directories is probably the topic covered most
 *		extensively by RFC959 (and in the most confusing way as
 *		well). I try to follow the conventions, but it isn't always
 *		easy :-) (This code isn't quite easy to understand, because
 *		temp2 is used twice, in two different roles.)
 */
int cmd_mkd(struct conn * const c)
{
	const char * const fname = translate_path(c, c->recv_buf);
	char temp[512], temp2[1024], *cdir;
	int i, j;

	TRAP_ERROR(fname == NULL || mkdir(fname, 0755) == -1, 550, return 1);

	chdir(fname);
	getcwd(temp2, 512);
	cdir = do_pwd(c, temp, temp2);

	/* double the quotes in the output */ 
	for (i = 0, j = 0; i <= strlen(cdir); i++, j++) {
		temp2[j] = cdir[i];
		if (cdir[i] == '"') {
			temp2[++j] = '"';
		}
	}
	numeric(c, 257, "\"%s\" created.", temp2);
	return 1;
}

/*
 * cmd_rmd():	Handle the RMD/XRMD command. Works just like DELE, only for
 *		directories.
 */
int cmd_rmd(struct conn * const c)
{
	const char * const fname = translate_path(c, c->recv_buf);

	TRAP_ERROR(fname == NULL || rmdir(fname) == -1, 550, return 1);
	numeric(c, 250, "Directory deleted.");
	return 1;
}

/*
 * cmd_allo():	Handle the ALLO command. The command does not do anything, except
 *		sit around and play compliant. Some Windows FTP servers (Serv-U,
 *		for instance), verifies that there is enough space on the disk,
 *		but since we have no idea on what the filesystem will be stored on,
 *		we just ignore the command.
 *
 *		We could theoretically use this information to give more information
 *		to the full-screen mode, but close to no FTP clients send this
 *		command, and it would touch too much code.
 */
int cmd_allo(struct conn * const c)
{
	numeric(c, 202, "No storage allocation necessary.");
	return 1;
}

/*
 * cmd_stat():	Handle the STAT command. Please see README for more details.
 *		Note that this command is run with euid=root, since it has
 *		to be able to run before USER.
 *
 *		Note that we need to bypass numeric(), to get a multi-line
 *		reply.
 */
#if WANT_STAT
char conn_state[5][27] = {
	"Not logged in",
	"Waiting for e-mail address",
	"Waiting for password",
	"Logged in",
	"Waiting for password",		/* actually non-existant user */
};

char ftran_state[6][42] = {
	"Not initialized",
	"Decided PASV address/port",
	"Waiting on PASV socket",
	"Got PORT address/port",
	"Connecting on PORT address/port",
	"Transferring file (or connecting on PORT)"
};
#endif

int cmd_stat(struct conn * const c)
{ 
#if WANT_STAT
	char buf[1024];
	int i, err;
	struct ftran *f = c->transfer;

	snprintf(buf, 1024, "211- FTP server status:\r\n"
			    "     BetaFTPD version " VERSION " (http://members.xoom.com/sneeze/betaftpd.html)\r\n"
			    "     Connected to %s\r\n"
			    "     Control connection state: %s\r\n"
#if WANT_ASCII
			    "     TYPE: %s; STRUcture: File; transfer MODE: Stream\r\n"
#else
			    "     TYPE: Image; STRUcture: File; transfer MODE: Stream\r\n"
#endif
			    "     Data connection state: %s\r\n"
			    "211 End of status\r\n",
				inet_ntoa(((struct sockaddr_in *)(&(c->addr)))->sin_addr),
				conn_state[c->auth],
#if WANT_ASCII
				(c->ascii_mode == 1) ? "ASCII, FORM: Nonprint" : "Image",
#endif
				(f) ? ftran_state[f->state] : ftran_state[0]);

	i = strlen(buf);

	err = send(c->sock, buf, i, 0);
       	if (err == -1 && errno == EPIPE) {
               	destroy_conn(c);
		return 0;
	}
#else
	numeric(c, 502, "STAT command disabled for security reasons.");
#endif
	return 1;
}

#if HAVE_MMAP
/*
 * _mwrite():	This define is for mmap-listing. It works as a write()
 *		(not in parameter, but in function), and is used in
 *		cmd_list() and cmd_nlst() only.
 *
 *		Note that this function returns the new position in the
 *		`file'. The caller is expected to send this information
 *		back in `pos' at the next call to _mwrite().
 */
int _mwrite(const char * const buf, const struct ftran * const f,
	    const int pos, const int count, const int size)
{
	if (pos + count >= size) return size;	/* out of space */
	memcpy(f->file_data + pos, buf, count);
	return pos + count;
}
#endif

/*
 * mwrite:	This is a short_hand define, making calls to _mwrite() very
 *		similiar to calls to write(). It works both with and without
 *		mmap().
 */
#if HAVE_MMAP
#define mwrite(buf, count) pos = _mwrite((buf), (f), (pos), (count), (size));
#else
#define mwrite(buf, count) write(f->local_file, buf, count);
#endif

/*
 * long_listing():
 *		Formats output in `ls -l' style. It returns one line for the
 *		file PATHNAME, and returns it in retbuf. Setting do_classify
 *		to nonzero has the same effect as `ls -F'.
 *
 *		This command is so long, because simply there is so much to
 *		be done. GNU ls has some extra functions, but it's close to
 *		3000 lines too...
 */
int long_listing(char * const retbuf, const char * const pathname, const int do_classify)
{
	int i, year;
	char newd[512], temp[1026];
	struct stat buf;
	struct tm *t;
	time_t now;
	char username[17], groupname[17];

	time(&now);
	year = localtime(&now)->tm_year;
	{
#if !WANT_NONROOT
		struct passwd *p;
		struct group *g;
#endif

		if (lstat(pathname, &buf) == -1) return 0;

#if WANT_NONROOT
		strcpy(username, nr_get_uname(buf.st_uid));
		strcpy(groupname, nr_get_gname(buf.st_gid));
#else
		p = getpwuid(buf.st_uid);
		if (p != NULL) {
			strncpy(username, p->pw_name, 16);
			username[16] = 0;
		} else {
			snprintf(username, 16, "%u", buf.st_uid);
		}

		g = getgrgid(buf.st_gid);
		if (g != NULL) {
			strncpy(groupname, g->gr_name, 16);
			groupname[16] = 0;
		} else {
			snprintf(groupname, 16, "%u", buf.st_gid);
		}
#endif
	}

	/*
	 * This POSIX approximation is based on GNU ls code (and obfuscated
	 * a bit...), to be compatible with `real' ls implementations.
	 */
	t = localtime(&(buf.st_mtime));
	strftime(newd, 512, ((now > buf.st_mtime + 6L * 30L * 24L * 60L * 60L) ||
			     (now < buf.st_mtime - 60L * 60L))
			? "%b %e  %Y" : "%b %e %H:%M", t);

	{
#if WANT_NONROOT
		char rights[16];

		if (nr_check_permission(0, pathname, 0, (S_ISDIR(buf.st_mode)), rights) == -1) {
			/* no permission to even see this file */
			return 0;
		}

		snprintf(temp, 1024, "%c%s %3u %-8s %-8s %8lu %12s %s\r\n",
#else
		snprintf(temp, 1024, "%c%c%c%c%c%c%c%c%c%c %3u %-8s %-8s %8lu %12s %s",
#endif
			decode_mode(buf.st_mode),
#if WANT_NONROOT
			rights,
#else
			(buf.st_mode & S_IRUSR) ? 'r' : '-', 
			(buf.st_mode & S_IWUSR) ? 'w' : '-',
			(buf.st_mode & S_IXUSR) ? ((buf.st_mode & S_ISUID) ? 's' : 'x') : '-',
			(buf.st_mode & S_IRGRP) ? 'r' : '-',
			(buf.st_mode & S_IWGRP) ? 'w' : '-',
			(buf.st_mode & S_IXGRP) ? ((buf.st_mode & S_ISGID) ? 's' : 'x') : '-',
			(buf.st_mode & S_IROTH) ? 'r' : '-',
			(buf.st_mode & S_IWOTH) ? 'w' : '-',
			(buf.st_mode & S_IXOTH) ? ((buf.st_mode & S_ISVTX) ? 't' : 'x') : '-',
#endif
			buf.st_nlink, username, groupname,
			(unsigned long)(buf.st_size), newd, pathname);
		i = strlen(temp);
	
#if 0
		/*
		 * vim needs this extra character for some reason... It's too 
		 * bad I'll have to do it this way, but syntax colouring
		 * that works properly is almost a `must' for me :-)
		 */
		)
#endif
 
	        /* add an extra classification `sign' if we got -F */
	        if (do_classify) {
			int len = strlen(temp);
	                temp[len] = classify(buf.st_mode);
	                temp[len + 1] = '\0';
	        }
	}

	strcpy(retbuf, temp);
	return 1;
}

/*
 * cmd_list():	Handles the LIST command (directory listing). Does a
 *		long listing (of type `ls -l'). The listing work is
 *		done by do_listing(), below.
 */
int cmd_list(struct conn * const c)
{
	struct list_options lo;

	lo.recursive = 0;
	lo.long_listing = 1;
	lo.classify = 0;

	do_listing(c, &lo);
	return 1;
}

/*
 * cmd_nlst():	Handles the NLST command (plain directory listing).
 *		Does a plain listing (no dates etc.), unless overridden
 *		by the `-l' or `-L' flag (case insensitivity because most
 *		FTP clients don't have a clue about what they send out). 
 *		The listing work is done by do_listing(), below.
 */	
int cmd_nlst(struct conn * const c)
{
	struct list_options lo;

 	lo.recursive = 0;
	lo.long_listing = 0;
	lo.classify = 0;

	do_listing(c, &lo);
	return 1;
}

/*
 * do_listing():
 *		Prepares any listing buffers, temp files, etc., before
 *		pushing the work one step further :-)
 *
 *		If the directory listing cache is enabled, the cache
 *		is checked first, to see if we still have a valid entry.
 */
void do_listing(struct conn * const c, struct list_options * const lo)
{
	int i;
	char *ptr;
#if HAVE_MMAP
	int size;
#endif
	struct ftran * const f = c->transfer;

#if WANT_DCACHE
	char cwd[256];
#endif

#if WANT_NONROOT
#warning No nonroot checking for list_core() yet
#endif

	i = prepare_for_listing(c, &ptr, lo);
	if (i == -1) {
		destroy_ftran(c->transfer);
		return;
	}

#if WANT_DCACHE
	getcwd(cwd, 256);
#endif

#if HAVE_MMAP
	strcpy(f->filename, "(directory listing)");
#endif

#if WANT_DCACHE
	{
		struct dcache *d = find_dcache(cwd, ptr, lo);
		if (d != NULL) {
			d->use_count++;
			f->dir_cache = d;
			f->file_data = d->dir_data;
			f->size = d->dir_size;
			f->dir_listing = 1;
			f->pos = 0;

			prepare_for_transfer(f);
			return;
		}
	}
#endif

#if HAVE_MMAP
	{
		int num_files = get_num_files(c, ptr, lo);
		if (num_files == -1) return;

		size = num_files * 160;
		f->file_data = malloc(size + 1);
		TRAP_ERROR(f->file_data == NULL, 550, return);
		list_core(c, ptr, "", lo, size, 0);
	}
#else
	list_core(c, ptr, "", lo);
#endif

#if WANT_DCACHE
	populate_dcache(f, cwd, ptr, lo);
#endif

#if HAVE_MMAP
	f->pos = 0;
#endif
	prepare_for_transfer(f);
}

/*
 * get_num_files():
 *		Get the number of files in PATHNAME (optionally matching
 *		a pattern). Note that c is needed for TRAP_ERROR.
 */
int get_num_files(struct conn * const c, const char * const pathname,
	       	   struct list_options * const lo)
{
	int num_files;
	glob_t pglob;

	/*
	 * glob() fails to set errno correctly, so we simply guess on
	 * `permission denied'... The others are far less likely to happen.
	 */
	switch (glob(pathname, 0, NULL, &pglob)) {
#ifdef GLOB_NOMATCH
	case GLOB_NOMATCH:
		return 0;
#endif
	case 0:
		num_files = pglob.gl_pathc;
		break;
	default:
		numeric(c, 550, strerror(EACCES));
		return -1;
	}

	if (lo->recursive) {
		int i;
       		for (i = 0; i < pglob.gl_pathc; i++) {
		        char *temp = pglob.gl_pathv[i];
			struct stat buf;

			lstat(temp, &buf);
			if (S_ISDIR(buf.st_mode)) {
				chdir(temp);
				num_files += get_num_files(c, "*", lo);
				chdir("..");
			}
		}
	}

	globfree(&pglob);

	return num_files;
}

/*
 * list_core():	Enumerate all the files in PATHNAME, and formats them
 *		according to list_options (calling format functions if
 *		required).
 *
 *		Note that we don't do any realloc() yet, so if your
 *		_average_ file name length is over a certain size (a little
 *		under 80 for long listings, and a little under 160 for
 *		short listings), the list will be truncated. Fix...
 *
 *		The return value only makes sense if mmap()'ing, since it
 *		returns the number of bytes written into the buffer.
 *
 *		This function is rather long.
 */
int list_core(struct conn * const c, const char * const pathname,
	      char * const disp_pathname, struct list_options * const lo
#if HAVE_MMAP
		, const int size, int pos
#endif
		)
{
	int i;
	glob_t pglob;
	struct ftran * const f = c->transfer;

        /*
         * glob() fails to set errno correctly, so we simply guess on
         * `permission denied'... The others are far less likely to happen.
         */
        switch (glob(pathname, GLOB_MARK, NULL, &pglob)) {
        case 0:
#ifdef GLOB_NOMATCH
	case GLOB_NOMATCH:
#endif
                break;		/* note: break, not return */
        default:
                numeric(c, 550, strerror(EACCES));
#if HAVE_MMAP
		return pos;
#else
                return 0;
#endif
        }

	if (lo->recursive) {
		if (disp_pathname[0] == '\0') {
			mwrite(".:\r\n", 4);
		} else {
			char temp[1024];
			int i;

			snprintf(temp, 1024, "%s:\r\n", disp_pathname);
			i = strlen(temp);
			mwrite(temp, i);
		}
	}
	if (lo->long_listing) {
		/* FIX: we may get too high total number if we are running nonroot! */
		struct stat buf;
		long unsigned int total = 0;
		char temp[1024];

		for (i = 0; i < pglob.gl_pathc; i++) {
	        	if (lstat(pglob.gl_pathv[i], &buf) != -1) {
	        		total += buf.st_blocks;
			}
		}
		snprintf(temp, 1024, "total %lu\r\n", total >> 1);
		i = strlen(temp);
		mwrite(temp, i);
	}

	for (i = 0; i < pglob.gl_pathc; i++) {
		char * const temp = pglob.gl_pathv[i];
		char buf[2048];

		/* strip `/' away from the pathname -- add it later if -F */
		{
			int len = strlen(temp);
			if (temp[len - 1] == '/') {
				temp[len - 1] = '\0';
			}
		}

		if (lo->long_listing) {
			if (long_listing(buf, temp, lo->classify) == 0) continue;
		} else {
			strcpy(buf, temp);
			if (lo->classify) {
				struct stat statbuf;

				if (lstat(buf, &statbuf) != -1) {
					const int len = strlen(buf);

					buf[len] = classify(statbuf.st_mode);
					buf[len + 1] = 0;
				}
			}
		}

		mwrite(buf, strlen(buf));
		mwrite("\r\n", 2);
	}

	/*
	 * If recursion is on, dive into any subdirectories now -- note
	 * that each entry is stat()'ed twice, hopefully the OS will manage,
	 * and we've got our own dcache anyways -- this could be fixed at
	 * the expense of some memory, consider for later inclusion.
 	 */
	if (lo->recursive) {
		for (i = 0; i < pglob.gl_pathc; i++) {
                	struct stat buf;
			const char * const temp = pglob.gl_pathv[i];

			/* don't dive into `.' or `..' */
                        if (lstat(temp, &buf) != -1 && S_ISDIR(buf.st_mode) &&
				(temp[0] != '.' || (temp[1] != '.' && temp[1] != '\0'))) {
				char tmp2[1024];

				mwrite("\r\n", 2);

				/* attach the pathname to the end of the displayed path */
				if (disp_pathname[0] == '\0') {
					snprintf(tmp2, 1024, "%s", temp);
				} else {
					snprintf(tmp2, 1024, "%s/%s", disp_pathname, temp);
				}

				chdir(temp);
#if HAVE_MMAP
				pos = list_core(c, "*", tmp2, lo, 
					size, pos);
#else
				list_core(c, "*", tmp2, lo); 
#endif
				chdir("..");
			}
		}
	}

#if HAVE_MMAP
	f->size = pos;
#else
	lseek(f->local_file, 0, SEEK_SET);
#endif

	globfree(&pglob);
#if HAVE_MMAP
	return pos;
#else
	return 0;
#endif
}

/*
 * cmd_noop():	Handles the NOOP command. Does nothing, doesn't even
 *		reset the timeout.
 */
int cmd_noop(struct conn * const c)
{
	numeric(c, 200, "NOOP command successful.");
	return 1;
}

/*
 * cmd_syst():	Handles the SYST command. Returns the system identification.
 */
int cmd_syst(struct conn * const c)
{
	numeric(c, 215, "UNIX Type: L%u", NBBY);
	return 1;
}

/*
 * cmd_type():	Handles the TYPE command.
 */
int cmd_type(struct conn * const c)
{
#if WANT_ASCII
	c->recv_buf[0] &= (255-32);	/* convert to upper case */
	if (c->recv_buf[0] == 'A') {
		c->ascii_mode = 1;
		numeric(c, 200, "Type is ASCII.");
	} else if (c->recv_buf[0] == 'I') {
		c->ascii_mode = 0;
		numeric(c, 200, "Type is IMAGE.");
	} else {
		numeric(c, 504, "Unknown type.");
	}
#else
	numeric(c, 200, "TYPE ignored (always I)");
#endif
	return 1;
}

/*
 * cmd_mode():	Handles the MODE command. Only stream mode is supported.
 */
int cmd_mode(struct conn * const c)
{
 	c->recv_buf[0] &= (255-32);	/* convert to upper case */
	if (c->recv_buf[0] == 'S') {
		numeric(c, 200, "Mode is STREAM.");
	} else {
		numeric(c, 504, "Unknown mode.");
	}
	return 1;
}

/*
 * cmd_stru():	Handles the STRU command. Only file mode is supported.
 */
int cmd_stru(struct conn * const c)
{
	c->recv_buf[0] &= (255-32);	/* convert to upper case */
	if (c->recv_buf[0] == 'F') {
		numeric(c, 200, "Structure is FILE.");
	} else {
		numeric(c, 504, "Unknown structure.");
	}
	return 1;
}

/*
 * cmd_help():	Handle the HELP command. I'm sorry, but I'm unwilling
 *		to use a lot of space to explain the RFCs in such a message,
 *		and BetaFTPD doesn't have any special things that should
 *		be noted anywhere. Thus, this message is close to empty. I
 *		feel that a 5xx entry would have been better, but that is
 *		disallowed.
 *
 *		As with ACCT, this command is supposed to be executed from
 *		everywhere, so we have to run without setuid. I don't like
 *		it, but at the same time I have to idea what could go
 *		wrong...
 *
 *		Perhaps I should make this message sound a little less
 *		like an error, since the error code is intended for helpful
 *		messages? :-)
 */
int cmd_help(struct conn * const c)
{
	numeric(c, 414, "Sorry, no detailed help; use standard FTP commands.");
	return 1;
}

/*
 * cmd_quit():	Handles the QUIT command, which shuts down the control
 *		and data sockets.
 */
int cmd_quit(struct conn * const c)
{
	numeric(c, 221, "Have a nice day!");
	destroy_conn(c);
	return 0;
}

/*
 * cmd_rein():  Handle the REIN command, which does close to a full reset
 *		of the connection. Much of the code here is intentionally
 *		copied directly from alloc_new_conn() -- perhaps we should
 *		modularize this?
 */
int cmd_rein(struct conn * const c)
{
	destroy_ftran(c->transfer);
	c->buf_len = c->auth = c->rest_pos = 0;

	/* equals: strcpy(c->curr_dir, "/") ; strcpy(c->last_cmd, ""); */
	c->curr_dir[0] = '/';
#if WANT_FULLSCREEN
	c->curr_dir[1] = c->last_cmd[0] = '\0';
#else
	c->curr_dir[1] = '\0';
#endif

	time(&(c->last_transfer));
	numeric(c, 220, "BetaFTPD " VERSION " ready.");

	return 1;
}

#if DOING_PROFILING
/*
 * cmd_exit():	Handles the EXIT command, my own `extension' to the
 *		FTP protocol... IMPORTANT: Only to be used for profiling
 *		purposes!! (It's needed to get some profiling data out
 *		of the server after compiling it with -pg, since such data
 *		is only written on a clear exit()). Any user (even those
 *		not logged in) can issue an EXIT, and make the server shut
 *		down without clearing any sockets etc. In other words:
 *		Don't use it on a production site.
 */
void cmd_exit(struct conn * const c)
{
	while (first_conn->next_conn)
		destroy_conn(first_conn->next_conn);
	exit(0);
}
#endif

/*
 * parse_command():
 *		Gets a command from c->recv_buf, determines which command
 *		it is, sets proper effective user-ID and calls the command
 *		handler. Finally, it cleans up.
 *
 *		To me, this command seems optimizable, but I'm not really
 *		sure where :-)
 */
void parse_command(struct conn *c)
{
	int cmlen;
	const struct handler *h = handler_table;  	/* first entry */

	if (c == NULL) return;

	/* strip any leading non-ASCII characters (including CR/LFs) */
	while (c->buf_len > 0 && (c->recv_buf[0] < 'a' || c->recv_buf[0] > 'z')
			      && (c->recv_buf[0] < 'A' || c->recv_buf[0] > 'Z')) {
		remove_bytes(c, 1);		/* not good */
	}

	/* scan, searching for CR or LF */	
	cmlen = strcspn(c->recv_buf, "\r\n");
	if (cmlen >= c->buf_len) return;

#if WANT_FULLSCREEN
	strncpy(c->last_cmd, c->recv_buf, cmlen);
	c->last_cmd[cmlen] = 0;
#endif

	do {
		if ((cmlen >= (strlen(h->cmd_name) + h->add_cmlen)) &&
		    (strncasecmp(c->recv_buf, h->cmd_name, strlen(h->cmd_name)) == 0)) {
			if (c->auth < h->min_auth) {
				numeric(c, 503, "Please login with USER and PASS.");
				while (c->recv_buf[0] != '\n') remove_bytes(c, 1);
			} else {
				char schar;

#if !WANT_NONROOT
				if (h->do_setuid) {
					seteuid(c->uid);
				} else {
					seteuid(0);
				}
#endif

				remove_bytes(c, strlen(h->cmd_name));
				cmlen -= strlen(h->cmd_name);
				while (c->recv_buf[0] == ' ') {
					remove_bytes(c, 1);
					cmlen--;
				}

				schar = c->recv_buf[cmlen];
				c->recv_buf[cmlen] = 0;

				/* result of zero means the connection is freed */
				if (h->callback(c)) {
					c->recv_buf[cmlen] = schar;
#if !WANT_NONROOT
					if (h->do_setuid) seteuid(getuid());
#endif
					remove_bytes(c, cmlen);
				}
			}
			return;
		}
	} while ((++h)->callback != NULL);

	numeric(c, 500, "Sorry, no such command.");
	remove_bytes(c, cmlen); 
}

/*
 * prepare_for_transfer():
 *		Prepares an ftran object for a file transfer, setting
 *		file size, opening sockets etc.
 *
 *		nonroot notice: prepare_for_transfer() assumes all access
 *		checks are already done.
 */
void prepare_for_transfer(struct ftran *f)
{
#if WANT_NONROOT
#warning No nonroot checking for prepare_for_transfer() yet
#endif

#if HAVE_MMAP
	/* mmap doesn't make temp files for dir listings */
	if (!f->dir_listing) {
#endif

		f->size = lseek(f->local_file, 0, SEEK_END);
		errno = 0;
#if WANT_UPLOAD
		if (f->upload == 0 || f->append == 0 || f->owner->rest_pos != 0)
#endif 
			lseek(f->local_file, f->owner->rest_pos, SEEK_SET);
#if HAVE_MMAP
	}
#endif
	
	if (f->state == 1) {		/* PASV connection */
		f->state = 2;		/* waiting */
	} else if (f->state == 3) {	/* PORT connection */
		f->state = 4;
		connect(f->sock, (struct sockaddr *)&f->sin, sizeof(f->sin));
		add_fd(f->sock, POLLOUT);
	}
	time(&(f->tran_start));
}

/*
 * decode_mode():
 *		Takes a mode_t argument (from a `struct stat'), and
 *		returns the proper dirlist letter for that type.
 *
 *		Note: S_IFLNK seems to be broken, or perhaps I just have
 *		missed something (S_IFLNK is always set for all *files* on
 *		my glibc 2.0.111 system).
 *
 *		The most common cases are put first, for speed :-)
 */
char decode_mode(mode_t mode) {
	if (S_ISREG(mode))  return '-';
	if (S_ISDIR(mode))  return 'd';
	if (S_ISLNK(mode))  return 'l';
	if (S_ISBLK(mode))  return 'b';
	if (S_ISCHR(mode))  return 'c';
	if (S_ISSOCK(mode)) return 's';
	if (S_ISFIFO(mode))  return 'f';

	return '-';
}

/*
 * translate_path():
 *		Take an FTP path, do all neccessary root_dir checks,
 *		change to the correct directory and return the proper
 *		file name to open/stat/whatever. The path returned is
 *		relative to the current directory (NOT absolute). chdir()
 *		in any way will `destroy' this argument.
 *
 *		Note that `path' will be _changed_, and used as a return pointer
 *		base. Do not attempt to free the result from this function --
 *		if you need to, free path instead.
 */
char *translate_path(struct conn * const c, char * const path)
{
	char *ptr = NULL;

	/* chdir to the right dir, then chop it off */
	chdir(c->curr_dir);

	ptr = strrchr(path, '/');
	if (ptr != NULL) {
		char save_char = ptr[0];
		ptr[0] = 0;

		if (do_chdir(c, path) == -1) {
			return NULL;
		}
		ptr[0] = save_char;
		ptr++;
	} else {
		ptr = path;
	}
	return ptr;
}

/*
 * do_openfile():
 *		Opens the file PATH with access parameters FLAGS, translating
 *		paths and checking permissions as neccessary. Generally, this
 *		should be used whenever you need an open().
 *
 *		The parameters might be a bit confusing. To clarify them a bit:
 *		c:		IN/OUT (will be changed)
 *		path:		IN (but _will_ be changed)
 *		filename:	OUT
 *		flags:		IN
 *		check_perm:	IN
 */
int do_openfile(struct conn * const c, char * const path,
		char * const filename, const int flags
#if WANT_NONROOT
		, const int check_permission
#endif
)
{
	char *ptr;
	struct stat buf;

#if WANT_NONROOT
	if (nr_check_permission(c->uid, path, check_permission, 0, NULL) == -1) {
		return -1;
	}
#endif

	ptr = translate_path(c, c->recv_buf);
	if (ptr == NULL) return -1;

#if WANT_UPLOAD
	if ((flags & O_CREAT) == 0) {
#endif
		TRAP_ERROR(stat(ptr, &buf) == -1, 550, return -2);
 		if (!S_ISREG(buf.st_mode)) {
			numeric(c, 550, "Not a plain file.", ptr);
			return -2;
 		}
#if WANT_UPLOAD
	}
#endif

	if (filename != NULL) {	/* filename should always be != NULL */
		strcpy(filename, ptr);
	}
	return open(ptr, flags, 0666);
}

/*
 * prepare_for_listing():
 *		Parse list options, put them back into the list_options
 *		structure lo, and make temporary room for the list.
 */
int prepare_for_listing(struct conn * const c, char ** const ptr,
			struct list_options * const lo)
{
#if !HAVE_MMAP
	char *tfname;
#endif
	struct ftran *f = c->transfer;
	char *tmp;
	char *optr = NULL, *fptr = NULL; 
#if WANT_NONROOT
	char chd[512];
#endif

#if WANT_NONROOT
#warning No nonroot checking for prepare_for_listing() yet
#endif

	if ((f == NULL) || ((f->state != 1) && (f->state != 3))) {
		numeric(c, 425, "No data connection set up; please use PASV or PORT.");
		return -1;
	}

	/*
	 * A little parameter scanning is required here. There can only
	 * be two parts: the directory name, and any options. We'll find
	 * any options first.
	 */
	if (c->recv_buf[0] == '-') {
		optr = c->recv_buf;
	} else {
		optr = strstr(c->recv_buf, " -");
	}

	/* Then see if there are any options to parse. */
	if (optr != NULL) {
		while (*++optr) {
			switch (*optr & (255-32)) {	/* uppercase */
			case 'R':	/* actually case sensitive... */
				lo->recursive = 1;
				break;
			case 'L':
				lo->long_listing = 1;
				break;
			case 'F':
				lo->classify = 1;
				break;
			case ' ':
				fptr = optr + 1;
				*(optr--) = 0;
				break;
			default:
				break;
			}
		}
	} else {
		fptr = c->recv_buf;
	}
	
	/* then we chdir to the dir in fptr (if any) */
	tmp = fptr ? strrchr(fptr, '/') : NULL;
	if (tmp != NULL) {
		tmp[0] = 0;
		if (do_chdir(c, fptr) == -1) return -1;
		fptr = tmp + 1;
	} else {
		/* current directory */
		TRAP_ERROR(chdir(c->curr_dir) == -1, 550, return -1);
	}

	/* if no argument, choose all files */
	if (fptr == NULL || fptr[0] == 0) {
		fptr = "*";
	} else {
		/* we need to check if the last part is a directory (no -d switch) */
		struct stat buf;
		if (stat(fptr, &buf) == 0 && S_ISDIR(buf.st_mode)) {
			TRAP_ERROR(chdir(fptr) == -1, 550, return -1);
			fptr = "*";
		}
	}
	*ptr = fptr;

#if WANT_NONROOT
	getcwd(chd, 512);
	if (nr_check_permission(c->uid, chd, 4, 1, NULL) == -1) {
	 	numeric(c, 550, "Permission denied");
		return -1;
	}
#endif

#if !HAVE_MMAP
	tfname = tempnam(NULL, "ftp");

#if WANT_NONROOT
	if (tfname == NULL) tfname = tempnam("/", "ftp");
#endif

	TRAP_ERROR(tfname == NULL, 550, return -1);
	strcpy(f->filename, tfname);
	free(tfname);

	f->local_file = open(f->filename, O_RDWR | O_CREAT | O_TRUNC, 0666);
	TRAP_ERROR(f->local_file == -1, 550, return -1);
#endif
	f->dir_listing = 1;
#if WANT_UPLOAD
	f->upload = 0;
#endif

	return 0;
}

/*
 * classify():	Takes a mode_t argument (from `struct stat'), and returns
 *		the parameter to be used in an `ls -F'-style listing.
 */
char classify(const mode_t mode)
{
	if (S_ISREG(mode)) {
		if (mode & (S_IXUSR | S_IXGRP | S_IXOTH)) {
			return '*';
		} else {
			return '\0';
		}
	}
	if (S_ISDIR(mode)) return '/';
	if (S_ISLNK(mode)) return '@';
	if (S_ISSOCK(mode)) return '=';
	if (S_ISFIFO(mode)) return '|';
	return '\0'; 
}
