/**setSAA7121.c
 * Small utility that configure a SAA7121 using Mobilygen's libi2c
 * For more information about SAA7121 check the Philips datasheet
 */
 
/*******************************************************************************
* 
* The content of this file or document is CONFIDENTIAL and PROPRIETARY
* to Mobilygen Corporation.  It is subject to the terms of a
* License Agreement between Licensee and Mobilygen Corporation.
* restricting among other things, the use, reproduction, distribution
* and transfer.  Each of the embodiments, including this information and
* any derivative work shall retain this copyright notice.
* 
* Copyright 2005 Mobilygen Corporation.
* All rights reserved.
* 
* QuArc is a registered trademark of Mobilygen Corporation.
* 
*******************************************************************************/

#include <linux/errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <i2c.h>
#include "SAA7121.h"

#define VERBOSE(arg...) do { if(verbose) { printf(arg); printf("\n"); }; } while(0)
#define ERROR(arg...) do { if(!quiet) { fprintf(stderr, "ERROR: ");fprintf(stderr,arg); fprintf(stderr,"\n"); } } while(0)
#define DISPLAY(arg...) do { if(!quiet) { printf(arg); printf("\n"); } } while(0)

static int verbose=0, quiet=0;

void help(char *prog)
{
    printf("Syntax is: %s [ <7121_dev> <saa7121_params> [ --verbose | --quiet | ] | --help ]\n",prog);
    printf("with:\n");
    printf("\t--verbose or -v: Set to make program more verbose.\n");
    printf("\t--quiet or -q: If set the program will not print anything (even errors).\n");
    printf("\t--help or -h: Display this help.\n");
    printf("\t<saa7121_params>: Parameter to configure SAA7121, can be:\n");
    printf("\t  --bus <i2c_bus> : Specify I2C bus to open(default /dev/i2c).\n");
    printf("\t  --dev <0xNN> : Specify device address on I2C bus.\n");
    printf("\t  --ntsc : Set to default NTSC config.\n");
    printf("\t  --pal  : Set to default PAL config.\n");
    printf("\t  --colorbars : Enable colarbars\n");
    printf("\t  --palcb     : Enable PAL colorbars\n");
    //printf("\t  --config  <mode> : Specify the mode, can be\n");
    //printf("\t                    \"ntsc\", \"pal\", \"colorbars\", \"palcb\"\n");
    printf("\t  --writeaddr <0xNN>  : Defines subaddress for data write.\n");
    printf("\t  --writedata <0xNN> [<0xNN>...<0xNN>] : Write data, range specifies multi-btye.\n");
    printf("\t  --readaddr  <0xNN>  : Defines subaddress for data read(one byte read by default).\n");
    printf("\t  --readsize  <NN>    : Define to read more than one byte.\n");
    printf("\n");
    return;
}
    
int 
main(int argc, char *argv[])
{   

    int count,res,ret;
    i2c_handle_t i2cfd;
    char *i2c_bus=NULL;
    int i2c_dev= 0x88;    // addr for device on i2c bus
    int i2c_subaddr = 0;  // subaddr to start read/write access
    int addr_type;
    int read_subaddr = 0;
    int write_subaddr = 0;
    short dowrite = 0;  // 0x00 is RO but a block write may start from there anyway
    int write_data = 0;
    int write_block_index = 0;
    int write_size = 0;
    int read_size = 0;
    short doread = 0;
    char read_data;

    enum config_e config = NO_CONFIG;

    for(count=1;count<argc;count++) {   
		if((strcmp(argv[count],"-v")==0) || (strcmp(argv[count],"--verbose")==0))
			verbose=1;
		else if((strcmp(argv[count],"-q")==0) || (strcmp(argv[count],"--quiet")==0))
			quiet=1;
		else if((strcmp(argv[count],"-h")==0) || (strcmp(argv[count],"--help")==0)) {   
			help(argv[0]);
			return 1;
		}
		else if(strcmp(argv[count],"--bus")==0)
			i2c_bus=argv[++count];
		else if(strcmp(argv[count],"--dev")==0) {
			i2c_dev=strtol(argv[++count], NULL, 16);
			#if 0
			if (i2c_dev != 0x88) {
				// XXX what are valid addresses?
				ERROR("Device address must be 0x88\n");
				return -1; 
			}
			#endif
		}
		// these table/values were taken from the 7114.i2c scripts
		// in /home/seve1/i2c
		else if(strcmp(argv[count],"--ntsc")==0)
			config = NTSC; 
		else if(strcmp(argv[count],"--pal")==0)
			config = PAL; 
		else if(strcmp(argv[count],"--colorbars")==0)
			config = NTSCCB; 
		else if(strcmp(argv[count],"--palcb")==0)
			config = PALCB; 
		// just a possible way to avoid lot of cmdline switches
		else if(strcmp(argv[count],"--config")==0) {
			count++;
			if(strcmp(argv[count],"ntsc")==0) 
				config = NTSC; 
			else if(strcmp(argv[count],"pal")==0) 
				config = PAL; 
			else if(strcmp(argv[count],"ntsccb")==0) 
				config = NTSCCB; 
			else if(strcmp(argv[count],"palcb")==0) 
				config = PALCB; 
			else {
				ERROR("Unknown config request\n");
				help(argv[0]);
				return -1;
			}
		}
		// provide a way to access any single subaddress
		else if(strcmp(argv[count],"--writeaddr")==0) {
				write_subaddr=strtol(argv[++count], NULL, 16);
			dowrite++;
		}
		else if(strcmp(argv[count],"--writedata")==0) {
			write_block_index = count+1;
			while(count+1 < argc && strncmp(argv[count+1], "--", 2) != 0) {
				write_size++;count++;
			}
		}
		else if(strcmp(argv[count],"--readaddr")==0) {
			read_subaddr=strtol(argv[++count], NULL, 16);
			doread++; // in case starting from 0x00
		}
		// or can do a block read if this is set
		else if(strcmp(argv[count],"--readsize")==0)
				read_size=atoi(argv[++count]);
		else {   
			fprintf(stderr,"Unknown option %s\n",argv[count]);
			help(argv[0]);
			return -1;
		};
    }//  for(count=1;count<argc;count++) END

    VERBOSE("%s: compiled on %s at %s.",argv[0],__DATE__,__TIME__);

    if (i2c_bus == NULL) {
	i2c_bus = "/dev/i2c";
    }

    if (i2c_dev == 0) {
	ERROR("Must specify an I2C device sub address\n");
	help(argv[0]);
        return -1;
    }
    VERBOSE("Accessing I2C bus %s, device address 0x%0x", i2c_bus, i2c_dev);
    addr_type = ((i2c_subaddr & 0xF0)==0xA0  ?  I2CDEV_ADDR16 : I2CDEV_STD);

    i2cfd=i2c_open(i2c_bus, i2c_dev, addr_type);
    if(i2cfd<0) {  
	ERROR("Failed to open I2C device %s, libi2c returned %d.\n", i2c_bus, i2c_dev);
        return -1;
    }

    if (config) {
	unsigned char *config_data;
	int config_data_size = 0;
	switch (config) {
	    case NTSC    : config_data = ntsc_data;
			   config_data_size = sizeof(ntsc_data);
			   break;
	    case PAL     : config_data = pal_data;
			   config_data_size = sizeof(pal_data);
			   break;
	    case NTSCCB  : config_data = ntsc_colorbars_data;
			   config_data_size = sizeof(ntsc_colorbars_data);
			   break;
	    case PALCB   : config_data = pal_colorbars_data;
			   config_data_size = sizeof(pal_colorbars_data);
			   break;
	    default: 
			   ERROR("Unknown config\n"); 
			   i2c_close(i2cfd);
			   return -1;
	}

	ret=i2c_write_block(i2cfd, 0, config_data, config_data_size);
	if(ret<0) {   
	    fprintf(stderr,"%s: failed to write I2C dev 0x%0x: error %d\n",argv[0], i2c_dev, ret);
	    i2c_close(i2cfd);
	    return -1;
	}
	i2c_close(i2cfd);
	return 0;

    }

    if (dowrite) { 

	if (write_size >= 1) { 
	    int i=0, j=0;
	    unsigned char *write_block_data = (char *) malloc(write_size*sizeof(char));

	    if (write_block_data == NULL) {
		ERROR("Failed to alloc memory for block write\n");
		i2c_close(i2cfd);
		return -1;
	    }
	    for (i=0,j=write_block_index; i<write_size;i++, j++)
		write_block_data[i] = strtol(argv[j], NULL, 16);

	    VERBOSE("Write block data of size %d starting at address 0x%02x\n", write_size, write_subaddr);
	    if (i2c_write_block(i2cfd, write_subaddr, write_block_data, write_size)) {
		ERROR("Failed to write block data to sub_address 0x%02x\n", write_subaddr);
		i2c_close(i2cfd);
		return -1;
	    }
	}
	else {
	    ERROR("Failed to provide data for write command\n");
	    i2c_close(i2cfd);
	    return -1;
	}
    }

    if (doread) {

	int i;
	unsigned char *tmpdata=NULL;

	if (read_size <= 0)
	    read_size = 1;

	tmpdata=(char *)malloc(read_size);
	if (tmpdata == NULL) {
	    ERROR("Failed to alloc memory for block read\n");
	    i2c_close(i2cfd);
	    return -1;
	}
	VERBOSE("Read %d bytes at address 0x%0x\n", read_size, read_subaddr);
	if (i2c_read_block(i2cfd,read_subaddr,tmpdata, read_size)) {
	    ERROR("Failed block read data starting at sub_address 0x%02x\n", read_subaddr);
	    i2c_close(i2cfd);
	    free(tmpdata);
	    return -1; 
	}
	else {
	    for (i=0; i<read_size; i++)
		 printf("0x%02x ",tmpdata[i]);
	    printf("\n");
	}
	free(tmpdata);
    }
    return 0;
}
