/**i2c.h
 * Header for an I2C library based on Linux I2C kernel driver
 */
 
/*******************************************************************************
* 
* The content of this file or document is CONFIDENTIAL and PROPRIETARY
* to Mobilygen Corporation.  It is subject to the terms of a
* License Agreement between Licensee and Mobilygen Corporation.
* restricting among other things, the use, reproduction, distribution
* and transfer.  Each of the embodiments, including this information and
* any derivative work shall retain this copyright notice.
* 
* Copyright 2004 Mobilygen Corporation.
* All rights reserved.
* 
* QuArc is a registered trademark of Mobilygen Corporation.
* 
*******************************************************************************/

#ifndef __I2C_H
#define __I2C_H

#include <stdarg.h>

#define I2C_DEFAULT_BUSDEV      "/dev/i2c"
#define I2C_BUSDEV_MAXNAME      20

#define I2C_SUCCESS         (0)   /* Everything is great ... */ 
#define I2C_ERR_HANDLE      (-1)  /* Handle not found */
#define I2C_ERR_DRV         (-2)  /* Driver error other than transmission */
#define I2C_ERR_TX          (-3)  /* Tx error (wrong address ?) */
#define I2C_ERR_BADPARAM    (-4)  /* Bad parameter */
#define I2C_ERR_MEM         (-5)  /* malloc failed */
#define I2C_ERR_NA          (-6)  /* Parameter value not supported */

typedef enum {
    I2CDEV_STD=0,     /* Normal device, follow I2C standard */
    I2CDEV_ADDR16=1,  /* 16bits address, used for EEPROMs */
    I2CDEV_SCCB=2,
    I2CDEV_NOSUBADDR=3,
} i2c_dev_t;

/* No one should modify a handle or what is inside */
typedef int i2c_handle_t;

#ifdef    __cplusplus
extern "C" {
#endif /* __cplusplus */

/**i2c_open
 * Return an i2c handler for a specific device on the bus
 * \param   busdev      Linux device access inode, if NULL use I2C_BUS_MAXNAME, 
 *                      must be less than I2C_BUSDEV_MAXNAME chars.
 * \param   dev_address I2C address of the device (10bits and 7bits modes are supported).
 * \param   type        Device type.
 * \return handle to use with other APIs if ok, negative value otherwise
 */
i2c_handle_t i2c_open(char *busdev, int dev_address, i2c_dev_t type);

/**i2c_read
 * Read a data from an I2C device
 * \param   i2c_h       Handler returned by i2c_open.
 * \param   sub_address Register that you want to access.
 * \param   datap       Address where data will be stored
 * \return I2C_SUCCESS if ok
 */
int i2c_read(i2c_handle_t i2c_h, int sub_address, unsigned char *datap);

/**i2c_write
 * Write a data to an I2C device
 * \param   i2c_h       Handler returned by i2c_open.
 * \param   sub_address Register that you want to access.
 * \param   data        Data to write
 * \return I2C_SUCCESS if ok
 */
int i2c_write(i2c_handle_t i2c_h, int sub_address, unsigned char data);

/**i2c_read_block
 * Read a block of data from an I2C device
 * \param   i2c_h       Handler returned by i2c_open.
 * \param   sub_address 1st Register that you want to access.
 * \param   datap       Address where data will be stored
 * \param   nBytes      Number of bytes to read
 * \return I2C_SUCCESS if ok
 * \WARNING the behaviour is device dependent. This API will generate an address
 *          cycle and then generate nBytes read, device will determine if address
 *          is incremented or not.
 */
int i2c_read_block(i2c_handle_t i2c_h, int start_sub_address, unsigned char *datap, int nBytes);

/**i2c_write_block
 * Write a block of data to an I2C device
 * \param   i2c_h       Handler returned by i2c_open.
 * \param   sub_address 1st Register that you want to access.
 * \param   datap       Address where data are stored
 * \param   nBytes      Number of bytes to write
 * \return I2C_SUCCESS if ok
 * \WARNING the behaviour is device dependent. This API will generate an address
 *          cycle and then generate nBytes write, device will determine if address
 *          is incremented or not.
 */
int i2c_write_block(i2c_handle_t i2c_h, int start_sub_address, unsigned char *datap, int nBytes);

/**i2c_page_read
 * Read a data from an I2C device
 * \param   i2c_h       Handler returned by i2c_open.
 * \param   page  Index to set of pages you want to access
 * \param   sub_address Register that you want to access.
 * \param   datap       Address where data will be stored
 * \return I2C_SUCCESS if ok
 */
int i2c_page_read(i2c_handle_t i2c_h, int page, 
		int sub_address, unsigned char *datap);

/**i2c_page_write
 * Write a data to an I2C device
 * \param   i2c_h       Handler returned by i2c_open.
 * \param   page  Index to set of pages you want to access
 * \param   sub_address Register that you want to access.
 * \param   data        Data to write
 * \return I2C_SUCCESS if ok
 */
int i2c_page_write(i2c_handle_t i2c_h, int page, 
		int sub_address, unsigned char data);

/**i2c_read_page_block
 * Read a block of data from an I2C device
 * \param   i2c_h       Handler returned by i2c_open.
 * \param   page  Index to set of pages you want to access
 * \param   sub_address 1st Register that you want to access.
 * \param   datap       Address where data will be stored
 * \param   nBytes      Number of bytes to read
 * \return I2C_SUCCESS if ok
 * \WARNING the behaviour is device dependent. This API will generate an address
 *          cycle and then generate nBytes read, device will determine if address
 *          is incremented or not.
 */
int i2c_read_page_block(i2c_handle_t i2c_h, int page, 
		int start_sub_address, unsigned char *datap, int nBytes);

/**i2c_write_page_block
 * Write a block of data to an I2C device
 * \param   i2c_h       Handler returned by i2c_open.
 * \param   page  Index to set of pages you want to access
 * \param   sub_address 1st Register that you want to access.
 * \param   datap       Address where data are stored
 * \param   nBytes      Number of bytes to write
 * \return I2C_SUCCESS if ok
 * \WARNING the behaviour is device dependent. This API will generate an address
 *          cycle and then generate nBytes write, device will determine if address
 *          is incremented or not.
 */
int i2c_write_page_block(i2c_handle_t i2c_h, int page,
		int start_sub_address, unsigned char *datap, int nBytes);


/**i2c_close
 * Close a device opened with i2c_open
 * \return I2C_SUCCESS if ok
 */
int i2c_close(i2c_handle_t i2c_h);

#ifdef    __cplusplus
} /* extern "C" */
#endif /* __cplusplus */

#endif  /* ifndef __I2C_H */
