/*
 * FILE NAME mobi_gpio.h
 *
 * BRIEF MODULE DESCRIPTION
 *	API to GPIO device.
 *
 * Copyright 2004 Mobilygen Corp.
 * Copyright 2009 Maxim IC
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __MOBI_GPIO_H
#define __MOBI_GPIO_H

#include <linux/ioctl.h>

/* Directory in /proc filesystem where stats and debug hooks will be */
#define GPIO_PROCDIR        "driver/gpio"

/* gpio pin selection. Selected using the 7 LSB of the minor number */
#define GPIO_PINFIRST	0
#define GPIO_PINLAST 	127
#define GPIO_PINMASK	0x7F
/* gpio bank selection. are the last 16 minor numbers with the last one being the magic bank */
#define GPIO_BANKFIRST	240
#define GPIO_BANKLAST	254
#define GPIO_BANKMASK	0xFF
#define GPIOBANK_MAGIC_MINOR    255 
        /* Magic minor that will make this driver ioctl behave like the old one */
/* Access mode arg. Selected using bits 7 of the minor number */
#define GPIO_INTOFFS	7
#define GPIO_INTMASK	0x1

/* GPIO ioctl */
#define AU1000_GPIO_IOC_MAGIC 'g'
#define GPIO_READ           _IO (AU1000_GPIO_IOC_MAGIC, 0)
#define GPIO_SET		    _IO (AU1000_GPIO_IOC_MAGIC, 1)
#define GPIO_CLEAR	        _IO (AU1000_GPIO_IOC_MAGIC, 2)
#define GPIO_INPUT  	    _IO (AU1000_GPIO_IOC_MAGIC, 4)
#define GPIO_RWTIMING	    _IOR (AU1000_GPIO_IOC_MAGIC, 5, unsigned long)
/* NOTE ABOUT GPIO READ AND WRITE:
 * read and write are sending the char stream to the GPIO pin
 * each byte being a GPIO level (>0 is 1 and 0 is 0), spacing
 * each R/W by GPIO_RWTIMING (in us) (0 initially)
 */

/*
 * GPIO bank driver (different major)
 */

/* Per bank IOCTLS (on a different major number and not always available) */
#define GPIOBANK_IOC_MAGIC 'G'
#define GPIOBANK_SETGRP     _IOW (GPIOBANK_IOC_MAGIC, 0, unsigned long)
#define GPIOBANK_GETGRP     _IOR (GPIOBANK_IOC_MAGIC, 1, unsigned long)
#define GPIOBANK_INPUT	    _IOW (GPIOBANK_IOC_MAGIC, 2, unsigned long)
#define GPIOBANK_WRITE	    _IOW (GPIOBANK_IOC_MAGIC, 3, unsigned long)
#define GPIOBANK_READ		_IOR (GPIOBANK_IOC_MAGIC, 4, unsigned long)
#define GPIOBANK_WRISATOMIC _IO (GPIOBANK_IOC_MAGIC, 5)
#define GPIOBANK_RWTIMING	_IOW (AU1000_GPIO_IOC_MAGIC, 6, unsigned long)
/* NOTE ABOUT GPIOBANK READ AND WRITE:
 * read and write are sending the WORD stream to the GPIO bank
 * each word being a BANK level (each bit being a GPIO: GP31...GP0), spacing
 * each bank R/W by GPIO_RWTIMING (in us) (0 initially)
 */

#ifdef __KERNEL__ 

#include <linux/types.h>
#include <linux/proc_fs.h>

#define GPIODRV_NAME_MAXLEN 32

struct gpio_func
{   int (*input) (int gpio);   /* Set GPIO pin as an input */
    int (*read) (int gpio);    /* Read GPIO pin status */
    int (*set) (int gpio);     /* Set GPIO pin level to 1 */
    int (*clear) (int gpio);   /* Set GPIO pin level to 0 */
    unsigned long (*bankrd) (int bank, unsigned long mask);            /* Read several pin status at the same time on the same bank */
    int (*bankwr) (int bank, unsigned long mask, unsigned long val);   /* Write several pin status at the same time on the same bank */
    int (*bankisatomic) (int bank);    /* Return 1 if gpio_bankwr will simultaneously write all pins */
    int (*open) (int gpio);        /* Register pin, should be called first before anything else */
    int (*release) (int gpio);      /* Release pin, should be called last */
    int (*proc_rd) (char *buf, char **start, off_t offset,int count, int *eof, void *data);    /* Used to see pin status in /proc fs */
    int (*proc_wr) (struct file *file, const char *buffer, unsigned long count, void *data);   /* Used to change pin status using /proc fs */
};

/**gpio_register_driver
 * Every GPIO driver should call this API to register itself
 */
extern int gpio_register_driver(char *strDrvName, struct gpio_func *gpioFuncList);

#ifdef CONFIG_PROC_FS
/**gpio_get_procdir
 * Each time a driver register to the GPIO framework
 * a new directory is added is the proc fs, the GPIO
 * driver can use this directory for debug purpose
 * This API return a pointer that can be used to access this directory
 */
extern struct proc_dir_entry *gpio_get_procdir(char *strDrvName);
#endif

/**gpio_get_driver
 * Any GPIO customer must call this API to get a pointer to the driver API
 * if strDrvName is NULL it will return the first driver that registered itself
 */
extern int gpio_get_driver(char *strDrvName, struct gpio_func **gpioFuncList);

/**gpio_release_driver
 * Decrement the use counter. Any code that called gpio_get_driver must call
 * release before exiting
 */
extern int gpio_release_driver(char *strDrvName);

/**gpio_unregister_driver
 * Unregister GPIO driver
 * Only works if no one is using the driver
 */
extern int gpio_unregister_driver(char *strDrvName, struct gpio_func *gpioFuncList);

#endif
#endif
