/*
 *  Maxim MG3500 framebuffer driver
 *
 *  Copyright (C) 2008-2009 Maxim Integrated Products.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/vmalloc.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/proc_fs.h>
#include <linux/time.h>
#include <linux/uaccess.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <linux/version.h>
#include <asm/pgtable.h>
#include <asm/tlbflush.h>
#include <asm/cacheflush.h>
#include <mach/platform.h>
#include <mach/mmu_bridge_regs.h>
#include <mach/mobi_qcc.h>
#include <mach/mobi_dma.h>
#include <mach/dw_dmac_regs.h>

#include <linux/dma-mapping.h>
#include <linux/mg3500fb.h>


#define DMA_MMU_BRIDGE_BASE     (MMU_BRIDGE_BASE|0xc0000000)
#define MMU_BRIDGE_INDEX_OFFSET(i) (MMU_BRIDGE_PID0_OFFSET + (i/2)*4)
#define MMU_BRIDGE_INDEX0_CLR_AND_SET(x, y) \
	({ MMU_BRIDGE_PID0_PARTITON_ID0_CLR_AND_SET(x, y); })
#define MMU_BRIDGE_INDEX1_CLR_AND_SET(x, y) \
	({ MMU_BRIDGE_PID0_PARTITON_ID1_CLR_AND_SET(x, y); })
#ifndef FB_ACCEL_MG3500_BLITTER
#define FB_ACCEL_MG3500_BLITTER 51
#endif

#define MODNAME                 "mg3500fb"

#define PPAR                    ((struct mg3500fb_par *)info->par)
#define LINE_LENGTH(var)        (((var).xres_virtual*(var).bits_per_pixel+7)/8)
#define NUM_PARTITIONS(x)       ((((x)-1)>>11)+1)
#define VIDEOMEMSIZE            (4*SZ_1M)
#define MAX_VIDEOMEMSIZE        (32*SZ_1M)

/* PMU Defines */
/* FIXME - derive this from registers.... */
#define SEG_W 64
#define SEG_H 32
#define SEG_SIZE (SEG_W*SEG_H)
/* PMU Registers */
#define FVSizeAddr(i)  (0x20 + (i)*6)
#define FHSizeAddr(i)  (0x22 + (i)*6)
#define FFormatAddr(i) (0x24 + (i)*6)
#define PartFormatAddr(i)  (0x100 + (i/2)*2)
#define PartBaseSegAddr(i) (0x200 + (i)*2)
/* VPU Registers */
#define CursorPosn      (0x3c)
#define CursorColor0    (0x40)
#define CursorColor1    (0x44)
#define CRAMLd          (0x48)
#define CRAMData        (0x4c)
#define CursorEn        (0x50)
#define PLUTLd          (0x54)
#define PLUTData        (0x58)
#define siScrnSrc(i)    (0x400 + 0x40*(i))
#define siSrcMod(i)     (0x404 + 0x40*(i))
#define siColorKey(i)   (0x408 + 0x40*(i))
#define siSize(i)       (0x40c + 0x40*(i))
#define siPosn(i)       (0x410 + 0x40*(i))
#define siSrcPosn(i)    (0x414 + 0x40*(i))
#define siLastPixel(i)  (0x418 + 0x40*(i))
#define siZoom(i)       (0x41c + 0x40*(i))
typedef union {
	struct {
		unsigned SrcFrame:8;
		unsigned Alpha:8;
		unsigned Z:2;
		unsigned _reserved1:2;
		unsigned Fd:2;
		unsigned Fs:2;
		unsigned PM:1;
		unsigned AlphaSelect:1;
		unsigned FTU:2;
		unsigned _reserved0:3;
		unsigned En:1;
	};
	unsigned r;
} ScrnSrc;


struct mg3500fb_par {
	unsigned        si;
	unsigned        partition;
	unsigned        num_partition;
	unsigned        format;
	int             lane_choice;
	int             part_index;
	int             part_x;
	int             part_y;
	unsigned long   phy;
	unsigned long   part_base;
	int             count;

	mobi_dma_handle dmah;
	wait_queue_head_t dma_wq;
	int             dma_int;
	unsigned long   list_start;	/* Start of Memory Mapped DMAC list  */
					/* (physical address) */
	__u32           list_len;	/* Length of Memory Mapped DMAC list */

	int             last_pixel;
	struct mg3500_screen_mode mode;
	struct mg3500_color_key   ck;
	u32             pseudo_palette[16];
};

#define BAD_OSD_PARTITION \
    "You must load a valid parition id prior to accessing the framebuffer\n" \
    "eg: echo osd_partition=<parition id> >> /proc/driver/osd/partition\n"

static u_long videomemory = VIDEOMEMSIZE;
module_param(videomemory, ulong, 0);

static int osd_partition = -1;
module_param(osd_partition, int, 0);

static int osd_format = -1;
module_param(osd_format, int, 0);

static int osd_numpartition = 2;
module_param(osd_numpartition, int, 0);

static int use_dma;
module_param(use_dma, int, 0);

static struct fb_var_screeninfo mg3500fb_default __initdata = {
	.xres = 640,
	.yres = 480,
	.xres_virtual = 640,
	.yres_virtual = 960,
	.bits_per_pixel = 16,
	.activate = FB_ACTIVATE_NOW,
	.vmode = FB_VMODE_NONINTERLACED,
	.red            = {
		.offset     = 12,
		.length     = 4,
		.msb_right  = 0
	},
	.green          = {
		.offset     = 8,
		.length     = 4,
		.msb_right  = 0
	},
	.blue           = {
		.offset     = 4,
		.length     = 4,
		.msb_right  = 0
	},
	.transp         = {
		.offset     = 0,
		.length     = 4,
		.msb_right  = 0
	},
};
static struct fb_fix_screeninfo mg3500fb_fix __initdata = {
	.id         = MODNAME,
	.smem_start = 0,	/* Start of frame buffer mem */
				/* (physical address) */
	.smem_len   = 0,	/* Length of frame buffer mem */
	.type       = FB_TYPE_PACKED_PIXELS,
	.visual     = FB_VISUAL_TRUECOLOR,
	.xpanstep   = 1,
	.ypanstep   = 1,
	.ywrapstep  = 0,
	.accel      = FB_ACCEL_NONE,
	.mmio_start = 0,	/* Start of Memory Mapped I/O   */
				/* (physical address) */
	.mmio_len   = 0,	/* Length of Memory Mapped I/O  */
	.line_length = 8192
};

static struct mg3500_screen_mode mg3500fb_screen_mode __initdata = {
	.enable = 1,
	.ftu = MG3500FB_FTU_USE_PARITY_FIELD,
	.alpha_select = MG3500FB_ALPHA_SELECT_SCREEN,
	.premultiply = 1,
	.fs = MG3500FB_FS_COLOR_MODIFIER,
	.fd = MG3500FB_FD_ONE_MINUS_ALPHA_MODIFIER,
	.z = 0,
	.alpha = 255,

	.color_modifier = 255,
	.alpha_modifier = 255,

	.hzoom = 1,
	.vzoom = 1,
	.hoffset = 0,
	.voffset = 0
};
static struct mg3500_color_key mg3500fb_color_key __initdata = {
	.ck_y = 0x10,
	.ck_cb = 0x80,
	.ck_cr = 0x80,
	.ck_r = 0,
	.ck_g = 0,
	.ck_b = 0,
	.ck_mode = MG3500FB_CK_MODE_NONE
};

static int mg3500fb_open(struct fb_info *info, int user);
static int mg3500fb_release(struct fb_info *info, int user);
static int mg3500fb_check_var(struct fb_var_screeninfo *var,
			      struct fb_info *info);
static int mg3500fb_set_par(struct fb_info *info);
static int mg3500fb_setcolreg(unsigned regno, unsigned red, unsigned green,
			      unsigned blue, unsigned transp,
			      struct fb_info *info);
static int mg3500fb_pan_display(struct fb_var_screeninfo *var,
				struct fb_info *info);
static int mg3500fb_mmap(struct fb_info *info, struct vm_area_struct *vma);
static int mg3500fb_ioctl(struct fb_info *info, unsigned int cmd,
			  unsigned long arg);
static ssize_t mg3500fb_read(struct fb_info *info, char __user *buf,
			     size_t count, loff_t *ppos);
static ssize_t mg3500fb_write(struct fb_info *info, const char __user *buf,
			      size_t count, loff_t *ppos);
static int mg3500fb_cursor(struct fb_info *info, struct fb_cursor *cursor);

static struct fb_ops mg3500fb_ops = {
	.fb_open        = mg3500fb_open,
	.fb_release     = mg3500fb_release,
	.fb_check_var   = mg3500fb_check_var,
	.fb_set_par     = mg3500fb_set_par,
	.fb_setcolreg   = mg3500fb_setcolreg,
	.fb_pan_display = mg3500fb_pan_display,
	.fb_mmap        = mg3500fb_mmap,
	.fb_ioctl       = mg3500fb_ioctl,
	.fb_read        = mg3500fb_read,
	.fb_write       = mg3500fb_write,
	.fb_cursor      = mg3500fb_cursor,
	.fb_fillrect	= cfb_fillrect,
	.fb_copyarea	= cfb_copyarea,
	.fb_imageblit	= cfb_imageblit,
};

static unsigned qcc_read_short(int bid, int addr)
{
	unsigned long v;
	mobi_qcc_read(bid, addr, &v, 2);
	return v & 0xffff;
}
static int qcc_write_short(int bid, int addr, unsigned long data)
{
	return mobi_qcc_write(bid, addr, data, 2);
}
static unsigned qcc_read_long(int bid, int addr)
{
	unsigned long v;
	mobi_qcc_read(bid, addr, &v, 4);
	return v;
}
static int qcc_write_long(int bid, int addr, unsigned long data)
{
	return mobi_qcc_write(bid, addr, data, 4);
}

static unsigned get_format_id(int partition)
{
	int frn = partition/2;
	int v = qcc_read_short(QCC_BID_PMU, PartFormatAddr(frn));
	if (!(frn & 1))
		v >>= 8;
	return v & 0xf;
}
static void set_format_id(int partition, unsigned format)
{
	int frn = partition/2;
	int v = qcc_read_short(QCC_BID_PMU, PartFormatAddr(frn));
	if (!(frn & 1)) {
		v &= 0x0f;
		v |= (format<<8);
	} else {
		v &= 0x0f00;
		v |= format;
	}
	qcc_write_short(QCC_BID_PMU, PartFormatAddr(frn), v);
}

static void program_pmu(struct fb_info *info)
{
	struct mg3500fb_par *mg = info->par;
	unsigned partbase, voff;
	unsigned x, y, n;
	unsigned hoff = 2048/SEG_W;
	unsigned line_length = LINE_LENGTH(info->var);
	voff = (line_length+SEG_W-1) & ~(SEG_W-1);
	voff = (voff*2048)/SEG_SIZE;
	partbase = mg->part_base/SEG_SIZE;
	n = 0;
	for (y = 0; y < mg->part_y; y++)
		for (x = 0; x < mg->part_x; x++) {
			int addr = PartBaseSegAddr(mg->partition + n++);
			unsigned long data = partbase + x*hoff + y*voff;
			qcc_write_short(QCC_BID_PMU, addr, data);
		}
	/* FIXME - do this up front */
	for (y = 0; y < n; y += 2)
		set_format_id(mg->partition+y, mg->format);
}
static void program_index_pid(void __iomem *base, unsigned index, unsigned pid)
{
	void __iomem *addr = base + MMU_BRIDGE_INDEX_OFFSET(index);
	uint32_t part_info = readl(addr);
	if (!(index&1))
		MMU_BRIDGE_INDEX0_CLR_AND_SET(part_info, pid);
	else
		MMU_BRIDGE_INDEX1_CLR_AND_SET(part_info, pid);
	writel(part_info, addr);
}
static int program_bridge(uint32_t base_address, struct mg3500fb_par *mg)
{
	unsigned part_index = mg->part_index;
	unsigned partition = mg->partition;
	unsigned num_partition = mg->num_partition;
	unsigned offset;
	void __iomem *base = ioremap(base_address, MMU_BRIDGE_SIZE);
	if (0 == base)
		return -EINVAL;
	for (offset = 0; offset < num_partition; offset++)
		program_index_pid(base, part_index+offset, partition+offset);
	iounmap(base);
	return 0;
}

static void convert_to_yuv(u8 r, u8 g, u8 b, u8 *y, u8 *cb, u8 *cr)
{
	/* select between CCIR601 and ITU BT.709 matrices */
	int YUVMat = qcc_read_long(QCC_BID_VPU, 0x854) >> 25;
	int red = r, green = g, blue = b;

	if (!YUVMat) {
		*y  =  (((66*red + 129*green +  25*blue) + 128) >> 8) +  16;
		*cb = (((-38*red -  74*green + 112*blue) + 128) >> 8) + 128;
		*cr = (((112*red -  94*green -  18*blue) + 128) >> 8) + 128;
	} else {
		*y  =  (((47*red + 157*green +  16*blue) + 128) >> 8) +  16;
		*cb = (((-26*red -  87*green + 112*blue) + 128) >> 8) + 128;
		*cr = (((112*red - 102*green -  10*blue) + 128) >> 8) + 128;
	}
}
/* FIXME - move */

#define TINY_SHIFT 10
#define TINY_SIZE (1<<TINY_SHIFT)
#define TINY_MASK (~(TINY_SIZE-1))
#define __tiny_index(addr) (((addr) >> TINY_SHIFT) & (2047))
#define pte_offset_tiny(pmd, addr) ((unsigned long *)(__va((*(pmd)) & ~(PAGE_SIZE-1))) + __tiny_index(addr))
#define tiny_pte(addr, prot) ((((unsigned long)addr)&TINY_MASK) | PTE_EXT_AP_URW_SRW | (prot & (PTE_CACHEABLE | PTE_BUFFERABLE)) | PTE_TYPE_EXT)

static inline void set_tiny_pte(unsigned long *pte, unsigned long val)
{
	*pte = val;
	asm volatile ("mcr	p15, 0, %0, c7, c10, 1" : : "r" (pte));
	asm volatile ("mcr	p15, 0, %0, c7, c10, 4" : : "r" (pte));
}

static int  __attribute__((unused))
clear_tiny_pte(pmd_t *pmd, unsigned long addr, unsigned long end)
{
	unsigned long *pte = pte_offset_tiny(pmd, addr);
	do {
		set_tiny_pte(pte, 0);
	} while (pte++, addr += TINY_SIZE, addr != end);
	return 0;
}
static int
remap_tiny_pte(pmd_t *pmd, unsigned long addr, unsigned long end, unsigned long phys_addr, pgprot_t prot)
{
	unsigned long *pte = pte_offset_tiny(pmd, addr);
	do {
		set_tiny_pte(pte, tiny_pte(phys_addr, prot));
	} while (pte++, phys_addr += TINY_SIZE, addr += TINY_SIZE, addr != end);
	return 0;
}
static int
ioremap_tiny_pte(unsigned long addr, unsigned long phys_addr, unsigned long size, pgprot_t prot)
{
	unsigned long end = addr+size;
	unsigned long next;
	pgd_t *pgd;
	pgd = pgd_offset_k(addr);
	while (addr < end) {
		pmd_t *pmd = pmd_offset(pgd, addr);
		next = pmd_addr_end(addr, end);
		remap_tiny_pte(pmd, addr, next, phys_addr, prot);
		phys_addr += next - addr;
		addr = next;
		pgd++;
	}
	return 0;
}
static int
remap_tiny_pte_xy(pmd_t *pmd, unsigned long addr, unsigned long end, unsigned long width, unsigned long part_index, unsigned long phys_addr, pgprot_t prot)
{
	unsigned long *pte = pte_offset_tiny(pmd, addr);
	do {
		unsigned line = phys_addr & ((1<<25)-1);
		unsigned x = (line % width);
		unsigned y = (line / width);
		unsigned p = (y>>11)*(((width-1)>>11)+1) + (x>>11) + part_index;
		x &= ((1<<11)-1);
		y &= ((1<<11)-1);
		line = phys_addr & ~((1<<25)-1);
		line |= (p<<22) | (y<<11) | x;
		set_tiny_pte(pte, tiny_pte(line, prot));
	} while (pte++, phys_addr += TINY_SIZE, addr += TINY_SIZE, addr != end);

	flush_tlb_all();

	return 0;
}
static int
ioremap_tiny_pte_xy(unsigned long addr, unsigned long phys_addr, unsigned long width, unsigned long height, pgprot_t prot)
{
	unsigned long next;
	unsigned long end = addr+width*height;
	unsigned long part_index = MMU_BRIDGE_XY_TRANSLATION_PARTID_R(phys_addr);
	pgd_t *pgd;
	phys_addr &= ~((1<<25)-1);
	pgd = pgd_offset_k(addr);
	do {
		pmd_t *pmd = pmd_offset(pgd, addr);
		next = pmd_addr_end(addr, end);
		remap_tiny_pte_xy(pmd, addr, next, width, part_index, phys_addr, prot);
		phys_addr += next - addr;
		addr = next;
		pgd++;
	} while (addr < end);

	flush_tlb_all();

	return 0;
}
static int
alias_area_tiny(struct vm_area_struct *vma, unsigned long addr, unsigned long size)
{
	/* FIXME - round vma->vm_end upto PGDIR_SIZE */
	memcpy(pgd_offset(vma->vm_mm, vma->vm_start),
	       pgd_offset_k(addr),
	       sizeof(pgd_t) * (pgd_index(addr+size) - pgd_index(addr)));
	flush_cache_all();
	return 0;
}

static int
update_area_tiny(struct fb_info *info)
{
	unsigned long addr = (unsigned long)info->screen_base;
	unsigned long width = info->fix.line_length;
	unsigned long height = info->var.yres_virtual;
	unsigned long phys_addr = info->fix.smem_start;
	unsigned long size = width*height;

	/* map the start into xy range */
	ioremap_tiny_pte_xy(addr, phys_addr, width, height, 0);

	/* map the remainder (if any) into linear range */
	addr += size;
	phys_addr = PPAR->part_base + size;
	size = info->fix.smem_len - size;
	ioremap_tiny_pte(addr, phys_addr, size, 0);

	return 0;
}

/* There are 8 partition indices in the MMU bridge */
#define NUM_PART_INDEX 8
static int index_in_use;
/* allocate num contiguous indices by scanning index_in_use */
static int alloc_part_index(int num)
{
	int i;
	int mask = (1<<num)-1;
	int end = NUM_PART_INDEX - num;
	for (i = 0; i <= end; i++, mask <<= 1) {
		if (index_in_use & mask)
			continue;
		index_in_use |= mask;
		return i;
	}
	return -1;
}
static void free_part_index(int part_index, int num)
{
	int mask = (1<<num)-1;
	mask <<= part_index;
	index_in_use &= ~mask;
}

static int mg3500fb_open(struct fb_info *info, int user)
{
	struct mg3500fb_par *mg = info->par;

	if (-1 == osd_partition) {
		printk(KERN_ERR "[%s] %s\n", MODNAME, BAD_OSD_PARTITION);
		return -EAGAIN;
	}

	if (mg->count) {
		/*
		 * FIXME - eventually we will recover from this...
		 */
		if (mg->partition != osd_partition)
			return -EBUSY;

		mg->count++;
		return 0;
	}

	/*
	** must be set by writing value to /proc/driver/osd/partition
	*/
	mg->partition = osd_partition;
	mg->part_base = qcc_read_short(QCC_BID_PMU, PartBaseSegAddr(mg->partition)) * SEG_SIZE;
	mg->format = osd_format;
	mg->num_partition = osd_numpartition;
	mg->part_index = alloc_part_index(mg->num_partition);
	if (mg->part_index < 0)
		return -EAGAIN;

	/*
	** how to deal with word swapping.
	*/
	mg->lane_choice =  info->var.bits_per_pixel <=  8 ? 0 :
			   info->var.bits_per_pixel == 16 ? 1 :
							    3 ;

	mg->phy = MMU_BRIDGE_XY_TRANSLATION_PARTID_W(mg->part_index) |
		  MMU_BRIDGE_XY_TRANSLATION_MODE_W(MMU_XY_ADDRESS_TRANSLATION_MODE) |
		  MMU_BRIDGE_XY_TRANSLATION_ENDIAN_W(mg->lane_choice);

	info->screen_base = ioremap_area_tiny(mg->phy, videomemory, 0);
	if (0 == info->screen_base) {
		printk(KERN_ERR "[%s] Couldn't map tiny kernel memory\n", MODNAME);
		return -ENOMEM;
	}

	if (0 != (program_bridge(MMU_BRIDGE_BASE, mg))) {
		printk(KERN_WARNING "[%s] Couldn't write partition %d into MMU!!!\n",
			__func__, mg->partition);
		iounmap_area_tiny(info->screen_base);
		return -ENOMEM;
	}

	if (0 != (program_bridge(DMA_MMU_BRIDGE_BASE, mg))) {
		printk(KERN_WARNING "[%s] Couldn't write partition %d into MMU DMA mirror reg.!!!\n",
			__func__, mg->partition);
		iounmap_area_tiny(info->screen_base);
		return -ENOMEM;
	}

	/* This isn't exactly right... but better than being uninitialized */
	info->fix.smem_len = videomemory;
	mg3500fb_set_par(info);

	mg->count++;

	return 0;
}

static int mg3500fb_release(struct fb_info *info, int user)
{
	struct mg3500fb_par *mg = info->par;

	mg->count--;

	if (mg->count)
		return 0;

	free_part_index(mg->part_index, mg->num_partition);

	iounmap_area_tiny(info->screen_base);

	mg->phy = 0;
	info->screen_base = 0;

	return 0;
}

#ifndef MG3500FB_DMA_SYNC
#define MG3500FB_DMA_SYNC	 	_IO('m', 0x0b)
#endif

static int mg3500fb_dma_sync(struct mg3500fb_par *mg)
{
	if (mg->dma_int) {
		mg->dma_int = 0;
		return 0;
	}
	wait_event_interruptible(mg->dma_wq, 0 != mg->dma_int);
	if (mg->dma_int) {
		mg->dma_int = 0;
	} else {
		printk(KERN_WARNING "[%s] lost interrupt ..\n", MODNAME);
		return -1;
	}
	return 0;
}

static void
dma_callback(mobi_dma_handle dmah, mobi_dma_event dma_event, void *p)
{
	struct mg3500fb_par *mg = p;
	switch (dma_event) {
	case MOBI_DMA_EVENT_TRANSFER_COMPLETE:
		break;
	case MOBI_DMA_EVENT_TRANSFER_ERROR:
		printk(KERN_ERR "[%s] dma transfer error\n", MODNAME);
		break;
	default:
		printk(KERN_ERR "[%s] Unrecognized dma_event: 0x%x\n",
			MODNAME, dma_event);
		break;
	}
	mg->dma_int = 1;
	wake_up_interruptible(&mg->dma_wq);
}

static int mg3500fb_waitforvblank(void *a, u32 crtc)
{
	/* FIXME... busy wait is for suck */
#if 0
	int top = (qcc_read_long(QCC_BID_VPU, 0x80)&0x3ff);
	if (!top)
		while ((qcc_read_long(QCC_BID_VPU, 0x80)&0x3ff) > top)
			;
	if (top)
		while ((qcc_read_long(QCC_BID_VPU, 0x80)&0x3ff) != 0)
			;
#endif
	int l = (qcc_read_long(QCC_BID_VPU, 0x80)&0x7ff);
	int d = (qcc_read_long(QCC_BID_VPU, 0x840) >> 16)-1;
	if (l == d)
		while ((qcc_read_long(QCC_BID_VPU, 0x80)&0x7ff) == d)
			;
	while ((qcc_read_long(QCC_BID_VPU, 0x80)&0x7ff) != d)
		;
	return 0;
}

#ifndef FBIO_WAITFORVSYNC
#define FBIO_WAITFORVSYNC _IOW('F', 0x20, __u32)
#endif

static int mg3500fb_ioctl(struct fb_info *info, unsigned int cmd, unsigned long arg)
{
	void __user *argp = (void __user *)arg;
	switch (cmd) {
	case FBIO_WAITFORVSYNC:
		{
			u32 crtc;

			if (get_user(crtc, (__u32 __user *) arg))
				return -EFAULT;

			return mg3500fb_waitforvblank(PPAR, crtc);
		}
		break;

	case MG3500FB_GET_SCREEN_MODE:
		{
			struct mg3500_screen_mode *mode = &PPAR->mode;
			if (copy_to_user(argp, mode, sizeof(*mode)))
				return -EFAULT;

			return 0;
		}
	case MG3500FB_SET_SCREEN_MODE:
		{
			__u32 fail = 0;
			struct mg3500_screen_mode mode;

			if (copy_from_user(&mode, argp, sizeof(mode)))
				return -EFAULT;
			if ((mode.flags & MG3500FB_FLAGS_ENABLE) && mode.enable > 1)
				fail |= MG3500FB_FLAGS_ENABLE;
			if ((mode.flags & MG3500FB_FLAGS_ALPHA_SELECT) && mode.alpha_select > 3)
				fail |= MG3500FB_FLAGS_ALPHA_SELECT;
			if ((mode.flags & MG3500FB_FLAGS_FTU) && mode.ftu > 3)
				fail |= MG3500FB_FLAGS_FTU;
			if ((mode.flags & MG3500FB_FLAGS_PREMULTIPLY) && mode.premultiply > 1)
				fail |= MG3500FB_FLAGS_PREMULTIPLY;
			if ((mode.flags & MG3500FB_FLAGS_FS) && mode.fs > 3)
				fail |= MG3500FB_FLAGS_FS;
			if ((mode.flags & MG3500FB_FLAGS_FD) && mode.fd > 3)
				fail |= MG3500FB_FLAGS_FD;
			if ((mode.flags & MG3500FB_FLAGS_Z) && mode.z > 3)
				fail |= MG3500FB_FLAGS_Z;
			if ((mode.flags & MG3500FB_FLAGS_HZOOM) && (mode.hzoom < 1 || mode.hzoom > 16))
				fail |= MG3500FB_FLAGS_HZOOM;
			if ((mode.flags & MG3500FB_FLAGS_VZOOM) && (mode.vzoom < 1 || mode.vzoom > 16))
				fail |= MG3500FB_FLAGS_VZOOM;
			if (fail) {
				if (put_user(fail, &(((struct mg3500_screen_mode * __user)argp)->flags)))
					return -EFAULT;
				return -EINVAL;
			}

			if (mode.flags & MG3500FB_FLAGS_ENABLE)
				PPAR->mode.enable = mode.enable;
			if (mode.flags & MG3500FB_FLAGS_ALPHA_SELECT)
				PPAR->mode.alpha_select = mode.alpha_select;
			if (mode.flags & MG3500FB_FLAGS_FTU)
				PPAR->mode.ftu = mode.ftu;
			if (mode.flags & MG3500FB_FLAGS_PREMULTIPLY)
				PPAR->mode.premultiply = mode.premultiply;
			if (mode.flags & MG3500FB_FLAGS_FS)
				PPAR->mode.fs = mode.fs;
			if (mode.flags & MG3500FB_FLAGS_FD)
				PPAR->mode.fd = mode.fd;
			if (mode.flags & MG3500FB_FLAGS_Z)
				PPAR->mode.z = mode.z;
			if (mode.flags & MG3500FB_FLAGS_ALPHA)
				PPAR->mode.alpha = mode.alpha;
			if (mode.flags & MG3500FB_FLAGS_COLOR_MODIFIER)
				PPAR->mode.color_modifier = mode.color_modifier;
			if (mode.flags & MG3500FB_FLAGS_ALPHA_MODIFIER)
				PPAR->mode.alpha_modifier = mode.alpha_modifier;
			if (mode.flags & MG3500FB_FLAGS_HZOOM)
				PPAR->mode.hzoom = mode.hzoom;
			if (mode.flags & MG3500FB_FLAGS_VZOOM)
				PPAR->mode.vzoom = mode.vzoom;
			if (mode.flags & MG3500FB_FLAGS_HOFFSET)
				PPAR->mode.hoffset = mode.hoffset;
			if (mode.flags & MG3500FB_FLAGS_VOFFSET)
				PPAR->mode.voffset = mode.voffset;

			/* FIXME - do we always want instantaneous update? */
			if (mode.flags)
				mg3500fb_set_par(info);
			return 0;
		}
	case MG3500FB_GET_COLOR_KEY:
		{
			struct mg3500_color_key *ck = &PPAR->ck;
			if (copy_to_user(argp, ck, sizeof(*ck)))
				return -EFAULT;

			return 0;
		}
	case MG3500FB_SET_COLOR_KEY:
		{
			struct mg3500_color_key ck;

			if (copy_from_user(&ck, argp, sizeof(ck)))
				return -EFAULT;
			if (ck.flags & MG3500FB_FLAGS_CK_MODE) {
				if (ck.ck_mode > 2)
					return -EINVAL;
				PPAR->ck.ck_mode = ck.ck_mode;
			}
			if (ck.flags & MG3500FB_FLAGS_CK_YUV) {
				PPAR->ck.ck_y = ck.ck_y;
				PPAR->ck.ck_cb = ck.ck_cb;
				PPAR->ck.ck_cr = ck.ck_cr;
			} else if (ck.flags & MG3500FB_FLAGS_CK_RGB) {
				int rl = info->var.red.length;
				int gl = info->var.green.length;
				int bl = info->var.blue.length;
				int red   = ((ck.ck_r >> (8-rl)) << (8-rl)) | (ck.ck_r >> rl);
				int green = ((ck.ck_g >> (8-gl)) << (8-gl)) | (ck.ck_g >> gl);
				int blue  = ((ck.ck_b >> (8-bl)) << (8-bl)) | (ck.ck_b >> bl);
				convert_to_yuv(red, green, blue, &PPAR->ck.ck_y, &PPAR->ck.ck_cb, &PPAR->ck.ck_cr);
				PPAR->ck.ck_r = ck.ck_r;
				PPAR->ck.ck_g = ck.ck_g;
				PPAR->ck.ck_b = ck.ck_b;
			}
			/* FIXME - do we always want instantaneous update? */
			if (ck.flags)
				mg3500fb_set_par(info);
			return 0;
		}
	case MG3500FB_GET_PHYSINFO:
		{
			struct mg3500_physinfo phys;
			phys.dmah = PPAR->dmah;
			phys.list_base = PPAR->list_start;
			phys.list_len = PPAR->list_len;
			phys.part_base = PPAR->part_base;
			phys.part_len = info->fix.smem_len;
			if (copy_to_user(argp, &phys, sizeof(phys)))
				return -EFAULT;

			return 0;
		}
	case MG3500FB_DMA_SYNC:
		if (use_dma)
			return mg3500fb_dma_sync(PPAR);
	}
	return -ENOTTY;
}
static ssize_t mg3500fb_read(struct fb_info *info, char __user *buf,
			     size_t count, loff_t *ppos)
{
	void *virt = info->screen_base;
	unsigned long p = *ppos;
	int cnt = 0, xoff, yoff;
	unsigned line_size;
	unsigned long total_size;

	line_size = LINE_LENGTH(info->var);
	total_size = line_size * info->var.yres_virtual;

	if (p >= total_size)
		return 0;

	if (count >= total_size)
		count = total_size;

	if (count + p > total_size)
		count = total_size - p;

	xoff = p % line_size;
	yoff = p / line_size;
	virt += yoff*info->fix.line_length;

	while (count) {
		void *out = virt + xoff;
		int c = line_size - xoff;
		c = c < count ? c : count;
		if (copy_to_user(buf, out, c))
			return -EFAULT;
		xoff = 0;
		virt += info->fix.line_length;
		*ppos += c;
		buf += c;
		cnt += c;
		count -= c;
	}

	return cnt;
}
static ssize_t mg3500fb_write(struct fb_info *info, const char __user *buf,
			      size_t count, loff_t *ppos)
{
	void *virt = info->screen_base;
	unsigned long p = *ppos;
	int cnt = 0, err = 0, xoff, yoff;
	unsigned line_size;
	unsigned long total_size;

	line_size = LINE_LENGTH(info->var);
	total_size = line_size * info->var.yres_virtual;

	if (p > total_size)
		return -EFBIG;

	if (count > total_size) {
		err = -EFBIG;
		count = total_size;
	}

	if (count + p > total_size) {
		if (!err)
			err = -ENOSPC;

		count = total_size - p;
	}

	xoff = p % line_size;
	yoff = p / line_size;
	virt += yoff*info->fix.line_length;

	while (count) {
		void *out = virt + xoff;
		int c = line_size - xoff;
		c = c < count ? c : count;
		if (copy_from_user(out, buf, c)) {
			err = -EFAULT;
			break;
		}
		xoff = 0;
		virt += info->fix.line_length;
		*ppos += c;
		buf += c;
		cnt += c;
		count -= c;
	}
	return (cnt) ? cnt : err;
}
static int mg3500fb_cursor(struct fb_info *info, struct fb_cursor *cursor)
{
	static u8 src[1024];

	qcc_write_long(QCC_BID_VPU, CursorEn, 0);

	if (cursor->set & FB_CUR_SETPOS) {
		u32 dx, dy;

		dx = cursor->image.dx - info->var.xoffset;
		dy = cursor->image.dy - info->var.yoffset;
		dx &= 0xfff;
		dy &= 0xfff;

		qcc_write_long(QCC_BID_VPU, CursorPosn, (dy << 16) | dx);
	}

	if (cursor->set & FB_CUR_SETSIZE) {
		if (cursor->image.width > 64 || cursor->image.height > 64)
			return -ENXIO;

		memset(src, 0, sizeof(src));
	}

	if (cursor->set & FB_CUR_SETCMAP) {
		u32 fg, bg;

		if (info->fix.visual == FB_VISUAL_TRUECOLOR ||
		    info->fix.visual == FB_VISUAL_DIRECTCOLOR) {
			fg = PPAR->pseudo_palette[cursor->image.fg_color];
			bg = PPAR->pseudo_palette[cursor->image.bg_color];
		} else {
			fg = cursor->image.fg_color;
			bg = cursor->image.bg_color;
		}

		qcc_write_long(QCC_BID_VPU, CursorColor0, fg);
		qcc_write_long(QCC_BID_VPU, CursorColor1, bg);
	}

	if (cursor->set & (FB_CUR_SETSHAPE | FB_CUR_SETIMAGE)) {
		const u8 *dat = cursor->image.data;
		const u8 *msk = cursor->mask;
		const int stride = (cursor->image.width+7)/8;
		u32 x, y;

		if (cursor->image.depth != 1)
			return -ENXIO;

		/* FIXME cursor->rop support */
		for (y = 0; y < cursor->image.height; y++) {
			for (x = 0; x < cursor->image.width; x++) {
				u32 d = (dat[y*stride+x/8] >> (7-(x&7))) & 1;
				u32 m = (msk[y*stride+x/8] >> (7-(x&7))) & 1;
				u32 v = m ? (d+1) : 0;
				u32 w = src[y*16+x/4];
				w &= ~(0xc0 >> ((x&3)*2));
				w |= v << (6-(x&3)*2);
				src[y*16+x/4] = w;
			}
		}
		/* load the CRAM */
		qcc_write_long(QCC_BID_VPU, CRAMLd, 0x100);
		for (x = 0; x < 256; x++) {
			u32 v = (src[4*x] << 24) |
				(src[4*x+1] << 16) |
				(src[4*x+2] <<  8) |
				(src[4*x+3]);
			qcc_write_long(QCC_BID_VPU, CRAMData, v);
		}
	}

	if (cursor->enable)
		qcc_write_long(QCC_BID_VPU, CursorEn, 1);

	return 0;
}
static int mg3500fb_check_var(struct fb_var_screeninfo *var,
			      struct fb_info *info)
{
	struct mg3500fb_par *mg = info->par;
	u_long w, h, line_length;

	/*
	 *  Some very basic checks
	 */
	if (!var->xres)
		var->xres = 1;
	if (!var->yres)
		var->yres = 1;
	if (var->xres > var->xres_virtual)
		var->xres_virtual = var->xres;
	if (var->yres > var->yres_virtual)
		var->yres_virtual = var->yres;
	if (var->bits_per_pixel <= 1)
		var->bits_per_pixel = 1;
	else if (var->bits_per_pixel <= 2)
		var->bits_per_pixel = 2;
	else if (var->bits_per_pixel <= 4)
		var->bits_per_pixel = 4;
	else if (var->bits_per_pixel <= 8)
		var->bits_per_pixel = 8;
	else if (var->bits_per_pixel <= 16)
		var->bits_per_pixel = 16;
	else if (var->bits_per_pixel <= 32)
		var->bits_per_pixel = 32;
	else
		return -EINVAL;

	if (var->grayscale && var->bits_per_pixel > 8)
		return -EINVAL;

	if (var->xres_virtual < var->xoffset + var->xres)
		var->xres_virtual = var->xoffset + var->xres;
	if (var->yres_virtual < var->yoffset + var->yres)
		var->yres_virtual = var->yoffset + var->yres;

	line_length = LINE_LENGTH(*var);

	/* Check for sufficient number of partitions */
	if (NUM_PARTITIONS(line_length)*NUM_PARTITIONS(var->yres_virtual) >
	    mg->num_partition)
		return -ENOSPC;

	/* Round width and height to segment size */
	w = (line_length + SEG_W-1) & ~(SEG_W-1);
	h = (var->yres_virtual + SEG_H-1) & ~(SEG_H-1);

	/* Round width to stride size */
	w = (w + 1023) & ~(1023);

	/* Check for sufficient memory */
	if (w*h > info->fix.smem_len)
		return -ENOMEM;

	/*
	 * Now that we checked it we alter var. The reason being is that the
	 * video mode passed in might not work but slight changes to it might
	 * make it work. This way we let the user know what is acceptable.
	 */

	var->red.msb_right = 0;
	var->green.msb_right = 0;
	var->blue.msb_right = 0;
	var->transp.msb_right = 0;

#define RGBA(r, g, b, a) (var->red.length == r && var->green.length == g && var->blue.length == b && var->transp.length == a)
	switch (var->bits_per_pixel) {
	case 1:
	case 2:
	case 4:
	case 8:
		var->red   .length =  8, var->red   .offset =  0;
		var->green .length =  8, var->green .offset =  0;
		var->blue  .length =  8, var->blue  .offset =  0;
		var->transp.length =  8, var->transp.offset =  0;
		return 0;
	case 16:
		/* RGB444, RGBA4444 */
		if (RGBA(4, 4, 4, 0) || RGBA(4, 4, 4, 4)) {
			var->red.offset    = 12;
			var->green.offset  =  8;
			var->blue.offset   =  4;
			var->transp.offset =  0;
			return 0;
		}
		/* RGB555, RGBA5551 */
		if (RGBA(5, 5, 5, 0) || RGBA(5, 5, 5, 1)) {
			var->red.offset    = 11;
			var->green.offset  =  6;
			var->blue.offset   =  1;
			var->transp.offset =  0;
			return 0;
		}
		/* RGB565 */
		if (RGBA(5, 6, 5, 0)) {
			var->red.offset    = 11;
			var->green.offset  =  5;
			var->blue.offset   =  0;
			var->transp.offset =  0;
			return 0;
		}
		/* RGBA5641 */
		if (RGBA(5, 6, 4, 1)) {
			var->red.offset    = 11;
			var->green.offset  =  5;
			var->blue.offset   =  1;
			var->transp.offset =  0;
			return 0;
		}
		break;
	case 32:
		/* RGB888/RGBA8888 */
		if (RGBA(8, 8, 8, 0) || RGBA(8, 8, 8, 8)) {
			var->red.offset    = 24;
			var->green.offset  = 16;
			var->blue.offset   =  8;
			var->transp.offset =  0;
			return 0;
		}
	}
#undef RGBA

	printk(KERN_WARNING "[" MODNAME "]" "Bad pixel format request %dbpp %d transparency length ...\n",
		var->bits_per_pixel, var->transp.length);

	return -1;
}

/* This routine actually sets the video mode. It's in here where we set
 * the hardware state info->par and fix which can be affected by the
 * change in par. For this driver it doesn't do much.
 */
static int mg3500fb_set_par(struct fb_info *info)
{
	struct mg3500_screen_mode *mode = &(PPAR->mode);
	struct mg3500_color_key *ck = &(PPAR->ck);
	int si = PPAR->si;
	unsigned hzoom = mode->hzoom, vzoom = mode->vzoom;
	unsigned hoffset = mode->hoffset, voffset = mode->voffset;
	unsigned format, value, sw, sh, dw, dh, last_pixel, line_length;
	int lane_choice;
	ScrnSrc scrn;
	if (info->var.bits_per_pixel <= 8)
		info->fix.visual = FB_VISUAL_PSEUDOCOLOR;
	else
		info->fix.visual = FB_VISUAL_TRUECOLOR;
#define RGBA(r, g, b, a) (info->var.red.length == r && info->var.green.length == g && info->var.blue.length == b && info->var.transp.length == a)
	switch (info->var.bits_per_pixel) {
	case 1:
		format = 0x00 + info->var.grayscale;
		break;
	case 2:
		format = 0x02 + info->var.grayscale;
		break;
	case 4:
		format = 0x04 + info->var.grayscale;
		break;
	case 8:
		format = 0x06 + info->var.grayscale;
		break;
	case 16:
		if (RGBA(4, 4, 4, 0))
			format = 0x10;
		if (RGBA(4, 4, 4, 4))
			format = 0x11;
		if (RGBA(5, 5, 5, 0))
			format = 0x12;
		if (RGBA(5, 5, 5, 1))
			format = 0x13;
		if (RGBA(5, 6, 5, 0))
			format = 0x14;
		if (RGBA(5, 6, 4, 1))
			format = 0x15;
		break;
	case 32:
	default:
		format = info->var.transp.length ? 0x19 : 0x18;
	}
#undef RGBA
	lane_choice = info->var.bits_per_pixel <=  8 ? 0 :
		      info->var.bits_per_pixel == 16 ? 1 :
						       3 ;

	qcc_write_long(QCC_BID_VPU, siScrnSrc(si), 0);

	scrn.r = 0;
	scrn.En = mode->enable;
	scrn.FTU = mode->ftu;
	scrn.AlphaSelect = mode->alpha_select;
	scrn.PM = mode->premultiply;
	scrn.Fs = mode->fs;
	scrn.Fd = mode->fd;
	scrn.Z = mode->z;
	scrn.Alpha = mode->alpha;
	scrn.SrcFrame = PPAR->partition;

	qcc_write_long(QCC_BID_VPU, siSrcMod(si), (mode->color_modifier << 8) | mode->alpha_modifier);

	value  = ck->ck_y    << 24;
	value |= ck->ck_cb   << 16;
	value |= ck->ck_cr   <<  8;
	value |= ck->ck_mode <<  6;
	qcc_write_long(QCC_BID_VPU, siColorKey(si), value);

	/* Solve partition geometry */
	line_length = LINE_LENGTH(info->var);
	PPAR->part_x = NUM_PARTITIONS(line_length);
	PPAR->part_y = NUM_PARTITIONS(info->var.yres_virtual);

	PPAR->lane_choice = lane_choice;

	PPAR->phy = MMU_BRIDGE_XY_TRANSLATION_PARTID_W(PPAR->part_index) |
		    MMU_BRIDGE_XY_TRANSLATION_MODE_W(MMU_XY_ADDRESS_TRANSLATION_MODE) |
		    MMU_BRIDGE_XY_TRANSLATION_ENDIAN_W(PPAR->lane_choice);
	/* FIXME */
	/* set fix */
	info->fix.smem_start = PPAR->phy;
	line_length = (line_length + 1023) & ~1023;
	info->fix.line_length = line_length;

	/* Adjust the frame format */
	qcc_write_short(QCC_BID_PMU, FVSizeAddr(PPAR->format), info->var.yres_virtual);
	qcc_write_short(QCC_BID_PMU, FHSizeAddr(PPAR->format), info->var.xres_virtual);
	qcc_write_short(QCC_BID_PMU, FFormatAddr(PPAR->format), format);

	/* Program parition bases for given resolution */
	program_pmu(info);

	/* redo mappings */
	update_area_tiny(info);

	/* get display size */
	dw = qcc_read_long(QCC_BID_VPU, 0x840);
	dh = dw >> 16;
	dw &= 0xffff;

	sw = info->var.xres*hzoom;
	sh = info->var.yres*vzoom;
	sw = (hoffset + sw) > dw ? dw-hoffset : sw;
	sh = (voffset + sh) > dh ? dh-voffset : sh;
	last_pixel = (sw+hzoom-1)/hzoom;
	PPAR->last_pixel = last_pixel;

	qcc_write_long(QCC_BID_VPU, siSize(si), (sh << 16) | sw);
	qcc_write_long(QCC_BID_VPU, siPosn(si), (voffset << 16) | hoffset);
	qcc_write_long(QCC_BID_VPU, siSrcPosn(si), (info->var.yoffset << 20) | (info->var.xoffset << 4));
	qcc_write_long(QCC_BID_VPU, siLastPixel(si), info->var.xoffset + last_pixel);
	qcc_write_long(QCC_BID_VPU, siZoom(si), ((vzoom-1) << 12) | (hzoom-1));

	qcc_write_long(QCC_BID_VPU, siScrnSrc(si), scrn.r);

	return 0;
}
static int mg3500fb_setcolreg(unsigned regno, unsigned red, unsigned green,
			      unsigned blue, unsigned transp,
			      struct fb_info *info)
{
	int PLUTChSel = (PPAR->si - 2) << 8;
	unsigned long v;

	/* no. of hw registers */
	if (regno >= 256)
		return -EINVAL;

	/*
	 ** grayscale works only partially under directcolor
	 ** grayscale = 0.30*R + 0.59*G + 0.11*B
	 **
	 */
	if (info->var.grayscale)
		red = green = blue = (red * 77 + green * 151 + blue * 28) >> 8;

#define CNVT_TOHW(val, width)    ((((val)<<(width))+0x7FFF-(val))>>16)
	red     = CNVT_TOHW(red   , 8);
	green   = CNVT_TOHW(green , 8);
	blue    = CNVT_TOHW(blue  , 8);
	transp  = CNVT_TOHW(transp, 8);
#undef CNVT_TOHW

	v = (red << 24) | (green << 16) | (blue << 8) | (0xff - transp);
	qcc_write_long(QCC_BID_VPU, PLUTLd, PLUTChSel | regno);
	qcc_write_long(QCC_BID_VPU, PLUTData, v);

	if (info->fix.visual == FB_VISUAL_TRUECOLOR ||
	    info->fix.visual == FB_VISUAL_DIRECTCOLOR) {

		if (regno >= 16)
			return -EINVAL;

		((u32 *)(info->pseudo_palette))[regno] = v;
	}

	return 0;
}
static int mg3500fb_pan_display(struct fb_var_screeninfo *var,
				struct fb_info *info)
{
	unsigned scrn;
	int si = PPAR->si;

	/* no range checks for xoffset and yoffset,   */
	/* as fb_pan_display has already done this */
	if (var->vmode & FB_VMODE_YWRAP)
		return -EINVAL;

	/* fast path, as we need not update last_pixel */
	if (var->xoffset == info->var.xoffset) {
		qcc_write_long(QCC_BID_VPU, siSrcPosn(si), (var->yoffset << 20) | (var->xoffset << 4));

		return 0;
	}

	/* slow path, FIXME use line count and sleep, rather than disabling
	 * the screen... which can glitch */

	scrn = qcc_read_long(QCC_BID_VPU, siScrnSrc(si));
	qcc_write_long(QCC_BID_VPU, siScrnSrc(si), 0);

	qcc_write_long(QCC_BID_VPU, siSrcPosn(si), (var->yoffset << 20) | (var->xoffset << 4));
	qcc_write_long(QCC_BID_VPU, siLastPixel(si), var->xoffset + PPAR->last_pixel);

	qcc_write_long(QCC_BID_VPU, siScrnSrc(si), scrn);

	return 0;
}

static struct vm_operations_struct mg3500fb_vm_ops;

static int mg3500fb_mmap(struct fb_info *info, struct vm_area_struct *vma)
{
	unsigned long off;
	unsigned long start;
	u32 len;
	struct mg3500fb_par *mg = info->par;

	printk(KERN_DEBUG "%s: vma request %p-%p (%ld bytes).\n",
			__func__, (void *)vma->vm_start, (void *)vma->vm_end,
			vma->vm_end - vma->vm_start);

	off = vma->vm_pgoff << PAGE_SHIFT;

	/* frame buffer memory */
	start = info->fix.smem_start;
	len = PAGE_ALIGN((start & ~PAGE_MASK) + info->fix.smem_len);
	if (off >= len) {
		/* memory mapped io */
		off -= len;
		start = info->fix.mmio_start;
		len = PAGE_ALIGN((start & ~PAGE_MASK) + info->fix.mmio_len);
	}
	if (off >= len) {
		/* memory mapped dmac list area */
		off -= len;
		start = mg->list_start;
		len = PAGE_ALIGN((start & ~PAGE_MASK) + mg->list_len);
	}
	start &= PAGE_MASK;
	if ((vma->vm_end - vma->vm_start + off) > len)
		return -EINVAL;
	off += start;
	vma->vm_pgoff = off >> PAGE_SHIFT;
	vma->vm_flags |= VM_IO | VM_RESERVED;
	vma->vm_page_prot = pgprot_writecombine(vma->vm_page_prot);
	if (start != info->fix.smem_start) {
		if (io_remap_pfn_range(vma, vma->vm_start, off >> PAGE_SHIFT,
					vma->vm_end - vma->vm_start, vma->vm_page_prot))
			return -EAGAIN;
		return 0;
	}

	/* cross link user pgd to kernel tables */
	vma->vm_flags |= VM_IO | VM_RESERVED | VM_PFNMAP;
	vma->vm_ops = &mg3500fb_vm_ops;
	alias_area_tiny(vma, (unsigned long)info->screen_base, info->fix.smem_len);
	return 0;
}

static int osd_get_state(char *buffer, char **start, off_t offset, int count,
			 int *eof, void *prvdata)
{
	ScrnSrc scrn;
	char *p = buffer ;
	struct fb_info *info = prvdata;
	struct fb_var_screeninfo *v = &info->var;
	struct mg3500fb_par *mg = info->par;
	int i;
	int si = mg->si;

	p += sprintf(p, "Build : %s %s\n", __DATE__, __TIME__);
	p += sprintf(p, "partition : %d\n", mg->partition);
	if (mg->partition != -1)
		for (i = 0; i < mg->num_partition; i++)
			p += sprintf(p, "  PartBaseSeg : 0x%04x\n", qcc_read_short(QCC_BID_PMU, PartBaseSegAddr(mg->partition+i)));
	p += sprintf(p, "format_index : %d\n", mg->format);
	if (mg->format != -1) {
		p += sprintf(p, "  FHSize  : %d\n", qcc_read_short(QCC_BID_PMU, FHSizeAddr(mg->format)));
		p += sprintf(p, "  FVSize  : %d\n", qcc_read_short(QCC_BID_PMU, FVSizeAddr(mg->format)));
		p += sprintf(p, "  FFormat : 0x%02x\n", qcc_read_short(QCC_BID_PMU, FFormatAddr(mg->format)));
	}
	scrn.r = qcc_read_long(QCC_BID_VPU, siScrnSrc(si));
	p += sprintf(p, "s%dScrnSrc   : 0x%08x\n", si, scrn.r);
	p += sprintf(p, "  En          : %d\n", scrn.En);
	p += sprintf(p, "  FTU         : %d\n", scrn.FTU);
	p += sprintf(p, "  AlphaSelect : %d\n", scrn.AlphaSelect);
	p += sprintf(p, "  PM          : %d\n", scrn.PM);
	p += sprintf(p, "  Fs          : %d\n", scrn.Fs);
	p += sprintf(p, "  Fd          : %d\n", scrn.Fd);
	p += sprintf(p, "  Z           : %d\n", scrn.Z);
	p += sprintf(p, "  Alpha       : %d\n", scrn.Alpha);
	p += sprintf(p, "  SrcFrame    : %d\n", scrn.SrcFrame);
	p += sprintf(p, "s%dSrcMod    : 0x%08x\n", si, qcc_read_long(QCC_BID_VPU, siSrcMod(si)));
	p += sprintf(p, "s%dColorKey  : 0x%08x\n", si, qcc_read_long(QCC_BID_VPU, siColorKey(si)));
	p += sprintf(p, "s%dSize      : 0x%08x\n", si, qcc_read_long(QCC_BID_VPU, siSize(si)));
	p += sprintf(p, "s%dPosn      : 0x%08x\n", si, qcc_read_long(QCC_BID_VPU, siPosn(si)));
	p += sprintf(p, "s%dSrcPosn   : 0x%08x\n", si, qcc_read_long(QCC_BID_VPU, siSrcPosn(si)));
	p += sprintf(p, "s%dLastPixel : 0x%08x\n", si, qcc_read_long(QCC_BID_VPU, siLastPixel(si)));
	p += sprintf(p, "s%dZoom      : 0x%08x\n", si, qcc_read_long(QCC_BID_VPU, siZoom(si)));
	p += sprintf(p, "lanechoice : %d\n", mg->lane_choice);
	p += sprintf(p, "part_index : %d\n", mg->part_index);
	p += sprintf(p, "phys       : %lx\n", mg->phy);
	p += sprintf(p, "screen_base: %p\n", info->screen_base);

	if (0 == v->bits_per_pixel)
		return p - buffer;

	p += sprintf(p, "smem_start             : %p\n", (void *)info->fix.smem_start);
	p += sprintf(p, "smem_len               : %p\n", (void *)info->fix.smem_len);
	p += sprintf(p, "mmio_start             : %p\n", (void *)info->fix.mmio_start);
	p += sprintf(p, "mmio_len               : %p\n", (void *)info->fix.mmio_len);
	p += sprintf(p, "list_start             : %p\n", (void *)mg->list_start);
	p += sprintf(p, "list_len               : %p\n", (void *)mg->list_len);
	p += sprintf(p, "part_x/part_y          : %d/%d\n", mg->part_x, mg->part_y);
	p += sprintf(p, "line_length            : %d\n", info->fix.line_length);
	p += sprintf(p, "xres/yres              : %d/%d\n", v->xres, v->yres);
	p += sprintf(p, "xres/yres virtual      : %d/%d\n", v->xres_virtual, v->yres_virtual);
	p += sprintf(p, "xoffset/yoffset        : %d/%d\n", v->xoffset, v->yoffset);
	p += sprintf(p, "HxW mode               : %dx%d %d\n", v->height, v->width, v->vmode);
	p += sprintf(p, "hsync/vsync len        : %d/%d\n", v->hsync_len, v->vsync_len);
	p += sprintf(p, "bpp/activate/pixclock  : %d/%d/%d\n",
			v->bits_per_pixel, v->activate, v->pixclock);
	p += sprintf(p, "margin lf/rt/up/lo     : %d/%d/%d/%d\n",
			v->left_margin,
			v->right_margin,
			v->upper_margin,
			v->lower_margin);
	p += sprintf(p, "color red              : %2d-%d-%d\n",
			v->red.offset,
			v->red.length,
			v->red.msb_right);
	p += sprintf(p, "color green            : %2d-%d-%d\n",
			v->green.offset,
			v->green.length,
			v->green.msb_right);
	p += sprintf(p, "color blue             : %2d-%d-%d\n",
			v->blue.offset,
			v->blue.length,
			v->blue.msb_right);
	p += sprintf(p, "transp                 : %2d-%d-%d\n",
			v->transp.offset,
			v->transp.length,
			v->transp.msb_right);

	return p - buffer ;
}
static int osd_get_partition(char *buffer, char **start, off_t offset,
			     int count, int *eof, void *prvdata)
{
	struct fb_info *info = (struct fb_info *)prvdata;
	char *p = buffer ;
	p += sprintf(p, "osd_partition=%d\n", osd_partition);
	p += sprintf(p, "width=%d\n", info->var.xres);
	p += sprintf(p, "height=%d\n", info->var.yres);
	p += sprintf(p, "videomemory=%lu\n", videomemory);
	p += sprintf(p, "osd_numpartition=%d\n", osd_numpartition);
	p += sprintf(p, "enable=%u\n", (unsigned)PPAR->mode.enable);
	p += sprintf(p, "ftu=%u\n", (unsigned)PPAR->mode.ftu);
	p += sprintf(p, "alpha_select=%u\n", (unsigned)PPAR->mode.alpha_select);
	p += sprintf(p, "premultiply=%u\n", (unsigned)PPAR->mode.premultiply);
	p += sprintf(p, "fs=%u\n", (unsigned)PPAR->mode.fs);
	p += sprintf(p, "fd=%u\n", (unsigned)PPAR->mode.fd);
	p += sprintf(p, "z=%u\n", (unsigned)PPAR->mode.z);
	p += sprintf(p, "alpha=%u\n", (unsigned)PPAR->mode.alpha);
	p += sprintf(p, "color_modifier=%u\n", (unsigned)PPAR->mode.color_modifier);
	p += sprintf(p, "alpha_modifier=%u\n", (unsigned)PPAR->mode.alpha_modifier);
	p += sprintf(p, "hzoom=%u\n", (unsigned)PPAR->mode.hzoom);
	p += sprintf(p, "vzoom=%u\n", (unsigned)PPAR->mode.vzoom);
	p += sprintf(p, "hoffset=%u\n", (unsigned)PPAR->mode.hoffset);
	p += sprintf(p, "voffset=%u\n", (unsigned)PPAR->mode.voffset);
	return p - buffer ;
}
static int osd_set_partition(struct file *f, const char *buffer, unsigned long count, void *prvdata)
{
	int x;
	struct fb_info *info = (struct fb_info *)prvdata;

	const char *p = buffer;

	if (1 == sscanf(p, "osd_partition=%d", &osd_partition))
		osd_format = get_format_id(osd_partition);
	if (1 == sscanf(p, "width=%d", &x)) {
		info->var.xres = x;
		info->var.xres_virtual = x;
	}
	if (1 == sscanf(p, "height=%d", &x)) {
		info->var.yres = x;
		info->var.yres_virtual = x;
	}
	if (1 == sscanf(p, "videomemory=%d", &x)) {
		videomemory = x;
		if (videomemory > MAX_VIDEOMEMSIZE)
			videomemory = MAX_VIDEOMEMSIZE;
	}
	sscanf(p, "osd_numpartition=%d", &osd_numpartition);
	sscanf(p, "enable=%hhu", &(PPAR->mode.enable));
	sscanf(p, "ftu=%hhu", &(PPAR->mode.ftu));
	sscanf(p, "alpha_select=%hhu", &(PPAR->mode.alpha_select));
	sscanf(p, "premultiply=%hhu", &(PPAR->mode.premultiply));
	sscanf(p, "fs=%hhu", &(PPAR->mode.fs));
	sscanf(p, "fd=%hhu", &(PPAR->mode.fd));
	sscanf(p, "z=%hhu", &(PPAR->mode.z));
	sscanf(p, "alpha=%hhu", &(PPAR->mode.alpha));
	sscanf(p, "color_modifier=%hhu", &(PPAR->mode.color_modifier));
	sscanf(p, "alpha_modifier=%hhu", &(PPAR->mode.alpha_modifier));
	sscanf(p, "hzoom=%hhu", &(PPAR->mode.hzoom));
	sscanf(p, "vzoom=%hhu", &(PPAR->mode.vzoom));
	sscanf(p, "hoffset=%hu", &(PPAR->mode.hoffset));
	sscanf(p, "voffset=%hu", &(PPAR->mode.voffset));

	return count ;
}

static int __init mg3500fb_probe(struct platform_device *dev)
{
	int ret;
	struct fb_info *info;
	struct mg3500fb_par *mg;
	struct proc_dir_entry *res;

	info = framebuffer_alloc(sizeof(struct mg3500fb_par), &dev->dev);
	if (!info)
		return -ENOMEM;

	ret = fb_alloc_cmap(&info->cmap, 256, 0);
	if (ret < 0) {
		framebuffer_release(info);
		return -ENOMEM;
	}

	mg = info->par;

	info->fbops = &mg3500fb_ops;
	info->var = mg3500fb_default;
	info->fix = mg3500fb_fix;
	info->pseudo_palette = mg->pseudo_palette;
	info->flags = FBINFO_FLAG_DEFAULT;

	mg->count = 0;
	mg->si = 2;
	mg->mode = mg3500fb_screen_mode;
	mg->ck = mg3500fb_color_key;

	ret = register_framebuffer(info);
	if (ret < 0) {
		fb_dealloc_cmap(&info->cmap);
		framebuffer_release(info);
		return -ENOMEM;
	}

	if (use_dma) {
		mg->dmah = mobi_dma_request(MODNAME, MOBI_DMA_O_NONE);
		if (mg->dmah < 0) {
			fb_dealloc_cmap(&info->cmap);
			framebuffer_release(info);
			return -EAGAIN;
		}
		init_waitqueue_head(&mg->dma_wq);
		if (0 > mobi_dma_setup_handler(mg->dmah, dma_callback, mg)) {
			mobi_dma_free(mg->dmah);
			fb_dealloc_cmap(&info->cmap);
			framebuffer_release(info);
			return -EAGAIN;
		}
		info->fix.accel = FB_ACCEL_MG3500_BLITTER;
		info->fix.mmio_start = DMAC_BASE;
		info->fix.mmio_len = DMAC_SIZE;
		mg->list_start = DTCM_BASE+(4096*mg->dmah);
		mg->list_len = 4096;
	}

	platform_set_drvdata(dev, info);

	proc_mkdir("driver/osd", NULL);
	res = create_proc_read_entry("driver/osd/state", 0, NULL, osd_get_state, info);
	res = create_proc_read_entry("driver/osd/partition", 0, NULL, osd_get_partition, info);
	res->write_proc = osd_set_partition;

	return 0;
}
static int mg3500fb_remove(struct platform_device *dev)
{
	struct fb_info *info = platform_get_drvdata(dev);

	if (!info)
		return 0;

	unregister_framebuffer(info);

	fb_dealloc_cmap(&info->cmap);

	iounmap_area_tiny(info->screen_base);

	framebuffer_release(info);

	if (use_dma)
		mobi_dma_free(PPAR->dmah);

	remove_proc_entry("driver/osd/state", NULL);
	remove_proc_entry("driver/osd/partition", NULL);
	remove_proc_entry("driver/osd", NULL);

	return 0;
}

static struct platform_driver mg3500fb_driver = {
    .probe  = mg3500fb_probe,
    .remove = mg3500fb_remove,
    .driver = { .name    = "mg3500fb", },
};

static struct platform_device *mg3500fb_device;

static int __init mg3500fb_init(void)
{
	int ret = platform_driver_register(&mg3500fb_driver);

	if (ret) {
		printk(KERN_WARNING "[" MODNAME "]" "platform_driver_register() fails ..\n");
		return ret;
	}

	mg3500fb_device = platform_device_alloc("mg3500fb", 0);

	ret = mg3500fb_device ? platform_device_add(mg3500fb_device) : -ENOMEM;

	if (ret) {
		printk(KERN_WARNING "[" MODNAME "]" "platform_device_add() fails ..\n");
		platform_device_put(mg3500fb_device);
		platform_driver_unregister(&mg3500fb_driver);
	}

	return ret;
}

static void __exit mg3500fb_exit(void)
{
	platform_device_unregister(mg3500fb_device);
	platform_driver_unregister(&mg3500fb_driver);
}

module_init(mg3500fb_init);
module_exit(mg3500fb_exit);
#ifdef MODULE
MODULE_AUTHOR("Stephan Lachowsky");
MODULE_DESCRIPTION("Maxim MG3500 Frame Buffer");
MODULE_LICENSE("GPL");
#endif
