/*
 * usb_cap_interface.c USB Video Class interface implementation
 *
 * (C) Copyright 2009 MCN Technologies Inc.
 *
 *
 * The file implements handling of class specific requests to uvc interface 
 *
 */
#include <linux/version.h>
#include <linux/types.h>
#if LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 20)
#include <linux/usb/gadgetfs.h>
#include <linux/usb/ch9.h>
#else
#include <linux/usb_gadgetfs.h>
#include <linux/usb_ch9.h>
#endif

#include "uvc_vs_ctrl.h"

#include "uvcdescript.h"
#include "uvc_codec_if.h"
#include "uvc_cap_interface.h"
#include "vid_input_ctrl.h"
#include "uvc_callback.h"
#include "dbgout.h"

//TODO: Move the following to a header file uvc_codec_if.h
extern struct stream_format_if_t vid_cap_strm_format_if;
extern struct stream_format_if_t vid_trans_strm_format_if_enc;
extern struct stream_format_if_t vid_trans_strm_format_if_dec;
extern void DumpStreamControlRequest(struct uvc_streaming_control_request *pReq);

extern int handle_vidcap_control_video_extension(int intf_num, char *req_buf, const struct usb_ctrlrequest *ctrl);
extern int init_vidcap_control_video_extension(vid_codec_config_t *codec_cfg);
extern int init_enc_control_video_extension(vid_codec_config_t *codec_cfg);

extern int handle_dec_control_video_extension(int intf_num, char *req_buf, const struct usb_ctrlrequest *ctrl);
extern int init_dec_control_video_extension(vid_codec_config_t *codec_cfg);


int handle_pu_control(void *pCtx, char *req_buf, const struct usb_ctrlrequest *ctrl);
void *get_pu_control(void);

__u8 bCrntVcErrCode = ERR_NONE;
__u8 bCrntPowerMode = 0x9B;	// 0x80(AC Powered) | 1B(Device Dependent)

typedef struct tag_interface_ctx_t
{
	__u8 bCrntFormatIndex;
	__u8 bCrntFrameIndex;
	__u8 bCrntAltSetting;
	__u8 bCrntVidInput;
	struct stream_format_if_t *stream_format_if;
} interface_ctx_t;


struct vid_cap_still_param
{
	// Still
	__u8 bFormatIndex;
	__u8  bFrameIndex;
	__u8  bCompressionIndex;
	__u32 dwMaxVideoFrameSize;
	__u32 dwMaxPayloadTransferSize;
};


struct vid_cap_pu_param
{
	//Processor
	__u16 wBacklightCompensation;
	__u16 wBrightness;
	__u16 wContrast;
	__u8  bPowerLineFrequency;
	__u16 wHue;
	__u8  bHueAuto;
	__u16 wSaturation;
	__u16 wSharpness;
	__u16 wGamma;
	__u16 wWhiteBalanceTemperature;
	__u8  bWhiteBalanceTemperatureAuto;
	__u16 wWhiteBalanceBlue;
	__u16 wWhiteBalanceRed;
	__u8  bWhiteBalanceComponentAuto;
	__u16 wMultiplierLimit;
};
struct vid_cap_su_param
{
	//Selector
	__u8  bSelector;
};

interface_ctx_t uvc_cap_strm_in = 
{
	.bCrntFormatIndex = 1,
	.bCrntFrameIndex = 1,
	.bCrntAltSetting = 1,
	.bCrntVidInput = 1,
	.stream_format_if = &vid_cap_strm_format_if
};

interface_ctx_t uvc_trans_strm_in = 
{
	.bCrntFormatIndex = 1,
	.bCrntFrameIndex = 1,
	.bCrntAltSetting = 1,
	.bCrntVidInput = 1,
	.stream_format_if = &vid_trans_strm_format_if_enc
};
interface_ctx_t uvc_trans_strm_out = 
{
	.bCrntFormatIndex = 1,
	.bCrntFrameIndex = 1,
	.bCrntAltSetting = 1,
	.bCrntVidInput = 1,
	.stream_format_if = &vid_trans_strm_format_if_dec
};

interface_ctx_t *get_interface_ctx(int intf_num)
{
	switch(intf_num) {

#if HAS_VID_CAP == 1
		case UVC_CTRL_INTERFACE_INDEX:
		case UVC_CAP_STRM_IN_INDEX:
			return &uvc_cap_strm_in;
			break;
#endif
#if HAS_VID_TRANS == 1
		case UVC_ENC_CTRL_IN_INDEX:
		case UVC_ENC_STRM_IN_INDEX:
			return &uvc_trans_strm_in;
			break;
		case UVC_TRANS_CTRL_OUT_INDEX:
		case UVC_TRANS_STRM_OUT_INDEX:
			return &uvc_trans_strm_out;
			break;
#endif
	default:
		return 0;

	}
}

/**
 * Handling of uvc probe control. 
 * Refer to Section 4.3.1.1 of USB Device Class definition for Video Devices Revision 1.1
 */

int uvcProbeVideoCtrl(int intf_num, struct uvc_streaming_control_request *ctrl, __u8 bRequest)
{
	int value = -1;
	__u8 bFormatIndex = 1;
	__u8 bFrameIndex = 1;
	struct stream_format_if_t *pStrmIf;
	interface_ctx_t *pIntfCtx = get_interface_ctx(intf_num);

	if(pIntfCtx == 0) {
		UAV_DBG_ERR("No context for the interface %d\n", intf_num);
		return value;
	}

	bFormatIndex = pIntfCtx->bCrntFormatIndex;
	bFrameIndex = pIntfCtx->bCrntFrameIndex;

	if(bFrameIndex <= 0) {
		UAV_DBG_ERR("Forcing frame index to 1\n");
		bFrameIndex = 1;
	}

	pStrmIf = pIntfCtx->stream_format_if;
	UAV_DBG_MSG("bRequest=0x%0x bFormatIndex=%d, bCrntFrameIndex=%d \n", bRequest, bFormatIndex, pIntfCtx->bCrntFrameIndex);
	// Validate bFormatIndex and bFrameIndex
	switch(bRequest)
	{
		case SET_CUR:
			/*Sets the streaming interface Probe state. This is the attribute used for stream parameter negotiation.*/
			UAV_DBG_MSG("Setting Probe State\n");
			bFormatIndex = pIntfCtx->bCrntFormatIndex = ctrl->bFormatIndex;
			bFrameIndex = pIntfCtx->bCrntFrameIndex = ctrl->bFrameIndex;
			//DumpStreamControlRequest(ctrl);
			value = pStrmIf->SetUvcStreamingControl(pStrmIf, ctrl, 1, bFormatIndex, bFrameIndex);
			break;

		case GET_CUR:
			/* Returns the current state of the streaming interface. All supported fields set to zero will be returned with an acceptable negotiated value.
			   Prior to the initial SET_CUR operation, the GET_CUR state is undefined. This request shall stall in case of negotiation failure. 
			*/
			value =  pStrmIf->GetUvcStreamingControl(pStrmIf, ctrl, 1, bFormatIndex, bFrameIndex, PROBE_INDEX_CUR);
			break;
		case GET_MIN:
			/* Returns the minimum value for negotiated fields. */
			value =  pStrmIf->GetUvcStreamingControl(pStrmIf, ctrl, 1, bFormatIndex, bFrameIndex, PROBE_INDEX_MIN);
			break;
		case GET_MAX:
			/* Returns the maximum value for negotiated fields. */
			value =  pStrmIf->GetUvcStreamingControl(pStrmIf, ctrl, 1, bFormatIndex, bFrameIndex, PROBE_INDEX_MAX);
			break;
		case GET_RES:
			/*Return the resolution of each supported field in the Probe/Commit data structure. */
			value = pStrmIf->GetUvcStreamingControl(pStrmIf, ctrl, 1, bFormatIndex, bFrameIndex, PROBE_INDEX_RES);
			break;
		case GET_LEN:
			/* Returns the length of the Probe data structure */
			*(__u8*)ctrl = 34;
			value = 1;
			break;
		case GET_INFO:
			/* Queries the capabilities and status of the Control. The value returned for this request shall have bits D0 and D1 each set to one (1), 
			   and the remaining bits set to zero (0) (see section 4.1.2, Get Request).
	*/
			*(__u8*)ctrl = 0x03;
			value = 1;
			break;
		case GET_DEF:
			/* Returns the default value for the negotiated fields. */
			value = pStrmIf->GetUvcStreamingControl(pStrmIf, ctrl, 1, bFormatIndex, bFrameIndex, PROBE_INDEX_DEF);
			/* Replace frame idex and format index. These are not filled by GetUvcStreamingControl */
			ctrl->bFormatIndex = bFormatIndex;
			ctrl->bFrameIndex = bFrameIndex;
			break;
	}
	return value;
}

/*******************************************************************************
 * Function   : uvcCommitVideoCtrl
 *
 * Description: Handling of uvc commit control. 
 *				Refer to Section 4.3.1.1 of USB Device Class definition for Video 
 *				Devices Revision 1.1
 *
 * Parameters :
 *		[IN/OUT]uvc_streaming_control_request *ctrl - setup data for the request
 *      [IN]	__u8 bRequest							- Request type e.g. SET_CUR
 *
 * Return Value:
 *      int     - return 0 on success
 *              - return -1 on error
 ******************************************************************************/

int uvcCommitVideoCtrl(int intf_num, struct uvc_streaming_control_request *ctrl, __u8 bRequest)
{
	__u8 bFormatIndex = ctrl->bFormatIndex;
	__u8 bFrameIndex = ctrl->bFrameIndex;
	struct stream_format_if_t *pStrmIf;
	int res = 0;
	interface_ctx_t *pIntfCtx = get_interface_ctx(intf_num);
	if(pIntfCtx == 0) {
		UAV_DBG_ERR("No context for the interface %d\n", intf_num);
		return -1;
	}

	if(bFrameIndex <= 0) {
		UAV_DBG_ERR("Forcing frame index to 1\n");
		bFrameIndex = 1;
	}
	pStrmIf = pIntfCtx->stream_format_if;
	// Validate bFormatIndex and bFrameIndex
	UAV_DBG_MSG("bRequest=0x%0x bFormatIndex=%d, bCrntFrameIndex=%d \n", bRequest, bFormatIndex, pIntfCtx->bCrntFrameIndex);
	switch(bRequest)
	{
		case SET_CUR:
			/* Sets the device state. This sets the active device state; the field values must be the result of a successful VS_PROBE_CONTROL(GET_CUR) request. 
			   This request shall stall in case an unsupported state is specified. 
			*/
			pIntfCtx->bCrntFormatIndex = bFormatIndex;
			pIntfCtx->bCrntFrameIndex = bFrameIndex;
			res = pStrmIf->SetUvcStreamingControl(pStrmIf, ctrl, 0, bFormatIndex, bFrameIndex);
			if (res >= 0 && !pStrmIf->IsAltSettingVaiable(pStrmIf)){
				UpdateStreamStateForBulkEps(intf_num);
			} else {
				// Do nothing. The state change will be handled during cap_set_interface
				UAV_DBG_ERR("ISOC interface\n");
			}
			return res;
			break;

		case GET_CUR:
			return pStrmIf->GetUvcStreamingControl(pStrmIf, ctrl, 0, bFormatIndex, bFrameIndex, PROBE_INDEX_CUR);
			break;
		case GET_MIN:
		case GET_MAX:
		case GET_RES:
		case GET_DEF:
			return -1;
			break;
		case GET_LEN:
			/* Returns the length of the Probe data structure */
			*(__u8*)ctrl = sizeof(struct uvc_streaming_control_request);
			break;
		case GET_INFO:
			/* Queries the capabilities and status of the Control. The value returned for this request shall have bits D0 and D1 each set to one (1), 
			   and the remaining bits set to zero (0) (see section 4.1.2, Get Request).
			*/
			*(__u8*)ctrl = 0x03;
			return 0;
			break;
	}
	return 0;
}


static int handle_vidcap_power_requests(char *req_buf, const struct usb_ctrlrequest *ctrl)
{
	int	value = -1;
	__u16	w_value = __le16_to_cpu(ctrl->wValue);
	//__u16	w_length = __le16_to_cpu(ctrl->wLength);
	int control_selector = w_value  >> 8;
	//__u16	w_index = __le16_to_cpu(ctrl->wIndex);
	//int intf_num = (w_index & 0x00FF);
	//interface_ctx_t *pIntfCtx = get_interface_ctx(intf_num);

	UAV_DBG_MSG("control_selector=%d\n",control_selector);


	switch(ctrl->bRequest){
		case SET_CUR:
		{
			value = 0;
			bCrntPowerMode = (bCrntPowerMode &  ~0x0F)| (req_buf[0] & 0x0F);
		}
		break;
		case GET_CUR:
			req_buf[0] = bCrntPowerMode;
			value = 1;
			break;
		case GET_MIN:
		case GET_MAX:
		case GET_RES:
		case GET_DEF:
			value = -1;
			break;
		case GET_INFO:
			*(__u8*)req_buf = 0x03;
			value = 1;
			break;
		}
	return value;
}
static int handle_vidcap_error_code(char *req_buf, const struct usb_ctrlrequest *ctrl)
{
	int	value = -1;
	__u16	w_value = __le16_to_cpu(ctrl->wValue);
	//__u16	w_length = __le16_to_cpu(ctrl->wLength);
	int control_selector = w_value  >> 8;
	//__u16	w_index = __le16_to_cpu(ctrl->wIndex);
	//int intf_num = (w_index & 0x00FF);
	//interface_ctx_t *pIntfCtx = get_interface_ctx(intf_num);

	UAV_DBG_MSG("control_selector=%d\n",control_selector);

	switch(ctrl->bRequest){
		case GET_CUR:
			req_buf[0] = bCrntVcErrCode;
			value = 1;
			break;
		case GET_MIN:
		case GET_MAX:
		case GET_RES:
		case GET_DEF:
		case SET_CUR:
			value = -1;
			break;
		case GET_INFO:
			*(__u8*)req_buf = 0x01;	// Supports Get
			value = 1;
			break;
		}
	return value;
}

static int handle_vidcap_control_interface_requests(char *req_buf, const struct usb_ctrlrequest *ctrl)
{
	int	value = -1;
	__u16	w_value = __le16_to_cpu(ctrl->wValue);
	//__u16	w_length = __le16_to_cpu(ctrl->wLength);
	int control_selector = w_value  >> 8;

	UAV_DBG_MSG("control_selector=%d\n",control_selector);
	switch(control_selector){
		case VC_VIDEO_POWER_MODE_CONTROL:
			return handle_vidcap_power_requests(req_buf, ctrl);
			break;
		case VC_REQUEST_ERROR_CODE_CONTROL:
			return handle_vidcap_error_code(req_buf, ctrl);
			break;
	}

	return value;
}


static int handle_vidcap_control_selector(int intf_num, char *req_buf, const struct usb_ctrlrequest *ctrl)
{
	int	value = -1;
	__u16	w_value = __le16_to_cpu(ctrl->wValue);
	//__u16	w_length = __le16_to_cpu(ctrl->wLength);
	int control_selector = w_value  >> 8;
	interface_ctx_t *pIntfCtx = get_interface_ctx(intf_num);
	vid_codec_config_t Param = {0};
	vid_cap_get_state(&Param);

	UAV_DBG_MSG("control_selector=%d\n",control_selector);
	switch(control_selector){
		case SU_INPUT_SELECT_CONTROL:
		{
			switch(ctrl->bRequest){
				case SET_CUR:
				{
					value = 0;
					pIntfCtx->bCrntVidInput = req_buf[0];
					switch(pIntfCtx->bCrntVidInput)
					{
#if HAS_CAMERA_INPUT	== 1
						case CAMERA_CONNECTOR_INPUT_ID:
							Param.dwCapSource = VID_SRC_CAMERA;
							break;
#endif
#if HAS_COMPOSITE_CONNECTOR_INPUT == 1
						case COMPOSITE_VID_CONNECTOR_INPUT_ID:
							Param.dwCapSource = VID_SRC_COMPOSITE;
							break;
#endif
#if HAS_SVIDEO_CONNECTOR_INPUT	== 1
						case SVIDEO_VID_CONNECTOR_INPUT_ID:
							Param.dwCapSource = VID_SRC_SVIDEO;
							break;
#endif
#if HAS_COMPONENT_CONNECTOR_INPUT == 1
						case COMPONENT_VID_CONNECTOR_INPUT_ID:
							Param.dwCapSource = VID_SRC_COMPONENT;
							break;
#endif
#if HAS_HDMI_CONNECTOR_INPUT == 1
						case HDMI_VID_CONNECTOR_INPUT_ID:
							Param.dwCapSource = VID_SRC_HDMI;
							break;
#endif
					}
				}
				break;
			case GET_CUR:
				req_buf[0] = pIntfCtx->bCrntVidInput;
				value = 1;
				break;
			case GET_MIN:
				req_buf[0] = 1;
				value = 1;
				break;
			case GET_MAX:
				req_buf[0] = VID_CONNECTOR_INPUT_ID_LAST - 1;
				value = 1;
				break;
			case GET_RES:
				req_buf[0] =1;
				value = 1;
				break;
			case GET_INFO:
				*(__u8*)req_buf = 0x03;
				value = 1;
				break;
			case GET_DEF:
				req_buf[0] =1;
				value = 1;
				break;
			}
		}
			break;
		default:
			break;
	}
	UAV_DBG_MSG("control_selector=%d return value=%d\n",control_selector,value);
	return value;
}

static int handle_enc_control_selector(int intf_num, char *req_buf, const struct usb_ctrlrequest *ctrl)
{
	int	value = -1;
	__u16	w_value = __le16_to_cpu(ctrl->wValue);
	int control_selector = w_value  >> 8;
	interface_ctx_t *pIntfCtx = get_interface_ctx(intf_num);
	vid_codec_config_t Param = {0};
	vid_enc_get_state(&Param);
	
if(!Param.fExtActive) {
		/* First time initialization */
		init_enc_control_video_extension(&Param);
	}

	UAV_DBG_MSG("control_selector=%d\n",control_selector);
	switch(control_selector){
		case SU_INPUT_SELECT_CONTROL:
		{
			switch(ctrl->bRequest){
				case SET_CUR:
				{
					value = 0;
					pIntfCtx->bCrntVidInput = req_buf[0];
					if(ENC_INPUT_ID_LVPP == pIntfCtx->bCrntVidInput) {
						Param.dwCapSource = VID_SRC_PREVIEW;
					} else if(ENC_INPUT_ID_AVDEC == pIntfCtx->bCrntVidInput) {
						Param.dwCapSource = VID_SRC_AVDEC;
					} else {
						UAV_DBG_MSG("Unknown input=%d\n",pIntfCtx->bCrntVidInput);
					}
					vid_enc_set_state(&Param, Param.dwStreamState == CODEC_STATE_RUN);
				}
				break;
			case GET_CUR:
				req_buf[0] = pIntfCtx->bCrntVidInput;
				value = 1;
				break;
			case GET_MIN:
				req_buf[0] = 1;
				value = 1;
				break;
			case GET_MAX:
				req_buf[0] = ENC_INPUT_ID_LAST - 1;
				value = 1;
				break;
			case GET_RES:
				req_buf[0] =1;
				value = 1;
				break;
			case GET_INFO:
				*(__u8*)req_buf = 0x03;
				value = 1;
				break;
			case GET_DEF:
				req_buf[0] =1;
				value = 1;
				break;
			}
		}
			break;
		default:
			break;
	}
	UAV_DBG_MSG("control_selector=%d return value=%d\n",control_selector,value);
	return value;
}


static int handle_vidcap_control_strm_out(int intf_num, char *req_buf, const struct usb_ctrlrequest *ctrl)
{
	int	value = 0;
	__u16	w_value = __le16_to_cpu(ctrl->wValue);
	int control_selector = w_value  >> 8;
	//interface_ctx_t *pIntfCtx = get_interface_ctx(intf_num);

	UAV_DBG_MSG("control_selector=%d\n",control_selector);
	return value;
}

extern int handle_vidcap_control_camera(void *pCtx, char *req_buf, const struct usb_ctrlrequest *ctrl);
extern void * get_vidcap_control_camera(void);
extern void *get_vidcap_control_input(void);

int handle_uvc_cap_control(char *req_buf, const struct usb_ctrlrequest *ctrl)
{
	int	value = -1;
	__u16	w_index = __le16_to_cpu(ctrl->wIndex);
	int ctrl_id = w_index >> 8;
	int intf_num = (w_index & 0x00FF);

	UAV_DBG_MSG("ctrl_id=%d\n",ctrl_id);
	switch (ctrl_id){
		case 0:	
			return handle_vidcap_control_interface_requests(req_buf, ctrl);
			break;
#if HAS_CAMERA_INPUT	== 1
		case CAMERA_CONNECTOR_INPUT_ID:
			return handle_vidcap_control_camera(get_vidcap_control_camera(), req_buf, ctrl);
			break;
#endif
#if HAS_COMPOSITE_CONNECTOR_INPUT == 1
			return handle_input_control_composite(get_vidcap_control_input(), req_buf, ctrl);
			break;
#endif
#if HAS_SVIDEO_CONNECTOR_INPUT == 1
			return handle_input_control_svideo(get_vidcap_control_input(), req_buf, ctrl);
			break;
#endif
#if HAS_COMPONENT_CONNECTOR_INPUT == 1
			return handle_input_control_component(get_vidcap_control_input(), req_buf, ctrl);
			break;
#endif
#if HAS_HDMI_CONNECTOR_INPUT == 1
			return handle_input_control_hdmi(get_vidcap_control_input(), req_buf, ctrl);
			break;
#endif
		case VIDCAP_UNIT_SELECTOR_ID:
			return handle_vidcap_control_selector(intf_num, req_buf, ctrl);
			break;
#if HAS_VIDCAP_UNIT_PROCESSOR == 1
		case VIDCAP_UNIT_PROCESSOR_ID:
			return handle_pu_control(get_pu_control(),req_buf, ctrl);
			break;
#endif
#if HAS_VIDCAP_UNIT_EXT == 1
		case VIDCAP_UNIT_EXT_ID:
			return handle_vidcap_control_video_extension(intf_num, req_buf, ctrl);
			break;
#endif
		case VIDCAP_OUT_TERMINAL_ID:
			return handle_vidcap_control_strm_out(intf_num, req_buf, ctrl);
			break;
		default:
			UAV_DBG_MSG("Unknown request\n");
			break;
	}
	return value;
}

int handle_uvc_enc_control(char *req_buf, const struct usb_ctrlrequest *ctrl)
{
	int	value = -1;
	__u16	w_index = __le16_to_cpu(ctrl->wIndex);
	int ctrl_id = w_index >> 8;
	int intf_num = (w_index & 0x00FF);
	UAV_DBG_MSG("ctrl_id=%d\n",ctrl_id);
#if HAS_VID_TRANS == 1
	switch (ctrl_id){
		case 0:	
			//return handle_vidcap_control_interface_requests(req_buf, ctrl);
			UAV_DBG_ERR("Unsupported request\n");
			break;
		case ENC_INPUT_ID_AVDEC:
			UAV_DBG_ERR("Unsupported request\n");
			break;
		case ENC_INPUT_ID_LVPP:
			UAV_DBG_ERR("Unsupported request\n");
			break;

		case ENC_UNIT_SELECTOR_ID:
			return handle_enc_control_selector(intf_num, req_buf, ctrl);
			break;

		case ENC_UNIT_ID_PROCESSOR:
			//return handle_pu_control(get_pu_control(),req_buf, ctrl);
			UAV_DBG_ERR("Unsupported request\n");
			break;
		case ENC_UNIT_ID_EXT:
			return handle_vidcap_control_video_extension(intf_num, req_buf, ctrl);
			break;
		case ENC_UNIT_ID_OUT_TERMINAL:
			//return handle_vidcap_control_strm_out(intf_num, req_buf, ctrl);
			UAV_DBG_ERR("Unsupported request\n");
			break;

		default:
			UAV_DBG_MSG("Unknown request\n");
			break;
	}
#endif
	return value;
}

int handle_uvc_dec_control(char *req_buf, const struct usb_ctrlrequest *ctrl)
{
	int	value = -1;
	__u16	w_index = __le16_to_cpu(ctrl->wIndex);
	int ctrl_id = w_index >> 8;
	int intf_num = (w_index & 0x00FF);
	UAV_DBG_MSG("ctrl_id=%d\n",ctrl_id);
#if HAS_VID_TRANS == 1
	switch (ctrl_id){
		case DEC_UNIT_ID_OUT_TERMINAL:
		case DEC_UNIT_ID_PROCESSOR:
		case DEC_UNIT_ID_INPUT:
			UAV_DBG_ERR("Unsupported request\n");
			break;
		case DEC_UNIT_ID_EXT:
			return handle_dec_control_video_extension(intf_num, req_buf, ctrl);
			break;

		default:
			UAV_DBG_MSG("Unknown request\n");
			break;
	}
#endif
	return value;
}

/**
 * Handles stream control requests for encoder, decoder and capture streams.
 */
int handle_uvc_stream_if_req(char *req_buf, const struct usb_ctrlrequest *ctrl)
{
	int	value = -1;
	__u16	w_selector;
	__u16	w_index;
	int intf_num;
	//__u16	w_index = __le16_to_cpu(ctrl->wIndex);
	w_selector = __le16_to_cpu(ctrl->wValue);
	w_selector = w_selector >> 8;
	w_index = __le16_to_cpu(ctrl->wIndex);
	intf_num = (w_index & 0x00FF);

	UAV_DBG_MSG("w_selector=%d\n",w_selector);
	switch(w_selector){
		case VS_PROBE_CONTROL:
			value = uvcProbeVideoCtrl(intf_num, (struct uvc_streaming_control_request *)req_buf, ctrl->bRequest);
			break;
		case VS_COMMIT_CONTROL:
			value = uvcCommitVideoCtrl(intf_num, (struct uvc_streaming_control_request *)req_buf, ctrl->bRequest);
			break;
		case VS_STILL_PROBE_CONTROL:
		case VS_STILL_COMMIT_CONTROL:
		case VS_STILL_IMAGE_TRIGGER_CONTROL:
		case VS_STREAM_ERROR_CODE_CONTROL:
		case VS_GENERATE_KEY_FRAME_CONTROL:
		case VS_UPDATE_FRAME_SEGMENT_CONTROL:
		case VS_SYNCH_DELAY_CONTROL:
			UAV_DBG_ERR("Unimplemented w_selector=%d\n",w_selector);
			break;
		default:
			UAV_DBG_ERR("Unknown w_selector=%d\n",w_selector);
			break;
	}
	return value;
}

int	UpdateStreamStateForBulkEps(int intf_num)
{
	UAV_DBG_ERR("intf_num=%d\n", intf_num);
	switch(intf_num)
	{

#if HAS_VID_CAP == 1
	case UVC_CAP_STRM_IN_INDEX:
			uvc_cap_init_interface(intf_num, 0);
			break;
#endif
#if HAS_VID_TRANS == 1
		case UVC_ENC_STRM_IN_INDEX:
			uvc_cap_init_interface(intf_num, 0);
			break;
		case UVC_TRANS_STRM_OUT_INDEX:
			uvc_dec_init_interface(intf_num, 0);
			break;
#endif
		default:
			break;
	}
	return 0;
}

int uvc_dec_init_interface(int intf_num, int alt_setting)
{
	int status = 0;
	struct stream_format_if_t *pStrmIf;
	vid_codec_config_t Param = {0};
	interface_ctx_t *pIntfCtx = get_interface_ctx(intf_num);
	struct usb_endpoint_descriptor *pEpDescript = 0;
	UAV_DBG_MSG("intf=%d alt_setting=%d \n", intf_num, alt_setting);
	
	vid_dec_get_state(&Param);
	if(!Param.fExtActive) {
		/* First time initialization */
		init_dec_control_video_extension(&Param);
	}
	if(pIntfCtx == 0){
		UAV_DBG_ERR("Context not found for intf_num=%d",intf_num);
		return -1;
	}
	pStrmIf = pIntfCtx->stream_format_if;
	Param.dwStreamId = STREAM_ID_TRANS_DEC;
	pIntfCtx->bCrntAltSetting = alt_setting;
	pEpDescript = get_tr_endpoint_descriptor(intf_num, alt_setting);

	if((pStrmIf->IsAltSettingVaiable(pStrmIf) &&  alt_setting >= 1) || (!pStrmIf->IsAltSettingVaiable(pStrmIf))){
		struct uvc_streaming_control_request Ctrl = {0};
		struct uvc_streaming_control_request *pCtrl = &Ctrl;
		long width, height;
		long format;
		long minBitRate = 0, maxBitRate = 0;
		/* Set Encode Parameters */
		status = pStrmIf->VerifyUvcStreamingControl(pStrmIf, pIntfCtx->bCrntFormatIndex, pIntfCtx->bCrntFrameIndex);
		if(status == status_error){
			UAV_DBG_ERR("status_error\n");
			return -1;
		}

		status = pStrmIf->GetFrameParams(pStrmIf, pCtrl, pIntfCtx->bCrntFormatIndex, pIntfCtx->bCrntFrameIndex,  &format, &width, &height, &minBitRate, &maxBitRate);
		if(status == status_error){
			UAV_DBG_ERR("status_error\n");
			return -1;
		}
		if(pEpDescript){
			Param.dwEpInterval = pEpDescript->bInterval;
			Param.dwEpPacketSize = pEpDescript->wMaxPacketSize;
			Param.dwEpIsoc = pStrmIf->IsAltSettingVaiable(pStrmIf);
		}
		Param.dwStreamState = CODEC_STATE_RUN; 
		/* Use format, width and height from decoder UVC extension */
		//Param.dwCodecFormat = format; 
		//Param.dwFrameWidth = width; 
		//Param.dwFrameHeight = height; 
		Param.dwMaxVideoFrameSize = pCtrl->dwMaxVideoFrameSize;
		Param.dwMaxPayloadTransferSize = pCtrl->dwMaxPayloadTransferSize;
		vid_dec_set_state(&Param, 1);
		return 0;//success
	} else {
		Param.dwStreamState = CODEC_STATE_STOP; 
		vid_dec_set_state(&Param, 1);
		return 0;//success
	}
	return -1;
}

int uvc_dec_set_interface(const struct usb_ctrlrequest *setup)
{
	__u16	alt_setting = __le16_to_cpu(setup->wValue);
	__u16	w_index = __le16_to_cpu(setup->wIndex);
	int intf_num = (w_index & 0x00FF);
	UAV_DBG_MSG("intf=%d alt_setting=%d \n", intf_num, alt_setting);
	return uvc_dec_init_interface(intf_num, alt_setting);
}

int uvc_cap_init_interface(int intf_num, int alt_setting)
{
	int status = 0;
	struct stream_format_if_t *pStrmIf;
	vid_codec_config_t Param = {0};
	interface_ctx_t *pIntfCtx = get_interface_ctx(intf_num);
	struct usb_endpoint_descriptor *pEpDescript = 0;
	UAV_DBG_MSG("intf=%d alt_setting=%d \n", intf_num, alt_setting);
	
	switch(intf_num){
		case UVC_CAP_STRM_IN_INDEX:
			vid_cap_get_state(&Param);
			if(!Param.fExtActive) {
				/* First time initialization */
				init_vidcap_control_video_extension(&Param);
			}
			break;

		case UVC_ENC_STRM_IN_INDEX:
			vid_enc_get_state(&Param);
			if(!Param.fExtActive) {
				/* First time initialization */
				init_enc_control_video_extension(&Param);
			}

			break;
	}
	if(pIntfCtx == 0){
		UAV_DBG_ERR("Context not found for intf_num=%d",intf_num);
		return -1;
	}
	pStrmIf = pIntfCtx->stream_format_if;
	switch(intf_num) {
#if HAS_VID_CAP == 1
		case UVC_CAP_STRM_IN_INDEX:
			Param.dwStreamId = STREAM_ID_VIDCAP_0;
			pIntfCtx->bCrntAltSetting = alt_setting;
			pEpDescript = get_endpoint_descriptor(intf_num, alt_setting);
			break;
#endif
#if HAS_VID_TRANS == 1
		case UVC_ENC_STRM_IN_INDEX:
			Param.dwStreamId = STREAM_ID_TRANS_ENC;
			pIntfCtx->bCrntAltSetting = alt_setting;
			pEpDescript = get_tr_endpoint_descriptor(intf_num, alt_setting);
			break;
#endif

		default:
			goto Error;
			break;
	}

	if((pStrmIf->IsAltSettingVaiable(pStrmIf) &&  alt_setting >= 1) || (!pStrmIf->IsAltSettingVaiable(pStrmIf))){
		struct uvc_streaming_control_request Ctrl = {0};
		struct uvc_streaming_control_request *pCtrl = &Ctrl;
		long width, height;
		long format;
		long minBitRate = 0, maxBitRate = 0;
		/* Set Encode Parameters */
		status = pStrmIf->VerifyUvcStreamingControl(pStrmIf, pIntfCtx->bCrntFormatIndex, pIntfCtx->bCrntFrameIndex);
		if(status == status_error){
			UAV_DBG_ERR("status_error\n");
			return -1;
		}

		status = pStrmIf->GetFrameParams(pStrmIf, pCtrl, pIntfCtx->bCrntFormatIndex, pIntfCtx->bCrntFrameIndex,  &format, &width, &height, &minBitRate, &maxBitRate);
		if(status == status_error){
			UAV_DBG_ERR("status_error\n");
			return -1;
		}
		if(pEpDescript){
			Param.dwEpInterval = pEpDescript->bInterval;
			Param.dwEpPacketSize = pEpDescript->wMaxPacketSize;
			Param.dwEpIsoc = pStrmIf->IsAltSettingVaiable(pStrmIf);
		}
		Param.dwStreamState = CODEC_STATE_RUN; 
		Param.dwCodecFormat = format; 
		Param.dwFrameWidth = width; 
		Param.dwFrameHeight = height; 
		Param.dwFrameInterval = pCtrl->dwFrameInterval;
		Param.dwKeyFrameRate = pCtrl->wKeyFrameRate;
		Param.dwPFrameRate = pCtrl->wPFrameRate;
		Param.dwCompQuality = pCtrl->wCompQuality;
		Param.dwCompWindowSize = pCtrl->wCompWindowSize;
		Param.dwMaxVideoFrameSize = pCtrl->dwMaxVideoFrameSize;
		Param.dwMaxPayloadTransferSize = pCtrl->dwMaxPayloadTransferSize;
		Param.dwMinBitRate = minBitRate;
		Param.dwMaxBitRate = maxBitRate;
	} else {
		Param.dwStreamState = CODEC_STATE_STOP; 
	}
	switch(intf_num){
#if HAS_VID_CAP == 1
		case UVC_CAP_STRM_IN_INDEX:
		vid_cap_set_state(&Param, 1);
		return 0;
		break;
#endif
#if HAS_VID_TRANS == 1
		case UVC_ENC_STRM_IN_INDEX:
		vid_enc_set_state(&Param, 1);
		return 0;
		break;
#endif
		default:
			break;
	}

Error:
	return -1;
}
int uvc_cap_set_interface(const struct usb_ctrlrequest *setup)
{
	__u16	alt_setting = __le16_to_cpu(setup->wValue);
	__u16	w_index = __le16_to_cpu(setup->wIndex);
	int intf_num = (w_index & 0x00FF);
	UAV_DBG_MSG("intf=%d alt_setting=%d \n", intf_num, alt_setting);
	return uvc_cap_init_interface(intf_num, alt_setting);
}
