/*
 * usb_cap_interface.c USB Video Class interface implementation
 *
 * (C) Copyright 2009 MCN Technologies Inc.
 *
 *
 * The file implements handling of video stream probe/commit requests 
 *
 */

#include <linux/string.h>
#include "uvcdescript.h"
//#include "codec_defs.h"
#include "uvc_codec_if.h"
#include "platform_caps.h"
#include "uvc_vs_ctrl.h"
#include "dbgout.h"
#include "mjpg_frame_descript.h"

/*
 * Enable the following parameter to validate probe/commit negation parameters.
 */
//#define VALIDATE_PARAM

typedef struct tag_stream_format_ctx_t {
	struct uvc_streaming_control_request * (* get_stream_contrl)(int format_index, int frame_index, int probe_index);
} stream_format_ctx_t;

#if HAS_ENC_ISOCH == 1
#define MAX_ENC_PL_SIZE PKT_SIZE_MP2TS_2
#define MAX_ENC_FRM_SIZE 128000
#else
#define MAX_ENC_PL_SIZE (2800 * TS_PACKET_SIZE + 2)	// Note: 350 is arbitrary
#define MAX_ENC_FRM_SIZE MAX_ENC_PL_SIZE
#endif

#if HAS_H264_MP2TS_TRANS_SINK == 1
static struct uvc_streaming_control_request	h264_param_table[1][PROBE_INDEX_LAST] = 
{
/*
 *  bmHnt     bFormatIndex   bFrameIndex    dwFrameInterval     wKeyFrmRate    wPFrmeRate     wCompQuality   wCompWindowSize     wDelay         dwMaxVideoFrameSize      dwMaxPayloadTransferSize
 */
 {
  { 0x0000,   0,           0,               333333,             1,              0,	          1000,          0,	                 50,            MAX_ENC_FRM_SIZE,        MAX_ENC_PL_SIZE},   /* MIN */
  { 0x0000,   0,	       0,	            3333330,            1000,	        0,	          10000,	     0,	                 50,	        MAX_ENC_FRM_SIZE,        MAX_ENC_PL_SIZE},   /* Max */
  { 0x0000,   0,	       0,	            333333,	            15,	            0,	          5000,	         0,	                 50,            MAX_ENC_FRM_SIZE,        MAX_ENC_PL_SIZE},   /* Def */
  { 0x0000,   0,	       0,	            333333,	            15,	            0,	          5000,	         0,	                 50,            MAX_ENC_FRM_SIZE,        MAX_ENC_PL_SIZE},   /* Cur */
  { 0x0000,   0,	       0,	            333333,	            1,	            0,	          100,	         0,	                 0,             0,                       0},                /* Res */
 }
};
#endif

#undef MAX_ENC_PL_SIZE
#undef MAX_ENC_FRM_SIZE
#if HAS_ENC_ISOCH == 1
#define MAX_ENC_PL_SIZE PKT_SIZE_MP2TS_2
#define MAX_ENC_FRM_SIZE 128000
#else
#define MAX_ENC_PL_SIZE (2800 * TS_PACKET_SIZE + 2)
#define MAX_ENC_FRM_SIZE MAX_ENC_PL_SIZE
#endif

#if HAS_H264_VEN_STREAM_TRANS_SINK ==  1
static struct uvc_streaming_control_request	h264_ven_stream_param_table[1][PROBE_INDEX_LAST] = 
{
/*
 *  bmHnt     bFormatIndex   bFrameIndex    dwFrameInterval     wKeyFrmRate    wPFrmeRate     wCompQuality   wCompWindowSize     wDelay         dwMaxVideoFrameSize      dwMaxPayloadTransferSize
 */
 {
  { 0x0000,   0,           0,               333333,             1,              0,	          1000,          0,	                 50,            MAX_ENC_FRM_SIZE,        MAX_ENC_PL_SIZE},   /* MIN */
  { 0x0000,   0,	       0,	            3333330,            1000,	        0,	          10000,	     0,	                 50,	        MAX_ENC_FRM_SIZE,        MAX_ENC_PL_SIZE},   /* Max */
  { 0x0000,   0,	       0,	            333333,	            15,	            0,	          5000,	         0,	                 50,            MAX_ENC_FRM_SIZE,        MAX_ENC_PL_SIZE},   /* Def */
  { 0x0000,   0,	       0,	            333333,	            15,	            0,	          5000,	         0,	                 50,            MAX_ENC_FRM_SIZE,        MAX_ENC_PL_SIZE},   /* Cur */
  { 0x0000,   0,	       0,	            333333,	            1,	            0,	          100,	         0,	                 0,             0,                       0},                /* Res */
 }
};
#endif

#if HAS_H264_VEN_FRAME_TRANS_SINK ==  1
static struct uvc_streaming_control_request	h264_ven_frame_param_table[H264_TRANS_SINK_FRAME_INDEX_LAST][PROBE_INDEX_LAST] = 
{
/*
 *  bmHnt     bFormatIndex   bFrameIndex    dwFrameInterval     wKeyFrmRate    wPFrmeRate     wCompQuality   wCompWindowSize     wDelay         dwMaxVideoFrameSize      dwMaxPayloadTransferSize
 */
#if HAS_H264_TRANS_SINK_RES_120	== 1
 {
  {0x0000,    0,           0,               10000000,           1,              0,            0,             0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* MIN */
  {0x0000,    0,           0,               330000,             1000,           0,            10000,         0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Max */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Def */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Cur */
  {0x0000,    0,           0,               330000,             1,              0,            100,           0,                  0,              0,                      0},	/* Res */
 },
#endif
#if HAS_H264_TRANS_SINK_RES_144	== 1
 {
  {0x0000,    0,           0,               10000000,           1,              0,            0,             0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* MIN */
  {0x0000,    0,           0,               330000,             1000,           0,            10000,         0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Max */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Def */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Cur */
  {0x0000,    0,           0,               330000,             1,              0,            100,           0,                  0,              0,                      0},	/* Res */
 },
#endif
#if HAS_H264_TRANS_SINK_RES_240	== 1
 {
  {0x0000,    0,           0,               10000000,           1,              0,            0,             0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* MIN */
  {0x0000,    0,           0,               330000,             1000,           0,            10000,         0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Max */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Def */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Cur */
  {0x0000,    0,           0,               330000,             1,              0,            100,           0,                  0,              0,                      0},	/* Res */
 },
#endif
#if HAS_H264_TRANS_SINK_RES_288	== 1
 {
  {0x0000,    0,           0,               10000000,           1,              0,            0,             0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* MIN */
  {0x0000,    0,           0,               330000,             1000,           0,            10000,         0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Max */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Def */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Cur */
  {0x0000,    0,           0,               330000,             1,              0,            100,           0,                  0,              0,                      0},	/* Res */
 },
#endif
#if HAS_H264_TRANS_SINK_RES_480I	== 1
 {
  {0x0000,    0,           0,               10000000,           1,              0,            0,             0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* MIN */
  {0x0000,    0,           0,               330000,             1000,           0,            10000,         0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Max */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Def */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Cur */
  {0x0000,    0,           0,               330000,             1,              0,            100,           0,                  0,              0,                      0},	/* Res */
 },
#endif
#if HAS_H264_TRANS_SINK_RES_480P	== 1
 {
  {0x0000,    0,           0,               10000000,           1,              0,            0,             0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* MIN */
  {0x0000,    0,           0,               330000,             1000,           0,            10000,         0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Max */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Def */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Cur */
  {0x0000,    0,           0,               330000,             1,              0,            100,           0,                  0,              0,                      0},	/* Res */
 },
#endif
#if HAS_H264_TRANS_SINK_RES_480W	== 1
 {
  {0x0000,    0,           0,               10000000,           1,              0,            0,             0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* MIN */
  {0x0000,    0,           0,               330000,             1000,           0,            10000,         0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Max */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Def */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Cur */
  {0x0000,    0,           0,               330000,             1,              0,            100,           0,                  0,              0,                      0},	/* Res */
 },
#endif
#if HAS_H264_TRANS_SINK_RES_576I	== 1
 {
  {0x0000,    0,           0,               10000000,           1,              0,            0,             0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* MIN */
  {0x0000,    0,           0,               330000,             1000,           0,            10000,         0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Max */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Def */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Cur */
  {0x0000,    0,           0,               330000,             1,              0,            100,           0,                  0,              0,                      0},	/* Res */
 },
#endif
#if HAS_H264_TRANS_SINK_RES_576P	== 1
 {
  {0x0000,    0,           0,               10000000,           1,              0,            0,             0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* MIN */
  {0x0000,    0,           0,               330000,             1000,           0,            10000,         0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Max */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Def */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Cur */
  {0x0000,    0,           0,               330000,             1,              0,            100,           0,                  0,              0,                      0},	/* Res */
 },
#endif
#if HAS_H264_TRANS_SINK_RES_720	== 1
 {
  {0x0000,    0,           0,               10000000,           1,              0,            0,             0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* MIN */
  {0x0000,    0,           0,               330000,             1000,           0,            10000,         0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Max */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Def */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Cur */
  {0x0000,    0,           0,               330000,             1,              0,            100,           0,                  0,              0,                      0},	/* Res */
 },
#endif
#if HAS_H264_TRANS_SINK_RES_800	== 1
 {
  {0x0000,    0,           0,               10000000,           1,              0,            0,             0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* MIN */
  {0x0000,    0,           0,               330000,             1000,           0,            10000,         0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Max */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Def */
  {0x0000,    0,           0,               330000,             15,             0,            5000,          0,                  50,            128000,                  PKT_SIZE_ENC_VEN_1},	/* Cur */
  {0x0000,    0,           0,               330000,             1,              0,            100,           0,                  0,              0,                      0},	/* Res */
 },
#endif
};
#endif

#if HAS_DEC_ISOCH == 1
#define MAX_DEC_PL_SIZE PKT_SIZE_MP2TS_2
#define MAX_DEC_FRM_SIZE 128000
#else
#define MAX_DEC_PL_SIZE (350 * TS_PACKET_SIZE + 2)	// 512K
#define MAX_DEC_FRM_SIZE MAX_DEC_PL_SIZE
#endif

static struct uvc_streaming_control_request	h264_dec_param_table[PROBE_INDEX_LAST] = 

#if HAS_H264_MP2TS_TRANS_SOURCE == 1
/*
**  bmHnt     bFormatIndex   bFrameIndex    dwFrameInterval     wKeyFrmRate    wPFrmeRate     wCompQuality   wCompWindowSize     wDelay         dwMaxVideoFrameSize      dwMaxPayloadTransferSize
*/
 {
  { 0x0000,   0,           0,               333333,             1,              0,	          1000,          0,	                 50,            MAX_DEC_FRM_SIZE,        MAX_DEC_PL_SIZE},   /* MIN */
  { 0x0000,   0,	       0,	            3333330,            1000,	        0,	          10000,	     0,	                 50,	        MAX_DEC_FRM_SIZE,        MAX_DEC_PL_SIZE},   /* Max */
  { 0x0000,   0,	       0,	            333333,	            15,	            0,	          5000,	         0,	                 50,            MAX_DEC_FRM_SIZE,        MAX_DEC_PL_SIZE},   /* Def */
  { 0x0000,   0,	       0,	            333333,	            15,	            0,	          5000,	         0,	                 50,            MAX_DEC_FRM_SIZE,        MAX_DEC_PL_SIZE},   /* Cur */
  { 0x0000,   0,	       0,	            333333,	            1,	            0,	          100,	         0,	                 0,             0,                       0},                /* Res */
 };
#endif
#if HAS_H264_VEN_TRANS_SOURCE == 1
struct uvc_streaming_control_request	h264_ven_dec_param_table[PROBE_INDEX_LAST] = 
/*
**  bmHnt     bFormatIndex   bFrameIndex    dwFrameInterval     wKeyFrmRate    wPFrmeRate     wCompQuality   wCompWindowSize     wDelay         dwMaxVideoFrameSize      dwMaxPayloadTransferSize
*/
 {
  { 0x0000,   0,           0,               333333,             1,              0,	          1000,          0,	                 50,            128000,                  PKT_SIZE_MP2TS_2},   /* MIN */
  { 0x0000,   0,	       0,	            3333330,            1000,	        0,	          10000,	     0,	                 50,	        128000,                  PKT_SIZE_MP2TS_2},   /* Max */
  { 0x0000,   0,	       0,	            333333,	            15,	            0,	          5000,	         0,	                 50,            128000,                  PKT_SIZE_MP2TS_2},   /* Def */
  { 0x0000,   0,	       0,	            333333,	            15,	            0,	          5000,	         0,	                 50,            128000,                  PKT_SIZE_MP2TS_2},   /* Cur */
  { 0x0000,   0,	       0,	            333333,	            1,	            0,	          100,	         0,	                 0,             0,                       0},                /* Res */
 };
#endif

void DumpStreamControlRequest(struct uvc_streaming_control_request *pReq)
{
	UAV_DBG_MSG(" bmHint				=0x%x \n",pReq->bmHint);
	UAV_DBG_MSG(" bFormatIndex			=0x%x\n",pReq->bFormatIndex);
	UAV_DBG_MSG(" bFrameIndex			=0x%x\n",pReq->bFrameIndex);
	UAV_DBG_MSG(" dwFrameInterval		=0x%x(%d 90kHzUnits\n",pReq->dwFrameInterval, pReq->dwFrameInterval * 9/1000);
	UAV_DBG_MSG(" wKeyFrameRate			=0x%x)\n",pReq->wKeyFrameRate);
	UAV_DBG_MSG(" wPFrameRate			=0x%x\n",pReq->wPFrameRate);
	UAV_DBG_MSG(" wCompQuality			=0x%x\n",pReq->wCompQuality);
	UAV_DBG_MSG(" wCompWindowSize		=0x%x\n",pReq->wCompWindowSize);
	UAV_DBG_MSG(" wDelay				=0x%x\n",pReq->wDelay);
	UAV_DBG_MSG(" dwMaxVideoFrameSize	=0x%x\n",pReq->dwMaxVideoFrameSize);
	UAV_DBG_MSG(" dwMaxPayloadTransferSize=0x%x\n",pReq->dwMaxPayloadTransferSize);

}

#ifdef VALIDATE_PARAM
/**
 * Validates the UVC Control Params
 */
static int ValidateParams(
		struct uvc_streaming_control_request *pCrnt, 
		struct uvc_streaming_control_request *pMin, 
		struct uvc_streaming_control_request *pMax)
{
	if((pCrnt->bmHint & BMHINT_FRAME_INTERVAL_MASK) == 0) {
		if(pCrnt->dwFrameInterval <= pMax->dwFrameInterval && 
			pCrnt->dwFrameInterval >= pMin->dwFrameInterval) {
			/* Frame interval is allowed be changed and requested value is in the range*/
			// Do nothing
		} else {
			UAV_DBG_ERR("dwFrameInterval out of range min=%d max=%d req=%d", pMin->dwFrameInterval, pMax->dwFrameInterval, pCrnt->dwFrameInterval);
			return -1;
		}
	}
	if((pCrnt->bmHint & BMHINT_KEYFRAME_RATE_MASK)  == 0){
		if(pCrnt->wKeyFrameRate <= pMax->wKeyFrameRate && 
			pCrnt->wKeyFrameRate >= pMin->wKeyFrameRate) {
			/* wKeyFrameRate is allowed be changed and requested value is in the range*/
			// Do nothing
		}else  {
			UAV_DBG_ERR("wKeyFrameRate out of range min=%d max=%d req=%d", pMin->wKeyFrameRate, pMax->wKeyFrameRate, pCrnt->wKeyFrameRate);
			return -1;
		}
	}	
	if((pCrnt->bmHint & BMHINT_PFRAME_RATE_MASK) == 0) {
		if(pCrnt->wPFrameRate <= pMax->wPFrameRate && 
			pCrnt->wPFrameRate >= pMin->wPFrameRate) {
			/* Frame interval allowed be changed and requested value is in the range*/
			// Do nothing
		}else  {
			UAV_DBG_ERR("wPFrameRate out of range min=%d max=%d req=%d", pMin->wPFrameRate, pMax->wPFrameRate, pCrnt->wPFrameRate);
			return -1;
		}
	}
	if((pCrnt->bmHint & BMHINT_COMP_QUALITY_MASK) == 0) {
		if(pCrnt->wCompQuality <= pMax->wCompQuality && 
			pCrnt->wCompQuality >= pMin->wCompQuality) {
			/* Frame interval allowed be changed and requested value is in the range*/
			// Do nothing
		}else  {
			UAV_DBG_ERR("wKeyFrameRate out of range min=%d max=%d req=%d", pMin->wCompQuality, pMax->wCompQuality, pCrnt->wCompQuality);
			return -1;
		}
	}
	if((pCrnt->bmHint & BMHINT_COMP_WINDOW_SIZE) == 0) {
		if(pCrnt->wCompWindowSize <= pMax->wCompWindowSize && 
			pCrnt->wCompWindowSize >= pMin->wCompWindowSize) {
			/* Frame interval allowed be changed and requested value is in the range*/
			// Do nothing
		}else  {
			UAV_DBG_ERR("wCompWindowSize out of range min=%d max=%d req=%d", pMin->wCompWindowSize, pMax->wCompWindowSize, pCrnt->wCompWindowSize);
			return -1;
		}
	}
	return 0;
}
#endif

static int GetStreamingControl(
		struct stream_format_if_t *pCtx,
		struct uvc_streaming_control_request *pCtrl,
		__u8  bIsProbe, 
		__u8  bFormatIndex, 
		__u8  bFrameIndex, 
		__u8  bProbeIndex)
{
	struct uvc_streaming_control_request *pCur;
	stream_format_ctx_t *pStreamFormatCtx = (stream_format_ctx_t *)pCtx->pStreamFormatCtx;
	UAV_DBG_MSG("In format = %d frame = %d\n", bFormatIndex, bFrameIndex);
	DumpStreamControlRequest(pCtrl);
	pCtrl->bFormatIndex = bFormatIndex;
	pCtrl->bFrameIndex = bFrameIndex;

	bFrameIndex -= 1;	// Make it base 0
	pCur =  pStreamFormatCtx->get_stream_contrl(bFormatIndex, bFrameIndex, bProbeIndex);
	if(pCur) {
		memcpy((unsigned char *)pCtrl + 4, (unsigned char *)pCur + 4, sizeof(struct uvc_streaming_control_request)-4);
		UAV_DBG_MSG("out format = %d frame = %d\n", bFormatIndex, bFrameIndex);
		DumpStreamControlRequest(pCtrl);
		return sizeof(struct uvc_streaming_control_request);
	}
	return 0;
}

static int SetStreamingControl(
	struct stream_format_if_t *pCtx,
	struct uvc_streaming_control_request *pCtrl, 
	__u8  bIsProbe, 
	__u8  bFormatIndex, 
	__u8  bFrameIndex)
{
	struct uvc_streaming_control_request *pCur = 0;
	stream_format_ctx_t *pStreamFormatCtx = (stream_format_ctx_t *)pCtx->pStreamFormatCtx;
	UAV_DBG_MSG("format = %d frame = %d\n", bFormatIndex, bFrameIndex);
	DumpStreamControlRequest(pCtrl);

	bFrameIndex -= 1;	// Make it base 0
	if(bIsProbe) {
		/* Do nothing */
		return 0;
	} else {
		pCur =  pStreamFormatCtx->get_stream_contrl(bFormatIndex, bFrameIndex, PROBE_INDEX_CUR);
		if(pCur) {
			memcpy(pCur, pCtrl, sizeof(struct uvc_streaming_control_request));
			return 0;
		}
	}	
	return 0;
}

static int VerifyUvcStreamingControl(
		struct stream_format_if_t *pCtx,
		__u8  bFormatIndex, __u8  bFrameIndex)
{
	struct uvc_streaming_control_request *pCur = 0;
	struct uvc_streaming_control_request *pMin = 0;
	struct uvc_streaming_control_request *pMax = 0;
	stream_format_ctx_t *pStreamFormatCtx = (stream_format_ctx_t *)pCtx->pStreamFormatCtx;
	UAV_DBG_MSG("format = %d frame = %d\n", bFormatIndex, bFrameIndex);
	bFrameIndex -= 1;
	pCur = pStreamFormatCtx->get_stream_contrl(bFormatIndex, bFrameIndex, PROBE_INDEX_CUR);
	pMin = pStreamFormatCtx->get_stream_contrl(bFormatIndex, bFrameIndex, PROBE_INDEX_MIN);
	pMax = pStreamFormatCtx->get_stream_contrl(bFormatIndex, bFrameIndex, PROBE_INDEX_MAX);
	// TODO: Enable the check for frame formats

#ifdef VALIDATE_PARAM
	if(pCur && pMin && pMax) {
		return ValidateParams(pCur, pMin, pMax);
	} else {
		UAV_DBG_ERR("\n%s pCur && pMin && pMax == 0",__FILE__);
		return -1;
	}
#else
	return 0;	
#endif

}


static int GetFrameParams(
	struct stream_format_if_t *pCtx,
	struct uvc_streaming_control_request *pCtrl, 
	__u8 bFormatIndex, 
	__u8 bFrameIndex, 
	long *pFormat, 
	long *pWidth, 
	long *pHeight,
	long *plMinBitrate,
	long *plMaxBitrate)
{	
	stream_format_ctx_t *pStreamFormatCtx = (stream_format_ctx_t *)pCtx->pStreamFormatCtx;
	struct uvc_streaming_control_request *pCur;
	UAV_DBG_MSG("format = %d frame = %d\n", bFormatIndex, bFrameIndex);

	bFrameIndex -= 1;
	pCur =  pStreamFormatCtx->get_stream_contrl(bFormatIndex, bFrameIndex, PROBE_INDEX_CUR);
	memcpy((unsigned char *)pCtrl + 4, (unsigned char *)pCur + 4, sizeof(struct uvc_streaming_control_request)-4);
	// TODO Set proper value
	*pWidth = 0;
	*pHeight = 0;
	switch(bFormatIndex)
	{
#if HAS_H264_MP2TS_TRANS_SINK == 1
	case H264_FORMAT_MP2TS_TRANS_SINK_INDEX:
		*pFormat = FORMAT_H264_TS;
		break;
#endif
#if HAS_H264_VEN_STREAM_TRANS_SINK ==  1
		case H264_FORMAT_VEN_STREAM_TRANS_SINK_INDEX:
		*pFormat = FORMAT_H264_VEN;
		break;
#endif
#if HAS_H264_VEN_FRAME_TRANS_SINK ==  1
		case H264_FORMAT_VEN_TRANS_SINK_INDEX:
		*pFormat = FORMAT_H264_VEN;
		break;
#endif

#if HAS_MJPEG_PREV ==  1
		case MJPEG_FORMAT_TRANS_SINK_INDEX:
		*pFormat = FORMAT_MJPEG;
		break;
#endif
	}
	return sizeof(struct uvc_streaming_control_request);

	UAV_DBG_MSG("Unknown Format or Frame %d %d\n", bFormatIndex, bFrameIndex);
	return 0;
}

static int IsEncAltSettingVaiable(struct stream_format_if_t *pObj)
{
#if HAS_ENC_ISOCH == 1
	return 1;	
#else
	return 0;
#endif
}

static int IsDecAltSettingVaiable(struct stream_format_if_t *pObj)
{
#if HAS_DEC_ISOCH == 1
	return 1;	
#else
	return 0;
#endif
}

uvc_streaming_control_t *get_stream_contrl_enc(int format_index, int frame_index, int probe_index)
{
	UAV_DBG_MSG("get_stream_contrl_enc Frame format_index=%d\n",format_index);

	switch (format_index)
	{
#if HAS_H264_MP2TS_TRANS_SINK == 1
	case H264_FORMAT_MP2TS_TRANS_SINK_INDEX:
		if (probe_index < PROBE_INDEX_LAST) {
			UAV_DBG_MSG("H264_FORMAT_MP2TS_TRANS_SINK_INDEX Frame probe_index=%d\n",probe_index);
			return &h264_param_table[0][probe_index];
		}
		break;
#endif

#if HAS_H264_VEN_STREAM_TRANS_SINK ==  1
		case H264_FORMAT_VEN_STREAM_TRANS_SINK_INDEX:
		if (probe_index < PROBE_INDEX_LAST) {
			UAV_DBG_MSG("H264_FORMAT_MP2TS_TRANS_SINK_INDEX Frame probe_index=%d\n",probe_index);
			return &h264_ven_stream_param_table[0][probe_index];
		}
		break;
		break;
#endif

#if HAS_H264_VEN_FRAME_TRANS_SINK ==  1
		case H264_FORMAT_VEN_TRANS_SINK_INDEX:
		if(frame_index < H264_TRANS_SINK_FRAME_INDEX_LAST && frame_index >= 0) {
			return &h264_ven_frame_param_table[frame_index][probe_index];
		} else {
			UAV_DBG_ERR("Index outof range frame=%d probe=%d\n", frame_index, probe_index);
			return NULL;
		}
		break;
#endif

#if HAS_MJPEG_PREV ==  1
		case MJPEG_FORMAT_TRANS_SINK_INDEX:
		if(frame_index < MJPEG_FRAME_INDEX_LAST && frame_index >= 0) {
			return &mjpeg_param_table[frame_index][probe_index];
		} else {
			UAV_DBG_ERR("Index outof range frame=%d probe=%d\n", frame_index, probe_index);
			return NULL;
		}
		break;
#endif

	}
	UAV_DBG_MSG("format = %d or frame = %d or probe index=%d out of range\n", format_index, frame_index,probe_index);
	return 0;
}

uvc_streaming_control_t *get_stream_contrl_dec(int format_index, int frame_index, int probe_index)
{
	switch (format_index)
	{
#if HAS_H264_MP2TS_TRANS_SOURCE == 1
		case H264_FORMAT_MP2TS_TRANS_SOURCE_INDEX:
		if (probe_index < PROBE_INDEX_LAST) {
			return &h264_dec_param_table[probe_index];
		}
		break;
#endif
#if HAS_H264_VEN_TRANS_SOURCE ==  1
		case H264_FORMAT_VEN_TRANS_SOURCE_INDEX:
			{
				UAV_DBG_ERR("Format not enabled %d %d\n", frame_index, probe_index);
				return -1;
			}
			break;
#endif
	}
	UAV_DBG_MSG("format = %d or frame = %d or probe index=%d out of range\n", format_index, frame_index,probe_index);
	return 0;
	
}

static stream_format_ctx_t StreamFormatCtxEnc = 
{
	.get_stream_contrl = get_stream_contrl_enc
}; 

static stream_format_ctx_t StreamFormatCtxDec = 
{
	.get_stream_contrl = get_stream_contrl_dec
}; 

struct stream_format_if_t vid_trans_strm_format_if_enc =
{
	GetStreamingControl,
	SetStreamingControl,
	VerifyUvcStreamingControl,
	GetFrameParams,
	IsEncAltSettingVaiable,
	
	/* Context information */
	(void *)&StreamFormatCtxEnc
};

struct stream_format_if_t vid_trans_strm_format_if_dec =
{
	GetStreamingControl,
	SetStreamingControl,
	VerifyUvcStreamingControl,
	GetFrameParams,
	IsDecAltSettingVaiable,

	/* Context information */
	(void *)&StreamFormatCtxDec
};
