/******************************************************************************
 * gmstream.c -- merlin usb streaming peripheral
 *
 * This implements the device mode usb video streaming of data coming from
 * the codec through the user space application.
 * The information about each device associated with the chip selects are
 * obtained through devices registered in /sys/devices/platform/ 
 *
 * The major number is dynamically allocated. It is unique for this driver
 * and is not shared with other drivers.  The char device is registered
 * using the cdev and class device.
 *
 ******************************************************************************/
#define CONFIG_USB_GADGET_DUALSPEED

#include <linux/version.h>
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)
#include <linux/config.h>
#endif

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/moduleparam.h>
#include <linux/usb_ch9.h>
#include <linux/usb_gadget.h>
#include "gadget_chips.h"
#include <linux/musb.h>					// merlin usb ioctl codes  
#include <linux/types.h>
#include <linux/platform_device.h>
#include <linux/cdev.h>
#include <linux/highmem.h>

extern struct usb_ep *usb_ep_config(struct usb_gadget *,
			struct usb_endpoint_descriptor *,
			const char *find_name) __devinit;

/******************************************************************************
 * id numbers - use normal USB-IF procedures
 ******************************************************************************/
#define DRIVER_VENDOR_NUM	 	0x1AEA		/* Mobilygen */
#define DRIVER_PRODUCT_NUM 		0x2468		

//#define DRIVER_VENDOR_NUM		0x1a0a		/* OTG test device IDs */
//#define DRIVER_PRODUCT_NUM	0xbadd

//#define DRIVER_VENDOR_NUM	 	0x0525		/* PLX/NetChip   X */
//#define DRIVER_PRODUCT_NUM 	0xa4a0		/* Gadget Zero */

//#define DRIVER_VENDOR_NUM	 	0x045E		/* Microsoft     X */
//#define DRIVER_PRODUCT_NUM 	0x00F8		/* Microsoft Lifecam NX-6000 */

/******************************************************************************/
int debug = 0;
module_param(debug, int, 0644);
MODULE_PARM_DESC(debug, "Enable debug");

static unsigned buflen = 4096;
module_param(buflen, uint, S_IRUGO);

// if nonzero, how many seconds to wait before trying to wake up the host after suspend
static unsigned autoresume = 0;
module_param(autoresume, uint, 0);

static char * serial = "12345";
module_param(serial, charp, S_IRUGO);

#define dprintk(a...)   do { if (debug) printk(a); } while(0)
//#define dprintk(a...)

//#define dev_printk(level, dev, format, arg...) printk(level "%s %s: " format , dev_driver_string(dev) , (dev)->bus_id , ## arg)
#define ERROR(d,fmt,args...) dev_printk(KERN_ERR, &(d)->gadget->dev , fmt , ## args)
#define INFO(d,fmt,args...)  dev_printk(KERN_INFO, &(d)->gadget->dev , fmt , ## args)
#define DBG(d,fmt,args...)   dev_printk(KERN_DEBUG, &(d)->gadget->dev , fmt , ## args)
//#define DBG(d,fmt,args...) do { } while (0)

/******************************************************************************/
struct device;
struct class_device;

#define POOL_ENTRIES 3
struct req_pool {
	struct usb_request *req;			// allocated request
	void * buf;							// keep track of allocated buf for kfree on exit
	int inuse;							
};

struct properties_set {
	int number;
	const char *ep_name;
	struct usb_ep *ep;
	wait_queue_head_t ep_waitq;         // interrupt waiting and notification
	int ep_waitq_flag;              	// interrupt wait queue flag
	struct usb_endpoint_descriptor *ep_desc;
	int pos;
	struct usb_request *ep_req;
	struct req_pool *reqpool;					// array of requests
};

struct musbdevice  {
	int ep_number;
	int minor;
	const char *name;					// usb.0 device name 
	struct device *dev;					// -> platform_device.dev -^
	struct class_device *class;			//
	struct semaphore sem;				// mutual exclusion 
	struct cdev cdev;					// char device structure

//	struct properties_set in;
//	struct properties_set out;
};


int musb_major = 0; 					//major number is dynamically allocated
int musb_nr_devs = MUSB_MAX_DEVS; 
module_param(musb_nr_devs, int, S_IRUGO);

static struct musbdevice * devlist[MUSB_MAX_DEVS]; //one for each EP number
// devlist[] array index = minor# = cs#

static struct class * musb_class; 		// created in the init(), one for all cs

// open() private_data struct allocated per open() as needed by user app 
struct open_pdata {
	struct musbdevice * mdev;
//	u_int8_t	endian;					// current preference setting of this user
};

/******************************************************************************
 * strings
 ******************************************************************************/
#define STN_MANUFACTURER		1 
#define STN_PRODUCT				2 
#define STN_SERIAL				3 
#define STN_CFG_MAIN			250
#define STN_UVC_MAIN			110   

/******************************************************************************/
static const char driver_name[] 	= "gmstream";
static const char usb_str_product[] = "Crusher";
static const char cfg_main_str[] 	= "src-sink"; 
static const char usb_str_uvc_main[] = "UVC Camera";

/******************************************************************************/
static char	usb_str_manufacturer[50];	// manufacturer char string
static char	usb_str_serial_num[40];		// serial number char string setup in init()

/******************************************************************************/
// big enough to hold our biggest descriptor
#define USB_BUFSIZ	256

/******************************************************************************/
struct gmstream {
	spinlock_t          lock;
	struct usb_gadget	*gadget;
	struct usb_request	*req;		// for control responses
	u8					config; 	// 0, CONFIG_VALUE_MAIN, CONFIG_VALUE_DFU
	struct timer_list	resume;		// autoresume timer
};


static struct usb_endpoint_descriptor ep1_in_desc = {
	.bLength 			= USB_DT_ENDPOINT_SIZE,
	.bDescriptorType 	= USB_DT_ENDPOINT,
	.bEndpointAddress 	= USB_DIR_IN,
	.bmAttributes 		= USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize 	= __constant_cpu_to_le16(512),
};

static struct usb_endpoint_descriptor ep2_in_desc = {
	.bLength 			= USB_DT_ENDPOINT_SIZE,
	.bDescriptorType 	= USB_DT_ENDPOINT,
	.bEndpointAddress 	= USB_DIR_IN,
	.bmAttributes 		= USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize 	= __constant_cpu_to_le16(512),
};

//static struct usb_endpoint_descriptor ep3_in_desc = {
//	.bLength 			= USB_DT_ENDPOINT_SIZE,
//	.bDescriptorType 	= USB_DT_ENDPOINT,
//	.bEndpointAddress 	= USB_DIR_IN,
//	.bmAttributes 		= USB_ENDPOINT_XFER_BULK,
//	.wMaxPacketSize 	= __constant_cpu_to_le16(512),
//};

static struct usb_endpoint_descriptor ep4_in_desc = {
	.bLength 			= USB_DT_ENDPOINT_SIZE,
	.bDescriptorType 	= USB_DT_ENDPOINT,
	.bEndpointAddress 	= USB_DIR_IN,
	.bmAttributes 		= USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize 	= __constant_cpu_to_le16(512),
};


static struct usb_endpoint_descriptor ep1_out_desc = {
	.bLength 			= USB_DT_ENDPOINT_SIZE,
	.bDescriptorType 	= USB_DT_ENDPOINT,
	.bEndpointAddress 	= USB_DIR_OUT,
	.bmAttributes 		= USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize 	= __constant_cpu_to_le16(512),
};

static struct usb_endpoint_descriptor ep2_out_desc = {
	.bLength 			= USB_DT_ENDPOINT_SIZE,
	.bDescriptorType 	= USB_DT_ENDPOINT,
	.bEndpointAddress 	= USB_DIR_OUT,
	.bmAttributes 		= USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize 	= __constant_cpu_to_le16(512),
};

static struct usb_endpoint_descriptor ep3_out_desc = {
	.bLength 			= USB_DT_ENDPOINT_SIZE,
	.bDescriptorType 	= USB_DT_ENDPOINT,
	.bEndpointAddress 	= USB_DIR_OUT,
	.bmAttributes 		= USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize 	= __constant_cpu_to_le16(512),
};

static struct usb_endpoint_descriptor ep5_out_desc = {
	.bLength 			= USB_DT_ENDPOINT_SIZE,
	.bDescriptorType 	= USB_DT_ENDPOINT,
	.bEndpointAddress 	= USB_DIR_OUT,
	.bmAttributes 		= USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize 	= __constant_cpu_to_le16(512),
};

struct properties_set ep1_in_set = {
	.number = 1,
	.pos = 0,
	.ep_req = 0,
	.ep_desc = &ep1_in_desc,
};

struct properties_set ep1_out_set = {
	.number = 1,
	.pos = 0,
	.ep_req = 0,
	.ep_desc = &ep1_out_desc,
};

struct properties_set ep2_in_set = {
	.number = 2,
	.pos = 0,
	.ep_req = 0,
	.ep_desc = &ep2_in_desc,
};

struct properties_set ep2_out_set = {
	.number = 2,
	.pos = 0,
	.ep_req = 0,
	.ep_desc = &ep2_out_desc,
};

//struct properties_set ep3_in_set = {
//	.number = 3,
//	.pos = 0,
//	.ep_req = 0,
//	.ep_desc = &ep3_in_desc,
//};

struct properties_set ep3_out_set = {
	.number = 3,
	.pos = 0,
	.ep_req = 0,
	.ep_desc = &ep3_out_desc,
};

struct properties_set ep4_in_set = {
	.number = 4,
	.pos = 0,
	.ep_req = 0,
	.ep_desc = &ep4_in_desc,
};

struct properties_set ep5_out_set = {
	.number = 5,
	.pos = 0,
	.ep_req = 0,
	.ep_desc = &ep5_out_desc,
};


/******************************************************************************
 * configurations - we have one configuration
 ******************************************************************************/
#define CONFIG_VALUE_MAIN 		1 	
//#define CONFIG_VALUE_DFU		2	

//------------------------------------------------------------------------------
// OTG descriptor:
// struct usb_otg_descriptor {
//	__u8  bLength;
//	__u8  bDescriptorType;
//
//	__u8  bmAttributes;	/* support for HNP, SRP, etc */
// } __attribute__ ((packed));
//------------------------------------------------------------------------------

static struct usb_otg_descriptor otg_descriptor = {
	.bLength 			= sizeof(otg_descriptor),		//
	.bDescriptorType 	= USB_DT_OTG,					// 9
	//------------------
	.bmAttributes 		= USB_OTG_SRP,					// 
};

//------------------------------------------------------------------------------
// DEVICE descriptor 
// This struct also goes into usbfs file system device entry
// used in gmstream_setup(), gmstream_bind()
//------------------------------------------------------------------------------
static struct usb_device_descriptor device_desc = 
{
	.bLength 			= sizeof(device_desc),
	.bDescriptorType 	= USB_DT_DEVICE,
	//------------------
	.bcdUSB				= __constant_cpu_to_le16(0x0200),
	.bDeviceClass		= USB_CLASS_VENDOR_SPEC, //0xEF, /* USB_CLASS_VIDEO,*/ //USB_CLASS_VENDOR_SPEC,
		.bDeviceSubClass = 0, //ph
		.bDeviceProtocol = 0, //ph
		.bMaxPacketSize0 = 0, //ph
	//------------------
	.idVendor			= __constant_cpu_to_le16(DRIVER_VENDOR_NUM),
	.idProduct			= __constant_cpu_to_le16(DRIVER_PRODUCT_NUM),
		.bcdDevice		 = 0, //ph
	//------------------
	.iManufacturer		= STN_MANUFACTURER,
	.iProduct			= STN_PRODUCT,
	.iSerialNumber		= STN_SERIAL,
	.bNumConfigurations = 1, //2,
};

//------------------------------------------------------------------------------
// CONFIGURATION descriptor
// used in config_buf(), gmstream_bind()
//------------------------------------------------------------------------------
static struct usb_config_descriptor config_desc_main = {
	.bLength 			= sizeof(config_desc_main),
	.bDescriptorType 	= USB_DT_CONFIG,
	//------------------
	/* compute wTotalLength on the fly */
	.bNumInterfaces 	= 1, //was 4, //was 1,
	.bConfigurationValue= CONFIG_VALUE_MAIN,
	.iConfiguration 	= 0, //was STN_CFG_MAIN,
	.bmAttributes 		= USB_CONFIG_ATT_ONE, //|USB_CONFIG_ATT_SELFPOWER,
	.bMaxPower 			= 500>>1, /* 500mA */			//was 1, /* self-powered */
};

//...more entries of configs go here, if any...

/******************************************************************************
 * IAD descriptor - describes VIC (Video Interface Collection) as Video Function
 * to group mulitple interfaces together in this 'Function'
 *
 * struct usb_interface_assoc_descriptor {
 *	 __u8  bLength;
 *	 __u8  bDescriptorType;
 *
 *	 __u8  bFirstInterface;
 *	 __u8  bInterfaceCount;
 *	 __u8  bFunctionClass;
 *	 __u8  bFunctionSubClass;
 *	 __u8  bFunctionProtocol;
 *	 __u8  iFunction;
 * } __attribute__ ((packed));
 ******************************************************************************/
static struct usb_interface_assoc_descriptor iad_desc = 
{
	.bLength 			= sizeof(iad_desc),
	.bDescriptorType 	= USB_DT_INTERFACE_ASSOCIATION,
	.bFirstInterface	= 0, // interface # of the VideoControl interface that is associated with this function
	.bInterfaceCount	= 2,
	.bFunctionClass		= UVC_CC_VIDEO,
	.bFunctionSubClass	= UVC_SC_VIDEO_INTERFACE_COLLECTION,
	.bFunctionProtocol	= UVC_PC_PROTOCOL_UNDEFINED,
	.iFunction			= STN_PRODUCT
};

/******************************************************************************
 * INTEFACE 0
 *
 * struct usb_interface_descriptor {
 * 	__u8  bLength;
 *	__u8  bDescriptorType;
 *
 *	__u8  bInterfaceNumber;
 *	__u8  bAlternateSetting;
 *	__u8  bNumEndpoints;
 *	__u8  bInterfaceClass;
 *	__u8  bInterfaceSubClass;
 *	__u8  bInterfaceProtocol;
 *	__u8  iInterface;
 *} __attribute__ ((packed));
 ******************************************************************************/
static const struct usb_interface_descriptor intf_main_srcsink = {
	.bLength 			= sizeof(intf_main_srcsink),
	.bDescriptorType 	= USB_DT_INTERFACE,
	.bInterfaceNumber	= 0,
	.bAlternateSetting	= 0,
	.bNumEndpoints 		= 6,						// # of EP's excluding EP0
	.bInterfaceClass 	= USB_CLASS_VENDOR_SPEC, //USB_CLASS_VIDEO, //VENDOR_SPEC,
	.iInterface 		= STN_CFG_MAIN,
};

/******************************************************************************
 * INTERFACE 1 - VC (VideoControl) Interface
 * Standard VC Interface descriptor (for the webcam)
 ******************************************************************************/
static const struct usb_interface_descriptor intf_main_uvc = {
	.bLength 			= USB_DT_INTERFACE_SIZE,	//sizeof(intf_main_uvc),
	.bDescriptorType 	= USB_DT_INTERFACE,
	.bInterfaceNumber	= 1, //fixme
	.bAlternateSetting	= 1, //fixme
	.bNumEndpoints		= 1,
	.bInterfaceClass	= UVC_CC_VIDEO,
	.bInterfaceSubClass	= UVC_SC_VIDEOCONTROL,
	.bInterfaceProtocol	= UVC_PC_PROTOCOL_UNDEFINED,
	.iInterface			= STN_PRODUCT,
};

/******************************************************************************
 * class specific vc inteface descriptor (Video Descriptor)
 ******************************************************************************/
static const struct uvc_vc_interface_descriptor intf_vci_class_spec = {
	.bLength 			= 13, //was 20==sizeof(intf_vci_class_spec),
	.bDescriptorType 	= UVC_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VC_HEADER,
	.bcdUVC				= 0x0100,
	.wTotalLength		= 57, //was 0x0042,
	.dwClockFrequency	= 6000000,
	.bInCollection		= 0x01,	// number of streaming interfaces
	.baInterfaceNr[0]	= 0x01,
};

// camera IT
static const struct uvc_input_terminal_camera_descriptor intf_it_camera = {
	.bLength			= sizeof(intf_it_camera),
	.bDescriptorType	= UVC_CS_INTERFACE,
	.bDescriptorSubType = UVC_VC_INPUT_TERMINAL,
	.bTerminalID		= 0x01,	// ID of this terminal
	.wTerminalType		= UVC_ITT_CAMERA,
	.bAssocTerminal		= 0x00, // no association
	.iTerminal			= 0x00, // unused
	.wObjectiveFocalLengthMin = 0x0000,
	.wObjectiveFocalLengthMax = 0x0000,
	.wOcularFocalLength = 0x0000,
	.bControlSize       = 0x02,	// 2 bytes, this terminal does not implement any control
	.bmControls         = 0x0000,
};

// composit IT
static const struct uvc_input_terminal_descriptor intf_it_composite = {
	.bLength			= sizeof(intf_it_composite),
	.bDescriptorType	= UVC_CS_INTERFACE,
	.bDescriptorSubType = UVC_VC_INPUT_TERMINAL,
	.bTerminalID		= 0x02, // ID of this terminal 
	.wTerminalType		= UVC_COMPOSITE_CONNECTOR, // this terminal is the composite connector
	.bAssocTerminal		= 0x00, // no association
	.iTerminal			= 0x00, // unused
};

// OT descriptor (to the host)
static const struct uvc_output_terminal_descriptor intf_ot = {
	.bLength			= sizeof(intf_ot),
	.bDescriptorType	= UVC_CS_INTERFACE,
	.bDescriptorSubType = UVC_VC_OUTPUT_TERMINAL,
	.bTerminalID		= 0x03, // ID of this terminal 
	.wTerminalType		= UVC_TT_STREAMING, // this terminal is the composite connector
	.bAssocTerminal		= 0x00, // no association
	.bSourceID			= 0x05, // our input pin == output pin of unit 5
	.iTerminal			= 0x00, // unused
};

/******************************************************************************
 * Full & high speed endpoints: 
 *
 * usb 2.0 devices need to expose both high speed and full speed
 * descriptors, unless they only run at full speed.
 *
 * that means alternate endpoint descriptors (bigger packets)
 * and a "device qualifier" ... plus more construction options
 * for the config descriptor.
 ******************************************************************************/



/******************************************************************************
 * USB function
 ******************************************************************************/
static const struct usb_descriptor_header *main_header[] = {
	(struct usb_descriptor_header *) &intf_main_srcsink,
	(struct usb_descriptor_header *) &ep1_out_desc,
	(struct usb_descriptor_header *) &ep1_in_desc,
	(struct usb_descriptor_header *) &ep2_out_desc,
	(struct usb_descriptor_header *) &ep2_in_desc,
	(struct usb_descriptor_header *) &ep3_out_desc,
//	(struct usb_descriptor_header *) &ep3_in_desc,
	(struct usb_descriptor_header *) &ep4_in_desc,
	(struct usb_descriptor_header *) &ep5_out_desc,
//	(struct usb_descriptor_header *) &iad_desc,
//	(struct usb_descriptor_header *) &intf_main_uvc,
//	(struct usb_descriptor_header *) &intf_vci_class_spec,
//	(struct usb_descriptor_header *) &intf_it_camera,
//	(struct usb_descriptor_header *) &intf_it_composite,
//	(struct usb_descriptor_header *) &intf_ot,
//	(struct usb_descriptor_header *) &otg_descriptor,
	NULL,
};

/******************************************************************************
 * used in USB 2.0 only:
 ******************************************************************************/
static struct usb_qualifier_descriptor dev_qualifier = {
	.bLength 			= sizeof(dev_qualifier),
	.bDescriptorType 	= USB_DT_DEVICE_QUALIFIER,
	//-------------------
	.bcdUSB 			= __constant_cpu_to_le16(0x0200),
	.bDeviceClass 		= USB_CLASS_VENDOR_SPEC,	//USB_CLASS_VIDEO, //

	.bNumConfigurations = 1, //2,
};

/******************************************************************************
 * static strings, in UTF-8
 ******************************************************************************/
static struct usb_string strings[] = {
	{ STN_MANUFACTURER, usb_str_manufacturer, 	},
	{ STN_PRODUCT, 		usb_str_product, 		},
	{ STN_SERIAL, 		usb_str_serial_num, 	},
	{ STN_CFG_MAIN,	 	cfg_main_str, 			},
	{ STN_UVC_MAIN,		usb_str_uvc_main, 		},
	{ } /* end of list */
};

/******************************************************************************
 * string table
 ******************************************************************************/
static struct usb_gadget_strings stringtab = {
	.language	= 0x0409,	/* en-us */
	.strings	= strings,
};

/******************************************************************************
 * invoked from gmstream_setup() case USB_DT_CONFIG 
 *
 * config descriptors are also handcrafted.  these must agree with code
 * that sets configurations, and with code managing interfaces and their
 * altsettings.  other complexity may come from:
 *
 *  - high speed support, including "other speed config" rules
 *  - multiple configurations
 *  - interfaces with alternate settings
 *  - embedded class or vendor-specific descriptors
 *
 * this handles high speed, and has a second config that could as easily
 * have been an alternate interface setting (on most hardware).
 *
 * NOTE:  to demonstrate (and test) more USB capabilities, this driver
 * should include an altsetting to test interrupt transfers, including
 * high bandwidth modes at high speed.  (Maybe work like Intel's test device?)
 ******************************************************************************/
static int config_buf(struct usb_gadget *gadget, u8 *buf, u8 type, unsigned index)
{
	int	len;
	const struct usb_descriptor_header **function;

	dprintk("%s()\n", __FUNCTION__);

	// select hs header and its descriptors/ep's if hs was requested
	dprintk("gadget->speed==USB_SPEED_HIGH:%s\n", (gadget->speed == USB_SPEED_HIGH)?"yes":"no");

	// if type is other-speed, then swap hs and fs header 
	dprintk("type==USB_DT_OTHER_SPEED_CONFIG:%s\n", (type==USB_DT_OTHER_SPEED_CONFIG)?"yes":"no");

	// 'index' input param tells us which configuration: 0=src_sink, 1=loopback
	// current we support only one (index=0)
	if (index > 0 ) // make sure the specified index# of configuration entry exists
	{
		dprintk("Error: index too big\n");  
		return -EINVAL;
	}

	function = main_header;

	if (!gadget->is_otg)	/* for now, don't advertise srp-only devices */
	{
		dprintk("gadget->is_otg is FALSE\n");
		function++;
	}

	// build config descriptor (1 of 2 here), from single descriptor vector 'buf'
	len = usb_gadget_config_buf(&config_desc_main, buf, USB_BUFSIZ, function);
	if (len < 0)
		return len;
	((struct usb_config_descriptor*)buf)->bDescriptorType = type;

	return len;
}

/******************************************************************************
 * allocate a request and also allocate buffer
 ******************************************************************************/
static struct usb_request * alloc_ep_req(struct usb_ep *ep, unsigned length)
{
	struct usb_request *req;

	req = usb_ep_alloc_request(ep, GFP_ATOMIC);
	if (req) 
	{
		req->length = length;
		req->buf = usb_ep_alloc_buffer(ep, length, &req->dma, GFP_ATOMIC);
		if (!req->buf) 
		{
			usb_ep_free_request(ep, req);
			req = NULL;
		}
	}
	return req;
}

/******************************************************************************
 * free buffer associated with the request and the request itself
 ******************************************************************************/
static void free_ep_req(struct usb_ep *ep, struct usb_request *req)
{
	// free an io buffer used by this ep struct
	if (req->buf)
		usb_ep_free_buffer(ep, req->buf, req->dma, req->length); // usb_gadget.h inline
	
	// free a struct usb_request used by this ep struct
	usb_ep_free_request(ep, req);
}


/******************************************************************************
 * setup a pool of requests 
 ******************************************************************************/
static int pool_init(struct properties_set *ep_set, struct usb_ep *ep)
{
	int i;
	int	status;
	struct usb_request *req;
	struct req_pool *reqpool;

	status = 0;
	reqpool = kzalloc(POOL_ENTRIES * sizeof(*reqpool), GFP_KERNEL);
	if (!reqpool)
		return -ENOMEM;

	ep_set->reqpool = reqpool;		// so that we can look it up later 
	
	for (i=0; i<POOL_ENTRIES; i++) {
		req = usb_ep_alloc_request(ep, GFP_ATOMIC);
		if (!req) {
			printk(KERN_ERR "pool_init can't alloc req\n");
			status = -EIO;
			break;
		}
		
		// let musb_write() initialize which ep_set this req is used for 
		req->context = (struct properties_set *)ep_set;

		req->buf = usb_ep_alloc_buffer(ep, buflen, &req->dma, GFP_ATOMIC);
		if (!req->buf) {
			printk(KERN_ERR "Error allocating buffer in musb_write\n");
			//then also free the allocated request
		}

		reqpool[i].buf = req->buf;
		reqpool[i].inuse = 0;
		reqpool[i].req = req;
	}

	return status;
}

/******************************************************************************
 * shut down pool of requests 
 ******************************************************************************/
static int pool_exit(struct properties_set *ep_set, struct usb_ep *ep)
{
	int i;
	int	status;
	struct usb_request *req;
	struct req_pool *reqpool;

	status = 0;
	reqpool = ep_set->reqpool;
	for (i=0; i<POOL_ENTRIES; i++) {
		if (reqpool[i].inuse) {
			printk("pool request to unqueue\n");
		}

		req = reqpool[i].req;
		if (req) {
			usb_ep_free_buffer(ep, reqpool[i].buf, req->dma, buflen);
			usb_ep_free_request(ep, req);
		}
	}

	kfree(reqpool);
	ep_set->reqpool = NULL;
	return status;
}

/******************************************************************************
 * check out an entry
 ******************************************************************************/
struct usb_request * pool_get_req(struct properties_set *ep_set)
{
	int i;
	struct req_pool *reqpool;
	
	reqpool = ep_set->reqpool;
	for (i=0; i<POOL_ENTRIES; i++) {
		if (reqpool[i].inuse == 0) {
			reqpool[i].inuse = 1;
			//printk(KERN_INFO "pool_get_req #%d\n", i);
			return reqpool[i].req;
		}
	}

	return 0; //none available at this time
}

/******************************************************************************
 * check back in an entry
 ******************************************************************************/
int pool_put_req(struct properties_set *ep_set, struct usb_request *req)
{
	int i;
	struct req_pool *reqpool;
	
	reqpool = ep_set->reqpool;
	for (i=0; i<POOL_ENTRIES; i++) {
		if (reqpool[i].req == req) {
			reqpool[i].inuse = 0;
			//printk(KERN_INFO "pool_put_req #%d\n", i);
			return 0;
		}
	}

	printk(KERN_ERR "Error pool_put_req()\n");
	return 0;
}

/******************************************************************************
 * completion handler
 ******************************************************************************/
static void ep_in_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct gmstream *gm = ep->driver_data;
	int	status = req->status;
	int ep_nb;
	struct properties_set *ep_set;
	ep_set = (struct properties_set *)req->context;
	ep_nb = ((struct properties_set *)req->context)->number;

	if (status==0) {
		pool_put_req(ep_set, req);
		ep_set->ep_waitq_flag = 1;
	   	wake_up_interruptible((wait_queue_head_t*)&(ep_set->ep_waitq));
	}
	else {
		switch (status) 
		{
		/* this endpoint is normally active while we're configured */
		case -ECONNABORTED:		/* hardware forced ep reset */
			dprintk("IN%i complete CONNABORT\n", ep_nb);
		case -ECONNRESET:		/* request dequeued */
			dprintk("IN%i complete CONNRESET\n", ep_nb);
		case -ESHUTDOWN:		/* disconnect from host */
			dprintk("IN%i complete SHUTDOWN \n", ep_nb);
			DBG(gm, "%s gone (%d), %d/%d\n", ep->name, status, req->actual, req->length);
			//free_ep_req(ep, req);	
			return;

		case -EOVERFLOW: // buffer overrun on read means we didn't provide a big enough buffer
			dprintk("IN%i complete OVERFLOW\n", ep_nb);
		default:
			dprintk("IN%i complete DEFAULT\n", ep_nb);
			DBG(gm, "%s complete --> %d, %d/%d\n", ep->name, status, req->actual, req->length);

		case -EREMOTEIO:		/* short read */
			dprintk("IN%i complete REMOTEIO\n", ep_nb);
			break;
		}
#if 0
		// enqueue/submit an I/O request to this endpoint
		status = usb_ep_queue(ep, req, GFP_ATOMIC);
		if (status) {
			ERROR(gm, "kill %s:  resubmit %d bytes --> %d\n", ep->name, req->length, status);
			usb_ep_set_halt(ep);
		}
#endif
		
	}
}

/*******************************************************************************
 * The caller of sendpage will write to the usb through this fops function
 *
 * ssize_t sendfile(int out_fd, int in_fd, off_t *offset, size_t count)
 * 
 * in_fd is a file descriptor opened for reading.
 * out_fd is a descriptor opened for writing.
 * Offset is a pointer to a variable holding the input file pointer position from which sendfile() will start reading data.
 * count is the number of bytes to copy between file descriptors.
 *
 ******************************************************************************/
//ssize_t musb_sendpage(struct socket *sock, struct page *page, int offset, size_t size, int flags)
ssize_t musb_sendpage(struct file *filp, struct page *page, int offset, size_t len, loff_t *ppos, int flags)
{
	int retval=0, status=0;
	struct properties_set *ep_set;
	struct usb_request *req;
	void *srcaddr;
	char *kaddr;
	
	kaddr = kmap(page);
	srcaddr = kaddr + offset;				// kernel space virtual addr, with length = len
	
	retval = len;
	switch (iminor(filp->f_dentry->d_inode)) {
	case 1:
		ep_set = &ep1_in_set;
		break;
	case 2:
		ep_set = &ep2_in_set;
		break;
//	case 3:
//		ep_set = &ep3_in_set;
//		break;
	/*case 4:
		ep_set = &ep4_in_set;
		break;*/
	default:
		printk(KERN_ERR "EP doesn't support write\n");
		return -EINVAL;
	}

	if (len > 0) {
		do {
			req = pool_get_req(ep_set);		// grab a pre-allocated usb request
			if (!req)
			{
				wait_event_interruptible(ep_set->ep_waitq, ep_set->ep_waitq_flag != 0);
				if (ep_set->ep_waitq_flag < 0) {	// if error reported by handler
					status = -EFAULT;				// return with error code
					goto exit;
				}
				ep_set->ep_waitq_flag = 0;
			}
		} while (!req); 

		req->complete = ep_in_complete;
		req->length = len;					// the whole thing, bypassing gadgetapi's concept
		req->buf = srcaddr;					// dma directly from sendfile caller's src buffer 
		req->context = (struct properties_set *)ep_set;

		status = usb_ep_queue(ep_set->ep, req, GFP_ATOMIC); // enqueue I/O request to this EP
		if (status) 
		{
			printk(KERN_ERR "kill %s:  resubmit %d bytes --> %d\n", ep_set->ep->name, req->length, status);
			usb_ep_set_halt(ep_set->ep);
		}
	}	
exit:
	kunmap(page);
	return retval;
}

/******************************************************************************
 * write() for /dev/musb.N
 * Process in 4K chunks if the incoming user's buffer is bigger than 4K. 
 ******************************************************************************/
static ssize_t musb_write(struct file *filp, const char *userbuf, size_t len, loff_t *ppos)
{
	int retval=0, status=0;
	struct properties_set *ep_set;
	int xferlen; 
	struct usb_request *req;

	retval = len;
	switch (iminor(filp->f_dentry->d_inode)) {
	case 1:
		ep_set = &ep1_in_set;
		break;
	case 2:
		ep_set = &ep2_in_set;
		break;
//	case 3:
//		ep_set = &ep3_in_set;
//		break;
	/*case 4:
		ep_set = &ep4_in_set;
		break;*/
	default:
		printk(KERN_ERR "EP doesn't support write\n");
		return -EINVAL;
	}

	while (len > 0) {
		xferlen = (len>buflen? buflen:len);

		do {
			req = pool_get_req(ep_set);
			if (!req)
			{
				//printk("."); 
				wait_event_interruptible(ep_set->ep_waitq, ep_set->ep_waitq_flag != 0);
				if (ep_set->ep_waitq_flag < 0) {	// if error reported by handler
					printk(KERN_ERR "write waitq error\n");
					status = -EFAULT;				// return with error code
					goto exit;
				}
				ep_set->ep_waitq_flag = 0;
			}
		} while (!req); 

		req->context = (struct properties_set *)ep_set;

		req->length = xferlen;
		if (copy_from_user(req->buf, userbuf, xferlen)) {
			printk(KERN_ERR "copy_from_user failed\n");
			retval = -EFAULT;
			goto exit;
		}

		req->complete = ep_in_complete;
		status = usb_ep_queue(ep_set->ep, req, GFP_ATOMIC); // enqueue I/O request to this EP
		if (status) {
			printk(KERN_ERR "kill %s:  resubmit %d bytes --> %d\n", ep_set->ep->name, req->length, status);
			usb_ep_set_halt(ep_set->ep);
		}

		userbuf += xferlen;
		len -= xferlen;
	}

exit:
	return retval;
}

/******************************************************************************
 * completion handler
 ******************************************************************************/
static void ep_out_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct gmstream *gm = ep->driver_data;
	int	status = req->status;
	int ep_nb;
	struct properties_set *ep_set;
	ep_set = (struct properties_set *)req->context;
	ep_nb = ((struct properties_set *)req->context)->number;


	// we get here when we have received enough packets to make one 4096(buflen) byte chunk 
	// resync with set_interface or set_config
	if (status==0) {
		// req->buf has capacity req->length, and contains req->actual bytes 
		
		// update the current req struct address for use by read
		ep_set->ep_req = req;

		if (req->actual != buflen)
			printk(KERN_ERR "ep%iout req->actual=%d\n", ep_nb, req->actual);

		ep_set->ep_waitq_flag = 1;		// inform of our status to read, to wake it up 
		wake_up_interruptible((wait_queue_head_t*)&(ep_set->ep_waitq));
	}
	else {
		switch (status) 
		{
		case -EOVERFLOW: // buffer overrun on read means we didn't provide a big enough buffer
			dprintk("OUT%i complete OVERFLOW\n", ep_nb);
		default:
			dprintk("OUT%i complete DEFAULT\n", ep_nb);
			DBG(gm, "%s complete --> %d, %d/%d\n", ep->name, status, req->actual, req->length);
		case -EREMOTEIO:		/* short read */
			dprintk("OUT%i complete REMOTEIO\n", ep_nb);
			break;

		/* this endpoint is normally active while we're configured */
		case -ECONNABORTED:		/* hardware forced ep reset */
			dprintk("OUT%i complete CONNABORT\n", ep_nb);
		case -ECONNRESET:		/* request dequeued */
			dprintk("OUT%i complete CONNRESET\n", ep_nb);
		case -ESHUTDOWN:		/* disconnect from host */
			dprintk("OUT%i complete SHUTDOWN \n", ep_nb);
			DBG(gm, "%s gone (%d), %d/%d\n", ep->name, status, req->actual, req->length);
			//free_ep_req(ep, req);	
			return;
		}
		
		status = usb_ep_queue(ep, req, GFP_ATOMIC);
		if (status) 
		{
			ERROR(gm, "kill %s:  resubmit %d bytes --> %d\n", ep->name, req->length, status);
			usb_ep_set_halt(ep);
		}		
	}
}

/******************************************************************************
 * read() 
 * Process in 4K chunks if the incoming user's buffer is bigger than 4K. 
 ******************************************************************************/
static ssize_t musb_read(struct file *filp, char *userbuf, size_t len, loff_t *ppos)
{
	int retval=0, status=0;
	int xferlen; 							// size of transfer in each iteration 
	int bytescopied;						// how many bytes copied to user so far
	struct usb_request *req;
	int avail;

	struct properties_set *ep_set;
	retval = len;
	
	switch (iminor(filp->f_dentry->d_inode)) {
	case 1:
		ep_set = &ep1_out_set;
		break;
	case 2:
		 ep_set = &ep2_out_set;
		 break;
	case 3:
		 ep_set = &ep3_out_set;
		 break;
	/*case 5:
		 ep_set = &ep5_out_set;
		 printk(KERN_ERR "read5\n");
		 break;*/
	default:
		printk(KERN_ERR "EP doesn't support read\n");
		return -EINVAL;
	}

	bytescopied = 0;							// start with 0 bytes copied so far with this read()

	while (bytescopied < len)					// keep filling caller's buf until we satisfied him
	{
		if (ep_set->ep_req == 0)					// if no data available, wait for it
		{
   			wait_event_interruptible(ep_set->ep_waitq, ep_set->ep_waitq_flag != 0);
			if (ep_set->ep_waitq_flag < 1) {		// if error reported by handler
				ep_set->ep_waitq_flag = 0;
				return -EFAULT;					// return with error code
			}
			ep_set->ep_waitq_flag = 0;
		}

		if (ep_set->ep_req != 0)					// ok we got a new large block of data coming in 
		{
			xferlen = len - bytescopied;		// copy the exact number of bytes requested if possible
			avail = buflen - ep_set->pos;		// but see how much is left in this block of req->buf
			if (xferlen > avail)				// not enough available?
				xferlen = avail;				// then copy only what's available

			if (copy_to_user(userbuf, (ep_set->ep_req->buf + ep_set->pos), xferlen))
			{
				printk(KERN_ERR "copy_to_user failed\n");
				retval = -EFAULT;
				goto exit;
			}
			ep_set->pos += xferlen;			// update index into req->buf for next iteration 
			userbuf += xferlen;					// update dst ptr for next iteration of copy_to_user
			bytescopied += xferlen;				// track how many bytes we have copied so far in this read()

			if (ep_set->pos >= buflen)			// done extracting data out from this req's buffer
			{
				req = ep_set->ep_req; 				// save
				ep_set->ep_req = 0;				// tell ourselves this is done so we don't redo it again if user
												// does the read faster than the usb xfer  
				status = usb_ep_queue(ep_set->ep, req, GFP_ATOMIC);
				if (status) 
				{
					printk(KERN_ERR "kill %s:  resubmit %d bytes --> %d\n", ep_set->ep->name, ep_set->ep_req->length, status);
					usb_ep_set_halt(ep_set->ep);
				}
				ep_set->pos = 0;				// wrap around for next large incoming block
			}
		}
	}

exit:
//	up(&mdev->sem);
	return retval;
}


/******************************************************************************
 * part of the gmstream_set_config()
 ******************************************************************************/
static int set_config_desc_main(struct gmstream *gm, gfp_t gfp_flags)
{
	int	result = 0;
	struct usb_ep	  *ep;
	struct usb_gadget *gadget = gm->gadget;
	int in = -1;
	struct  properties_set  *ep_set;
	
	dprintk("%s()\n", __FUNCTION__);

	list_for_each_entry(ep, &(gadget)->ep_list, ep_list) //gadget_for_each_ep(ep, gadget) 
	{
		struct usb_request *req;					// usb request associated with an EP

		if (strcmp(ep->name, ep1_in_set.ep_name) == 0) {
			ep_set = &ep1_in_set;
			in = 1;
		}else if (strcmp(ep->name, ep2_in_set.ep_name) == 0) {
			ep_set = &ep2_in_set;
			in = 1;
//		}else if (strcmp(ep->name, ep3_in_set.ep_name) == 0) {
//			ep_set = &ep3_in_set;
//			in = 1;
		}else if (strcmp(ep->name, ep1_out_set.ep_name) == 0) {
			ep_set = &ep1_out_set;
			in = 0;
		}else if (strcmp(ep->name, ep2_out_set.ep_name) == 0) {
			ep_set = &ep2_out_set;
			in = 0;
		}else if (strcmp(ep->name, ep3_out_set.ep_name) == 0) {
			ep_set = &ep3_out_set;
			in = 0;
		/*}else if (strcmp(ep->name, ep5_out_set.ep_name) == 0) {
			ep_set = &ep5_out_set;
			in = 0;*/
		}else{
			continue;
		}

	 	ep->driver_data = gm;
		ep_set->ep = ep;

		if (in == 1) {

			result = usb_ep_enable(ep, ep_set->ep_desc);
			if (result != 0){
				ERROR(gm, "can't enable EP %s\n", ep->name);
				result = -EIO;
				break;
			}

			//ep_set->ep_req = req;			
		} 
		else if (in == 0) {

			req = alloc_ep_req(ep, buflen);
			if (!req){
				usb_ep_disable(ep);
				ERROR(gm, "can't alloc req for %s\n", ep->name);
				result = -EIO;
				break;
			}
			req->context = (struct properties_set *)ep_set;
			req->complete = ep_out_complete;

			result = usb_ep_enable(ep, ep_set->ep_desc);
			if (result != 0){
				ERROR(gm, "can't enable EP %s\n", ep->name);
				result = -EIO;
				break;
			}
		
			result = usb_ep_queue(ep, req, gfp_flags);
			if (result){
				ERROR(gm, "can't queue req for %s\n", ep->name);
				free_ep_req(ep, req);
				usb_ep_disable(ep);
				result = -EIO;
			}

		}

	}
	if (result != 0)
		printk(KERN_ERR "after foreach ep, result==%d\n", result);
	
	/* caller is responsible for cleanup on error */
	return result;
}

/******************************************************************************
 * helper function used by gmstream_setup() 
 ******************************************************************************/
static void gmstream_reset_config(struct gmstream *gm)
{
	dprintk("%s()\n", __FUNCTION__);

	if (gm->config == 0)
		return;

	DBG(gm, "reset config\n");

	/* just disable endpoints, forcing completion of pending i/o.
	 * all our completion handlers free their requests in this case.
	 */
	if (ep1_in_set.ep) 
	{
		usb_ep_disable(ep1_in_set.ep);
		ep1_in_set.ep = NULL;
	}
	if (ep1_out_set.ep) 
	{
		usb_ep_disable(ep1_out_set.ep);
		ep1_out_set.ep = NULL;
	}

	if (ep2_in_set.ep) 
	{
		usb_ep_disable(ep2_in_set.ep);
		ep2_in_set.ep = NULL;
	}
	if (ep2_out_set.ep) 
	{
		usb_ep_disable(ep2_out_set.ep);
		ep2_out_set.ep = NULL;
	}

//	if (ep3_in_set.ep) 
//	{
//		usb_ep_disable(ep3_in_set.ep);
//		ep3_in_set.ep = NULL;
//	}
	if (ep3_out_set.ep) 
	{
		usb_ep_disable(ep3_out_set.ep);
		ep3_out_set.ep = NULL;
	}

	if (ep4_in_set.ep) 
	{
		usb_ep_disable(ep4_in_set.ep);
		ep4_in_set.ep = NULL;
	}
	if (ep5_out_set.ep) 
	{
		usb_ep_disable(ep5_out_set.ep);
		ep5_out_set.ep = NULL;
	}

	gm->config = 0;
	del_timer(&gm->resume);
}


/******************************************************************************
 * helper function used by gmstream_setup() to setup the configuration
 *
 * change our operational config.  this code must agree with the code
 * that returns config descriptors, and altsetting code.
 *
 * it's also responsible for power management interactions. some
 * configurations might not work with our current power sources.
 *
 * note that some device controller hardware will constrain what this
 * code can do, perhaps by disallowing more than one configuration or
 * by limiting configuration choices (like the pxa2xx).
 ******************************************************************************/
static int gmstream_set_config(struct gmstream *gm, unsigned number, gfp_t gfp_flags)
{
	int result = 0; 
	struct usb_gadget *gadget = gm->gadget;

	dprintk("%s()\n", __FUNCTION__);

	if (number == gm->config)
	{
		dprintk("set config is already set...\n");
		return 0;
	}

	// here gadget->name is "dwc_otg_pcd"

	if (gadget_is_sa1100(gadget) && gm->config) 
	{
		/* tx fifo is full, but we can't clear it...*/
		INFO(gm, "can't change configurations\n");
		return -ESPIPE;
	}
	gmstream_reset_config(gm);

	switch(number) 
	{
	case CONFIG_VALUE_MAIN:
		result = set_config_desc_main(gm, gfp_flags);
		break;
//	case CONFIG_VALUE_DFU:
//		result = set_config_desc_dfu(gm, gfp_flags);
//		break;
	default:
		result = -EINVAL;
		/* FALL THROUGH */
	case 0:
		return result;
	}

	if (!result && !ep1_in_set.ep)   // && !ep1_out)
		result = -ENODEV;
		
	if (result)
		gmstream_reset_config(gm);
	else 
	{
		char *speed;

		switch (gadget->speed) 
		{
		case USB_SPEED_LOW:	 speed = "low";  break;
		case USB_SPEED_FULL: speed = "full"; break;
		case USB_SPEED_HIGH: speed = "high"; break;
		default: 			 speed = "?";    break;
		}

		gm->config = number;
		INFO(gm, "%s speed config #%d\n", speed, number);
	}
	return result;
}

/******************************************************************************
 * installed by gmstream_bind(), and invoked by gmstream_setup
 ******************************************************************************/
static void gmstream_setup_complete(struct usb_ep *ep, struct usb_request *req)
{
	if (req->status || req->actual != req->length)
	{
		DBG((struct gmstream *) ep->driver_data, 
				"setup complete --> %d, %d/%d\n",
				req->status, req->actual, req->length);
	}
}

/******************************************************************************
 * The setup() callback implements all the ep0 functionality that's
 * not handled lower down, in hardware or the hardware driver (like
 * device and endpoint feature flags, and their status).  It's all
 * housekeeping for the gadget function we're implementing.  Most of
 * the work is in config-specific setup.
 ******************************************************************************/
static int gmstream_setup(struct usb_gadget *gadget, const struct usb_ctrlrequest *ctrl)
{
	struct gmstream *gm = get_gadget_data(gadget);
	struct usb_request	*req = gm->req;
	int	value = -EOPNOTSUPP;
	u16	w_index = le16_to_cpu(ctrl->wIndex);
	u16	w_value = le16_to_cpu(ctrl->wValue);
	u16	w_length = le16_to_cpu(ctrl->wLength);

//	dprintk("SETUP bRequest=0x%02X\n", ctrl->bRequest);

	// usually this stores reply data in the pre-allocated ep0 buffer,
	// but config change events will reconfigure hardware.

	req->zero = 0;
	switch (ctrl->bRequest) 
	{
	case USB_REQ_GET_DESCRIPTOR:
		dprintk(KERN_INFO "USB_REQ_GET_DESCRIPTOR value=%04x index=%04x len=%d\n", w_value, w_index, w_length);

		if (ctrl->bRequestType != USB_DIR_IN)
			goto unknown;
		switch (w_value >> 8) 
		{
		
		case USB_DT_DEVICE:
			dprintk(KERN_INFO "w_value:USB_DT_DEVICE\n");
			value = min(w_length, (u16)sizeof(device_desc));
			memcpy(req->buf, &device_desc, value);
			break;

		case USB_DT_DEVICE_QUALIFIER:	// this is for DUALSPEED 
			dprintk(KERN_INFO "w_value:USB_DT_DEVICE_QUALIFIER\n");
			if (!gadget->is_dualspeed)
				break;
			value = min(w_length, (u16)sizeof(dev_qualifier));
			memcpy(req->buf, &dev_qualifier, value);
			break;

		case USB_DT_OTHER_SPEED_CONFIG:	// this is for DUALSPEED 
			dprintk(KERN_INFO "w_value:USB_DT_OTHER_SPEED_CONFIG\n");
			if (!gadget->is_dualspeed)
				break;
			// FALLTHROUGH

		case USB_DT_CONFIG:
			dprintk(KERN_INFO "w_value:USB_DT_CONFIG\n");
			value = config_buf(gadget, req->buf, w_value >> 8, w_value & 0xff);
			if (value >= 0)
				value = min(w_length, (u16)value);
			break;

		case USB_DT_STRING:
			dprintk(KERN_INFO "w_value:USB_DT_STRING\n");
			/* wIndex == language code.
			 * this driver only handles one language, you can
			 * add string tables for other languages, using
			 * any UTF-8 characters
			 */
			value = usb_gadget_get_string(&stringtab, w_value & 0xff, req->buf);
			if (value >= 0)
				value = min(w_length, (u16)value);
			break;
		}
		break;

	/* currently two configs, two speeds */
	case USB_REQ_SET_CONFIGURATION:
		dprintk(KERN_INFO "USB_REQ_SET_CONFIGURATION value=%04x index=%04x len=%d\n", w_value, w_index, w_length);

		if (ctrl->bRequestType != 0)
			goto unknown;
		if (gadget->a_hnp_support)
			dprintk(KERN_INFO "HNP available\n");
		else if (gadget->a_alt_hnp_support)
			dprintk(KERN_INFO "HNP needs a different root port\n");
		else
			dprintk(KERN_INFO "HNP inactive\n");
		spin_lock(&gm->lock);
		value = gmstream_set_config(gm, w_value, GFP_ATOMIC);
		spin_unlock(&gm->lock);
		break;

	case USB_REQ_GET_CONFIGURATION:
		dprintk(KERN_INFO "USB_REQ_GET_CONFIGURATION value=%04x index=%04x len=%d\n", w_value, w_index, w_length);

		if (ctrl->bRequestType != USB_DIR_IN)
			goto unknown;
		*(u8*)req->buf = gm->config;
		value = min(w_length, (u16)1);
		break;

	/* until we add altsetting support, or other interfaces,
	 * only 0/0 are possible.  pxa2xx only supports 0/0 (poorly)
	 * and already killed pending endpoint I/O.
	 */
	case USB_REQ_SET_INTERFACE:
		
		dprintk(KERN_INFO "USB_REQ_SET_INTERFACE value=%04x index=%04x len=%d\n", w_value, w_index, w_length);

		if (ctrl->bRequestType != USB_RECIP_INTERFACE)
			goto unknown;
		spin_lock(&gm->lock);
		if (gm->config && w_index == 0 && w_value == 0) 
		{
			//u8 config = gm->config;

			/* resets interface configuration, forgets about
			 * previous transaction state (queued bufs, etc)
			 * and re-inits endpoint state (toggle etc)
			 * no response queued, just zero status == success.
			 * if we had more than one interface we couldn't
			 * use this "reset the config" shortcut.
			 */
			//gmstream_reset_config(gm);
			//gmstream_set_config(gm, config, GFP_ATOMIC);
			value = 0;
		}
		spin_unlock(&gm->lock);
		break;

	case USB_REQ_GET_INTERFACE:
		dprintk(KERN_INFO "USB_REQ_GET_INTERFACE value=%04x index=%04x len=%d\n", w_value, w_index, w_length);

		if (ctrl->bRequestType != (USB_DIR_IN|USB_RECIP_INTERFACE))
			goto unknown;
		if (!gm->config)
			break;
		if (w_index != 0) 
		{
			value = -EDOM;
			break;
		}
		*(u8*)req->buf = 0;
		value = min(w_length, (u16)1);
		break;

	/*
	 * These are the same vendor-specific requests supported by
	 * Intel's USB 2.0 compliance test devices.  We exceed that
	 * device spec by allowing multiple-packet requests.
	 */
	case 0x5b:	/* control WRITE test -- fill the buffer */
		dprintk(KERN_INFO "**case 0x5b** value=%04x index=%04x len=%d\n", w_value, w_index, w_length);
		if (ctrl->bRequestType != (USB_DIR_OUT|USB_TYPE_VENDOR))
			goto unknown;
		if (w_value || w_index)
			break;
		/* just read that many bytes into the buffer */
		if (w_length > USB_BUFSIZ)
			break;
		value = w_length;
		break;

	case 0x5c:	/* control READ test -- return the buffer */
		dprintk(KERN_INFO "**case 0x5c** value=%04x index=%04x len=%d\n", w_value, w_index, w_length);
		if (ctrl->bRequestType != (USB_DIR_IN|USB_TYPE_VENDOR))
			goto unknown;
		if (w_value || w_index)
			break;
		/* expect those bytes are still in the buffer; send back */
		if (w_length>USB_BUFSIZ  ||  w_length != req->length)
			break;
		value = w_length;
		break;
	
// Mobilygen Vendor Requests
//#define VX_CSR_READ 					0xB5 // Control Status Register Read (8051)
//#define VX_CSR_WRITE 					0xB6 // Control Status Register Write (8051)
//#define VX_CSR_READ_MEM_BLOCK 		0xB7 // Control Status Register Read memory Block (8051)
//#define VX_CSR_WRITE_MEM_BLOCK 	   *0xB8 // Control Status Register Write memory Block (8051)
//#define VX_HOST_READ_REG 				0xB9 // Host Read Register (GPIF)
//#define VX_HOST_WRITE_REG 			0xBA // Host Write Register (GPIF)
//#define VX_EEPROM       				0xBB // Read/Write EEPROM
//#define VX_MG1264_RESET				0xBC // Reset MG1264
//#define VX_CSR_CFG_READ_MEM_BLOCK	   *0xBD // Configure the Read Mem Block  VR Command for address, Block Id, ...
//#define VX_CSR_WRITE_MEM_BLOCK_DATA 	0xBE // Sending down the paylod for the Write Mem Block, you must first setup and configure the write mem block prior to sending down this command.
	
	case VX_CSR_WRITE_MEM_BLOCK:
		dprintk(KERN_INFO "VX_CSR_WRITE_MEM_BLOCK control req (bRequestType=0x%02x bRequest=0x%02x\n"
				"value=0x%04x index=0x%04x length=%d)\n",
					ctrl->bRequestType, ctrl->bRequest,
					w_value, w_index, w_length);
		if (ctrl->bRequestType != (USB_DIR_OUT |USB_TYPE_VENDOR|USB_RECIP_DEVICE))
		{
			DBG(gm, "VX_CSR_WRITE_MEM_BLOCK's bRequestType unexpected\n");
			goto unknown;
		}
		value = w_length;
		break;


    case VX_CSR_CFG_READ_MEM_BLOCK:
		//fall through for now 
	
	case VX_CSR_READ_MEM_BLOCK:
		dprintk(KERN_INFO "VX_CSR_READ_MEM_BLOCK control req (bRequestType=0x%02x bRequest=0x%02x\n"
				"value=0x%04x index=0x%04x length=%d)\n",
					ctrl->bRequestType, ctrl->bRequest,
					w_value, w_index, w_length);
		if (ctrl->bRequestType != (USB_DIR_OUT |USB_TYPE_VENDOR|USB_RECIP_DEVICE))
		{
			DBG(gm, "VX_CSR_READ_MEM_BLOCK's bRequestType unexpected\n");
			goto unknown;
		}
		value = w_length;
		break;

		
	case VX_CSR_WRITE:
		dprintk(KERN_INFO "VX_CSR_WRITE control req (bRequestType=0x%02x bRequest=0x%02x\n"
				"value=0x%04x index=0x%04x length=%d)\n",
					ctrl->bRequestType, ctrl->bRequest,
					w_value, w_index, w_length);
		if (ctrl->bRequestType != (USB_DIR_OUT |USB_TYPE_VENDOR|USB_RECIP_DEVICE))
		{
			DBG(gm, "VX_CSR_WRITE's bRequestType unexpected\n");
			goto unknown;
		}
		value = w_length;
		break;
		
	case VX_CSR_READ:
		dprintk(KERN_INFO "VX_CSR_READ control req (bRequestType=0x%02x bRequest=0x%02x\n"
				"value=0x%04x index=0x%04x length=%d)\n",
					ctrl->bRequestType, ctrl->bRequest,
					w_value, w_index, w_length);
		if (ctrl->bRequestType != (USB_DIR_IN|USB_TYPE_VENDOR|USB_RECIP_DEVICE))
		{
			DBG(gm, "VX_CSR_READ's bRequestType unexpected\n");
			goto unknown;
		}
		value = w_length;
		break;

	default:
unknown:
		dprintk(KERN_INFO "unknown control req (bRequestType=0x%02X bRequest=0x%02X\n"
				"value=0x%04X index=0x%04X length=%d)\n",
					ctrl->bRequestType, ctrl->bRequest,
					w_value, w_index, w_length);
	}

	/* respond with data transfer before status phase? */
	if (value >= 0) {
		req->length = value;
		req->zero = value < w_length;
		value = usb_ep_queue(gadget->ep0, req, GFP_ATOMIC);
		if (value < 0) {
			dprintk(KERN_INFO "ep_queue --> %d\n", value);
			req->status = 0;
			gmstream_setup_complete(gadget->ep0, req);
		}
	}
	else {
		dprintk(KERN_INFO "at end of gmstream_setup(), value is negative --> stalls\n");
	}

	/* device either stalls (value < 0) or reports success */
	return value;
}

/******************************************************************************
 * 
 ******************************************************************************/
static void gmstream_disconnect(struct usb_gadget *gadget)
{
	struct gmstream *gm = get_gadget_data(gadget);
	unsigned long flags;

	dprintk("%s()\n", __FUNCTION__);

	spin_lock_irqsave(&gm->lock, flags);
	gmstream_reset_config(gm);

	// a more significant application might have some non-usb
	// activities to quiesce here, saving resources like power
	// or pushing the notification up a network stack.
	spin_unlock_irqrestore(&gm->lock, flags);

	// next we may get setup() calls to enumerate new connections;
	// or an unbind() during shutdown (including removing module).
}

/******************************************************************************
 * this function is installed by gmstream_bind()
 ******************************************************************************/
static void gmstream_autoresume(unsigned long _gm)
{
	struct gmstream *gm = (struct gmstream *) _gm;
	int status;

	dprintk("%s()\n", __FUNCTION__);

	// normally the host would be woken up for something
	// more significant than just a timer firing...
	if (gm->gadget->speed != USB_SPEED_UNKNOWN) {
		status = usb_gadget_wakeup(gm->gadget);
		DBG(gm, "wakeup --> %d\n", status);
	}
}

/******************************************************************************
 * may be invoked indirectly during driver __init and __exit
 ******************************************************************************/
static void gmstream_unbind(struct usb_gadget *gadget)
{
	struct gmstream *gm = get_gadget_data(gadget);

	dprintk("%s()\n", __FUNCTION__);

	/* we've already been disconnected ... no i/o is active */
	if (gm->req) {
		gm->req->length = USB_BUFSIZ;
		free_ep_req(gadget->ep0, gm->req);
	}

	if (0 != pool_exit(&ep1_in_set, ep1_in_set.ep)) {
		printk(KERN_ERR "pool_exit error on %s\n", ep1_in_set.ep->name);
	}

	if (0 != pool_exit(&ep2_in_set, ep2_in_set.ep)) {
		printk(KERN_ERR "pool_exit error on %s\n", ep2_in_set.ep->name);
	}
	
	del_timer_sync(&gm->resume);
	kfree(gm);
	set_gadget_data(gadget, NULL);
}

/******************************************************************************
 *
 ******************************************************************************/
static int __init gmstream_bind(struct usb_gadget *gadget)
{
	struct gmstream *gm;
	struct usb_ep  *ep;
	int gcnum;

	dprintk("%s()\n", __FUNCTION__);

	if (gadget_is_sh(gadget))
		return -ENODEV;

	usb_ep_autoconfig_reset(gadget);
	//usb_ep_listall(gadget);

	dprintk("configure ep1 in\n");
	ep = usb_ep_config(gadget, ep1_in_set.ep_desc, "ep1in");
	if (!ep) 
		goto epconf_fail;
	ep1_in_set.ep_name = ep->name;		// assign a str name for this EP
	ep->driver_data = ep;				// claim
	if (0 != pool_init(&ep1_in_set, ep)) {
		printk(KERN_ERR "pool_init error on %s\n", ep->name);
		goto epconf_fail;
	}

	dprintk("configure ep1 out\n");
	ep = usb_ep_config(gadget, ep1_out_set.ep_desc, "ep1out");
	if (!ep)
		goto epconf_fail;
	ep1_out_set.ep_name = ep->name;		// assign a str name for this EP
	ep->driver_data = ep;				// claim

	dprintk("configure ep2 in\n");
	ep = usb_ep_config(gadget, ep2_in_set.ep_desc, "ep2in");
	if (!ep)
		goto epconf_fail;
	ep2_in_set.ep_name = ep->name;		// assign a str name for this EP
	ep->driver_data = ep;				// claim
	if (0 != pool_init(&ep2_in_set, ep)) {
		printk(KERN_ERR "pool_init error on %s\n", ep->name);
		goto epconf_fail;
	}
	
	dprintk("configure ep2 out\n");
	ep = usb_ep_config(gadget, ep2_out_set.ep_desc, "ep2out");
	if (!ep)
		goto epconf_fail;
	ep2_out_set.ep_name = ep->name;	// assign a str name for this EP
	ep->driver_data = ep;		// claim

	dprintk("configure ep3 in\n");
//	ep = usb_ep_config(gadget, &ep3_in_desc, "ep3in");
//	if (!ep)
//		goto epconf_fail;
//	ep3_in_set.ep_name = ep->name;		// assign a str name for this EP
//	ep->driver_data = ep;				// claim
//	if (0 != pool_init(&ep3_in_set, ep)) {
//		printk(KERN_ERR "pool_init error on %s\n", ep->name);
//		goto epconf_fail;
//	}

	dprintk("configure ep3 out\n");
	ep = usb_ep_config(gadget, &ep3_out_desc, "ep3out");
	if (!ep)
		goto epconf_fail;
	ep3_out_set.ep_name = ep->name;		// assign a str name for this EP
	ep->driver_data = ep;				// claim

#if 0 
	// hardware configuration - ep not available
	dprintk("configure ep4 in\n");
	ep = usb_ep_config(gadget, &ep4_in_desc, "ep4in");
	if (!ep)
		goto epconf_fail;
	ep4_in_set.ep_name = ep->name;		// assign a str name for this EP
	ep->driver_data = ep;				// claim
	if (0 != pool_init(&ep4_in_set, ep)) {
		printk(KERN_ERR "pool_init error on %s\n", ep->name);
		goto epconf_fail;
	}
#endif

	dprintk("configure ep5 out\n");
	ep = usb_ep_config(gadget, &ep5_out_desc, "ep5out");
	if (!ep)
		goto epconf_fail;
	ep5_out_set.ep_name = ep->name;	// assign a str name for this EP
	ep->driver_data = ep;		// claim


	// look up in gadget_chips.h the list of known pheripherals for the id code.
	// If not found, then we get an error code.  gadget->name is 'dwc_otg_pcd'

	gcnum = usb_gadget_controller_number(gadget);

	if (gcnum >= 0)
		device_desc.bcdDevice = cpu_to_le16(0x0200 + gcnum);
	else 
	{
		/* gadget merlin is so simple (for now, no altsettings) that
		 * it SHOULD NOT have problems with bulk-capable hardware.
		 * so warn about unrcognized controllers, don't panic.
		 *
		 * things like configuration and altsetting numbering
		 * can need hardware-specific attention though.
		 */
		printk(KERN_WARNING "%s: controller '%s' not recognized\n", driver_name, gadget->name);
		device_desc.bcdDevice = __constant_cpu_to_le16(0x9999);
	}

	/* ok, we made sense of the hardware ... */
	gm = kzalloc(sizeof(*gm), GFP_KERNEL);
	//same as: kmalloc(sizeof(*gm), SLAB_KERNEL);  if (gm) memset(gm, 0, sizeof(*gm));

	if (!gm)
		return -ENOMEM;

	spin_lock_init(&gm->lock);
	gm->gadget = gadget;
	set_gadget_data(gadget, gm);

	/* preallocate control response and buffer */
	gm->req = usb_ep_alloc_request(gadget->ep0, GFP_KERNEL);
	if (!gm->req)
		goto enomem;
	gm->req->buf = usb_ep_alloc_buffer(gadget->ep0, USB_BUFSIZ, &gm->req->dma, GFP_KERNEL);
	if (!gm->req->buf)
		goto enomem;

	gm->req->complete = gmstream_setup_complete;

	device_desc.bMaxPacketSize0 = gadget->ep0->maxpacket;	// 64

//#ifdef CONFIG_USB_GADGET_DUALSPEED
	/* assume ep0 uses the same value for both speeds ... */
	dev_qualifier.bMaxPacketSize0 = device_desc.bMaxPacketSize0;

//	/* and that all endpoints are dual-speed */
//	ephs_source.bEndpointAddress = epfs_source.bEndpointAddress;
//	ephs_sink.bEndpointAddress   = epfs_sink.bEndpointAddress;
//#endif

	if (gadget->is_otg) {
		otg_descriptor.bmAttributes |= USB_OTG_HNP,
		config_desc_main.bmAttributes |= USB_CONFIG_ATT_WAKEUP;
	}

	usb_gadget_set_selfpowered(gadget);

	init_timer(&gm->resume);
	gm->resume.function = gmstream_autoresume;
	gm->resume.data = (unsigned long)gm;
	if (autoresume) {
		config_desc_main.bmAttributes |= USB_CONFIG_ATT_WAKEUP;
	}

	gadget->ep0->driver_data = gm;

	INFO(gm, "Product function = %s\n", usb_str_product);
	INFO(gm, "gadget->name = %s\n", gadget->name);
	INFO(gm, "EP1 IN=%s OUT=%s\n", ep1_in_set.ep_name, ep1_out_set.ep_name);
	INFO(gm, "EP2 IN=%s OUT=%s\n", ep2_in_set.ep_name, ep2_out_set.ep_name);
	INFO(gm, "EP3 OUT=%s\n", 	   ep3_out_set.ep_name);
	if (ep4_in_set.ep_name)
		INFO(gm, "EP4 IN=%s\n",    ep4_in_set.ep_name);
	INFO(gm, "EP5 OUT=%s\n",       ep5_out_set.ep_name);

	snprintf(usb_str_manufacturer, sizeof(usb_str_manufacturer), "Mobilygen");

	return 0;

epconf_fail:
		printk(KERN_ERR "%s: can't configure EP on %s\n", driver_name, gadget->name);
		return -ENODEV;

enomem:
	gmstream_unbind(gadget);
	return -ENOMEM;
}

/******************************************************************************
 *
 ******************************************************************************/
static void gmstream_suspend(struct usb_gadget *gadget)
{
	struct gmstream *gm = get_gadget_data(gadget);

	dprintk("%s()\n", __FUNCTION__);

	if (gadget->speed == USB_SPEED_UNKNOWN)
		return;

	if (autoresume) {
		mod_timer(&gm->resume, jiffies + (HZ * autoresume));
		DBG(gm, "suspend, wakeup in %d seconds\n", autoresume);
	} 
	else
		DBG(gm, "suspend\n");
}

/******************************************************************************
 *
 ******************************************************************************/
static void gmstream_resume(struct usb_gadget *gadget)
{
	struct gmstream *gm = get_gadget_data(gadget);

	dprintk("%s()\n", __FUNCTION__);

	DBG(gm, "resume\n");
	del_timer(&gm->resume);
}


/******************************************************************************
 * used for driver registration in driver __init() and __exit()
 ******************************************************************************/
static struct usb_gadget_driver gmstream_driver = {
#ifdef CONFIG_USB_GADGET_DUALSPEED
	.speed		= USB_SPEED_HIGH,	// dual speed (full and high)
#else
	.speed		= USB_SPEED_FULL,	// full speed only
#endif
	.function	= (char*)usb_str_product,
	
	.bind		= gmstream_bind,
	.unbind		= __exit_p(gmstream_unbind),

	.setup		= gmstream_setup,
	.disconnect	= gmstream_disconnect,

	.suspend	= gmstream_suspend,
	.resume		= gmstream_resume,

	.driver 	= {
		.name		= (char*)driver_name,
		.owner		= THIS_MODULE,
	},
};

/******************************************************************************
 * close() for /dev/musb.N
 ******************************************************************************/
static int musb_close(struct inode *inode, struct file *filp)
{
	struct open_pdata *odata;
	struct musbdevice *mdev;
	
	dprintk(KERN_INFO "musb_close()\n");
	odata = (struct open_pdata*)filp->private_data;
	mdev  = (struct musbdevice*)odata->mdev;
//	down(&mdev->sem);

	// free the kernel space buffer used by read() write() for this cs device
	kfree(odata); //free the container that was holding the mdev

//	up(&mdev->sem);
	return 0;
}

/******************************************************************************
 * seek() for /dev/musb.N 
 ******************************************************************************/
static loff_t musb_llseek(struct file *filp, loff_t offset, int origin)
{
	struct open_pdata *odata;
	struct musbdevice *mdev;
	loff_t newpos;
	
	odata = (struct open_pdata*)filp->private_data;
	mdev  = (struct musbdevice*)odata->mdev;
//	down(&mdev->sem);

	switch (origin)	{
	case 0: /* SEEK_SET */
		newpos = offset;
		break;
	case 1: /* SEEK_CUR */
		newpos = filp->f_pos + offset;
		break;
	default: 
		return -EINVAL;
	}

	if (newpos < 0) {
		printk(KERN_ERR "Error: file seek out of bound\n");		
//		up(&mdev->sem);
	 	return -EINVAL;
	}
	
	filp->f_pos = newpos;

//	up(&mdev->sem);
	return newpos;
}

/******************************************************************************
 * ioctl for /dev/musb.N 
 ******************************************************************************/
static int musb_ioctl(struct inode *inode, struct file *filp,
					  unsigned int cmd, unsigned long arg)
{
	int retval = 0;
	int cs;

	cs = MINOR(inode->i_rdev);
	dprintk(KERN_INFO "ioctl device number = %d\n", cs);

	return retval;
}

/******************************************************************************
 * open() for /dev/musb.N
 ******************************************************************************/
static int musb_open(struct inode *inode, struct file *filp)
{
	int retval=0;
	int cs;
	struct musbdevice *mdev;
	struct merlin_usb_data *cs_data;
	struct open_pdata *odata;

	cs = MINOR(inode->i_rdev);
	//dprintk(KERN_INFO "device number cs = %d\n", cs);

	if (cs >= MUSB_MAX_DEVS)
		return -EINVAL;
		
	// tell read/write/ioctl the address of this device's data
	mdev = (struct musbdevice*)devlist[cs];
//	down(&mdev->sem);						// exclusive usage
	
	cs_data = mdev->dev->platform_data;		// to access (cs_data->endian)
	filp->f_pos = 0;

	//--------------------------------------------------------------------------
	// allocate an open_private_data struct for this open instance 
	//--------------------------------------------------------------------------
	odata = kmalloc(sizeof(struct open_pdata), GFP_KERNEL);
	if (odata == NULL) {
		printk(KERN_ERR "kmalloc failed\n");
		retval = -ENOMEM;
		goto exit;
	}
	
	odata->mdev = mdev;						// put mdev "inside" odata
//	odata->endian = cs_data->endian;		// copy platform driver properties
	filp->private_data = odata;				// odata is our main container 
	
exit:
//	up(&mdev->sem);
	return retval;
}


/******************************************************************************
 * user access functions
 ******************************************************************************/
static const struct file_operations musb_fops = {
	.owner		= THIS_MODULE,
	.llseek		= musb_llseek,
	.ioctl		= musb_ioctl,
	.open		= musb_open,
	.release	= musb_close,
	.write		= musb_write,
	.read		= musb_read,
	.sendpage	= musb_sendpage,
};

/******************************************************************************
 * this is invoked upon rmmod 
 ******************************************************************************/
static int musb_remove(struct platform_device *pdev)
{
	// do the equivalent of: misc_deregister(devlist[pdev->id])
	if (devlist[pdev->id] != NULL)
	{
		struct musbdevice * d;
		d = devlist[pdev->id];	
		cdev_del(&d->cdev);
		class_device_destroy(musb_class, MKDEV(musb_major, d->minor));
		printk(KERN_INFO "/dev/%s\n", d->name);
		kfree(devlist[pdev->id]);
		devlist[pdev->id] = NULL;
	}
	return 0;
}

/******************************************************************************
 * Set up the char_dev structure for this device.
 ******************************************************************************/
static void musb_setup_cdev(struct musbdevice *mdev, 			 
							const struct file_operations *musb_fops) // input table
{
	int err; 
	dev_t devno = MKDEV(musb_major, mdev->minor); // minors start at 0

	cdev_init(&mdev->cdev, musb_fops);
	mdev->cdev.owner = THIS_MODULE;
	mdev->cdev.ops = musb_fops;
	if ((err=cdev_add (&mdev->cdev, devno, 1)))
	{
		printk(KERN_NOTICE "Error %d adding musb at minor %d", err, mdev->minor);
	}

	printk(KERN_INFO "cdev(%d,%d) /dev/%s\n", musb_major, mdev->minor, mdev->name);

	// the last step is to connect our driver to the musb class 
	mdev->class = class_device_create(musb_class, NULL, devno, mdev->dev, "%s", mdev->name);
	if (IS_ERR(mdev->class)) {
		err = PTR_ERR(mdev->class);
		goto exit;
	}
	return;
exit:
	printk(KERN_ERR "musb_setup_cdev fail\n");
	return;
}

/******************************************************************************
 * musb_probe()
 * create 1 info structure, 2 musbdevice structures
 * setup 2 cdev's
 ******************************************************************************/
static int musb_probe(struct platform_device *pdev)
{
	struct merlin_usb_data *merlin_usb_data;
    struct musbdevice *mdev;	// data for setting up user file_operations
	int err;

	merlin_usb_data = pdev->dev.platform_data;
	if (merlin_usb_data == NULL)
	{
		printk("error: dev.platform_data is null\n");
		return -ENODEV;
	}

	// pdev->dev.bus_id is "musb.0", "musb.1", ..
	// pdev->id is the minor#
	// {
	// struct merlin_usb_data *cs_data = pdev->dev.platform_data;
	// dprintk(KERN_INFO "bus_width=%d\n", cs_data->width);
	// }

	if (pdev->id > MUSB_MAX_DEVS)	// chip select number too big?
	{
		err = -ENOMEM;
		goto err_out;
	}

	//=========================================================================
	// for each device, dynamically allocate the equivalent of this:
	// static struct miscdevice musb_miscdev = {
	//	.minor		= 0,			 
	//	.name		= "musb.0",
	//	.fops		= &musb_fops,
	// };

	// this device id entry already exist? 
	if (devlist[pdev->id]) {		
		err = -EBUSY;
		goto err_out;	
	}
	
	mdev = kmalloc(sizeof(struct musbdevice), GFP_KERNEL);
	if (mdev == NULL) {
		err = -ENOMEM;
		goto err_out;
	}
	memset(mdev, 0, sizeof(*mdev));
	mdev->dev   	= &pdev->dev;			// associate device with this driver 
	mdev->minor 	= (pdev->id)+0,			// one device number map to 1 minor number
	mdev->name  	= pdev->dev.bus_id;		// device name "musb.<id>"
	mdev->ep_number = pdev->id;
	//do the equivalent of: misc_register(mdev);
//	init_MUTEX(&mdev->sem);
	musb_setup_cdev(mdev, &musb_fops); 	// minor number to have a cdev
	
	devlist[pdev->id] = mdev;			// table of device structs, for remove()
	//=========================================================================

	return 0;

err_out:
	musb_remove(pdev);
	return err;
}

/******************************************************************************
 *
 ******************************************************************************/
static void musb_shutdown(struct platform_device *pdev)
{
	dprintk(KERN_INFO "gmstream shutdown\n");
}

/******************************************************************************
 *
 ******************************************************************************/
static int musb_suspend(struct platform_device *pdev, pm_message_t message)
{
	dprintk(KERN_INFO "gmstream suspend\n");
	return 0;
}

/******************************************************************************
 *
 ******************************************************************************/
static int musb_resume(struct platform_device *pdev)
{
	dprintk(KERN_INFO "gmstream resume\n");
	return 0;
}

/******************************************************************************
 *
 ******************************************************************************/
static struct platform_driver musb_driver = {
	.probe	    = musb_probe,
	.remove	    = musb_remove,
	.shutdown   = musb_shutdown,
	.suspend    = musb_suspend,
	.resume     = musb_resume,
	.driver	    = {
	    .name = MUSB_PLATFORM_NAME,
	},
};

/******************************************************************************
 * module initialization 
 ******************************************************************************/
static int __init gmstream_init(void)
{
	int i, err;
	dev_t dev = 0;

	// initialize the list of pointers to each device musbdev structs
	for (i=0; i<MUSB_MAX_DEVS; i++)
		devlist[i] = NULL;

	ep1_in_set.ep_waitq_flag = 0;
	ep1_out_set.ep_waitq_flag = 0;
	ep2_in_set.ep_waitq_flag = 0;
	ep2_out_set.ep_waitq_flag = 0;
//	ep3_in_set.ep_waitq_flag = 0;
	ep3_out_set.ep_waitq_flag = 0;
	ep4_in_set.ep_waitq_flag = 0;
	ep5_out_set.ep_waitq_flag = 0;

	init_waitqueue_head(&(ep1_in_set.ep_waitq));
	init_waitqueue_head(&(ep1_out_set.ep_waitq));
	init_waitqueue_head(&(ep2_in_set.ep_waitq));
	init_waitqueue_head(&(ep2_out_set.ep_waitq));
//	init_waitqueue_head(&(ep3_in_set.ep_waitq));
	init_waitqueue_head(&(ep3_out_set.ep_waitq));
	init_waitqueue_head(&(ep4_in_set.ep_waitq));
	init_waitqueue_head(&(ep5_out_set.ep_waitq));

	//-----------------------------------------------------------------------
	// create a class called 'musb'
	//-----------------------------------------------------------------------
	musb_class = class_create(THIS_MODULE, "musb");
	if (IS_ERR(musb_class)) {
		printk(KERN_INFO "musb_init class_create failed\n");  
		return PTR_ERR(musb_class);
	}
	//-----------------------------------------------------------------------
	// grab several minor numbers and one dynamically allocated major number 
	//-----------------------------------------------------------------------
	err = alloc_chrdev_region(&dev, 0, musb_nr_devs, "musb"); // 2nd param is minor #
	musb_major = MAJOR(dev);	// remember our allocated major number

	if (err < 0) {
		printk(KERN_WARNING "musb: can't get major %d\n", musb_major);
		class_destroy(musb_class);	// undo the created class 
		return err;
	}

	//-------------------------------------------------------
	// here starts the code for setting up platform_driver...
	//-------------------------------------------------------
	err = platform_driver_register(&musb_driver);
	// look for a match in name and execute probe

	//----------------------------------------------------------------------
	// a real value would likely come through some id prom or module option.  
	strlcpy(usb_str_serial_num, serial,	sizeof(usb_str_serial_num));

	return usb_gadget_register_driver(&gmstream_driver);
}

/******************************************************************************
 * module exit 
 ******************************************************************************/
static void __exit gmstream_exit(void)
{
	dev_t devno;
	
	dprintk("%s()\n", __FUNCTION__);
	usb_gadget_unregister_driver(&gmstream_driver);

	//-------------------------------------------------------
	// platform_driver to unregister itself, runs .REMOVE
	//-------------------------------------------------------
	platform_driver_unregister(&musb_driver);

	//-------------------------------------------------------
	// give back the several minor numbers we grabbed in init 
	//-------------------------------------------------------
	devno = MKDEV(musb_major, 0); // 2nd param is minor number
	// musb_exit() is never called if registering failed
	unregister_chrdev_region(devno, musb_nr_devs);

	//-------------------------------------------------------
	// remove the class we created
	//-------------------------------------------------------
	class_destroy(musb_class);	// undo the created class 
	
	ep1_in_set.ep_waitq_flag = -1;         // inform of our status to read, to wake it up 
	wake_up_interruptible((wait_queue_head_t*)&(ep1_in_set.ep_waitq));
	ep1_out_set.ep_waitq_flag = -1;         // inform of our status to read, to wake it up 
	wake_up_interruptible((wait_queue_head_t*)&(ep1_out_set.ep_waitq));

	ep2_in_set.ep_waitq_flag = -1;         // inform of our status to read, to wake it up 
	wake_up_interruptible((wait_queue_head_t*)&(ep2_in_set.ep_waitq));
	ep2_out_set.ep_waitq_flag = -1;         // inform of our status to read, to wake it up 
	wake_up_interruptible((wait_queue_head_t*)&(ep2_out_set.ep_waitq));

//	ep3_in_set.ep_waitq_flag = -1;         // inform of our status to read, to wake it up 
//	wake_up_interruptible((wait_queue_head_t*)&(ep3_in_set.ep_waitq));
	ep3_out_set.ep_waitq_flag = -1;         // inform of our status to read, to wake it up 
	wake_up_interruptible((wait_queue_head_t*)&(ep3_out_set.ep_waitq));

	ep4_in_set.ep_waitq_flag = -1;         // inform of our status to read, to wake it up 
	wake_up_interruptible((wait_queue_head_t*)&(ep4_in_set.ep_waitq));

	ep5_out_set.ep_waitq_flag = -1;         // inform of our status to read, to wake it up 
	wake_up_interruptible((wait_queue_head_t*)&(ep5_out_set.ep_waitq));
}		

/******************************************************************************/
module_init(gmstream_init);
module_exit(gmstream_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Mobilygen");
MODULE_DESCRIPTION("MG3500 usb gadget driver");

