/*
 * epconfig.c - custom config of synopsys end points for gadget drivers
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/device.h>

#include <linux/ctype.h>
#include <linux/string.h>

#include <linux/usb_ch9.h>
#include <linux/usb_gadget.h>

#include "gadget_chips.h"


/* we must assign addresses for configurable endpoints (like net2280) */
static __devinitdata unsigned epnum;

/*
 * This should work with endpoints from controller drivers sharing the
 * same endpoint naming convention.  By example:
 *
 *	- ep1, ep2, ... address is fixed, not direction or type
 *	- ep1in, ep2out, ... address and direction are fixed, not type
 *	- ep1-bulk, ep2-bulk, ... address and type are fixed, not direction
 *	- ep1in-bulk, ep2out-iso, ... all three are fixed
 *	- ep-* ... no functionality restrictions
 *
 * Type suffixes are "-bulk", "-iso", or "-int".  Numbers are decimal.
 * Less common restrictions are implied by gadget_is_*().
 *
 * NOTE:  each endpoint is unidirectional, as specified by its USB
 * descriptor; and isn't specific to a configuration or altsetting.
 */
static int __devinit ep_matches(struct usb_ep	  *ep,
								struct usb_gadget *gadget,
								struct usb_endpoint_descriptor *desc)
{
	u8		type;
	const char	*tmp;
	u16		max;

	/* endpoint already claimed? */
	if (0 != ep->driver_data)
		return 0;

	/* only support ep0 for portable CONTROL traffic */
	type = desc->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK;
	if (USB_ENDPOINT_XFER_CONTROL == type)
		return 0;

	/* some other naming convention */
	if ('e' != ep->name[0])
		return 0;

	/* type-restriction:  "-iso", "-bulk", or "-int".
	 * direction-restriction:  "in", "out".
	 */
	if ('-' != ep->name[2]) 
	{
		tmp = strrchr (ep->name, '-');
		if (tmp) 
		{
			switch (type) 
			{
			case USB_ENDPOINT_XFER_INT:
				/* bulk endpoints handle interrupt transfers,
				 * except the toggle-quirky iso-synch kind
				 */
				if ('s' == tmp[2])	// == "-iso"
					return 0;
				/* for now, avoid PXA "interrupt-in";
				 * it's documented as never using DATA1.
				 */
				if (gadget_is_pxa (gadget)
						&& 'i' == tmp [1])
					return 0;
				break;
			case USB_ENDPOINT_XFER_BULK:
				if ('b' != tmp[1])	// != "-bulk"
					return 0;
				break;
			case USB_ENDPOINT_XFER_ISOC:
				if ('s' != tmp[2])	// != "-iso"
					return 0;
			}
		} 
		else 
		{
			tmp = ep->name + strlen (ep->name);
		}

		/* direction-restriction:  "..in-..", "out-.." */
		tmp--;
		if (!isdigit (*tmp)) 
		{
			if (desc->bEndpointAddress & USB_DIR_IN) 
			{
				if ('n' != *tmp)
					return 0;
			} 
			else 
			{
				if ('t' != *tmp)
					return 0;
			}
		}
	}

	/* endpoint maxpacket size is an input parameter, except for bulk
	 * where it's an output parameter representing the full speed limit.
	 * the usb spec fixes high speed bulk maxpacket at 512 bytes.
	 */
	max = 0x7ff & le16_to_cpup (&desc->wMaxPacketSize);
	
	// here desc->wMaxPacketSize == 512
	
	switch (type) 
	{
	case USB_ENDPOINT_XFER_INT:
		/* INT:  limit 64 bytes full speed, 1024 high speed */
		if (!gadget->is_dualspeed && max > 64)
			return 0;
		/* FALLTHROUGH */

	case USB_ENDPOINT_XFER_ISOC:
		/* ISO:  limit 1023 bytes full speed, 1024 high speed */
		if (ep->maxpacket < max)
			return 0;
		if (!gadget->is_dualspeed && max > 1023)
			return 0;

		/* BOTH:  "high bandwidth" works only at high speed */
		if ((desc->wMaxPacketSize & __constant_cpu_to_le16(3<<11))) {
			if (!gadget->is_dualspeed)
				return 0;
			/* configure your hardware with enough buffering!! */
		}
		break;
	}

	/* MATCH!! */

	/* report address */
	if (isdigit(ep->name[2])) 
	{
		u8	num = simple_strtol (&ep->name [2], NULL, 10);
		desc->bEndpointAddress |= num;
	} 
	else 
	{
		if (++epnum > 15)
			return 0;
		desc->bEndpointAddress |= epnum;
	}

	/* report (variable) full speed bulk maxpacket */
	if (USB_ENDPOINT_XFER_BULK == type) 
	{
		int size = ep->maxpacket;

		  // here for some reason, ep->maxpacket == 64
		  // set it to the 512 byte length
		  size = ep->maxpacket = 512;

		if (!gadget->is_dualspeed)	// don't go below 512 if we are above full-speed
		{
			if (size > 64)
				size = 64;
		}
		desc->wMaxPacketSize = cpu_to_le16(size);
	}
	return 1;
}

#if 0
static struct usb_ep * __devinit
find_ep (struct usb_gadget *gadget, const char *name)
{
	struct usb_ep *ep;

	list_for_each_entry(ep, &gadget->ep_list, ep_list) 
	{
		if (0 == strcmp(ep->name, name))
			return ep;
	}
	return NULL;
}
#endif

/**
 * usb_ep_config - choose an endpoint matching the name of the 
 * endpoint and matching the descriptor definition
 * 
 */
extern struct usb_ep *usb_ep_config (struct usb_gadget *,
			struct usb_endpoint_descriptor *,
			const char *find_name) __devinit;
			
struct usb_ep * __devinit usb_ep_config(struct usb_gadget *gadget,
										struct usb_endpoint_descriptor *desc,
										const char *find_name)
{
	struct usb_ep	*ep;
	u8		type;

	type = desc->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK;

	if (0 == strcmp("ep3in", find_name))
	{
		printk(KERN_ERR "EP3IN is not available\n");
		return NULL;
	}

	/* look at endpoints until an unclaimed one looks usable */
	list_for_each_entry(ep, &gadget->ep_list, ep_list) 
	{
//		if (USB_ENDPOINT_XFER_INT == type) 
//		{
//			/* single buffering is enough */
//			ep = find_ep (gadget, "ep3-bulk");
//			if (ep && ep_matches (ep, gadget, desc))
//				return ep;
//		} 
//		else if (USB_ENDPOINT_XFER_BULK == type && (USB_DIR_IN & desc->bEndpointAddress)) 
//		{
//			/* DMA may be available */
//			ep = find_ep (gadget, "ep2-bulk");
//			if (ep && ep_matches (ep, gadget, desc))
//				return ep;
//		}
	
		if (0 == strcmp(ep->name, find_name))
		{
			//printk(KERN_INFO "EP:%s\n", find_name);  
			if (ep_matches (ep, gadget, desc))
				return ep;
		}
			
	}

	/* Fail */
	return NULL;  // no match found
}



/**
 *
 */
void __devinit usb_ep_listall(struct usb_gadget *gadget)
{
	struct usb_ep	*ep;

	list_for_each_entry (ep, &gadget->ep_list, ep_list) 
	{
		printk(KERN_INFO "End point: %s\n", ep->name);
		ep->driver_data = NULL;
	}
}


/**
 * usb_ep_autoconfig_reset - reset endpoint autoconfig state
 * @gadget: device for which autoconfig state will be reset
 *
 * Use this for devices where one configuration may need to assign
 * endpoint resources very differently from the next one.  It clears
 * state such as ep->driver_data and the record of assigned endpoints
 * used by usb_ep_autoconfig().
 */
void __devinit usb_ep_autoconfig_reset(struct usb_gadget *gadget)
{
	struct usb_ep	*ep;

	list_for_each_entry (ep, &gadget->ep_list, ep_list) {
		ep->driver_data = NULL;
	}
	epnum = 0;
}

