#define DRIVER_NAME 	"dwapbssi"
#define DRIVER_DESC 	"Driver for the DW_apb_ssi component"
#define DRIVER_AUTHOR 	"Gregoire Pean <gpean@mobilygen.com>"
#define DRIVER_VERSION 	"4:1.1"

/*
 *  This file Copyright (C) 2007 Mobilygen Corp.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE	LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/ctype.h>
#include <linux/version.h>
#include <linux/vermagic.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/spinlock.h>
#include <linux/workqueue.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/clk.h>
#include <linux/err.h>

#include <linux/amba/bus.h>
#include <linux/spi/spi.h>

#include <asm/atomic.h>

#include <linux/spi/dwapbssi.h>

#define USE_MOBI_DMA

#ifdef USE_MOBI_DMA
#include <mach/mobi_dma.h>
#endif

/*-------------------------------------------------------------------------*/

#define error(format, arg...)	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#define info(format, arg...) 	printk(KERN_INFO DRIVER_NAME ": " format "\n" , ## arg)
#define warn(format, arg...) 	printk(KERN_WARNING DRIVER_NAME ": " format "\n" , ## arg)

#ifdef DEBUG
#define debug(format, arg...) 	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#else
#define debug(format, arg...) 	do {} while (0)
#endif

/*-------------------------------------------------------------------------*/

struct dwapbssi_spi_device {
	struct list_head		list;
	struct spi_device		*dev;
	char				use_tx_dma, use_rx_dma;
	u8				cs_cache;
};

struct dmahelper_data {
#ifdef USE_MOBI_DMA
	mobi_dma_handle			dmah;
#endif
	char 				write;
	dma_addr_t 			mem_addr;
	int				size;
};

struct dwapbssi_drv_data {
	struct amba_device 		*dev;
	void __iomem 			*base;

#define DISABLED_HALT 	1
#define DISABLED_ERROR 	2
#define DISABLED_USER 	4
	atomic_t			disabled;
	
	struct mutex			access_lock;
	
	struct dwapbssi_bus_data	*bdata;
	struct clk 			*clock;
	
	struct spi_master		*master;
	struct list_head 		spi_devices;
	
	struct workqueue_struct 	*workqueue;
	
	char				initializing;
	
	atomic_t			completion_pending;
	wait_queue_head_t		wait_queue;
	struct spi_transfer 		*cur_tx_xfer, *cur_rx_xfer;
	volatile unsigned long		total_len, cur_tx, cur_rx, start_tx, start_rx;
	u8				last_fault_isr;
	spinlock_t			irq_lock;
	unsigned long			last_unsupported_speed;
	
	char 				is_slave;
	unsigned long 			force_speed;
	unsigned long			slave_timeout;
	
	atomic_t			ignore_rxfifoover;
	
	atomic_t			tx_dma_error, rx_dma_error;
	struct dmahelper_data 		*dma_tx, *dma_rx;
	u8				*dma_tx_buf, *dma_rx_buf;
	int				dma_tx_buf_order, dma_rx_buf_order;
};

struct dwapbssi_work_data {
	struct spi_device 		*spi;
	struct spi_message		*msg;
	struct work_struct		work;
};

static void				*work_data_cache;

/*-------------------------------------------------------------------------*/

static inline u32 apb_read32(void __iomem *base, int reg)
{
	return ioread32(base + reg);
}

static inline void apb_write32(void __iomem *base, int reg, u32 value)
{
	iowrite32(value, base + reg);
}

static inline u16 apb_read16(void __iomem *base, int reg)
{
	return ioread16(base + reg);
}

static inline void apb_write16(void __iomem *base, int reg, u16 value)
{
	iowrite16(value, base + reg);
}

static inline u8 apb_read8(void __iomem *base, int reg)
{
	return ioread8(base + reg);
}

static inline void apb_write8(void __iomem *base, int reg, u8 value)
{
	iowrite8(value, base + reg);
}

static inline struct dwapbssi_bus_data *masterdev_to_busdata(struct device *dev)
{
        return dev->platform_data;
}

static inline struct dwapbssi_drv_data *spidev_to_drvdata(struct spi_master *master)
{
	return *((struct dwapbssi_drv_data **)spi_master_get_devdata(master));
}

static u16 dwapbssi_get_baudrate(unsigned long req_hz, unsigned long clock_rate,
	unsigned long *effective_freq)
{
	unsigned long baudrate;
	if (unlikely(req_hz * DWAPBSSI_BAUDRATE_MIN >= clock_rate || !clock_rate)) {
		baudrate = DWAPBSSI_BAUDRATE_MIN;
	} else if (unlikely(!req_hz)) {
		baudrate = DWAPBSSI_BAUDRATE_MAX;
	} else {
		/*
		 * Since this is a divider, we round up the division
		 * to ensure we do not go above the requested frequency.
		 */
		baudrate = 1UL + (clock_rate - 1UL) / req_hz;
		if (baudrate & 1) /* value must be even */
			baudrate++;
		if (baudrate < DWAPBSSI_BAUDRATE_MIN)
			baudrate = DWAPBSSI_BAUDRATE_MIN;
		else if (baudrate > DWAPBSSI_BAUDRATE_MAX)
			baudrate = DWAPBSSI_BAUDRATE_MAX;
	}
	if (effective_freq)
		*effective_freq = clock_rate / baudrate;
	return baudrate;
}

static int dwapbssi_setup(struct spi_device *spi)
{
	struct dwapbssi_drv_data *ddata = spidev_to_drvdata(spi->master);
	
	if (ddata->initializing) {
		if (ddata->is_slave) {
			dev_info(&ddata->dev->dev, "new master: driver: %s\n", spi->modalias);
		} else {
			dev_info(&ddata->dev->dev, "new slave(%d): speed %lu%sHz, driver: %s\n",
				spi->chip_select, (unsigned long)(spi->max_speed_hz < 10000 ?
				spi->max_speed_hz : (spi->max_speed_hz / 1000)),
				spi->max_speed_hz < 10000 ? "" : "K", spi->modalias);
		}
	}
	
	return 0;
}

static int dwapbssi_wait_for_idle(struct dwapbssi_drv_data *ddata, unsigned long timeout)
{
	unsigned long comp_timeout_j;
	u8 status;
	status = apb_read8(ddata->base, DWAPBSSI_IC_SR);
	comp_timeout_j = jiffies + msecs_to_jiffies(timeout);
	while (status & DWAPBSSI_SR_BUSY) {
		if (unlikely(atomic_read(&ddata->disabled)))
			return -ECANCELED;
		msleep(0);
		if (unlikely(!time_before_eq(jiffies, comp_timeout_j)))
			return -ETIMEDOUT;
		status = apb_read8(ddata->base, DWAPBSSI_IC_SR);
	}
	return 0;
}

#ifdef USE_MOBI_DMA
static void dwapbssi_dma_event(mobi_dma_handle dmah, mobi_dma_event e, void *data)
{
	struct dwapbssi_drv_data *ddata = data;
	if (e == MOBI_DMA_EVENT_TRANSFER_COMPLETE) {
		if (ddata->dma_rx && dmah == ddata->dma_rx->dmah)
			ddata->cur_rx = mobi_dma_get_bytes(dmah);
		atomic_dec(&ddata->completion_pending);
		wake_up_interruptible(&ddata->wait_queue);
	} else if (e == MOBI_DMA_EVENT_TRANSFER_ERROR) {
		if (ddata->dma_tx && dmah == ddata->dma_tx->dmah) {
			ddata->cur_tx = mobi_dma_get_bytes(dmah);
			atomic_set(&ddata->tx_dma_error, 1);
		} else {
			ddata->cur_rx = mobi_dma_get_bytes(dmah);
			atomic_set(&ddata->rx_dma_error, 1);
		}
		atomic_set(&ddata->completion_pending, 0);
		wake_up_interruptible(&ddata->wait_queue);
	}
}
#endif

static void dmahelper_update_cur_bytes(struct dwapbssi_drv_data *ddata)
{
#ifdef USE_MOBI_DMA
	if (ddata->dma_tx)
		ddata->cur_tx = mobi_dma_get_bytes(ddata->dma_tx->dmah);
	if (ddata->dma_rx)
		ddata->cur_rx = mobi_dma_get_bytes(ddata->dma_rx->dmah);
#endif
}

static struct dmahelper_data *dmahelper_setup(struct dwapbssi_drv_data *ddata,
	const char *dma_id, int write, u8 *bytes, int size, dma_addr_t ip_addr)
{
#ifdef USE_MOBI_DMA
	int ret, ip_flags, mem_flags;
	struct dmahelper_data *hdata = kzalloc(sizeof(struct dmahelper_data), GFP_KERNEL);
	if (unlikely(!hdata))
		return NULL;
	
	hdata->size = size;
	hdata->write = write;
	
	hdata->dmah = mobi_dma_request(dma_id, MOBI_DMA_O_NONE);
	if (unlikely(hdata->dmah < 0))
		goto err_request;

	ret = mobi_dma_setup_handler(hdata->dmah, dwapbssi_dma_event, ddata);
	if (unlikely(ret))
		goto err_handler;
	
	ip_flags = mem_flags = 0;
	
	mem_flags |= MOBI_DMA_CONFIG_ADDRADJ_INC;
	mem_flags |= MOBI_DMA_CONFIG_BURST_SIZE_8;
	mem_flags |= MOBI_DMA_CONFIG_TRANSFER_WIDTH_32;
	
	ip_flags |= MOBI_DMA_CONFIG_ADDRADJ_NONE;
	ip_flags |= MOBI_DMA_CONFIG_BURST_SIZE_4;
	ip_flags |= MOBI_DMA_CONFIG_TRANSFER_WIDTH_8;

        if (unlikely(mobi_dma_config(hdata->dmah, DMA_CONFIG_DST, write ? ip_flags : mem_flags, NULL)
		|| mobi_dma_config(hdata->dmah, DMA_CONFIG_SRC, write ? mem_flags : ip_flags, NULL)
		|| mobi_dma_config(hdata->dmah, DMA_CONFIG_XFER, MOBI_DMA_CONFIG_DATA_WIDTH_8, NULL)))
		goto err_config;
	
	hdata->mem_addr = dma_map_single(NULL, (void *)bytes, size, write ? DMA_TO_DEVICE : DMA_FROM_DEVICE);
	if (unlikely(dma_mapping_error(NULL, hdata->mem_addr)))
		goto err_map;
	
	if (unlikely(mobi_dma_setup_single(hdata->dmah,
		(uint32_t)(write ? hdata->mem_addr : ip_addr),
		(uint32_t)(write ? ip_addr : hdata->mem_addr),
		size)))
		goto err_setup;
	
	return hdata;

err_setup:
	dma_unmap_single(NULL, hdata->mem_addr, hdata->size, hdata->write ? DMA_TO_DEVICE : DMA_FROM_DEVICE);
err_map:
err_config:
err_handler:
    	mobi_dma_free(hdata->dmah);
err_request:
    	kfree(hdata);
#endif
	return NULL;
}

static inline int dmahelper_start(struct dmahelper_data *hdata)
{
#ifdef USE_MOBI_DMA
	return mobi_dma_enable(hdata->dmah);
#endif
	return 0;
}

static void dmahelper_free(struct dmahelper_data *hdata)
{
#ifdef USE_MOBI_DMA
	dma_unmap_single(NULL, hdata->mem_addr, hdata->size, hdata->write ? DMA_TO_DEVICE : DMA_FROM_DEVICE);
	mobi_dma_disable(hdata->dmah);
	mobi_dma_free(hdata->dmah);
	kfree(hdata);
#endif
}

static inline u16 dwapbssi_write_next_chunk(struct dwapbssi_drv_data *ddata, int simulate)
{
	u16	data;
	char	*bdata = (char*)&data;
	
	while (ddata->cur_tx - ddata->start_tx >= ddata->cur_tx_xfer->len) {
		ddata->start_tx = ddata->cur_tx;
		ddata->cur_tx_xfer = list_entry(ddata->cur_tx_xfer->transfer_list.next,
			struct spi_transfer, transfer_list);
	}
	bdata[0] = (ddata->cur_tx_xfer->tx_buf != NULL) ?
		((u8*)ddata->cur_tx_xfer->tx_buf)[ddata->cur_tx - ddata->start_tx] : 0;
	bdata[1] = 0;
	
	if (!simulate)
		apb_write16(ddata->base, DWAPBSSI_IC_DR, data);
	ddata->cur_tx++;
	
	return data;
}

static inline u16 dwapbssi_read_next_chunk(struct dwapbssi_drv_data *ddata, int simulate, u16 data)
{
	char	*bdata = (char*)&data;
	
	if (!simulate)
		data = apb_read16(ddata->base, DWAPBSSI_IC_DR);
	
	while (ddata->cur_rx - ddata->start_rx >= ddata->cur_rx_xfer->len) {
		ddata->start_rx = ddata->cur_rx;
		ddata->cur_rx_xfer = list_entry(ddata->cur_rx_xfer->transfer_list.next,
			struct spi_transfer, transfer_list);
	}
	if (ddata->cur_rx_xfer->rx_buf != NULL)
		((u8*)ddata->cur_rx_xfer->rx_buf)[ddata->cur_rx - ddata->start_rx] = bdata[0];
	ddata->cur_rx++;
	
	return data;
}

static inline void dwapbssi_xfer_reset(struct dwapbssi_drv_data *ddata)
{
	atomic_set(&ddata->completion_pending, 0);
	atomic_set(&ddata->tx_dma_error, 0);
	atomic_set(&ddata->rx_dma_error, 0);
	ddata->cur_tx_xfer = NULL;
	ddata->cur_rx_xfer = NULL;
	ddata->cur_tx = 0;
	ddata->cur_rx = 0;
	ddata->start_tx = 0;
	ddata->start_rx = 0;
	ddata->total_len = 0;
	ddata->dma_tx = NULL;
	ddata->dma_rx = NULL;
	ddata->last_fault_isr = 0;
}

static void dwapbssi_access(struct work_struct *work)
{
	struct dwapbssi_work_data 	*wdata = container_of(work, struct dwapbssi_work_data, work);
	struct spi_device 		*spi = wdata->spi;
	struct dwapbssi_spi_device 	*spi_d;
	struct dwapbssi_drv_data 	*ddata = spidev_to_drvdata(spi->master);
	struct spi_message 		*msg = wdata->msg;
	struct spi_transfer 		*xfer;
	unsigned long 			effective_hz = 0, timeout = 0, tx_speed = 0;
	int				ret = 0, tx_only = 1, rx_only = 1, do_spinlock = 0;
	const char 			*stop_reason;
	
	if (unlikely(!work || !wdata || !ddata
		|| atomic_read(&ddata->disabled) 
		|| mutex_lock_interruptible(&ddata->access_lock)))
		goto end_complete;
	
	list_for_each_entry(spi_d, &ddata->spi_devices, list)
		if (spi_d->cs_cache == spi->chip_select)
			break;
	
	dwapbssi_xfer_reset(ddata);
	ddata->cur_tx_xfer = list_entry(msg->transfers.next, struct spi_transfer, transfer_list);
	ddata->cur_rx_xfer = ddata->cur_tx_xfer;
	
	list_for_each_entry(xfer, &msg->transfers, transfer_list) {
		if (xfer->speed_hz && !tx_speed)
			tx_speed = xfer->speed_hz;
		ddata->total_len += xfer->len;
		if (xfer->len) {
			if (xfer->tx_buf && rx_only)
				rx_only = 0;
			if (xfer->rx_buf && tx_only)
				tx_only = 0;
		}
	}
	
	if (unlikely(!ddata->total_len)) {
		mutex_unlock(&ddata->access_lock);
		goto end_complete;
	}
	
	apb_write8(ddata->base, DWAPBSSI_IC_SSIENR, 0);
    	apb_write16(ddata->base, DWAPBSSI_IC_CTRLR0,
		0x7 				/* data size in bits count - 1 */
		| ((spi->mode & 3) << 6) 	/* clock phase and polarity */
		| (tx_only << 8)
		| (rx_only << 9)
	);
	
	apb_write32(ddata->base, DWAPBSSI_IC_TXFTLR, 7);
	apb_write32(ddata->base, DWAPBSSI_IC_RXFTLR, 0);
	
	if (spi_d->use_tx_dma || spi_d->use_rx_dma) {
		
		int i, this_order = get_order(ddata->total_len);
		
		if (spi_d->use_tx_dma && !rx_only) {
			if (ddata->dma_tx_buf && this_order > ddata->dma_tx_buf_order) {
				free_pages((unsigned long)ddata->dma_tx_buf, ddata->dma_tx_buf_order);
				ddata->dma_tx_buf = NULL;
			}
			if (!ddata->dma_tx_buf) {
				ddata->dma_tx_buf = (u8 *)__get_dma_pages(GFP_KERNEL, this_order);
				ddata->dma_tx_buf_order = ddata->dma_tx_buf ? this_order : 0;
			}
			if (ddata->dma_tx_buf) {
				for (i=0; i<ddata->total_len; i++)
					ddata->dma_tx_buf[i] = 0xff & dwapbssi_write_next_chunk(ddata, 1);
				ddata->dma_tx = dmahelper_setup(ddata, DRIVER_NAME "_tx", 1,
					ddata->dma_tx_buf, ddata->total_len,
					ddata->dev->res.start + DWAPBSSI_IC_DR);
			}
			if (!ddata->dma_tx) {
				dev_err(&ddata->dev->dev, "%s: could not setup DMA for TX, "
					"cancelling transfer\n", __func__);
				atomic_set(&ddata->disabled, atomic_read(&ddata->disabled) | DISABLED_ERROR);
				goto end_disable;
			}
		}
		
		if (spi_d->use_rx_dma && !tx_only) {
			if (ddata->dma_rx_buf && this_order > ddata->dma_rx_buf_order) {
				free_pages((unsigned long)ddata->dma_rx_buf, ddata->dma_rx_buf_order);
				ddata->dma_rx_buf = NULL;
			}
			if (!ddata->dma_rx_buf) {
				ddata->dma_rx_buf = (u8 *)__get_dma_pages(GFP_KERNEL, this_order);
				ddata->dma_rx_buf_order = ddata->dma_rx_buf ? this_order : 0;
			}
			if (ddata->dma_rx_buf) {
				memset(ddata->dma_rx_buf, 0, ddata->total_len);
				ddata->dma_rx = dmahelper_setup(ddata, DRIVER_NAME "_rx", 0,
					ddata->dma_rx_buf, ddata->total_len,
					ddata->dev->res.start + DWAPBSSI_IC_DR);
			}
			if (!ddata->dma_rx) {
				if (ddata->dma_tx)
					dmahelper_free(ddata->dma_tx);
				atomic_set(&ddata->disabled, atomic_read(&ddata->disabled) | DISABLED_ERROR);
				dev_err(&ddata->dev->dev, "%s: could not setup DMA for RX, "
					"cancelling transfer\n", __func__);
				goto end_disable;
			}
		}
	}
	
	if (ddata->is_slave) {
		timeout = ddata->slave_timeout;
	} else {
		unsigned long orig_req_hz, req_hz = ddata->force_speed;
		unsigned long clock_rate = clk_get_rate(ddata->clock);
		if (!req_hz) {
			if (tx_speed && tx_speed < spi->max_speed_hz)
				req_hz = tx_speed;
			else
				req_hz = spi->max_speed_hz;
		}
		if (!req_hz) {
			dev_err(&ddata->dev->dev, "%s: no speed specified for transfer\n", __func__);
			goto end_disable;
		}
		orig_req_hz = req_hz;
		if (req_hz > (clock_rate >> 1))
			/* Cap to max theorical master mode speed. */
			req_hz = clock_rate >> 1;
		if ((ddata->dma_tx && ddata->dma_rx) 
			|| (ddata->dma_tx && tx_only) 
			|| (ddata->dma_rx && rx_only)) {
			/* Allow max theorical speed. */
		} else if (ddata->dma_tx || ddata->dma_rx) {
			if (req_hz > DWAPBSSI_MAX_FREQUENCY_1DMA)
				req_hz = DWAPBSSI_MAX_FREQUENCY_1DMA;
		} else if (req_hz > DWAPBSSI_MAX_FREQUENCY_NODMA) {
			do_spinlock = 1;
			if (req_hz > DWAPBSSI_MAX_FREQUENCY_NODMA_SPIN)
				req_hz = DWAPBSSI_MAX_FREQUENCY_NODMA_SPIN;
		}
		apb_write16(ddata->base, DWAPBSSI_IC_BAUDR, dwapbssi_get_baudrate(req_hz,
			clock_rate, &effective_hz));
		if ((orig_req_hz >> 9) != (effective_hz >> 9)) {
			if (ddata->last_unsupported_speed != orig_req_hz) {
				dev_warn(&ddata->dev->dev, "%s: WARNING: speed %luKHz cannot be reached "
					"with current settings, will use %luKHz instead\n", 
					__func__, orig_req_hz / 1000, effective_hz / 1000);
				ddata->last_unsupported_speed = orig_req_hz;
			}
		}
		timeout = (ddata->total_len * 10000) / effective_hz;
		if (timeout < 100)
			timeout = 100;
		
		apb_write32(ddata->base, DWAPBSSI_IC_SER, 1 << spi->chip_select);
		if (rx_only)
			apb_write16(ddata->base, DWAPBSSI_IC_CTRLR1, ddata->total_len - 1);
	}
	
	if (do_spinlock) {
		apb_write8(ddata->base, DWAPBSSI_IC_IMR, 0);
		apb_write8(ddata->base, DWAPBSSI_IC_DMACR, 0);
		/* TX only mode is found to be buggy using this mode. 
		 * Just do duplex transfers everytime, the overhead is very small. */
		tx_only = rx_only = 0;
		apb_write16(ddata->base, DWAPBSSI_IC_CTRLR0, 
			apb_read16(ddata->base, DWAPBSSI_IC_CTRLR0) & ~(0x3 << 8));
	} else {
		atomic_set(&ddata->completion_pending, (tx_only || rx_only) ? 1 : 2);
		apb_write8(ddata->base, DWAPBSSI_IC_DMACR, (ddata->dma_tx ? DWAPBSSI_DMACR_TDMAE : 0)
			| (ddata->dma_rx ? DWAPBSSI_DMACR_RDMAE : 0));
		apb_write8(ddata->base, DWAPBSSI_IC_IMR, 
			(atomic_read(&ddata->ignore_rxfifoover) ?  0 : DWAPBSSI_IR_RXFIFO_OVERFLOW)
			| DWAPBSSI_IR_RXFIFO_UNDERFLOW | DWAPBSSI_IR_TXFIFO_OVERFLOW
			| ((ddata->dma_tx || rx_only) ? 0 : DWAPBSSI_IR_TXFIFO_EMPTY)
			| ((ddata->dma_rx || tx_only) ? 0 : DWAPBSSI_IR_RXFIFO_FULL));
	}
	
	if (ddata->dma_tx)
		/* DMATDLR = DWAPBSSI_TXFIFO_DEPTH - DMA burst size */
		apb_write32(ddata->base, DWAPBSSI_IC_DMATDLR, DWAPBSSI_TXFIFO_DEPTH - 4);
	if (ddata->dma_rx)
		/* DMARDLR = DMA burst size - 1 */
		apb_write32(ddata->base, DWAPBSSI_IC_DMARDLR, 3);
	
	debug("START cs=%d len=%lu speed=%luKHz tx_only=%d rx_only=%d DMA:TX/RX=%d/%d->%d/%d SPIN=%s", 
		spi_d->cs_cache, ddata->total_len, effective_hz / 1000, 
		tx_only, rx_only, spi_d->use_tx_dma, spi_d->use_rx_dma,
		ddata->dma_tx != 0, ddata->dma_rx != 0, do_spinlock ? "YES" : "NO");
	
	/* If we are a master, enable first. DMA has to be started after 
	 * component is enabled in this case. */
	if (!ddata->is_slave)
		apb_write8(ddata->base, DWAPBSSI_IC_SSIENR, 1);
	
	if (do_spinlock) {
		u8 sr, busy = 0;
		unsigned long flags;
		ret = 1;
		spin_lock_irqsave(&ddata->irq_lock, flags);
		while (ddata->cur_tx < ddata->total_len && ddata->cur_tx < 4)
			dwapbssi_write_next_chunk(ddata, 0);
		while (ddata->cur_rx < ddata->total_len) {
			sr = apb_read8(ddata->base, DWAPBSSI_IC_SR);
			if (sr & DWAPBSSI_SR_RXFIFO_NOTEMPTY)
				dwapbssi_read_next_chunk(ddata, 0, 0);
			if (ddata->cur_tx < ddata->total_len 
				&& (sr & DWAPBSSI_SR_TXFIFO_NOTFULL))
				dwapbssi_write_next_chunk(ddata, 0);
			if (!busy && (sr & DWAPBSSI_SR_BUSY)) {
				busy = 1;
			} else if (busy == 1 && !(sr & DWAPBSSI_SR_BUSY)) {
				busy = 2;
			}
			if (busy == 2 && !apb_read8(ddata->base, DWAPBSSI_IC_RXFLR) 
				&& ddata->cur_rx < ddata->total_len) {
				debug("--- TXRX BUSY BREAK ---");
				ret = -EINTR;
				break;
			}
			if (unlikely(apb_read8(ddata->base, DWAPBSSI_IC_RISR) 
				& (DWAPBSSI_IR_RXFIFO_OVERFLOW | DWAPBSSI_IR_RXFIFO_UNDERFLOW | DWAPBSSI_IR_TXFIFO_OVERFLOW))) {
				debug("--- TXRX FAULT BREAK 0x%x ---", sr);
				ddata->last_fault_isr = sr;
				break;
			}
		}
		spin_unlock_irqrestore(&ddata->irq_lock, flags);
		goto end_transfer;
	}
	
	if (ddata->dma_rx)
		dmahelper_start(ddata->dma_rx);
	
	if (rx_only) {
		if (!ddata->is_slave)
	/* In (master) RX-only mode, you need to write a dummy byte to start xfer. */
			apb_write8(ddata->base, DWAPBSSI_IC_DR, 0);
	} else if (ddata->dma_tx) {
		dmahelper_start(ddata->dma_tx);
	}
	
	if (ddata->is_slave)
		apb_write8(ddata->base, DWAPBSSI_IC_SSIENR, 1);
	
	if (timeout == DWAPBSSI_SLAVE_TIMEOUT_INFINITE)
		ret = wait_event_interruptible(ddata->wait_queue,
			(atomic_read(&ddata->completion_pending) <= 0));
	else
		ret = wait_event_interruptible_timeout(ddata->wait_queue,
			(atomic_read(&ddata->completion_pending) <= 0),
			msecs_to_jiffies(timeout));
	
	apb_write16(ddata->base, DWAPBSSI_IC_IMR, 0);
	
	dmahelper_update_cur_bytes(ddata);
	
	if (ddata->dma_tx)
		dmahelper_free(ddata->dma_tx);
	if (ddata->dma_rx) {
		unsigned long read_until = ddata->cur_rx;
		ddata->cur_rx = 0;
		dmahelper_free(ddata->dma_rx);
		while (ddata->cur_rx < read_until)
			dwapbssi_read_next_chunk(ddata, 1, ddata->dma_rx_buf[ddata->cur_rx]);
	}
	
end_transfer:
	if (unlikely((ret <= 0 && timeout != DWAPBSSI_SLAVE_TIMEOUT_INFINITE)
		|| (timeout == DWAPBSSI_SLAVE_TIMEOUT_INFINITE && ret < 0))) {
		stop_reason = (ret == 0) ? "operation timed out" : "operation interrupted";
	} else if (unlikely(ddata->last_fault_isr)) {
		if (ddata->last_fault_isr & DWAPBSSI_IR_RXFIFO_OVERFLOW)
			stop_reason = "RX FIFO overflow (speed too high?)";
		else if (ddata->last_fault_isr & DWAPBSSI_IR_RXFIFO_UNDERFLOW)
			stop_reason = "RX FIFO underflow";
		else if (ddata->last_fault_isr & DWAPBSSI_IR_TXFIFO_OVERFLOW)
			stop_reason = "TX FIFO overflow";
		else
			stop_reason = "unknown error";
	} else if (unlikely(atomic_read(&ddata->tx_dma_error)
		&& atomic_read(&ddata->rx_dma_error))) {
		stop_reason = "RX and TX DMA failed";
	} else if (unlikely(atomic_read(&ddata->tx_dma_error))) {
		stop_reason = "TX DMA failed";
	} else if (unlikely(atomic_read(&ddata->rx_dma_error))) {
		stop_reason = "RX DMA failed";
	} else if (unlikely(dwapbssi_wait_for_idle(ddata, timeout))) {
		stop_reason = "transfer completion timeout";
	} else {
		goto end_disable;
	}
	
  	dev_err(&ddata->dev->dev, "%s: %s, sent/read/total bytes: %lu/%lu/%lu,"
		" chipselect: %d\n", ddata->is_slave ? "slave" : "master", 
		stop_reason, ddata->cur_tx, ddata->cur_rx, 
		ddata->total_len, spi->chip_select);
	dev_err(&ddata->dev->dev, "status: 0x%x, interrupt status: 0x%x,"
		" speed: %luHz\n", apb_read8(ddata->base, DWAPBSSI_IC_SR),
		apb_read16(ddata->base, DWAPBSSI_IC_RISR), effective_hz);
	dev_err(&ddata->dev->dev, "TX/RX only: %s/%s, DMA: %s/%s, SPIN: %s\n",
		tx_only ? "yes" : "no", rx_only ? "yes" : "no",
		ddata->dma_tx ? "yes" : "no", ddata->dma_rx ? "yes" : "no",
		do_spinlock ? "yes" : "no");
	
end_disable:
	apb_write8(ddata->base, DWAPBSSI_IC_SSIENR, 0);
	
	/* Fill data which could not be read with zeros */
	if (!tx_only) {
		while (ddata->cur_rx < ddata->total_len)
			dwapbssi_read_next_chunk(ddata, 1, 0);
	}
	
	dwapbssi_xfer_reset(ddata);
	
	mutex_unlock(&ddata->access_lock);
	
end_complete:
	msg->state = 0;
	msg->complete(msg->context);
	
	/* Free memory associated with this work */
	kmem_cache_free(work_data_cache, wdata);
}

static irqreturn_t dwapbssi_interrupt(int irq, void *dev_id)
{
	struct dwapbssi_drv_data 	*ddata = dev_id;
	u8				isr;

	if (unlikely(!ddata))
		return IRQ_NONE;
	
	if (unlikely(!atomic_read(&ddata->completion_pending)
		|| atomic_read(&ddata->disabled)))
		goto wakeup;
	
	isr = apb_read8(ddata->base, DWAPBSSI_IC_ISR);
	
	if (unlikely(isr & DWAPBSSI_IR_RXFIFO_OVERFLOW)) {
		ddata->last_fault_isr = isr;
		apb_read8(ddata->base, DWAPBSSI_IC_RXOICR);
		goto wakeup;
	} else if (unlikely(isr & DWAPBSSI_IR_RXFIFO_UNDERFLOW)) {
		ddata->last_fault_isr = isr;
		apb_read8(ddata->base, DWAPBSSI_IC_RXUICR);
		goto wakeup;
	} else if (unlikely(isr & DWAPBSSI_IR_TXFIFO_OVERFLOW)) {
		ddata->last_fault_isr = isr;
		apb_read8(ddata->base, DWAPBSSI_IC_TXOICR);
		goto wakeup;
	}
	
	if ((isr & DWAPBSSI_IR_RXFIFO_FULL) || (isr & DWAPBSSI_IR_TXFIFO_EMPTY)) {
		
		int r, w, read = 0, wrote = 0;
		u16 imr;
		
		for (;;) {
			r = ((isr & DWAPBSSI_IR_RXFIFO_FULL)
				&& ddata->cur_rx < ddata->total_len);
			w = ((isr & DWAPBSSI_IR_TXFIFO_EMPTY)
				&& ddata->cur_tx < ddata->total_len);
			if (!r && !w) {
				break;
			} else if (r) {
				dwapbssi_read_next_chunk(ddata, 0, 0);
				read++;
			} else if (w) {
				dwapbssi_write_next_chunk(ddata, 0);
				wrote++;
			}
			isr = apb_read8(ddata->base, DWAPBSSI_IC_ISR);
		}
		
		imr = apb_read16(ddata->base, DWAPBSSI_IC_IMR);
		
		if (ddata->cur_tx == ddata->total_len
			&& (imr & DWAPBSSI_IR_TXFIFO_EMPTY)) {
			apb_write16(ddata->base, DWAPBSSI_IC_IMR,
				imr & ~DWAPBSSI_IR_TXFIFO_EMPTY);
			apb_write32(ddata->base, DWAPBSSI_IC_RXFTLR, 0);
		}
		
		if (ddata->cur_rx == ddata->total_len
			&& (imr & DWAPBSSI_IR_RXFIFO_FULL)) {
			apb_write16(ddata->base, DWAPBSSI_IC_IMR,
				imr & ~DWAPBSSI_IR_RXFIFO_FULL);
			/* This is to make sure the interrupt will wake up below */
			read++;
		}
		
		if ((ddata->cur_tx == ddata->total_len && wrote)
			|| (ddata->cur_rx == ddata->total_len && read)) {
			atomic_dec(&ddata->completion_pending);
			wake_up_interruptible(&ddata->wait_queue);
		}
	}
	 
	return IRQ_HANDLED;

  wakeup:
  	apb_write16(ddata->base, DWAPBSSI_IC_IMR, 0);
	atomic_set(&ddata->completion_pending, 0);
	wake_up_interruptible(&ddata->wait_queue);
	return IRQ_HANDLED;
}

static int dwapbssi_transfer(struct spi_device *spi, struct spi_message *msg)
{
	struct dwapbssi_drv_data 	*ddata = spidev_to_drvdata(spi->master);
	struct dwapbssi_work_data 	*wdata;
	struct spi_transfer 		*xfer;
	
	if (unlikely(atomic_read(&ddata->disabled))) {
		int dis = atomic_read(&ddata->disabled);
		const char *reason = NULL;
		if (dis & DISABLED_HALT)
			reason = "halting";
		else if (dis & DISABLED_ERROR)
			reason = "error state";
		if (reason)
			dev_err(&ddata->dev->dev, "%s: driver is disabled and cannot "
				"perform any transfer (%s)\n", __func__, reason);
		return -EPERM;
	}
	
	list_for_each_entry(xfer, &msg->transfers, transfer_list) {
		if ((!xfer->tx_buf && !xfer->rx_buf) || !xfer->len)
			return -EINVAL;
	}
	
	wdata = kmem_cache_alloc(work_data_cache, GFP_KERNEL);
	wdata->spi = spi;
	wdata->msg = msg;
	INIT_WORK(&wdata->work, dwapbssi_access);
	queue_work(ddata->workqueue, &wdata->work);
	
	return 0;
}

static void dwapbssi_cleanup(struct spi_device *spi)
{
}

/*-------------------------------------------------------------------------*/

static ssize_t dev_attr_show_config(struct device *dev, struct device_attribute *attr, char *buf)
{
	int ret = 0;
	struct dwapbssi_drv_data *ddata = dev_get_drvdata(dev);
	struct dwapbssi_spi_device *spi_d;
	unsigned long clock_rate = clk_get_rate(ddata->clock);
	
	mutex_lock(&ddata->access_lock);
	
	ret += snprintf(buf + ret, PAGE_SIZE - ret, DRIVER_NAME " " DRIVER_VERSION ": %s %s Bus Configuration (ssi_clk=%luHz)\n",
		dev_name(dev), ddata->is_slave ? "SLAVE" : "MASTER", clock_rate);
	if (ddata->is_slave) {
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "Theorical Max. RX+TX Speed:      %luHz\n", clock_rate >> 3);
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "Theorical Max. RX only Speed:    %luHz\n\n", clock_rate / 6);
		if (ddata->slave_timeout == DWAPBSSI_SLAVE_TIMEOUT_INFINITE)
			ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (T) Transfer Timeout:          INFINITE\n");
		else
			ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (T) Transfer Timeout:          %lumsecs\n",
				ddata->slave_timeout);
	} else {
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "Theorical Max. Transfer Speed:   %luHz\n\n", clock_rate >> 1);
		if (ddata->force_speed) {
			ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (S) Forced Speed:              %luHz\n",
				ddata->force_speed);
		} else {
			ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (S) Speed:                     (Default per-chip configuration.)\n");
		}
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "      Transfer Timeout:          (Automatic in master mode.)\n");
	}
	ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (X) Disabled:                  %s\n",
		atomic_read(&ddata->disabled) ? 
		((atomic_read(&ddata->disabled) & DISABLED_ERROR) ? 
		"YES, ERROR STATE" : "YES") : "NO");
	ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (I) Ignore RXFIFO Over. Intr.: %s\n",
		atomic_read(&ddata->ignore_rxfifoover) ? "YES" : "NO");
	
	list_for_each_entry(spi_d, &ddata->spi_devices, list) {
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "\n%d.%d %s Device (chipselect %d):\n",
			ddata->master->bus_num, spi_d->dev->chip_select, 
			ddata->is_slave ? "MASTER" : "SLAVE", spi_d->dev->chip_select);
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (M) SPI Mode:                  %d\n",
			spi_d->dev->mode);
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (C) Client Driver/modalias:    %s\n",
			spi_d->dev->modalias);
		if (!ddata->is_slave)
			ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (S) Speed:                     %luHz\n",
				(unsigned long)spi_d->dev->max_speed_hz);
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (D) Use TX DMA:                %s\n",
			spi_d->use_tx_dma ? "YES" : "NO");
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (d) Use RX DMA:                %s\n",
			spi_d->use_rx_dma ? "YES" : "NO");
	}
	
	ret += snprintf(buf + ret, PAGE_SIZE - ret, "\necho '<item> [<chipselect>] <value>' in this file to change setting.\n");
	
	mutex_unlock(&ddata->access_lock);
	
	return ret;
}

static void dwapbssi_private_abort(struct dwapbssi_drv_data *ddata)
{
	if (mutex_trylock(&ddata->access_lock)) {
		/* No transfer is ongoing. */
		mutex_unlock(&ddata->access_lock);
		return;
	}
	atomic_set(&ddata->completion_pending, 0);
	apb_write16(ddata->base, DWAPBSSI_IC_IMR, 0);
	wake_up_interruptible(&ddata->wait_queue);
	mutex_lock(&ddata->access_lock);
	mutex_unlock(&ddata->access_lock);
}

static ssize_t dev_attr_store_config(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct dwapbssi_drv_data *ddata = dev_get_drvdata(dev);
	char c = 0;
	unsigned n = 0;
	char v[256];
	static const char abort_string[] = "abort";
	
	/* Special request: abort current transfer/wait (useful in slave mode). */
	if (!memcmp(buf, abort_string, strlen(abort_string))) {
		dwapbssi_private_abort(ddata);
		return count;
	}
	
	mutex_lock(&ddata->access_lock);
	
	if (sscanf(buf, "%c %u %256s", &c, &n, v) == 3) {
		int found = 0;
		struct dwapbssi_spi_device *spi_d;
		list_for_each_entry(spi_d, &ddata->spi_devices, list) {
			if (spi_d->dev->chip_select == n) {
				found = 1;
				break;
			}
		}
		if (!found)
			goto err_input;
		switch (c) {
		case 'S': {
			unsigned long i = 0;
			if (sscanf(v, "%lu", &i) != 1 || !i)
				goto err_input;
			spi_d->dev->max_speed_hz = i;
			ddata->bdata->devices[n].max_speed_hz = i;
			break;
		}
		case 'M': {
			int i = 0;
			if (sscanf(v, "%i", &i) != 1 || i < 0 || i > 3)
				goto err_input;
			spi_d->dev->mode = i;
			ddata->bdata->devices[n].mode = i;
			break;
		}
		case 'C': {
			int len = strlen(v);
			while (isspace(v[len - 1])) {
				v[len - 1] = 0;
				if (--len == 0)
					goto err_input;
			}
			strncpy(ddata->bdata->devices[n].modalias, v, 20);
			mutex_unlock(&ddata->access_lock);
			spi_unregister_device(spi_d->dev);
			spi_d->dev = spi_new_device(ddata->master, &ddata->bdata->devices[n]);
			mutex_lock(&ddata->access_lock);
			if (unlikely(!spi_d->dev)) {
				list_del(&spi_d->list);
				kfree(spi_d);
			}
			break;
		}
		case 'D': {
			int i = 0;
			if (sscanf(v, "%i", &i) != 1)
				goto err_input;
			spi_d->use_tx_dma = (i != 0);
			break;
		}
		case 'd': {
			int i = 0;
			if (sscanf(v, "%i", &i) != 1)
				goto err_input;
			spi_d->use_rx_dma = (i != 0);
			break;
		}
		default:
			goto err_input;
		}
		goto eofunction;
		
	} else if (sscanf(buf, "%c %256s", &c, v) == 2) {
		switch (c) {
		case 'S': {
			unsigned long i = 0;
			if (sscanf(v, "%lu", &i) != 1 || !i)
				goto err_input;
			ddata->force_speed = i;
			break;
		}
		case 'T': {
			if (ddata->is_slave) {
				char b[16] = { 0 };
				if (sscanf(v, "%3s", b) == 1 && (!strcmp(b, "inf") || !strcmp(b, "INF"))) {
					ddata->slave_timeout = DWAPBSSI_SLAVE_TIMEOUT_INFINITE;
				} else {
					unsigned long t = 0;
					sscanf(v, "%lu", &t);
					if (t >= 1)
						ddata->slave_timeout = t;
					else
						dev_err(&ddata->dev->dev, "%s: invalid timeout value\n", __func__);
				}					
			} else {
				dev_err(&ddata->dev->dev, "%s: cannot set timeout in master mode\n", __func__);
			}
			break;
		}
		case 'X': {
			int i = 0;
			if (sscanf(v, "%i", &i) != 1)
				goto err_input;
			if (i)
				atomic_set(&ddata->disabled, atomic_read(&ddata->disabled) | DISABLED_USER);
			else
				atomic_set(&ddata->disabled, atomic_read(&ddata->disabled) & ~DISABLED_USER);
			atomic_set(&ddata->disabled, atomic_read(&ddata->disabled) & ~DISABLED_ERROR);
			break;
		}
		case 'I': {
			int i = 0;
			if (sscanf(v, "%i", &i) != 1)
				goto err_input;
			atomic_set(&ddata->ignore_rxfifoover, i);
			break;
		}
		default:
			goto err_input;
		}
		goto eofunction;
	}
err_input:
	dev_err(&ddata->dev->dev, "%s: invalid input\n", __func__);
eofunction:
  	mutex_unlock(&ddata->access_lock);
	return count;
}

static DEVICE_ATTR(config, 0644, dev_attr_show_config, dev_attr_store_config);

/*-------------------------------------------------------------------------*/

int dwapbssi_configure(const char *bus_id, unsigned long force_speed,
	unsigned long slave_timeout)
{
	struct amba_device *dev;
	if (unlikely(!bus_id || !*bus_id))
		return -EINVAL;
	dev = amba_find_device(bus_id, NULL, 0, 0);
	if (dev) {
		struct dwapbssi_drv_data *ddata = amba_get_drvdata(dev);
		mutex_lock(&ddata->access_lock);
		ddata->force_speed = force_speed;
		if (slave_timeout)
			ddata->slave_timeout = slave_timeout;
		mutex_unlock(&ddata->access_lock);
		return 0;
	} else {
		return -ENODEV;
	}
}
EXPORT_SYMBOL(dwapbssi_configure);

int dwapbssi_configure_device(const char *bus_id, u8 chip_select, unsigned long speed_hz, 
	const char *modalias, int mode, int use_tx_dma, int use_rx_dma)
{
	struct amba_device *dev;
	if (unlikely(!bus_id || !*bus_id))
		return -EINVAL;
	dev = amba_find_device(bus_id, NULL, 0, 0);
	if (dev) {
		struct dwapbssi_drv_data *ddata = amba_get_drvdata(dev);
		int ret = 0, found = 0;
		struct dwapbssi_spi_device *spi_d;
		mutex_lock(&ddata->access_lock);
		list_for_each_entry(spi_d, &ddata->spi_devices, list) {
			if (spi_d->dev->chip_select == chip_select) {
				found = 1;
				break;
			}
		}
		if (found) {
			if (use_tx_dma >= 0)
				spi_d->use_tx_dma = (use_tx_dma != 0);
			if (use_rx_dma >= 0)
				spi_d->use_rx_dma = (use_rx_dma != 0);
			if (mode >= 0 && mode <= 3) {
				spi_d->dev->mode = mode;
				ddata->bdata->devices[chip_select].mode = mode;
			}
			if (speed_hz) {
				spi_d->dev->max_speed_hz = speed_hz;
				ddata->bdata->devices[chip_select].max_speed_hz = speed_hz;
			}
			if (modalias && *modalias) {
				strncpy(ddata->bdata->devices[chip_select].modalias, modalias, 20);
				mutex_unlock(&ddata->access_lock);
				spi_unregister_device(spi_d->dev);
				spi_d->dev = spi_new_device(ddata->master, &ddata->bdata->devices[chip_select]);
				mutex_lock(&ddata->access_lock);
				if (unlikely(!spi_d->dev)) {
					list_del(&spi_d->list);
					kfree(spi_d);
				}
			}
		} else {
			ret = -ENODEV;
		}
		mutex_unlock(&ddata->access_lock);
		return ret;
	} else {
		return -ENODEV;
	}
}
EXPORT_SYMBOL(dwapbssi_configure_device);

int dwapbssi_abort_transfer(const char *bus_id, int disable)
{
	struct amba_device *dev;
	if (unlikely(!bus_id || !*bus_id))
		return -EINVAL;
	dev = amba_find_device(bus_id, NULL, 0, 0);
	if (dev) {
		struct dwapbssi_drv_data *ddata = amba_get_drvdata(dev);
		if (disable)
			atomic_set(&ddata->disabled, atomic_read(&ddata->disabled)
				| DISABLED_USER);
		dwapbssi_private_abort(ddata);
		return 0;
	} else {
		return -ENODEV;
	}
}
EXPORT_SYMBOL(dwapbssi_abort_transfer);

int dwapbssi_reenable(const char *bus_id)
{
	struct amba_device *dev;
	if (unlikely(!bus_id || !*bus_id))
		return -EINVAL;
	dev = amba_find_device(bus_id, NULL, 0, 0);
	if (dev) {
		struct dwapbssi_drv_data *ddata = amba_get_drvdata(dev);
		atomic_set(&ddata->disabled, atomic_read(&ddata->disabled)
			& ~DISABLED_ERROR & ~DISABLED_USER);
		return 0;
	} else {
		return -ENODEV;
	}
}
EXPORT_SYMBOL(dwapbssi_reenable);

struct spi_device *dwapbssi_find_device(const char *bus_id, u8 chip_select)
{
	struct amba_device *dev;
	if (unlikely(!bus_id || !*bus_id))
		return NULL;
	dev = amba_find_device(bus_id, NULL, 0, 0);
	if (dev) {
		struct dwapbssi_drv_data *ddata = amba_get_drvdata(dev);
		struct dwapbssi_spi_device *entry;
		list_for_each_entry(entry, &ddata->spi_devices, list) {
			if (entry->dev->chip_select == chip_select)
				return entry->dev;
		}
	}
	return NULL;
}
EXPORT_SYMBOL(dwapbssi_find_device);

/*-------------------------------------------------------------------------*/

static int dwapbssi_probe(struct amba_device *dev, struct amba_id *id)
{
	int 				ret, i;
	struct dwapbssi_drv_data	*ddata;
	struct dwapbssi_bus_data	*bdata = dev->dev.platform_data;
	
	if (unlikely(!bdata || !bdata->num_devices))
		return -EINVAL;
	
	ddata = kzalloc(sizeof(struct dwapbssi_drv_data), GFP_KERNEL);
	if (unlikely(!ddata)) {
		ret = -ENOMEM;
		goto err_alloc;
	}
	ddata->dev = dev;
	ddata->bdata = bdata;
	
	amba_set_drvdata(dev, ddata);
	
	ddata->base = ioremap(dev->res.start, dev->res.end - dev->res.start + 1);
	if (unlikely(!ddata->base)) {
		dev_err(&dev->dev, "%s: unable to remap memory (address=0x%08x, size=%d)\n",
			__func__, dev->res.start, dev->res.end - dev->res.start + 1);
		ret = -EBUSY;
		goto err_ioremap;
	}
	
	/* Disable component & interrupts */
	apb_write8(ddata->base, DWAPBSSI_IC_SSIENR, 0);
	apb_write16(ddata->base, DWAPBSSI_IC_IMR, 0);
	
	for (i=0; i<AMBA_NR_IRQS; i++) { 
		if (dev->irq[i] != NO_IRQ) {
			if (unlikely(ret = request_irq(dev->irq[i], dwapbssi_interrupt,
				IRQF_DISABLED, DRIVER_NAME, (void*)ddata))) {
				dev_err(&dev->dev, "%s: could not get shared interrupt "
					"%d, error is %d\n", __func__, dev->irq[i], ret);
				goto err_irq;
			}
		}
	}
	
	ddata->clock = clk_get(&dev->dev, ddata->bdata->clock_id);
	if (unlikely(IS_ERR(ddata->clock))) {
		ret = PTR_ERR(ddata->clock);
		dev_err(&dev->dev, "%s: failed to get clock id=%s, "
			"error is %d\n", __func__, 
			bdata->clock_id, ret);
		goto err_clkget;
	}
	
	ddata->master = spi_alloc_master(&dev->dev, sizeof(struct dwapbssi_drv_data *));
	if (unlikely(!ddata->master)) {
		dev_err(&dev->dev, "%s: failed to allocate SPI master\n", __func__);
		ret = -ENOMEM;
		goto err_alloc_master;
	}
	*((struct dwapbssi_drv_data **)spi_master_get_devdata(ddata->master)) = ddata;
	
	ddata->workqueue = create_singlethread_workqueue(dev_name(&dev->dev));
	if (unlikely(!ddata->workqueue)) {
		dev_err(&dev->dev, "%s: failed to create workqueue\n", __func__);
		ret = -ENOMEM;
		goto err_workqueue;
	}
	
	/* Initialize driver data */
	atomic_set(&ddata->disabled, DISABLED_HALT);
	atomic_set(&ddata->completion_pending, 0);
	atomic_set(&ddata->ignore_rxfifoover, 0);
	mutex_init(&ddata->access_lock);
	init_waitqueue_head(&ddata->wait_queue);
	spin_lock_init(&ddata->irq_lock);
	ddata->initializing = 1;
	
	/* Configure SPI controller */
	ddata->master->bus_num = bdata->devices[0].bus_num;
	ddata->master->setup = dwapbssi_setup;
	ddata->master->transfer = dwapbssi_transfer;
	ddata->master->cleanup = dwapbssi_cleanup;
	ddata->master->num_chipselect = bdata->num_chipselect;
	
	ddata->slave_timeout = (bdata->slave_timeout ? bdata->slave_timeout
		: DWAPBSSI_DEFAULT_SLAVE_TIMEOUT_MSECS);
	
	/* Determine if component is slave by design */
	ddata->is_slave = 0;
	apb_write16(ddata->base, DWAPBSSI_IC_CTRLR1, 10);
	if (apb_read16(ddata->base, DWAPBSSI_IC_CTRLR1) == 0) {
		ddata->is_slave = 1;
	} else {
		apb_write16(ddata->base, DWAPBSSI_IC_CTRLR1, 0);
	}
	
	/* Register our controller with the SPI framework */
	if (unlikely(ret = spi_register_master(ddata->master))) {
		dev_err(&dev->dev, "%s: could not register SPI controller\n", __func__);
		goto err_register_master;
	}
		
	if (unlikely(ret = device_create_file(&dev->dev, &dev_attr_config))) {
		dev_err(&dev->dev, "%s: unable to create bus config file\n", __func__);
		goto err_devfile;
	}
	
	dev_info(&dev->dev, "device is %s\n",
		ddata->is_slave ? "SLAVE" : "MASTER");
	
	atomic_set(&ddata->disabled, 0);
	
	INIT_LIST_HEAD(&ddata->spi_devices);
	
	mutex_lock(&ddata->access_lock);
	for (i=0; i<bdata->num_devices; i++) {
		struct dwapbssi_spi_device *spi_d = 
			kzalloc(sizeof(struct dwapbssi_spi_device), GFP_KERNEL);
		if (likely(spi_d)) {
			bdata->devices[i].bus_num = ddata->master->bus_num;
			spi_d->use_tx_dma = (bdata->use_tx_dma != 0);
			spi_d->use_rx_dma = (bdata->use_rx_dma != 0);
			spi_d->cs_cache = bdata->devices[i].chip_select;
			list_add_tail(&spi_d->list, &ddata->spi_devices);
			mutex_unlock(&ddata->access_lock);
			spi_d->dev = spi_new_device(ddata->master, &bdata->devices[i]);
			mutex_lock(&ddata->access_lock);
			if (unlikely(!spi_d->dev)) {
				list_del(&spi_d->list);
				kfree(spi_d);
			}
		}
	}
	mutex_unlock(&ddata->access_lock);
	
	ddata->initializing = 0;
	
	return 0;
  
err_devfile:
  	spi_unregister_master(ddata->master);
err_register_master:
	destroy_workqueue(ddata->workqueue);
err_workqueue:
	spi_master_put(ddata->master);
	put_device(&dev->dev);
err_alloc_master:
  	clk_put(ddata->clock);
err_clkget:
err_irq:
 	while (--i >= 0) {
		if (dev->irq[i] != NO_IRQ)
 			free_irq(dev->irq[i], (void *)ddata);
	}
	iounmap(ddata->base);
err_ioremap:
 	kfree(ddata);
err_alloc:
	return ret;
}

static int dwapbssi_remove(struct amba_device *dev)
{
	int 				ret, i;
	struct dwapbssi_drv_data 	*ddata = amba_get_drvdata(dev);
	struct dwapbssi_spi_device 	*entry, *tmp_entry;
	
	/* Make sure no transfer is ongoing. */
	atomic_set(&ddata->disabled, atomic_read(&ddata->disabled) | DISABLED_HALT);
	if (unlikely(ret = mutex_lock_interruptible(&ddata->access_lock))) {
		atomic_set(&ddata->disabled, atomic_read(&ddata->disabled) & ~DISABLED_HALT);
		return ret;
	}
	
	dev_info(&dev->dev, "removing device\n");
	
	/* Unregister SPI devices currently on the bus. */
	list_for_each_entry_safe(entry, tmp_entry, &ddata->spi_devices, list) {
		spi_unregister_device(entry->dev);
		list_del(&entry->list);
		kfree(entry);
	}
	
	mutex_unlock(&ddata->access_lock);
	
	amba_set_drvdata(dev, NULL);
	
	for (i=0; i<AMBA_NR_IRQS; i++) {
		if (dev->irq[i] != NO_IRQ)
 			free_irq(dev->irq[i], (void *)ddata);
	}
	
	if (ddata->dma_tx_buf)
		free_pages((unsigned long)ddata->dma_tx_buf, ddata->dma_tx_buf_order);
	if (ddata->dma_rx_buf)
		free_pages((unsigned long)ddata->dma_rx_buf, ddata->dma_rx_buf_order);
	
	device_remove_file(&dev->dev, &dev_attr_config);
	flush_workqueue(ddata->workqueue);
	destroy_workqueue(ddata->workqueue);
	spi_unregister_master(ddata->master);
	spi_master_put(ddata->master);
	
	/* spi_master_put wont release the parent device */
	put_device(&dev->dev);
	
	clk_put(ddata->clock);
	
	iounmap(ddata->base);
	kfree(ddata);
	
	return 0;
}

#ifdef CONFIG_PM

static int dwapbssi_suspend(struct amba_device *dev, pm_message_t msg)
{
	int ret;
	struct dwapbssi_drv_data *ddata = amba_get_drvdata(dev);
	dev_info(&dev->dev, "suspending, all new SPI messages will be discarded\n");
	atomic_set(&ddata->disabled, atomic_read(&ddata->disabled) | DISABLED_HALT);
	/* Just wait for current transfer to end. */
	if (unlikely(ret = mutex_lock_interruptible(&ddata->access_lock))) {
		atomic_set(&ddata->disabled, atomic_read(&ddata->disabled) & ~DISABLED_HALT);
		return ret;
	}
	mutex_unlock(&ddata->access_lock);
	return 0;
}

static int dwapbssi_resume(struct amba_device *dev)
{
	struct dwapbssi_drv_data *ddata = amba_get_drvdata(dev);
	dev_info(&dev->dev, "resuming\n");
	atomic_set(&ddata->disabled, atomic_read(&ddata->disabled) & ~DISABLED_HALT);
	return 0;
}

#endif

/*-------------------------------------------------------------------------*/

static struct __initdata amba_id dwapbssi_ids[] = {
	{
		.id	= DWAPBSSI_ID,
		.mask	= DWAPBSSI_ID_MASK,
	},
	{ 0, 0 },
};

static struct __initdata amba_driver dwapbssi_drv = {
	.drv		= {
		.name	= DRIVER_NAME,
	},
	.probe		= dwapbssi_probe,
	.remove		= dwapbssi_remove,
#ifdef CONFIG_PM
	.suspend	= dwapbssi_suspend,
	.resume		= dwapbssi_resume,
#endif
	.id_table	= dwapbssi_ids,
};

static int __init dwapbssi_init(void)
{
	int ret;

	debug(DRIVER_DESC " compiled for Linux %s, version %s (%s at %s)",
		UTS_RELEASE, DRIVER_VERSION, __DATE__, __TIME__);
	
	work_data_cache = kmem_cache_create(DRIVER_NAME, 
		sizeof(struct dwapbssi_work_data),
		0, 0, NULL);
	if (unlikely(!work_data_cache)) {
		error("%s: kmem_cache_create failed", __func__);
		return -ENOMEM;
	}
	
	if (unlikely(ret = amba_driver_register(&dwapbssi_drv)))
		error("could not register AMBA driver");

	return ret;
}

static void __exit dwapbssi_exit(void)
{
	amba_driver_unregister(&dwapbssi_drv);
	kmem_cache_destroy(work_data_cache);
}

module_init(dwapbssi_init);
module_exit(dwapbssi_exit);

/*-------------------------------------------------------------------------*/

MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_LICENSE("GPL");
