/*
 * drivers/net/stmmac/stmmac_mdio.c
 *
 * STMMAC Ethernet Driver -- MDIO bus implementation
 * Provides Bus interface for MII registers
 *
 * Author: Carl Shaw <carl.shaw@st.com>
 * Maintainer: Giuseppe Cavallaro <peppe.cavallaro@st.com>
 *
 * Copyright (c) 2006-2007 STMicroelectronics
 *
 */
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/netdevice.h>
#include <linux/module.h>
#include <linux/mii.h>
#include <linux/phy.h>
#include <linux/platform_device.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/delay.h>

#include "stmmac.h"

#define BUSY_TIMEOUT_MS 500

/**
 * stmmac_mdio_read
 * @bus: points to the mii_bus structure
 * @phyaddr: MII addr reg bits 15-11
 * @phyreg: MII addr reg bits 10-6
 * Description: it reads data from the MII register from within the phy device.
 */
int stmmac_mdio_read(struct mii_bus *bus, int phyaddr, int phyreg)
{
	struct net_device *ndev = bus->priv;
	struct eth_driver_local *lp = netdev_priv(ndev);
	unsigned long ioaddr = ndev->base_addr;
	//unsigned long timeout_j;
	struct mii_regs_t *mii = &lp->mac->hw.mii;
	u32 value;
	
	value = ((phyaddr << mii->phyaddr_lsb) & mii->phyaddr_msk)
			| ((phyreg << mii->phyreg_lsb) & mii->phyreg_msk)
			| ((mii->get_phy_csr << mii->phycsr_lsb) & mii->phycsr_msk)
			| mii->addr_busy;/* both synopsys drivers set this ??? */

	//timeout_j = jiffies + msecs_to_jiffies(BUSY_TIMEOUT_MS);
	while (((readl(ioaddr + mii->addr)) & mii->addr_busy) == 1 /*&& jiffies < timeout_j*/)
		udelay(100);
	//if (jiffies >= timeout_j)  {
	//	printk(KERN_ERR "XXX: timeout on MII read begin: phyaddr=0x%x phyreg=0x%x\n", phyaddr, phyreg);
	//	dump_stack();
	//}
	
	/* Write the address to the MII address register */
	writel(value, ioaddr + mii->addr);

	//timeout_j = jiffies + msecs_to_jiffies(BUSY_TIMEOUT_MS);
	while (((readl(ioaddr + mii->addr)) & mii->addr_busy) == 1 /*&& jiffies < timeout_j*/)
		udelay(100);
	//if (jiffies >= timeout_j) {
	//	printk(KERN_ERR "XXX: timeout on MII read end: phyaddr=0x%x phyreg=0x%x\n", phyaddr, phyreg);
	//	dump_stack();
	//}

	/* Read the data from the MII data register */
	value = readl(ioaddr + mii->data);
	
	/* NOTE: dirty workaround for libphy to see properly 
	 * when there is no PHY at this address. */
	if ((value & 0xff) == 0xff && phyreg == 0x2) {
		value = 0xffff;
	}
	
	return value;
}

/**
 * stmmac_mdio_write
 * @bus: points to the mii_bus structure
 * @phyaddr: MII addr reg bits 15-11
 * @phyreg: MII addr reg bits 10-6
 * @phydata: phy data
 * Description: it writes the data intto the MII register from within the device.
 */
int stmmac_mdio_write(struct mii_bus *bus, int phyaddr, int phyreg, u16 phydata)
{
	struct net_device *ndev = bus->priv;
	struct eth_driver_local *lp = netdev_priv(ndev);
	unsigned long ioaddr = ndev->base_addr, timeout_j;
	struct mii_regs_t *mii = &lp->mac->hw.mii;
	u32 value;
	
	
	value = ((phyaddr << mii->phyaddr_lsb) & mii->phyaddr_msk)
			| ((phyreg << mii->phyreg_lsb) & mii->phyreg_msk)
			| ((mii->get_phy_csr << mii->phycsr_lsb) & mii->phycsr_msk)
			| mii->addr_busy /* both synopsys drivers set this without polling */
			| mii->addr_write;

	timeout_j = jiffies + msecs_to_jiffies(BUSY_TIMEOUT_MS);
	while (((readl(ioaddr + mii->addr)) & mii->addr_busy) == 1 && jiffies < timeout_j) { }
	if (jiffies >= timeout_j)  {
		printk(KERN_ERR " timeout on MII write begin\n");
	}
	
	writel(phydata, ioaddr + mii->data);
	writel(value, ioaddr + mii->addr);
	
	/* Wait until any existing MII operation is complete */
	timeout_j = jiffies + msecs_to_jiffies(BUSY_TIMEOUT_MS);
	while (((readl(ioaddr + mii->addr)) & mii->addr_busy) == 1 && jiffies < timeout_j) { }
	if (jiffies >= timeout_j)  {
		printk(KERN_ERR " timeout on MII write end\n");
	}
	
	return 0;
}

/* Resets the MII bus */
int stmmac_mdio_reset(struct mii_bus *bus)
{
	struct net_device *ndev = bus->priv;
	struct eth_driver_local *lp = netdev_priv(ndev);

	if (lp->phy_reset) {
		printk("stmmac_mdio_reset: calling phy_reset\n");
		return lp->phy_reset(lp->bsp_priv);
	}

	return 0;
}

/**
 * stmmac_mdio_register
 * @ndev: net device structure
 * Description: it registers the MII bus
 */
int stmmac_mdio_register(struct net_device *ndev)
{
	int ret = 0;
	struct mii_bus *new_bus;
	int *irqlist;
	int i;
	struct eth_driver_local *lp = netdev_priv(ndev);
	
	new_bus = mdiobus_alloc();
	if (unlikely(!new_bus))
		return -ENOMEM;
	
	irqlist = kzalloc(sizeof(int) * PHY_MAX_ADDR, GFP_KERNEL);
	if (unlikely(!irqlist)) {
		kfree(new_bus);
		return -ENOMEM;
	}

	/* Assign IRQ to phy at address phy_addr */
	for(i=0; i < PHY_MAX_ADDR; i++)
		irqlist[i] = -1;
	/* There is currently one phy associated to this dev, so set its irq */
	irqlist[lp->phy_addr] = lp->phy_irq;

	new_bus->name = "STMMAC MII Bus";
	new_bus->read = &stmmac_mdio_read;
	new_bus->write = &stmmac_mdio_write;
	new_bus->reset = &stmmac_mdio_reset;
	snprintf(new_bus->id, MII_BUS_ID_SIZE, "%x", lp->bus_id);
	new_bus->priv = ndev;
	new_bus->irq = irqlist;
	new_bus->phy_mask = lp->phy_mask;
	platform_set_drvdata(ndev, new_bus);
	
	if (unlikely(ret = mdiobus_register(new_bus))) {
		printk(KERN_ERR "%s: Cannot register as MDIO bus\n",
		       new_bus->name);
		goto bus_register_fail;
	}
	/* show the phys connected to the bus */
	for(i=0; i < PHY_MAX_ADDR; i++) {
		struct phy_device *phydev = new_bus->phy_map[i];
		if(phydev) {
			printk("%3d. phy_id %08x drv %s\n", i, phydev->phy_id, phydev->dev.driver ? phydev->dev.driver->name : "NONE" );
		}
	}
	
	lp->mii = new_bus;
	
	return 0;
	
bus_register_fail:
	kfree(new_bus);
	kfree(irqlist);
	return ret;
}

/**
 * stmmac_mdio_unregister
 * @ndev: net device structure
 * Description: it unregisters the MII bus
 */
int stmmac_mdio_unregister(struct net_device *ndev)
{
	struct eth_driver_local *lp = netdev_priv(ndev);
	
	mdiobus_unregister(lp->mii);
	lp->mii->priv = NULL;
	kfree(lp->mii->irq);
	kfree(lp->mii);
	
	return 0;
}

