/*
 *  linux/drivers/mtd/nand/mobi_aumb3000.c
 *
 *  Copyright (C) 2006 Mobilygen Corp
 *
 *  nand driver for Aurora MB3000 flash controller.  Basic structure
 *  adopted from nand_base.c
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 */
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/clk.h>
#include <linux/sched.h>
#include <linux/proc_fs.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/amba/bus.h>
#include <linux/io.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/nand_ecc.h>
#include <linux/mtd/partitions.h>

/* platform data */
#include <linux/aumb3000_nand_device.h>

#include <mach/platform.h>
#include <mach/au_flash_regs.h>

/* APIs */
#include <mach/mobi_dma.h>
#include <mach/mobi_hmux.h>
#include <mach/mobi_reset.h>
#include <mach/mobi_qcc.h>

#include "aumb3000_nand.h"

#define DRIVER_NAME "AU-MB3000 NAND "

#undef LOCAL_AUMB3000_NAND_DEBUG_ENABLE

static struct mtd_info *aumb3000_mtd;
static void __iomem *flashreg_base;
static void __iomem *flashdata_base;
static struct proc_dir_entry *nand_proc_dir;
static int interrupt_enabled;
static int dma_waitq_flag;
static DECLARE_WAIT_QUEUE_HEAD(dma_waitq);

/* values in ns */
static uint8_t default_timings[] = {
	16,   8,   0,   0,  /* timing 0: tds, tcs, tals, tcls     */
	16,   8,   8,  16,  /* timing 1: tdh, tch, talkh, tclh    */
	40, 200,  24,  32,  /* timing 2: trr, twb, twh, twp       */
	0,   0,  18,  40,  /* timing 3: tceh, trb, treh, trp     */
	0,  76,  64,  56,  /* timing 4: tir, twhr, tclr, trdelay */
	56,  40,   0,	    /* timing 5: tar, trhz, tww           */
};

uint32_t modparam_set_timing_bitmask;
static uint8_t modparam_timings[NUMBER_NAND_TIMING_PARAMS];

/* the size has been updated to uint64_t to handle larger flash sizes
 * but I'm having an issue dividing a number that big.  our controller
 * won't go above 8Gbit anyway, so we really don't have to worry about
 * bigger sizes with this controller
 */
#define MTD_INFO_SIZE_CAST	uint32_t

#define UNINITIALIZED_MODPARAM 	((char) -1)

static int __attribute__((unused)) loglevel = -1;
#if defined(LOCAL_AUMB3000_NAND_DEBUG_ENABLE) || \
	defined(CONFIG_AUMB3000_NAND_DEBUG)
#define AUMB3000_NAND_DEBUG 1
#else
#define AUMB3000_NAND_DEBUG 0
#endif

#if AUMB3000_NAND_DEBUG
module_param(loglevel, int, 0644);
MODULE_PARM_DESC(loglevel, "Set module debug level(0-5)");
static int debug_nand_scan;

#define DBGPFX "AUMB:"
#define aumb3000_debug(n, fmt, args...)	\
	do { 						\
		if (loglevel >= n) { 	\
			printk(DBGPFX"%s: " fmt, __func__, ##args); \
		}						\
	} while (0)

#else
#define aumb3000_debug(n, fmt, args...)	do {} while (0)
#endif

#define dprintk(x...)	printk(x)
#define dprintk1(fmt, x...)	aumb3000_debug(1, fmt, ##x)
#define dprintk2(fmt, x...)	aumb3000_debug(2, fmt, ##x)
#define dprintk3(fmt, x...)	aumb3000_debug(3, fmt, ##x)
#define dprintk4(fmt, x...)	aumb3000_debug(4, fmt, ##x)
#define dprintk5(fmt, x...)	aumb3000_debug(5, fmt, ##x)

#define error(str, args...)  \
	printk(KERN_ERR DRIVER_NAME "Error: " str "\n", ##args)


static uint32_t nand_reg_read(uint32_t offset, uint8_t lock)
{
	uint32_t data = 0;

	if (lock)
		NAND_LOCK;

	data = readl(flashreg_base+offset);
	if (lock)
		NAND_UNLOCK;

	return data;
}

static void nand_reg_write(uint32_t data, uint32_t offset, uint8_t lock)
{
	if (lock)
		NAND_LOCK;

	writel(data, flashreg_base+offset);
	if (lock)
		NAND_UNLOCK;
}

/**
 * \brief aumb3000_nand_select_chip:
 * 	Select chip/bank to operate on.
 *
 * \param mtd	- mtd_info struct
 * \chip 	- bank/chip to select
 *
 * \remark
 * 	Normally this function is used to actually select
 * the chip.  Since we have the controller we don't have to
 * worry about that.  The controller will be able to determine
 * which bank to access based on the read/write address.  However,
 * we need to track the currently selected chip because commands
 * like STATUS return a 32-bit word which contains the 8-bit status
 * for the max possible banks of 4.  Since this is not address based
 * will have to used select_chip(which the nand_base framework will
 * call) to keep track of which bank is being queried. Controller
 * can have 1, 2 or 4 banks.
 */
static void aumb3000_nand_select_chip(struct mtd_info *mtd, int which)
{
	struct nand_chip *chip = (struct nand_chip *)mtd->priv;
	struct aumb3000_nand_info *info =
		(struct aumb3000_nand_info *)chip->priv;

	if (which == -1) {
		info->select_chip = -1;
	} else {
		/* our max # of banks is 4 */
		if (which >= 0 && which <= 3)
			info->select_chip = which;
		else
			/* have to select something ?? */
			info->select_chip = 0;
		dprintk5("chip %d\n", info->select_chip);
	}
}

/* Issue command and address cycles to the chip */
/* but since we have a flash controller we do nothing */
static void aumb3000_nand_hwcontrol(struct mtd_info *mtd,
		int cmd, unsigned int ctrl)
{
}

/* convert chip size to controller register value */
static int8_t get_chipsize(unsigned long size)
{
	/* the chipsize stored in the chip structure is shifted and
	 *  is in megabytes, we'll just make a tweak here to
	 *  handle both cases and convert to megabit
	 */
	int adj_size = 0;
	adj_size = (size > 16384 ? size >> 20 : size);
	dprintk3("size = %lu, adj_size = %d\n", size, adj_size);
	switch (adj_size*8) {
	case 64:   return AUMB3000_NAND_SIZE_64M;
	case 128:  return AUMB3000_NAND_SIZE_128M;
	case 256:  return AUMB3000_NAND_SIZE_256M;
	case 512:  return AUMB3000_NAND_SIZE_512M;
	case 1024: return AUMB3000_NAND_SIZE_1G;
	case 2048: return AUMB3000_NAND_SIZE_2G;
	case 4096: return AUMB3000_NAND_SIZE_4G;
	case 8192: return AUMB3000_NAND_SIZE_8G;
	default:  return -1;
	}
}

/* convert number of external banks to controller register value */
static int8_t get_exbanks(int n)
{
	switch (n) {
	case 1: return AUMB3000_NANDNUM_EXTERNAL_BANKS1;
	case 2: return AUMB3000_NANDNUM_EXTERNAL_BANKS2;
	case 4: return AUMB3000_NANDNUM_EXTERNAL_BANKS4;
	default: return -1;
	}
}

static void get_databus_width(struct nand_chip *chip)
{
	struct aumb3000_nand_info *info =
		(struct aumb3000_nand_info *)chip->priv;
	/*
	 *  options are set externally, xs is field in control
	 *  register that indicate if flash chip data bus is
	 *  16 bit, 0 == 8 bit, 1 == 16 bit
	 */
	if (info->xs == 0 && info->chips_per_bank == 1)
		info->wd = AUMB3000_NANDCTRL_DATABUS_WIDTH8;
	else if (info->xs == 0 && info->chips_per_bank == 2)
		info->wd = AUMB3000_NANDCTRL_DATABUS_WIDTH16;
	else if (info->xs == 1 && info->chips_per_bank == 1)
		info->wd = AUMB3000_NANDCTRL_DATABUS_WIDTH16;
	else if (info->xs == 0 && info->chips_per_bank == 4)
		info->wd = AUMB3000_NANDCTRL_DATABUS_WIDTH32;
	else if (info->xs == 1 && info->chips_per_bank == 2)
		info->wd = AUMB3000_NANDCTRL_DATABUS_WIDTH32;
	else if (info->xs == 1 && info->chips_per_bank == 4)
		info->wd = AUMB3000_NANDCTRL_DATABUS_WIDTH64;

	dprintk3("xs = %d, chips_per_bank = %d, wd = %d\n",
			info->xs, info->chips_per_bank, info->wd);
}

/**
 * \brief aumb3000_nand_devready
 * 	See if nand is busy
 *
 * \param mtd: mtd_info structure
 *
 * \retval 0 if the nand is busy, 1 if it is ready
 *
 * \remark
 * 	There are a couple of ways to verify that an operation
 * has completed or if the chip is ready.  it's a bit
 * klunky because there are multiple NOP which indicate that
 * the chip is not busy so we have to look for them all
 * 	Function for nand_chip->dev_ready
 *
 */
static int aumb3000_nand_devready(struct mtd_info *mtd)
{
	uint32_t cmd_reg = 0;

	cmd_reg = REG_READ(MOBI_FLASH_NAND_COMMAND_OFFSET)
		& MOBI_FLASH_NAND_COMMAND_CMD_MASK;

	if (cmd_reg == AUMB3000_NAND_CMD_NOP  ||
			cmd_reg == AUMB3000_NAND_CMD_NOP2 ||
			cmd_reg == AUMB3000_NAND_CMD_NOP3 ||
			cmd_reg == AUMB3000_NAND_CMD_NOP4 ||
			cmd_reg == AUMB3000_NAND_CMD_NOP5 ||
			cmd_reg == AUMB3000_NAND_CMD_NOP6 ||
			cmd_reg == AUMB3000_NAND_CMD_NOP7 ||
			cmd_reg == AUMB3000_NAND_CMD_NOP8) {

		dprintk4("return 1\n");
		return 1;
	} else
		return 0;
}

/**
 * \brief aumb3000_nand_waitready:
 * 	Poll the device to see if it is ready
 *
 * \params mtd: pointer to mtd struct
 *
 * \retval 0 if the nand is busy, 1 if it is ready
 *
 * \remark
 *   Will use a counter loop here to act as a crude timer
 * so that we don't hang if the device never becomes ready.
 * also duplicate the dev_ready functionality so we can save
 * cycles by avoiding lock grabs and function calls
 */
static int aumb3000_nand_waitready(struct mtd_info *mtd)
{
	int32_t i = 1073741824;
	uint32_t cmd_reg = 0;

	NAND_LOCK;
	while (--i != 0) {
		if (interrupt_enabled == 0) {
			cmd_reg =
				readl(flashreg_base +
						MOBI_FLASH_NAND_COMMAND_OFFSET) &
				MOBI_FLASH_NAND_COMMAND_CMD_MASK;
			dprintk4("cmd_reg 0x%x\n", cmd_reg);

			if (cmd_reg == AUMB3000_NAND_CMD_NOP  ||
					cmd_reg == AUMB3000_NAND_CMD_NOP2 ||
					cmd_reg == AUMB3000_NAND_CMD_NOP3 ||
					cmd_reg == AUMB3000_NAND_CMD_NOP4 ||
					cmd_reg == AUMB3000_NAND_CMD_NOP5 ||
					cmd_reg == AUMB3000_NAND_CMD_NOP6 ||
					cmd_reg == AUMB3000_NAND_CMD_NOP7 ||
					cmd_reg == AUMB3000_NAND_CMD_NOP8) {
				NAND_UNLOCK;
				return 1;
			}
		} else {
			/* interrupts are enabled(but not using interrupt handler */
			if (readl(flashreg_base +
						MOBI_FLASH_NAND_INTERRUPT_OFFSET) &
					MOBI_FLASH_NAND_INTERRUPT_MASK) {
				dprintk4("Interrupt bit is set!\n");
				NAND_UNLOCK;
				return 1;
			}
		}
	}
	NAND_UNLOCK;
	error("Flash response timeout");
	return 0;
}

int cmd_write(uint32_t cmd, uint64_t addr)
{
	int32_t i = 1073741824;
	uint32_t cmd_reg = 0;

	NAND_LOCK;

	/* hmmm, this will still eat up cycles polling but, we might be able to
	 * grab the lock, issue the command and do a waitq_int.  then add an
	 * actually interrupt handler that wakes up the waitq. after waking up
	 * we can release the lock.  this way other processes could use the
	 * cycles we are wasting spinning for the NOP.
	 *
	 */
	REG_WRITE(0x0, MOBI_FLASH_NAND_INTERRUPT_OFFSET);
	REG_WRITE(((addr & MOBI_FLASH_NAND_COMMAND_ADDR_MASK) | (cmd & 0xf)),
			MOBI_FLASH_NAND_COMMAND_OFFSET);

	while(--i != 0) {
		if (interrupt_enabled == 0) {
			cmd_reg =
				readl(flashreg_base+MOBI_FLASH_NAND_COMMAND_OFFSET) &
				MOBI_FLASH_NAND_COMMAND_CMD_MASK;
			dprintk4("cmd_reg 0x%x\n", cmd_reg);

			if (cmd_reg == AUMB3000_NAND_CMD_NOP  ||
					cmd_reg == AUMB3000_NAND_CMD_NOP2 ||
					cmd_reg == AUMB3000_NAND_CMD_NOP3 ||
					cmd_reg == AUMB3000_NAND_CMD_NOP4 ||
					cmd_reg == AUMB3000_NAND_CMD_NOP5 ||
					cmd_reg == AUMB3000_NAND_CMD_NOP6 ||
					cmd_reg == AUMB3000_NAND_CMD_NOP7 ||
					cmd_reg == AUMB3000_NAND_CMD_NOP8) {
				NAND_UNLOCK;
				return 0;
			}
		} else {
			/* interrupts are enabled(but not using interrupt handler */
			if (readl(flashreg_base+MOBI_FLASH_NAND_INTERRUPT_OFFSET) &
					MOBI_FLASH_NAND_INTERRUPT_MASK) {
				dprintk4("Interrupt bit is set!\n");
				NAND_UNLOCK;
				return 0;
			}
		}
	}
	error("Flash response timeout");
	NAND_UNLOCK;

	return -1;
}

/* our NAND controller generates, checks AND corrects ECC so we don't need
 *  to have these functions do anything; however, they must be defined for
 *  HW_ECC otherwise nand_scan will fail
 */
static void aumb3000_enable_hwecc(struct mtd_info *mtd, int mode)
{
}

static int aumb3000_calculate_ecc(struct mtd_info *mtd,
		const u_char *dat, u_char *ecc_code)
{
	return 0;
}

static int aumb3000_nand_correct_data(struct mtd_info *mtd, u_char *dat,
		u_char *read_ecc, u_char *calc_ecc)
{
	return 0;
}

void dma_handler(mobi_dma_handle dmah, mobi_dma_event dma_event, void *data)
{
	switch (dma_event) {
	case MOBI_DMA_EVENT_TRANSFER_COMPLETE:
		dprintk2("XFER_COMPLETE: status 0x%x, bytes transfered %d\n",
				mobi_dma_get_status(dmah), mobi_dma_get_bytes(dmah));
		break;
	case MOBI_DMA_EVENT_TRANSFER_ERROR:
		dprintk2("XFER_ERROR: DMA failed\n");
		break;
	}
	dma_waitq_flag = dma_event;
	wake_up_interruptible(&dma_waitq);
}

static void cleanup_nand_dma(struct aumb3000_nand_info *info,
		uint8_t area_access_type)
{
	if (info->buf_dma_addr == 0x0)
		return;

	if (area_access_type == AREA_MODE_SINGLE) {
		dma_unmap_single(NULL,
				info->buf_dma_addr,
				info->dma_length,
				(info->dma_mode == DMA_MODE_WRITE ?
				 DMA_TO_DEVICE : DMA_FROM_DEVICE));
	} else {
		struct mobi_dma_list *mlist =
			(struct mobi_dma_list *)info->buf_dma_addr;
		int i;
		dma_addr_t buf_addr;
		uint8_t dma_mode =
			(info->dma_mode == DMA_MODE_WRITE ?
			 DMA_TO_DEVICE : DMA_FROM_DEVICE);

		for (i = 0; i < 3; i++) {
			if (info->dma_mode == DMA_MODE_WRITE)
				buf_addr = mlist->src_addr;
			else
				buf_addr = mlist->dst_addr;

			dma_unmap_single(NULL, buf_addr,
					mlist->size, dma_mode);
			mlist++;
		}
		kfree(info->ctrl_reg_mem);
		kfree((struct mobi_dma_list *)info->buf_dma_addr);
	}
}

static int dma_config(mobi_dma_handle dma_handle, uint8_t dma_mode)
{
	int32_t retval = 0;

	if (dma_mode == DMA_MODE_READ) {
		/* nand controller is BE  so let the mmu do the work */
		if ((retval = mobi_dma_config(dma_handle, DMA_CONFIG_SRC,
						MOBI_DMA_CONFIG_TRANSFER_WIDTH_32  |
						MOBI_DMA_CONFIG_ADDRADJ_INC  |
						MOBI_DMA_CONFIG_ENDIAN_BE,
						NULL))) {

			dprintk3("Failed to configure source\n");
			return retval;
		}
		if ((retval = mobi_dma_config(dma_handle, DMA_CONFIG_DST,
						MOBI_DMA_CONFIG_ADDRADJ_INC  |
						MOBI_DMA_CONFIG_TRANSFER_WIDTH_32  |
						MOBI_DMA_CONFIG_ENDIAN_LE,
						NULL))) {

			dprintk3("Failed to configure destination\n");
			return retval;
		}
		if ((retval = mobi_dma_config(dma_handle, DMA_CONFIG_XFER,
						MOBI_DMA_CONFIG_DATA_WIDTH_8,
						NULL))) {

			dprintk3("Failed to configure xfer\n");
			return retval;
		}

	} else {
		if ((retval = mobi_dma_config(dma_handle, DMA_CONFIG_SRC,
						MOBI_DMA_CONFIG_TRANSFER_WIDTH_32  |
						MOBI_DMA_CONFIG_ADDRADJ_INC  |
						MOBI_DMA_CONFIG_ENDIAN_LE,
						NULL))) {

			dprintk3("Failed to configure source\n");
			return retval;
		}
		if ((retval = mobi_dma_config(dma_handle, DMA_CONFIG_DST,
						MOBI_DMA_CONFIG_TRANSFER_WIDTH_32  |
						MOBI_DMA_CONFIG_ADDRADJ_INC  |
						MOBI_DMA_CONFIG_ENDIAN_BE,
						NULL))) {

			dprintk3("Failed to configure destination\n");
			return retval;
		}

		if ((retval = mobi_dma_config(dma_handle, DMA_CONFIG_XFER,
						MOBI_DMA_CONFIG_DATA_WIDTH_8,
						NULL))) {

			dprintk3("Failed to configure xfer\n");
			return retval;
		}
	}
	return 0;
}
/**
 * \brief setup_nand_dma:
 * 	Setup a DMA transfer
 *
 * \params mtd: pointer to mtd struct
 * \params buf: r/w data buffer
 * \params len: length of transfer
 * \params mode: read or write
 * \params area_access_mode:  single or multiple area
 *
 * \remark
 * 	This function is used to setup r/w DMA.  The area_access_mode
 *  refers to if we are going to access a single area(main or spare) or
 *  multiple areas(both main and spare).  If accessing a spare area
 *  only, calling functions are expected to have already set the SA bit
 *  in the control register.  If accessing both areas, a dma_list will be
 *  used so that a write to the control register can be done to set the
 *  SA bit before doing the spare area DMA.
 */
static int setup_nand_dma(struct mtd_info *mtd, uint8_t *buf,
		int len, uint8_t mode, uint8_t area_access_mode)
{
	struct nand_chip *chip = (struct nand_chip *)mtd->priv;
	struct aumb3000_nand_info *info =
		(struct aumb3000_nand_info *)chip->priv;
	dma_addr_t src_dma_addr = 0, dst_dma_addr = 0;
	/* internal page address */
	uint64_t nand_addr = 0x0;
	struct mobi_dma_list *mlist = NULL;
	int retval = 0;

	dprintk3("mode %s, length %d\n",
			(mode == DMA_MODE_READ ? "read" : "write"),
			len);

	info->dma_mode = mode;
	/* construct the physical dma address for the chip */
	nand_addr = (info->cmd.page_addr << chip->page_shift) +
		(info->select_chip*chip->chipsize);
	dprintk3("data_phys 0x%x nand_addr 0x%llx, page 0x%x(%d)\n",
			FLASH_DATA_BASE, nand_addr,
			info->cmd.page_addr, info->cmd.page_addr);
	if (mtd->writesize == 512 &&
			(MOBI_FLASH_NAND_CONTROL_SA_R(info->ctrl_shadow))) {
		nand_addr |= ((info->cmd.page_addr << 4) & 0xff);
		dprintk3("small page modified nand_addr 0x%llx\n", nand_addr);
	}

	/* bits [33:26] of the address are programmed into the control reg */
	MOBI_FLASH_NAND_CONTROL_ADDR_CLR_AND_SET(info->ctrl_shadow,
			((nand_addr >> 26) & 0xff));
	CONTROL_WRITE(info->ctrl_shadow);
	dprintk4("control reg = 0x%08x\n", CONTROL_READ());

	if (info->dma_mode == DMA_MODE_READ) {
		if (info->prev_dma_mode == DMA_MODE_WRITE ||
				info->prev_dma_mode == DMA_MODE_NONE) {

			if ((retval = dma_config(info->dma_handle, info->dma_mode))) {
				info->prev_dma_mode = DMA_MODE_NONE;
				goto exit;
			}
			info->prev_dma_mode = DMA_MODE_READ;
		}

		if (area_access_mode == AREA_MODE_SINGLE) {
			dst_dma_addr = dma_map_single(NULL,
					buf,
					len,
					DMA_FROM_DEVICE);

			if (dst_dma_addr <= 0) {
				error("Failed DMA mapping read");
				retval = dst_dma_addr;
				goto exit;
			}
			info->buf_dma_addr = dst_dma_addr;
			info->dma_length = len;

			/* the dma address is the combination of the data
			 *  physical address and the internal nand page
			 */
			src_dma_addr = (uint32_t) FLASH_DATA_BASE +
				(uint32_t)(nand_addr & 0x3ffffff);
		} else {

			mlist = (struct mobi_dma_list *)
				kzalloc(3*(sizeof(struct mobi_dma_list)), GFP_KERNEL);
			if (!mlist) {
				retval = -ENOMEM;
				goto exit;
			}
			info->buf_dma_addr = (uint32_t)mlist;

			info->ctrl_reg_mem = NULL;
			info->ctrl_reg_mem =
				(uint32_t *) kzalloc(sizeof(uint32_t), GFP_KERNEL);
			if (!info->ctrl_reg_mem) {
				retval = -ENOMEM;
				goto exit;
			}

			dst_dma_addr = dma_map_single(NULL,
					buf,
					len,
					DMA_FROM_DEVICE);

			if (dst_dma_addr <= 0) {
				error("Failed DMA mapping read");
				retval = dst_dma_addr;
				goto exit;
			}

			/* the dma address is the combination of the control
			 *  physical address and the internal nand page
			 */
			/* main area write */
			(&mlist[0])->src_addr = (uint32_t) FLASH_DATA_BASE +
				(uint32_t)(nand_addr & 0x3ffffff);
			(&mlist[0])->dst_addr = dst_dma_addr;
			(&mlist[0])->size = mtd->writesize;
			(&mlist[0])->next = &mlist[1];
			dprintk3("sar: 0x%x, dar: 0x%x\n",
					(&mlist[0])->src_addr, (&mlist[0])->dst_addr);

			/* set spare area access bit in ctrl reg */
			/* note for read, will need to do REV dir flag */
			/* also, we need to swap the data since the nand
			 *  controller is BE so dma is setup to do the swap
			 */
			MOBI_FLASH_NAND_CONTROL_SA_SET(info->ctrl_shadow);
			*(info->ctrl_reg_mem) = cpu_to_be32(info->ctrl_shadow);
			MOBI_FLASH_NAND_CONTROL_SA_CLR(info->ctrl_shadow);

			dst_dma_addr = dma_map_single(NULL,
					info->ctrl_reg_mem,
					sizeof(uint32_t),
					DMA_FROM_DEVICE);

			if (dst_dma_addr <= 0) {
				error("Failed DMA mapping write");
				retval = dst_dma_addr;
				goto exit;
			}
			(&mlist[1])->src_addr = FLASH_REG_BASE +
				MOBI_FLASH_NAND_CONTROL_OFFSET;
			(&mlist[1])->dst_addr = dst_dma_addr;
			(&mlist[1])->size = sizeof(uint32_t);
			(&mlist[1])->xfer_flags = MOBI_DMA_LIST_XFER_FLAG_REVERSE_DIR;
			(&mlist[1])->next = &mlist[2];
			dprintk3("sar: 0x%x, dar: 0x%x\n",
					(&mlist[1])->src_addr, (&mlist[1])->dst_addr);

			/* now the spare area data */
			dst_dma_addr = dma_map_single(NULL,
					chip->oob_poi,
					mtd->oobsize,
					DMA_FROM_DEVICE);

			if (dst_dma_addr <= 0) {
				error("Failed DMA mapping write");
				retval = dst_dma_addr;
				goto exit;
			}
			if (mtd->writesize == 512) {
				nand_addr |= ((info->cmd.page_addr << 4) & 0xff);
				dprintk3("small page modified nand_addr 0x%llx\n", nand_addr);
				(&mlist[2])->src_addr = (uint32_t) FLASH_DATA_BASE +
					(uint32_t)(nand_addr & 0x3ffffff);

			} else {
				(&mlist[2])->src_addr = (uint32_t) FLASH_DATA_BASE +
					(uint32_t)(nand_addr & 0x3ffffff);
			}
			(&mlist[2])->dst_addr = dst_dma_addr;
			(&mlist[2])->size = mtd->oobsize;
			dprintk3("sar: 0x%x, dar: 0x%x\n",
					(&mlist[2])->src_addr, (&mlist[2])->dst_addr);
			(&mlist[2])->next = NULL;

			info->dma_length =
				mtd->writesize + mtd->oobsize + sizeof(uint32_t);
		}
	} else {

		if (info->prev_dma_mode == DMA_MODE_READ ||
				info->prev_dma_mode == DMA_MODE_NONE) {

			if ((retval = dma_config(info->dma_handle, info->dma_mode))) {
				info->prev_dma_mode = DMA_MODE_NONE;
				goto exit;
			}
			info->prev_dma_mode = DMA_MODE_WRITE;
		}

		if (area_access_mode == AREA_MODE_SINGLE) {

			src_dma_addr = dma_map_single(NULL,
					buf,
					len,
					DMA_TO_DEVICE);
			if (src_dma_addr <= 0) {
				error("Failed DMA mapping write");
				retval = src_dma_addr;
				goto exit;
			}
			info->buf_dma_addr = src_dma_addr;
			info->dma_length = len;

			dst_dma_addr = (uint32_t) FLASH_DATA_BASE +
				(uint32_t)(nand_addr & 0x3ffffff);

		} else {

			mlist = (struct mobi_dma_list *)
				kzalloc(3*(sizeof(struct mobi_dma_list)), GFP_KERNEL);
			if (!mlist) {
				retval = -ENOMEM;
				goto exit;
			}
			info->buf_dma_addr = (uint32_t)mlist;

			info->ctrl_reg_mem = NULL;
			info->ctrl_reg_mem =
				(uint32_t *) kzalloc(sizeof(uint32_t), GFP_KERNEL);
			if (!info->ctrl_reg_mem) {
				retval = -ENOMEM;
				goto exit;
			}

			src_dma_addr = dma_map_single(NULL,
					buf,
					len,
					DMA_TO_DEVICE);

			if (src_dma_addr <= 0) {
				error("Failed DMA mapping write");
				retval = src_dma_addr;
				goto exit;
			}

			/* main area write */
			(&mlist[0])->src_addr = src_dma_addr;
			(&mlist[0])->dst_addr = (uint32_t) FLASH_DATA_BASE +
				(uint32_t)(nand_addr & 0x3ffffff);
			(&mlist[0])->size = mtd->writesize;
			(&mlist[0])->next = &mlist[1];
			dprintk3("sar: 0x%x, dar: 0x%x\n",
					(&mlist[0])->src_addr, (&mlist[0])->dst_addr);

			/* set spare area access bit in ctrl reg */
			/* note for read, will need to do REV dir flag */
			/* also, we need to swap the data since the nand
			 *  controller is BE so dma is setup to do the swap
			 */
			MOBI_FLASH_NAND_CONTROL_SA_SET(info->ctrl_shadow);
			*(info->ctrl_reg_mem) = cpu_to_be32(info->ctrl_shadow);
			MOBI_FLASH_NAND_CONTROL_SA_CLR(info->ctrl_shadow);

			src_dma_addr = dma_map_single(NULL,
					info->ctrl_reg_mem,
					sizeof(uint32_t),
					DMA_TO_DEVICE);

			if (src_dma_addr <= 0) {
				error("Failed DMA mapping write");
				retval = src_dma_addr;
				goto exit;
			}
			(&mlist[1])->src_addr = src_dma_addr;
			(&mlist[1])->dst_addr = FLASH_REG_BASE +
				MOBI_FLASH_NAND_CONTROL_OFFSET;
			(&mlist[1])->size = sizeof(uint32_t);
			(&mlist[1])->next = &mlist[2];
			dprintk3("sar: 0x%x, dar: 0x%x\n",
					(&mlist[1])->src_addr, (&mlist[1])->dst_addr);

			/* now the spare area data */
			src_dma_addr = dma_map_single(NULL,
					chip->oob_poi,
					mtd->oobsize,
					DMA_TO_DEVICE);

			if (src_dma_addr <= 0) {
				error("Failed DMA mapping write");
				retval = src_dma_addr;
				goto exit;
			}
			(&mlist[2])->src_addr = src_dma_addr;
			if (mtd->writesize == 512) {
				nand_addr |= ((info->cmd.page_addr << 4) & 0xff);
				dprintk3("small page modified nand_addr 0x%llx\n", nand_addr);
				(&mlist[2])->dst_addr = (uint32_t) FLASH_DATA_BASE +
					(uint32_t)(nand_addr & 0x3ffffff);

			} else {
				(&mlist[2])->dst_addr = (uint32_t) FLASH_DATA_BASE +
					(uint32_t)(nand_addr & 0x3ffffff);
			}

			(&mlist[2])->size = mtd->oobsize;
			(&mlist[2])->next = NULL;
			dprintk3("sar: 0x%x, dar: 0x%x\n",
					(&mlist[2])->src_addr, (&mlist[2])->dst_addr);

			info->dma_length =
				mtd->writesize + mtd->oobsize + sizeof(uint32_t);
		}
	}

	dprintk3("mode %s, length %d\n",
			(mode == DMA_MODE_READ ? "read" : "write"),
			info->dma_length);
	if (area_access_mode == AREA_MODE_SINGLE) {
		dprintk4("sar 0x%08x, dar 0x%08x\n", src_dma_addr, dst_dma_addr);
		if ((retval = mobi_dma_setup_single(info->dma_handle,
						src_dma_addr,
						dst_dma_addr,
						info->dma_length))) {
			error("Failed setting up dma single transfer");
			goto cleandma;
		}
	} else {
		if ((retval = mobi_dma_setup_mlist(info->dma_handle,
						(struct mobi_dma_list *)info->buf_dma_addr,
						3,
						info->dma_length))) {
			error("Failed setting up dma list transfer");
			goto cleandma;
		}
	}
	return retval;

cleandma:
	cleanup_nand_dma(info, area_access_mode);

exit:
	if (retval) {
		if (mlist)
			kfree(mlist);
		if (info->ctrl_reg_mem)
			kfree(info->ctrl_reg_mem);
	}
	return retval;
}

static void do_dma_write(struct mtd_info *mtd,
		const uint8_t *buf, int len, uint8_t area_access_mode)
{
	struct nand_chip *chip = (struct nand_chip *)mtd->priv;
	struct aumb3000_nand_info *info =
		(struct aumb3000_nand_info *)chip->priv;
	int retry = 0, ret = 0;

	/* grab the hmux lock for the duration of the dma transfer */
	NAND_LOCK;

	if (setup_nand_dma(mtd, (uint8_t  *)buf, len,
				DMA_MODE_WRITE, area_access_mode) < 0) {
		NAND_UNLOCK;
		return;
	}

	while (((ret = mobi_dma_enable(info->dma_handle)) < 0) &&
			(retry++ < 3)) {
		dprintk1("mobi_dma_enable returned %d, retry %d\n",
				ret, retry);
		if (ret < 0 && ret != -EBUSY)
			break;
	}
	dprintk1("exiting loop mobi_dma_enable returned %d, retry %d\n",
			ret, retry);

	if (ret >= 0) {
		/* don't return until we get error or xfer complete */
		dprintk1("prepare to wait\n");
		wait_event_interruptible(dma_waitq, dma_waitq_flag != 0);
		if (dma_waitq_flag == MOBI_DMA_EVENT_TRANSFER_ERROR) {
			error("DMA write transfer error, page 0x%x",
					info->cmd.page_addr);
			info->xfer_status = dma_waitq_flag;

		} else {
			info->xfer_status = 0;
		}
		dma_waitq_flag = 0;

		mobi_dma_disable(info->dma_handle);
		dprintk1("dma completed, xfer_status 0x%x\n",
				info->xfer_status);
	} else {
		info->xfer_status = MOBI_DMA_EVENT_TRANSFER_ERROR;
		if (retry == 4)
			error("DMA write retry count exceeded\n");
		else
			error("DMA write error\n");
	}

	cleanup_nand_dma(info, area_access_mode);
	NAND_UNLOCK;
	return;
}

static void do_pio_write(struct mtd_info *mtd,
		const uint8_t *buf, int len, uint8_t area_access_mode)
{
	struct nand_chip *chip = (struct nand_chip *)mtd->priv;
	struct aumb3000_nand_info *info =
		(struct aumb3000_nand_info *)chip->priv;
	const uint32_t *buf32 = (uint32_t *)buf;
	uint64_t nand_data_addr = 0x0;
	uint32_t amba_addr = 0x0;
	int i;

	/* grab the hmux lock for the duration of the dma transfer */
	NAND_LOCK;

	nand_data_addr = (info->cmd.page_addr << chip->page_shift) +
		(info->select_chip*chip->chipsize);
	dprintk4("select_chip %d, page 0x%x, nand_addr 0x%llx\n",
			info->select_chip, info->cmd.page_addr, nand_data_addr);
	if (mtd->writesize == 512 &&
			(MOBI_FLASH_NAND_CONTROL_SA_R(info->ctrl_shadow))) {
		nand_data_addr |= ((info->cmd.page_addr << 4) & 0xff);
		dprintk4("small page modified nand_addr 0x%llx\n", nand_data_addr);
	}
	/* bits [33:26] of the address are programmed into the control reg */
	MOBI_FLASH_NAND_CONTROL_ADDR_CLR_AND_SET(info->ctrl_shadow,
			((nand_data_addr >> 26) & 0xff));
	CONTROL_WRITE(info->ctrl_shadow);

	amba_addr = (uint32_t)flashdata_base |
		(uint32_t)(nand_data_addr & 0x3ffffff);

	dprintk4("ctrl_addr_bits 0x%x, amba address 0x%x\n",
			((uint32_t)(nand_data_addr >> 26) & 0xff), amba_addr);

	dprintk4("Data writebuf, len = %d bytes\n", len);
	/* length in bytes, make it words */
	len >>= 2; 

	for (i = 0; i < len; i++) {
		writel(cpu_to_be32(buf32[i]), amba_addr+(i*4));
	}

	if (area_access_mode == AREA_MODE_MULTIPLE) {
		if (chip->oob_poi) {
			if (mtd->writesize == 512) {
				nand_data_addr |= ((info->cmd.page_addr << 4) & 0xff);
				amba_addr = (uint32_t)flashdata_base |
					(uint32_t)(nand_data_addr & 0x3ffffff);
			}
			dprintk4("nand_addr 0x%llx amba address 0x%x\n",
					nand_data_addr, amba_addr);
			/* enable spare area access */
			MOBI_FLASH_NAND_CONTROL_SA_SET(info->ctrl_shadow);
			CONTROL_WRITE(info->ctrl_shadow);
			len = mtd->oobsize >> 2;
			buf32 = (uint32_t *)chip->oob_poi;
			for (i = 0; i < len; i++) {
				writel(cpu_to_be32(buf32[i]), amba_addr+(i*4));
			}
			MOBI_FLASH_NAND_CONTROL_SA_CLR(info->ctrl_shadow);
			CONTROL_WRITE(info->ctrl_shadow);
		}
	}
	NAND_UNLOCK;
}

static void aumb3000_nand_write_buf(struct mtd_info *mtd,
		const uint8_t *buf, int len)
{
	struct nand_chip *chip = (struct nand_chip *)mtd->priv;
	struct aumb3000_nand_info *info =
		(struct aumb3000_nand_info *)chip->priv;

	dprintk3("mode: single, len: %d\n", len);
	if (info->use_dma) {
		do_dma_write(mtd, buf, len, AREA_MODE_SINGLE);
	} else {
		do_pio_write(mtd, buf, len, AREA_MODE_SINGLE);
	}
}

const static char empty_oob[64] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};
/* we don't have to calculate the ECC and write it to the spare
 *  area like the nand_base function.  The controller should be
 *  taking care of all this as long as it is enabled.  therefor
 *  we do a single write of pagesize instead of breaking the
 *  writes up to calculate the ecc
 */
static void aumb3000_nand_write_page_hwecc(struct mtd_info *mtd,
		struct nand_chip *chip,
		const uint8_t *buf)
{
	struct aumb3000_nand_info *info = (struct aumb3000_nand_info *)chip->priv;
	uint8_t area_access_mode;

	/* there is no guarentee that there is real data in the oob
	   buffer. nand_do_write_ops in nand_base will either set it
	   to 0xff or real data.  since a whole new dma needs to be
	   done to write the oob lets see if it is real data or not.
	   in pio the memcpy is probabl cheaper then than doing all
	   the writes??
   */
	if (chip->oob_poi &&
			(memcmp(empty_oob, chip->oob_poi, mtd->oobsize) != 0)) {
		area_access_mode = AREA_MODE_MULTIPLE;

	} else {
		area_access_mode = AREA_MODE_SINGLE;
	}

	dprintk3("mode: %s, writesize: %d\n",
			(area_access_mode == AREA_MODE_MULTIPLE ? "multiple" : "single"),
			mtd->writesize);
	if (info->use_dma) {
		do_dma_write(mtd, buf, mtd->writesize, area_access_mode);

	} else {
		do_pio_write(mtd, buf, mtd->writesize, area_access_mode);
	}
}

static void aumb3000_nand_write_page_raw(struct mtd_info *mtd,
		struct nand_chip *chip, const uint8_t *buf)
{
	struct aumb3000_nand_info *info =
		(struct aumb3000_nand_info *)chip->priv;
	/* always write main and spare area from here */
	dprintk3("mode: multiple\n");
	if (info->use_dma) {
		do_dma_write(mtd, buf, mtd->writesize, AREA_MODE_MULTIPLE);

	} else {
		do_pio_write(mtd, buf, mtd->writesize, AREA_MODE_MULTIPLE);
	}
}

/**
 * \brief aumb3000_nand_write_oob_std:
 * 	the most common OOB data write function
 * \param mtd   - mtd info structure
 * \param chip  - nand chip info structure
 * \param page  - page number to write
 *
 * \remark
 * 	When accessing data in the spare area we have to set a
 * bit in the control register to indicate the the address is a
 * spare area address.
 *
 */
static int aumb3000_nand_write_oob_std(struct mtd_info *mtd,
		struct nand_chip *chip, int page)
{
	int status = 0, ret = 0;
	struct aumb3000_nand_info *info =
		(struct aumb3000_nand_info *)chip->priv;

	/*
	 * just a note on the steps going on here, looking at nand_base
	 *  the seqence would be send command SEQIN, write_buf, send command
	 *  PROG.  SEQ is a nop for the amb3000 but we do need to save the
	 *  page address(which we do below), writebuf calls setup_dma which
	 *  then handles the address correctly and the PROG is sent below
	 *  to flush the data
	 *
	 *  another note.  nand_write_buf is used to write to main area
	 *  or spare area or main and spare area.  when writing to a single
	 *  area, the functions just write a specific sized buffer to
	 *  the data area.  so when we call write_buf from here we take
	 *  care of setting the control bit correctly so that write_buf
	 *  does not need to handle it.
	 */

	dprintk3("writing page 0x%x(%d), nand_addr 0x%x\n",
			page, page, page << chip->page_shift);
	info->cmd.page_addr = page;
	/* writing to the spare area, set the bit in the ctrl reg */
	MOBI_FLASH_NAND_CONTROL_SA_SET(info->ctrl_shadow);
	CONTROL_WRITE(info->ctrl_shadow);
	chip->write_buf(mtd, chip->oob_poi, mtd->oobsize);

	/* force flushing, nand_data address and program command */
	dprintk4("send PROGRAM, address = 0x%lx\n",
			((info->cmd.page_addr << chip->page_shift) +
			 (info->select_chip*chip->chipsize)));

	ret = cmd_write(AUMB3000_NAND_CMD_PROGRAM,
			((info->cmd.page_addr << chip->page_shift) +
			 (info->select_chip*chip->chipsize)));

	/* turn off spare area address bit */
	MOBI_FLASH_NAND_CONTROL_SA_CLR(info->ctrl_shadow);
	CONTROL_WRITE(info->ctrl_shadow);
	if (ret) {
		ret = -EBUSY;

	} else {
		status = chip->waitfunc(mtd, chip);
		if (status & NAND_STATUS_FAIL) {
			ret = -EIO;

		} else if (info->use_dma &&
				(info->xfer_status == MOBI_DMA_EVENT_TRANSFER_ERROR)) {
			ret = -EIO;
		}
	}

	return ret;
}

/**
 * nand_write_page - [REPLACEABLE] write one page
 * @mtd:	MTD device structure
 * @chip:	NAND chip descriptor
 * @buf:	the data to write
 * @page:	page number to write
 * @cached:	cached programming
 * @raw:	use _raw version of write_page
 */
static int aumb3000_nand_write_page(struct mtd_info *mtd,
		struct nand_chip *chip,
		const uint8_t *buf, int page, int cached, int raw)
{
	struct aumb3000_nand_info *info =
		(struct aumb3000_nand_info *)chip->priv;
	int status = 0, ret = 0;

	chip->cmdfunc(mtd, NAND_CMD_SEQIN, 0x00, page);

	if (unlikely(raw))
		chip->ecc.write_page_raw(mtd, chip, buf);
	else
		chip->ecc.write_page(mtd, chip, buf);

	chip->cmdfunc(mtd, NAND_CMD_PAGEPROG, -1, -1);
	status = chip->waitfunc(mtd, chip);

	if (status & NAND_STATUS_FAIL) {
		dprintk2("nand_status_fail\n");
		ret = -EIO;
	} else if (info->use_dma &&
			(info->xfer_status == MOBI_DMA_EVENT_TRANSFER_ERROR)) {
		dprintk2("xfer_status = %d\n", info->xfer_status);
		ret = -EIO;
	}

	return ret;
}

/* these functions will be use to read/write data */
static void do_dma_read(struct mtd_info *mtd,
		uint8_t *buf, int len, uint8_t area_access_mode)
{
	struct nand_chip *chip = (struct nand_chip *)mtd->priv;
	struct aumb3000_nand_info *info =
		(struct aumb3000_nand_info *)chip->priv;
	int retry = 0, ret = 0;

	NAND_LOCK;
	if (setup_nand_dma(mtd, buf, len, DMA_MODE_READ, area_access_mode) < 0) {
		NAND_UNLOCK;
		return;
	}

	while (((ret = mobi_dma_enable(info->dma_handle)) < 0) &&
			(retry++ < 3)) {
		dprintk1("mobi_dma_enable returned %d, retry %d\n",
				ret, retry);
		if (ret < 0 && ret != -EBUSY)
			break;
	}
	dprintk1("exiting loop mobi_dma_enable returned %d, retry %d\n",
			ret, retry);

	if (ret >= 0) {
		/* don't return until we get error or xfer complete */
		wait_event_interruptible(dma_waitq, dma_waitq_flag != 0);
		if (dma_waitq_flag == MOBI_DMA_EVENT_TRANSFER_ERROR) {
			error("DMA read transfer error, page 0x%x",
					info->cmd.page_addr);
			info->xfer_status = dma_waitq_flag;
		} else {
			info->xfer_status = 0;
		}
		dma_waitq_flag = 0;

		mobi_dma_disable(info->dma_handle);
		dprintk1("dma completed, xfer_status 0x%x\n",
				info->xfer_status);
	} else {
		info->xfer_status = MOBI_DMA_EVENT_TRANSFER_ERROR;
		if (retry == 4)
			error("DMA read retry count exceeded\n");
		else
			error("DMA read error\n");
	}

	cleanup_nand_dma(info, area_access_mode);
	NAND_UNLOCK;
	return;
}

static void do_pio_read(struct mtd_info *mtd,
		uint8_t *buf, int len, uint8_t area_access_mode)
{
	struct nand_chip *chip = (struct nand_chip *)mtd->priv;
	struct aumb3000_nand_info *info =
		(struct aumb3000_nand_info *)chip->priv;
	uint32_t *buf32_main = (uint32_t *)buf;
	uint32_t *buf32_oob  = (uint32_t*)chip->oob_poi;
	uint64_t nand_data_addr = 0x0;
	uint32_t amba_addr = 0x0;
	int i;

	NAND_LOCK;
	nand_data_addr = (info->cmd.page_addr << chip->page_shift) +
		(info->select_chip*chip->chipsize);
	if (mtd->writesize == 512 &&
			(MOBI_FLASH_NAND_CONTROL_SA_R(info->ctrl_shadow))) {
		nand_data_addr |= ((info->cmd.page_addr << 4) & 0xff);
		dprintk3("small page modified nand_addr 0x%llx\n", nand_data_addr);
	}
	/* bits [33:26] of the address are programmed into the control reg */
	MOBI_FLASH_NAND_CONTROL_ADDR_CLR_AND_SET(info->ctrl_shadow,
			((nand_data_addr >> 26) & 0xff));
	CONTROL_WRITE(info->ctrl_shadow);

	amba_addr = (uint32_t)flashdata_base +
		(uint32_t)(nand_data_addr & 0x3ffffff);

	dprintk4("select_chip %d, page 0x%x, nand_addr 0x%llx\n",
			info->select_chip, info->cmd.page_addr, nand_data_addr);
	dprintk4("ctrl_addr_bits 0x%x, amba address 0x%x\n",
			((uint32_t)(nand_data_addr >> 26) & 0xff), amba_addr);

	dprintk4("Data readbuf, len = %d bytes\n", len);
	len >>= 2;
	for (i = 0; i < len; i++) {
		buf32_main[i] = be32_to_cpu(readl(amba_addr+(i*4)));
	}

	if (area_access_mode == AREA_MODE_MULTIPLE) {
		if (mtd->writesize == 512) {
			nand_data_addr |= ((info->cmd.page_addr << 4) & 0xff);
			amba_addr = (uint32_t)flashdata_base |
				(uint32_t)(nand_data_addr & 0x3ffffff);
		}
		dprintk4("nand_addr 0x%llx amba address 0x%x\n",
				nand_data_addr, amba_addr);

		/* enable spare area access */
		MOBI_FLASH_NAND_CONTROL_SA_SET(info->ctrl_shadow);
		CONTROL_WRITE(info->ctrl_shadow);

		len = mtd->oobsize >> 2;
		for (i = 0; i < len; i++) {
			buf32_oob[i] = be32_to_cpu(readl(amba_addr+(i*4)));
		}
		/* turn off spare area address bit */
		MOBI_FLASH_NAND_CONTROL_SA_CLR(info->ctrl_shadow);
		CONTROL_WRITE(info->ctrl_shadow);
	}

	NAND_UNLOCK;
}

/* these functions will be use to read/write data */
static void aumb3000_nand_read_buf(struct mtd_info *mtd, uint8_t *buf, int len)
{
	struct nand_chip *chip = (struct nand_chip *)mtd->priv;
	struct aumb3000_nand_info *info = (struct aumb3000_nand_info *)chip->priv;

	dprintk3("mode: single, len: %d\n", len);
	if (info->use_dma) {
		do_dma_read(mtd, buf, len, AREA_MODE_SINGLE);
	} else {
		do_pio_read(mtd, buf, len, AREA_MODE_SINGLE);
	}
}

/**
 * \brief aumb3000_nand_read_page_hwecc
 * 	Read a page from nand with hardware ecc enabled.
 *
 * \param mtd :	mtd info structure
 * \param chip:	nand chip info structure
 * \param buf :	buffer to store read data
 *
 * \retval Zero   - upon success
 *
 * \remark
 *    Our Flash controller takes care off all the ecc stuff,
 * so if ecc checking and correction is enabled, we do nothing but
 * a simple read(or read loop)
 *   The path to this func is mtd->read which is nand_read in
 * nand_base.c. nand_read calls nand_do_read_ops which calls
 * chip->ecc.read_page which points to this function
 *
 */
static int aumb3000_nand_read_page_hwecc(struct mtd_info *mtd,
		struct nand_chip *chip,
		uint8_t *buf)
{
	struct aumb3000_nand_info *info = (struct aumb3000_nand_info *)chip->priv;
	uint8_t area_access_mode;
	int ret = 0;

	if (chip->oob_poi) {
		area_access_mode = AREA_MODE_MULTIPLE;
	} else {
		area_access_mode = AREA_MODE_SINGLE;
	}
	dprintk3("mode: %s, writesize: %d\n",
			(area_access_mode == AREA_MODE_MULTIPLE ? "multiple" : "single"),
			mtd->writesize);

	if (info->use_dma) {
		do_dma_read(mtd, buf, mtd->writesize, area_access_mode);
		if (info->xfer_status == MOBI_DMA_EVENT_TRANSFER_ERROR)
			ret = -EIO;
	} else
		do_pio_read(mtd, buf, mtd->writesize, area_access_mode);

	return ret;
}

static int aumb3000_nand_read_page_raw(struct mtd_info *mtd,
		struct nand_chip *chip, uint8_t *buf)
{
	struct aumb3000_nand_info *info = (struct aumb3000_nand_info *)chip->priv;
	int ret = 0;

	/* always write main and spare area from here */
	dprintk3("mode: multiple\n");
	if (info->use_dma) {
		do_dma_read(mtd, buf, mtd->writesize, AREA_MODE_MULTIPLE);
		if (info->xfer_status == MOBI_DMA_EVENT_TRANSFER_ERROR)
			ret = -EIO;
	} else {
		do_pio_read(mtd, buf, mtd->writesize, AREA_MODE_MULTIPLE);
	}
	return ret;
}

/**
 * \brief aumb3000_nand_read_oob_std:
 * 	the most common OOB data read function
 * \param mtd    - mtd info structure
 * \param chip   - nand chip info structure
 * \param page   - page number to read
 * \param sndcmd - flag whether to issue read command or not
 *
 * \remark
 * 	When accessing data in the spare area we have to set a
 * bit in the control register to indicate the the address is a
 * spare area address.
 *
 */
static int aumb3000_nand_read_oob_std(struct mtd_info *mtd,
		struct nand_chip *chip, int page, int sndcmd)
{
	struct aumb3000_nand_info *info =
		(struct aumb3000_nand_info *)chip->priv;

	dprintk3("Reading page 0x%x(%d), nand_addr 0x%x\n",
			page, page, page << chip->page_shift);
	info->cmd.page_addr = page;
	MOBI_FLASH_NAND_CONTROL_SA_SET(info->ctrl_shadow);
	CONTROL_WRITE(info->ctrl_shadow);

	chip->read_buf(mtd, chip->oob_poi, mtd->oobsize);

	/* turn off spare area address bit */
	MOBI_FLASH_NAND_CONTROL_SA_CLR(info->ctrl_shadow);
	CONTROL_WRITE(info->ctrl_shadow);

	if (info->use_dma &&
			(info->xfer_status == MOBI_DMA_EVENT_TRANSFER_ERROR)) {
		return -EIO;
	}

	return 0;
}

/**
 * \brief aumb3000_nand_read_byte:
 * 	Read a byte size chunch from the NAND chip
 *
 * \param mtd - pointer to mtd struct for this device
 *
 * \retval data value upon success
 * \retval -EINVAL on error
 * \retval -EIO on IO access error
 *
 * \remark
 * 	This function has to handle reads from both controller
 * and data space.  Since we need to emulate having a direct
 * connection to the chip things are a bit convoluted.  We have
 * to store the last/current command that was sent to the command
 * function.  In direct connection this will set the address lines
 * accordingly so the correct chunk of data is read. So we have to
 * save that command so we know where to read from when the read
 * command is issued.
 * 	We only have to deal with three different read commands.
 * Two control types, READID and STATUS, and data reads.
 */
static uint8_t aumb3000_nand_read_byte(struct mtd_info *mtd)
{
	struct nand_chip *chip = (struct nand_chip *)mtd->priv;
	struct aumb3000_nand_info *info = (struct aumb3000_nand_info *)chip->priv;
	uint8_t ret = 0;

	if (info->cmd.nand_cmd == NAND_CMD_READID) {
		/*
		 *  we get back 4 bytes, each consecutive read returns the
		 *  idea of the next chip
		 *  only read the register the first time read_byte
		 *  is called with command READID
		 */
		if (info->cmd.last_cmd_byte_read == 3) {
			info->cmd.read_data =
				REG_READ_LOCKED(MOBI_FLASH_NAND_ID_OFFSET);
		}

		ret = (info->cmd.read_data >> (info->cmd.last_cmd_byte_read*8)) & 0xff;
		dprintk2("cmd READID(0x%x), chip %d, byte %d, ",
				info->cmd.nand_cmd, info->select_chip,
				info->cmd.last_cmd_byte_read);
		/*
		 *  ID is accessed via single byte reads so have to
		 *  keep track of last byte we returned.  read start with
		 *  "msb"
		 */
		info->cmd.last_cmd_byte_read--;
		info->cmd.last_cmd_byte_read &= 0x3;

	} else if (info->cmd.nand_cmd == NAND_CMD_STATUS) {
		/*
		 *  status is returned for all chips in a bank, the
		 *  chip select determines which bank is the one
		 *  of interest
		 */
		info->cmd.read_data = REG_READ_LOCKED(MOBI_FLASH_NAND_STATUS_OFFSET);

		dprintk3("STATUS(0x%x), chip %d, status 0x%08x ",
				info->cmd.nand_cmd, info->select_chip, info->cmd.read_data);

		ret = ((info->cmd.read_data >> (info->select_chip*8)) & 0xff);

	} else if (info->cmd.nand_cmd == NAND_CMD_READOOB) {
		/* page and column set when command sent in */
		/* malloc a oobsize buffer, and point chip->oob_poi
		*/
		chip->oob_poi = (uint8_t *) kmalloc(mtd->oobsize, GFP_KERNEL);
		if (chip->oob_poi != NULL) {
			if (aumb3000_nand_read_oob_std(mtd, chip,
						info->cmd.page_addr, 0) >= 0)
				ret = chip->oob_poi[info->cmd.readoob_pos];
			else
				ret = -EINVAL;
			kfree(chip->oob_poi);
		}
	} else {
		ret = -EINVAL;
	}

	/* doesn't seem to really be an error code here since it is just returning
	 *  the byte that was read, however what if there was an error reading??
	 */
	dprintk3("ret: 0x%x\n", (uint8_t) ret);
	return (uint8_t) ret;
}

static uint16_t aumb3000_nand_read_word(struct mtd_info *mtd)
{
	struct nand_chip *chip = (struct nand_chip *)mtd->priv;
	struct aumb3000_nand_info *info = (struct aumb3000_nand_info *)chip->priv;
	uint16_t ret = 0;

	if (info->xs == 0) {
		return -EACCES;
	} else {
		if (info->cmd.nand_cmd == NAND_CMD_READOOB) {
			/* page and column set when command sent in */
			/* malloc a oobsize buffer, and point chip->oob_poi
			*/
			chip->oob_poi =
				(uint8_t *) kmalloc(mtd->oobsize, GFP_KERNEL);
			if (chip->oob_poi != NULL) {
				if (aumb3000_nand_read_oob_std(mtd, chip, info->cmd.page_addr, 0) >= 0)
					ret = *((uint16_t *)chip->oob_poi+info->cmd.readoob_pos);
				else
					ret = -EINVAL;
				kfree(chip->oob_poi);
			}
		} else {
			error("Unsupported read word request");
			ret = -EINVAL;
		}
	}

	return ret;
}

/**
 * nand_command - [DEFAULT] Send command to NAND device
 * @mtd:	MTD device structure
 * @command:	the command to be sent
 * @column:	the column address for this command, -1 if none
 * @page_addr:	the page address for this command, -1 if none
 *
 */
static void aumb3000_nand_command(struct mtd_info *mtd,
		uint32_t command, int column, int page_addr)
{
	struct nand_chip *chip = mtd->priv;
	struct aumb3000_nand_info *info = chip->priv;
	uint64_t nand_cmd_addr = 0x0;
	int ret = 0;

	info->cmd.page_addr = page_addr;
	switch (command) {
	case NAND_CMD_READID:
		dprintk2("CMD_READID(0x%x)\n", command);
		info->cmd.page_addr = info->select_chip;
		info->cmd.last_cmd_byte_read = 3;
		info->cmd.au3000_cmd = AUMB3000_NAND_CMD_READID;
		break;

	case NAND_CMD_STATUS:
		dprintk2("CMD_STATUS(0x%x)\n", command);
		info->cmd.page_addr = info->select_chip;
		info->cmd.au3000_cmd = AUMB3000_NAND_CMD_RDSTATUS;
		break;

	case NAND_CMD_RESET:
		dprintk2("CMD_RESET(0x%x)\n", command);
		info->cmd.au3000_cmd = AUMB3000_NAND_CMD_RESET;
		break;

	case NAND_CMD_ERASE1:
		dprintk2("CMD_ERASE1(0x%x)\n", command);
		/* erase1, grab address to use w/erase2 */
		info->cmd.erase_addr = page_addr;
		info->cmd.au3000_cmd = AUMB3000_NAND_CMD_NOP;
		break;

	case NAND_CMD_ERASE2:
		dprintk2("CMD_ERASE2(0x%x)\n", command);
		/* erase address comes in on ERASE1 cmd, just put it
		 *  in page_addr for ease of processing below
		 */
		info->cmd.page_addr = info->cmd.erase_addr;
		info->cmd.au3000_cmd = AUMB3000_NAND_CMD_ERASE;
		break;

		/* notes about programming
		 *  seqin prepares us for the write, need to save the address
		 *  a column can be passed into this function so at some point
		 *  need to handle that but don't worry for now, columns are
		 *  probably only passed in via write_oob, but our write_oob
		 *  will never call this function
		 *
		 *  the data is then writen via ecc.write_page
		 *
		 *  pageprog is used to flush the data so that it is fully
		 *  written to the chip.
		 *
		 *    first, we just need to get the address, writes will be done
		 *  via amba pios or dma, no command to send to the controller
		 *    second, we can also ignore pageprog because we enabled
		 *  automatic writes in the controller which means it will take
		 *  care of any flushing.
		 *
		 *  if the write is going to be to the spare area we will need to
		 *  toggle the spare area address bit in the control register.
		 *  read/write oob is accessed thru the specific function so we
		 *  can handle setting controls bits there and then the actual
		 *  address will cover the middle of the page read
		 */
	case NAND_CMD_SEQIN:
		dprintk2("CMD_SEQIN(0x%x)\n", command);
		if (column >= mtd->writesize) {
			/* OOB area */
			column -= mtd->writesize;
		} else if (column >= mtd->writesize/2) {
			/* read in middle of page, READ1 */
			column -= mtd->writesize/2;
		} else {
			/* READ0 */
		}
		info->cmd.seq_addr = page_addr;
		info->cmd.au3000_cmd = AUMB3000_NAND_CMD_NOP;
		break;

	case NAND_CMD_READ0:
	case NAND_CMD_READ1:
		dprintk2("CMD_READ0/1(0x%x)\n", command);
		info->cmd.au3000_cmd = AUMB3000_NAND_CMD_NOP;
		break;

		/* our read_oob doesn't call this function but nand_block_bad
		 *  does and the column is the position of the bad block indicator
		 *  in the spare area.  save the info and handle when read_byte is
		 *  called
		 */
	case NAND_CMD_READOOB:
		dprintk2("CMD_READOOB(0x%x)\n", command);
		info->cmd.au3000_cmd = AUMB3000_NAND_CMD_NOP;
		info->cmd.readoob_pos = column;
		break;

	case NAND_CMD_PAGEPROG:
		/* even though au is enabled, probably should handle
		 *  this command just to make sure we flush the data as
		 *  the mtd layer expects.  There is one SEQ cmd in the
		 *  controller so we have to grab the address when the
		 *  command is sent to the driver and then use it when we
		 *  want to flush the chip register
		 */
		dprintk2("CMD_PAGEPROG(0x%x)\n", command);
		info->cmd.page_addr = info->cmd.seq_addr;
		info->cmd.au3000_cmd = AUMB3000_NAND_CMD_PROGRAM;
		break;
	case NAND_CMD_RNDIN:
	case NAND_CMD_RNDOUT:
	case NAND_CMD_STATUS_MULTI:
		dprintk2("CMD_OTHERS(0x%x)\n", command);
		info->cmd.au3000_cmd = AUMB3000_NAND_CMD_NOP;
		break;
	default:
		info->cmd.errno = -EINVAL;
		return;
		break;
	}
	/* save real nand command */
	info->cmd.nand_cmd = command;
	dprintk2("column %d, incoming page 0x%x, cmd.page_addr 0x%x\n",
			column, page_addr, info->cmd.page_addr);

	/* send the command to the cmd reg */
	if (info->cmd.au3000_cmd != AUMB3000_NAND_CMD_NOP) {
		/*
		 *  to suppor > 4G of flash we need to put the
		 *  upper 2 bits of address into the control register
		 */
		nand_cmd_addr = (info->cmd.page_addr << chip->page_shift) +
			(info->select_chip*chip->chipsize);
		MOBI_FLASH_NAND_CONTROL_ADDR_CLR_AND_SET(info->ctrl_shadow,
				((nand_cmd_addr >> 26) & 0xff));
		CONTROL_WRITE(info->ctrl_shadow);
		dprintk2("select_chip %d, au3000_cmd 0x%x, address 0x%llx\n",
				info->select_chip, info->cmd.au3000_cmd, nand_cmd_addr);

		ret = cmd_write(info->cmd.au3000_cmd, nand_cmd_addr);
		if (ret)
			info->cmd.errno = -EBUSY;

		/* sometimes we need to do some post-command processing */
		switch (command) {
		case NAND_CMD_ERASE2:
			info->cmd.erase_addr = 0x0;
			break;
		}
	} else {
		dprintk3("au3000_cmd set to NOP\n");
	}

	return;
}

#ifdef CONFIG_PROC_FS
static void proc_advance_ptr(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	while (((**ptr == ' ') || (**ptr == '\t') || (**ptr == '\r')
				|| (**ptr == '\n')) && (*ptr-buf < count)) {
		(*ptr)++;
	}
}

static int proc_get_next_arg(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	proc_advance_ptr(ptr, buffer, count);
	if (*ptr-buf >= count) {
		return -EINVAL;
	}
	return 0;
}

/* get and return the next argument as an unsigned long */
static int get_ularg(char **ptr, uint32_t buffer,
		unsigned long count, uint32_t *arg)
{
	char *buf = (char *)buffer;
	proc_advance_ptr(ptr, buffer, count);
	if (*ptr-buf >= count) {
		error("Invalid argument string");
		return -1;
	} else {
		*arg = simple_strtoul(*ptr, ptr, 0);
		if (*arg < 0) {
			error("invalid argument");
			return -1;
		}
	}
	return 0;
}

static int aumb3000_proc_rd_timerintval(char *buf, char **start,
		off_t offset, int count, int *eof, void *data)
{
	int len = 0;
	struct nand_chip *chip;
	struct aumb3000_nand_info *info;
	chip = (struct nand_chip *)(&aumb3000_mtd[1]);
	/* and then the pointer to our private data */
	info = (struct aumb3000_nand_info *) (&chip[1]);

	len += sprintf(buf+len, "\n");
	len += sprintf(buf+len, "ns_per_cycle: %d\n", info->ns_per_cycle);
	len += sprintf(buf+len, "timing reg 0: 0x%08x\n",
			REG_READ(MOBI_FLASH_NAND_TIMERINTVAL0_OFFSET));
	len += sprintf(buf+len, "timing reg 1: 0x%08x\n",
			REG_READ(MOBI_FLASH_NAND_TIMERINTVAL1_OFFSET));
	len += sprintf(buf+len, "timing reg 2: 0x%08x\n",
			REG_READ(MOBI_FLASH_NAND_TIMERINTVAL2_OFFSET));
	len += sprintf(buf+len, "timing reg 3: 0x%08x\n",
			REG_READ(MOBI_FLASH_NAND_TIMERINTVAL3_OFFSET));
	len += sprintf(buf+len, "timing reg 4: 0x%08x\n",
			REG_READ(MOBI_FLASH_NAND_TIMERINTVAL4_OFFSET));
	len += sprintf(buf+len, "timing reg 5: 0x%08x\n",
			REG_READ(MOBI_FLASH_NAND_TIMERINTVAL5_OFFSET));

	*eof = 1;
	return len;
}

static int aumb3000_proc_wr_timerintval(struct file *file,
		const char *buffer, unsigned long count, void *data)
{
	char *ptr = (char *)buffer;
	char *ptr2;
	char timer[8];
	uint32_t value = 0, err = 0;
	uint16_t timer_reg;
	struct nand_chip *chip = (struct nand_chip *)(&aumb3000_mtd[1]);
	struct aumb3000_nand_info *info =
		(struct aumb3000_nand_info *) (&chip[1]);

	if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0) {

		ptr2 = ptr;
		/* get end of string so we can determine length */
		while (((*ptr != ' ') && (*ptr != '\t') && (*ptr != '\r')
					&& (*ptr != '\n')) && (ptr-buffer < count))
			ptr++;

		memset(timer, 0x0, 8);
		strncpy(timer, ptr2, ptr-ptr2);
		get_ularg(&ptr, (uint32_t)buffer, count, &value);
		value &= 0xff;
	}

#define TIMING_INTVAL0_REG 0x01
#define TIMING_INTVAL1_REG 0x02
#define TIMING_INTVAL2_REG 0x04
#define TIMING_INTVAL3_REG 0x08
#define TIMING_INTVAL4_REG 0x10
#define TIMING_INTVAL5_REG 0x20
#define TIMING_INTVAL_REG_ALL_MASK 0x3f

	modparam_set_timing_bitmask = 0x0;
	if (!strcmp(timer, "tds")) {
		default_timings[NAND_TIMING_TDS] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TDS_SET;
		timer_reg = TIMING_INTVAL0_REG;

	} else if (!strcmp(timer, "tcs")) {
		default_timings[NAND_TIMING_TCS] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TCS_SET;
		timer_reg = TIMING_INTVAL0_REG;

	} else if (!strcmp(timer, "tals")) {
		default_timings[NAND_TIMING_TALS] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TALS_SET;
		timer_reg = TIMING_INTVAL0_REG;

	} else if (!strcmp(timer, "tcls")) {
		default_timings[NAND_TIMING_TCLS] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TCLS_SET;
		timer_reg = TIMING_INTVAL0_REG;

	} else if (!strcmp(timer, "tdh")) {
		default_timings[NAND_TIMING_TDH] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TDH_SET;
		timer_reg = TIMING_INTVAL1_REG;

	} else if (!strcmp(timer, "tch")) {
		default_timings[NAND_TIMING_TCH] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TCH_SET;
		timer_reg = TIMING_INTVAL1_REG;

	} else if (!strcmp(timer, "talh")) {
		default_timings[NAND_TIMING_TALH] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TALH_SET;
		timer_reg = TIMING_INTVAL1_REG;

	} else if (!strcmp(timer, "tclh")) {
		default_timings[NAND_TIMING_TCLH] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TCLH_SET;
		timer_reg = TIMING_INTVAL1_REG;

	} else if (!strcmp(timer, "trr")) {
		default_timings[NAND_TIMING_TRR] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TRR_SET;
		timer_reg = TIMING_INTVAL2_REG;

	} else if (!strcmp(timer, "twb")) {
		default_timings[NAND_TIMING_TWB] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TWB_SET;
		timer_reg = TIMING_INTVAL2_REG;

	} else if (!strcmp(timer, "twh")) {
		default_timings[NAND_TIMING_TWH] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TWH_SET;
		timer_reg = TIMING_INTVAL2_REG;

	} else if (!strcmp(timer, "twp")) {
		default_timings[NAND_TIMING_TWP] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TWP_SET;
		timer_reg = TIMING_INTVAL2_REG;

	} else if (!strcmp(timer, "tceh")) {
		default_timings[NAND_TIMING_TCEH] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TCEH_SET;
		timer_reg = TIMING_INTVAL3_REG;

	} else if (!strcmp(timer, "trb")) {
		default_timings[NAND_TIMING_TRB] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TRB_SET;
		timer_reg = TIMING_INTVAL3_REG;

	} else if (!strcmp(timer, "treh")) {
		default_timings[NAND_TIMING_TREH] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TREH_SET;
		timer_reg = TIMING_INTVAL3_REG;

	} else if (!strcmp(timer, "trp")) {
		default_timings[NAND_TIMING_TRP] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TRP_SET;
		timer_reg = TIMING_INTVAL3_REG;

	} else if (!strcmp(timer, "tir")) {
		default_timings[NAND_TIMING_TIR] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TIR_SET;
		timer_reg = TIMING_INTVAL4_REG;

	} else if (!strcmp(timer, "twhr")) {
		default_timings[NAND_TIMING_TWHR] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TWHR_SET;
		timer_reg = TIMING_INTVAL4_REG;

	} else if (!strcmp(timer, "tclr")) {
		default_timings[NAND_TIMING_TCLR] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TCLR_SET;
		timer_reg = TIMING_INTVAL4_REG;

	} else if (!strcmp(timer, "trdelay")) {
		default_timings[NAND_TIMING_TRDELAY] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TRDELAY_SET;
		timer_reg = TIMING_INTVAL4_REG;

	} else if (!strcmp(timer, "tar")) {
		default_timings[NAND_TIMING_TAR] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TAR_SET;
		timer_reg = TIMING_INTVAL5_REG;

	} else if (!strcmp(timer, "trhz")) {
		default_timings[NAND_TIMING_TRHZ] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TRHZ_SET;
		timer_reg = TIMING_INTVAL5_REG;

	} else if (!strcmp(timer, "tww")) {
		default_timings[NAND_TIMING_TWW] = value;
		modparam_set_timing_bitmask = NAND_TIMING_TWW_SET;
		timer_reg = TIMING_INTVAL5_REG;

	} else {
		error("Unknow timing interval specified: %s", timer);
		err = 1;
	}

	if (err == 0)
		program_timing_intervals(info, timer_reg);

	while (ptr-buffer < count)
		ptr++;

	return ptr - buffer;
}

#define ECC_CHK_MAIN  0x1
#define ECC_CHK_SPARE 0x2
#define ECC_GEN_MAIN  0x3
#define ECC_GEN_SPARE 0x4

static void set_ecc_state(uint32_t ecc_type, uint8_t enable)
{
	struct nand_chip *chip;
	struct aumb3000_nand_info *info;
	uint32_t value = 0x0;
	chip = (struct nand_chip *)(&aumb3000_mtd[1]);
	/* and then the pointer to our private data */
	info = (struct aumb3000_nand_info *) (&chip[1]);

	switch (ecc_type) {
	case ECC_CHK_MAIN:
		value = MOBI_FLASH_NAND_CONTROL_EC_R(info->ctrl_shadow);
		MOBI_FLASH_NAND_CONTROL_EC_CLR(info->ctrl_shadow);
		value &= ~(0x2); /* clear main area bit */
		value |= (enable << 1);
		info->ctrl_shadow |= MOBI_FLASH_NAND_CONTROL_EC_W(value);
		info->check_ecc = MOBI_FLASH_NAND_CONTROL_EC_R(info->ctrl_shadow);
		break;
	case ECC_CHK_SPARE:
		value = MOBI_FLASH_NAND_CONTROL_EC_R(info->ctrl_shadow);
		MOBI_FLASH_NAND_CONTROL_EC_CLR(info->ctrl_shadow);
		value &= ~(0x1); /* clear spare area bit */
		value |= enable;
		info->ctrl_shadow |= MOBI_FLASH_NAND_CONTROL_EC_W(value);
		info->check_ecc = MOBI_FLASH_NAND_CONTROL_EC_R(info->ctrl_shadow);
		break;
	case ECC_GEN_MAIN:
		value = MOBI_FLASH_NAND_CONTROL_EG_R(info->ctrl_shadow);
		MOBI_FLASH_NAND_CONTROL_EG_CLR(info->ctrl_shadow);
		value &= ~(0x2); /* clear main area bit */
		value |= (enable << 1);
		info->ctrl_shadow |= MOBI_FLASH_NAND_CONTROL_EG_W(value);
		info->gen_ecc = MOBI_FLASH_NAND_CONTROL_EG_R(info->ctrl_shadow);
		break;
	case ECC_GEN_SPARE:
		value = MOBI_FLASH_NAND_CONTROL_EG_R(info->ctrl_shadow);
		MOBI_FLASH_NAND_CONTROL_EG_CLR(info->ctrl_shadow);
		value &= ~(0x1); /* clear spare area bit */
		value |= enable;
		info->ctrl_shadow |= MOBI_FLASH_NAND_CONTROL_EG_W(value);
		info->gen_ecc = MOBI_FLASH_NAND_CONTROL_EG_R(info->ctrl_shadow);
		break;
	}
	CONTROL_WRITE(info->ctrl_shadow);
}

static int mobinand_proc_wr_ecc_chk_main(struct file *file,
		const char *buffer, unsigned long count, void *data)
{
	char *ptr = (char *)buffer;
	uint32_t enable = 0;

	if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0) {
		enable = simple_strtoul(ptr, &ptr, 0);
		if(enable != 0 && enable != 1) {
			error("ECC check main illegal value, must be 0 or 1");
		} else {
			set_ecc_state(ECC_CHK_MAIN, enable);
		}
	}
	while (ptr-buffer<count)
		ptr++;
	return ptr - buffer;
}

static int mobinand_proc_wr_ecc_chk_spare(struct file *file,
		const char *buffer, unsigned long count, void *data)
{
	char *ptr = (char *)buffer;
	uint32_t enable = 0;

	if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0) {
		enable = simple_strtoul(ptr, &ptr, 0);
		if (enable != 0 && enable != 1) {
			error("ECC check spare illegal value, must be 0 or 1");
		} else {
			set_ecc_state(ECC_CHK_SPARE, enable);
		}
	}
	while (ptr-buffer < count)
		ptr++;
	return ptr - buffer;
}

static int mobinand_proc_wr_ecc_gen_main(struct file *file,
		const char *buffer, unsigned long count, void *data)
{
	char *ptr = (char *)buffer;
	uint32_t enable = 0;

	if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0) {
		enable = simple_strtoul(ptr, &ptr, 0);
		if (enable != 0 && enable != 1) {
			error("ECC generatate main illegal value, must be 0 or 1");
		} else {
			set_ecc_state(ECC_GEN_MAIN, enable);
		}
	}
	while (ptr-buffer < count)
		ptr++;
	return ptr - buffer;
}

static int mobinand_proc_wr_ecc_gen_spare(struct file *file,
		const char *buffer, unsigned long count, void *data)
{
	char *ptr = (char *)buffer;
	uint32_t enable = 0;

	if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0) {
		enable = simple_strtoul(ptr, &ptr, 0);
		if (enable != 0 && enable != 1) {
			error("ECC generatate main illegal value, must be 0 or 1");
		} else {
			set_ecc_state(ECC_GEN_SPARE, enable);
		}
	}
	while (ptr-buffer < count)
		ptr++;
	return ptr - buffer;
}

static int mobinand_proc_wr_recover(struct file *file,
		const char *buffer, unsigned long count, void *data)
{
	char *ptr = (char *)buffer;
	struct nand_chip *chip;
	int32_t total_erase_blocks, blocks_erased, pages_done;
	int32_t pagesperblock =
		aumb3000_mtd->erasesize/aumb3000_mtd->writesize;
	uint32_t enable = 0;

	if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0)
		enable = simple_strtoul(ptr, &ptr, 0);
	else
		goto err;

	if (enable != 0x4d4f4249) {
		printk(KERN_ERR "Invalid access code: 0x%x\n", enable);
		goto err;
	}

	chip = (struct nand_chip *)(&aumb3000_mtd[1]);

	total_erase_blocks =
		((MTD_INFO_SIZE_CAST)aumb3000_mtd->size)/aumb3000_mtd->writesize;
	total_erase_blocks /= pagesperblock;

	chip->select_chip(aumb3000_mtd, 0);

	for (blocks_erased = 0, pages_done = 0;
			blocks_erased < total_erase_blocks;
			blocks_erased++, pages_done += pagesperblock) {

		printk(KERN_ERR "\rErasing block %4d at page %6u @ 0x%08x... ",
				pages_done/pagesperblock,
				pages_done, pages_done << chip->page_shift);

		aumb3000_nand_command(aumb3000_mtd, NAND_CMD_ERASE1, -1, pages_done);
		aumb3000_nand_command(aumb3000_mtd, NAND_CMD_ERASE2, -1, -1);
		if (chip->waitfunc(aumb3000_mtd, chip) & NAND_STATUS_FAIL) {
			printk(KERN_ERR "FAILED\n");
		}
	}
	printk(KERN_ERR "done\n");

err:
	while (ptr-buffer < count)
		ptr++;

	return ptr - buffer;
}

static int aumb3000_proc_rd_info(char *buf, char **start,
		off_t offset, int count, int *eof, void *data)
{

	int len = 0;
	struct nand_chip *chip;
	struct aumb3000_nand_info *info;
	chip = (struct nand_chip *)(&aumb3000_mtd[1]);
	/* and then the pointer to our private data */
	info = (struct aumb3000_nand_info *) (&chip[1]);

	len += sprintf(buf+len, "\n");
	len += sprintf(buf+len, "NAND Chip description\n");
	len += sprintf(buf+len, "\tPage size: %d\n", aumb3000_mtd->writesize);
	len += sprintf(buf+len, "\tTotal pages: %d\n",
			((MTD_INFO_SIZE_CAST)aumb3000_mtd->size)/aumb3000_mtd->writesize);
	len += sprintf(buf+len, "\tBlock size: %d\n", aumb3000_mtd->erasesize);
	len += sprintf(buf+len, "\tPages per block : %d\n",
			aumb3000_mtd->erasesize/aumb3000_mtd->writesize);
	len += sprintf(buf+len, "\tTotal blocks: %d\n",
			((MTD_INFO_SIZE_CAST)aumb3000_mtd->size)/aumb3000_mtd->erasesize);
	len += sprintf(buf+len, "\tmtd->name: %s\n", aumb3000_mtd->name);
	len += sprintf(buf+len, "\tmtd->writesize: %d\n", aumb3000_mtd->writesize);
	len += sprintf(buf+len, "\tmtd->oobsize: %d\n", aumb3000_mtd->oobsize);
	len += sprintf(buf+len, "\tmtd->erasesize: %d\n", aumb3000_mtd->erasesize);
	len += sprintf(buf+len, "\tmtd->size: %llu\n", aumb3000_mtd->size);
	len += sprintf(buf+len, "\tchip->options: 0x%x\n", chip->options);
	len += sprintf(buf+len, "\tchip->chipsize: 0x%llx\n", chip->chipsize);
	len += sprintf(buf+len, "\tchip->numchips: 0x%x\n", chip->numchips);
	len += sprintf(buf+len, "\tchip->page_shift: 0x%x\n", chip->page_shift);
	len += sprintf(buf+len, "\tchip->pagemask: 0x%x\n", chip->pagemask);
	len += sprintf(buf+len, "\tchip->bbt_erase_shift: %d\n",  chip->bbt_erase_shift);
	len += sprintf(buf+len, "\tchip->chip_shift: %d\n",  chip->chip_shift);
	len += sprintf(buf+len, "\tchip->badblockpos: %d\n",  chip->badblockpos);
	len += sprintf(buf+len, "\tchip->numchips: %d\n",  chip->numchips);
	len += sprintf(buf+len, "\tchip->ecc.mode: 0x%x\n",  chip->ecc.mode);
	len += sprintf(buf+len, "\tchip->ecc.size: 0x%x\n",  chip->ecc.size);
	len += sprintf(buf+len, "\tchip->ecc.steps: 0x%x\n",  chip->ecc.steps);
	len += sprintf(buf+len, "\tchip->ecc.total: 0x%x\n\n",  chip->ecc.total);

	len += sprintf(buf+len, "Chip setup params\n");
	len += sprintf(buf+len, "\tdatabus_16bit: %d\n", info->xs);
	len += sprintf(buf+len, "\tuse_dma: %d\n", info->use_dma);
	len += sprintf(buf+len, "\tns_per_cycle: %d\n", info->ns_per_cycle);
	len += sprintf(buf+len, "\n\n");

	return len;
}

#if AUMB3000_NAND_DEBUG

/* debug commands */
#define NAND_DBG_CMD_READ_REGISTER 	0x1
#define NAND_DBG_CMD_WRITE_REGISTER 0x2
#define NAND_DBG_CMD_LOG_LEVEL 		0x3
#define NAND_DBG_CMD_IO_BASE	 	0x4
#define NAND_DBG_CMD_ECC_CHECK	 	0x5
#define NAND_DBG_CMD_ECC_GEN	 	0x6
#define NAND_DBG_CMD_ERASE_BLOCK 	0x7
#define NAND_DBG_CMD_ERASE_ALL	 	0x8
#define NAND_DBG_CMD_INFO	 		0x9

static int mobinand_proc_wr_debug(struct file *file,
		const char *buffer, unsigned long count, void *data)
{
	char *ptr = (char *)buffer;
	uint32_t cmd = 0, arg1 = -1, arg2 = -1, arg3 = -1;
	uint32_t value, pagesperblock, errcnt = 0;

	struct nand_chip *chip;
	struct aumb3000_nand_info *info;
	chip = (struct nand_chip *)(&aumb3000_mtd[1]);
	/* and then the pointer to our private data */
	info = (struct aumb3000_nand_info *) (&chip[1]);

	if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0) {
		cmd = simple_strtoul(ptr, &ptr, 0);
		if (cmd < 0) {
			error("Invalid debug command");
			goto err_help;
		}
	} else {
		printk(KERN_ERR "Invalid cmd string");
		goto err_help;
	}

	/*
	printk("debug: cmd: 0x%x, arg1: 0x%x(%d), arg2: 0x%x(%d)\n",
			cmd, arg1, arg1, arg2, arg2);
	*/
	switch (cmd) {
	case NAND_DBG_CMD_READ_REGISTER:
		get_ularg(&ptr, (uint32_t)buffer, count, &arg1);
		value = REG_READ_LOCKED(arg1);
		printk(KERN_ERR "Read at offset 0x%x is 0x%x(%d)\n",
				arg1, value, value);
		break;
	case NAND_DBG_CMD_WRITE_REGISTER:
		get_ularg(&ptr, (uint32_t)buffer, count, &arg1);
		get_ularg(&ptr, (uint32_t)buffer, count, &arg2);
		printk(KERN_ERR "Write 0x%x(%d) to offset 0x%x\n",
				arg2, arg2, arg1);
		REG_WRITE_LOCKED(arg2, arg1);
		break;
	case NAND_DBG_CMD_LOG_LEVEL:
		get_ularg(&ptr, (uint32_t)buffer, count, &loglevel);
		printk(KERN_INFO "NAND setting debug level to %d\n",
				loglevel);
		break;
	case NAND_DBG_CMD_IO_BASE:
		printk(KERN_INFO "NAND flashreg_base  = 0x%p\n", flashreg_base);
		printk(KERN_INFO "NAND flashdata_base = 0x%p\n", flashdata_base);
		break;
	case NAND_DBG_CMD_ECC_CHECK:
		/* arg is 0-3, see ECC_TYPE defines */
		get_ularg(&ptr, (uint32_t)buffer, count, &arg1);
		printk(KERN_INFO
				"Current ECC check value is 0x%x, new value is 0x%x\n",
				MOBI_FLASH_NAND_CONTROL_EC_R(info->ctrl_shadow), arg1);
		info->check_ecc = arg1;
		MOBI_FLASH_NAND_CONTROL_EC_CLR_AND_SET(info->ctrl_shadow, arg1);
		CONTROL_WRITE(info->ctrl_shadow);
		break;

	case NAND_DBG_CMD_ECC_GEN:
		/* arg is 0-3, see ECC_TYPE defines */
		get_ularg(&ptr, (uint32_t)buffer, count, &arg1);
		printk(KERN_INFO
				"Current ECC generate value is 0x%x, new value is 0x%x\n",
				MOBI_FLASH_NAND_CONTROL_EG_R(info->ctrl_shadow), arg1);
		info->gen_ecc = arg1;
		MOBI_FLASH_NAND_CONTROL_EG_CLR_AND_SET(info->ctrl_shadow, arg1);
		CONTROL_WRITE(info->ctrl_shadow);
		break;

	case NAND_DBG_CMD_ERASE_BLOCK:
	case NAND_DBG_CMD_ERASE_ALL:
		/* a way to skip the bbt table */
		pagesperblock = aumb3000_mtd->erasesize/aumb3000_mtd->writesize;
		if (cmd == NAND_DBG_CMD_ERASE_BLOCK) {
			/* echo "7 <start_addr> <block_count>"
			 *  page_num must be on an erase block boundry(0,64,128..)
			 */

			/* get start address and convert to page number */
			get_ularg(&ptr, (uint32_t)buffer, count, &arg1);
			arg1 >>= chip->page_shift;
			if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0) {
				arg2 = simple_strtoul(ptr, &ptr, 0);
				if (arg2 < 0) {
					arg2 = 1;
				}
			} else {
				arg2 = 1;
			}
		} else {
			/* erase all */
			arg1 = 0;
			/* total pages */
			arg2 = ((MTD_INFO_SIZE_CAST)aumb3000_mtd->size)/aumb3000_mtd->writesize;
			/* and then number of erase blocks */
			arg2 /= pagesperblock;
		}
		/* disable logs, and var1 is just for printk info */
		value = loglevel;
		loglevel = 0;

		chip->select_chip(aumb3000_mtd, 0);
		/* arg1 bumped by page number for nand_command */
		printk(KERN_INFO "Erasing %d blocks starting at block %d, page %6u @ 0x%08x\n",
				arg2, arg1/pagesperblock, arg1, arg1 << chip->page_shift);

		for (arg3 = 0; arg3 < arg2; arg3++, arg1 += pagesperblock) {
			printk("\rErasing block %d at page %6u @ 0x%08x... ",
					arg1/pagesperblock, arg1, arg1 << chip->page_shift);

			aumb3000_nand_command(aumb3000_mtd, NAND_CMD_ERASE1, -1, arg1);
			aumb3000_nand_command(aumb3000_mtd, NAND_CMD_ERASE2, -1, -1);
			if (chip->waitfunc(aumb3000_mtd, chip) & NAND_STATUS_FAIL) {
				printk(KERN_ERR "FAILED\n");
				errcnt++;
			}
		}
		printk(KERN_ERR "done\n");
		loglevel = value;
		break;
	case NAND_DBG_CMD_INFO:
		printk(KERN_INFO "\tPage size: %d\n", aumb3000_mtd->writesize);
		printk(KERN_INFO "\tTotal pages: %d\n",
				((MTD_INFO_SIZE_CAST)aumb3000_mtd->size)/aumb3000_mtd->writesize);
		printk(KERN_INFO "\tBlock size: %d\n", aumb3000_mtd->erasesize);
		printk(KERN_INFO "\tPages per block : %d\n",
				aumb3000_mtd->erasesize/aumb3000_mtd->writesize);
		printk(KERN_INFO "\tTotal blocks: %d\n",
				((MTD_INFO_SIZE_CAST)aumb3000_mtd->size)/aumb3000_mtd->erasesize);
		printk(KERN_INFO "\tmtd->name: %s\n", aumb3000_mtd->name);
		printk(KERN_INFO "\tmtd->writesize: %d\n", aumb3000_mtd->writesize);
		printk(KERN_INFO "\tmtd->oobsize: %d\n", aumb3000_mtd->oobsize);
		printk(KERN_INFO "\tmtd->erasesize: %d\n", aumb3000_mtd->erasesize);
		printk(KERN_INFO "\tmtd->size: %llu\n", aumb3000_mtd->size);
		printk(KERN_INFO "\tchip->options: 0x%x\n", chip->options);
		printk(KERN_INFO "\tchip->chipsize: 0x%lx\n", chip->chipsize);
		printk(KERN_INFO "\tchip->numchips: 0x%x\n", chip->numchips);
		printk(KERN_INFO "\tchip->page_shift: 0x%x\n", chip->page_shift);
		printk(KERN_INFO "\tchip->pagemask: 0x%x\n", chip->pagemask);
		printk(KERN_INFO "\tchip->bbt_erase_shift: %d\n",  chip->bbt_erase_shift);
		printk(KERN_INFO "\tchip->chip_shift: %d\n",  chip->chip_shift);
		printk(KERN_INFO "\tchip->badblockpos: %d\n",  chip->badblockpos);
		printk(KERN_INFO "\tchip->numchips: %d\n",  chip->numchips);
		printk(KERN_INFO "\tchip->ecc.mode: 0x%x\n",  chip->ecc.mode);
		printk(KERN_INFO "\tchip->ecc.size: 0x%x\n",  chip->ecc.size);
		printk(KERN_INFO "\tchip->ecc.steps: 0x%x\n",  chip->ecc.steps);
		printk(KERN_INFO "\tchip->ecc.total: 0x%x\n\n",  chip->ecc.total);
		break;
	default:
		break;
	}

	while (ptr-buffer < count)
		ptr++;

	return ptr - buffer;

err_help:
#define PRINT_CMD(x)	#x
	printk(KERN_INFO "\nAvailable debug commands are(see code for details):\n");
	/*
	   printk("cmd name: %s, cmd_num: %d\n",
	   PRINT_CMD(UART_DBG_CMD_LOOPBACK), UART_DBG_CMD_LOOPBACK);
	   printk("syntax: echo \"ttynum cmd_num [param1] [param2]\"
	   > /proc/driver/serial/debug");
	   */

	/* read everything we don't care about */
	while (ptr-buffer<count)
		ptr++;

	return -EINVAL;
}
#endif /* DEBUG */
#endif /* PROC_FS */

/**
 * \brief aumb3000_nand_init_chip
 * 	Initialized data structures base on mkr and dev ID
 *
 * \param info: pointer to aumb3000s_nand_info
 *
 * \retval Zero  - upon success
 *
 * \remark
 * 	Since I've got the mfg and device ID code was structured
 * so that it should be easy to add and handle other chips in the
 * future
 */
static int aumb3000_init_chip(struct nand_chip *chip)
{
	struct aumb3000_nand_info *info =
		(struct aumb3000_nand_info *) chip->priv;

	struct nand_flash_dev *type = NULL;
	int pagesize = 0, i;

	/*
	 *  some things may be chip specific, these values
	 *  can be redefined in the device specific switch
	 *  statement
	 */
	chip->cmdfunc 		= aumb3000_nand_command;
	chip->write_page    = aumb3000_nand_write_page;
	chip->write_buf     = aumb3000_nand_write_buf;
	chip->ecc.write_page = aumb3000_nand_write_page_hwecc;
	chip->ecc.write_oob  = aumb3000_nand_write_oob_std;
	chip->ecc.write_page_raw = aumb3000_nand_write_page_raw;

	chip->read_byte		= aumb3000_nand_read_byte;
	chip->read_word		= aumb3000_nand_read_word;
	chip->read_buf      = aumb3000_nand_read_buf;
	chip->ecc.read_page = aumb3000_nand_read_page_hwecc;
	chip->ecc.read_oob  = aumb3000_nand_read_oob_std;
	chip->ecc.read_page_raw = aumb3000_nand_read_page_raw;

	chip->select_chip   = aumb3000_nand_select_chip;
	chip->cmd_ctrl      = aumb3000_nand_hwcontrol;

	chip->dev_ready     = aumb3000_nand_devready;

	chip->controller    = &info->controller;
	chip->IO_ADDR_R		= info->data_base;
	chip->IO_ADDR_W     = info->data_base;

	/* these must be defined even if they do nothing */
	/* the flash control takes care of generating, checking
	 *  and correction so no need to do anything in SW
	 */
	chip->ecc.hwctl     = aumb3000_enable_hwecc;
	chip->ecc.calculate = aumb3000_calculate_ecc;
	chip->ecc.correct   = aumb3000_nand_correct_data;

	chip->ecc.mode	= NAND_ECC_HW;
	spin_lock_init(&info->controller.lock);
	init_waitqueue_head(&info->controller.wq);

	/* stolen from nand_base
	 *  we can get the chip size and see if we need to
	 *  determine the pages size from the ID bytes, once
	 *  again, we need this info before nand_scan because
	 *  nand_scan will try to access the chip to do things
	 *  like build the bad block table so the controller
	 *  must be setup before that
	 */
	for (i = 0; nand_flash_ids[i].name != NULL; i++) {
		if (info->dev_id == nand_flash_ids[i].id) {
			type =  &nand_flash_ids[i];
			break;
		}
	}
	if (!type)
		return -1;

	dprintk3("chipsize from table is %lu Mbytes\n", type->chipsize);
	if ((info->sz = get_chipsize(type->chipsize)) == -1) {
		error("Unable to determine NAND flash size");
		return -1;
	}
	dprintk3("info->sz = %d\n", info->sz);
	/*  flash chip size */
	info->ctrl_shadow |= (info->sz << MOBI_FLASH_NAND_CONTROL_SZ_SHIFT);

	/* we can get info from 3rd ID bit if tables pagesize is 0 */
	if (type->pagesize == 0) {
		dprintk3("type->pagesize == 0, get pagesize from ID 0x%x\n",
				info->chip_desc);
		/* if large page, enable in the qcc registers */
		pagesize = 1024 << (info->chip_desc & 0x3);
	} else {
		pagesize = type->pagesize;
	}
	/*
	 *  looking at the samsung doc, there are 3 ecc bytes
	 *  for each 512 bytes area.  on a small page device the
	 *  ecc is written only once to the spare area.  for large
	 *  page, it is calculated in 512 block for the 2048 page
	 *  and each ecc for the 512 block is written at different
	 *  offset in the spare area.  so I think our ecc.bytes
	 *  will always be 3 and ecc.size is 512. but since the
	 *  nand controller generates, correct, read/writes the
	 *  the ecc to the spare area this won't be used
	 */
	chip->options     = 0;
	chip->ecc.bytes   = 3;
	chip->ecc.size    = 512;

	if (pagesize > 512) {

		dprintk3("Large page device detected, enabling\n");
		/* large page device */
		/* column address in 2 cycles for large page */
		info->ctrl_shadow |= (1 << MOBI_FLASH_NAND_CONTROL_C2_SHIFT);

#ifdef CONFIG_ARCH_MERLIN
		{
			uint32_t qcc_sysconfig_shadow = 0x0;
			qcc_sysconfig_shadow =
				mobi_qcc_readreg(MOBI_QCCSISC_SYS_CONFIG_OFFSET);
			qcc_sysconfig_shadow |= MOBI_QCCSISC_SYS_CONFIG_NAND_2KPG_MASK;
			mobi_qcc_writereg(qcc_sysconfig_shadow, MOBI_QCCSISC_SYS_CONFIG_OFFSET);
		}
#endif
		/*
		 * struct nand_ecc_ctrl - Control structure for ecc
		 * @mode:       ecc mode
		 * @steps:      number of ecc steps per page
		 * @size:       data bytes per ecc step
		 * @bytes:      ecc bytes per step
		 * @total:      total number of ecc bytes per page
		 *
		 * really not clear on this, but since we do HW EEC
		 * gen and check it never gets used anyway..
		 *
		 * from nand_scan in nand_base.c
		 *  chip->ecc.steps = mtd->writesize / chip->ecc.size;
		 *  chip->ecc.total = chip->ecc.steps * chip->ecc.bytes;
		 *
		 *  writesize is pagesize
		 */

		/*
		 *  even if 16 wide, data is looked at in bytes
		 *  so use the 8-wide layout for both
		 */
		chip->ecc.layout  = &samsungx8_layout_oob_64;
		if (info->xs) {
			chip->options     = NAND_BUSWIDTH_16;
			chip->badblock_pattern = &samsungx16_largepage_bbt;
		} else {
			chip->badblock_pattern = &samsungx8_largepage_bbt;
		}
		if (info->sz > 4) /* greater 128MB/1Gbits */
			info->ctrl_shadow |= (1 << MOBI_FLASH_NAND_CONTROL_P3_SHIFT);
		else
			info->ctrl_shadow |= (0 << MOBI_FLASH_NAND_CONTROL_P3_SHIFT);

	} else { /* small page */
		dprintk3("Small page nand device detected\n");
		/*
		 *  even if 16 wide, data is looked at in bytes
		 *  so use the 8-wide layout for both
		 */
		chip->ecc.layout  = &samsungx8_layout_oob_16;

		if (info->xs) {
			chip->options     = NAND_BUSWIDTH_16;
			chip->badblock_pattern = &samsungx16_smallpage_bbt;
		} else {
			chip->badblock_pattern = &samsungx8_smallpage_bbt;
		}
		if (info->sz > 2) /* greater 32MB/256Mbits */
			info->ctrl_shadow |= (1 << MOBI_FLASH_NAND_CONTROL_P3_SHIFT);
		else
			info->ctrl_shadow |= (0 << MOBI_FLASH_NAND_CONTROL_P3_SHIFT);
	}
	chip->chip_delay   = 50;
	if (info->ctrl_shadow & MOBI_FLASH_NAND_CONTROL_P3_MASK) {
		dprintk3("Page address in 3 cycles\n");
	} else {
		dprintk3("Page address in 2 cycles\n");
	}


	/*
	 * TODO need to look into read confirm a bit more, and see if it is a
	 *  function of pagesize or it is mfg and/or chip specfic.  the
	 *  samsung devices i've looked at only do rc on large page
	 */
	if (pagesize > 512)
		MOBI_FLASH_NAND_CONTROL_RC_SET(info->ctrl_shadow);
	else
		MOBI_FLASH_NAND_CONTROL_RC_CLR(info->ctrl_shadow);

	/* OK, we got everything we need so do any mfg specific stuff now */
	switch (info->mfg_id) {
	case NAND_MFR_TOSHIBA:
	case NAND_MFR_SAMSUNG:
	case NAND_MFR_STMICRO:
	case NAND_MFR_MICRON:
	default:
		break;
	}
	CONTROL_WRITE(info->ctrl_shadow);
	dprintk3("info->ctrl_shadow based on device detection: 0x%x\n",
			info->ctrl_shadow);
	return 0;
}

/* overwrite any default timings with modparams or values
 * specified in board driver, modparams take precedence
 */
static void update_timing_data(struct aumb3000_nand_info *info)
{
	struct aumb3000_nand_device_data_t *pdata = info->pdata;
	int i;

	for (i = 0; i < NUMBER_NAND_TIMING_PARAMS; i++) {
		if ((modparam_set_timing_bitmask >> i) & 0x1)
			default_timings[i] = modparam_timings[i];
		else if ((pdata->board_set_timing_bitmask >> i) & 0x1)
			default_timings[i] = pdata->board_timings[i];
	}
}

static void program_timing_intervals(struct aumb3000_nand_info *info,
		uint16_t timer_mask)
{

#define BITSAT(want, max) ((uint32_t)want > (uint32_t)max ? max : want)
	/* our timing fields are max 8 bit, thus max value ix 0xff */
#define NS_TO_CYCLES8(ns)   BITSAT(ns/info->ns_per_cycle, 0xff)

	/* this is a bit crude but..., basically 'x' is the array element
	 * of the lowest byte in the register */
#define ASSEMBLE_3MEMBER_TIMING_REG(x) \
	((((NS_TO_CYCLES8(default_timings[x]))   <<  0) & 0x000000ff) | \
	 (((NS_TO_CYCLES8(default_timings[x+1])) <<  8) & 0x0000ff00) | \
	 (((NS_TO_CYCLES8(default_timings[x+2])) << 16) & 0x00ff0000))

#define ASSEMBLE_4MEMBER_TIMING_REG(x) \
	((((NS_TO_CYCLES8(default_timings[x]))   <<  0) & 0x000000ff) | \
	 (((NS_TO_CYCLES8(default_timings[x+1])) <<  8) & 0x0000ff00) | \
	 (((NS_TO_CYCLES8(default_timings[x+2])) << 16) & 0x00ff0000) | \
	 (((NS_TO_CYCLES8(default_timings[x+3])) << 24) & 0xff000000))

	update_timing_data(info);

	if (timer_mask & TIMING_INTVAL0_REG) {
		info->timing.interval0 =
			ASSEMBLE_4MEMBER_TIMING_REG(NAND_TIMING_TDS);

		REG_WRITE_LOCKED(info->timing.interval0,
				MOBI_FLASH_NAND_TIMERINTVAL0_OFFSET);
		dprintk4("timing0: wrote 0x%x\n", info->timing.interval0);
	}

	if (timer_mask & TIMING_INTVAL1_REG) {
		info->timing.interval1 =
			ASSEMBLE_4MEMBER_TIMING_REG(NAND_TIMING_TDH);

		REG_WRITE_LOCKED(info->timing.interval1,
				MOBI_FLASH_NAND_TIMERINTVAL1_OFFSET);
		dprintk4("timing1: wrote 0x%x\n", info->timing.interval1);
	}

	if (timer_mask & TIMING_INTVAL2_REG) {
		info->timing.interval2 =
			ASSEMBLE_4MEMBER_TIMING_REG(NAND_TIMING_TRR);

		REG_WRITE_LOCKED(info->timing.interval2,
				MOBI_FLASH_NAND_TIMERINTVAL2_OFFSET);
		dprintk4("timing2: wrote 0x%x\n", info->timing.interval2);
	}

	if (timer_mask & TIMING_INTVAL3_REG) {
		info->timing.interval3 =
			ASSEMBLE_4MEMBER_TIMING_REG(NAND_TIMING_TCEH);

		REG_WRITE_LOCKED(info->timing.interval3,
				MOBI_FLASH_NAND_TIMERINTVAL3_OFFSET);
		dprintk4("timing3: wrote 0x%x\n", info->timing.interval3);
	}

	if (timer_mask & TIMING_INTVAL4_REG) {
		info->timing.interval4 =
			ASSEMBLE_4MEMBER_TIMING_REG(NAND_TIMING_TIR);

		REG_WRITE_LOCKED(info->timing.interval4,
				MOBI_FLASH_NAND_TIMERINTVAL4_OFFSET);
		dprintk4("timing4: wrote 0x%x\n", info->timing.interval4);
	}

	if (timer_mask & TIMING_INTVAL5_REG) {
		info->timing.interval5 =
			ASSEMBLE_3MEMBER_TIMING_REG(NAND_TIMING_TAR);

		REG_WRITE_LOCKED(info->timing.interval5,
				MOBI_FLASH_NAND_TIMERINTVAL5_OFFSET);
		dprintk4("timing5: wrote 0x%x\n", info->timing.interval5);
	}
}

/*
 * some default parameters will be set in the board file, however,
 *  these can be overridden by modparams.  Here will will update
 *  the defaults with any modparams that have been set
 */
static void check_modparam_timings(struct aumb3000_nand_device_data_t *pdata)
{
	modparam_set_timing_bitmask = 0x0;
	if (tds != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TDS] = tds;
		modparam_set_timing_bitmask |= NAND_TIMING_TDS_SET;
	}
	if (tcs != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TCS] = tcs;
		modparam_set_timing_bitmask |= NAND_TIMING_TCS_SET;
	}
	if (tals != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TALS] = tals;
		modparam_set_timing_bitmask |= NAND_TIMING_TALS_SET;
	}
	if (tcls != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TCLS] = tcls;
		modparam_set_timing_bitmask |= NAND_TIMING_TCLS_SET;
	}
	if (tdh != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TDH] = tdh;
		modparam_set_timing_bitmask |= NAND_TIMING_TDH_SET;
	}
	if (talh != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TALH] = talh;
		modparam_set_timing_bitmask |= NAND_TIMING_TALH_SET;
	}
	if (tclh != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TCLH] = tclh;
		modparam_set_timing_bitmask |= NAND_TIMING_TCLH_SET;
	}
	if (trr != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TRR] = trr;
		modparam_set_timing_bitmask |= NAND_TIMING_TRR_SET;
	}
	if (twb != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TWB] = twb;
		modparam_set_timing_bitmask |= NAND_TIMING_TWB_SET;
	}
	if (twh != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TWH] = twh;
		modparam_set_timing_bitmask |= NAND_TIMING_TWH_SET;
	}
	if (tceh != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TCEH] = tceh;
		modparam_set_timing_bitmask |= NAND_TIMING_TCEH_SET;
	}
	if (trp != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TRP] = trp;
		modparam_set_timing_bitmask |= NAND_TIMING_TRP_SET;
	}
	if (tir != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TIR] = tir;
		modparam_set_timing_bitmask |= NAND_TIMING_TIR_SET;
	}
	if (twhr != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TWHR] = twhr;
		modparam_set_timing_bitmask |= NAND_TIMING_TWHR_SET;
	}
	if (tclr != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TCLR] = tclr;
		modparam_set_timing_bitmask |= NAND_TIMING_TCLR_SET;
	}
	if (trdelay != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TRDELAY] = trdelay;
		modparam_set_timing_bitmask |= NAND_TIMING_TRDELAY_SET;
	}
	if (tar != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TAR] = tar;
		modparam_set_timing_bitmask |= NAND_TIMING_TAR_SET;
	}
	if (trhz != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TRHZ] = trhz;
		modparam_set_timing_bitmask |= NAND_TIMING_TRHZ_SET;
	}
	if (tww != UNINITIALIZED_MODPARAM) {
		modparam_timings[NAND_TIMING_TWW] = tww;
		modparam_set_timing_bitmask |= NAND_TIMING_TWW_SET;
	}
}

int device_powerup(void)
{
	uint32_t ret = 0;

#ifdef CONFIG_ARCH_MERLIN
	unsigned long sysconfig;
	/* make sure the NAND and RAMs are enabled */
	sysconfig = mobi_qcc_readreg(MOBI_QCCSISC_SYS_CONFIG_OFFSET);

	MOBI_QCCSISC_SYS_CONFIG_NAND_FLASH_EN_SET(sysconfig);
	MOBI_QCCSISC_SYS_CONFIG_NANDNOR_RAM_EN_SET(sysconfig);
	ret = mobi_qcc_writereg(sysconfig, MOBI_QCCSISC_SYS_CONFIG_OFFSET);
	if (ret < 0)
		return -1;

	/* make sure we are out of reset */
	ret = mobi_reset_disable(RESET_ID_NAND);
	if (ret < 0) {
		MOBI_QCCSISC_SYS_CONFIG_NAND_FLASH_EN_CLR(sysconfig);
		MOBI_QCCSISC_SYS_CONFIG_NANDNOR_RAM_EN_CLR(sysconfig);
		mobi_qcc_writereg(sysconfig,
				MOBI_QCCSISC_SYS_CONFIG_OFFSET);
	}
#endif

#ifdef CONFIG_ARCH_FALCON
	ret = mobi_reset_disable(RESET_ID_NAND);
#endif

	return ret;
}


void device_powerdown(void)
{

#ifdef CONFIG_ARCH_MERLIN
	unsigned long sysconfig;
	sysconfig =
		mobi_qcc_readreg(MOBI_QCCSISC_SYS_CONFIG_OFFSET);

	MOBI_QCCSISC_SYS_CONFIG_NAND_FLASH_EN_CLR(sysconfig);
	MOBI_QCCSISC_SYS_CONFIG_NANDNOR_RAM_EN_CLR(sysconfig);

	mobi_qcc_writereg(sysconfig, MOBI_QCCSISC_SYS_CONFIG_OFFSET);
#endif

	mobi_reset_enable(RESET_ID_NAND);
}

static int __devinit aumb3000_nandreg_probe(struct amba_device *pdev, 
		struct amba_id *id)
{
	struct nand_chip *chip;
	struct aumb3000_nand_info *info;
	struct clk *nand_clk = NULL;
	int ret = 0;

	/* Allocate memory for MTD device structure and private data */
	aumb3000_mtd = kmalloc(sizeof(struct mtd_info) +
			sizeof(struct nand_chip) +
			sizeof(struct aumb3000_nand_info), GFP_KERNEL);

	if (!aumb3000_mtd) {
		printk("Unable to allocate NAND MTD dev structure.\n");
		return -ENOMEM;
	}

	/* get pointer to nand_chip private data */
	chip = (struct nand_chip *)(&aumb3000_mtd[1]);
	/* and then the pointer to our private data */
	info = (struct aumb3000_nand_info *) (&chip[1]);

	/* initialize structures */
	memset(aumb3000_mtd, 0, sizeof(struct mtd_info));
	memset(chip, 0, sizeof(struct nand_chip));
	memset(info, 0, sizeof(struct aumb3000_nand_info));

	/* link the nand_chip private data with the MTD structure */
	aumb3000_mtd->priv = chip;
	aumb3000_mtd->owner = THIS_MODULE;
	/* and link out private data with the nand_chip structure */
	chip->priv = info;
	info->pdata = (struct aumb3000_nand_device_data_t*)pdev->dev.platform_data;

	if (device_powerup()) {
		error("Could not take device out of reset");
		ret = -EIO;
		goto err;
	}

	info->reg_base =
		ioremap(pdev->res.start, pdev->res.end-pdev->res.start+1);

	if (!info->reg_base) {
		error("Control ioremap failed");
		ret = -EIO;
		goto err;
	}
	flashreg_base  = info->reg_base;
	dprintk2("flashreg_phys 0x%x, flashreg_iobase = 0x%p\n",
			FLASH_REG_BASE, flashreg_base);

	/* check that the controller is not busy before we
	 *  start setting it up
	 */
	if (!aumb3000_nand_waitready(aumb3000_mtd)) {
		ret = -EIO;
		goto err;
	}

	/* modparams take precedence over platform data, so check if
	 * set and update local var accordingly
	 */
	if (use_dma != UNINITIALIZED_MODPARAM)
		info->use_dma = use_dma;
	else
		info->use_dma = (info->pdata)->use_dma;

	if (databus_16bit != UNINITIALIZED_MODPARAM)
		info->xs = databus_16bit;
	else
		info->xs = (info->pdata)->databus_16bit;

	if (banks_per_system != UNINITIALIZED_MODPARAM)
		info->banks_per_system = banks_per_system;
	else
		info->banks_per_system = (info->pdata)->banks_per_system;

	if (chips_per_bank != UNINITIALIZED_MODPARAM)
		info->chips_per_bank  = chips_per_bank;
	else
		info->chips_per_bank   = (info->pdata)->chips_per_bank;

	check_modparam_timings(info->pdata);

	printk("AU-MB3000 NAND Driver running in %s mode.\n",
			(info->use_dma == 1 ? "DMA" : "PIO"));

	nand_clk = clk_get(&pdev->dev, (info->pdata)->clk_name);
	info->ns_per_cycle = (1000000000/clk_get_rate(nand_clk));

	/* from modparams */
	program_timing_intervals(info, TIMING_INTVAL_REG_ALL_MASK);

	info->check_ecc = ECC_TYPE_MAIN_ONLY;
	info->gen_ecc = ECC_TYPE_MAIN_ONLY;

	if ((info->eb = get_exbanks(info->banks_per_system)) == -1) {
		error("Invalid number of NAND flash banks: %d",
				info->banks_per_system);
		ret = -ENXIO;
		goto err;
	}
	/* get wd fields */
	get_databus_width(chip);

	/*
	 *  spare area enable,
	 *  automatic writeback disabled
	 *  write fill enable(no cache fills on write)
	 *  write protect disable
	 *  ID in 4 cycles
	 *  interrupts disabled
	 *  16bit wide databus (from modparam)
	 *  number of external banks(from modparam)
	 *  data bus width determed from #of banks and databus width
	 */
	info->ctrl_shadow = ((1 << MOBI_FLASH_NAND_CONTROL_SE_SHIFT) |
			(0 << MOBI_FLASH_NAND_CONTROL_AU_SHIFT) |
			(0 << MOBI_FLASH_NAND_CONTROL_FD_SHIFT) |
			(0 << MOBI_FLASH_NAND_CONTROL_WP_SHIFT) |
			(1 << MOBI_FLASH_NAND_CONTROL_I4_SHIFT) |
			(0 << MOBI_FLASH_NAND_CONTROL_IE_SHIFT) |
			(info->gen_ecc << MOBI_FLASH_NAND_CONTROL_EG_SHIFT) |
			(info->xs << MOBI_FLASH_NAND_CONTROL_XS_SHIFT) |
			(info->eb << MOBI_FLASH_NAND_CONTROL_EB_SHIFT) |
			(info->wd << MOBI_FLASH_NAND_CONTROL_WD_SHIFT) |
			(interrupt_enabled << MOBI_FLASH_NAND_CONTROL_IE_SHIFT));

	CONTROL_WRITE(info->ctrl_shadow);
	dprintk3("Wrote to ctrl reg: data 0x%x\n", info->ctrl_shadow);

	/* set the private data */
	platform_set_drvdata(pdev, aumb3000_mtd);

	return 0;
err:
	if (info->reg_base != NULL) {
		iounmap(info->reg_base);
		info->reg_base = NULL;
	}
	if (aumb3000_mtd != NULL) {
		kfree(aumb3000_mtd);
		aumb3000_mtd = NULL;
	}
	device_powerdown();

	return ret;
}

static int __devexit aumb3000_nandreg_remove(struct amba_device *pdev)
{
	struct nand_chip *chip;
	struct aumb3000_nand_info *info;

	if (!aumb3000_mtd)
		return 0;
	/* get pointer to nand_chip private data */
	chip = (struct nand_chip *)(&aumb3000_mtd[1]);
	/* and then the pointer to our private data */
	info = (struct aumb3000_nand_info *) (&chip[1]);

	/* Release resources(including partitions), unregister device */
	nand_release(aumb3000_mtd);

	if (info->reg_base != NULL) {
		iounmap(info->reg_base);
		info->reg_base = NULL;
	}
	if (aumb3000_mtd != NULL) {
		kfree(aumb3000_mtd);
		aumb3000_mtd = NULL;
	}

	platform_set_drvdata(pdev, NULL);

	device_powerdown();

	return 0;
}

#ifdef CONFIG_MTD_PARTITIONS
const char *part_probes[] = { "cmdlinepart", NULL };

/* partitions are enabled but none given on the cmdline
 * so just make one big one??
 */
static struct mtd_partition nand_partition_info[] = {
	{
		.name = "Nand Flash Partition",
		.offset = 0,
		.size = MTDPART_SIZ_FULL,
	}
};

#define NUM_PARTITIONS 1
#endif

static int __devinit aumb3000_nanddata_probe(struct amba_device *pdev, 
		struct amba_id *id)
{
	struct nand_chip *chip;
	struct aumb3000_nand_info *info;
	int ret = 0, reg = 0, retry = 1;

#ifdef CONFIG_MTD_PARTITIONS
	struct mtd_partition *partition_info;
	int nr_partitions = 0;
#endif

	if (!aumb3000_mtd) {
		printk("Unable to allocate NAND MTD dev structure.\n");
		return -ENOMEM;
	}

	/* get pointer to nand_chip private data */
	chip = (struct nand_chip *)(&aumb3000_mtd[1]);
	/* and then the pointer to our private data */
	info = (struct aumb3000_nand_info *) (&chip[1]);
	info->data_base =
		ioremap(pdev->res.start, pdev->res.end-pdev->res.start+1);

	if (!info->data_base) {
		error("Data ioremap failed");
		ret = -EIO;
		goto err;
	}
	/* save these to a global also, just to make read/writes easier */
	flashdata_base = info->data_base;
	dprintk2("flashdata_phys 0x%x flashdata_iobase = 0x%p\n",
			FLASH_DATA_BASE, flashdata_base);

	info->select_chip = 0;

	/* lets get some info from the flash chip zero which can be used */
	/* for init upper 28 bits of address and lower 3 are cmd */
	/* the assumption is that all the banks are the same type of chip */
retry_readid:
	dprintk3("Query for device id\n");
	ret = cmd_write(AUMB3000_NAND_CMD_READID, 0x0);
	if (ret) {
		ret = -EIO;
		goto err;
	} else {
		reg = REG_READ(MOBI_FLASH_NAND_ID_OFFSET);
		/*
		 *  some chips do the READID in 4 cyles, if we didn't get it
		 *  on the first try, set the control bit to try in 4 cycles
		 *  and retry the read again
		 */
		if (reg == 0x0 && retry) {
			info->ctrl_shadow &= ~MOBI_FLASH_NAND_CONTROL_I4_MASK;
			CONTROL_WRITE(info->ctrl_shadow);
			dprintk3("disable ID4 and retry, ctrl_reg: 0x%x\n",
					info->ctrl_shadow);
			retry = 0;
			goto retry_readid;
		}

		if (reg == 0x0) {
			error("Unable to discover any NAND chips");
			ret = -ENODEV;
			goto err;
		}
		dprintk3("NAND id reg = 0x%x\n", reg);
		info->mfg_id = MOBI_FLASH_NAND_ID_MKR_R(reg);
		info->dev_id = MOBI_FLASH_NAND_ID_DEV_R(reg);
		info->chip_desc = MOBI_FLASH_NAND_ID_ID4_R(reg);
	}
	dprintk3("got mfg_id of 0x%x, dev_id of 0x%x\n",
			info->mfg_id, info->dev_id);

	/* early init nand_chip structure */
	if (aumb3000_init_chip(chip) != 0) {
		error("Unknown NAND chip, cannot initialize flash setup");
		ret = -ENODEV;
		goto err;
	}

	info->cmd.nand_cmd = -1;
	info->cmd.au3000_cmd = -1;
	info->cmd.last_cmd_byte_read = 3;
	info->dma_handle = -1;
	info->prev_dma_mode = DMA_MODE_NONE;

	if (info->use_dma) {
		/* do as much setup as possible for DMA now, share r/w handle */
		/* dma channel must be setup before nand_scan is called because
		 *  it will read the chip to build the bbt
		 *  because the NAND, HSBS and MHIF do not have a DMA HW handshaking
		 *  interface, only one device can be running dma at a time.  mainly
		 *  when decoding, since this uses the HSBS path, the rootfs on the
		 *  nand can get locked out.  Add a flag that will allow these device
		 *  to waitfor the bus to become available instead of returning an
		 *  error when dma cannot be started.  This has a performance effect
		 *  when trying to use the nand rootfs when decoding but rootfs
		 *  interaction should be kept to a minimum in these case
		 */
		if ((info->dma_handle =
					mobi_dma_request("nanddma", MOBI_DMA_O_NONE)) < 0) {
			ret = info->dma_handle;
			goto err;
		}
		if ((ret = mobi_dma_setup_handler(info->dma_handle,
						(void *)dma_handler, NULL)))
			goto err;
	}

	/* we want all the logs off while calling nand_scan */
#if AUMB3000_NAND_DEBUG
	if (debug_nand_scan == 0) {
		debug_nand_scan = loglevel;
		loglevel = 0;
	}
#endif
	/* now look for real devices, up to 4 banks */
	dprintk3("Calling nand_scan\n");
	if (nand_scan(aumb3000_mtd,
				(info->banks_per_system)*(info->chips_per_bank))) {
		ret = -ENXIO;
		goto err;
	}
#if AUMB3000_NAND_DEBUG
	if (loglevel == 0)
		loglevel = debug_nand_scan;
#endif

	/* the bbt has been built, turn on ECC checking */
	MOBI_FLASH_NAND_CONTROL_EC_CLR_AND_SET(info->ctrl_shadow, info->check_ecc);
	CONTROL_WRITE(info->ctrl_shadow);

#ifdef CONFIG_MTD_PARTITIONS
	/* if cmdline partitions are enabled/loaded, it will register a
	 *  cmdlinepart mtd partition parser.  a patch has enabled the
	 *  cmdline parser to work with a modparam which will define "cmdline"
	 */
	aumb3000_mtd->name = "aumb3000-nand";
	nr_partitions = parse_mtd_partitions(aumb3000_mtd, part_probes,
			&partition_info, 0);
	dprintk3("Added %d partitions after attempting to parse cmdline\n",
			(nr_partitions <= 0 ? 0 : nr_partitions));

	if (nr_partitions <= 0) {
		partition_info = nand_partition_info;
		nr_partitions = NUM_PARTITIONS;
		printk(KERN_WARNING
				"No cmdline partitions, will add %d static partitions\n",
				nr_partitions);
	}

	if (nr_partitions > 0)
		add_mtd_partitions(aumb3000_mtd, partition_info, nr_partitions);
#else
	/* if no partitions, add it as a single device */
	add_mtd_device(aumb3000_mtd);
#endif

	dprintk1("\n");
	dprintk1("Values after calling nand_scan\n");
	dprintk1("\tmtd->name: %s\n", aumb3000_mtd->name);
	dprintk1("\tmtd->writesize: %d\n", aumb3000_mtd->writesize);
	dprintk1("\tmtd->oobsize: %d\n", aumb3000_mtd->oobsize);
	dprintk1("\tmtd->erasesize: %d\n", aumb3000_mtd->erasesize);
	dprintk1("\tmtd->size: %llu\n", aumb3000_mtd->size);
	dprintk1("\tchip->options: 0x%x\n", chip->options);
	dprintk1("\tchip->chipsize: 0x%lx\n", chip->chipsize);
	dprintk1("\tchip->numchips: 0x%x\n", chip->numchips);
	dprintk1("\tchip->page_shift: 0x%x\n", chip->page_shift);
	dprintk1("\tchip->pagemask: 0x%x\n", chip->pagemask);
	dprintk1("\tchip->bbt_erase_shift: %d\n",  chip->bbt_erase_shift);
	dprintk1("\tchip->chip_shift: %d\n",  chip->chip_shift);
	dprintk1("\tchip->badblockpos: %d\n",  chip->badblockpos);
	dprintk1("\tchip->numchips: %d\n",  chip->numchips);
	dprintk1("\tchip->ecc.mode: 0x%x\n",  chip->ecc.mode);
	dprintk1("\tchip->ecc.size: 0x%x\n",  chip->ecc.size);
	dprintk1("\tchip->ecc.steps: 0x%x\n",  chip->ecc.steps);
	dprintk1("\tchip->ecc.total: 0x%x\n\n",  chip->ecc.total);
	return 0;

err:
	if (info->dma_handle >= 0) {
		mobi_dma_free(info->dma_handle);
		info->dma_handle = -1;
	}
	if (info->data_base != NULL) {
		iounmap(info->data_base);
		info->data_base = NULL;
	}
	/* free these here so we don't have to deal in init */
	if (info->reg_base != NULL) {
		iounmap(info->reg_base);
		info->reg_base = NULL;
	}
	if (aumb3000_mtd != NULL) {
		kfree(aumb3000_mtd);
		aumb3000_mtd = NULL;
	}
	return ret;
}

static int __devexit aumb3000_nanddata_remove(struct amba_device *pdev)
{
	struct nand_chip *chip;
	struct aumb3000_nand_info *info;

	if (!aumb3000_mtd)
		return 0;

	/* get pointer to nand_chip private data */
	chip = (struct nand_chip *)(&aumb3000_mtd[1]);
	/* and then the pointer to our private data */
	info = (struct aumb3000_nand_info *) (&chip[1]);

	if (info->dma_handle >= 0) {
		mobi_dma_free(info->dma_handle);
		info->dma_handle = -1;
	}
	if (info->data_base != NULL) {
		iounmap(info->data_base);
		info->data_base = NULL;
	}
	return 0;
}

#ifdef CONFIG_PM
static int aumb3000_nandreg_suspend(struct amba_device *dev, pm_message_t state)
{
	return 0;
}

static int aumb3000_nandreg_resume(struct amba_device *dev)
{
	return 0;
}

static int aumb3000_nanddata_suspend(struct amba_device *dev,
		pm_message_t state)
{
	return 0;
}

static int aumb3000_nanddata_resume(struct amba_device *dev)
{
	return 0;
}
#else
#define aumb3000_nandreg_suspend NULL
#define aumb3000_nandreg_resume NULL
#define aumb3000_nanddata_suspend NULL
#define aumb3000_nanddata_resume NULL
#endif

static struct amba_id nandreg_ids[] __initdata = {
	{
		.id     = AUMB3000_NANDREG_AMBA_DEVID,
		.mask   = AUMB3000_NANDREG_AMBA_DEVID_MASK,
	},
	{ 0, 0 },
};

static struct amba_driver aumb3000_nandreg_driver = {
	.probe		= aumb3000_nandreg_probe,
	.remove		= __devexit_p(aumb3000_nandreg_remove),
	.suspend	= aumb3000_nandreg_suspend,
	.resume		= aumb3000_nandreg_resume,
	.id_table	= nandreg_ids,
	.drv	= {
		.name	= AUMB3000_NANDREG_AMBA_NAME,
	},
};

static struct amba_id nanddata_ids[] __initdata = {
	{
		.id     = AUMB3000_NANDDATA_AMBA_DEVID,
		.mask   = AUMB3000_NANDDATA_AMBA_DEVID_MASK,
	},
	{ 0, 0 },
};

static struct amba_driver aumb3000_nanddata_driver = {
	.probe		= aumb3000_nanddata_probe,
	.remove		= __devexit_p(aumb3000_nanddata_remove),
	.suspend	= aumb3000_nanddata_suspend,
	.resume		= aumb3000_nanddata_resume,
	.id_table	= nanddata_ids,
	.drv	= {
		.name	= AUMB3000_NANDDATA_AMBA_NAME,
	},
};

static int __init aumb3000_nand_init(void)
{
	int ret = 0;
#ifdef CONFIG_PROC_FS
	struct proc_dir_entry *pentry;
#endif

	printk("Loading AU-MB3000 NAND Driver\n");

	/* if not set locally or by modparam, then init */
#if defined(LOCAL_AUMB3000_NAND_DEBUG_ENABLE) || defined(CONFIG_AUMB3000_NAND_DEBUG)
	if (loglevel == -1)
#if defined(CONFIG_AUMB3000_NAND_DEBUG)
		loglevel = CONFIG_AUMB3000_NAND_DEBUG_LEVEL;
#else
	loglevel = 0;
#endif
#endif
	if (loglevel != -1)
		printk(KERN_INFO "AU-MB3000 NAND debug enabled,  loglevel is %d\n", loglevel);

	ret = amba_driver_register(&aumb3000_nandreg_driver);
	if (ret)
		goto out;

	ret = amba_driver_register(&aumb3000_nanddata_driver);
	if (ret) {
		amba_driver_unregister(&aumb3000_nandreg_driver);
		goto out;
	}

#ifdef CONFIG_PROC_FS
	if (nand_proc_dir == NULL)
		nand_proc_dir = proc_mkdir("driver/nand", NULL);

	if (nand_proc_dir == NULL) {
		nand_proc_dir = proc_mkdir("driver", NULL);
		if (nand_proc_dir != NULL)
			nand_proc_dir = proc_mkdir("driver/nand", NULL);
	}
	create_proc_read_entry("driver/nand/info",
			S_IRUSR | S_IRGRP | S_IROTH,
			NULL, aumb3000_proc_rd_info, NULL);

	pentry = create_proc_entry("driver/nand/timerintval",
			S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->write_proc = aumb3000_proc_wr_timerintval;
		pentry->read_proc  = aumb3000_proc_rd_timerintval;
	}

	pentry = create_proc_entry("driver/nand/recover",
			S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->write_proc = mobinand_proc_wr_recover;
	}

	proc_mkdir("driver/nand/ecc", NULL);
	proc_mkdir("driver/nand/ecc/check", NULL);
	proc_mkdir("driver/nand/ecc/generate", NULL);
	pentry = create_proc_entry("driver/nand/ecc/check/mainarea",
			S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->write_proc = mobinand_proc_wr_ecc_chk_main;
	}

	pentry = create_proc_entry("driver/nand/ecc/check/sparearea",
			S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->write_proc = mobinand_proc_wr_ecc_chk_spare;
	}
	pentry = create_proc_entry("driver/nand/ecc/generate/mainarea",
			S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->write_proc = mobinand_proc_wr_ecc_gen_main;
	}

	pentry = create_proc_entry("driver/nand/ecc/generate/sparearea",
			S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->write_proc = mobinand_proc_wr_ecc_gen_spare;
	}

#if AUMB3000_NAND_DEBUG
	pentry = create_proc_entry("driver/nand/debug",
			S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->write_proc = mobinand_proc_wr_debug;
	}
#endif
#endif

out:
	return ret;
}

static void __exit aumb3000_nand_exit(void)
{

#ifdef CONFIG_PROC_FS
#if AUMB3000_NAND_DEBUG
	remove_proc_entry("driver/nand/debug", NULL);
#endif
	remove_proc_entry("driver/nand/ecc/check/mainarea", NULL);
	remove_proc_entry("driver/nand/ecc/check/sparearea", NULL);
	remove_proc_entry("driver/nand/ecc/generate/mainarea", NULL);
	remove_proc_entry("driver/nand/ecc/generate/sparearea", NULL);
	remove_proc_entry("driver/nand/ecc/check", NULL);
	remove_proc_entry("driver/nand/ecc/generate", NULL);
	remove_proc_entry("driver/nand/ecc", NULL);
	remove_proc_entry("driver/nand/timerintval", NULL);
	remove_proc_entry("driver/nand/recover", NULL);
	remove_proc_entry("driver/nand/info", NULL);
	remove_proc_entry("driver/nand", NULL);
#endif

	amba_driver_unregister(&aumb3000_nanddata_driver);
	amba_driver_unregister(&aumb3000_nandreg_driver);

	printk(KERN_INFO "AU-MB3000 NAND Driver unloaded\n");
}
module_init(aumb3000_nand_init);
module_exit(aumb3000_nand_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Jeff Hane <jhane@mobilygen.com>");
MODULE_DESCRIPTION("Mobilygen AU-MB3000 MTD NAND driver");

module_param(use_dma, byte, 0644);
MODULE_PARM_DESC(use_dma, "Use dma for chip access(default 1)");;

module_param(databus_16bit, byte, 0644);
MODULE_PARM_DESC(databus_16bit, "Sixteen bit NAND chip data bus(0,1), "
		"default 0(8-bit)");

module_param(banks_per_system, byte, 0644);
MODULE_PARM_DESC(banks_per_system, "Number of flash banks in the "
		"system(1,2,4), default 1");

module_param(chips_per_bank, byte, 0644);
MODULE_PARM_DESC(chips_per_bank, "Number of flash chips per bank(1,2,4), "
		"default 1");

/* module params below this line are for the different timing intervals */
module_param(tds, byte, 0644);
MODULE_PARM_DESC(tds, "Data setup to write pulse edge time");
module_param(tcs, byte, 0644);
MODULE_PARM_DESC(tcs, "Chip enable setup to write pulse rising edge time");
module_param(tals, byte, 0644);
MODULE_PARM_DESC(tals, "Address setup to write pulse rising edge time");
module_param(tcls, byte, 0644);
MODULE_PARM_DESC(tcls, "Command setup to write pulse rising edge time");
module_param(tdh, byte, 0644);
MODULE_PARM_DESC(tdh , "Data hold form write pulse rising edge time");
module_param(tch, byte, 0644);
MODULE_PARM_DESC(tch , "Chipe enabel form write pulse rising edge time");
module_param(talh, byte, 0644);
MODULE_PARM_DESC(talh, "Address hold from write pulse rising edge time");
module_param(tclh, byte, 0644);
MODULE_PARM_DESC(tclh, "Command hold from write pulse rising edge time");
module_param(trr, byte, 0644);
MODULE_PARM_DESC(trr, "Ready setup to read pulse falling edge time");
module_param(twb, byte, 0644);
MODULE_PARM_DESC(twb, "Write pulse risign edge to busy time");
module_param(twh, byte, 0644);
MODULE_PARM_DESC(twh , "Write high hold time");
module_param(twp, byte, 0644);
MODULE_PARM_DESC(twp, "Wite pulse width");
module_param(tceh, byte, 0644);
MODULE_PARM_DESC(tceh, "Chip enable hold from read pulse rising edge time");
module_param(trb, byte, 0644);
MODULE_PARM_DESC(trb, "Read pulse risign edge to busy time");
module_param(treh, byte, 0644);
MODULE_PARM_DESC(treh, "Read high hold time");
module_param(trp, byte, 0644);
MODULE_PARM_DESC(trp, "Read pulse width");
module_param(tir, byte, 0644);
MODULE_PARM_DESC(tir, "Flash chip output high Z to read pulse "
		"falling edge time");
module_param(twhr, byte, 0644);
MODULE_PARM_DESC(twhr, "Write pulse rising edge to read pulse "
		"falling edge time");
module_param(tclr, byte, 0644);
MODULE_PARM_DESC(tclr, "Command setup to read pulse falling edge time");
module_param(trdelay, byte, 0644);
MODULE_PARM_DESC(trdelay, "Read pulse falling edge to data valid time");
module_param(tar, byte, 0644);
MODULE_PARM_DESC(tar, "Address setup to read pulse falling edge time");
module_param(trhz, byte, 0644);
MODULE_PARM_DESC(trhz, "Read puse rising edge to flash chip output high Z time");
module_param(tww, byte, 0644);
MODULE_PARM_DESC(tww, "Ready to write pulse falling edge time");

