/* 
*  linux/drivers/mtd/maps/merlin_flash.c
*  
*  Copyright (C) 2007 Mobilygen Corp
*  
*  MTD mapping driver for NOR on Mobilygen's Merlin platform
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License version 2 as
*  published by the Free Software Foundation.
*/
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/cfi.h>
#include <linux/mtd/partitions.h>
#include <linux/mtd/physmap.h>
#include <asm/io.h>

#include <mach/mobi_hmux.h>

struct physmap_flash_info {
	struct mtd_info		*mtd;
	struct map_info		map;
	struct resource		*res;
#ifdef CONFIG_MTD_PARTITIONS
	int			nr_parts;
	struct mtd_partition	*parts;
#endif
};

#if 0
#define dprintk(x...)  printk(x)
#else
#define dprintk(x...)  do {} while(0)
#endif


static map_word locked_map_read(struct map_info *map, unsigned long ofs)
{
	map_word val;
	
	mobi_hmux_request();
	val = inline_map_read(map, ofs);
	mobi_hmux_release();
	return val;
}

static void locked_map_write(struct map_info *map, 
		const map_word datum, unsigned long ofs)
{
	mobi_hmux_request();
	inline_map_write(map, datum, ofs);
	mobi_hmux_release();
}

static void locked_map_copy_from(struct map_info *map, 
		void *to, unsigned long from, ssize_t len)
{
	mobi_hmux_request();
	inline_map_copy_from(map, to, from, len);
	mobi_hmux_release();
}

static void locked_map_copy_to(struct map_info *map, 
		unsigned long to, const void *from, ssize_t len)
{
	mobi_hmux_request();
	inline_map_copy_to(map, to, from, len);
	mobi_hmux_release();
}

void locked_map_init(struct map_info *map)
{
	map->read = locked_map_read;
	map->write = locked_map_write;
	map->copy_from = locked_map_copy_from;
	map->copy_to = locked_map_copy_to;
}

static int physmap_flash_remove(struct platform_device *dev)
{
	struct physmap_flash_info *info;
	struct physmap_flash_data *physmap_data;

	info = platform_get_drvdata(dev);
	if (info == NULL)
		return 0;
	platform_set_drvdata(dev, NULL);
	physmap_data = dev->dev.platform_data;

	if (info->mtd != NULL) {
#ifdef CONFIG_MTD_PARTITIONS
		if (info->nr_parts) {
			del_mtd_partitions(info->mtd);
			kfree(info->parts);
		} 
		else if (physmap_data->nr_parts) {
			del_mtd_partitions(info->mtd);
		} 
		else {
			del_mtd_device(info->mtd);
		}
#else
		del_mtd_device(info->mtd);
#endif
		map_destroy(info->mtd);
	}

	if (info->map.virt != NULL)
		iounmap(info->map.virt);

	if (info->res != NULL) {
		release_resource(info->res);
		kfree(info->res);
	}

	return 0;
}

static const char *rom_probe_types[] = { "cfi_probe", NULL };
#ifdef CONFIG_MTD_PARTITIONS
static const char *part_probe_types[] = { "cmdlinepart", NULL };
#endif

/* XXX this should be define somewhere else */
#define MERLIN_MAX_NOR_SIZE 0x1000000

static int physmap_flash_probe(struct platform_device *dev)
{
	struct physmap_flash_data *physmap_data;
	struct physmap_flash_info *info;
	const char **probe_type;
	struct cfi_private *cfi;
	int err;
	unsigned long long res_end;

	dprintk("%s entered\n", __func__);
	physmap_data = dev->dev.platform_data;
	if (physmap_data == NULL)
		return -ENODEV;

	if (dev->resource->end > (dev->resource->start + MERLIN_MAX_NOR_SIZE-1)) {
		printk(KERN_WARNING "NOR size exceeds maximum allowed, reset to %d\n",
				MERLIN_MAX_NOR_SIZE);
		res_end = dev->resource->start + MERLIN_MAX_NOR_SIZE-1;
	}
	else {
		res_end = dev->resource->end;
	}

	printk(KERN_NOTICE "physmap platform flash device size 0x%.8llx at 0x%.8llx\n",
		(unsigned long long)(res_end - dev->resource->start + 1),
		(unsigned long long) dev->resource->start);

	info = kzalloc(sizeof(struct physmap_flash_info), GFP_KERNEL);
	if (info == NULL) {
		err = -ENOMEM;
		goto err_out;
	}

	platform_set_drvdata(dev, info);

	info->map.name = dev_name(&dev->dev);
	info->map.phys = dev->resource->start;
	info->map.size = res_end - dev->resource->start + 1;
	info->map.bankwidth = physmap_data->width;
	info->map.set_vpp = physmap_data->set_vpp;

	info->map.virt = ioremap(info->map.phys, info->map.size);
	if (info->map.virt == NULL) {
		dev_err(&dev->dev, "Failed to ioremap flash region\n");
		err = -EIO;
		goto err_out;
	}
	locked_map_init(&info->map);

	probe_type = rom_probe_types;
	for (; info->mtd == NULL && *probe_type != NULL; probe_type++) {
		info->mtd = do_map_probe(*probe_type, &info->map);
	}
	if (info->mtd == NULL) {
		dev_err(&dev->dev, "map_probe failed\n");
		err = -ENXIO;
		goto err_out;
	}
	info->mtd->owner = THIS_MODULE;

	/* we initially ioremap the address space for the largest NOR we
	*  can support.  then after probing we need to check if the flash 
	*  is actually smaller so we don't have a mapping into an area where
	*  there is no flash!
	*/
	cfi = (struct cfi_private *)info->map.fldrv_priv;
	if ((1 << cfi->cfiq->DevSize) < info->map.size) {
		info->map.size = (1 << cfi->cfiq->DevSize);
		iounmap(info->map.virt);
		info->map.virt = NULL;

		info->map.virt = ioremap(info->map.phys, info->map.size);
		if (info->map.virt == NULL) {
			dev_err(&dev->dev, "Failed to ioremap new flash region\n");
			err = -EIO;
			goto err_out;
		}
	}

#ifdef CONFIG_MTD_PARTITIONS
	err = parse_mtd_partitions(info->mtd, part_probe_types, &info->parts, 0);
	if (err > 0) {
		add_mtd_partitions(info->mtd, info->parts, err);
		return 0;
	}

	if (physmap_data->nr_parts) {
		printk(KERN_NOTICE "Using physmap partition information\n");
		add_mtd_partitions(info->mtd, physmap_data->parts,
						physmap_data->nr_parts);
		return 0;
	}
#endif

	add_mtd_device(info->mtd);
	return 0;

err_out:
	physmap_flash_remove(dev);
	return err;
}

#ifdef CONFIG_PM
static int physmap_flash_suspend(struct platform_device *dev, 
		pm_message_t state)
{
	struct physmap_flash_info *info = platform_get_drvdata(dev);
	int ret = 0;

	if (info)
		ret = info->mtd->suspend(info->mtd);
	return ret;
}

static int physmap_flash_resume(struct platform_device *dev)
{
	struct physmap_flash_info *info = platform_get_drvdata(dev);
	if (info)
		info->mtd->resume(info->mtd);
	return 0;
}
#else 
#define physmap_flash_suspend NULL
#define physmap_flash_resume  NULL
#endif

static void physmap_flash_shutdown(struct platform_device *dev)
{
	struct physmap_flash_info *info = platform_get_drvdata(dev);
	if (info && info->mtd->suspend(info->mtd) == 0)
		info->mtd->resume(info->mtd);
}

static struct platform_driver physmap_flash_driver = {
	.probe		= physmap_flash_probe,
	.remove		= physmap_flash_remove,
	.suspend	= physmap_flash_suspend,
	.resume		= physmap_flash_resume,
	.shutdown	= physmap_flash_shutdown,
	.driver		= {
		.name	= "physmap-flash",
	},
};

static int __init physmap_init(void)
{
	int err = 0;

	dprintk("%s entered\n", __func__);
	err = platform_driver_register(&physmap_flash_driver);
	if (err)
		dprintk("error!! %d\n", err);
	return err;
}

static void __exit physmap_exit(void)
{
	platform_driver_unregister(&physmap_flash_driver);
}

module_init(physmap_init);
module_exit(physmap_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Jeff Hane");
MODULE_DESCRIPTION("Mobilygen MTD map driver for Merlin NOR");
