/*
 *  drivers/mmc/dwc_msh.h - DesignWare Core Mobile Storage Host driver include file
 * This file is to be included by code that registers devices using this DWC MSH driver
 *
 *  Copyright (C) 2007 Mobilygen Corp., All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/ioport.h>

#ifndef DWC_MSH_H
#define DWC_MSH_H

/* Endianess doesn't seem to be very well supported in the kernel */
#ifndef __LITTLE_ENDIAN
#define		__LITTLE_ENDIAN	1
#endif

#ifndef __BIG_ENDIAN
#define		__BIG_ENDIAN	0
#endif

#ifndef __BYTE_ORDER
#define 	__BYTE_ORDER	__LITTLE_ENDIAN
#endif

#define IORESOURCE_IOINDEX_CCLOCK	0x0	/* This is a IORESOURCE_IO resource */

#define CMD_TIMEOUT_MS			100
#define WAITFORREADY_TIMEOUT_MS	1000
/*
*DWC Mobile Storage Host Registers Map
 */
typedef enum dwcmsh_reg_enum {
	DWCMSH_REG_CTRL		=0x00,	/**Control Register (doc 2.10a p136) */
	DWCMSH_REG_PWREN	=0x04,	/**Power enable Register (doc 2.10a p139) */
	DWCMSH_REG_CLKDIV	=0x08,	/**Clock-divider Register (doc 2.10a p140) */
	DWCMSH_REG_CLKSRC	=0x0C,	/**Clock source Register (doc 2.10a p141) */
	DWCMSH_REG_CLKENA	=0x10,	/**Clock enable Register (doc 2.10a p141) */
	DWCMSH_REG_TMOUT	=0x14,	/**Time-out Register (doc 2.10a p142) */
	DWCMSH_REG_CTYPE	=0x18,	/**Card type Register (doc 2.10a p143) */
	DWCMSH_REG_BLKSIZE	=0x1C,	/**Block-size Register (doc 2.10a p144) */
	DWCMSH_REG_BYTCNT	=0x20,	/**Byte count Register (doc 2.10a p144) */
	DWCMSH_REG_INTMASK	=0x24,	/**Interrupt mask Register (doc 2.10a p145) */
	DWCMSH_REG_CMDARG	=0x28,	/**Command argument Register (doc 2.10a p146) */
	DWCMSH_REG_CMD		=0x2C,	/**Command Register (doc 2.10a p146) */
	DWCMSH_REG_RESP0	=0x30,	/**Response 0 Register (doc 2.10a p149) */
	DWCMSH_REG_RESP1	=0x34,	/**Response 1 Register (doc 2.10a p150) */
	DWCMSH_REG_RESP2	=0x38,	/**Response 2 Register (doc 2.10a p150) */
	DWCMSH_REG_RESP3	=0x3C,	/**Response 3 Register (doc 2.10a p150) */
	DWCMSH_REG_MINTSTS	=0x40,	/**Masked interrupt status Register (doc 2.10a p151) */
	DWCMSH_REG_RINTSTS	=0x44,	/**Raw interrupt status Register (doc 2.10a p152) */
	DWCMSH_REG_STATUS	=0x48,	/**Status Register (doc 2.10a p153) */
	DWCMSH_REG_FIFOTH	=0x4C,	/**FIFO threshold Register (doc 2.10a p155) */
	DWCMSH_REG_CDETECT	=0x50,	/**Card detect Register (doc 2.10a p157) */
	DWCMSH_REG_WRTPRT	=0x54,	/**Write protect Register (doc 2.10a p158) */
	DWCMSH_REG_GPIO		=0x58,	/**GPIO Register (doc 2.10a p158) */
	DWCMSH_REG_TCBCNT	=0x5C,	/**Transfered CIU card byte count Register (doc 2.10a p159) */
	DWCMSH_REG_TBBCNT	=0x60,	/**Transfered host/DMA to/from BIU/FIFO byte count Register (doc 2.10a p159) */
	DWCMSH_REG_DEBNCE	=0x64,	/**Card detect debounce Register (doc 2.10a p160) */
	DWCMSH_REG_USRID	=0x68,	/**User ID Register (doc 2.10a p160) */
	DWCMSH_REG_VERID	=0x6C,	/**Synopsys version ID Register (doc 2.10a p161) */
	DWCMSH_REG_HCON		=0x70,	/**Hardware Configuration Register (doc 2.10a p162) */
	DWCMSH_REG_BMOD		=0x80,	/**Bus Mode Register (doc 2.10a p164) */
	DWCMSH_REG_PLDMND	=0x84,	/**Poll Demand Register (doc 2.10a p165) */
	DWCMSH_REG_DBADDR	=0x88,	/**Descrip[tor List Base Address Register (doc 2.10a p165) */
	DWCMSH_REG_IDSTS	=0x8C,	/**Internal DMAC Status Register Register (doc 2.10a p166) */
	DWCMSH_REG_IDINTEN	=0x90,	/**Internal DMA Interrupt Enable Register (doc 2.10a p168) */
	DWCMSH_REG_DSCADDR	=0x94,	/**Current Host Descriptor Address Register (doc 2.10a p169) */
	DWCMSH_REG_BUFADDR	=0x98,	/**Current Host Buffer Address Register (doc 2.10a p169) */
	DWCMSH_REG_DATA		=0x100,	/**FIFO Data Register (doc 2.10a p132) */
} dwcmsh_reg_t;

/******DWCMSH_REG_CTRL Control Register ************/

#define	DWCMSH_CTRL_CTRLRST		  (1  <<  0)   /* DWCMSH Controller Reset */
#define	DWCMSH_CTRL_FIFORST		  (1  <<  1)   /* FIFO Reset */
#define	DWCMSH_CTRL_DMARST		   (1  <<  2)   /* DMA Reset */
/* Bit 3 Reserved */
#define	DWCMSH_CTRL_IE		   (1  <<  4)   /* Interrupts Enable */
#define	DWCMSH_CTRL_DMAE		 (1  <<  5)   /* DMA Enable */
#define	DWCMSH_CTRL_RDWAIT		   (1  <<  6)   /* Read Wait */
#define	DWCMSH_CTRL_SND_IRQ_RESP	 (1  <<  7)   /* Send IRQ Response */
#define	DWCMSH_CTRL_ABT_RD_DATA	  (1  <<  8)   /* Abort Read Data */
#define	DWCMSH_CTRL_SND_CCSD		 (1  <<  9)   /* Send Command Completion Signal Disable (CCSD)*/
#define	DWCMSH_CTRL_SND_AUTO_STOP_CCSD   (1  <<  10)   /* Send Auto stop CCSD*/
#define	DWCMSH_CTRL_CEATA_DEV_IS	 (1  <<  11)   /* CEATA Device Interrupt Status*/
/* Bit 15:12 Reserved */
#define	DWCMSH_CTRL_CRD_VOL_A		(0x0f  <<  16)  /*Card Voltage A*/
#define	DWCMSH_CTRL_CRD_VOL_B		(0x0f  <<  20)  /*Card Voltage B*/
#define	DWCMSH_CTRL_ENB_OD_PULLUP	(1  <<  24)	 /*External Open-drain pullup*/
#define	DWCMSH_CTRL_IDMA		 (1  <<  25)	 /*Internal DMA Controller*/
/* Bit 26:31 Reserved */
/****** Power Enable Register *******/
#define	DWCMSH_PWREN_EN		  (1  <<  0)	 /* Power Enable, only NUM_CARDS bits implemented*/
/* Bit 30:31 Reserved */
/****** Clock Divider register ******/
#define	DWCMSH_CLKDIV_0		 (0xff  <<  0)	 /*cclk Enable*/
#define	DWCMSH_CLKDIV_1		 (0xff  <<  8)	 /*cclk Enable*/
#define	DWCMSH_CLKDIV_2		 (0xff  <<  16)	 /*cclk Enable*/
#define	DWCMSH_CLKDIV_3		 (0xff  <<  24)	 /*cclk Enable*/

/****** Clock Enable Register ******/
#define	DWCMSH_CLKENA_EN		 (1  <<  0)	 /*cclk Enable*/
#define	DWCMSH_CLKENA_LPWR		   (1  <<  16)	 /*cclk Low Power*/

/****** Timeout Register **********/
#define	DWCMSH_TMOUT_RSP		   (0xff  <<  0)	  /*Response Timeout */
#define	DWCMSH_TMOUT_DATA		  (0xffffff  <<  8)  /*Data Timeout*/

/****** Card Type Register ********/
/*Only NUM_CARDS*2 bits implemented */
#define	DWCMSH_CTYPE_4BIT		   (1  <<  0)	 /* Card Width  ALL card in 4 BIT mode*/
#define	DWCMSH_CTYPE_8BIT		   (1  <<  1)	 /*Card Width */

/****** Block Size register *******/
#define	DWCMSH_BLKSIZ_SIZE		   (0x200  <<  0)	 /* 512 Block Size*/
/* Bit 16:31 Reserved */
/****** Byte Count register ******/
#define	DWCMSH_BYTCNT_CNT		   (0x200  <<  0)	 /* 512 Block Size */

/****** Interrupt Mask register ************/
/****** Mask Interrupt Status register ************/
/****** Raw Interrupt Status register ************/

#define	DWCMSH_RINTSTS_CD		  (1  <<  0)   /* Card Detect */
#define	DWCMSH_RINTSTS_RE		  (1  <<  1)   /* Response Error */
#define	DWCMSH_RINTSTS_CDONE	   (1  <<  2)   /* Command Done */
#define	DWCMSH_RINTSTS_DTO		 (1  <<  3)   /* Data Transfer Over */
#define	DWCMSH_RINTSTS_TXDR		(1  <<  4)   /* Tx FIFO Data Request */
#define	DWCMSH_RINTSTS_RXDR		(1  <<  5)   /* Rx FIFO Data Request */
#define	DWCMSH_RINTSTS_RCRC		(1  <<  6)   /* Response CRC Error */
#define	DWCMSH_RINTSTS_DCRC		(1  <<  7)   /* Data CRC Error */
#define	DWCMSH_RINTSTS_RTO		 (1  <<  8)   /* Response Timeout */
#define	DWCMSH_RINTSTS_DRTO		(1  <<  9)   /* Data Read Timeout */
#define	DWCMSH_RINTSTS_HTO		 (1  <<  10)   /* Data starvation by Host Timeout */
#define	DWCMSH_RINTSTS_FRUN		(1  <<  11)   /* FIFO underrun/overrun Error */
#define	DWCMSH_RINTSTS_HLE		 (1  <<  12)   /* Hardware Locked write  Error */
#define	DWCMSH_RINTSTS_SBE		 (1  <<  13)   /* Start Bit error */
#define	DWCMSH_RINTSTS_ACD		 (1  <<  14)   /* Auto Command Done */
#define	DWCMSH_RINTSTS_EBE		 (1  <<  15)   /* End Bit error (read)/write no CRC */
#define	DWCMSH_RINTSTS_SDIO		(0x0000  <<  16)   /*TBD SDIO cards bit 16,17 is Card 0,1.. */

#define	DWCMSH_INT_CMD_MASK	 (DWCMSH_RINTSTS_RE | \
								DWCMSH_RINTSTS_CDONE | \
								DWCMSH_RINTSTS_RCRC  | \
								DWCMSH_RINTSTS_RTO   | \
								DWCMSH_RINTSTS_EBE   | \
								DWCMSH_RINTSTS_SBE) 

#define	DWCMSH_INT_DATA_MASK (DWCMSH_RINTSTS_DTO | \
								DWCMSH_RINTSTS_TXDR | \
								DWCMSH_RINTSTS_RXDR | \
								DWCMSH_RINTSTS_DCRC | \
								DWCMSH_RINTSTS_DRTO | \
								DWCMSH_RINTSTS_HTO  | \
								DWCMSH_RINTSTS_FRUN | \
								DWCMSH_RINTSTS_SBE  | \
								DWCMSH_RINTSTS_EBE)  

/****** Command register ************/
#define	DWCMSH_CMD_CMDI		  		(0x3f  <<  0)   /* Command Index */
#define	DWCMSH_CMD_RSPE		  		(1  <<  6)   /* Response Expect */
#define	DWCMSH_CMD_RSPL		  		(1  <<  7)   /* Response length */
#define	DWCMSH_CMD_RSPCRC			(1  <<  8)   /*Check Response CRC */
#define	DWCMSH_CMD_DATAE		 	(1  <<  9)   /*Data Expected */
#define	DWCMSH_CMD_RW				(1  <<  10)   /*Read/Write */
#define	DWCMSH_CMD_TNFM		  		(1  <<  11)   /*Transfer Mode */
#define	DWCMSH_CMD_SND_AUTO_STOP	(1  <<  12)   /*Send Auto Stop */
#define	DWCMSH_CMD_WAIT_PRV_DATA 	(1  <<  13)   /*Wait for previous data complete */
#define	DWCMSH_CMD_STOP_ABT	  		(1  <<  14)   /*Stop or Abort command */
#define	DWCMSH_CMD_SND_INIT	  		(1  <<  15)   /* Send Initialization Sequence */
#define	DWCMSH_CMD_CNUM		  		(1  <<  16)   /* Card Number in Use */
#define	DWCMSH_CMD_UPD_CLK	   		(1  <<  21)   /* Update clock register */
#define	DWCMSH_CMD_RCEATA_DEV		(1  <<  22)   /* Read CEATA Device */
#define	DWCMSH_CMD_CCSE		  		(1  <<  23)   /* Command Completion signal expected */
/* Bit 24:30 Reserved */
#define	DWCMSH_CMD_START_CMD	 	(1  <<  31)   /* Start Command */

/****** Status register ************/
#define	DWCMSH_STAT_FIFORX_WM		  (1  <<  0)   /* FIFO Reached receive Watermark */
#define	DWCMSH_STAT_FIFOTX_WM		  (1  <<  1)   /* FIFO Reached Transmitted Watermark */
#define	DWCMSH_STAT_FIFO_EMTY		  (1  <<  2)   /* FIFO is Empty */
#define	DWCMSH_STAT_FIFO_FULL		  (1  <<  3)   /* FIFO is Full */

#define	DWCMSH_STAT_FSM_IDLE		   	(0x0  <<  4)   /* Command FSM States */
#define	DWCMSH_STAT_FSM_SND_INIT	   	(0x1  <<  4)   /*Send Init Sequence */
#define	DWCMSH_STAT_FSM_TX_START	   	(0x2  <<  4)   /* TX command Start Bit */
#define	DWCMSH_STAT_FSM_TX_TX		  	(0x3  <<  4)   /* TX command TX Bit */
#define	DWCMSH_STAT_FSM_TX_IARG			(0x4  <<  4)   /* TX command Index + Arg  */
#define	DWCMSH_STAT_FSM_TX_CRC7			(0x5  <<  4)   /* TX command CRC7  */
#define	DWCMSH_STAT_FSM_TX_END		 	(0x6  <<  4)   /* TX command END  */
#define	DWCMSH_STAT_FSM_RX_START	   	(0x7  <<  4)   /* RX Response Start  */
#define	DWCMSH_STAT_FSM_RX_IRQ		 	(0x8  <<  4)   /* RX Response IRQ response  */
#define	DWCMSH_STAT_FSM_RX_TX		  	(0x9  <<  4)   /* RX Response tx bit  */
#define	DWCMSH_STAT_FSM_RX_CMDI			(0xa  <<  4)   /* RX Response Command Index  */
#define	DWCMSH_STAT_FSM_RX_DATA			(0xb  <<  4)   /* RX Response Data  */
#define	DWCMSH_STAT_FSM_RX_CRC7			(0xc  <<  4)   /* RX Response CRC7  */
#define	DWCMSH_STAT_FSM_RX_END		 	(0xd  <<  4)   /* RX Response End  */
#define	DWCMSH_STAT_FSM_RX_WAITNCC	 	(0xe  <<  4)   /* Command path wait NCC  */
#define	DWCMSH_STAT_FSM_RX_WAIT			(0xf  <<  4)   /* Wait  */

#define	DWCMSH_STAT_DATA_3		 	(1  <<  8)	 /* Raw Selected card[3]  */
#define	DWCMSH_STAT_DATA_BUSY		(1  <<  9)	 /* Card Data Busy  */
#define	DWCMSH_STAT_DATA_SM_BUSY	(1  <<  10)	 /* Data TX/RX State Machine Busy   */
#define	DWCMSH_STAT_RSPI		   	(0x3f  <<  11)  /* Response Index   */
#define	DWCMSH_STAT_FIFO_CNT		(0x1fff  <<  17)  /* FIFO count   */
#define	DWCMSH_STAT_DMA_ACK			(1  <<  30)  /* DMA ACK   */
#define	DWCMSH_STAT_DMA_REQ			(1  <<  31)  /* DMA REQ   */

/****** Card detect register ********/
/*Only NUM_CARDS bits implemented */
#define	DWCMSH_CDETECT_CD		  (1  <<  0)	   /* Card Detect Only one card  */
/* Bit 30:31 Reserved */
/****** Write Protect register ******/
/*Only NUM_CARDS bits implemented */
#define	DWCMSH_WRTPRT_WP		   (1  <<  0)	   /*  Write Protect  */

/****** General Purpose IO register ***/
#define	DWCMSH_GPIO_GPI		   (0xff  <<  0)	 /* Value on GPI Input  */
#define	DWCMSH_GPIO_GPO		   (0xffff  <<  8)	   /* Value on GPI Input  */

/****** Debounce Count register *****/
//#define	DWCMSH_DEBNCE_DCNT		(0xffffff  <<  0)	 /*  Debounce Count */
#define	DWCMSH_DEBNCE_DCNT		(0xa4cb8  <<  0)	 /*  Debounce Count */

/****** User ID register ***********/
#define	DWCMSH_USERID_UID		(0xffffffff  <<  0)	/*  User ID */

/****** Bus Mode register **********/
#define	DWCMSH_BMOD_SWR		  (1  <<  0)	 /* Software Reset */
#define	DWCMSH_BMOD_FB		   (1  <<  1)	 /* Fixed Burst */
#define	DWCMSH_BMOD_DSL		  (0x1f  <<  2)  /* Descriptor Skip Length */
#define	DWCMSH_BMOD_DE		   (1  <<  7)	 /* IDMAC Enable */
#define	DWCMSH_BMOD_PBL_1TNF		 (0  <<  8)	 /* 1 transfer */
#define	DWCMSH_BMOD_PBL_4TNF		 (1  <<  8)	 /* 4 transfer */
#define	DWCMSH_BMOD_PBL_8TNF		 (2  <<  8)	 /* 8 transfer */
#define	DWCMSH_BMOD_PBL_16TNF		(3  <<  8)	 /* 16 transfer */
#define	DWCMSH_BMOD_PBL_32TNF		(4  <<  8)	 /* 32 transfer */
#define	DWCMSH_BMOD_PBL_64TNF		(5  <<  8)	 /* 64 transfer */
#define	DWCMSH_BMOD_PBL_128TNF	   (6  <<  8)	 /* 128 transfer */
#define	DWCMSH_BMOD_PBL_256TNF	   (7  <<  8)	 /* 256 transfer */
/* Bit 11:31 Reserved */

/****** Internal DMAC Interrupt Enable register ***/
/****** Internal DMAC Status register ************/
#define	DWCMSH_IDSTS_TI		 (1  <<  0)  /* Transmit Interrupt */
#define	DWCMSH_IDSTS_RI		 (1  <<  1)  /* Receive Interrupt */
#define	DWCMSH_IDSTS_FBE	 (1  <<  2)  /* Fatal Bus Error Interrupt */
/* Bit 3 Reserved */
#define	DWCMSH_IDSTS_DU		 (1  <<  4)  /* Descriptor Unavailable Interrupt */
#define	DWCMSH_IDSTS_CES	 (1  <<  5)  /* Card Error Summary */
/* Bit 6:7 Reserved */
#define	DWCMSH_IDSTS_NIS	 (1  <<  8)  /* Normal Interrupt Summary */
#define	DWCMSH_IDSTS_AIS	 (1  <<  9)  /* AbNormal Interrupt Summary */

#define	DWCMSH_IDSTS_EBTX	(1  <<  10)  /* Error Bit - Host abort Received durin TX*/
#define	DWCMSH_IDSTS_EBRX	(2  <<  10)  /*  Error Bit - Host abort Received durin RX*/
#define	DWCMSH_IDSTS_FSM_IDLE		(0  <<  13)  /* TBD IDLE*/
#define	DWCMSH_IDSTS_FSM_SUSPEND	(1  <<  13)  /* TBD Suspend*/
#define	DWCMSH_IDSTS_FSM_DRD		(2  <<  13)  /* TBD Descriptor read*/
#define	DWCMSH_IDSTS_FSM_DCHK		(3  <<  13)  /* TBD Descriptor Check*/
#define	DWCMSH_IDSTS_FSM_RD_REQWAIT	  (4  <<  13)  /* TBD Read Request Wait*/
#define	DWCMSH_IDSTS_FSM_WR_REQWAIT	  (5  <<  13)  /* TBD Write Request Wait*/
#define	DWCMSH_IDSTS_FSM_RD		  (6  <<  13)  /* TBD Read*/
#define	DWCMSH_IDSTS_FSM_WR		  (7  <<  13)  /* TBD write*/
#define	DWCMSH_IDSTS_FSM_DCLOSE	  (8  <<  13)  /* TBD Descriptor Close*/
/* Bit 17:31 Reserved */
/****** FIFOTH ************/
#define	DWCMSH_FIFOTH_TX_WM	  (0xfff  <<  0)  /*FIFO threshold when transmitting data to card*/
/* Bit 12:15 Reserved */
#define	DWCMSH_FIFOTH_RX_WM	  (0xfff  <<  16)  /*FIFO threshold when receiving data to card*/


typedef	struct	{
#if __BYTE_ORDER == __BIG_ENDIAN
	uint32_t	reserved		: 0x0a ;	/* LSB 22 */
	uint32_t	fifo_ram_inside		: 0x01 ;	/* LSB 21 */
	uint32_t	ge_dma_data_width	: 0x03 ;	/* LSB 18 */
	uint32_t	dma_interface		: 0x02 ;	/* LSB 16 */
	uint32_t	h_addr_width		: 0x06 ;	/* LSB 10 */
	uint32_t	h_data_width		: 0x03 ;	/* LSB 07 */
	uint32_t	h_bus_type		: 0x01 ;	/* LSB 06 */
	uint32_t	num_cards		: 0x05 ;	/* LSB 01 */
	uint32_t	card_type		: 0x01 ;	/* LSB 00 */
#else
	uint32_t	card_type		: 0x01 ;	/* LSB 00 */
	uint32_t	num_cards		: 0x05 ;	/* LSB 01 */
	uint32_t	h_bus_type		: 0x01 ;	/* LSB 06 */
	uint32_t	h_data_width		: 0x03 ;	/* LSB 07 */
	uint32_t	h_addr_width		: 0x06 ;	/* LSB 10 */
	uint32_t	dma_interface		: 0x02 ;	/* LSB 16 */
	uint32_t	ge_dma_data_width	: 0x03 ;	/* LSB 18 */
	uint32_t	fifo_ram_inside		: 0x01 ;	/* LSB 21 */
	uint32_t	reserved		: 0x0a ;	/* LSB 22 */
#endif
} dwcmsh_reg_hcon_fields;

typedef	union	{
	uint32_t reg;
	dwcmsh_reg_hcon_fields fields;
} dwcmsh_reg_hcon_t ;


/* Few defines for registers access
* Please use DWCMSH_READ_REG and DWCMSH_WRITE_REG to access registers
 */
#define DWCMSH_READ_REG(__base_p__, __reg__) \
	(readl(((unsigned long)(__base_p__))+(__reg__)))

#define DWCMSH_WRITE_REG(__base_p__, __reg__, __val__)	\
	(writel((__val__), ((unsigned long)(__base_p__))+(__reg__)))

/**DWCMSH_CHECK_SIGNATURE
* Return a non-zero value if the Version ID is recognized
* by this driver, return 0 otherwise
 */
#define DWCMSH_CHECK_SIGNATURE(__sig__)			((__sig__)==0x5342210a)

#ifdef CONFIG_MMC_DEBUG
#define info(str, args...) \
	( {	printk(KERN_INFO "%s(%s@%d):\n",DRIVER_NAME,__FILE__,__LINE__); \
	  printk(KERN_INFO str "\n",##args); \
	  } )
#define warning(str, args...) \
	( { 	printk(KERN_WARNING "%s(%s@%d):\n",DRIVER_NAME,__FILE__,__LINE__); \
	  printk(KERN_WARNING str "\n",##args); \
	  } )
#define error(str, args...) \
	( {	printk(KERN_ERR "%s(%s@%d):\n",DRIVER_NAME,__FILE__,__LINE__); \
	  printk(KERN_ERR str "\n",##args); \
	  } )
#else
#define debug(str, args...) 
#define info(str, args...)		( { printk(KERN_INFO str "\n",##args); } )
#define warning(str, args...)	( { printk(KERN_WARNING str "\n",##args); } )
#define error(str, args...) 	( { printk(KERN_ERR str "\n",##args); } )
#endif	/* #ifdef CONFIG_MMC_DEBUG */

typedef struct dwc_msh_data_struct 
{
	dma_addr_t phys_reg_addr;	/* Physical address of the IP */
	u32 *reg_addr;			/* Virtual address of the IP */
	int dma_enable;			/* DMA enable */
	mobi_dma_handle dma_handle;
	uint8_t dma_mode;
	int bus_width;			/* Data FIFO width in Bytes */
	int dma_width;			/* DMA data wisth in Bytes */
	/* Card Clock fed into the IP(defines the useable clock range for a card) */
	uint32_t cclock;		
	struct mmc_host *mmc;
	struct mmc_command *cmd;	/* Current command */
	struct mmc_request *mrq;	/* Current request */
	struct mmc_data	*data;	 	/* Current data request */

	/*Flag indicating when the commnd has been sent. This is used to work out
	*whether or not to send stop
	 */
	unsigned int flags; 
#define DWCMSH_SENT_CMD	(1 << 0)
#define DWCMSH_SENT_STOP   (1 << 1)
#define DWCMSH_USE_DMA	 (1 << 2)
	u32 bus_mode;			   /* Flag for current bus setting */
	spinlock_t		  lock;	   /* Mutex */
	struct tasklet_struct   card_tasklet;   /* Tasklet structures */
	struct tasklet_struct   finish_tasklet;
	struct tasklet_struct   cmd_data_tasklet;
	u32 card_detect;			/* value of CD register */
	u32 isr;				/* Accumulated ISR */

	struct scatterlist	  *cur_sg;	/* We're working on this */
	char *mapped_sg;	 /* This is where it's mapped */
	int	num_sg;	 /* Entries left */
	int offset;	 /* Offset into current sg */
	int remain;	 /* Bytes left in current */

	int size;	   /* Remaining bytes in transfer */

	struct timer_list	   timer;	  /* Timer for timeouts */
	int irq;
	uint8_t disable_hs;
	uint8_t dedicated_dma;

#define DMA_MODE_NONE	0x1
#define DMA_MODE_READ	0x2
#define DMA_MODE_WRITE	0x3
	uint8_t prev_dma_mode;

	struct clk *sdmmc_clk;

} dwc_msh_data_t;

#endif	/* #ifndef DWC_MSH_H */
