/*
 *  drivers/misc/maxregrw.c
 *
 *  Copyright (C) 2009 Maxim IC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/types.h>
#include <linux/proc_fs.h>
#include <linux/version.h>
#include <linux/bitops.h>
#include <linux/uaccess.h>
#include <linux/io.h>

#ifdef CONFIG_PPC32
#include <asm/dcr.h>
#endif

/* 
 * on powerPC, enable this if you want to be able to insmod this 
 * modules without loading the pci driver.  must use insmod!
 */
#undef DISABLE_PCI_ACCESS

#define PFX "REGRW: "
static void __iomem *reg_base;
static uint32_t prev_addr;

#if 1
#define dprintk(x...)	printk(x)
#else
#define dprintk(x...)	do {} while (0)
#endif

#define REGRW_PROC_DIR "driver/regrw"
static struct proc_dir_entry *regrw_proc_dir;

#define DEVICE_DBG_READL_LE_REGISTER 	0x01
#define DEVICE_DBG_READL_BE_REGISTER 	0x02
#define DEVICE_DBG_READL_REGISTER 	    DEVICE_DBG_READ_LE_REGISTER

#define DEVICE_DBG_READW_LE_REGISTER 	0x03
#define DEVICE_DBG_READW_BE_REGISTER 	0x04
#define DEVICE_DBG_READW_REGISTER 	    DEVICE_DBG_READW_LE_REGISTER

#define DEVICE_DBG_READB_LE_REGISTER 	0x05
#define DEVICE_DBG_READB_BE_REGISTER 	0x06
#define DEVICE_DBG_READB_REGISTER 	    DEVICE_DBG_READW_LE_REGISTER

#define DEVICE_DBG_WRITEL_LE_REGISTER 	0x07
#define DEVICE_DBG_WRITEL_BE_REGISTER 	0x08
#define DEVICE_DBG_WRITEL_REGISTER 	    DEVICE_DBG_WRITEL_LE_REGISTER

#define DEVICE_DBG_WRITEW_LE_REGISTER 	0x09
#define DEVICE_DBG_WRITEW_BE_REGISTER 	0x0a
#define DEVICE_DBG_WRITEW_REGISTER 	    DEVICE_DBG_WRITEW_LE_REGISTER

#define DEVICE_DBG_WRITEB_LE_REGISTER 	0x0b
#define DEVICE_DBG_WRITEB_BE_REGISTER 	0x0c
#define DEVICE_DBG_WRITEB_REGISTER 	    DEVICE_DBG_WRITEB_LE_REGISTER

#define DEVICE_DBG_READ_REGISTER 	    DEVICE_DBG_READ_LE_REGISTER
#define DEVICE_DBG_WRITE_REGISTER 	    DEVICE_DBG_WRITEL_LE_REGISTER

#define DEVICE_DBG_READ_DUMP_LE_REGISTER 	0x0d
#define DEVICE_DBG_READ_DUMP_BE_REGISTER 	0x0e
#define DEVICE_DBG_READ_DUMP_REGISTER 	    DEVICE_DBG_READ_DUMP_LE_REGISTER


/* PPC only */
#define DEVICE_DBG_DCR_READ_REGISTER 	0x10
#define DEVICE_DBG_DCR_WRITE_REGISTER 	0x11
#define DEVICE_DBG_SDR_READ_REGISTER 	0x12
#define DEVICE_DBG_SDR_WRITE_REGISTER 	0x13
#define DEVICE_DBG_CPR_READ_REGISTER 	0x14
#define DEVICE_DBG_CPR_WRITE_REGISTER 	0x15
#define DEVICE_DBG_PCI_READ_CONFIG 	    0x16
#define DEVICE_DBG_PCI_WRITE_CONFIG 	0x17
#define DEVICE_DBG_PCI_READ_BAR0 	    0x18
#define DEVICE_DBG_PCI_WRITE_BAR0 	    0x19
#define DEVICE_DBG_PCI_READ_BAR2 	    0x1a
#define DEVICE_DBG_PCI_WRITE_BAR2 	    0x1b
#define DEVICE_DBG_EBC_READ_REGISTER 	0x1c
#define DEVICE_DBG_EBC_WRITE_REGISTER 	0x1d
/* end PPC only */

/* these are defined in the PPC PCI driver */
#define PCI_READ_CONFIG    0x1
#define PCI_WRITE_CONFIG   0x2
#define PCI_READ_BAR0      0x3
#define PCI_WRITE_BAR0     0x4
#define PCI_READ_BAR2      0x5
#define PCI_WRITE_BAR2     0x6

#ifdef CONFIG_PPC32
#ifdef CONFIG_CANYONLANDS
#ifndef DISABLE_PCI_ACCESS
int32_t falcon_pci_regrw(uint8_t cmd, uint32_t offset, uint32_t data, uint8_t size);
#else
static int32_t falcon_pci_regrw(uint8_t cmd, uint32_t offset, uint32_t data, uint8_t size)
{
	printk(KERN_ERR "PCI access is currently disabled\n");
	return 0;
}
#endif /* DISABLE_PCI_ACCESS */
#endif /* CONFIG_CANYONLANDS */
#else
static int32_t falcon_pci_regrw(uint8_t cmd, uint32_t offset, uint32_t data, uint8_t size)
{
	return 0;
}
#endif

/* XXX to be tested on arm, use in/out_le instead of read/write */
uint32_t read_reg(void *addr, uint8_t size, uint8_t endianess)
{
	/* 
	 * the 460ex seems to have both le and be devices, like
	 * gpio is BE but the sata is LE
	 * the mg chips have BE and LE devices but the registers 
	 * are all LE
	 */
#ifdef CONFIG_PPC32
	if (endianess) {
		switch (size) {
		case 1: return in_8(addr);break;
		case 2: return in_be16(addr);break;
		default: return in_be32(addr);break;
		}
	} else {
		switch (size) {
		case 1: return in_8(addr);break;
		case 2: return in_le16(addr);break;
		default: return in_le32(addr);break;
		}
	}
#else
	switch (size) {
	case 1: return readb(addr);break;
	case 2: return readw(addr);break;
	default: return readl(addr);break;
	}
#endif
	return 0;
}

void write_reg(uint32_t data, void *addr, uint8_t size, uint8_t endianess)
{
#ifdef CONFIG_PPC32
	if (endianess) {
		switch (size) {
		case 1: out_8(addr, data);break;
		case 2: out_be16(addr, data);break;
		default: out_be32(addr, data);break;
		}
	} else {
		switch (size) {
		case 1: out_8(addr, data);break;
		case 2: out_le16(addr, data);break;
		default: out_le32(addr, data);break;
		}
	}

#else
	switch (size) {
	case 1: writeb(data, addr);break;
	case 2: writew(data, addr);break;
	default: writel(data, addr);break;
	}
#endif
}

static void proc_advance_ptr(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	while (((**ptr == ' ') || (**ptr == '\t') ||
				(**ptr == '\r') || (**ptr == '\n'))
			&& (*ptr-buf < count)) {
		(*ptr)++;
	}
}

static int proc_get_next_arg(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	proc_advance_ptr(ptr, buffer, count);
	if (*ptr-buf >= count) {
		return -EINVAL;
	}
	return 0;
}

/* use this function after getting fixed number of input */
static int proc_check_for_arg(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	proc_advance_ptr(ptr, buffer, count);
	if (*ptr-buf != count) {
		if (*ptr-buf >= count) {
			return -EINVAL;
		}
		return 1;
	}
	return 0;
}

static int device_proc_wr_debug(struct file *file,
		const char *buffer, unsigned long count, void *data)
{
	char *ptr = (char *)buffer;
	int cmd = 0, i, ret = 0;
	uint32_t addr, size, offset, rwdata = 0, dump_data;

	if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0) {
		cmd = simple_strtoul(ptr, &ptr, 0);
		if (cmd < 0) {
			printk(KERN_ERR PFX "Invalid debug command\n");
			goto err;
		}
	} else {
		printk(KERN_ERR PFX "Invalid cmd string\n");
		goto err;
	}

	/* address */
	if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0) {
		addr = simple_strtoul(ptr, &ptr, 0);
	} else {
		printk(KERN_ERR PFX "Invalid address argument string\n");
		goto err;
	}

	/* offset */
	if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0) {
		offset = simple_strtoul(ptr, &ptr, 0);
	} else {
		printk(KERN_ERR PFX "Invalid offset argument string\n");
		goto err;
	}

	/* size */
	if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0) {
		size = simple_strtoul(ptr, &ptr, 0);
	} else {
		printk(KERN_ERR PFX "Invalid size argument string\n");
		goto err;
	}

	dprintk(KERN_ERR PFX "cmd 0x%x addr 0x%08x, offset 0x%x, size 0x%x",
			cmd, addr, offset, size);

	/* write data */
	ret = proc_check_for_arg(&ptr, (uint32_t)buffer, count);
	if (ret == 1) {
		rwdata = simple_strtoul(ptr, &ptr, 0);
		dprintk(", write data: 0x%08x", rwdata);
	} else if (ret < 0) {
		printk(KERN_ERR PFX "Invalid data argument string\n");
		goto err;
	}
	dprintk("\n");

	/* read everything we don't care about */
	while (ptr-buffer < count)
		ptr++;

	switch (cmd) {
	/* no ioremapping for these commands */
	case DEVICE_DBG_DCR_READ_REGISTER:
	case DEVICE_DBG_DCR_WRITE_REGISTER:
	case DEVICE_DBG_SDR_READ_REGISTER:
	case DEVICE_DBG_SDR_WRITE_REGISTER:
	case DEVICE_DBG_CPR_READ_REGISTER:
	case DEVICE_DBG_CPR_WRITE_REGISTER:
	case DEVICE_DBG_EBC_READ_REGISTER:
	case DEVICE_DBG_EBC_WRITE_REGISTER:
	case DEVICE_DBG_PCI_READ_CONFIG:
	case DEVICE_DBG_PCI_WRITE_CONFIG:
	case DEVICE_DBG_PCI_READ_BAR0:
	case DEVICE_DBG_PCI_WRITE_BAR0:
	case DEVICE_DBG_PCI_READ_BAR2:
	case DEVICE_DBG_PCI_WRITE_BAR2:
		break;
	default:
		if (addr == 0) {
			printk(KERN_ERR PFX "Cannot remap address 0x0\n");
			goto err;
		}
		if (prev_addr != addr) {
			if (reg_base != NULL) {
				iounmap(reg_base);
				reg_base = NULL;
			}

#ifdef CONFIG_4xx
			/* get this warning when building but if don't do this then
			 * get kernel crash....
			 * 206: warning: integer constant is too large for 'long' type
			 */
			reg_base = ioremap((phys_addr_t)(0x400000000|addr), size);
#else
			reg_base = ioremap(addr, size);
#endif
			if (!reg_base) {
				printk(KERN_ERR PFX
						"Failed to ioremap address space at 0x%08x of size %d\n",
						addr, size);
				goto err;
			}
			prev_addr = addr;
		}
		break;
	}

	switch (cmd) {
	/* case DEVICE_DBG_READ_REGISTER: */
	case DEVICE_DBG_READL_LE_REGISTER:
	case DEVICE_DBG_READL_BE_REGISTER:
		rwdata = read_reg(reg_base + offset, 4,
				(cmd == DEVICE_DBG_READL_BE_REGISTER ? 1 : 0));

		printk(KERN_ERR PFX
				"readl address 0x%08x, offset 0x%x = 0x%08x(%u)\n",
				addr, offset, rwdata, rwdata);
		break;
	case DEVICE_DBG_READW_LE_REGISTER:
	case DEVICE_DBG_READW_BE_REGISTER:
		rwdata = (read_reg(reg_base + offset, 2,
					(cmd == DEVICE_DBG_READW_BE_REGISTER ? 1 : 0))
				& 0x0000ffff);

		printk(KERN_ERR PFX
				"readw at address 0x%08x, offset 0x%x = 0x%04x(%u)\n",
				addr, offset, (uint16_t)rwdata, (uint16_t)rwdata);
		break;
	case DEVICE_DBG_READB_LE_REGISTER:
	case DEVICE_DBG_READB_BE_REGISTER:
		rwdata = (read_reg(reg_base + offset, 1,
					(cmd == DEVICE_DBG_READB_BE_REGISTER ? 1 : 0))
				& 0x000000ff);

		printk(KERN_ERR PFX
				"readb address 0x%08x, offset 0x%x = 0x%02x(%u)\n",
				addr, offset, (uint8_t)rwdata, (uint8_t)rwdata);
		break;
	case DEVICE_DBG_READ_DUMP_LE_REGISTER:
	case DEVICE_DBG_READ_DUMP_BE_REGISTER:
		printk(KERN_ERR PFX
			"Dumping %d registers at 0x%08x, starting offset 0x%04x\n", 
			rwdata, addr, offset);

		for (i = 0; i < rwdata; i++) {
			dump_data = read_reg(reg_base + (offset + (i*4)), 4,
					(cmd == DEVICE_DBG_READ_DUMP_BE_REGISTER ? 1 : 0));

			printk(KERN_ERR PFX
					"offset 0x%04x = 0x%08x(%u)\n",
					offset + (i*4), dump_data, dump_data);
		}
		break;
	/*case DEVICE_DBG_WRITE_REGISTER: */
	case DEVICE_DBG_WRITEL_LE_REGISTER:
	case DEVICE_DBG_WRITEL_BE_REGISTER:
		write_reg(rwdata, reg_base + offset, 4,
					(cmd == DEVICE_DBG_WRITEL_BE_REGISTER ? 1 : 0));

		printk(KERN_ERR PFX
				"writel 0x%x(%u) to address 0x%08x offset 0x%x\n",
				rwdata, rwdata, addr, offset);
		break;
	case DEVICE_DBG_WRITEW_REGISTER:
		write_reg((uint16_t)(rwdata & 0x0000ffff), 
					reg_base + offset, 2,
					(cmd == DEVICE_DBG_WRITEW_BE_REGISTER ? 1 : 0));

		printk(KERN_ERR PFX
				"writew 0x%04x(%u) to address 0x%08x offset 0x%x\n",
				(uint16_t)rwdata, (uint16_t)rwdata, addr, offset);
		break;
	case DEVICE_DBG_WRITEB_REGISTER:
		write_reg((uint8_t)(rwdata & 0x000000ff), 
					reg_base + offset, 1,
					(cmd == DEVICE_DBG_WRITEB_BE_REGISTER ? 1 : 0));

		printk(KERN_ERR PFX
				"writeb 0x%x(%u) to address 0x%08x offset 0x%x\n",
				(uint8_t)rwdata, (uint8_t)rwdata, addr, offset);
		break;
	case DEVICE_DBG_PCI_READ_CONFIG:
		ret = falcon_pci_regrw(PCI_READ_CONFIG, offset, rwdata, size);
		if (ret < 0) {
			if (ret == -EPERM)
				printk(KERN_ERR PFX
						"PCI driver debug support not enabled\n");
			else
				printk(KERN_ERR PFX
						"pci_read_config error\n");
		}
		break;
	case DEVICE_DBG_PCI_WRITE_CONFIG:
		ret = falcon_pci_regrw(PCI_WRITE_CONFIG, offset, rwdata, size);
		if (ret < 0) {
			if (ret == -EPERM)
				printk(KERN_ERR PFX
						"PCI driver debug support not enabled\n");
			else
				printk(KERN_ERR PFX
						"pci_write_config error\n");
		}
		break;
	case DEVICE_DBG_PCI_READ_BAR0:
		ret = falcon_pci_regrw(PCI_READ_BAR0, offset, rwdata, size);
		if (ret < 0) {
			if (ret == -EPERM)
				printk(KERN_ERR PFX
						"PCI driver debug support not enabled\n");
			else
				printk(KERN_ERR PFX
						"pci_read_bar0 error\n");
		}
		break;
	case DEVICE_DBG_PCI_WRITE_BAR0:
		ret = falcon_pci_regrw(PCI_WRITE_BAR0, offset, rwdata, size);
		if (ret < 0) {
			if (ret == -EPERM)
				printk(KERN_ERR PFX
						"PCI driver debug support not enabled\n");
			else
				printk(KERN_ERR PFX
						"pci_write_bar0 error\n");
		}
		break;
	case DEVICE_DBG_PCI_READ_BAR2:
		ret = falcon_pci_regrw(PCI_READ_BAR2, offset, rwdata, size);
		if (ret < 0) {
			if (ret == -EPERM)
				printk(KERN_ERR PFX
						"PCI driver debug support not enabled\n");
			else
				printk(KERN_ERR PFX
						"pci_read_bar2 error\n");
		}
		break;
	case DEVICE_DBG_PCI_WRITE_BAR2:
		ret = falcon_pci_regrw(PCI_WRITE_BAR2, offset, rwdata, size);
		if (ret < 0) {
			if (ret == -EPERM)
				printk(KERN_ERR PFX
						"PCI driver debug support not enabled\n");
			else
				printk(KERN_ERR PFX
						"pci_write_bar2 error\n");
		}
		break;
#ifdef CONFIG_PPC32
	case DEVICE_DBG_DCR_READ_REGISTER:
		rwdata = mfdcr(addr);
		printk(KERN_ERR PFX
				"DCR read offset 0x%x = 0x%08x(%u)\n",
				offset, rwdata, rwdata);
		break;
	case DEVICE_DBG_DCR_WRITE_REGISTER:
		mtdcr(addr, rwdata);
		printk(KERN_ERR PFX
				"DCR write 0x%x(%u) to offset 0x%x\n",
				rwdata, rwdata, offset);
		break;
	case DEVICE_DBG_SDR_READ_REGISTER:
		/* 0xe SDR0_CFGADDR, 0xf SDR0_CFGDATA */
		/* could also get these value from the device tree */
		mtdcr(0xe, offset);
		rwdata = mfdcr(0xf);
		printk(KERN_ERR PFX
				"SDR read offset 0x%x = 0x%08x(%u)\n",
				offset, rwdata, rwdata);
		break;
	case DEVICE_DBG_SDR_WRITE_REGISTER:
		/* 0xe SDR0_CFGADDR, 0xf SDR0_CFGDATA */
		mtdcr(0xe, offset);
		mtdcr(0xf, rwdata);
		printk(KERN_ERR PFX
				"SDR write 0x%x(%u) to offset 0x%x\n",
				rwdata, rwdata, offset);
		break;
	case DEVICE_DBG_CPR_READ_REGISTER:
		/* 0xe CPR0_CFGADDR, 0xf SDR0_CFGDATA */
		/* could also get these value from the device tree */
		mtdcr(0xc, offset);
		rwdata = mfdcr(0xd);
		printk(KERN_ERR PFX
				"CPR read offset 0x%x = 0x%08x(%u)\n",
				offset, rwdata, rwdata);
		break;
	case DEVICE_DBG_CPR_WRITE_REGISTER:
		/* 0xe SDR0_CFGADDR, 0xf SDR0_CFGDATA */
		mtdcr(0xc, offset);
		mtdcr(0xd, rwdata);
		printk(KERN_ERR PFX
				"SDR write 0x%x(%u) to offset 0x%x\n",
				rwdata, rwdata, offset);
		break;
	case DEVICE_DBG_EBC_READ_REGISTER:
		/* 0x12 EBC0_CFGADDR, 0xf EBC0_CFGDATA */
		/* could also get these value from the device tree */
		mtdcr(0x12, offset);
		rwdata = mfdcr(0x13);
		printk(KERN_ERR PFX
				"EBC read offset 0x%x = 0x%08x(%u)\n",
				offset, rwdata, rwdata);
		break;
	case DEVICE_DBG_EBC_WRITE_REGISTER:
		/* 0x12 EBC0_CFGADDR, 0x13 EBC0_CFGDATA */
		mtdcr(0x12, offset);
		mtdcr(0x13, rwdata);
		printk(KERN_ERR PFX
				"EBC write 0x%x(%u) to offset 0x%x\n",
				rwdata, rwdata, offset);
		break;

#else
	case DEVICE_DBG_DCR_READ_REGISTER:
	case DEVICE_DBG_DCR_WRITE_REGISTER:
	case DEVICE_DBG_SDR_READ_REGISTER:
	case DEVICE_DBG_SDR_WRITE_REGISTER:
	case DEVICE_DBG_CPR_READ_REGISTER:
	case DEVICE_DBG_CPR_WRITE_REGISTER:
	case DEVICE_DBG_EBC_READ_REGISTER:
	case DEVICE_DBG_EBC_WRITE_REGISTER:
		printk(KERN_ERR PFX
				"Cmd 0x%x only supported on powerPC\n", cmd);
		break;
#endif
	default:
		printk(KERN_ERR PFX
				"ERROR: Unrecognized command: 0x%x\n",
				cmd);
		break;
	}

	return ptr - buffer;

err:
	/* read everything we don't care about */
	while (ptr-buffer < count)
		ptr++;

	return -EINVAL;
}

static int __init maxregrw_init(void)
{
	struct proc_dir_entry *pentry;
	char string[20];

	/* create /proc/driver/regrw */
	if (regrw_proc_dir == NULL)
		regrw_proc_dir = proc_mkdir(REGRW_PROC_DIR, NULL);

	sprintf(string, "%s", "cmd");
	pentry = create_proc_entry(string,
			S_IRUSR | S_IRGRP | S_IROTH,
			regrw_proc_dir);
	if (pentry) {
		pentry->write_proc  = device_proc_wr_debug;
	}

	/*
	np = of_find_compatible_node(NULL, NULL, "ibm,sdr-460ex");
	if (np == NULL) {
		dev_err(dev, "cannot find node\n");
		ret = -ENODEV;
	}
	*/
	printk(KERN_ERR PFX "Maxim debug regrw driver loaded\n");

	return 0;
}

static void __exit maxregrw_exit(void)
{
	char string[20];

	if (reg_base != NULL)
		iounmap(reg_base);

	sprintf(string, "%s", "cmd");
	remove_proc_entry(string, regrw_proc_dir);
	remove_proc_entry(REGRW_PROC_DIR, NULL);

}
module_init(maxregrw_init);
module_exit(maxregrw_exit);

MODULE_AUTHOR("Jeff Hane");
MODULE_DESCRIPTION("Maxim register r/w debug driver");
MODULE_LICENSE("GPL");
