#define DRIVER_NAME 	"gpioi2c"
#define DRIVER_DESC 	"I2C over GPIO driver"
#define DRIVER_AUTHOR 	"Gregoire Pean <gpean@mobilygen.com>"
#define DRIVER_VERSION 	"1:0.1"

/*
 *  This file Copyright (C) 2007 Mobilygen Corp.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE	LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/vermagic.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/platform_device.h>

#include <linux/gpio-core.h>

#include <linux/i2c.h>
#include <linux/i2c-algo-bit.h>

#include <linux/gpioi2c.h>

/*-------------------------------------------------------------------------*/

#define error(format, arg...)	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#define info(format, arg...) 	printk(KERN_INFO DRIVER_NAME ": " format "\n" , ## arg)
#define warn(format, arg...) 	printk(KERN_WARNING DRIVER_NAME ": " format "\n" , ## arg)

#ifdef DEBUG
#define debug(format, arg...) 	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#else
#define debug(format, arg...) 	do {} while (0)
#endif

/*-------------------------------------------------------------------------*/

struct gpioi2c_driver_data {
	struct platform_device		*pdev;
	struct gpioi2c_bus_data		*bdata;
	struct i2c_adapter 		adap;
	struct i2c_algo_bit_data	algo;
	struct gpio_client_pin		pins[2];
};

/*-------------------------------------------------------------------------*/

static void gpioi2c_setscl(void *data, int state)
{
	struct gpioi2c_driver_data *ddata = data;
	if (state) /* Let pullup tie the pin high. */
		gpio_direction_input(ddata->pins[0].hdrv, ddata->pins[0].index);
	else
		gpio_direction_output(ddata->pins[0].hdrv, ddata->pins[0].index, 0);
}

static void gpioi2c_setsda(void *data, int state)
{
	struct gpioi2c_driver_data *ddata = data;
	if (state) /* Let pullup tie the pin high. */
		gpio_direction_input(ddata->pins[1].hdrv, ddata->pins[1].index);
	else
		gpio_direction_output(ddata->pins[1].hdrv, ddata->pins[1].index, 0);
}

static int gpioi2c_getscl(void *data)
{
	struct gpioi2c_driver_data *ddata = data;
	return gpio_get_value(ddata->pins[0].hdrv, ddata->pins[0].index);
}

static int gpioi2c_getsda(void *data)
{
	struct gpioi2c_driver_data *ddata = data;
	return gpio_get_value(ddata->pins[1].hdrv, ddata->pins[1].index);
}

/*-------------------------------------------------------------------------*/

static int gpioi2c_gpiocli_probe(struct gpio_client *client)
{
	int 				ret;
	struct gpioi2c_driver_data	*ddata = client->parent_data;
	
	memset(&ddata->algo, 0, sizeof(ddata->algo));
	memset(&ddata->adap, 0, sizeof(ddata->adap));
	
	ddata->algo.data = ddata;
	ddata->algo.setsda = gpioi2c_setsda;
	ddata->algo.setscl = gpioi2c_setscl;
	ddata->algo.getsda = gpioi2c_getsda;
	ddata->algo.getscl = gpioi2c_getscl;
	ddata->algo.udelay = ddata->bdata->udelay ? 
		ddata->bdata->udelay : GPIOI2C_DEFAULT_UDELAY;
	ddata->algo.timeout = msecs_to_jiffies(ddata->bdata->timeout_msecs ? 
		ddata->bdata->timeout_msecs : GPIOI2C_DEFAULT_TIMEOUT);
	
	snprintf(ddata->adap.name, sizeof(ddata->adap.name), 
		"%s.adap", ddata->pdev->dev.bus_id);
	ddata->adap.id = I2C_HW_B_GPIO;
	ddata->adap.algo_data = &ddata->algo;
	ddata->adap.dev.parent = &ddata->pdev->dev;
	ddata->adap.class = I2C_CLASS_HWMON | I2C_CLASS_SPD;

	if (unlikely(ret = i2c_bit_add_bus(&ddata->adap)) < 0) {
		dev_err(&ddata->pdev->dev, "%s: I2C bus registration failed "
			"with error %d\n", __func__, ret);
		return ret;
	}
	
	dev_info(&ddata->pdev->dev, "bus ready\n");
	
	return 0;
}

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,20)
extern int i2c_bit_del_bus(struct i2c_adapter *);
#endif

static int gpioi2c_gpiocli_remove(struct gpio_client *client)
{
	int ret;
	struct gpioi2c_driver_data *ddata = client->parent_data;
	
	dev_info(&ddata->pdev->dev, "removing bus\n");
	
	if (unlikely(ret = i2c_bit_del_bus(&ddata->adap))) {
		dev_err(&ddata->pdev->dev, "%s: I2C bus unregistration failed "
			"with error %d\n", __func__, ret);
		return ret;
	}
	
	return 0;
}

static void gpioi2c_release(struct device *dev)
{
	struct gpio_client 		*gpiocli = dev_get_drvdata(dev);
	struct gpioi2c_driver_data 	*ddata = gpiocli->parent_data;
	dev_set_drvdata(dev, NULL);
	kfree(ddata);
	kfree(gpiocli);
}

static int gpioi2c_probe(struct platform_device *pdev)
{
	int 				ret;
	struct gpio_client 		*gpiocli;
	struct gpioi2c_bus_data 	*bdata = pdev->dev.platform_data;
	struct gpioi2c_driver_data	*ddata;
	
	gpiocli = kzalloc(sizeof(struct gpio_client), GFP_KERNEL);
	if (!gpiocli)
		return -ENOMEM;
	platform_set_drvdata(pdev, gpiocli);
	gpiocli->probe = gpioi2c_gpiocli_probe;
	gpiocli->remove = gpioi2c_gpiocli_remove;
	gpiocli->match_group = bdata->gpio_group;
	
	ddata = kzalloc(sizeof(struct gpioi2c_driver_data), GFP_KERNEL);
	if (!ddata) {
		ret = -ENOMEM;
		goto err_alloc_ddata;
	}
	ddata->pdev = pdev;
	ddata->bdata = bdata;
	
	gpiocli->parent_data = ddata;
	
	ddata->pins[0].gpio = bdata->pin_scl;
	snprintf(ddata->pins[0].label, sizeof(ddata->pins[0].label), "%s.SCL", pdev->dev.bus_id);
	ddata->pins[1].gpio = bdata->pin_sda;
	snprintf(ddata->pins[1].label, sizeof(ddata->pins[1].label), "%s.SDA", pdev->dev.bus_id);
	
	gpiocli->match_pins = ddata->pins;
	gpiocli->pin_count = 2;
	
	if (unlikely(ret = gpio_client_register(gpiocli))) {
		dev_err(&pdev->dev, "%s: failed to register GPIO client, "
			"error is %d\n", __func__, ret);
		goto err_reg;
	}
	
	if (!pdev->dev.release)
		pdev->dev.release = gpioi2c_release;
	
	return 0;
	
err_reg:
	kfree(ddata);
err_alloc_ddata:
	kfree(gpiocli);
	return ret;
}

static int gpioi2c_remove(struct platform_device *pdev)
{
	int ret;
	struct gpio_client 		*gpiocli = platform_get_drvdata(pdev);
	struct gpioi2c_driver_data	*ddata = gpiocli->parent_data;
	
	if (unlikely(ret = gpio_client_unregister(gpiocli))) {
		dev_err(&ddata->pdev->dev, "%s: failed to unregister GPIO client, "
			"error is %d\n", __func__, ret);
		return ret;
	}
	
	return 0;
}

/*-------------------------------------------------------------------------*/

static struct platform_driver gpioi2c_drv = {
	.probe 		= gpioi2c_probe,
        .remove		= gpioi2c_remove,
        .driver		= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
        },
};

static int __init gpioi2c_init(void)
{
	int ret;

	debug(DRIVER_DESC " compiled for Linux %s, version %s (%s at %s)", 
		UTS_RELEASE, DRIVER_VERSION, __DATE__, __TIME__);
	
	if (unlikely(ret = platform_driver_register(&gpioi2c_drv)))
		error("%s: platform_driver_register failed with error %d", __func__, ret);
	
	return ret;
}

static void __exit gpioi2c_exit(void)
{
        platform_driver_unregister(&gpioi2c_drv);
}

module_init(gpioi2c_init);
module_exit(gpioi2c_exit);

/*-------------------------------------------------------------------------*/

MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_LICENSE("GPL");

