#define DRIVER_NAME 	"gpiosample"
#define DRIVER_DESC 	"Sample client driver for the GPIO framework"
#define DRIVER_AUTHOR 	"Gregoire Pean <gpean@mobilygen.com>"
#define DRIVER_VERSION 	"1:0.0"

/*
 *  This file Copyright (C) 2007 Mobilygen Corp.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE	LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/vermagic.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/platform_device.h>

/* Most important: GPIO framework header. */
#include <linux/gpio-core.h>

#define DEBUG

/*-------------------------------------------------------------------------*/

#define error(format, arg...)	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#define info(format, arg...) 	printk(KERN_INFO DRIVER_NAME ": " format "\n" , ## arg)
#define warn(format, arg...) 	printk(KERN_WARNING DRIVER_NAME ": " format "\n" , ## arg)

#ifdef DEBUG
#define debug(format, arg...) 	printk(KERN_ERR DRIVER_NAME ":%d: " format "\n" , __LINE__, ## arg)
#else
#define debug(format, arg...) 	do {} while (0)
#endif

/*-------------------------------------------------------------------------*/

static struct gpio_client_pin gpiosample_pins[] = {
	/* DECLARE THE GPIO PINS YOU NEED HERE. */
	{
		.label	= DRIVER_NAME ".test",
	},

	/* SAMPLE: */
/*
	{
		.gpio 	= GPIOPIN_BUSYLED,
		.label	= DRIVER_NAME ".BUSYLED",
	},
	{
		.gpio 	= GPIOPIN_RESETSW,
		.label	= DRIVER_NAME ".RESETSW",
	},
	{
		.gpio 	= GPIOPIN_POWERSW,
		.label	= DRIVER_NAME ".POWERSW",
	},
	{
		.gpio 	= GPIOPIN_POWERLED,
		.label	= DRIVER_NAME ".POWERLED",
	},
*/
};

/* For the modparam, see bottom of file. */
static int gpiosample_gpio = 0;

/*-------------------------------------------------------------------------*/

/**
 * This is the interrupt callback that will be called when an interrupt occurs on the GPIO pin.
 * Please note that you can also do the request_irq() call yourself, 
 * but letting the GPIO framework do the job is easier.
 * You can have a look at the gpio-core.c file to see how the request_irq() 
 * is done if you want to do the same manually in your driver.
 */
static int gpiosample_interrupt_callback(unsigned gpio, 
	unsigned gpio_index, void *cbdata)
{
	/* Simply print out something... */
	debug("INT! GPIO %u", gpio);

	/* Clear the interrupt. */
	gpio_interrupt_clear(gpiosample_pins[0].hdrv, 
		gpiosample_pins[0].gpio);
	
	/* Careful! Any non-zero value will disable the IRQ! */
	return 0;
}

/**
 * This probe function will be called when the pins you requested
 * get available.
 */
static int gpiosample_probe(struct gpio_client *client)
{
	int ret;
	
	/* Interrupt configuration. */
	static const struct gpio_interrupt_config int_config = {
		.type 		= GPIO_INT_TYPE_EDGE,
		.polarity 	= GPIO_INT_POLARITY_HIGH_OR_RISING,
		.debounce_on	= 1,
	};
	
	debug("probe: gpio=%u index=%u", gpiosample_pins[0].gpio, 
		gpiosample_pins[0].index);
	
	/* Register our interrupt callback. This will not enable the interrupt. */
	debug("register int");
	ret = gpio_interrupt_register(gpiosample_pins[0].hdrv,
		gpiosample_pins[0].index, 
		gpiosample_interrupt_callback, NULL /* private data */);
	if (unlikely(ret)) {
		error("could not register the interrupt callback (%d)", ret);
		return ret;
	}
	
	/* Configure interrupt type and edge. */
	debug("configure int");
	ret = gpio_interrupt_configure(gpiosample_pins[0].hdrv,
		gpiosample_pins[0].index,  &int_config);
	if (unlikely(ret)) {
		error("could not configure the interrupt (%d)", ret);
		goto end_unreg;
	}

	/* Enable the interrupt. */
	debug("enable int");
	ret = gpio_interrupt_enable(gpiosample_pins[0].hdrv, 
		gpiosample_pins[0].index);
	if (unlikely(ret)) {
		error("could not enable the interrupt (%d)", ret);
		goto end_unreg;
	}
	
	debug("probe success");
	return 0;
	
end_unreg:
	debug("unregister int");
	gpio_interrupt_unregister(gpiosample_pins[0].hdrv,
		gpiosample_pins[0].index, 
		gpiosample_interrupt_callback);
	return ret;
}

/**
 * This remove function will be called when the pins you requested
 * are being unregistered.
 */
static int gpiosample_remove(struct gpio_client *client)
{
	debug("remove");
	
	debug("unregister int");
	return gpio_interrupt_unregister(gpiosample_pins[0].hdrv, 
		gpiosample_pins[0].index, 
		gpiosample_interrupt_callback);
}

/*-------------------------------------------------------------------------*/

/* This is like a platform_driver struct... */
static struct gpio_client gpiosample_cli = {
	/* NULL means match any group. On the MG3500 there is
	 * only one GPIO group so that's fine. */
	.match_group	= NULL,
	.match_pins	= gpiosample_pins,
	.pin_count	= ARRAY_SIZE(gpiosample_pins),
	.probe		= gpiosample_probe,
	.remove		= gpiosample_remove,

	/* Pass any private data using those members: */
	/*
	.parent_data	= ...,
	.client_data	= ...,
	*/
};

static int __init gpiosample_init(void)
{
	int ret;

	info(DRIVER_DESC " compiled for Linux %s, version %s (%s at %s)",
		UTS_RELEASE, DRIVER_VERSION, __DATE__, __TIME__);
	
	gpiosample_pins[0].gpio = gpiosample_gpio;
	
	/* Register our client driver with the GPIO framework. */
	debug("registering");
	if (unlikely(ret = gpio_client_register(&gpiosample_cli))) {
		error("%s: failed to register GPIO client, "
			"error is %d\n", __func__, ret);
	}
	
	return ret;
}

static void __exit gpiosample_exit(void)
{
	int ret;
	debug("unregistering");
        if (unlikely(ret = gpio_client_unregister(&gpiosample_cli))) {
		error("%s: failed to unregister GPIO client, "
			"error is %d\n", __func__, ret);
	}
}

module_init(gpiosample_init);
module_exit(gpiosample_exit);

module_param(gpiosample_gpio, int, 0444);	
MODULE_PARM_DESC(gpiosample_gpio, "GPIO number");

/*-------------------------------------------------------------------------*/

MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_LICENSE("GPL");





