/*
 * arch/arm/mach-merlin/timers.c
 *
 * Copyright (C) Mobilygen Corp
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/mm.h>
#include <linux/types.h>
#include <linux/time.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/version.h>
#include <linux/io.h>

#include <mach/hardware.h>
#include <asm/mach-types.h>
#include <asm/setup.h>
#include <asm/irq.h>

#include <asm/mach/arch.h>
#include <asm/mach/irq.h>
#include <asm/mach/map.h>
#include <asm/mach/time.h>

#include <mach/platform.h>
#include <mach/timex.h>
#include <mach/dw_timers_regs.h>
#include <mach/mobi_timer.h>
#include <mach/mobi_clock.h>


#define MOBILYGEN_DEBUG_TIMERS 0

#if MOBILYGEN_DEBUG_TIMERS
#define dprintk(x...)   printk(x)
#else
#define dprintk(x...)   do {} while (0)
#endif

#define TIMER_ID_4 4
#define TIMER_ID_5 5

/* User definable flags must fit in this mask */
#define TIMER_FLAG_MASK 0x7
/* Internal flag indicating timer has a connection to the FIQ */
#define TIMER_ON_FIQ 0x1000

#define RAW_TICK_TIMER_ID TIMER_ID_5

#define WRITEL(data, base, offset) \
	writel(data, base+offset)

#define READL(base, offset) \
	readl(base+offset)

struct mobi_timer {
	char *name;
	void __iomem *base;
	unsigned int irq;
	void (*callback)(void);
	uint32_t count;
	uint32_t flags;
	struct mutex lock;
};

/*
 * Ensure the TIMER_ON_FIQ timers come first, mobi_get_timer will request
 * them last.
 */
static struct mobi_timer timers[] = {
	{
	.name	= "timer4",
	.base	= __io_address(TIMERS_BASE+TIMER4_OFFSET),
	.irq	= MOBI_IRQ_TIMER4,
	.flags	= TIMER_ON_FIQ,
	},
	{
	.name	= "timer5",
	.base	= __io_address(TIMERS_BASE+TIMER5_OFFSET),
	.irq	= MOBI_IRQ_TIMER5,
	}
};

/**
 * \brief get_timer:
 * 	Returns a pointer to the mobi_timer struct for a given timerID
 *
 * \param timerID - timer handle
 *
 * \retval pointer to struct mobi_timer for timerID
 * \retval NULL if timerID is invalid
 *
 */
static inline struct mobi_timer *get_timer(timer_handle timerID)
{
	int idx = timerID - TIMER_ID_4;
	if (idx < 0 || idx >= ARRAY_SIZE(timers))
		return NULL;
	return &timers[idx];
}

/**
 * \brief load_timer:
 * 	Load specified timer with initial value
 *
 * \param t		- timer
 *
 */
static void load_timer(struct mobi_timer *t)
{
	uint32_t ctrl_reg;

	WRITEL(TIMER_DISABLE, t->base, TIMER_CONTROL);
	WRITEL(t->count, t->base, TIMER_LOAD_COUNT);
	ctrl_reg = TIMER_ENABLE | TIMER_MODE_USER;
	if (t->flags & TIMER_NOINT)
		ctrl_reg |= TIMER_INTERRUPT_MASK;
	WRITEL(ctrl_reg, t->base, TIMER_CONTROL);
}

/**
 * \brief mobi_request_timer
 *  	Try to find a free timer.
 *
 * \param func	- callback function
 * \param flags	- option flags
 * 	- TIMER_COUNT_OUNCE - timer will only trigger once
 * 	- TIMER_USE_FIQ     - request timer the can trigger FIQ
 *	- TIMER_NOINT       - this will mask the interrupt
 *	                      (func is ignored and not called)
 *
 * \retval -ENODEV - if no free timer is found
 * \retval timer_handle  - id of timer
 *
 */
timer_handle mobi_request_timer(void (*func)(void), uint32_t flags)
{
	struct mobi_timer *t;
	int i;

	/* Proceed backward so the FIQ timer will be the last tried */
	for (i = ARRAY_SIZE(timers)-1; i >= 0; i--) {
		t = &timers[i];
		if ((flags & TIMER_USE_FIQ) && !(t->flags & TIMER_ON_FIQ))
			continue;
		if (mutex_trylock(&t->lock) != 1)
			continue;
		if (flags & TIMER_USE_FIQ)
			disable_irq(t->irq);
		t->callback = func;
		t->flags |= flags & TIMER_FLAG_MASK;
		return TIMER_ID_4+i;
	}
	return -ENODEV;
}
EXPORT_SYMBOL(mobi_request_timer);

/**
 * \brief mobi_set_timer
 *  	Program a timer for a give time interval.
 *
 * \param td - timer handle
 * \param tv - mobi_timeval with initial starting value define
 * \param flags - accepted flags
 *  		- TIMER_COUNT_ONCE: only trigger the counter one time
 *
 * \retval -ENODEV
 * 	- if invalid timer handle is provided
 * \retval -EINVAL
 * 	- if startvalue is great than 32 bit for 32 bit timer
 * 	- if td is a 32bit timer and try to set 64BIT_COUNTER flag
 * \retval -EIO
 * 	- if timer clock is disabled or rate cannot be determined
 *  \retval Zero
 *  	- on success
 *
 * \remark
 *  	Convert time into ticks which will be programmed into
 *  the LoadCount, enables interrupts and starts the timer.
 */
int32_t mobi_set_timer(timer_handle td, struct mobi_timeval tv, uint32_t flags)
{
	struct mobi_timer *t;
	uint64_t count;
	uint32_t clock_rate;

	t = get_timer(td);
	if (t == NULL)
		return -ENODEV;

	clock_rate = mobi_clock_get_rate(CLOCK_ID_TIMER);
	if (clock_rate <= 0) {
		printk(KERN_ERR "%s: Timer clock is not running\n",
				__func__);
		return -EIO;
	}

	if (tv.tv_hour == 0 && tv.tv_min == 0 &&
			tv.tv_sec == 0 && tv.tv_usec == 0)
		return -EINVAL;

	t->flags |= flags & TIMER_FLAG_MASK;

	count  = (uint64_t)tv.tv_hour*60*60;
	count += (uint64_t)tv.tv_min*60;
	count += (uint64_t)tv.tv_sec;
	count *= clock_rate;
	count += tv.tv_usec;

	dprintk("timer: %d reqested time is %ld hours, "
			"%ld min, %ld secs, %ld usec,"
			"count = 0x%llx\n",
			td, tv.tv_hour, tv.tv_min,
			tv.tv_sec, tv.tv_usec, count);

	/* is requested amount greater
	 * than we can do in a 32-bit counter?
	 */
	if (count > 0xffffffff)
		return -EINVAL;

	t->count = count;
	load_timer(t);

	return 0;
}
EXPORT_SYMBOL(mobi_set_timer);

/**
 * \brief mobi_restart_timer
 * 	If a process wants to restart the timer.
 *
 * \param td - timer_handle
 *
 * \retval -ENODEV - if invalid td is given
 * \retval Zero    - upon success
 *
 * \remark
 *   This applies to contineous and one-shot counters.  Timer will
 * be disabled, reloaded with start value and restarted.  Note,
 * the start value cannot be changed, must delete and create new
 * timer to do this.
 *
 */
int32_t mobi_restart_timer(timer_handle td)
{
	struct mobi_timer *t;

	t = get_timer(td);
	if (t == NULL)
		return -ENODEV;

	load_timer(t);

	return 0;
}
EXPORT_SYMBOL(mobi_restart_timer);

/**
 * \brief mobi_get_timer
 * 	Return the current value of the timer.
 *
 * \param td  - timer handle
 * \param tv  - empty mobi_timeval struct
 *
 * \retval -ENODEV
 * 	- given an invalid timer handle
 * \retval -EACCESS
 * 	- if timer has not been enabled
 * \retval -EIO
 * 	- if timer clock is disabled or rate cannot be determined
 * \retval Zero
 * 	- up success and tv will contain current timer counts
 *
 * \remark
 *   Current value is returned in provided mobi_timeval struct.
 *  Do not have to be holding lock to read this value but
 *  timer must be enabled.
 */
int32_t mobi_get_timer(timer_handle td, struct mobi_timeval *tv)
{
	uint32_t clock_rate;
	struct mobi_timer *t;

	t = get_timer(td);
	if (t == NULL)
		return -ENODEV;

	/* There is the implicit assumption clock_rate = 1 MHz */
	clock_rate = mobi_clock_get_rate(CLOCK_ID_TIMER);
	if (clock_rate <= 0) {
		printk(KERN_ERR "%s: Timer clock is not running\n",
				__func__);
		return -EIO;
	}

	if (READL(t->base, TIMER_CONTROL) & TIMER_ENABLE) {
		uint32_t cur;

		cur = READL(t->base, TIMER_CURRENT_VALUE);
		tv->tv_hour = (time_t) cur/(60*60*clock_rate);
		cur -= tv->tv_hour*(60*60*clock_rate);

		tv->tv_min = (time_t) cur/(60*clock_rate);
		cur -= tv->tv_min*(60*clock_rate);

		tv->tv_sec = cur/clock_rate;
		cur -= tv->tv_sec*clock_rate;
		tv->tv_usec = cur;
	} else {
		return -EACCES;
	}

	return 0;
}
EXPORT_SYMBOL(mobi_get_timer);

/**
 * \brief mobi_del_timer
 * 	Disable and release selected timer
 *
 * \param td - timer handle
 *
 * \retval -ENODEV  - if td is invalid
 * \retval -EACCESS - if caller is not the timer owner
 * \retval Zero     - upon success
 *
 * \remark
 *   Caller must be the current owner of the timer.  After successful
 * completion, timer is avail for use by other processes.
 *
 */
int32_t mobi_del_timer(timer_handle td)
{
	struct mobi_timer *t;

	t = get_timer(td);
	if (t == NULL)
		return -ENODEV;

	t->callback = NULL;

	WRITEL(TIMER_DISABLE, t->base, TIMER_CONTROL);
	if (t->flags & TIMER_USE_FIQ)
		enable_irq(t->irq);
	t->flags &= ~TIMER_FLAG_MASK;
	mutex_unlock(&t->lock);

	return 0;
}
EXPORT_SYMBOL(mobi_del_timer);

/**
 * \brief mobi_timer_interrupt:
 * 	Internal IRQ handler for the timer
 */
static irqreturn_t mobi_timer_interrupt(int irq, void *dev_id)
{
	struct mobi_timer *t;

	t = get_timer(irq-MOBI_IRQ_TIMER4+TIMER_ID_4);
	if (!t)
		return IRQ_NONE;

	if (READL(t->base, TIMER_INT_STATUS) == 1) {
		READL(t->base, TIMER_EOI); /* clear the interrupt */
		if (t->callback != NULL)
			(*t->callback)();
	}

	if (t->flags & TIMER_COUNT_ONCE)
		WRITEL(TIMER_DISABLE, t->base, TIMER_CONTROL);

	return IRQ_HANDLED;
}

/**
 * \brief mobi_request_tick
 * 	Set up a timer to be used a precise tick
 *
 * \param t	- none
 *
 * \retval	- -ENODEV if already requested
 *
 */
int mobi_request_tick(void)
{
	struct mobi_timer *t;
	uint32_t ctrl_reg;

	t = get_timer(RAW_TICK_TIMER_ID);
	if (!mutex_trylock(&t->lock))
		return -ENODEV;

	WRITEL(TIMER_DISABLE, t->base, TIMER_CONTROL);
	WRITEL(~0, t->base, TIMER_LOAD_COUNT);
	ctrl_reg = TIMER_INTERRUPT_MASK | TIMER_MODE_FREERUN | TIMER_ENABLE;
	WRITEL(ctrl_reg, t->base, TIMER_CONTROL);

	return 0;
}
EXPORT_SYMBOL(mobi_request_tick);

/**
 * \brief mobi_get_tick
 * 	Get the current tick
 *
 * \retval	- the current tick value
 *
 */
inline unsigned long mobi_get_tick(void)
{
	struct mobi_timer *t;

	t = get_timer(RAW_TICK_TIMER_ID);
	return READL(t->base, TIMER_CURRENT_VALUE);
}
EXPORT_SYMBOL(mobi_get_tick);

/**
 * \brief mobi_release_tick
 * 	Release a successfuly requested tick
 *
 */
void mobi_release_tick(void)
{
	struct mobi_timer *t;

	t = get_timer(RAW_TICK_TIMER_ID);
	mutex_unlock(&t->lock);
}
EXPORT_SYMBOL(mobi_release_tick);

static inline int __init timer_register(struct mobi_timer* t)
{
	WRITEL(TIMER_DISABLE, t->base, TIMER_CONTROL);
	mutex_init(&t->lock);
	return request_irq(t->irq, mobi_timer_interrupt,
			   IRQF_DISABLED, t->name, NULL);
}

static int __init mobi_timers_init(void)
{
	unsigned long clk_rate = 0;
	int i, ret;

	clk_rate = mobi_clock_get_rate(CLOCK_ID_TIMER);
	if (clk_rate <= 0) {
		printk(KERN_ERR "Timers clock failed to start\n");
		return -EIO;
	}

	for (i = 0; i < ARRAY_SIZE(timers); i++) {
		ret = timer_register(&timers[i]);
		if (ret)
			return ret;
	}

	return 0;
}
arch_initcall(mobi_timers_init);
