/*HEADER_START*/
/*
 * arch/arm/plat-mgx/time.c
 *
 * Copyright (C) Mobilygen Corp
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*HEADER_STOP*/

#include <linux/init.h>
#include <linux/module.h>
#include <linux/mm.h>
#include <linux/types.h>
#include <linux/time.h>
#include <linux/interrupt.h>
#include <linux/version.h>
#include <linux/io.h>
#include <linux/clocksource.h>
#include <linux/clockchips.h>

#include <mach/hardware.h>
#include <asm/mach-types.h>
#include <asm/setup.h>
#include <asm/irq.h>

#include <asm/mach/arch.h>
#include <asm/mach/irq.h>
#include <asm/mach/map.h>
#include <asm/mach/time.h>

#include <mach/platform.h>
#include <mach/dw_timers_regs.h>
#include <mach/mobi_clock.h>

#define TIMER_PERIODIC (TIMER_ENABLE | TIMER_MODE_USER)
#define TIMER_FREERUN  (TIMER_ENABLE | TIMER_MODE_FREERUN | TIMER_INTERRUPT_MASK)

#define WRITEL(data, base, offset) \
	writel((uint32_t) data, __io_address(base)+(uint32_t)offset)

#define READL(base, offset) \
	readl(__io_address(base)+(uint32_t)offset)

#define TIMER_CLOCK_RATE 1000000

static cycle_t read_cycles(struct clocksource *cs)
{
	unsigned long value;
	value = READL(TIMER1_BASE, TIMER_CURRENT_VALUE);
	return (cycle_t)(~value);
}

static struct clocksource clocksource_mgx = {
	.name		= "timer1",
	.rating		= 200,
	.read		= read_cycles,
	.mask		= CLOCKSOURCE_MASK(32),
	.shift		= 24,
	.flags		= CLOCK_SOURCE_IS_CONTINUOUS,
};

/*
 * clockevent
 */
static unsigned timer_mode = TIMER_DISABLE;

static int mgx_set_next_event(unsigned long cycles,
				  struct clock_event_device *evt)
{
	WRITEL(TIMER_DISABLE, TIMER3_BASE, TIMER_CONTROL);
	WRITEL(cycles, TIMER3_BASE, TIMER_LOAD_COUNT);
	WRITEL(timer_mode, TIMER3_BASE, TIMER_CONTROL);
	return 0;
}

static void mgx_set_mode(enum clock_event_mode mode,
			     struct clock_event_device *evt)
{
	switch (mode) {
	case CLOCK_EVT_MODE_PERIODIC:
		timer_mode = TIMER_ENABLE | TIMER_MODE_USER;
		mgx_set_next_event(TIMER_CLOCK_RATE/HZ, evt);
		break;
	case CLOCK_EVT_MODE_ONESHOT:
		timer_mode = TIMER_ENABLE | TIMER_MODE_FREERUN;
		break;
	case CLOCK_EVT_MODE_UNUSED:
	case CLOCK_EVT_MODE_SHUTDOWN:
		timer_mode = TIMER_DISABLE;
		break;
	case CLOCK_EVT_MODE_RESUME:
		break;
	}
}

static struct clock_event_device clockevent_mgx = {
	.name		= "timer3",
	.features       = CLOCK_EVT_FEAT_PERIODIC | CLOCK_EVT_FEAT_ONESHOT,
	.shift		= 32,
	.set_next_event	= mgx_set_next_event,
	.set_mode	= mgx_set_mode,
};

/*
 * IRQ handler for the timer
 */
static irqreturn_t mgx_timer_interrupt(int irq, void *dev_id)
{
	/* clear the interrupt */
	READL(TIMER3_BASE, TIMER_EOI);

	/* Disable timer after one-shot */
	if (clockevent_mgx.mode != CLOCK_EVT_MODE_PERIODIC)
		WRITEL(TIMER_DISABLE, TIMER3_BASE, TIMER_CONTROL);
	clockevent_mgx.event_handler(&clockevent_mgx);

	return IRQ_HANDLED;
}

static struct irqaction mgx_timer_irq = {
	.name           = "Timer Tick",
	.flags          = IRQF_DISABLED | IRQF_TIMER,
	.handler        = mgx_timer_interrupt,
};

/*
 * Set up timer interrupt, and return the current time in seconds.
 */
static void __init mgx_systimer_init(void)
{
	unsigned long clk_rate, pclk_rate;

	/*
	 * use to query the sys_timer clock and set the reload value
	 */
	mobi_clock_set_rate(CLOCK_ID_TIMER, TIMER_CLOCK_RATE);
	clk_rate = mobi_clock_get_rate(CLOCK_ID_TIMER);
	pclk_rate = mobi_clock_get_rate(CLOCK_ID_APB);

	/*
	 * Initialize all timers to a known state
	 */
	WRITEL(TIMER_DISABLE, TIMER1_BASE, TIMER_CONTROL);
	WRITEL(TIMER_DISABLE, TIMER2_BASE, TIMER_CONTROL);
	WRITEL(TIMER_DISABLE, TIMER3_BASE, TIMER_CONTROL);
	WRITEL(TIMER_DISABLE, TIMER4_BASE, TIMER_CONTROL);
	WRITEL(TIMER_DISABLE, TIMER5_BASE, TIMER_CONTROL);

	WRITEL(~0, TIMER1_BASE, TIMER_LOAD_COUNT);
	WRITEL(~0, TIMER2_BASE, TIMER_LOAD_COUNT);

	setup_irq(MOBI_IRQ_TIMER3, &mgx_timer_irq);

	WRITEL(TIMER_FREERUN, TIMER2_BASE, TIMER_CONTROL);
	WRITEL(TIMER_FREERUN, TIMER1_BASE, TIMER_CONTROL);

	/* setup clocksource */
	clocksource_mgx.mult =
		clocksource_hz2mult(pclk_rate,
				    clocksource_mgx.shift);
	clocksource_register(&clocksource_mgx);

	/* setup clockevent */
	clockevent_mgx.mult = div_sc(clk_rate, NSEC_PER_SEC,
					 clockevent_mgx.shift);
	clockevent_mgx.max_delta_ns =
		clockevent_delta2ns(0xfffffffe, &clockevent_mgx);
	clockevent_mgx.min_delta_ns =
		clockevent_delta2ns(1, &clockevent_mgx);

	clockevent_mgx.cpumask = cpumask_of(0);
	clockevents_register_device(&clockevent_mgx);
}

struct sys_timer mgx_sys_timer = {
	.init       = mgx_systimer_init,
};
