/*HEADER_START*/
/*
 * Copyright (C) 2009 Maxim IC
 *
 * linux/arch/arm/mach-falcon/irq.c
 *
 * Derived from at91rm9200 irq.c
 * linux/arch/arm/mach-at91rm9200/irq.c
 *  Copyright (C) 2004 SAN People
 *  Copyright (C) 2004 ATMEL
 *  Copyright (C) Rick Bronson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*HEADER_STOP*/

#include <linux/module.h>
#include <linux/init.h>
#include <linux/mm.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/io.h>

#include <mach/hardware.h>
#include <asm/irq.h>
#include <asm/fiq.h>
#include <asm/mach-types.h>

#include <asm/mach/arch.h>
#include <asm/mach/irq.h>
#include <asm/mach/map.h>

#include <mach/platform.h>
#include <mach/dw_ictl_regs.h>
#include <mach/mobi_qcc.h>
#include <mach/mobi_fiq.h>

#define WRITEL(data, offset) \
	writel((uint32_t) data, __io_address(ICTL_BASE)+(uint32_t)offset)

#define READL(offset) \
	readl(__io_address(ICTL_BASE)+(uint32_t)offset)

static struct mutex fiq_lock;
static int32_t fiq_lock_pid = -1;
static int32_t fiq_cur_num  = -1;

static struct fiq_handler fh = {
	.name = "falcon_fiq",
};

int mobi_claim_fiq(uint8_t fiq)
{
	int ret = 0;
	/* first come, first served */
	if (mutex_trylock(&fiq_lock) != 1)
		return -ENODEV;

	switch (fiq) {
#ifdef CONFIG_MACH_MERLIN
	case MOBI_FIQ_DMA0:
	case MOBI_FIQ_DMA1:
	case MOBI_FIQ_DMA2:
	case MOBI_FIQ_USB:
	case MOBI_FIQ_GPIO0:
	case MOBI_FIQ_GPIO1:
	case MOBI_FIQ_GPIO2:
	case MOBI_FIQ_TIMER:
#endif
#ifdef CONFIG_MACH_FALCON
	case MOBI_FIQ_nPMU:
	case MOBI_FIQ_TIMER1:
	case MOBI_FIQ_TIMER2:
	case MOBI_FIQ_TIMER3:
	case MOBI_FIQ_TIMER4:
	case MOBI_FIQ_TIMER5:
	case MOBI_FIQ_WDT:
#endif
		break;
	default:
		printk(KERN_ERR "Invalid FIQ: %d\n", fiq);
		mutex_unlock(&fiq_lock);
		return -EINVAL;
	}
	ret = claim_fiq(&fh);
	if (ret == 0) {
		fiq_lock_pid = current->pid;
		fiq_cur_num = fiq;
	} else {
		mutex_unlock(&fiq_lock);
	}
	return ret;
}
EXPORT_SYMBOL(mobi_claim_fiq);

int mobi_release_fiq(void)
{
	if (fiq_lock_pid != current->pid) {
		printk(KERN_ERR "Process %d is not the current owner of the FIQ\n",
				current->pid);
		return -EPERM;
	}
	if (fiq_cur_num != -1) {
		printk(KERN_ERR "FIQ must be disabled before it can be released\n");
		return -EPERM;
	}

	release_fiq(&fh);
	fiq_lock_pid  = -1;

	mutex_unlock(&fiq_lock);

	return 0;
}
EXPORT_SYMBOL(mobi_release_fiq);

int mobi_enable_fiq(void)
{
	int ret = 0;

	if (fiq_lock_pid != current->pid) {
		printk(KERN_ERR
				"Process %d is not the current"
				" owner of the FIQ\n", current->pid);
		return -EPERM;
	}

#ifdef CONFIG_MACH_MERLIN
	ret = mobi_qcc_writereg(fiq_cur_num, MOBI_QCCSISC_FIQ_OFFSET);
	if (!ret) {
		WRITEL(0xfffffffe, MOBI_ICTL_FIQ_INTMASK_OFFSET);
		WRITEL(0x1, MOBI_ICTL_FIQ_INTEN_OFFSET);
	}
#endif

#ifdef CONFIG_MACH_FALCON
	/*
	 * ALL 8 fiqs feed the fiq in falcon, so we must set
	 * mask accordingly, we will continue to support only
	 * one fiq at a time.  multiple one seem pointless and
	 * confusing anyway
	 */
	if (fiq_cur_num < 0) {
		return -EINVAL;
	} else {
		WRITEL(~(0x1 << fiq_cur_num), MOBI_ICTL_FIQ_INTMASK_OFFSET);
		WRITEL((0x1 << fiq_cur_num), MOBI_ICTL_FIQ_INTEN_OFFSET);
	}
#endif

	return ret;
}
EXPORT_SYMBOL(mobi_enable_fiq);

int mobi_set_fiq_handler(void *fiq_handler, struct pt_regs regs)
{
	uint32_t fiqhandler[2];

	if (fiq_lock_pid != current->pid) {
		printk(KERN_ERR
				"Process %d is not the current"
				" owner of the FIQ\n", current->pid);
		return -EPERM;
	}
	fiqhandler[0] = 0xE51FF004;    /* ldr pc, [pc, #-4] */
	fiqhandler[1] = (uint32_t)fiq_handler;

	/* how this works.  we install a 2 word fiq handler at the
	 *  default vector address(0xffff001c), the first word is a
	 *  ldr instruction that loads the second word, which is the
	 *  fiq handler address, into the pc! I think the fiq vector
	 *  space is only about 200 bytes, this way we don't have to
	 *  use that vectore space, we can "jump" out to larger space
	 *  which must be compiled in(no module code for FIQ handler)
	 */
	set_fiq_handler((void *) fiqhandler, 8);
	set_fiq_regs(&regs);

	return 0;
}
EXPORT_SYMBOL(mobi_set_fiq_handler);

int mobi_disable_fiq(void)
{
	if (fiq_lock_pid != current->pid) {
		printk(KERN_ERR
				"Process %d is not the current"
				" owner of the FIQ\n", current->pid);
		return -EPERM;
	}

	WRITEL(0x0, MOBI_ICTL_FIQ_INTEN_OFFSET);
	WRITEL(0xffffffff, MOBI_ICTL_FIQ_INTMASK_OFFSET);
	fiq_cur_num  = -1;

	return 0;
}
EXPORT_SYMBOL(mobi_disable_fiq);

/* ================  normal IRQ stuff ============================ */
static void mgx_ictl_mask_irq(unsigned int irq)
{
	unsigned int reg = 0x0;

	reg = READL((irq < 32 ?
				MOBI_ICTL_IRQ_INTEN_L_OFFSET :
				MOBI_ICTL_IRQ_INTEN_H_OFFSET));
	reg &= ~(1 << (irq < 32 ? irq : (irq - 32)));
	/* Disable interrupt */
	WRITEL(reg, (irq < 32 ?
				MOBI_ICTL_IRQ_INTEN_L_OFFSET :
				MOBI_ICTL_IRQ_INTEN_H_OFFSET));
}

static void mgx_ictl_unmask_irq(unsigned int irq)
{
	unsigned int reg = 0x0;

	reg = READL((irq < 32 ?
				MOBI_ICTL_IRQ_INTEN_L_OFFSET :
				MOBI_ICTL_IRQ_INTEN_H_OFFSET));
	reg |= (1 << (irq < 32 ? irq : (irq - 32)));
	/* Enable interrupt */
	WRITEL(reg, (irq < 32 ?
				MOBI_ICTL_IRQ_INTEN_L_OFFSET :
				MOBI_ICTL_IRQ_INTEN_H_OFFSET));
}

static int mgx_ictl_set_irq_type(unsigned irq, unsigned type)
{
	/*
	 * don't support this.  the polarity of the interrupts are set by the
	 * hardware and are not programmable.
	 */

	return 0;
}

#ifdef CONFIG_PM

static u32 wakeups_l;
static u32 wakeups_h;
static u32 backups_l;
static u32 backups_h;

static int mgx_ictl_set_wake(unsigned irq, unsigned value)
{
	if (unlikely(irq >= 39))
		return -EINVAL;

	if (irq < 32) {
		if (value)
			wakeups_l |= (1 << irq);
		else
			wakeups_l &= ~(1 << irq);
	} else {
		if (value)
			wakeups_h |= (1 << (irq - 32));
		else
			wakeups_h &= ~(1 << (irq - 32));
	}

	return 0;
}

void mgx_irq_suspend(void)
{
	backups_l = READL(MOBI_ICTL_IRQ_INTMASK_L_OFFSET);
	WRITEL(~(backups_l), MOBI_ICTL_IRQ_INTEN_L_OFFSET);
	WRITEL(wakeups_l, MOBI_ICTL_IRQ_INTEN_L_OFFSET);
	backups_h = READL(MOBI_ICTL_IRQ_INTMASK_H_OFFSET);
	WRITEL(~(backups_h), MOBI_ICTL_IRQ_INTEN_H_OFFSET);
	WRITEL(wakeups_h, MOBI_ICTL_IRQ_INTEN_H_OFFSET);
}

void mgx_irq_resume(void)
{
	WRITEL(wakeups_l, MOBI_ICTL_IRQ_INTEN_L_OFFSET);
	WRITEL(backups_l, MOBI_ICTL_IRQ_INTEN_L_OFFSET);
	WRITEL(wakeups_h, MOBI_ICTL_IRQ_INTEN_H_OFFSET);
	WRITEL(backups_h, MOBI_ICTL_IRQ_INTEN_H_OFFSET);
}

#else
#define mgx_ictl_set_wake	NULL
#endif

static struct irq_chip mgx_ictl_chip = {
	.name		= "ICTL",
	.ack		= mgx_ictl_mask_irq,
	.mask		= mgx_ictl_mask_irq,
	.unmask		= mgx_ictl_unmask_irq,
	.set_type	= mgx_ictl_set_irq_type,
	.set_wake	= mgx_ictl_set_wake,
};

/*
 * Initialize the interrupt controller.
 */
void __init mgx_ictl_init(void)
{
	unsigned int i;

	/* disable all the interrupts */
	WRITEL(0x0, MOBI_ICTL_IRQ_INTEN_L_OFFSET);
	WRITEL(0x0, MOBI_ICTL_IRQ_INTEN_H_OFFSET);
	/* fiq stuff too */
	WRITEL(0x0, MOBI_ICTL_FIQ_INTEN_OFFSET);
	WRITEL(0xffffffff, MOBI_ICTL_FIQ_INTMASK_OFFSET);
	mutex_init(&fiq_lock);

	for (i = 0; i < NR_IRQS; i++) {
		set_irq_chip(i, &mgx_ictl_chip);
		set_irq_handler(i, handle_level_irq);
		set_irq_flags(i, IRQF_VALID | IRQF_PROBE);
	}

#ifdef CONFIG_FIQ
	/* Initialize FIQ */
	init_FIQ();
#endif
}
