#define DRIVER_NAME 	"i2c-mux"
#define DRIVER_DESC 	"Virtual muxing I2C adapter for MG3500"
#define DRIVER_AUTHOR 	"Gregoire Pean <gpean@mobilygen.com>"
#define DRIVER_VERSION 	"1:3.1"

/*
 *  Copyright (C) 2007 Mobilygen Corp., All Rights Reserved.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE	LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/vermagic.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/platform_device.h>

#include <linux/i2c.h>

#include <asm/atomic.h>

#include "mach/mobi_ioctrl.h"

/*-------------------------------------------------------------------------*/

#define error(format, arg...)	\
	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#define info(format, arg...) 	\
	printk(KERN_INFO DRIVER_NAME ": " format "\n" , ## arg)
#define warn(format, arg...) 	\
	printk(KERN_WARNING DRIVER_NAME ": " format "\n" , ## arg)

#ifdef DEBUG
#define debug(format, arg...) 	\
	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#else
#define debug(format, arg...) 	do {} while (0)
#endif

/*-------------------------------------------------------------------------*/

#define I2C_DRIVERID_I2CMMUX		3500
#define PARENT_ADAPTER			"dwapbi2c.I2C_1"
#define NUM_CHILD_ADAPTER		2 /* max being 10, see i2cmmux_access */

static struct i2c_adapter 		i2cmmux_adapters[NUM_CHILD_ADAPTER];
static struct mutex			access_lock;
struct workqueue_struct 		*attach_workqueue;
static atomic_t				stop;
static const struct ioctrl_macro_s	*macros[NUM_CHILD_ADAPTER];
static const char *macro_names[] = {
	"I2C_1.V01",
	"I2C_1.V23",
};

struct i2cmmux_work_data {
	char				do_attach;
	struct i2c_adapter 		*adap;
	struct work_struct		work;
};

/*-------------------------------------------------------------------------*/

/*
 * "Fake" platform driver to avoid an i2c subsystem warning if you dont pass
 * a "physical device" upon adapter registration. And we cannot use the parent
 * adapter device otherwise it will hang (i2c subsystem deadlocking).
 */

static int i2cmmux_fakepdrv_probe(struct platform_device *pdev)
{
	return 0;
}

static int i2cmmux_fakepdrv_remove(struct platform_device *pdev)
{
	return 0;
}

static struct platform_driver i2cmmux_fakepdrv_driver = {
	.probe		= i2cmmux_fakepdrv_probe,
	.remove		= i2cmmux_fakepdrv_remove,
	.suspend	= NULL,
	.resume		= NULL,
	.driver	= {
		.name	= "i2cmmux_fakepdrv",
		.owner	= THIS_MODULE,
	},
};

static void i2cmmux_fakepdrv_dev_release(struct device *dev) {}

static struct platform_device i2cmmux_fakepdrv_device = {
	.name           = "i2cmmux_fakepdrv",
	.id             = -1,
	.dev            = { .release = i2cmmux_fakepdrv_dev_release },
	.num_resources  = 0,
};

/*-------------------------------------------------------------------------*/
static int i2cmmux_access(struct i2c_adapter *adap,
		struct i2c_msg *messages, int count)
{
	int ret = 0, child_adap_id;
	struct i2c_adapter *parent_adap = adap->algo_data;

	if (unlikely(atomic_read(&stop)))
		return 0;

	/* XXX: sscanf or something like that to
	 * increase the number of muxable busses?
	 */
	child_adap_id = adap->name[strlen(adap->name) - 1] - '0';

	/* Merlin dependant code begins here */
	if (unlikely(child_adap_id < 0 || child_adap_id > 1))
		return -ENODEV;

	mutex_lock(&access_lock);

	/*
	 * Safety first: in case we just got the lock after the detach
	 * function released it and the adapter no longer exists.
	 */
	if (unlikely(!i2cmmux_adapters[child_adap_id].name[0])) {
		mutex_unlock(&access_lock);
		return -ENODEV;
	}

	mobi_ioctrl_apply_macro(macros[child_adap_id]);

	debug("calling master_xfer count=%d qcc_reg=0x%x old_reg=%x",
			count, new_reg, reg);
	ret = parent_adap->algo->master_xfer(parent_adap, messages, count);

	mutex_unlock(&access_lock);
	return ret;
}

static u32 i2cmmux_function(struct i2c_adapter *adap)
{
	struct i2c_adapter *parent_adap = adap->algo_data;
	return parent_adap->algo->functionality(parent_adap);
}

static struct i2c_algorithm i2cmmux_algorithm = {
	.master_xfer	= i2cmmux_access,
	.functionality	= i2cmmux_function,
};

static void i2cmmux_attach_pump(struct work_struct *work)
{
	struct i2cmmux_work_data *wdata =
		container_of(work, struct i2cmmux_work_data, work);
	int i;

	mutex_lock(&access_lock);

	/*
	 * (Un)register our fake platform driver/device then
	 * create(remove) as many virtual i2c adapters as necessary.
	 */

	if (wdata->do_attach) {
		platform_driver_register(&i2cmmux_fakepdrv_driver);
		platform_device_register(&i2cmmux_fakepdrv_device);
		for (i = 0; i < NUM_CHILD_ADAPTER; i++) {
			sprintf(i2cmmux_adapters[i].name,
					"%s.%d", PARENT_ADAPTER, i);
			i2cmmux_adapters[i].algo = &i2cmmux_algorithm;
			i2cmmux_adapters[i].algo_data = (void *)wdata->adap;
			i2cmmux_adapters[i].class = I2C_CLASS_HWMON;
			/* We cannot pass &wdata->adap->dev;
			 * otherwise it hangs.
			 */
			i2cmmux_adapters[i].dev.parent =
				&i2cmmux_fakepdrv_device.dev;
			if (unlikely(i2c_add_adapter(&i2cmmux_adapters[i]))) {
				error("%s: child I2C adapter [%s] registration failed",
						__func__,
						i2cmmux_adapters[i].name);
				i2cmmux_adapters[i].name[0] = 0;
			} else {
				info("created virtual adapter [%s]",
						i2cmmux_adapters[i].name);
			}
			macros[i] = mobi_ioctrl_get_macro(macro_names[i]);
		}
	} else {
		for (i = 0; i < NUM_CHILD_ADAPTER; i++) {
			if (likely(i2cmmux_adapters[i].name[0])) {
				if (unlikely(i2c_del_adapter(&i2cmmux_adapters[i])))
					error("%s: child I2C adapter [%s] removal failed",
							__func__,
							i2cmmux_adapters[i].name);
				else
					info("removed virtual adapter [%s]",
							i2cmmux_adapters[i].name);
				i2cmmux_adapters[i].name[0] = 0;
			}
		}
		platform_device_unregister(&i2cmmux_fakepdrv_device);
		platform_driver_unregister(&i2cmmux_fakepdrv_driver);
	}
	mutex_unlock(&access_lock);
	kfree(wdata);
}

/*
 * We have to defer virtual adapters creation in 
 * a workqueue otherwise the i2c subsystem
 * locking system will make the things hang.
 */
static int i2cmmux_attach_adapter(struct i2c_adapter *adap)
{
	int ret = 0;

	if (!strcmp(adap->name, PARENT_ADAPTER)) {
		struct i2cmmux_work_data *wdata;
		wdata = kzalloc(sizeof(struct i2cmmux_work_data), GFP_KERNEL);
		wdata->adap = adap;
		wdata->do_attach = 1;
		INIT_WORK(&wdata->work, i2cmmux_attach_pump);
		queue_work(attach_workqueue, &wdata->work);
	}

	return ret;
}

static int i2cmmux_detach_adapter(struct i2c_adapter *adap)
{
	int ret = 0;

	if (!strcmp(adap->name, PARENT_ADAPTER)) {
		struct i2cmmux_work_data *wdata;
		wdata = kzalloc(sizeof(struct i2cmmux_work_data), GFP_KERNEL);
		wdata->adap = adap;
		wdata->do_attach = 0;
		INIT_WORK(&wdata->work, i2cmmux_attach_pump);
		queue_work(attach_workqueue, &wdata->work);
	}

	return ret;
}

static struct i2c_driver i2cmmux_driver = {
	.driver = {
		.name	= DRIVER_NAME,
	},
	.id		= I2C_DRIVERID_I2CMMUX,
	.attach_adapter	= i2cmmux_attach_adapter,
	.detach_adapter	= i2cmmux_detach_adapter,
};

static int __init i2cmmux_init(void)
{
	int ret = 0;

	debug(DRIVER_DESC " compiled for Linux %s, version %s (%s at %s)",
			UTS_RELEASE, DRIVER_VERSION, __DATE__, __TIME__);

	atomic_set(&stop, 0);
	mutex_init(&access_lock);

	attach_workqueue = create_singlethread_workqueue(DRIVER_NAME);
	if (unlikely(!attach_workqueue)) {
		error("%s: cannot create workqueue", __func__);
		ret = -ENOMEM;
		goto err_workqueue;
	}

	ret = i2c_add_driver(&i2cmmux_driver);
	if (unlikely(ret)) {
		error("%s: could not add I2C driver", __func__);
		goto err_adddrv;
	}

err_adddrv:
err_workqueue:
	return ret;
}

static void __exit i2cmmux_exit(void)
{
	atomic_set(&stop, 1);
	i2c_del_driver(&i2cmmux_driver);
	flush_workqueue(attach_workqueue);
	mutex_lock(&access_lock);
	destroy_workqueue(attach_workqueue);
	mutex_unlock(&access_lock);
}
module_init(i2cmmux_init);
module_exit(i2cmmux_exit);

/*-------------------------------------------------------------------------*/
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_LICENSE("GPL");
