#define DRIVER_NAME 	"gpio-core"
#define DRIVER_DESC 	"GPIO Framework"
#define DRIVER_AUTHOR 	"Gregoire Pean <gpean@mobilygen.com>"
#define DRIVER_VERSION 	"1:7.8"

/*
 *  This file Copyright (C) 2007 Mobilygen Corp.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE	LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/vermagic.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/poll.h>

#ifdef CONFIG_PROC_FS
/* /sys interface should be used instead but keep proc around */
#define CONFIG_GPIO_CREATE_PROC_ENTRIES
#include <linux/proc_fs.h>
#else
#undef CONFIG_GPIO_CREATE_PROC_ENTRIES
#endif

#include <mach/gpio.h>

/*-------------------------------------------------------------------------*/

#define error(format, arg...)	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#define info(format, arg...) 	printk(KERN_INFO DRIVER_NAME ": " format "\n" , ## arg)
#define warn(format, arg...) 	printk(KERN_WARNING DRIVER_NAME ": " format "\n" , ## arg)

#ifdef DEBUG
#define debug(format, arg...) 	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#else
#define debug(format, arg...) 	do {} while (0)
#endif

#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
#define gpio_arg_error(drv) \
	do { \
		struct gpio_driver *__drv = drv; \
		error("%s(%d): BUG, please FIX: function received an " \
			"invalid argument! hdrv=%p, drv->name=%s", \
			__func__, __LINE__, __drv, __drv ? __drv->name : "?"); \
		dump_stack(); \
	} while (0)
#else
#define gpio_arg_error(drv) \
	do { } while(0);
#endif

/*-------------------------------------------------------------------------*/

struct gpio_group
{
	struct list_head 	list;
	char	 		name[GPIO_DRV_NAME_MAXLEN];
	int			refcount;
	struct class		*class;
	struct proc_dir_entry	*pentry;
};

static				LIST_HEAD(gpio_drivers);
static				LIST_HEAD(gpio_groups);
static				LIST_HEAD(gpio_pending_clients);
static				LIST_HEAD(gpio_loaded_clients);
static				DECLARE_MUTEX(global_lock);
static				DECLARE_MUTEX(client_lock);
#ifdef CONFIG_GPIO_CREATE_PROC_ENTRIES
static struct proc_dir_entry	*root_pentry = NULL;
#endif

/*-------------------------------------------------------------------------*/

/* Misc. functions */

static int gpio_bank_safe_lock(struct gpio_driver *drv)
{
	int ret = 0;
	if (in_atomic() || irqs_disabled() || drv->dont_lock) {
		if (mutex_is_locked(&drv->bank_lock))
			ret = -EBUSY;
	} else {
		if (unlikely(mutex_lock_interruptible(&drv->bank_lock)))
			ret = -EINTR;
	}
	return ret;
}

static inline void gpio_bank_safe_unlock(struct gpio_driver *drv)
{
	if (!in_atomic() && !irqs_disabled() && !drv->dont_lock)
		mutex_unlock(&drv->bank_lock);
}

static inline void gpio_bank_safe_spin_lock(struct gpio_driver *drv)
{
	if (atomic_inc_return(&drv->pending_spin_unlock) == 1) {
		if (!in_atomic() && !irqs_disabled())
			spin_lock_irqsave(&drv->irq_lock, drv->spin_flags);
	}
}

static inline void gpio_bank_safe_spin_unlock(struct gpio_driver *drv)
{
	if (atomic_dec_return(&drv->pending_spin_unlock) == 0)
		spin_unlock_irqrestore(&drv->irq_lock, drv->spin_flags);
	BUG_ON(atomic_read(&drv->pending_spin_unlock) < 0);
}

static unsigned int list_size(struct list_head *head)
{
        unsigned int count = 0;
        struct list_head *pos;
        list_for_each(pos, head)
                count++;
        return count;
}
 
/*-------------------------------------------------------------------------*/

/* /sys access functions */

static ssize_t gpio_attr_show_control_count(struct device *class_dev, char *buf, int count)
{
	int ret = 0, x;
	char b[32], y;
	strcpy(b, dev_name(class_dev));	
	strcat(b, "X");
	if (sscanf(b, GPIO_FILE_FORMAT "%c", &x, &y) == 2 && y == 'X') {
		struct gpio_pin_info *info = class_dev->driver_data;
		ret += snprintf(buf + ret, count - ret, "%d\n",
			arch_gpio_get_value((gpio_driver_handle)info->drv,
			info->cached_index));
	} else if (sscanf(b, GPIO_INT_FILE_FORMAT "%c", &x, &y) == 2 && y == 'X') {
		struct gpio_pin_info *info = class_dev->driver_data;
		struct gpio_interrupt_callback *scb;
		struct gpio_interrupt_config config;
		const char *pol, *type;
		if (unlikely(ret = gpio_interrupt_get_config(info->drv, 
			info->cached_index, &config))) {
				error("%s: interrupt_get_config unavailable/failed with error %d for %s/%s", 
				      __func__, ret, info->drv->name, dev_name(class_dev));
			return 0;
		}
		if ((config.type & GPIO_INT_TYPE_LEVEL) && 
			(config.type & GPIO_INT_TYPE_EDGE)) {
			type = "LEVEL+EDGE";
		} else if (config.type & GPIO_INT_TYPE_LEVEL) {
			type = "LEVEL";
		} else if (config.type & GPIO_INT_TYPE_EDGE) {
			type = "EDGE";
		} else {
			type = "UNKNOWN";
		}
		if ((config.polarity & GPIO_INT_POLARITY_LOW_OR_FALLING) && 
			(config.polarity & GPIO_INT_POLARITY_HIGH_OR_RISING)) {
			pol = "HIGH+LOW";
		} else if (config.polarity & GPIO_INT_POLARITY_LOW_OR_FALLING) {
			pol = "LOW/FALLING";
		} else if (config.polarity & GPIO_INT_POLARITY_HIGH_OR_RISING) {
			pol = "HIGH/RISING";
		} else {
			pol = "UNKNOWN";
		}
		ret += snprintf(buf + ret, count - ret, "enabled=%d irq=%d type=%s "
			"polarity=%s debounce=%s active=%d counter=%u\n",
			gpio_interrupt_is_enabled((gpio_driver_handle)info->drv, info->cached_index),
			info->irq, type, pol, config.debounce_on ? "ON" : "OFF",
			gpio_interrupt_is_active((gpio_driver_handle)info->drv, info->cached_index, -1),
			atomic_read(&info->int_counter));
		gpio_bank_safe_spin_lock(info->drv);
		list_for_each_entry(scb, &info->callbacks, list) {
			ret += snprintf(buf + ret, count - ret, "  cb=0x%x cbdata=0x%x\n",
				(unsigned int)scb->cb, (unsigned int)scb->cbdata);
		}
		gpio_bank_safe_spin_unlock(info->drv);
	} else {
		struct gpio_driver *drv = class_dev->driver_data;
		u32 *data;
		int i, r, dw_count = gpio_driver_bank_size_dwords((gpio_driver_handle)drv);
		data = kzalloc(dw_count << 2, GFP_KERNEL);
		if (unlikely(!data))
			return -ENOMEM;
		r = gpio_read_bank((gpio_driver_handle)drv, data, dw_count);
		if (r > 0) {
			for (i=0; i<r; i++)
				ret += snprintf(buf + ret, count - ret, "0x%08x ", data[i]);
		} 
		kfree(data);
		if (unlikely(r < 0)) {
			error("%s: gpio_read_bank unavailable/failed with "
				"error %d for %s", __func__, r, drv->name);
			return r;
		}
		ret += snprintf(buf + ret, count - ret, "\n");
	}
	return ret;
}

static ssize_t gpio_attr_show_control(struct device *class_dev, struct device_attribute *attr, char *buf)
{
	return gpio_attr_show_control_count(class_dev, buf, PAGE_SIZE);
}

static ssize_t gpio_attr_store_control(struct device *class_dev, struct device_attribute *attr, 
				       const char *buf, size_t count)
{
	int ret = 0, x;
	char b[32], y;
	if (!count)
		return 0;	
	strcpy(b, dev_name(class_dev));
	strcat(b, "X");
	if (sscanf(b, GPIO_FILE_FORMAT "%c", &x, &y) == 2 && y == 'X') {
		struct gpio_pin_info *info = class_dev->driver_data;
		switch (*buf) {
		case '0':   /* off */
		case 0x0:
			ret = arch_gpio_direction_output((gpio_driver_handle)info->drv, 
				info->cached_index, 0);
			break;
		case '1':   /* on */
		case 0x1:
			ret = arch_gpio_direction_output((gpio_driver_handle)info->drv, 
				info->cached_index, 1);
			break;
		case 'Z':   /* tristate */
		case 'z':
			ret = arch_gpio_direction_input((gpio_driver_handle)info->drv, 
				info->cached_index);
			break;
		case 'p':   /* pulse in us */
		case 'P': {
			unsigned long t = 0;
			if (sscanf(buf + 1, "%lu", &t) == 1 || sscanf(buf + 1, " %lu", &t) == 1) {
				int val = arch_gpio_get_value((gpio_driver_handle)info->drv, 
					info->cached_index);
				if (unlikely(val < 0))
					return val;
				ret = arch_gpio_direction_output((gpio_driver_handle)info->drv, 
					info->cached_index, !val);
				if (unlikely(ret < 0))
					return ret;
				udelay(t);
				ret = arch_gpio_direction_output((gpio_driver_handle)info->drv, 
					info->cached_index, val);
			}
		}	break;
 		default:
			error("%s/%s: syntax for controlling a pin is: '0'/'1' to driver low/high, 'z' "
				"to set to input (tristate), 'p100' to pulse during 100us", 
				info->drv->name, dev_name(class_dev));
			ret = -EINVAL;
		}
	} else if (sscanf(b, GPIO_INT_FILE_FORMAT "%c", &x, &y) == 2 && y == 'X') {
		struct gpio_pin_info *info = class_dev->driver_data;
		switch (*buf) {
		case '0':   /* disable */
		case 0x0:
			ret = gpio_interrupt_disable((gpio_driver_handle)info->drv, info->cached_index);
			break;
		case '1':   /* enable */
		case 0x1:
			ret = gpio_interrupt_enable((gpio_driver_handle)info->drv, info->cached_index);
			break;
		case 'x':   /* clear */
		case 'X':
			ret = gpio_interrupt_clear((gpio_driver_handle)info->drv, info->cached_index);
			break;
		case 'c':   /* configure */
		case 'C': {
			struct gpio_interrupt_config config;
			if (unlikely(ret = gpio_interrupt_get_config(info->drv, 
				info->cached_index, &config)))
				break;
			if (*(buf+1) == 'e')
				config.type = GPIO_INT_TYPE_EDGE;
			else if (*(buf+1) == 'l')
				config.type = GPIO_INT_TYPE_LEVEL;
			else if (*(buf+1) == 'b')
				config.type = GPIO_INT_TYPE_LEVEL | GPIO_INT_TYPE_EDGE;
			else if (*(buf+1) == 'p')
				config.polarity = GPIO_INT_POLARITY_LOW_OR_FALLING;
			else if (*(buf+1) == 'P')
				config.polarity = GPIO_INT_POLARITY_HIGH_OR_RISING;
			else if (*(buf+1) == 'q')
				config.polarity = GPIO_INT_POLARITY_HIGH_OR_RISING
					| GPIO_INT_POLARITY_LOW_OR_FALLING;
			else if (*(buf+1) == 'd')
				config.debounce_on = 0;
			else if (*(buf+1) == 'D')
				config.debounce_on = 1;
			ret = gpio_interrupt_configure((gpio_driver_handle)info->drv, 
				info->cached_index, &config);
		}	break;
		default:
			error("%s/%s: bad input for interrupt control, valid commands are:", 
			      info->drv->name, dev_name(class_dev));
			error("  * '0'/'1' to enable/disable");
			error("  * 'x' to clear");
			error("  * 'c[elb]' to set type to edge, level, both");
			error("  * 'c[pPq]' to set polarity to low, high, both");
			error("  * 'c[dD]' to set debounce mode on, off (may not be available)");
			ret = -EINVAL;
		}
	} else {
		struct gpio_driver *drv = class_dev->driver_data;
		int i, skip = 0, cur_dw = 0;
		int max_dwcount = gpio_driver_bank_size_dwords((gpio_driver_handle)drv);
		unsigned x;
		u32 *data, *mask;
		if (max_dwcount <= 0)
			return -EINVAL;
		data = kzalloc(max_dwcount * sizeof(u32), GFP_KERNEL);
		if (unlikely(!data))
			return -ENOMEM;
		mask = kzalloc(max_dwcount * sizeof(u32), GFP_KERNEL);
		if (unlikely(!mask)) {
			kfree(data);
			return -ENOMEM;
		}
		
#define issp(c) (c == ' ' || c == '\n' || c == ';' || c == '\t' || c == '\r')
		for (i=0; i<count; i++) {
			if (!buf[i])
				break;
			if (issp(buf[i])) {
				skip = 0;
				continue;
			}
			if (skip)
				continue;
			if (sscanf(buf + i, "%i", &x) == 1) {
				if (cur_dw & 1)
					data[cur_dw >> 1] = x;
				else
					mask[cur_dw >> 1] = x;
				cur_dw++;
				skip = 1;
			} else {
				goto err_input;
			}
		}
		
		if (cur_dw & 1) /* Odd number of args, meaning missing data. */
			goto err_input;
		
		/* Call write_bank only if one mask is not 0. */
		for (i=0; i<max_dwcount; i++) {
			if (mask[i] != 0) {
				ret = gpio_write_bank((gpio_driver_handle)drv, data, mask, max_dwcount);
				if (ret > 0)
					ret = 0;
				goto end;
			}
		}
		
err_input:
		error("%s/%s: bad input for writing bank, syntax is '<mask> <data>; <mask> <data>; ...'", 
		      drv->name, dev_name(class_dev));
		error("  where data and mask are 32 bits wide words (prefix with 0x if hex).");
		printk(KERN_ERR DRIVER_NAME ":   (raw) input was: ");
		for (i=0; i<count; i++)
			printk("0x%02x ", buf[i]);
		printk("\n");
		ret = -EINVAL;
end:
		kfree(data);
		kfree(mask);
	}
	
	return ret ? ret : count;
}

static ssize_t gpio_attr_show_status_count(struct device *class_dev, char *buf, int count)
{
	struct gpio_driver *drv = class_dev->driver_data;
	char c, d, e;
	int i, r, ret = 0;
	static const char *again = "\n...\nRead again to see the rest\n";
	int again_size = strlen(again);
	int limit = count - 256;
	
	if (!drv->status_offset)
		ret += snprintf(buf + ret, count - ret, GPIO_BANK_FILE_FORMAT ": <enabled> <dir>,"
			"<val> <int-enabled>,<active> <owner>\n", drv->bank_id);
	
	for (i=drv->status_offset; i<drv->bank_size; i++) {
		
		if (ret + again_size >= limit) {
			drv->status_offset = i;
			strcpy(buf + ret, again);
			ret += again_size + 1;
			return ret;
		}
	
		r = gpio_is_enabled((gpio_driver_handle)drv, i);
		if (unlikely(r < 0))
			e = '?';
		else if (r)
			e = 'e';
		else
			e = 'd';
		r = gpio_get_direction((gpio_driver_handle)drv, i);
		if (unlikely(r < 0))
			c = 'E';
		else if (r == GPIO_DIRECTION_INPUT)
			c = 'i';
		else
			c = 'o';
		r = arch_gpio_get_value((gpio_driver_handle)drv, i);
		if (unlikely(r < 0))
			d = 'E';
		else if (r)
			d = '1';
		else
			d = '0';
		ret += snprintf(buf + ret, count - ret, GPIO_INDEX_LINK_FORMAT "," 
			GPIO_FILE_FORMAT ": %c %c,%c", i, drv->bank[i].gpio, e, c, d);
		if (c != 'E' && d != 'E' && drv->bank[i].irq != NO_IRQ) {
			r = gpio_interrupt_is_enabled((gpio_driver_handle)drv, i);
			if (unlikely(r < 0))
				c = 'E';
			else if (r)
				c = '1';
			else
				c = '0';
			r = gpio_interrupt_is_active((gpio_driver_handle)drv, i, -1);
			if (unlikely(r < 0))
				d = 'E';
			else if (r)
				d = '1';
			else
				d = '0';
			ret += snprintf(buf + ret, count - ret, " %c,%c", c, d);
		}
		if (drv->bank[i].label)
			ret += snprintf(buf + ret, count - ret, " %s\n",
				drv->bank[i].label);
		else
			ret += snprintf(buf + ret, count - ret, "\n");
	}
	
	drv->status_offset = 0;
	
	return ret;
}

static ssize_t gpio_attr_show_status(struct device *class_dev, struct device_attribute *attr, char *buf)
{
	return gpio_attr_show_status_count(class_dev, buf, PAGE_SIZE);
}

static DEVICE_ATTR(control, 0666, gpio_attr_show_control, gpio_attr_store_control);
static DEVICE_ATTR(status,  0444, gpio_attr_show_status, NULL);

/*-------------------------------------------------------------------------*/

/* Backward compatible /proc stuff */

#ifdef CONFIG_GPIO_CREATE_PROC_ENTRIES
static int gpio_proc_read(char *buf, char **start, off_t offset,
	int count, int *eof, void *data)
{
	struct gpio_pin_info *info = (struct gpio_pin_info *)data;
	if (offset > 0) {
		*eof = 1;
		return 0;
	}
	return gpio_attr_show_control_count(info->class_dev, buf, count);
}

static int gpio_proc_write(struct file *file, const char *buf,
	unsigned long count, void *data)
{
	struct gpio_pin_info *info = (struct gpio_pin_info *)data;
	return gpio_attr_store_control(info->class_dev, NULL, buf, count);
}

int gpio_proc_setup_pentry(gpio_driver_handle hdrv, unsigned gpio_index, 
	struct proc_dir_entry *pentry)
{
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
	if (unlikely(!hdrv || gpio_index >= drv->bank_size))
		return -EINVAL;
	pentry->read_proc = gpio_proc_read;
	if (pentry->mode & (S_IWUSR | S_IWGRP | S_IWOTH))
		pentry->write_proc = gpio_proc_write;
	else
		pentry->write_proc = NULL;
	pentry->data = (void *)&drv->bank[gpio_index];
	return 0;
}
EXPORT_SYMBOL(gpio_proc_setup_pentry);

static int gpio_proc_read_bank(char *buf, char **start, off_t offset,
	int count, int *eof, void *data)
{
	struct gpio_driver *drv = (struct gpio_driver *)data;
	if (offset > 0) {
		*eof = 1;
		return 0;
	}
	return gpio_attr_show_control_count(drv->class_dev_bank, buf, count);
}

static int gpio_proc_write_bank(struct file *file, const char *buf,
	unsigned long count, void *data)
{
	struct gpio_driver *drv = (struct gpio_driver *)data;
	return gpio_attr_store_control(drv->class_dev_bank, NULL, buf, count);
}

static int gpio_proc_read_status(char *buf, char **start, off_t offset,
	int count, int *eof, void *data)
{
	struct gpio_driver *drv = (struct gpio_driver *)data;
	if (offset > 0) {
		*eof = 1;
		return 0;
	}
	return gpio_attr_show_status_count(drv->class_dev_bank, buf, count);
}
#endif

/*-------------------------------------------------------------------------*/

/* /dev stuff */

struct gpio_filp_data
{
	struct gpio_driver 	*drv;
	struct gpio_pin_info 	*info;
	char 			interrupt;
	atomic_t		int_last_count;
};

static int gpio_fop_ioctl(struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg)
{
	int ret = -ENOIOCTLCMD, i;
	struct gpio_filp_data *fdata;
	if (!filp || !filp->private_data)
		return -EINVAL;
	fdata = (struct gpio_filp_data *)filp->private_data;	
	if (fdata->info && fdata->interrupt) {
		unsigned long val = 0;
		struct gpio_interrupt_config config;
		switch (cmd) {
		case GPIOINT_ENABLE:
			ret = gpio_interrupt_enable((gpio_driver_handle)fdata->drv, 
				fdata->info->cached_index);
			break;
		case GPIOINT_DISABLE:
			ret = gpio_interrupt_disable((gpio_driver_handle)fdata->drv, 
				fdata->info->cached_index);
			break;
		case GPIOINT_IS_ACTIVE:
			ret = gpio_interrupt_is_active((gpio_driver_handle)fdata->drv, 
				fdata->info->cached_index, -1);
			break;
		case GPIOINT_CLEAR:
			ret = gpio_interrupt_clear((gpio_driver_handle)fdata->drv, 
				fdata->info->cached_index);
			break;
		case GPIOINT_TYPE: {
			if (unlikely(ret = get_user(val, (u32 *)arg)))
				break;
			if (unlikely(ret = gpio_interrupt_get_config(fdata->drv, 
				fdata->info->cached_index, &config)))
				break;
			config.type = val;
			ret = gpio_interrupt_configure((gpio_driver_handle)fdata->drv,
				fdata->info->cached_index, &config);
		}	break;
		case GPIOINT_POLARITY: {
			if (unlikely(ret = get_user(val, (u32 *)arg)))
				break;
			if (unlikely(ret = gpio_interrupt_get_config(fdata->drv, 
				fdata->info->cached_index, &config)))
				break;
			config.polarity = val;
			ret = gpio_interrupt_configure((gpio_driver_handle)fdata->drv,
				fdata->info->cached_index, &config);
		}	break;
		case GPIOINT_DEBOUNCE_ON: {
			if (unlikely(ret = get_user(val, (u32 *)arg)))
				break;
			if (unlikely(ret = gpio_interrupt_get_config(fdata->drv, 
				fdata->info->cached_index, &config)))
				break;
			config.debounce_on = (val != 0);
			ret = gpio_interrupt_configure((gpio_driver_handle)fdata->drv,
				fdata->info->cached_index, &config);
		}	break;
		}
	} else if (fdata->info) {
		switch (cmd) {
		case GPIO_READ:
			ret = arch_gpio_get_value((gpio_driver_handle)fdata->drv, 
				fdata->info->cached_index);
			break;
		case GPIO_SET:
			ret = arch_gpio_direction_output((gpio_driver_handle)fdata->drv, 
				fdata->info->cached_index, 1);
			break;
		case GPIO_CLEAR:
			ret = arch_gpio_direction_output((gpio_driver_handle)fdata->drv, 
				fdata->info->cached_index, 0);
			break;
		case GPIO_INPUT:
			ret = arch_gpio_direction_input((gpio_driver_handle)fdata->drv, 
				fdata->info->cached_index);
			break;
		case GPIO_RWTIMING:
			fdata->drv->fs_rw_interval_us = arg;
			ret = 0;
			break;
		}
	} else {
		/* For backward compatibility, only banks 32 bits wide are supported */
		unsigned long val = 0;
		switch (cmd) {
		case GPIOBANK_SETGRP:
			if (unlikely(ret = get_user(val, (u32 *)arg)))
				break;
			fdata->drv->fs_bank_mask = val;
			break;
		case GPIOBANK_GETGRP:
			ret = put_user(fdata->drv->fs_bank_mask, (u32 *)arg);
			break;
		case GPIOBANK_INPUT:
			for (i=0; i<fdata->drv->bank_size; i++)
				if (unlikely(ret = arch_gpio_direction_input((gpio_driver_handle)fdata->drv, i)))
					break;
			break;
		case GPIOBANK_WRITE:			
			if (unlikely(ret = get_user(val, (u32 *)arg)))
				break;
			ret = gpio_write_bank((gpio_driver_handle)fdata->drv, (u32 *)&val,
				(u32 *)&fdata->drv->fs_bank_mask, 1);
			if (ret > 0)
				ret = 0;
			break;
		case GPIOBANK_READ: {
			u32 *data;
			int dw_count = gpio_driver_bank_size_dwords((gpio_driver_handle)fdata->drv);
			data = kzalloc(dw_count << 2, GFP_KERNEL);
			if (unlikely(!data)) {
				ret = -ENOMEM;
				break;
			}
			ret = gpio_read_bank((gpio_driver_handle)fdata->drv, data, dw_count);
			if (likely(!ret))
				put_user(data[0], (u32 *)arg);
			kfree(data);
		}	break;
		case GPIOBANK_WRISATOMIC:
			ret = fdata->drv->bank_is_atomic;
			break;
		case GPIOBANK_RWTIMING:
			fdata->drv->fs_bank_rw_interval_us = arg;
			ret = 0;
			break;
		}
	}
	if (unlikely(ret < 0))
		error("%s: failed with error %d", __func__, ret);
	return ret;
}

static int gpio_fs_interrupt_callback(unsigned gpio, unsigned gpio_index, void *cbdata);

static int gpio_fop_open(struct inode *inode, struct file *filp)
{
	dev_t devt;
	int i, ret = -ENODEV;
	struct gpio_driver *drv;
	struct gpio_filp_data *fdata = NULL;
	if (filp->private_data)
		return -EINVAL;
	devt = MKDEV(imajor(inode), iminor(inode));
	if (unlikely(down_interruptible(&global_lock)))
		return -EINTR;
	list_for_each_entry(drv, &gpio_drivers, list) {
    		if (devt == drv->devt) {
			/* A bank is being opened. */
			fdata = kzalloc(sizeof(struct gpio_filp_data), GFP_KERNEL);
			if (unlikely(!fdata)) {
				ret = -ENOMEM;
				goto end;
			}
			fdata->drv = drv;
			filp->private_data = fdata;
			ret = 0;
			goto end;
		} else {
			/* A pin/interrupt is being opened. */
			/*
			 * TODO: spin this code off in a separate function to avoid
			 * those many levels of indentation...
			 */
			for (i=0; i<drv->bank_size; i++) {
				unsigned long int_count = 0;
				if (drv->bank[i].class_dev_int && drv->bank[i].class_dev_int->devt == devt) {
					if (atomic_inc_return(&drv->bank[i].fop_open_count) == 1) {
						/* Disable just in case. */
						ret = gpio_interrupt_disable((gpio_driver_handle)drv,
							drv->bank[i].cached_index);
						if (unlikely(ret))
							goto end;
						ret = gpio_interrupt_register((gpio_driver_handle)drv, drv->bank[i].cached_index, 
							gpio_fs_interrupt_callback, &drv->bank[i]);
						if (unlikely(ret))
							goto end;
						/* Save the counter before we enable interrupts. */
						int_count = atomic_read(&drv->bank[i].int_counter);
						ret = gpio_interrupt_enable((gpio_driver_handle)drv, drv->bank[i].cached_index);
						if (unlikely(ret)) {
							gpio_interrupt_unregister((gpio_driver_handle)drv, drv->bank[i].cached_index, 
								gpio_fs_interrupt_callback);
							goto end;
						}
					} else {
						gpio_bank_safe_spin_lock(drv);
						int_count = atomic_read(&drv->bank[i].int_counter);
						gpio_interrupt_enable((gpio_driver_handle)drv, drv->bank[i].cached_index);
 						if (!atomic_read(&drv->bank[i].int_enabled)) {
							atomic_inc(&drv->bank[i].int_enabled);
 							enable_irq(drv->bank[i].irq);
 						}
						gpio_bank_safe_spin_unlock(drv);
					}
					ret = 1; /* Success but set to 1 to indicate interrupt was opened, see below. */
				} else if (drv->bank[i].class_dev->devt == devt) {
					ret = 0;
				} else {
					continue;
				}
				fdata = kzalloc(sizeof(struct gpio_filp_data), GFP_KERNEL);
				if (unlikely(!fdata)) {
					if (ret) {
						gpio_interrupt_disable((gpio_driver_handle)drv,
							drv->bank[i].cached_index);
						gpio_interrupt_unregister((gpio_driver_handle)drv, 
							drv->bank[i].cached_index, gpio_fs_interrupt_callback);
					}
					kfree(fdata);
					ret = -ENOMEM;
					goto end;
				}
				atomic_set(&fdata->int_last_count, int_count);
				fdata->drv = drv;
				fdata->info = &drv->bank[i];
				fdata->interrupt = ret;
				filp->private_data = fdata;
				ret = 0;
				goto end;
			}
		}
	}
end:
	up(&global_lock);
	return ret;
}

static int gpio_fs_interrupt_callback(unsigned gpio, unsigned gpio_index, void *cbdata)
{
	struct gpio_pin_info *info = cbdata;
	if (atomic_dec_return(&info->int_enabled) == 0)
		disable_irq_nosync(info->irq);
	wake_up_interruptible(&info->drv->fs_int_wait_queue);
	if (info->drv->fs_async_queue)
		kill_fasync(&info->drv->fs_async_queue, SIGIO, POLL_IN);
	return 0;
}

static struct gpio_interrupt_callback *gpio_scb_from_cb(struct gpio_pin_info *info,
	gpio_interrupt_callback_t cb)
{
	struct gpio_interrupt_callback *scb;
	gpio_bank_safe_spin_lock(info->drv);
	list_for_each_entry(scb, &info->callbacks, list) {
		if (scb->cb == cb) {
			gpio_bank_safe_spin_unlock(info->drv);
			return scb;
		}
	}
	gpio_bank_safe_spin_unlock(info->drv);
	return NULL;
}

static ssize_t gpio_fop_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos)
{
	int ret = 0, i;
	struct gpio_filp_data *fdata;
	if (!filp || !filp->private_data) {
		ret = -EINVAL;
		goto out;
	}
	fdata = (struct gpio_filp_data *)filp->private_data;
	if (fdata->info && fdata->interrupt) {
		/* Read pin interrupt = wait for interrupt if not enough are pending. */
		ret = atomic_read(&fdata->info->int_counter) - atomic_read(&fdata->int_last_count);
		if (ret < count) {
			if (filp->f_flags & O_NONBLOCK) {
				ret = -EAGAIN;
				goto out;
			}
			gpio_bank_safe_spin_lock(fdata->info->drv);
			if (!atomic_read(&fdata->info->int_enabled))
				enable_irq(fdata->info->irq);
			atomic_add(count, &fdata->info->int_enabled);
			gpio_bank_safe_spin_unlock(fdata->info->drv);
			if (unlikely(wait_event_interruptible(fdata->drv->fs_int_wait_queue,
				((ret = (atomic_read(&fdata->info->int_counter) - 
					atomic_read(&fdata->int_last_count))) >= count)))) {
				ret = -EINTR;
				goto out;
			}
		}
		atomic_add(count, &fdata->int_last_count);
		ret = 0;
	} else if (fdata->info) {
		/* Read pin */
		for (i = 0; i < count; i++) {
			if (i)
				udelay(fdata->drv->fs_rw_interval_us);
			ret = arch_gpio_get_value((gpio_driver_handle)fdata->drv, 
				fdata->info->cached_index);
			if (unlikely(ret < 0))
				goto out;
			put_user((char)ret, buf + i);
		}
	} else {
		/* Read bank */
		u32 *data;
		int r, len;
		int dw_count = gpio_driver_bank_size_dwords((gpio_driver_handle)fdata->drv);
		data = kzalloc(dw_count << 2, GFP_KERNEL);
		if (unlikely(!data))
			return -ENOMEM;
		i = 0;
		do {
			if (i)
				udelay(fdata->drv->fs_bank_rw_interval_us);
			r = gpio_read_bank((gpio_driver_handle)fdata->drv, data, dw_count);
			if (r <= 0)
				break;
			len = r * sizeof(u32);
			if (i + len > count)
				len = count - i;
			if (unlikely(copy_to_user(buf + i, data, len))) {
				ret = -EFAULT;
				break;
			}
			i += len;
		} while (i < count);
		kfree(data);
	}
out:
	if (unlikely(ret < 0))
		error("%s: failed with error %d", __func__, ret);
	return ret ? ret : count;
}

static ssize_t gpio_fop_write(struct file *filp, const char __user *buf, size_t count, loff_t *f_pos)
{
	int ret = 0, i;
	struct gpio_filp_data *fdata;
	if (!filp || !filp->private_data) {
		ret = -EINVAL;
		goto out;
	}
	fdata = (struct gpio_filp_data *)filp->private_data;
	if (fdata->info && fdata->interrupt) {
		/* Reenable interrupts. */
		gpio_bank_safe_spin_lock(fdata->info->drv);
		if (!atomic_read(&fdata->info->int_enabled))
			enable_irq(fdata->info->irq);
		atomic_add(count, &fdata->info->int_enabled);
		gpio_bank_safe_spin_unlock(fdata->info->drv);
	} else if (fdata->info) {
		/* Write pin */
		char c = 0;
		for (i = 0; i < count; i++) {
			get_user(c, buf + i);
			if (i) {
				udelay(fdata->drv->fs_rw_interval_us);
				ret = arch_gpio_set_value((gpio_driver_handle)fdata->drv,
					fdata->info->cached_index, c != 0);
			} else {
				ret = arch_gpio_direction_output((gpio_driver_handle)fdata->drv,
					fdata->info->cached_index, c != 0);
			}
			if (unlikely(ret))
				goto out;
		}
	} else {
		/* Write bank */
		unsigned max_dwcount = gpio_driver_bank_size_dwords((gpio_driver_handle)fdata->drv);
		int len = max_dwcount << 2;
		u32 *data, *mask;
		if (max_dwcount <= 0) {
			ret = -EINVAL;
			goto out;
		}
		data = kzalloc(len, GFP_KERNEL);
		if (unlikely(!data)) {
			ret = -ENOMEM;
			goto out;
		}
		mask = kmalloc(len, GFP_KERNEL);
		if (unlikely(!mask)) {
			kfree(data);
			ret = -ENOMEM;
			goto out;
		}
		memset(mask, ~0, len);
		mask[0] = fdata->drv->fs_bank_mask;
		i = 0;
		do {
			if (i + len > count)
				len = count - i;
			if (unlikely(copy_from_user(data, buf + i, len))) {
				ret = -EFAULT;
				break;
			}
			if (i)
				udelay(fdata->drv->fs_bank_rw_interval_us);
			ret = gpio_write_bank((gpio_driver_handle)fdata->drv, data, mask, len);
			if (unlikely(ret < 0))
				break;
			else
				ret = 0;
			i += len;
		} while (i < count);
		kfree(data);
		kfree(mask);
	}
out:
	if (unlikely(ret < 0))
		error("%s: failed with error %d", __func__, ret);
	return ret ? ret : count;
}

static int gpio_fop_fasync(int fd, struct file *filp, int on)
{
	struct gpio_filp_data *fdata;
	if (!filp || !filp->private_data)
		return -EINVAL;
	fdata = (struct gpio_filp_data *)filp->private_data;
	if (fdata->info && fdata->interrupt)
		return fasync_helper(fd, filp, on, &fdata->drv->fs_async_queue);
	return -ENOSYS;
}

static int gpio_fop_release(struct inode *inode, struct file *filp)
{
	struct gpio_filp_data *fdata;
	if (!filp->private_data)
		return -EINVAL;
	fdata = (struct gpio_filp_data *)filp->private_data;
	if (fdata->info && fdata->interrupt) {
		gpio_fop_fasync(-1, filp, 0);
		down(&global_lock);
		if (atomic_dec_return(&fdata->info->fop_open_count) == 0) {
			gpio_interrupt_disable((gpio_driver_handle)fdata->info->drv,
				fdata->info->cached_index);
			gpio_interrupt_unregister((gpio_driver_handle)fdata->info->drv,
				fdata->info->cached_index, gpio_fs_interrupt_callback);
		}
		up(&global_lock);
	}
	kfree(filp->private_data);
	filp->private_data = NULL;
	return 0;
}

static unsigned int gpio_fop_poll(struct file *filp, struct poll_table_struct *table)
{
	int ret = 0;
	struct gpio_filp_data *fdata;
	if (!filp || !filp->private_data)
		return -EINVAL;
	fdata = (struct gpio_filp_data *)filp->private_data;
	if (fdata->info && fdata->interrupt) {
		poll_wait(filp, &fdata->drv->fs_int_wait_queue, table);
		if (atomic_read(&fdata->info->int_counter) - atomic_read(&fdata->int_last_count) > 0)
			ret = POLLIN | POLLRDNORM;
		else
			ret = 0;
	} else {
		ret = -ENOSYS;
	}
	return ret;
}

static struct file_operations gpio_fops =
{
	.owner 		= THIS_MODULE,
	.write 		= gpio_fop_write,
	.read 		= gpio_fop_read,
	.ioctl 		= gpio_fop_ioctl,
	.open 		= gpio_fop_open,
	.release 	= gpio_fop_release,
	.poll 		= gpio_fop_poll,
	.fasync		= gpio_fop_fasync,
};

/*-------------------------------------------------------------------------*/

static int gpio_client_try_probe(struct gpio_client *client)
{
	int i, ret = 0, do_probe = 0;
	
	if (client->match_name) {
		client->gpio_hdrv = gpio_driver_get(client->match_name);
		if (client->gpio_hdrv)
			do_probe = 1;
	}
	
	if (!do_probe && client->pin_count) {
		for (i=0; i<client->pin_count; i++) {
			if (!client->match_pins[i].hdrv)
				client->match_pins[i].hdrv = gpio_driver_find(client->match_group, 
					client->match_pins[i].gpio, &client->match_pins[i].index);
			if (client->match_pins[i].hdrv)
				do_probe++;
		}
		if (do_probe == client->pin_count)
			do_probe = 1;
		else
			do_probe = 0;
	}
	
	if (do_probe) {
		int ret;
		for (i=0; i<client->pin_count; i++) {
			if (!*client->match_pins[i].label)
				continue;
			if (unlikely(arch_gpio_request(client->match_pins[i].hdrv, 
				client->match_pins[i].index, client->match_pins[i].label)))
				goto free_all;
		}
		if (unlikely(ret = client->probe(client)))
			goto free_all;
		list_del(&client->list); /* delete from pending */
		list_add_tail(&client->list, &gpio_loaded_clients);
		return 0;
	}
	goto put_all;
	
free_all:
	for (--i; i>=0; i--)
		if (*client->match_pins[i].label)
			arch_gpio_free(client->match_pins[i].hdrv, client->match_pins[i].index);
put_all:
	if (client->match_name && client->gpio_hdrv) {
		gpio_driver_put(client->gpio_hdrv);
		client->gpio_hdrv = NULL;
	}
	for (i=0; i<client->pin_count; i++) {
		if (client->match_pins[i].hdrv) {
			gpio_driver_put(client->match_pins[i].hdrv);
			client->match_pins[i].hdrv = NULL;
		}
	}
	return ret;
}

static void gpio_driver_probe_clients(void)
{
	struct gpio_client *entry, *tmp_entry;
	down(&client_lock);
	list_for_each_entry_safe(entry, tmp_entry, &gpio_pending_clients, list)
		gpio_client_try_probe(entry);
	up(&client_lock);
}

static int gpio_driver_remove_client(struct gpio_client *client)
{
	int ret, i;
	if (unlikely(ret = client->remove(client)))
		return ret;
	if (client->match_name && client->gpio_hdrv) {
		gpio_driver_put(client->gpio_hdrv);
		client->gpio_hdrv = NULL;
	}
	for (i=0; i<client->pin_count; i++) {
		if (client->match_pins[i].hdrv) {
			if (*client->match_pins[i].label)
				arch_gpio_free(client->match_pins[i].hdrv, client->match_pins[i].index);
			gpio_driver_put(client->match_pins[i].hdrv);
			client->match_pins[i].hdrv = NULL;
		}
	}
	list_del(&client->list);
	list_add_tail(&client->list, &gpio_pending_clients);
	return 0;
}

static void gpio_driver_remove_clients(struct gpio_driver *drv)
{
	struct gpio_client *entry, *tmp_entry;
	down(&client_lock);
	list_for_each_entry_safe(entry, tmp_entry, &gpio_loaded_clients, list) {
		int do_remove = 0;
		if (entry->gpio_hdrv && entry->gpio_hdrv == (gpio_driver_handle)drv) {
			do_remove = 1;
		} else if (entry->pin_count) {
			int i;
			for (i=0; i<entry->pin_count; i++) {
				if (entry->match_pins[i].hdrv == (gpio_driver_handle)drv) {
					do_remove = 1;
					break;
				}
			}
		}
		if (do_remove)
			gpio_driver_remove_client(entry);
	}
	up(&client_lock);
}

static int gpio_create_pin_dev(struct gpio_pin_info *info,
	int support_int, int *cur_minor)
{
	int ret;
	struct gpio_driver *drv = info->drv;
	
	info->class_dev = device_create(drv->class, drv->dev,
		MKDEV(MAJOR(drv->devt), (*cur_minor)++), NULL,
		GPIO_FILE_FORMAT, info->gpio);
	if (unlikely(IS_ERR(info->class_dev)))
		return PTR_ERR(info->class_dev);
	
	info->class_dev->driver_data = info;
	if (unlikely(ret = device_create_file(info->class_dev, &dev_attr_control)))
		goto err_pin_create_file_control;
	
	if (!support_int)
		return 0;
	
	info->class_dev_int = device_create(drv->class, drv->dev, 
			MKDEV(MAJOR(drv->devt), (*cur_minor)++), NULL,
			GPIO_INT_FILE_FORMAT, info->gpio);

	if (unlikely(IS_ERR(info->class_dev_int))) {
		ret = PTR_ERR(info->class_dev_int);
		goto err_pin_classdev_int;
	}
	
	info->class_dev_int->driver_data = info;
	if (unlikely(ret = device_create_file(info->class_dev_int, &dev_attr_control)))
		goto err_pin_create_file_control_int;
	
	return 0;
	
err_pin_create_file_control_int:
	device_unregister(info->class_dev_int);
err_pin_classdev_int:
	device_remove_file(info->class_dev, &dev_attr_control);
err_pin_create_file_control:
	device_unregister(info->class_dev);
	return ret;
}

static void gpio_destroy_pin_dev(struct gpio_pin_info *info, int support_int)
{
	if (support_int) {
		device_remove_file(info->class_dev_int, &dev_attr_control);
		device_unregister(info->class_dev_int);
	}
	device_remove_file(info->class_dev, &dev_attr_control);
	device_unregister(info->class_dev);
}

int gpio_driver_register(struct gpio_driver *drv)
{
	int ret = 0, i;
	int i_pindev = 0;
	struct gpio_group *group = NULL;
	char bank_file[GPIO_ANYFILE_MAXLEN];
	char status_file[GPIO_ANYFILE_MAXLEN];

#ifdef CONFIG_GPIO_CREATE_PROC_ENTRIES
	int i_pinproc = 0;
	char buffer[GPIO_ANYFILE_MAXLEN];
#endif

#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	struct gpio_driver *entry;
	if (unlikely(!drv || !*drv->name || !drv->bank
		|| !drv->chip.get|| !drv->chip.set)) {
		gpio_arg_error(drv);
		return -EINVAL;
	}
#endif
	
	down(&global_lock);

#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	list_for_each_entry(entry, &gpio_drivers, list) {
    		if (unlikely(!strcmp(entry->name, drv->name))) {
			ret = -EALREADY;
			goto end_unlock;
		}
	}
#endif
	
	if (*drv->group) {
		struct gpio_group *group_entry;
		list_for_each_entry(group_entry, &gpio_groups, list) {
			if (!strcmp(group_entry->name, drv->group)) {
				group = group_entry;
				break;
			}
		}
		if (!group) {
			group = kzalloc(sizeof(struct gpio_group), GFP_KERNEL);
			if (unlikely(!group)) {
				ret = -ENOMEM;
				goto end_unlock;
			}
			strcpy(group->name, drv->group);
			list_add_tail(&group->list, &gpio_groups);
		}
		group->refcount++;
	}
	
	drv->bank_id = group ? (group->refcount - 1) : 0;
	sprintf(bank_file, GPIO_BANK_FILE_FORMAT, drv->bank_id);
	sprintf(status_file, GPIO_STATUS_FILE_FORMAT, drv->bank_id);
	
#ifdef CONFIG_GPIO_CREATE_PROC_ENTRIES
	if (!root_pentry) {
		ret = -EPERM;
		goto err_proc;
	}

	if (!drv->pentry) {
		if (group) {
			if (group->refcount == 1)
				group->pentry = drv->pentry = 
					proc_mkdir(drv->group, root_pentry);
			else
				drv->pentry = group->pentry;
		} else {
			drv->pentry = proc_mkdir(drv->name, root_pentry);
		}
		if (unlikely(!drv->pentry)) {
			ret = -EPERM;
			goto err_proc;
		}
	}
	
	if (!drv->pentries) {
		drv->pentries = kzalloc(drv->bank_size * sizeof(struct proc_dir_entry *), GFP_KERNEL);
		if (unlikely(!drv->pentries)) {
			ret = -ENOMEM;
			goto err_proc_alloc;
		}
	}
	
	if (!drv->bank_pentry) {
		drv->bank_pentry = create_proc_entry(bank_file, 0666, drv->pentry);
		if (unlikely(!drv->bank_pentry)) {
			ret = -EPERM;
			goto err_proc_add_bank_pentry;
		} else {
			drv->bank_pentry->read_proc = gpio_proc_read_bank;
			drv->bank_pentry->write_proc = gpio_proc_write_bank;
			drv->bank_pentry->data = (void *)drv;
		}
	}
	
	if (!drv->status_pentry) {
		drv->status_pentry = create_proc_entry(status_file, 0444, drv->pentry);
		if (unlikely(!drv->status_pentry)) {
			ret = -EPERM;
			goto err_proc_add_status_pentry;
		} else {
			drv->status_pentry->read_proc = gpio_proc_read_status;
			drv->status_pentry->write_proc = NULL;
			drv->status_pentry->data = (void *)drv;
		}
	}
	
	for (i_pinproc = 0; i_pinproc < drv->bank_size; i_pinproc++) {
		sprintf(buffer, GPIO_PROC_FILE_FORMAT, drv->bank[i_pinproc].gpio);
		drv->pentries[i_pinproc] = create_proc_entry(buffer, 0666, drv->pentry);
		if (unlikely(!drv->pentries[i_pinproc])) {
			ret = -EPERM;
			goto err_proc_add_pentry;
		} else {
			drv->pentries[i_pinproc]->read_proc = gpio_proc_read;
			drv->pentries[i_pinproc]->write_proc = gpio_proc_write;
			drv->pentries[i_pinproc]->data = (void *)&drv->bank[i_pinproc];
		}
	}
#endif
	
	if (drv->dev) {
		int cur_minor = 0;
		drv->devt = 0;
		
		if (!drv->no_dev_entries) {
			int minors, int_count = 0;
			dev_t devt = 0;
			
			for (i=0; i<drv->bank_size; i++)
				if (drv->bank[i].irq != NO_IRQ)
					int_count++;
			
			minors = drv->bank_size + int_count + 1;			
			if (likely(!alloc_chrdev_region(&devt, 0, minors, drv->name))) {
				cdev_init(&drv->chrdev, &gpio_fops);
				drv->chrdev.owner = THIS_MODULE;
				if (likely(!cdev_add(&drv->chrdev, devt, minors))) {
					drv->devt = devt;
					cur_minor = MINOR(devt);
					drv->minors = minors;
				} else {
					unregister_chrdev_region(drv->devt, drv->minors);
					error("%s: failed to create character devices for %s, "
						"loading anyway", __func__, drv->name);
				}					
			} else {
				error("%s: could not allocate device ID region for %s, "
					"loading anyway", __func__, drv->name);
			}
		}
		
		if (group) {
			if (group->refcount == 1)
				group->class = drv->class = class_create(THIS_MODULE, drv->group);
			else
				drv->class = group->class;
		} else {
			drv->class = class_create(THIS_MODULE, drv->name);
		}		
		if (unlikely(IS_ERR(drv->class))) {
			ret = PTR_ERR(drv->class);
			goto err_create_class;
		}
		
		drv->class_dev_bank = device_create(drv->class, drv->dev,
			MKDEV(MAJOR(drv->devt), cur_minor++), NULL, bank_file);
		if (unlikely(IS_ERR(drv->class_dev_bank))) {
			ret = PTR_ERR(drv->class_dev_bank);
			goto err_bank_create_classdev;
		}
		
		drv->class_dev_bank->driver_data = drv;
		if (unlikely(ret = device_create_file(drv->class_dev_bank, &dev_attr_control)))
			goto err_bank_create_file_control;
		if (unlikely(ret = device_create_file(drv->class_dev_bank, &dev_attr_status)))
			goto err_bank_create_file_status;
		
		for (i_pindev = 0; i_pindev < drv->bank_size; i_pindev++) {
			int support_int = (drv->bank[i_pindev].irq != NO_IRQ);
			if (unlikely(ret = gpio_create_pin_dev(&drv->bank[i_pindev],
				support_int, &cur_minor)))
				goto err_pin_dev;
		}
	}
	
	for (i=0; i<drv->bank_size; i++) {
		INIT_LIST_HEAD(&drv->bank[i].callbacks);
		mutex_init(&drv->bank[i].used_lock);
		drv->bank[i].label = NULL;
	}
	mutex_init(&drv->bank_lock);
	spin_lock_init(&drv->irq_lock);
	atomic_set(&drv->pending_spin_unlock, 0);
	atomic_set(&drv->refcount, 0);
	init_waitqueue_head(&drv->fs_int_wait_queue);
	list_add_tail(&drv->list, &gpio_drivers);
	
	goto end_unlock;
	
err_pin_dev:
	for (i_pindev--; i_pindev >= 0; i_pindev--) {
		int support_int = (drv->bank[i_pindev].irq != NO_IRQ);
		gpio_destroy_pin_dev(&drv->bank[i_pindev], support_int);
	}
	device_remove_file(drv->class_dev_bank, &dev_attr_status);
err_bank_create_file_status:
	device_remove_file(drv->class_dev_bank, &dev_attr_control);
err_bank_create_file_control:
	device_unregister(drv->class_dev_bank);
err_bank_create_classdev:
	if (!group || group->refcount == 1)
		class_destroy(drv->class);
err_create_class:
    	if (drv->devt) {
		cdev_del(&drv->chrdev);
		unregister_chrdev_region(drv->devt, drv->minors);
		drv->devt = 0;
	}
#ifdef CONFIG_GPIO_CREATE_PROC_ENTRIES
err_proc_add_pentry:
	for (i_pinproc--; i_pinproc >= 0; i_pinproc--) {
		sprintf(buffer, GPIO_PROC_FILE_FORMAT, drv->bank[i_pinproc].gpio);
		remove_proc_entry(buffer, drv->pentry);
	}
	remove_proc_entry(status_file, drv->pentry);
	drv->status_pentry = NULL;
err_proc_add_status_pentry:
	remove_proc_entry(bank_file, drv->pentry);
	drv->bank_pentry = NULL;
err_proc_add_bank_pentry:
	kfree(drv->pentries);
	drv->pentries = NULL;
err_proc_alloc:
	if (!group)
    		remove_proc_entry(drv->name, root_pentry);
	else if (group->refcount == 1)
		remove_proc_entry(drv->group, root_pentry);
	drv->pentry = NULL;
err_proc:
#endif
	if (group && --group->refcount == 0) {
    		list_del(&group->list);
		kfree(group);
	}
end_unlock:
	up(&global_lock);
	if (!ret)
		gpio_driver_probe_clients();
	return ret;
}
EXPORT_SYMBOL(gpio_driver_register);

int gpio_driver_unregister(struct gpio_driver *drv)
{
	int ret = 0, i;
	struct gpio_driver *entry;
	struct gpio_group *group = NULL;
	
#ifdef CONFIG_GPIO_CREATE_PROC_ENTRIES
	int i_pinproc = 0;
	char buffer[GPIO_ANYFILE_MAXLEN];
#endif
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!drv || !drv->name || !*drv->name))
		return -EINVAL;
#endif
	
	down(&global_lock);
	list_for_each_entry(entry, &gpio_drivers, list) {
    		if (!strcmp(entry->name, drv->name)) {
			if (entry != drv) {
				gpio_arg_error(drv); /* Names match, but not addresses! */
				ret = -EINVAL;
			} else if (atomic_read(&entry->refcount) > 0) {
				ret = -EBUSY;
			}
			break;
		}
	}
	up(&global_lock);
	
	if (unlikely(ret))
		return ret;
	
	gpio_driver_remove_clients(drv);
	
	down(&global_lock);
	
	if (*drv->group) {
		struct gpio_group *group_entry;
		list_for_each_entry(group_entry, &gpio_groups, list) {
			if (!strcmp(group_entry->name, drv->group)) {
				group = group_entry;
				break;
			}
		}
	}
	
    	for (i=0; i<drv->bank_size; i++) {
		struct gpio_interrupt_callback *scb, *temp_scb;
		gpio_bank_safe_spin_lock(drv);
		list_for_each_entry_safe(scb, temp_scb, &drv->bank[i].callbacks, list) {
			gpio_bank_safe_spin_unlock(drv);
			gpio_interrupt_unregister((gpio_driver_handle)drv, i, scb->cb);
			gpio_bank_safe_spin_lock(drv);
		}
		gpio_bank_safe_spin_unlock(drv);
	}
	
	if (drv->dev) {
		int i_pindev;
		for (i_pindev = drv->bank_size - 1; i_pindev >= 0; i_pindev--) {
			int support_int = (drv->bank[i_pindev].irq != NO_IRQ);
			gpio_destroy_pin_dev(&drv->bank[i_pindev], support_int);
		}
		device_remove_file(drv->class_dev_bank, &dev_attr_status);
		device_remove_file(drv->class_dev_bank, &dev_attr_control);
		device_unregister(drv->class_dev_bank);
		if (!group || group->refcount == 1)
			class_destroy(drv->class);
		if (drv->devt) {
			cdev_del(&drv->chrdev);
			unregister_chrdev_region(drv->devt, drv->minors);
			drv->devt = 0;
		}
	}
	
#ifdef CONFIG_GPIO_CREATE_PROC_ENTRIES
    	for (i_pinproc = drv->bank_size - 1; i_pinproc >= 0; i_pinproc--) {
		sprintf(buffer, GPIO_PROC_FILE_FORMAT, drv->bank[i_pinproc].gpio);
		remove_proc_entry(buffer, drv->pentry);
	}
	kfree(drv->pentries);
	drv->pentries = NULL;
	
	sprintf(buffer, GPIO_BANK_FILE_FORMAT, group ? (group->refcount - 1) : 0);
	remove_proc_entry(buffer, drv->pentry);
	drv->bank_pentry = NULL;
	sprintf(buffer, GPIO_STATUS_FILE_FORMAT, group ? (group->refcount - 1) : 0);
	remove_proc_entry(buffer, drv->pentry);
	drv->status_pentry = NULL;
	
	if (!group)
    		remove_proc_entry(drv->name, root_pentry);
	else if (group->refcount == 1)
		remove_proc_entry(drv->group, root_pentry);
	drv->pentry = NULL;
#endif
	
	if (group && --group->refcount == 0) {
    		list_del(&group->list);
		kfree(group);
	}
	
	/* Finally remove driver from available drivers list. */
	list_del(&drv->list);
	up(&global_lock);
	return 0;
}
EXPORT_SYMBOL(gpio_driver_unregister);

/*-------------------------------------------------------------------------*/

/*****************************************************
 * CLIENT INTERFACE
 *****************************************************/

unsigned gpio_driver_bank_size(gpio_driver_handle hdrv)
{
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv)) {
		gpio_arg_error(hdrv);
		return 0;
	}
#endif
	return ((struct gpio_driver *)hdrv)->bank_size;
}
EXPORT_SYMBOL(gpio_driver_bank_size);

unsigned gpio_driver_bank_size_bytes(gpio_driver_handle hdrv)
{
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv)) {
		gpio_arg_error(hdrv);
		return 0;
	}
#endif
	return 1 + ((((struct gpio_driver *)hdrv)->bank_size - 1) >> 3);
}
EXPORT_SYMBOL(gpio_driver_bank_size_bytes);

unsigned gpio_driver_bank_size_dwords(gpio_driver_handle hdrv)
{
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv)) {
		gpio_arg_error(hdrv);
		return 0;
	}
#endif
	return 1 + ((((struct gpio_driver *)hdrv)->bank_size - 1) >> 5);
}
EXPORT_SYMBOL(gpio_driver_bank_size_dwords);

int gpio_to_index(gpio_driver_handle hdrv, unsigned gpio)
{
	int i;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	for (i=0; i<((struct gpio_driver *)hdrv)->bank_size; i++)
		 if (((struct gpio_driver *)hdrv)->bank[i].gpio == gpio)
		 	return i;
	return -ENODEV;
}
EXPORT_SYMBOL(gpio_to_index);

int gpio_from_index(gpio_driver_handle hdrv, unsigned gpio_index)
{
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >=
		((struct gpio_driver *)hdrv)->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	return ((struct gpio_driver *)hdrv)->bank[gpio_index].gpio;
}
EXPORT_SYMBOL(gpio_from_index);

int arch_gpio_to_irq(gpio_driver_handle hdrv, unsigned gpio_index)
{
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	return drv->bank[gpio_index].irq;
}

int gpio_from_irq(gpio_driver_handle hdrv, unsigned irq)
{
	int i, found = -1;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || irq == NO_IRQ)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	for (i=0; i<drv->bank_size; i++) {
		if (drv->bank[i].irq == irq) {
			if (found >= 0)
				/* 
				 * This error code is not really 
				 * intended for that but its name
				 * is quite relevant here.
				 */
				return -ETOOMANYREFS;
			found = i;
		}
	}
	return found >= 0 ? found : -ENODEV;
}
EXPORT_SYMBOL(gpio_from_irq);

gpio_driver_handle gpio_driver_find(const char *group,
	unsigned gpio, unsigned *gpio_index)
{
	struct gpio_driver *entry;
	gpio_driver_handle ret = NULL;
	int i, found = 0;
	might_sleep();
	if (unlikely(down_interruptible(&global_lock)))
		return NULL;
	list_for_each_entry(entry, &gpio_drivers, list) {
    		if (group && *group && strcmp(group, entry->group))
			continue;
		for (i=0; i<entry->bank_size; i++) {
			if (entry->bank[i].gpio == gpio) {
				up(&global_lock);
				ret = gpio_driver_get(entry->name);
				if (likely(ret)) {
					if (gpio_index)
						*gpio_index = i;
					found = 1;
				}
				break;
			}
		}
	}
	if (!found)
		up(&global_lock);
	if (!ret && gpio_index)
		*gpio_index = -ENODEV;
	return ret;
}
EXPORT_SYMBOL(gpio_driver_find);

gpio_driver_handle gpio_driver_get(const char *name)
{
	gpio_driver_handle ret = NULL;
	struct gpio_driver *entry, *found = NULL;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!name || !*name)) {
		gpio_arg_error(NULL);
		return NULL;
	}
#endif
	might_sleep();
	if (unlikely(down_interruptible(&global_lock)))
		return NULL;
	list_for_each_entry(entry, &gpio_drivers, list) {
    		if (!strcmp(entry->name, name)) {
			found = entry;
			break;
		}
	}
	if (found) {
		atomic_inc(&found->refcount);
		ret = (gpio_driver_handle)found;
	}
	up(&global_lock);
	return ret;
}
EXPORT_SYMBOL(gpio_driver_get);

void gpio_driver_put(gpio_driver_handle hdrv)
{
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
	struct gpio_driver *entry;
	int i;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv)) {
		gpio_arg_error(hdrv);
		return;
	}
#endif
	might_sleep();
	down(&global_lock);
	list_for_each_entry(entry, &gpio_drivers, list) {
    		if (strcmp(entry->name, drv->name))
			continue;
		if (atomic_read(&entry->refcount) == 1) {
			for (i=0; i<drv->bank_size; i++) {
				if (mutex_is_locked(&drv->bank[i].used_lock)) {
					error("%s: BUG, please fix! GPIO %d (%d in %s) "
						"has been requested (label: %s) but not freed!",
						__func__, drv->bank[i].gpio, i, drv->name,
						drv->bank[i].label);
					break;
				}
			}
		}
		atomic_dec(&entry->refcount);
		break;
	}
	up(&global_lock);
}
EXPORT_SYMBOL(gpio_driver_put);

struct device *gpio_get_device(gpio_driver_handle hdrv)
{
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv)) {
		gpio_arg_error(hdrv);
		return NULL;
	}
#endif
	return drv->dev;
}
EXPORT_SYMBOL(gpio_get_device);

int gpio_client_register(struct gpio_client *client)
{
	int ret;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	struct gpio_client *entry;
	if (unlikely(!client || !client->probe || !client->remove
		|| ((!client->match_name || !*client->match_name) && !client->match_pins)
		|| (client->match_pins && !client->pin_count))) {
		gpio_arg_error(NULL);
		return -EINVAL;
	}
	down(&client_lock);
	list_for_each_entry(entry, &gpio_loaded_clients, list)
		if (client == entry)
			return -EALREADY;
	list_for_each_entry(entry, &gpio_pending_clients, list)
		if (client == entry)
			return -EALREADY;
	up(&client_lock);
#endif
	might_sleep();
	down(&client_lock);
	list_add_tail(&client->list, &gpio_pending_clients);
	ret = gpio_client_try_probe(client);
	up(&client_lock);
	return ret;	
}
EXPORT_SYMBOL(gpio_client_register);

int gpio_client_unregister(struct gpio_client *client)
{
	int ret = 0;
	struct gpio_client *entry;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!client)) {
		gpio_arg_error(NULL);
		return -EINVAL;
	}
#endif
	down(&client_lock);
	list_for_each_entry(entry, &gpio_loaded_clients, list) {
		if (client == entry) {
			ret = 1;
			goto is_found;
		}
	}
	list_for_each_entry(entry, &gpio_pending_clients, list)
		if (client == entry)
			goto is_found;
	up(&client_lock);
	return -EINVAL;
is_found:
	if (ret)
		ret = gpio_driver_remove_client(client);
	if (!ret)
		list_del(&client->list); /* Remove from pending */
	up(&client_lock);
	return 0;
}
EXPORT_SYMBOL(gpio_client_unregister);


/* Access functions */

int gpio_request_(gpio_driver_handle hdrv, unsigned gpio_index, const char *label, int wait)
{
	int ret = 0;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || !label || !*label || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (wait && !in_atomic() && !irqs_disabled()) {
		if (unlikely(mutex_lock_interruptible(&drv->bank[gpio_index].used_lock)))
			return -EINTR;
	} else {
		if (!mutex_trylock(&drv->bank[gpio_index].used_lock))
			return -EBUSY;
	}
	
		if (unlikely(ret = gpio_bank_safe_lock(drv))) {
			mutex_unlock(&drv->bank[gpio_index].used_lock);
			return ret;
		}
		ret = drv->chip.request(&drv->chip, gpio_index);
		gpio_bank_safe_unlock(drv);
	
	if (!ret) {
		drv->bank[gpio_index].label = label;
	} else {
		drv->bank[gpio_index].label = NULL;
		mutex_unlock(&drv->bank[gpio_index].used_lock);
	}
	return ret;
}

int arch_gpio_request(gpio_driver_handle hdrv, unsigned gpio_index, const char *label)
{
	return gpio_request_(hdrv, gpio_index, label, 0);
}


int gpio_request_wait(gpio_driver_handle hdrv, unsigned gpio_index, const char *label)
{
	return gpio_request_(hdrv, gpio_index, label, 1);
}
EXPORT_SYMBOL(gpio_request_wait);

void arch_gpio_free(gpio_driver_handle hdrv, unsigned gpio_index)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return;
	}
#endif
	if (!mutex_is_locked(&drv->bank[gpio_index].used_lock))
		return;
	gpio_bank_safe_spin_lock(drv);
	ret = list_size(&drv->bank[gpio_index].callbacks);
	gpio_bank_safe_spin_unlock(drv);
	if (ret) {
		error("%s: cannot free GPIO %d (%d in %s, label: %s) until "
			"all interrupt callbacks have been unregistered!",
			__func__, drv->bank[gpio_index].gpio, gpio_index, 
			drv->name, drv->bank[gpio_index].label);
		return;
	}
	
		if (unlikely(gpio_bank_safe_lock(drv)))
			return;

		drv->chip.free(&drv->chip, gpio_index);
		gpio_bank_safe_unlock(drv);
	
	drv->bank[gpio_index].label = NULL;
	mutex_unlock(&drv->bank[gpio_index].used_lock);
}


int gpio_is_enabled(gpio_driver_handle hdrv, unsigned gpio_index)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;
	ret = drv->is_enabled(drv, gpio_index);
	gpio_bank_safe_unlock(drv);
	return ret;
}
EXPORT_SYMBOL(gpio_is_enabled);

int arch_gpio_direction_input(gpio_driver_handle hdrv, unsigned gpio_index)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (!drv->chip.direction_input)
		return -ENOSYS;
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;

	ret = drv->chip.direction_input(&drv->chip, gpio_index);
	gpio_bank_safe_unlock(drv);

	return ret;
}


int arch_gpio_direction_output(gpio_driver_handle hdrv, unsigned gpio_index, int value)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (!drv->chip.direction_output)
		return -ENOSYS;
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;

	ret = drv->chip.direction_output(&drv->chip, gpio_index, value != 0);
	gpio_bank_safe_unlock(drv);
	return ret;
}


int arch_gpio_get_value(gpio_driver_handle hdrv, unsigned gpio_index)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;

	ret = drv->chip.get(&drv->chip, gpio_index);
	gpio_bank_safe_unlock(drv);
	return ret;
}


int arch_gpio_set_value(gpio_driver_handle hdrv, unsigned gpio_index, int value)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;

	drv->chip.set(&drv->chip, gpio_index, value != 0);
	gpio_bank_safe_unlock(drv);
	return ret;
}


int gpio_get_direction(gpio_driver_handle hdrv, unsigned gpio_index)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (!drv->get_direction)
		return -ENOSYS;
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;
	ret = drv->get_direction(drv, gpio_index);
	gpio_bank_safe_unlock(drv);
	return ret;
}
EXPORT_SYMBOL(gpio_get_direction);

int gpio_read_bank(gpio_driver_handle hdrv, u32 *data, int dword_count)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || !data || dword_count <= 0)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (!drv->read_bank)
		return -ENOSYS;
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;
	ret = drv->read_bank(drv, data, dword_count);
	gpio_bank_safe_unlock(drv);
	return ret;
}
EXPORT_SYMBOL(gpio_read_bank);

int gpio_write_bank(gpio_driver_handle hdrv, const u32 *data,
	const u32 *mask, int dword_count)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || !data || dword_count <= 0 || !mask)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (!drv->write_bank)
		return -ENOSYS;
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;
	ret = drv->write_bank(drv, data, mask, dword_count);
	gpio_bank_safe_unlock(drv);
	return ret;
}
EXPORT_SYMBOL(gpio_write_bank);

int gpio_interrupt_registered(gpio_driver_handle hdrv, 
	unsigned gpio_index, gpio_interrupt_callback_t cb)
{
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	return gpio_scb_from_cb(&drv->bank[gpio_index], cb) != NULL;
}
EXPORT_SYMBOL(gpio_interrupt_registered);

static irqreturn_t gpio_default_interrupt(int irq, void *dev_id)
{
	int ret, i;
	struct gpio_driver *drv = dev_id;
	struct gpio_interrupt_callback *scb;
	for (i=0; i<drv->bank_size; i++) {
		if (drv->interrupt_is_active(drv, i, irq)) {
			atomic_inc(&drv->bank[i].int_counter);
			list_for_each_entry(scb, &drv->bank[i].callbacks, list) {
				ret = scb->cb(drv->bank[i].gpio, i, scb->cbdata);
// 				drv->interrupt_clear(drv, i);
				if (unlikely(ret))
					drv->interrupt_disable(drv, i);
			}
		}
	}
	return IRQ_HANDLED;
}

int gpio_interrupt_register(gpio_driver_handle hdrv, unsigned gpio_index, 
	gpio_interrupt_callback_t cb, void *cbdata)
{
	int ret = 0, found = 0;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
	struct gpio_interrupt_callback *scb;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || !cb)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (!drv->interrupt_enable || !drv->interrupt_disable 
		|| !drv->interrupt_clear || !drv->interrupt_is_active)
		return -ENOSYS;
	if (drv->bank[gpio_index].irq == NO_IRQ)
		return -EPERM;
	gpio_bank_safe_spin_lock(drv);
	list_for_each_entry(scb, &drv->bank[gpio_index].callbacks, list) {
		if (scb->cb == cb)
			found = 1;
		ret++;
	}
	gpio_bank_safe_spin_unlock(drv);
	if (found)
		return -EALREADY;
	if (!ret) {
		atomic_set(&drv->bank[gpio_index].int_enabled, 1);
		if (unlikely(ret = request_irq(drv->bank[gpio_index].irq, 
			gpio_default_interrupt, IRQF_SHARED /*| IRQF_DISABLED*/, DRIVER_NAME, (void*)drv))) {
			error("%s: could not get shared interrupt %d, error is %d",
				__func__, drv->bank[gpio_index].irq, ret);
			return ret;
		}
	}
	scb = kzalloc(sizeof(struct gpio_interrupt_callback), GFP_KERNEL);
	if (unlikely(!scb)) {
		free_irq(drv->bank[gpio_index].irq, (void *)drv);
		return -ENOMEM;
	}
	scb->cb = cb;
	scb->cbdata = cbdata;
	gpio_bank_safe_spin_lock(drv);
	list_add_tail(&scb->list, &drv->bank[gpio_index].callbacks);
	gpio_bank_safe_spin_unlock(drv);
	return 0;
}
EXPORT_SYMBOL(gpio_interrupt_register);

int gpio_interrupt_unregister(gpio_driver_handle hdrv, unsigned gpio_index,
	gpio_interrupt_callback_t cb)
{
	int ret = 0, found = 0;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
	struct gpio_interrupt_callback *scb, *temp_scb;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	gpio_bank_safe_spin_lock(drv);
	list_for_each_entry_safe(scb, temp_scb, 
		&drv->bank[gpio_index].callbacks, list) {
		if (scb->cb == cb) {
			found = 1;
			list_del(&scb->list);
			kfree(scb);
		} else {
			ret++;
		}
	}
	if (!found)
		ret = -EPERM;
	if (!ret) {
		free_irq(drv->bank[gpio_index].irq, (void *)drv);
		atomic_set(&drv->bank[gpio_index].int_enabled, 0);
	}
	gpio_bank_safe_spin_unlock(drv);
	return ret;
}
EXPORT_SYMBOL(gpio_interrupt_unregister);

#define IS_VALID_CONFIG(config) \
	(config && (config->type & (GPIO_INT_TYPE_LEVEL | GPIO_INT_TYPE_EDGE)) \
	&& (config->type & (GPIO_INT_POLARITY_HIGH_OR_RISING | GPIO_INT_POLARITY_LOW_OR_FALLING)))

int gpio_interrupt_configure(gpio_driver_handle hdrv, unsigned gpio_index, 
	const struct gpio_interrupt_config *config)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size
		|| !IS_VALID_CONFIG(config))) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (!drv->interrupt_configure)
		return -ENOSYS;
	if (drv->bank[gpio_index].irq == NO_IRQ)
		return -EPERM;
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;
	ret = drv->interrupt_configure(drv, gpio_index, config);
	gpio_bank_safe_unlock(drv);
	return ret;
}
EXPORT_SYMBOL(gpio_interrupt_configure);

int gpio_interrupt_get_config(gpio_driver_handle hdrv, unsigned gpio_index, 
	struct gpio_interrupt_config *config)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!config)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	memset(config, 0, sizeof(struct gpio_interrupt_config));
	if (unlikely(!hdrv || gpio_index >= drv->bank_size))
		return -EINVAL;
	if (!drv->interrupt_get_config)
		return -ENOSYS;
	if (drv->bank[gpio_index].irq == NO_IRQ)
		return -EPERM;
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;
	ret = drv->interrupt_get_config(drv, gpio_index, config);
	gpio_bank_safe_unlock(drv);
	return ret;
}
EXPORT_SYMBOL(gpio_interrupt_get_config);

int gpio_interrupt_enable(gpio_driver_handle hdrv, unsigned gpio_index)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (!drv->interrupt_enable)
		return -ENOSYS;
	if (drv->bank[gpio_index].irq == NO_IRQ)
		return -EPERM;
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;
	ret = drv->interrupt_enable(drv, gpio_index);
	gpio_bank_safe_unlock(drv);
	return ret;
}
EXPORT_SYMBOL(gpio_interrupt_enable);

int gpio_interrupt_is_active(gpio_driver_handle hdrv, unsigned gpio_index, int irq)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (!drv->interrupt_is_active)
		return -ENOSYS;
	if (drv->bank[gpio_index].irq == NO_IRQ)
		return -EPERM;
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;
	ret = drv->interrupt_is_active(drv, gpio_index, irq);
	gpio_bank_safe_unlock(drv);
	return ret;
}
EXPORT_SYMBOL(gpio_interrupt_is_active);

int gpio_interrupt_is_enabled(gpio_driver_handle hdrv, unsigned gpio_index)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (!drv->interrupt_is_enabled)
		return -ENOSYS;
	if (drv->bank[gpio_index].irq == NO_IRQ)
		return -EPERM;
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;
	ret = drv->interrupt_is_enabled(drv, gpio_index);
	gpio_bank_safe_unlock(drv);
	return ret;
}
EXPORT_SYMBOL(gpio_interrupt_is_enabled);

int gpio_interrupt_disable(gpio_driver_handle hdrv, unsigned gpio_index)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (!drv->interrupt_disable)
		return -ENOSYS;
	if (drv->bank[gpio_index].irq == NO_IRQ)
		return -EPERM;
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;
	ret = drv->interrupt_disable(drv, gpio_index);
	gpio_bank_safe_unlock(drv);
	return ret;
}
EXPORT_SYMBOL(gpio_interrupt_disable);

int gpio_interrupt_clear(gpio_driver_handle hdrv, unsigned gpio_index)
{
	int ret;
	struct gpio_driver *drv = (struct gpio_driver *)hdrv;
#ifdef CONFIG_GPIO_DEBUG_CHECK_ARGS
	if (unlikely(!hdrv || gpio_index >= drv->bank_size)) {
		gpio_arg_error(hdrv);
		return -EINVAL;
	}
#endif
	if (!drv->interrupt_clear)
		return -ENOSYS;
	if (drv->bank[gpio_index].irq == NO_IRQ)
		return -EPERM;
	if (unlikely(ret = gpio_bank_safe_lock(drv)))
		return ret;
	ret = drv->interrupt_clear(drv, gpio_index);
	gpio_bank_safe_unlock(drv);
	return ret;
}
EXPORT_SYMBOL(gpio_interrupt_clear);

/*-------------------------------------------------------------------------*/

static int __init gpio_init(void)
{
#ifdef CONFIG_GPIO_CREATE_PROC_ENTRIES
	root_pentry = proc_mkdir(GPIO_ROOT_PROC_DIR, NULL);
	if (unlikely(!root_pentry)) {
		error("%s: could not create proc root dir '%s'",
			__func__, GPIO_ROOT_PROC_DIR);
		return -EINVAL;
	}
#endif
	return 0;
}

static void __exit gpio_exit(void)
{
	int ret;
	struct gpio_driver *entry, *tmp_entry;
	down(&global_lock);
	list_for_each_entry_safe(entry, tmp_entry, &gpio_drivers, list) {
		up(&global_lock);
		if (unlikely(ret = gpio_driver_unregister(entry))) {
			error("%s: BUG, please FIX: gpio_driver_unregister returned %d, at this stage a "
				"failure unregistering a GPIO controller is not acceptable!", __func__, ret);
		}
		down(&global_lock);
		list_del(&entry->list);
	}
	up(&global_lock);
#ifdef CONFIG_GPIO_CREATE_PROC_ENTRIES
	remove_proc_entry(GPIO_ROOT_PROC_DIR, NULL);
	root_pentry = NULL;
#endif
}

module_init(gpio_init);
module_exit(gpio_exit);

/*-------------------------------------------------------------------------*/

MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_LICENSE("GPL");
