#define DRIVER_NAME 	"bench"
#define DRIVER_DESC 	"Benchmarking driver"
#define DRIVER_AUTHOR 	"Gregoire Pean <gpean@mobilygen.com>"
#define DRIVER_VERSION 	"1:2.1"

/*
 *  This file Copyright (C) 2007 Mobilygen Corp.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE	LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/vermagic.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/io.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/proc_fs.h>
#include <linux/hrtimer.h>

#include <asm/checksum.h>
#include <asm/uaccess.h>
#include <asm/cacheflush.h>

#include <mach/bench.h>

#define DEBUG

/*-------------------------------------------------------------------------*/

#define error(format, arg...)	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#define info(format, arg...) 	printk(KERN_INFO DRIVER_NAME ": " format "\n" , ## arg)
#define warn(format, arg...) 	printk(KERN_WARNING DRIVER_NAME ": " format "\n" , ## arg)

#ifdef DEBUG
#define debug(format, arg...) 	printk(KERN_ERR DRIVER_NAME ":%d: " format "\n" , __LINE__, ## arg)
#else
#define debug(format, arg...) 	do {} while (0)
#endif

/*-------------------------------------------------------------------------*/

static int flush_cache = 1;
static int irq_enable = 1;

#define DEFINE_BENCH_PROC(name, code, from_flags, to_flags) \
static int bench_##name(struct file *file, \
	const char *buffer, unsigned long count, void *data) \
{ \
	ktime_t start, end; \
	unsigned long flags; \
	int i, callcount, size; \
	void *from, *to; \
	int order; \
 \
	if (sscanf(buffer, "%d %d", &size, &callcount) != 2  \
		|| callcount < 1 || size < 1) { \
		printk(KERN_ERR "%s: invalid input\n", __func__); \
		return count; \
	} \
 \
	order = get_order(size); \
	 \
	printk(KERN_ERR "%s: testing with size=%d count=%d (page order=%d)\n",  \
		__func__, size, callcount, order); \
	 \
	from = (void *)__get_free_pages(from_flags, order); \
	if (!from) { \
		printk(KERN_ERR "%s: not enough memory\n", __func__); \
		return count; \
	} \
	to = (void *)__get_free_pages(to_flags, order); \
	if (!to) { \
		free_pages((unsigned long)from, order); \
		printk(KERN_ERR "%s: not enough memory\n", __func__); \
		return count; \
	} \
 \
	local_save_flags(flags); \
	if (!irq_enable) \
		local_irq_disable(); \
	start = ktime_get(); \
	for (i = 0; i < callcount; i++) { \
		if (flush_cache) \
			flush_cache_all(); \
		code; \
	} \
	end = ktime_get(); \
	local_irq_restore(flags); \
 \
	free_pages((unsigned long)from, order); \
	free_pages((unsigned long)to, order); \
 \
	if (ktime_equal(start, end)) { \
		printk(KERN_ERR "%s: too fast; use a bigger count " \
			"of calls / a bigger size\n", __func__); \
	} else { \
		unsigned long usecs = ktime_us_delta(end, start); \
		unsigned long bytes = callcount * size; \
		unsigned long rate_kbs = (bytes < (1<<22)) ? bytes*1000/usecs : bytes/(usecs/1000); \
		printk(KERN_ERR "%s: --> %lu bytes, %lu usecs (%lu KB/s)\n",  \
			__func__, bytes, usecs, rate_kbs); \
	} \
 \
	return count; \
}

static inline void memzero(void *ptr, __kernel_size_t n)
{
	__memzero(ptr, n);
}

DEFINE_BENCH_PROC(memcpy, memcpy(to, from, size);, GFP_KERNEL, GFP_KERNEL)
DEFINE_BENCH_PROC(copyfromuser, { int r = __copy_from_user(to, from, size); r=r; }, GFP_USER, GFP_KERNEL)
DEFINE_BENCH_PROC(copyfromuser_csumpartial, { int r = __copy_from_user(to, from, size); csum_partial(to, size, i); r=r; }, GFP_USER, GFP_KERNEL)
DEFINE_BENCH_PROC(csumpartial, csum_partial(from, size, i);, GFP_KERNEL, GFP_KERNEL)
DEFINE_BENCH_PROC(csumpartialcopyuser, { int err = 0; csum_partial_copy_from_user(from, to, size, 0, &err); if (err) { printk(KERN_ERR "csum_partial_copy_from_user fault\n"); break;} }, GFP_USER, GFP_KERNEL)
DEFINE_BENCH_PROC(csumpartialcopy, csum_partial_copy_nocheck(from, to, size, 0);, GFP_KERNEL, GFP_KERNEL)
DEFINE_BENCH_PROC(memset, memset(to, i, size);, GFP_KERNEL, GFP_KERNEL)
DEFINE_BENCH_PROC(memmove, memmove(to, from, size);, GFP_KERNEL, GFP_KERNEL)
DEFINE_BENCH_PROC(memzero, memzero(to, size);, GFP_KERNEL, GFP_KERNEL)
DEFINE_BENCH_PROC(copypage, size = PAGE_SIZE; copy_page(to, from);, GFP_KERNEL, GFP_KERNEL)
DEFINE_BENCH_PROC(copytouser, { int r = __copy_to_user(to, from, size); r=r; }, GFP_KERNEL, GFP_USER)

static int bench_memmove_overlap(struct file *file,
	const char *buffer, unsigned long count, void *data)
{
	ktime_t start, end;
	int i, callcount, size, actual_size;
	void *from;
	int order, offset;

	if (sscanf(buffer, "%d %d", &size, &callcount) != 2
		|| callcount < 1 || size < 1) {
		printk(KERN_ERR "%s: invalid input\n", __func__);
		return count;
	}

	if (size / 10 == 0)
		offset = 1;
	else
		offset = size / 10;
	actual_size = size + offset;
	order = get_order(actual_size);

	printk(KERN_ERR "%s: testing with count=%d size=%d offset=%d\n",
		__func__, callcount, size, offset);

	from = (void *)__get_free_pages(GFP_KERNEL, order);
	if (!from) {
		printk(KERN_ERR "%s: not enough memory\n", __func__);
		return count;
	}
	
	start = ktime_get();
	for (i = 0; i < callcount; i++) {
		if (unlikely(flush_cache))
			flush_cache_all();
		memmove(from + offset, from, size);
	}
	end = ktime_get();

	free_pages((unsigned long)from, order);

	if (ktime_equal(start, end)) {
		printk(KERN_ERR "%s: too fast; use a bigger count "
			"of calls / a bigger size\n", __func__);
	} else {
		unsigned long usecs = ktime_us_delta(end, start);
		unsigned long bytes = callcount * size;
		unsigned long rate_kbs = (bytes < (1<<22)) ? bytes*1000/usecs : bytes/(usecs/1000);
		printk(KERN_ERR "%s: --> %lu bytes, %lu usecs (%lu KB/s)\n",
			__func__, bytes, usecs, rate_kbs);
	}
	
	return count;
}

static int bench_flush_cache(struct file *file,
	const char *buffer, unsigned long count, void *data)
{
	int i;
	if (sscanf(buffer, "%d", &i) != 1) {
		printk(KERN_ERR "%s: invalid input\n", __func__);
		return count;
	}
	flush_cache = (i != 0);
	printk(KERN_ERR "%s: flush cache now %s\n", __func__, flush_cache ? "ON" : "OFF");
	return count;
}

static int bench_irq_enable(struct file *file,
	const char *buffer, unsigned long count, void *data)
{
	int i;
	if (sscanf(buffer, "%d", &i) != 1) {
		printk(KERN_ERR "%s: invalid input\n", __func__);
		return count;
	}
	irq_enable = (i != 0);
	printk(KERN_ERR "%s: IRQ's are %s\n", __func__, irq_enable ? "ON" : "OFF");
	return count;
}

#define MMU_REGBASE 	0x0c000000
#define MMU_SIZE	0x64

static int bench_readmmureg(struct file *file,
	const char *buffer, unsigned long count, void *data)
{
	int i;
	unsigned long addr;
	void *vaddr;
	if (sscanf(buffer, "%i", &i) != 1) {
		printk(KERN_ERR "%s: invalid input\n", __func__);
		return count;
	}
	addr = MMU_REGBASE + i;
	vaddr = ioremap(addr, 4);
	if (!vaddr) {
		printk(KERN_ERR "%s: ioremap(0x%08x, 4) failed\n",
			__func__, (unsigned int)addr);
		return count;
	}
	printk(KERN_ERR "%s: reading register 0x%08x: 0x%08x\n",
		__func__, (unsigned int)addr, ioread32(vaddr));
	iounmap(vaddr);
	return count;
}

static int bench_writemmureg(struct file *file,
	const char *buffer, unsigned long count, void *data)
{
	int i, value;
	unsigned long addr;
	void *vaddr;
	if (sscanf(buffer, "%i %i", &i, &value) != 2) {
		printk(KERN_ERR "%s: invalid input\n", __func__);
		return count;
	}
	addr = MMU_REGBASE + i;
	vaddr = ioremap(addr, 4);
	if (!vaddr) {
		printk(KERN_ERR "%s: ioremap(0x%08x, 4) failed\n",
			__func__, (unsigned int)addr);
		return count;
	}
	printk(KERN_ERR "%s: writing 0x%08x to register 0x%08x\n",
		__func__, value, (unsigned int)addr);
	iowrite32(value, vaddr);
	printk(KERN_ERR "%s: reading back: 0x%08x\n",
		__func__, ioread32(vaddr));
	iounmap(vaddr);
	return count;
}

static const char *mmu_registers[] = {
	"PartitionID0", // Addr 0x00
	"PartitionID1", // Addr 0x04
	"PartitionID2", // Addr 0x08
	"PartitionID3", // Addr 0x0c
	"Lpartition", // Addr 0x10
	"MMUConfig", // Addr 0x14
	"MasterWrAck", // Addr 0x18
	"ConfigDone", // Addr 0x1c
	"PerfCounter", // Addr 0x20
        "NumInstrReadTrans", // Addr 0x24
	"NumInstrWriteTrans", // Addr 0x28
	"NumInstrReadBytes", // Addr 0x2c
	"NumInstrWriteBytes", // Addr 0x30
	"NumInstrWrap8", // Addr 0x34
	"NumErrorInstrWrap8", // Addr 0x38
	"NumDataReadTrans", // Addr 0x40
	"NumGoodInstrWrap8", // Addr 0x3c
	"NumDataWriteTrans", // Addr 0x44
	"NumDataReadBytes", // Addr 0x48
	"NumDataWriteBytes", // Addr 0x4c
	"NumDataWrap8", // Addr 0x50
	"NumErrorDataWrap8", // Addr 0x54
	"NumGoodDataWrap8", // Addr 0x58
	"CounterReset", // Addr 0x5c
	"PerfCounterEnReg", // Addr 0x60
	NULL
};

static int bench_dumpmmuregs(struct file *file,
	const char *buffer, unsigned long count, void *data)
{
	unsigned long addr;
	void *vaddr;
	int i;
	addr = MMU_REGBASE;
	vaddr = ioremap(addr, MMU_SIZE);
	if (!vaddr) {
		printk(KERN_ERR "%s: ioremap(0x%08x, %d) failed\n",
			__func__, (unsigned int)addr, MMU_SIZE);
		return count;
	}
	for (i = 0; mmu_registers[i]; i++) {
		printk(KERN_ERR "  0x%08x: %18s: 0x%08x\n",
			(unsigned int)(addr + i * 4),
			mmu_registers[i], ioread32(vaddr + i * 4));
	}
	iounmap(vaddr);
	return count;
}

static int __init bench_init(void)
{
	int ret = 0;
	struct proc_dir_entry *pentry;

	info(DRIVER_DESC " compiled for Linux %s, version %s (%s at %s)",
		UTS_RELEASE, DRIVER_VERSION, __DATE__, __TIME__);

#define CREATE_BENCH_PROC(name) \
	pentry = create_proc_entry(#name, 0, NULL); \
	if (pentry) { \
		pentry->write_proc = name; \
		pentry->data = NULL; \
	}

	CREATE_BENCH_PROC(bench_memcpy);
	CREATE_BENCH_PROC(bench_copyfromuser);
	CREATE_BENCH_PROC(bench_copyfromuser_csumpartial);
	CREATE_BENCH_PROC(bench_csumpartial);
	CREATE_BENCH_PROC(bench_csumpartialcopy);
	CREATE_BENCH_PROC(bench_csumpartialcopyuser);
	CREATE_BENCH_PROC(bench_memset);
	CREATE_BENCH_PROC(bench_memmove);
	CREATE_BENCH_PROC(bench_memmove_overlap);
	CREATE_BENCH_PROC(bench_memzero);
	CREATE_BENCH_PROC(bench_copypage);
	CREATE_BENCH_PROC(bench_copytouser);
	CREATE_BENCH_PROC(bench_flush_cache);
	CREATE_BENCH_PROC(bench_irq_enable);
	CREATE_BENCH_PROC(bench_readmmureg);
	CREATE_BENCH_PROC(bench_writemmureg);
	CREATE_BENCH_PROC(bench_dumpmmuregs);
	
	return ret;
}

static void __exit bench_exit(void)
{
	remove_proc_entry("bench_memcpy", NULL);
	remove_proc_entry("bench_copyfromuser", NULL);
	remove_proc_entry("bench_copyfromuser_csumpartial", NULL);
	remove_proc_entry("bench_csumpartial", NULL);
	remove_proc_entry("bench_csumpartialcopy", NULL);
	remove_proc_entry("bench_csumpartialcopyuser", NULL);
	remove_proc_entry("bench_memset", NULL);
	remove_proc_entry("bench_memmove", NULL);
	remove_proc_entry("bench_memmove_overlap", NULL);
	remove_proc_entry("bench_memzero", NULL);
	remove_proc_entry("bench_copypage", NULL);
	remove_proc_entry("bench_copytouser", NULL);
	remove_proc_entry("bench_flush_cache", NULL);
	remove_proc_entry("bench_irq_enable", NULL);
	remove_proc_entry("bench_readmmureg", NULL);
	remove_proc_entry("bench_writemmureg", NULL);
	remove_proc_entry("bench_dumpmmuregs", NULL);
}

module_init(bench_init);
module_exit(bench_exit);

/*-------------------------------------------------------------------------*/

MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_LICENSE("GPL");
