/*HEADER_START*/
/*
 * linux/arch/arm/mach-merlin/qcc.c
 *
 * Copyright (C) Mobilygen Corp
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*HEADER_STOP*/

#ifndef DOXYGEN_SKIP

#include <linux/init.h>
#include <linux/module.h>
#include <linux/mm.h>
#include <linux/sched.h>
#include <linux/hardirq.h>
#include <linux/types.h>
#include <linux/time.h>
#include <linux/proc_fs.h>
#include <linux/major.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/uaccess.h>
#include <linux/io.h>

#include <asm/mach-types.h>
#include <asm/setup.h>
#include <asm/mach/arch.h>

#include <mach/hardware.h>
#include <mach/platform.h>
#include <mach/mobi_reset.h>
#include <mach/mobi_qcc.h>
#include <mach/mobi_qccsisc_regs.h>

#include "qcc.h"

#endif

static void __iomem *qcc_base;

static struct proc_dir_entry *qcc_proc_dir;
spinlock_t qcc_lock;

int qcc_wait(unsigned long time_ms)
{
	uint32_t end = jiffies+(time_ms*HZ)/1000;

	do {
		schedule();    /* schedule at least once */
		if (signal_pending(current))
			return -EPERM;
	} while (jiffies < end);

	return 0;
}

uint32_t mobi_qcc_readreg(uint32_t offset)
{
	return readl(qcc_base+offset);
}
EXPORT_SYMBOL(mobi_qcc_readreg);

int mobi_qcc_writereg(uint32_t data, uint32_t offset)
{
	writel(data, qcc_base+offset);
	return 0;
}
EXPORT_SYMBOL(mobi_qcc_writereg);

/* exthost/qhal/merlin/qhal_qcc.c */
int mobi_qcc_read(int bid, int addr, unsigned long *datap, int len)
{
	int ret = 0, locked = 0;
	unsigned long flags = 0;
	unsigned long qccaddr =
		((((unsigned long)bid) & 0xff) << 24) | ((unsigned long)addr);

	if (!CHECK_QCC_ADDR(addr))
		return -EINVAL;

	if (!in_atomic() && !irqs_disabled()) {
		spin_lock_irqsave(&qcc_lock, flags);
		locked = 1;
	}

	WRITE_CSRStat(1); /* Clear done bit */

	switch (len) {
	case 4:
		CSR_CMD_READ_WORD(qccaddr);
		CSR_CMD_POLLDONE();
		if (CSR_STAT_ERR(READ_CSRStat()))
			ret = -EIO;
		*datap = READ_CSRRdData();
		break;
	case 2:
		CSR_CMD_READ_SHORT(qccaddr);
		CSR_CMD_POLLDONE();
		if (CSR_STAT_ERR(READ_CSRStat()))
			ret = -EIO;
		*datap = (READ_CSRRdData() & 0xFFFF);
		break;
	case 1:
		CSR_CMD_READ_BYTE(qccaddr);
		CSR_CMD_POLLDONE();
		if (CSR_STAT_ERR(READ_CSRStat()))
			ret = -EIO;
		*datap = (READ_CSRRdData() & 0xFF);
		break;
	default:
		ret = -EPERM;
	}

	if (locked)
		spin_unlock_irqrestore(&qcc_lock, flags);

	return ret;
}
EXPORT_SYMBOL(mobi_qcc_read);

int mobi_qcc_write(int bid, int addr, unsigned long data, int len)
{
	int ret = 0, locked = 0;
	unsigned long flags = 0;
	unsigned long qccaddr =
		((((unsigned long)bid) & 0xff) << 24) | ((unsigned long)addr);

	if (!CHECK_QCC_ADDR(addr))
		return -EINVAL;

	if (!in_atomic() && !irqs_disabled()) {
		spin_lock_irqsave(&qcc_lock, flags);
		locked = 1;
	}

	WRITE_CSRStat(1); /* Clear done bit */

	switch (len) {
	case 4:
		CSR_CMD_WRITE_WORD(qccaddr, data);
		break;
	case 2:
		CSR_CMD_WRITE_SHORT(qccaddr, data);
		break;
	case 1:
		CSR_CMD_WRITE_BYTE(qccaddr, data);
		break;
	default:
		ret = -EPERM;
	}

	CSR_CMD_POLLDONE();
	if (CSR_STAT_ERR(READ_CSRStat()))
		ret = -EIO;

	if (locked)
		spin_unlock_irqrestore(&qcc_lock, flags);

	return ret;
}
EXPORT_SYMBOL(mobi_qcc_write);

#ifdef CONFIG_PROC_FS

/* set these defaults to be the resetSet vectors in chipctl */
int procfs_qccrd_bid = 8;
int procfs_qccrd_addr = 0xb0;
int procfs_qccrd_len = 4;

int procfs_qccwr_bid = 8;
int procfs_qccwr_addr = 0xb0;
int procfs_qccwr_len = 4;
unsigned long  procfs_qccwr_data;

/* advance the pointer to next non-empty character  in the input buffer */
static void proc_advance_ptr(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;

	while (((**ptr == ' ') || (**ptr == '\t') || (**ptr == '\r')
				|| (**ptr == '\n')) && (*ptr-buf < count)) {
		(*ptr)++;
	}

}

/* returns pointer to next arg or error if not found */
static int __attribute__((unused)) proc_get_next_arg(
		char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	proc_advance_ptr(ptr, buffer, count);
	if (*ptr-buf >= count) {
		return -EINVAL;
	}
	return 0;
}

/* return the next arg found as unsigned long */
static int get_ularg(
		char **ptr, uint32_t buffer, unsigned long count, uint32_t *arg)
{
	char *buf = (char *)buffer;
	proc_advance_ptr(ptr, buffer, count);
	if (*ptr-buf >= count) {
		printk(KERN_ERR "Invalid argument string\n");
		return -1;
	} else {
		*arg = simple_strtoul(*ptr, ptr, 0);
		if (*arg < 0) {
			printk(KERN_ERR "Invalid number argument\n");
			return -1;
		}
	}
	return 0;
}

static int qcc_proc_get_parts(const char *buffer, unsigned long count,
		int *bid, int *addr,
		int *len, unsigned long *data, char iswrite)
{

	char *ptr = (char *)buffer;
	int error = 0;

	if (get_ularg(&ptr, (uint32_t)buffer, count, bid))
		goto err_help;
	if (*bid <= 0) {
		printk(KERN_ERR "qccsisc: Invalid bid\n");
		error = -EINVAL;
		goto err_help;
	}

	if (get_ularg(&ptr, (uint32_t)buffer, count, addr))
		goto err_help;
	if (*addr <= 0) {
		printk(KERN_ERR "qccsisc: Invalid addr\n");
		error = -EINVAL;
		goto err_help;
	}

	if (get_ularg(&ptr, (uint32_t)buffer, count, len))
		goto err_help;
	if (*len <= 0) {
		printk(KERN_ERR "qccsisc: Invalid len\n");
		error = -EINVAL;
		goto err_help;
	}

	if (iswrite) {
		if (get_ularg(&ptr,
					(uint32_t)buffer, count, (uint32_t *)data) < 0) {
			error = -EINVAL;
			goto err_help;
		}
	}
	while (ptr-buffer < count)
		ptr++;

	return ptr-buffer;

err_help:
	while (ptr-buffer < count)
		ptr++;

	printk(KERN_ERR "qccsisc: syntax is 'echo \"<bid> <addr> <len> %s'\n",
			iswrite ? "<data>\"" : "");
	printk(KERN_ERR "qccsisc: where addr 0x<hex>\n");
	if (!iswrite)
		printk(KERN_ERR "qccsisc: cat readbid entry to see read data\n");
	return error;
}

static int qcc_proc_get_readaddr(struct file *file, const char *buffer,
		unsigned long count, void *data)
{
	qcc_proc_get_parts(buffer, count,
			&procfs_qccrd_bid, &procfs_qccrd_addr,
			&procfs_qccrd_len, NULL, 0);

	return count;
}

static int qcc_proc_show_readdata(char *buf, char **start, off_t offset,
		int count, int *eof, void *data)
{
	int len = 0, ret = 0;
	unsigned long qccdata = 0;
	ret = mobi_qcc_read(procfs_qccrd_bid, procfs_qccrd_addr,
			&qccdata, procfs_qccrd_len);
	if (ret == 0)
		len += sprintf(buf+len,
				"qcc_read: bid %d, addr 0x%x, data 0x%lx (%lu)\n",
				procfs_qccrd_bid, procfs_qccrd_addr, qccdata, qccdata);
	else
		printk(KERN_ERR "qccsisc: problem reading bid %d address 0x%x\n",
				procfs_qccrd_bid, procfs_qccrd_addr);

	*eof = 1;
	return len;
}

static int qcc_proc_writebid(struct file *file, const char *buffer,
		unsigned long count, void *data)
{
	int ret = 0;

	ret = (qcc_proc_get_parts(buffer, count,
				&procfs_qccwr_bid, &procfs_qccwr_addr,
				&procfs_qccwr_len, &procfs_qccwr_data, 1));

	if (ret > 0) {
		ret = mobi_qcc_write(procfs_qccwr_bid,
				procfs_qccwr_addr, procfs_qccwr_data, procfs_qccwr_len);

		if (ret < 0) {
			printk(KERN_ERR "qccsisc: problem writing data to bid %d address 0x%x\n",
					procfs_qccwr_bid, procfs_qccwr_addr);
		}
	}

	return count;
}

static int qcc_proc_writereg(struct file *file, const char *buffer,
		unsigned long count, void *data)
{
	char *ptr = (char *)buffer;
	uint32_t offset, writedata;
	int error = 0;

	get_ularg(&ptr, (uint32_t)buffer, count, &offset);
	if (offset <= 0) {
		printk(KERN_ERR "qccsisc: Invalid addr\n");
		error = -EINVAL;
		goto err;
	}

	get_ularg(&ptr, (uint32_t)buffer, count, &writedata);

	mobi_qcc_writereg(writedata, offset);
	printk(KERN_INFO "Write 0x%x(%d) to base 0x%p, offset 0x%x\n",
			writedata, writedata, qcc_base, offset);

err:
	/* flush */
	while (ptr-buffer < count)
		ptr++;

	return ptr - buffer;
}

static int qcc_proc_readreg(struct file *file, const char *buffer,
		unsigned long count, void *data)
{
	char *ptr = (char *)buffer;
	int qccdata = 0;
	uint32_t offset;
	int error;

	get_ularg(&ptr, (uint32_t)buffer, count, &offset);
	if (offset <= 0) {
		printk(KERN_ERR "qccsisc: Invalid addr\n");
		error = -EINVAL;
		goto err;
	}

	qccdata = mobi_qcc_readreg(offset);
	printk(KERN_INFO "Read at base 0x%p offset 0x%x is 0x%x(%d)\n",
			qcc_base, offset, qccdata, qccdata);

err:
	while (ptr-buffer < count)
		ptr++;

	return ptr - buffer;
}

static void qcc_procfs_init(void)
{

	struct proc_dir_entry *pentry;

	if (qcc_proc_dir == NULL)
		qcc_proc_dir = proc_mkdir("driver/qcc", NULL);

	if (qcc_proc_dir == NULL) {
		qcc_proc_dir = proc_mkdir("driver", NULL);
		if (qcc_proc_dir != NULL)
			proc_mkdir("driver/qcc", NULL);
		else
			return;
	}

	pentry = create_proc_entry("driver/qcc/readbid",
			S_IRUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->read_proc  = qcc_proc_show_readdata;
		pentry->write_proc = qcc_proc_get_readaddr;
	}

	pentry = create_proc_entry("driver/qcc/writebid",
			S_IRUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->write_proc = qcc_proc_writebid;
	}

	pentry = create_proc_entry("driver/qcc/readreg",
			S_IRUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->write_proc  = qcc_proc_readreg;
	}

	pentry = create_proc_entry("driver/qcc/writereg",
			S_IRUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->write_proc = qcc_proc_writereg;
	}
}
#endif

/*
 *  call this from the board file so we can access the qcc
 *  chain, which we need to set the timer clock which is
 *  needed to run the system timer
 */
void __init __early_qccsisc_init(void)
{
	qcc_base = __io_address(QCC_BASE);
	spin_lock_init(&qcc_lock);
}

/**
 * \brief mobi_qccsisc_init:
 * 	Called at system startup.  Priority is arch_initcall
 */
static int __init mobi_qccsisc_init(void)
{
#ifdef CONFIG_PROC_FS
	qcc_procfs_init();
#endif
	return 0;
}
core_initcall(mobi_qccsisc_init);
