/*HEADER_START*/
/*
 * linux/arch/arm/mach-merlin/hmux.c
 *
 * Copyright (C) Mobilygen Corp
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*HEADER_STOP*/

/* enable doxygen */
/** \file
 * Driver for Mobilygen Host Mux Interface
 *
 *   The host interface support serveral different iterface types. 
 * A common set of I/O pins are used to implement the various interfaces.
 * All interfaces time-share the common set of pins.  The hmux API provide
 * syncronization method for access to the shared data bus.  
 *
 * \verbatim
 * Setup the hmux definition like this:
 *
 *    static void nand_hmux_release(struct device *dev)
 *    {
 *    }
 *
 *    static struct mobi_hmux_drvdata hmux_data = {
 *			.chipen = HMUX_NAND_CE0,
 *    };
 *
 *    static struct platform_device mobi_hmux_device = {
 *			.name = "mobi_hmux",
 *			.id   = HMUX_HOST_CS1,
 *			.dev  = {
 *	    		.release     = nand_hmux_release,
 *	    		.platform_data = &hmux_data,
 *	 		},
 *    };
 *
 * during device init or probe register the hmux
 *
 *  	mobi_hmux_register(&mobi_hmux_device);
 *
 * when exiting or unload the device unregister the hmux
 *
 * 		mobi_hmux_unregister(&mobi_hmux_device);
 *
 * to use the hmux:
 *
 * 	mobi_hmux_request();
 * 	.
 * 	.. do work that access the device ..
 * 	.
 * 	mobi_hmux_release();
 *
 * \endverbatim
 */

#ifndef DOXYGEN_SKIP

#include <linux/init.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/mm.h>
#include <linux/types.h>
#include <linux/time.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>

#include <mach/hardware.h>
#include <asm/mach-types.h>
#include <asm/setup.h>
#include <asm/irq.h>
#include <asm/io.h>

#include <asm/mach/arch.h>
#include <asm/mach/irq.h>
#include <asm/mach/map.h>
#include <asm/mach/time.h>

#include <mach/platform.h>
#include <mach/mobi_hmux.h>
#include <mach/mobi_qcc.h>


#define HMUX_HOSTCS_MASK 			0xf
#define HMUX_CYCLE_DELAY_BIT0_MASK	0x01000000
#define HMUX_CYCLE_DELAY_BIT0_SHIFT	24
#define HMUX_CYCLE_DELAY_BIT1_MASK	0x02000000
#define HMUX_CYCLE_DELAY_BIT1_SHIFT	25
#define HMUX_CYCLE_DELAY_MASK		0x1f000000

static struct mutex hmux_lock;
static uint32_t hmux_shadow = 0xffffff;
static uint16_t hmux_ref_cnt = 0;
struct thread_info *hmux_owner = NULL;

#endif

/** 
 * \brief mobi_hmux_request_try:
 * 	Request access to the hmux
 *
 * \retval Zero - upon success, -1 on failure
 *
 * \remark
 *   The function tries to grant access to the hmux and locks 
 * it so that only a single device can use it at a time.  
 * The call will not block but still must not be used
 * in interrupt context.
 * 
 */
int mobi_hmux_request_try(void)
{
	int ret = 0;

	if (hmux_ref_cnt > 1) 
		ret = mutex_trylock(&hmux_lock) ? 0 : -1;

	if (ret == 0) 
		hmux_owner = current->stack;

	return ret;
}
/// @cond
EXPORT_SYMBOL(mobi_hmux_request_try);
/// @endcond

/** 
 * \brief mobi_hmux_request:
 * 	Request access to the hmux.  This function 
 * 	must not be used in interrupt context.
 *
 * \retval Zero - upon success
 *
 * \remark
 *   The function grants the access to the hmux and locks 
 * it so that only a single device can use it at a time.  
 * The call will block until access can be granted to the hmux.
 * 
 */
int mobi_hmux_request(void)
{
	if (hmux_ref_cnt > 1) 
		mutex_lock(&hmux_lock);

	hmux_owner = current->stack;
	return 0;
}
/// @cond
EXPORT_SYMBOL(mobi_hmux_request);
/// @endcond

/** 
 * \brief mobi_hmux_release:
 * 	Release access to the hmux
 *
 * \retval Zero - upon success
 * \retval -EACCES - not the current owner and trying to unlock mutex 
 *
 * \remark
 *   This function releases the hmux.
 * 
 */
int mobi_hmux_release(void)
{
	int ret = 0;

	if (hmux_owner == current->stack) {
		hmux_owner = NULL;
		if (hmux_ref_cnt > 1) 
			mutex_unlock(&hmux_lock);
	}
	else {
		/* should we print a message here ?? */
		ret = -EACCES;
	}

	return ret;
}
/// @cond
EXPORT_SYMBOL(mobi_hmux_release);
/// @endcond

/** 
 * \brief mobi_hmux_owner:
 * 	Get current owner of the hmux
 *
 * \retval NULL    - no owner
 * \retval POINTER - thread info of the owner 
 *
 */
void *mobi_hmux_owner(void)
{
	return (void*)hmux_owner;
}
/// @cond
EXPORT_SYMBOL(mobi_hmux_owner);
/// @endcond

/** 
 * \brief mobi_hmux_register:
 * 	Register a platform device for the hmux
 *
 * \retval Zero - upon success
 * \retval Negetive value - upon failure
 *
 * \remark
 *   Create a function which can be called from a driver
 * which create a dependency on the hmux driver.  When 
 * platform_device_register is called from a driver there is 
 * not gaurentee that the platform_driver is or will be registered
 * in the future; in this case, trying to access the hmux could
 * have seriously bad side-effects!
 */
int mobi_hmux_register(struct platform_device *pdev)
{
	return(platform_device_register(pdev));
}
/// @cond
EXPORT_SYMBOL(mobi_hmux_register);
/// @endcond

/** 
 * \brief mobi_hmux_unregister:
 * 	Unregister a platform device for the hmux
 *
 * \retval void
 */
void mobi_hmux_unregister(struct platform_device *pdev)
{
	platform_device_unregister(pdev);
} 
/// @cond
EXPORT_SYMBOL(mobi_hmux_unregister);
/// @endcond


static irqreturn_t mobi_hmux_interrupt(int irq, void *dev_id)
{
    panic("Host Mux Collision Interrupt(contact Maxim support)\n");
    return IRQ_HANDLED;
}

static int __devinit mobi_hmux_probe(struct platform_device *pdev)
{
	int ret = 0;
	char i;
	struct mobi_hmux_drvdata *drvdata = pdev->dev.platform_data;

	for(i=0;i<=5;i++) {
		if (((hmux_shadow >> (i*4)) & 0xf) == (drvdata->chipen & 0xf)) {
			printk(KERN_ERR "Chip enable %d already in use by chip select %d\n",
					drvdata->chipen & 0xf, pdev->id);
			return(-1);
		}
	}

	switch (drvdata->chipen) {
		case HMUX_MHIF_CE0:
		case HMUX_MHIF_ADDR_22_CE0: 
		case HMUX_FLASH_ADDR_22_CE0:
			hmux_shadow &= ~HMUX_CYCLE_DELAY_BIT0_MASK;
			break;
		case HMUX_MHIF_16BIT_CE0:
		case HMUX_MHIF_ADDR_23_CE0: 
		case HMUX_FLASH_ADDR_23_CE0:
			hmux_shadow |= HMUX_CYCLE_DELAY_BIT0_MASK;
			break;
		case HMUX_MHIF_ADDR_23_CE1: 
		case HMUX_FLASH_ADDR_23_CE1:
			hmux_shadow &= ~HMUX_CYCLE_DELAY_BIT1_MASK;
			break;
		case HMUX_DEV_13_ZERO_CE0:
		case HMUX_FLASH_ADDR_24_CE0:
			hmux_shadow |= HMUX_CYCLE_DELAY_BIT1_MASK;
			break;
		default:
			/* don't have to deal with other devices in this way */
			break;
	}

	hmux_shadow &= ~(HMUX_HOSTCS_MASK << (pdev->id*4));
	hmux_shadow |= ((drvdata->chipen & 0xf) << (pdev->id*4));

	//printk("mobi_hmux_probe: hostcs: 0x%x, chipen: 0x%x, hmux_shadow 0x%x\n", 
	//		pdev->id, drvdata->chipen, hmux_shadow);
	/* update for each device that registers */
	ret = mobi_qcc_write(QCC_BID_CHIPCTL,
			QCC_CHIPCTL_HOSTMUX_CONTROL, 
			hmux_shadow,
			4);

	if (ret == 0)
		hmux_ref_cnt++;

	return ret;
}

static void mobi_hmux_shutdown(struct platform_device *pdev)
{
}

static int mobi_hmux_remove(struct platform_device *pdev)
{
	struct mobi_hmux_drvdata *drvdata = pdev->dev.platform_data;

	if (hmux_ref_cnt > 0)
		hmux_ref_cnt-- ;

	hmux_shadow &= ~(drvdata->chipen << (pdev->id*4));
	return 0;
}

static struct platform_driver mobi_hmux_driver = {
	.probe          = mobi_hmux_probe,
	.shutdown       = mobi_hmux_shutdown,
	.remove         = mobi_hmux_remove,
	.driver         = {
		.name   = "mobi_hmux",
		.owner  = THIS_MODULE,
	},
};

static void __exit mobi_hmux_exit(void)
{
	mobi_qcc_writereg(MOBI_QCCSISC_PERI_INTENCLR_HMUXERR_MASK, 
			MOBI_QCCSISC_PERI_INTENCLR_OFFSET);

	free_irq(MOBI_IRQ_HMUX, NULL);

	platform_driver_unregister(&mobi_hmux_driver);
}

static int __init mobi_hmux_init(void)
{
	int ret = 0;

	hmux_ref_cnt = 0;
	if ((ret = platform_driver_register(&mobi_hmux_driver)) == 0) {
		mutex_init(&hmux_lock);
		ret = request_irq(MOBI_IRQ_HMUX, mobi_hmux_interrupt, 
				IRQF_DISABLED, "mobihmux", NULL);
		mobi_qcc_writereg(MOBI_QCCSISC_PERI_INTENSET_HMUXERR_MASK, 
				MOBI_QCCSISC_PERI_INTENSET_OFFSET);
	}
	/* set to default */
	ret = mobi_qcc_write(QCC_BID_CHIPCTL,
			QCC_CHIPCTL_HOSTMUX_CONTROL, 
			hmux_shadow, 4);

	return ret;
}
/// @cond
arch_initcall(mobi_hmux_init);
/// @endcond

