/*
 *  linux/arch/arm/mach-merlin/clock.c
 *
 *  Copyright (C) 2006 Mobilygen Corp.
 *  Derived from mach-versatile/clock.c
 *
 *  Copyright (C) 2004 ARM Limited.
 *  Written by Deep Blue Solutions Limited.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
/** \file
 *
 * Driver for Mobilygen clock driver
 * \verbatim
 * 
 *   This is a summary of the Mobilygen clock driver.  More details 
 * can be found in the function documentation.  The driver provides
 * the following API calls.
 * 	mobi_clock_set_rate
 * 	  - this function is used to set the rate of a clock.  If 
 * 	the new rate is 0, the clock will be disabled.   The driver
 * 	will attempt to program the clock to be as close to the requested
 * 	rate as possible without being greater than.  It is strongly 
 * 	recommended to call mobi_clock_get_rate to get the new clock
 * 	rate.  
 *
 * 	mobi_clock_get_rate 
 * 	  - returns the current rate for a given clock
 *
 * 	mobi_clock_register_callback 
 * 	  - a driver should register a callback function against any
 *      clock that it is interested in. If that clock changes, the 
 *      callback will be triggered.
 *
 * 	mobi_clock_unregister_callback
 * 	  - if no longer interested in clock changes OR if unloading
 * 	a module, the callback should be unregistered.
 *
 * The clk_* functions are only provided for compatibility
 * with existing linux code.  New code written for the 
 * Mobilygen platform line should use the mobi_clock_* API 
 * only
 *
 * \endverbatim
 *
 */

#ifndef DOXYGEN_SKIP

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/list.h>
#include <linux/errno.h>
#include <linux/err.h>
#include <linux/string.h>
#include <linux/clk.h>
#include <linux/mutex.h>
#include <linux/timex.h>
#include <linux/proc_fs.h>

#include <mach/platform.h>
#include <mach/mobi_clock.h>
#include <mach/mobi_qcc.h>
#include <mach/mobi_clk_regs.h>

#include "clock.h"

#define CLOCK_READ_FIELD(clk, field, reg) \
		MOBI_CLK_##clk##_##field##_R(reg)

#define CLOCK_WRITE_FIELD(clk, field, reg, data) \
		MOBI_CLK_##clk##_##field##_CLR_AND_SET(reg, data)

static LIST_HEAD(clocks);
static DEFINE_MUTEX(clocks_mutex);
static struct proc_dir_entry *clock_proc_dir=NULL;

/* the default pll input clock is 12Mhz unless set on cmdline */
#define USB_OSC_RATE 12000000
unsigned long __system_pll_fin = USB_OSC_RATE; 

#if 0
#define dprintk(x...)	printk(x)
#else
#define dprintk(x...)
#endif

#endif

static int32_t get_hl_freq(struct clk *clk, unsigned long *fout);

/*
* Internal, looks up clk struct base on clk_id
 */
static struct clk *clock_get_by_id(enum clock_id clk_id)
{
	struct clk *p, *clk = NULL;

	mutex_lock(&clocks_mutex);

	list_for_each_entry(p, &clocks, node) {
		if (clk_id == p->id) {
			clk = p;
			break;
		}
	}
	mutex_unlock(&clocks_mutex);

	return clk;
}

/* there is some chance that clocks could be changed under us by the
*  codec firmware.  this could cause really bad things to happen but
*  it might not.  so just to check, before we do anything with a clock
*  lets grab an updated copy of it control register and work from that
 */
static int32_t update_shadow_reg(struct clk *clk)
{
	/* don't do this on the FPGA because bits like the pll lock bit will
	*  not change */
#ifndef CONFIG_MERLIN_IS_FPGA
	if (mobi_qcc_read(QCC_BID_CHIPCTL, clk->qcc_addr, &clk->shadow_reg, 4)) {
		printk(KERN_ERR "Unable to read clock register, clk_id: %d\n", 
				clk->id);
		return -1;
	}
#endif
	return 0;
}

/* just toggle the clk enable bit if it has one */
static int clock_disable(struct clk *clk)
{

	int32_t ret = 0;

	mutex_lock(&clk->lock);
	switch (clk->id) {
		case CLOCK_ID_PLL0:
		case CLOCK_ID_PLL1:
		case CLOCK_ID_PLL2:
		case CLOCK_ID_PLL3:
			/* set PD field to 1, save power */
			clk->shadow_reg |= MOBI_CLK_PLL_PD_MASK;
			clk->shadow_reg &= ~(clk->enable_mask);
			break;
		case CLOCK_ID_UARTDBG:
		case CLOCK_ID_UART0:
		case CLOCK_ID_UART1:
		case CLOCK_ID_I2C0:
		case CLOCK_ID_I2C1:
		case CLOCK_ID_PLL2HL:
		case CLOCK_ID_PLL3HL:
		case CLOCK_ID_TIMER:
		case CLOCK_ID_SDMMC:
		case CLOCK_ID_SSI0:
		case CLOCK_ID_SSI1:
		case CLOCK_ID_BS:
		case CLOCK_ID_VIN0:
		case CLOCK_ID_VIN1:
		case CLOCK_ID_VOUT0:
			/* these clocks 0 means off */
			clk->shadow_reg &= ~(clk->enable_mask);
			break;
		default:
			printk(KERN_WARNING "Invalid clock, cannot disable"
					"clock_id %d\n", clk->id);
			mutex_unlock(&clk->lock);
			return -1;
			break;
	}
	ret = mobi_qcc_write(QCC_BID_CHIPCTL, clk->qcc_addr,
			clk->shadow_reg, 4);

	if (ret == 0)
		handle_callbacks(clk);
	else 
		printk(KERN_ERR "ERROR - failed to disable clock_id %d\n", clk->id);

	mutex_unlock(&clk->lock);

	return ret;
}


/* calculates the output frequency of a PLL.  for PLL0 and PLL1, 
*  the driving clock rate is determined at boot time and is stored
*  in __system_pll_fin.  For PLL2 and PLL3, need to look at what is
*  driving these
 */
static int get_pll_freq(enum clock_id clk_id, unsigned long *fout)
{
	uint32_t sdiv, intprog, sout, enabled;
	struct clk *clk = clock_get_by_id(clk_id);
	unsigned long pll_freq = 0;
	struct clk *tmp_clk;
	int ret = 0;

	*fout = 0;
	mutex_lock(&clk->lock);

	if (update_shadow_reg(clk)) {
		ret = -EIO;
		goto out;
	}
	sdiv = pll_sdiv_values[CLOCK_READ_FIELD(PLL, SDIV, clk->shadow_reg)];
	intprog = CLOCK_READ_FIELD(PLL, INTPROG, clk->shadow_reg);
	sout = pll_sout_values[CLOCK_READ_FIELD(PLL, SOUT, clk->shadow_reg)];
	enabled = CLOCK_READ_FIELD(PLL, ENABLE, clk->shadow_reg);

	/* lets just be safe and make that somehow we didn't get a 0 for sout */
	if (sout == 0) {
		ret = -ERANGE;
		goto out;
	}
	else {
		if (enabled) {
			switch (clk_id) {
				case CLOCK_ID_PLL2:
					tmp_clk = clock_get_by_id(CLOCK_ID_PLL2HL);
					ret = get_hl_freq(tmp_clk, &pll_freq);
					if (ret >= 0)
						*fout = (pll_freq * sdiv * intprog)/sout;
					break;
				case CLOCK_ID_PLL3:
					tmp_clk = clock_get_by_id(CLOCK_ID_PLL3HL);
					ret = get_hl_freq(tmp_clk, &pll_freq);
					if (ret >= 0)
						*fout = (pll_freq * sdiv * intprog)/sout;
					break;
				case CLOCK_ID_PLL0:
				case CLOCK_ID_PLL1: 
				default:
					*fout = (__system_pll_fin * sdiv * intprog)/sout;
					break;
			}
		}

	}
out:
	mutex_unlock(&clk->lock);

	dprintk("pll id %d, fout %lu\n", clk_id, *fout);
	return ret;
}

/* determine the current output freq. of a scaler */
static int get_scaler_freq(enum clock_id clk_id, unsigned long *fout)
{
	int pre, div, ret = 0;
	unsigned long pll_fout = 0;
	struct clk *clk = clock_get_by_id(clk_id);

	*fout = 0;
	mutex_lock(&clk->lock);

	if (update_shadow_reg(clk)) {
		ret = -EIO;
		goto out;
	}

	pre = scaler_pre_values[CLOCK_READ_FIELD(SCALER, PRE, clk->shadow_reg)];
	div = CLOCK_READ_FIELD(SCALER, DIV, clk->shadow_reg);

	/* prevent divide by 0 */
	if (pre == 0 || ((div == 0) && (clk_id!=CLOCK_ID_SCALER1))) {
		ret = -ERANGE;
		goto out;
	}

	switch (clk->id) {
		case CLOCK_ID_SCALER0:
			ret = get_pll_freq(CLOCK_ID_PLL0, &pll_fout);
			break;
		case CLOCK_ID_SCALER1:
			ret = get_pll_freq(CLOCK_ID_PLL1, &pll_fout);
			break;
		case CLOCK_ID_SCALER2:
			ret = get_pll_freq(CLOCK_ID_PLL2, &pll_fout);
			break;
		case CLOCK_ID_SCALER3:
			ret = get_pll_freq(CLOCK_ID_PLL3, &pll_fout);
			break;
		default:
			break;
	}
	// scaler output freq for ARM is 
	// (fin / PRE) / DIV  = Fin/(PRE * DIV)
	if (!ret)
	{	/* WARNING: On Scaler1 the second stage is not used (div) */
		if (clk->id==CLOCK_ID_SCALER1) 
			*fout = pll_fout/pre;
		else *fout = pll_fout/(pre*div);
	};

out:
	mutex_unlock(&clk->lock);

	return ret;
}

/* 
*  determine the current output freq of a N/R divider. Note
*  that all N/R dividors are driven my the HF4PH signal of
*  scaler 0(which will be PLL0-Fout divided by 4
* in the case of an error, return an fout of 0 
 */
static int get_nr_freq(struct clk *clk, unsigned long *fout)
{
	unsigned long fin;
	int32_t N = 0, R = 0, enabled = 0, ret = 0;

	*fout = 0;
	mutex_unlock(&clk->lock);

	/* all NR clocks are driven by scaler0 fin/4 */
	if ((ret = get_pll_freq(CLOCK_ID_PLL0, &fin)) < 0) {
		goto out;
	}
	if (fin == 0)
		goto out;

	if (update_shadow_reg(clk)) {
		ret = -EIO;
		goto out;
	}

	/* 
	* NR dividers are driven by the HF4PH signal on the scaler
	* so we divide the scaler Fout by 4 to get this freq
	 */
	fin /= 4;  

	switch (clk->id) {
		case CLOCK_ID_QMM:
			N = CLOCK_READ_FIELD(CORE, NDIV, clk->shadow_reg);
			R = CLOCK_READ_FIELD(CORE, RDIV, clk->shadow_reg);
			enabled = CLOCK_READ_FIELD(CORE, ENABLE, clk->shadow_reg);
			break;
		case CLOCK_ID_VIN0:
		case CLOCK_ID_VIN1:
		case CLOCK_ID_VOUT0:
			N = CLOCK_READ_FIELD(VIDCTRL, NDIV, clk->shadow_reg);
			R = CLOCK_READ_FIELD(VIDCTRL, RDIV, clk->shadow_reg);
			enabled = CLOCK_READ_FIELD(VIDCTRL, ENABLE, clk->shadow_reg);
			break;
		default:
			break;
	}

	if (R == 0) {
		goto out;
	}
	/* Fclko = Fin / (2*(((8192-R)/N)+1)) */
	if (enabled) {
		*fout = fin/(2*(((8192-R)/N)+1));
	}
out:
	dprintk("%s(%d): N = %d, R = %d, fin = %lu, enabled = %d, fout = %lu\n", 
			__func__, clk->id, N, R, fin, enabled, *fout);

	mutex_unlock(&clk->lock);

	return ret;
}

/* 
*  determine the current output freq of a H/L divider. Note
*  that all H/L dividors are driven my the HF4PH signal of
*  scaler 0(which will be PLL0-Fout divided by 4
* in the case of an error, return an fout of 0 
 */
static int32_t get_hl_freq(struct clk *clk, unsigned long *fout)
{
	unsigned long fin;
	int H = 0, L = 0, enabled = 0, ret = 0;

	*fout = 0;
	mutex_lock(&clk->lock);

	/* all NR clocks are driven by scaler0 fin/4 */
	if ((ret = get_pll_freq(CLOCK_ID_PLL0, &fin)) < 0) {
		goto out;
	}

	if (fin == 0)
		goto out;

	if (update_shadow_reg(clk)) {
		ret = -EIO;
		goto out;
	}

	/* 
	* HL dividers are driven by the HF4PH signal on the scaler
	* so we divide the scaler Fout by 4 to get this freq
	 */
	fin /= 4;  

	switch (clk->id) {
		case CLOCK_ID_UARTDBG:
		case CLOCK_ID_UART0:
		case CLOCK_ID_UART1:
			H = CLOCK_READ_FIELD(HL6, HDIV, clk->shadow_reg);
			L = CLOCK_READ_FIELD(HL6, LDIV, clk->shadow_reg);
			enabled = CLOCK_READ_FIELD(HL6, ENABLE, clk->shadow_reg);
			break;
		case CLOCK_ID_I2C0:
		case CLOCK_ID_I2C1:
		case CLOCK_ID_PLL2HL:
		case CLOCK_ID_PLL3HL:
			H = CLOCK_READ_FIELD(HL8, HDIV, clk->shadow_reg);
			L = CLOCK_READ_FIELD(HL8, LDIV, clk->shadow_reg);
			enabled = CLOCK_READ_FIELD(HL8, ENABLE, clk->shadow_reg);
			break;
		case CLOCK_ID_TIMER:
		case CLOCK_ID_SDMMC:
		case CLOCK_ID_SSI0:
		case CLOCK_ID_SSI1:
		case CLOCK_ID_BS:
			H = CLOCK_READ_FIELD(HL9, HDIV, clk->shadow_reg);
			L = CLOCK_READ_FIELD(HL9, LDIV, clk->shadow_reg);
			enabled = CLOCK_READ_FIELD(HL9, ENABLE, clk->shadow_reg);
			break;
		default:
			break;
	} 

#ifdef CONFIG_MERLIN_IS_FPGA
	*fout = MERLIN_SYSTEM_CLOCK;
#else
	/* Fclko = Fclki / (H+1) + (L+1) */
	if (enabled) 
		*fout = fin /((H+1) + (L+1));
#endif

out:
	dprintk("%s(%d): H = %d, L = %d\n\tfin = %lu, enabled = %d, fout = %lu\n", 
			__func__, clk->id, H, L, fin, enabled, *fout);

	mutex_unlock(&clk->lock);
	return ret;
}

static int32_t get_vid_freq(struct clk *clk, unsigned long *fout)
{
	int ret = 0;
	uint8_t srcsel, enable, pcoe;

	*fout = 0;
	mutex_lock(&clk->lock);

	if (update_shadow_reg(clk)) {
		ret = -EIO;
		goto out;
	}

	enable = CLOCK_READ_FIELD(VIDCTRL, ENABLE, clk->shadow_reg);
	if (enable == 0) 
		goto out;

	/* if pcoe is 0, vid clock src is external and can't 
	*  determine that freqency
	 */
	pcoe = CLOCK_READ_FIELD(VIDCTRL, PCOE, clk->shadow_reg);
	if (pcoe == 0) {
		printk(KERN_NOTICE "Pixel clock is being driven by external clock source\n");
		goto out;
	}

	srcsel = CLOCK_READ_FIELD(VIDCTRL, SRCSEL, clk->shadow_reg);
	if (srcsel == VID_SRC_SELECT_PLL2) {
		dprintk("%s(%d): srcsel is pll2\n", __func__, clk->id);
		if ((ret = get_scaler_freq(CLOCK_ID_SCALER2, fout)) < 0) 
			goto out;
	}
	else if (srcsel == VID_SRC_SELECT_PLL3) {
		dprintk("%s(%d): srcsel is pll3\n", __func__, clk->id);
		if ((ret = get_scaler_freq(CLOCK_ID_SCALER3, fout)) < 0) 
			goto out;
	}
	else {
		dprintk("%s(%d): srcsel is nr\n", __func__, clk->id);
		ret = get_nr_freq(clk, fout);
	}

out:
	mutex_unlock(&clk->lock);
	return ret;
}

static int32_t get_aud_freq(struct clk *clk, unsigned long *fout)
{
	int ret = 0;
	uint8_t srcsel, enable, acoe;

	*fout = 0;
	mutex_lock(&clk->lock);

	if (update_shadow_reg(clk)) {
		ret = -EIO;
		goto out;
	}

	enable = CLOCK_READ_FIELD(AUDCTRL, ENABLE, clk->shadow_reg);
	if (enable == 0) 
		goto out;

	/* if acoe is 0, audio clock src is external and can't 
	*  determine that freqency
	 */
	acoe = CLOCK_READ_FIELD(AUDCTRL, ACOE, clk->shadow_reg);
	if (acoe == 0) {
		printk(KERN_NOTICE "Audio clock is being driven by external clock source\n");
		goto out;
	}


	srcsel = CLOCK_READ_FIELD(AUDCTRL, SRCSEL, clk->shadow_reg);
	if (srcsel == AUD_SRC_SELECT_PLL2) {
		if ((ret = get_scaler_freq(CLOCK_ID_SCALER2, fout)) < 0) 
			goto out;
	}
	else if (srcsel == AUD_SRC_SELECT_PLL3) {
		if ((ret = get_scaler_freq(CLOCK_ID_SCALER3, fout)) < 0) 
			goto out;
	}

out:
	dprintk("fout = %lu\n", *fout);
	mutex_unlock(&clk->lock);
	return ret;
}

/* 
*  Find the values for H and L that will give us the closest
*  freqency to the request rate without going over
*  HL and NR dividers will adjust freq locally, but the input 
*  freq must be determined upstream
 */
static void hl_calculate_div(unsigned long fout, 
		unsigned long fin, uint32_t *hdiv, uint32_t *ldiv, uint32_t mask)
{

	uint32_t cycles;

	/* from Merlin clock spec, section HL Clock Divider
	* To generate a clock, clko, from another clock clki where
	* Fclko = Fclki / (H+1) + (L+1)
	* initiall try equal duty cycle, so H = L which gives
	* H = (Fclki - 2*Fclko) / 2 * Fclko
	* H = (Fclki / 2* Fclko) - 1
	 */ 

	cycles=(fin+(fout>>1))/fout;
    *hdiv = (cycles>>1)-1;
    *ldiv = cycles-(*hdiv)-2;
	/* we don't want to be over, so increase one of the divisors */
    if (fin/((*hdiv+1)+(*ldiv+1)) > fout)
        (*hdiv)++;

	/* we have 3 different H/L dividors, 6, 8 and 9, if for some reason
	*  we exceed the size, just set it to the max
	*/
	/*
	if (*hdiv & ~(mask)) {
		dprintk("hdiv of %u, exceeds max size, set to %u\n", *hdiv, mask);
		*hdiv = mask;
	}

	if (*ldiv & ~(mask)) {
		dprintk("ldiv of %u, exceeds max size, set to %u\n", *hdiv, mask);
		*ldiv = mask;
	}
	*/

	dprintk("%s: H = %d, L = %d\n\tfin = %lu, fout requested = %lu, fout given = %lu\n", 
		    __func__, *hdiv, *ldiv, fin, fout, fin/((*hdiv+1)+(*ldiv+1)));
}		

/* 
*  Find the values for N and N that will give us the closest
*  freqency to the request rate without going over
*  HL and NR dividers will adjust freq locally, but the input 
*  freq must be determined upstream
 */
static void nr_calculate_div(unsigned long fout, 
		unsigned long fin, int *ndiv, int *rdiv, uint32_t size)
{

	int N = 0, R = 1;
	unsigned long estimate = 0, min_diff = 0;
	/* Fclko = Fin / (2*(((8192-R)/N)+1)) */
	/* maybe not the most efficient but should work */
	dprintk("nr_calc: fin = %lu, fout requested = %lu\n", fin, fout);
	if (fout == fin) {
		/* best we can do is fin/2 */
		*ndiv = 1;
		*rdiv = size;
	}
	else {
		for (N=1; N < size; N++) {
			*ndiv = N;
			for (R=size-1; R > 0; R--) {
				estimate = fin/(2*(((8192-R)/N)+1));
				if (estimate == fout) {
					goto done;
				}
				else if (estimate < fout) {
					if (min_diff > (fout-estimate)) 
						min_diff = fout - estimate;
					break;
				}
			}
		}
	}

done:
	dprintk("%s: N = %d, R = %d, fout given = %lu\n", 
			__func__, *ndiv , *rdiv , estimate);
}

struct clk_calc_values {
	unsigned long calculated;
	int32_t intdiv;
	uint8_t sout;
	unsigned long difference;
};

/* determine the pll value to output a frequency
* equation for pll output frequency is the following:
* fout = (fin * intprog * sdiv)/sout
 */
static int32_t 
set_pll_freq(struct clk *pll, unsigned long fout_wanted)
{
	int32_t sdiv, min_sdiv, sout;
	int32_t ret = 0, sdiv_reg, j, sout_reg;
	struct clk_calc_values *calc = NULL;
	unsigned long min_diff;

	calc = (struct clk_calc_values*) 
		kmalloc(ARRAY_SIZE(pll_sdiv_values)*(sizeof(struct clk_calc_values)), 
				GFP_KERNEL);
	if (calc == NULL) 
		return -ENOMEM;

	dprintk("%s: clk_id %d, requested freq %lu\n", 
			__func__, pll->id, fout_wanted);
	/* loop to do simple search for values */
	for (sdiv_reg=0; sdiv_reg < ARRAY_SIZE(pll_sdiv_values); sdiv_reg++) {
		sdiv = pll_sdiv_values[sdiv_reg];

		for (sout_reg=0; sout_reg < ARRAY_SIZE(pll_sout_values); sout_reg++) {
			sout = pll_sout_values[sout_reg];

			for (j=PLL_INTPROG_MIN; j <= PLL_INTPROG_MAX; j++) {
				/* go thru the numbers until we find the first one greater than
				*  the requested fout.  we want the freq to be <= to request
				*  so deduct 1(unless already at 1) to get previous value
				 */
				if (((__system_pll_fin * sdiv * j)/sout) > fout_wanted) {
					/* j-1 to get previous value which is less than requested */
					calc[sdiv_reg].calculated = 
						(__system_pll_fin * sdiv * (j==1 ? j:(j-1)));
					calc[sdiv_reg].calculated /= sout;
					calc[sdiv_reg].intdiv = (j==1 ? j:(j-1));;
					calc[sdiv_reg].sout = sout_reg;
					calc[sdiv_reg].difference = 
						fout_wanted - calc[sdiv_reg].calculated;
					break;
				}
			}
		}
	}
	min_sdiv = 0;
	min_diff = calc[min_sdiv].difference;
	/* the index into the arrays is the register value of sdiv */
	for (sdiv=0; sdiv < ARRAY_SIZE(pll_sdiv_values); sdiv++) {
		if (calc[sdiv].difference < min_diff) {
			min_diff = calc[sdiv].difference;
			min_sdiv = sdiv;
		}
	}

	mutex_lock(&pll->lock);

	if (update_shadow_reg(pll)) {
		ret = -EIO;
		goto out;
	}
	CLOCK_WRITE_FIELD(PLL, SDIV, pll->shadow_reg, min_sdiv);
	CLOCK_WRITE_FIELD(PLL, INTPROG, pll->shadow_reg, calc[min_sdiv].intdiv);
	CLOCK_WRITE_FIELD(PLL, SOUT, pll->shadow_reg, calc[min_sdiv].sout);
	CLOCK_WRITE_FIELD(PLL, ENABLE, pll->shadow_reg, PLL_ENABLE);
	CLOCK_WRITE_FIELD(PLL, PD, pll->shadow_reg, 0);

	/* enable bypass */
	CLOCK_WRITE_FIELD(PLL, BYPASS, pll->shadow_reg, PLL_BYPASS_ENABLE);
	ret = mobi_qcc_write(QCC_BID_CHIPCTL, pll->qcc_addr, pll->shadow_reg, 4);
	if (ret < 0)
		goto out;

	/* update pll */
	ret = mobi_qcc_write(QCC_BID_CHIPCTL, pll->qcc_addr, pll->shadow_reg, 4);
	if (ret < 0)
		goto out;

	/* disable bypass */
	CLOCK_WRITE_FIELD(PLL, BYPASS, pll->shadow_reg, PLL_BYPASS_DISABLE);
	ret = mobi_qcc_write(QCC_BID_CHIPCTL, pll->qcc_addr, pll->shadow_reg, 4);
	if (ret < 0)
		goto out;

	/* re-write the pll */
	ret = mobi_qcc_write(QCC_BID_CHIPCTL, pll->qcc_addr, pll->shadow_reg, 4);
	if (ret < 0)
		goto out;

#ifndef CONFIG_MERLIN_IS_FPGA
	/* poll for lock bit */
	do {
		ret = mobi_qcc_read(QCC_BID_CHIPCTL, 
				pll->qcc_addr, &pll->shadow_reg, 4);
	} while (((pll->shadow_reg & MOBI_CLK_PLL_LOCK_MASK) != 1)  || (ret == 0));
#endif

out:
	if (ret < 0)
		printk(KERN_ERR "Unable to reprogram PLL rate\n");
	else
		handle_callbacks(pll);

	mutex_unlock(&pll->lock);

	return ret;
}

/* 
*  determine the scaler values to output a frequency
* equation for scaler output frequency is the following:
* fout = fin/(pre * div)
 */
static int32_t 
set_scaler_freq(struct clk *clk, unsigned long fout_wanted)
{
	int32_t pre = 0, min_pre;
	int32_t i, div, ret = 0;
	unsigned long scaler_fin = 0;
	unsigned long min_diff = 0;
	struct clk_calc_values *calc = NULL;

	calc = (struct clk_calc_values*) 
		kmalloc(ARRAY_SIZE(pll_sdiv_values)*(sizeof(struct clk_calc_values)), 
				GFP_KERNEL);
	if (calc == NULL) 
		return -ENOMEM;

	dprintk("%s: clk_id %d, requested freq %lu\n", 
			__func__, clk->id, fout_wanted);
	// get the pll freq driving the scaler, then calc pre and div
	switch (clk->id) {
		case CLOCK_ID_SCALER0:
			ret = get_pll_freq(CLOCK_ID_PLL0, &scaler_fin);
			break;
		case CLOCK_ID_SCALER1:
			ret = get_pll_freq(CLOCK_ID_PLL1, &scaler_fin);
			break;
		case CLOCK_ID_SCALER2:
			ret = get_pll_freq(CLOCK_ID_PLL2, &scaler_fin);
			break;
		case CLOCK_ID_SCALER3:
			ret = get_pll_freq(CLOCK_ID_PLL3, &scaler_fin);
			break;
		default:
			break;
	}

	if (ret)
		return ret;

	/* 
	*  PRE can be 1,2,3,4 and 8.  Fm2 is the input freq to DIV and it must 
	*  be less than 750 Mhz, so choose pre to meet the criteria
	 */
	for (i=0; i < ARRAY_SIZE(scaler_pre_values); i++) {
		pre = scaler_pre_values[i];
		if (scaler_fin/pre < SCALER_MAX_PRE_OUT_RATE) {
			for (div=SCALER_DIV_MIN; div <= SCALER_DIV_MAX; div++) {
				/* On Scaler1 the second stage is not used and should be set to SCALER_DIV_MIN */
				if((clk->id==CLOCK_ID_SCALER1) && (div!=SCALER_DIV_MIN))
					break;
				/* the test is to find the first value lower than the fout, since
				*  div is incrementing, the quotient would decrement so that's why
				*  we use the first one found
				 */
				if ((scaler_fin/(pre*div)) <= fout_wanted) {
					calc[i].calculated = (scaler_fin/(pre*div));
					calc[i].intdiv = div;
					calc[i].difference = fout_wanted - calc[i].calculated;
					break;
				}
			}
		}
		else { 
			/* need some way to mark entry unusable */
			calc[i].intdiv = -1;
		}
	}
	min_pre = 0;
	while (calc[min_pre].intdiv == -1)
		min_pre++;

	min_diff = calc[min_pre].difference;
	/* the index into the arrays is the register value of pre */
	for (pre=0; pre < ARRAY_SIZE(scaler_pre_values); pre++) {
		if(calc[pre].intdiv == -1)
			continue;
		if (calc[pre].difference < min_diff) {
			min_diff = calc[pre].difference;
			min_pre = pre;
		}
	}

	mutex_lock(&clk->lock);
	if (update_shadow_reg(clk)) {
		ret = -EIO;
		goto out;
	}

	/* write twice, toggling WE & WRPD per clock document */
	CLOCK_WRITE_FIELD(SCALER, PRE, clk->shadow_reg, min_pre);
	CLOCK_WRITE_FIELD(SCALER, DIV, clk->shadow_reg, calc[min_pre].intdiv);
	CLOCK_WRITE_FIELD(SCALER, WE, clk->shadow_reg, 0);
	CLOCK_WRITE_FIELD(SCALER, WRPD, clk->shadow_reg, 0);

	if ((ret = mobi_qcc_write(QCC_BID_CHIPCTL, 
					clk->qcc_addr, clk->shadow_reg, 4)) != 0) {
		goto out;
	}
	else {
		CLOCK_WRITE_FIELD(SCALER, WE, clk->shadow_reg, 1);
		CLOCK_WRITE_FIELD(SCALER, WRPD, clk->shadow_reg, 1);

		if ((ret = mobi_qcc_write(QCC_BID_CHIPCTL, 
						clk->qcc_addr, clk->shadow_reg, 4)) != 0)
			goto out;
		else
			handle_callbacks(clk);
	}

out:

	if (ret < 0)
		printk(KERN_ERR "Unable to set new scaler frequency for clk_id: %d\n", 
				clk->id);

	mutex_unlock(&clk->lock);

	return ret;
}

/* handle the control for core clocks, we only need to deal with the QMM
*  clock, the codec will handle the others
 */
/*
* > > #-------------------
> > # N/R dividers(core)
> > #-------------------
> > Can be updated with one write. Just write the new values in.
> >
*
 */
static int32_t set_core_freq(struct clk *clk, unsigned long fout)
{
	uint32_t ndiv, rdiv;
	int32_t ret = 0;
	unsigned long fin;

	dprintk("%s: clk_id %d, requested freq %lu\n", 
			__func__, clk->id, fout);
	switch(clk->id) {
		case CLOCK_ID_QMM: 
			/* all NR clocks are driven by scaler0 fin/4 */
			if ((ret = get_pll_freq(CLOCK_ID_PLL0, &fin)) == 0) {
				/* 
				* HL dividers are driven by the HF4PH signal on the scaler
				* so we divide the scaler Fout by 4 to get this freq
				 */
				fin /= 4;
				nr_calculate_div(fout, fin, &ndiv, &rdiv, MOBI_CLK_CORE_RDIV_MASK);
			}
			break;
		default:
			break;
	}

	if (ret == 0) {
		mutex_lock(&clk->lock);
		if (update_shadow_reg(clk)) {
			mutex_unlock(&clk->lock);
			return -EIO;
		}

		CLOCK_WRITE_FIELD(CORE, NDIV, clk->shadow_reg, ndiv);
		CLOCK_WRITE_FIELD(CORE, RDIV, clk->shadow_reg, rdiv);
		/* and make sure it's on! */
		CLOCK_WRITE_FIELD(CORE, ENABLE, clk->shadow_reg, 1);
		ret = mobi_qcc_write(QCC_BID_CHIPCTL,
				clk->qcc_addr, clk->shadow_reg, 4);

		/* the write may have returned an error for unknown reasons
		*  but we probably should let people know in case something
		*  did change, that way they can check!
		 */
		handle_callbacks(clk);

		mutex_unlock(&clk->lock);
	}

	return ret;
}


/* this function handles the control for clocks driven by H/L dividers */
/*
*  from mail msg [Fwd: Re: N/R and H/L divider programming
*  on Thu, 12 Apr 2007 17:57:04 -0700
> > #-------------
> > # H/L dividers
> > #-------------
> > H/L dividers should use a four access re-program sequence.
> > H0 and L0 are defined as the old values.
> > H1 and L1 are defined as the new values.
> > 
> > 1) read H/L divider to get H0, L0.
> > 2) write ClkEn=0, H0, L0
> > 3) write ClkEn=0, H1, L1
> > 4) write ClkEn=1, H1, L1
> > 
*
 */
static int32_t set_hl_freq(struct clk *clk, unsigned long fout)
{
	int32_t ret = 0;
	uint32_t hdiv, ldiv; 
	unsigned long fin;

	mutex_lock(&clk->lock);

	if (update_shadow_reg(clk)) {
		mutex_unlock(&clk->lock);
		return -EIO;
	}

	/* get the freq of the driving clock and then the current 
	*  values of the clock */
	if ((ret = get_pll_freq(CLOCK_ID_PLL0, &fin)) == 0) 
		fin /= 4;

	if (ret != 0 || fin == 0)
		goto out;

	switch (clk->id) {
		case CLOCK_ID_UARTDBG:
		case CLOCK_ID_UART0:
		case CLOCK_ID_UART1:
			/* size, 2 shifted by (num bits set minus 1) */
			hl_calculate_div(fout, fin, &hdiv, &ldiv, MOBI_CLK_HL6_HDIV_MASK);

			CLOCK_WRITE_FIELD(HL6, ENABLE, clk->shadow_reg, 0);
			ret = mobi_qcc_write(QCC_BID_CHIPCTL, 
					clk->qcc_addr, clk->shadow_reg, 4);

			if (ret == 0) {
				CLOCK_WRITE_FIELD(HL6, HDIV, clk->shadow_reg, hdiv);
				CLOCK_WRITE_FIELD(HL6, LDIV, clk->shadow_reg, ldiv);
				ret = mobi_qcc_write(QCC_BID_CHIPCTL, 
						clk->qcc_addr, clk->shadow_reg, 4);
				if (ret == 0) 
					CLOCK_WRITE_FIELD(HL6, ENABLE, clk->shadow_reg, 1);
			}
			break;
		case CLOCK_ID_I2C0:
		case CLOCK_ID_I2C1:
		case CLOCK_ID_PLL2HL:
		case CLOCK_ID_PLL3HL:
			hl_calculate_div(fout, fin, &hdiv, &ldiv, MOBI_CLK_HL8_HDIV_MASK);

			CLOCK_WRITE_FIELD(HL8, ENABLE, clk->shadow_reg, 0);
			ret = mobi_qcc_write(QCC_BID_CHIPCTL, 
					clk->qcc_addr, clk->shadow_reg, 4);

			if (ret == 0) {
				CLOCK_WRITE_FIELD(HL8, HDIV, clk->shadow_reg, hdiv);
				CLOCK_WRITE_FIELD(HL8, LDIV, clk->shadow_reg, ldiv);
				ret = mobi_qcc_write(QCC_BID_CHIPCTL, 
						clk->qcc_addr, clk->shadow_reg, 4);
				if (ret == 0) 
					CLOCK_WRITE_FIELD(HL8, ENABLE, clk->shadow_reg, 1);
			}
			break;
		case CLOCK_ID_TIMER:
		case CLOCK_ID_SDMMC:
		case CLOCK_ID_SSI0:
		case CLOCK_ID_SSI1:
		case CLOCK_ID_BS: 
			hl_calculate_div(fout, fin, &hdiv, &ldiv, MOBI_CLK_HL9_HDIV_MASK);

			CLOCK_WRITE_FIELD(HL9, ENABLE, clk->shadow_reg, 0);
			ret = mobi_qcc_write(QCC_BID_CHIPCTL, 
					clk->qcc_addr, clk->shadow_reg, 4);
			if (ret == 0) {
				CLOCK_WRITE_FIELD(HL9, HDIV, clk->shadow_reg, hdiv);
				CLOCK_WRITE_FIELD(HL9, LDIV, clk->shadow_reg, ldiv);
				ret = mobi_qcc_write(QCC_BID_CHIPCTL, 
						clk->qcc_addr, clk->shadow_reg, 4);
				if (ret == 0) 
					CLOCK_WRITE_FIELD(HL9, ENABLE, clk->shadow_reg, 1);
			}
			break;
		default:
			break;
	} 

	/* now enable the clock */
	if (ret == 0) {
		ret = mobi_qcc_write(QCC_BID_CHIPCTL,
				clk->qcc_addr, clk->shadow_reg, 4);
	}

out:
	/* either something changed or broke so let everyone know */
	handle_callbacks(clk);

	dprintk("##### %s(%d) requested rate %lu, got rate %lu\n", 
			__func__, clk->id, fout, (fin/((hdiv+1) + (ldiv+1))));

	mutex_unlock(&clk->lock);

	return ret;
}

static int32_t handle_callbacks(struct clk *clk)
{
	struct list_head *lh;
	struct clk_callback *cb;

	list_for_each(lh, &clk->callback_list) {
		cb = list_entry(lh, struct clk_callback, node); 
		(*cb->callback_func)(cb->callback_data);
	}
	return 0;
}

/**
* \brief mobi_clock_register_callback:
* 	Register a user's callback function which is called when the
* 	status of a clock is changed
*
* \param clk_id  : id of the clock that an app is interested in.
* \param *func   : function pointer to callback
*
* \retval EINVAL - if invalid clk_id is given
* \retval Zero   - upon success
*
* \remark
* 	Any application can call this function and register a callback
* against a clock that it is interested in.  This provides a mechnism 
* for an app to be notified if a clock is changed underneath it.
*
 */
int32_t mobi_clock_register_callback(enum clock_id clk_id, 
		void (*func)(void *), void *data)
{
	struct clk *clk = NULL;
	struct clk *dep_clk = NULL;
	struct clk_callback *cb = NULL;
	int i;

	clk = clock_get_by_id(clk_id);
	if (clk == NULL)
		return -EINVAL;

	cb = (struct clk_callback *) 
		kmalloc(sizeof(struct clk_callback), GFP_KERNEL);

	cb->callback_func = func;
	cb->callback_data = data;

	mutex_lock(&clk->lock);
	for (i=0;i < CLOCK_ID_MAX;i++) {
		if (clk->dependencies[i] != 0) {
			dep_clk = clock_get_by_id(clk->dependencies[i]);
			mutex_lock(&dep_clk->lock);
			list_add(&cb->node, &dep_clk->callback_list);
			mutex_unlock(&dep_clk->lock);
		}
		else {
			break;
		}
	}
	list_add(&cb->node, &clk->callback_list);
	mutex_unlock(&clk->lock);

	return 0;
}

/**
* \brief mobi_clock_unregister_callback:
* 	Remove a callback from a clocks callback list.
*
* \param clk_id  : id of the clock that an app is interested in.
* \param *func   : function pointer to callback to be removed
*
* \retval EINVAL - if invalid clk_id is given
* \retval Zero   - upon success
*
 */
int32_t mobi_clock_unregister_callback(enum clock_id clk_id, void (*func))
{
	struct clk *clk = NULL;
	struct clk *dep_clk = NULL;
	int i;
	struct list_head *lh;
	struct clk_callback *cb;

	clk = clock_get_by_id(clk_id);
	if (clk == NULL)
		return -EINVAL;

	mutex_lock(&clk->lock);
	/* delete the callback entries in the controlling clocks dependency list */
	for (i=0;i < CLOCK_ID_MAX;i++) {
		if (clk->dependencies[i] != 0) { // XXX don't really like this array...
			dep_clk = clock_get_by_id(clk->dependencies[i]);
			mutex_lock(&dep_clk->lock);
			list_for_each(lh, &dep_clk->callback_list) {
				cb = list_entry(lh, struct clk_callback, node); 
				if (cb->callback_func == func) {
					list_del(&cb->node);
					kfree(cb);
					break;
				}
			}
			mutex_unlock(&dep_clk->lock);
		}
		else {
			break;
		}
	}
	/* and then delete the entry in the clock for the current driver */
	list_for_each(lh, &clk->callback_list) {
		cb = list_entry(lh, struct clk_callback, node); 
		if (cb->callback_func == func) {
			list_del(&cb->node);
			kfree(cb);
			break;
		}
	}
	mutex_unlock(&clk->lock);

	return 0;
}

/**
* \brief mobi_clock_get_rate:
* 	Get the current output frequency of a clock
*
* \param clk_id  : id of the clock that an app is interested in.
*
* \retval zero   - if unable to determine frequency or invalid clk_id is given
* \retval "freq" - a number greater than zero that is the 
* 		    frequency of the clock.
*
* \remark
* 	Just to be clear, a return value of 0 is an indication
* that something when wrong when trying to determing the 
* frequency of the clock.
 */
unsigned long mobi_clock_get_rate(enum clock_id clk_id)
{
	struct clk *clk = NULL;
	unsigned long freq = 0;
	int ret = 0;

	clk = clock_get_by_id(clk_id);
	if (clk == NULL)
		return 0;

	switch (clk_id) {
		case CLOCK_ID_PLL0:
		case CLOCK_ID_PLL1:
		case CLOCK_ID_PLL2:
		case CLOCK_ID_PLL3:
			ret = get_pll_freq(clk_id, &freq);
			break;
		case CLOCK_ID_ARM:
			ret = get_scaler_freq(CLOCK_ID_SCALER0, &freq);
			break;
		case CLOCK_ID_AHB:
			ret = get_scaler_freq(CLOCK_ID_SCALER0, &freq);
			freq = freq >> 1;
			break;
		case CLOCK_ID_APB:
			ret = get_scaler_freq(CLOCK_ID_SCALER0, &freq);
			freq = freq >> 3;
			break;
		case CLOCK_ID_SCALER0:
		case CLOCK_ID_SCALER1:
		case CLOCK_ID_SCALER2:
		case CLOCK_ID_SCALER3:
			ret = get_scaler_freq(clk_id, &freq);
			break;
		case CLOCK_ID_QMM:
			ret = get_nr_freq(clk, &freq);
			break;
		case CLOCK_ID_VIN0:
		case CLOCK_ID_VIN1:
		case CLOCK_ID_VOUT0:
			ret = get_vid_freq(clk, &freq);
			break; 
		case CLOCK_ID_AUD0:
		case CLOCK_ID_AUD1:
			ret = get_aud_freq(clk, &freq);
			break; 
		case CLOCK_ID_UARTDBG:
		case CLOCK_ID_UART0:
		case CLOCK_ID_UART1:
		case CLOCK_ID_I2C0:
		case CLOCK_ID_I2C1:
		case CLOCK_ID_PLL2HL:
		case CLOCK_ID_PLL3HL:
		case CLOCK_ID_TIMER:
		case CLOCK_ID_SDMMC:
		case CLOCK_ID_SSI0:
		case CLOCK_ID_SSI1:
		case CLOCK_ID_BS:
			ret = get_hl_freq(clk, &freq);
			break;
		case CLOCK_ID_MEMBUS:
			ret = get_scaler_freq(CLOCK_ID_SCALER1, &freq);
			freq>>=2;
			break;
		case CLOCK_ID_CORE:
			ret = get_pll_freq(CLOCK_ID_PLL0, &freq);
			freq>>=2;
			break;
		default:
			return 0;
	} 

	if (ret != 0)
		printk(KERN_ERR "%s:  Error when retrieving rate for clock %d: %d\n", 
				__func__, clk->id, ret);

	return freq;
}

/**
* \brief mobi_clock_set_rate
* 	Set the frequency of a clock.
*
* \param clk_id  : id of the clock that an app is interested in.
* \param freq    : new frequency for the the clock
*
* \retval -EINVAL  - if invalid clk_id is given
* \retval -EIO	   - if an error was encounted programming new freqency
* \return Zero	   - upon success
*
* \remark
* 	Passing in a frequency of '0' will disable the clock.
 */
int32_t mobi_clock_set_rate(enum clock_id clk_id, unsigned long freq)
{
	struct clk *clk = NULL;
	int ret = 0;
	struct clk *tmp_clk;

	clk = clock_get_by_id(clk_id);
	if (clk == NULL)
		return -EINVAL;

	if (freq == 0) {
		/* short way back, disable will handle callbacks */
		clock_disable(clk);
		return 0;
	}

	switch (clk_id) {
		case CLOCK_ID_ARM:
			tmp_clk = clock_get_by_id(CLOCK_ID_SCALER0);
			ret = set_scaler_freq(tmp_clk, freq);
			break;
		case CLOCK_ID_AHB:
		case CLOCK_ID_APB: 
			printk(KERN_ERR "AHB and APB clocks cannot be changed\n");
			return -EINVAL;
			break;
		case CLOCK_ID_PLL0:
		case CLOCK_ID_PLL1:
		case CLOCK_ID_PLL2:
		case CLOCK_ID_PLL3:
			ret = set_pll_freq(clk, freq);
			break;
		case CLOCK_ID_SCALER0:
		case CLOCK_ID_SCALER1:
		case CLOCK_ID_SCALER2:
		case CLOCK_ID_SCALER3:
			ret = set_scaler_freq(clk, freq);
			break;
		case CLOCK_ID_QMM:
			ret = set_core_freq(clk, freq);
			break;
			/*
		case CLOCK_ID_VIN0:
		case CLOCK_ID_VIN1:
		case CLOCK_ID_VOUT0:
			ret = set_vid_freq(clk, freq, flags);
			break; 
		case CLOCK_ID_AUD0:
		case CLOCK_ID_AUD1:
			ret = set_aud_freq(clk, freq, flags);
			break; 
			*/
		case CLOCK_ID_UARTDBG:
		case CLOCK_ID_UART0:
		case CLOCK_ID_UART1:
		case CLOCK_ID_I2C0:
		case CLOCK_ID_I2C1:
		case CLOCK_ID_PLL2HL:
		case CLOCK_ID_PLL3HL:
		case CLOCK_ID_TIMER:
		case CLOCK_ID_SDMMC:
		case CLOCK_ID_SSI0:
		case CLOCK_ID_SSI1:
		case CLOCK_ID_BS:
			ret =  set_hl_freq(clk, freq);
			break;
		case CLOCK_ID_MEMBUS:
			ret =  set_scaler_freq(clock_get_by_id(CLOCK_ID_SCALER1), freq<<2);
			break;
		case CLOCK_ID_CORE:
			ret =  set_pll_freq(clock_get_by_id(CLOCK_ID_PLL0), freq<<2);
			break;
		default:
			printk(KERN_ERR "Unsupported operation on clock_id 0x%x\n", 
					clk->id);
			return -EINVAL;
	} 

	/* if we get an error back it's likely we had some i/o error
	*  with the qcc
	 */
	if (ret != 0)
		ret = -EIO;

	return ret;
}

/*
* 	linux/include/linux/clk.h seems to define an API for a couple 
*  of clock functions, clk_get, clk_enable, clk_disable, clk_get_rate, 
*  clk_put.  It's not clear really how much these have to be supported? 
*  lots of other clock files do nothing, but we'll implement them since 
*  it is not to difficult
 */
void clk_put(struct clk *clk)
{
	module_put(clk->owner);
}

unsigned long clk_get_rate(struct clk *clk)
{
	return mobi_clock_get_rate(clk->id);
}

int clk_enable(struct clk *clk)
{
	/* return 0 if the clock has a non-zero rate */
	if (mobi_clock_get_rate(clk->id)) 
		return 0;
	else {
		printk(KERN_WARNING "Unable to enable clock %s, "
				"verify that rate has been set\n", clk->name);
		return -1;
	}
}

void clk_disable(struct clk *clk)
{
	mobi_clock_set_rate(clk->id, 0);
	return;
}

int clk_set_rate(struct clk *clk, unsigned long rate)
{
	int ret = -EIO;
	ret = mobi_clock_set_rate(clk->id, rate);
	return ret;
}

struct clk *clk_get(struct device *dev, const char *id)
{
	struct clk *p, *clk = ERR_PTR(-ENOENT);

	mutex_lock(&clocks_mutex);
	list_for_each_entry(p, &clocks, node) {
		if (strcmp(id, p->name) == 0 && try_module_get(p->owner)) {
			clk = p;
			break;
		}
	}
	mutex_unlock(&clocks_mutex);
	return clk;
}

static struct clk arm_clk = {
	.name	= "arm",
	.id = CLOCK_ID_ARM,
	.qcc_addr = 0x0,
	.callback_list  = LIST_HEAD_INIT(arm_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
		CLOCK_ID_SCALER0,
		CLOCK_ID_CORE,
	}
};

static struct clk ahb_clk = {
	.name	= "ahb",
	.id = CLOCK_ID_AHB,
	.qcc_addr = 0x0,
	.callback_list  = LIST_HEAD_INIT(ahb_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
		CLOCK_ID_SCALER0,
	}
};

static struct clk apb_clk = {
	.name	= "apb",
	.id = CLOCK_ID_APB,
	.qcc_addr = 0x0,
	.callback_list  = LIST_HEAD_INIT(apb_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
		CLOCK_ID_SCALER0,
	}
};

static struct clk pll0_clk = {
	.name	= "pll0",
	.id = CLOCK_ID_PLL0,
	.qcc_addr = QCC_CHIPCTL_PLL0_CONTROL,
	.enable_mask = MOBI_CLK_PLL_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(pll0_clk.callback_list),
};

static struct clk pll1_clk = {
	.name	= "pll1",
	.id = CLOCK_ID_PLL1,
	.qcc_addr = QCC_CHIPCTL_PLL1_CONTROL,
	.enable_mask = MOBI_CLK_PLL_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(pll1_clk.callback_list),
};

static struct clk pll2_clk = {
	.name	= "pll2",
	.id = CLOCK_ID_PLL2,
	.qcc_addr = QCC_CHIPCTL_PLL2_CONTROL,
	.enable_mask = MOBI_CLK_PLL_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(pll2_clk.callback_list),
};

static struct clk pll3_clk = {
	.name	= "pll3",
	.id = CLOCK_ID_PLL3,
	.qcc_addr = QCC_CHIPCTL_PLL3_CONTROL,
	.enable_mask = MOBI_CLK_PLL_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(pll3_clk.callback_list),
};


static struct clk scaler0_clk = {
	.name	= "scaler0",
	.id = CLOCK_ID_SCALER0,
	.qcc_addr = QCC_CHIPCTL_SCALER0_CONTROL, 
	.callback_list  = LIST_HEAD_INIT(scaler0_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
	}
};

static struct clk scaler1_clk = {
	.name	= "scaler1",
	.id = CLOCK_ID_SCALER1,
	.qcc_addr = QCC_CHIPCTL_SCALER1_CONTROL,
	.callback_list  = LIST_HEAD_INIT(scaler1_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL1,
	}
};

static struct clk scaler2_clk = {
	.name	= "scaler2",
	.id = CLOCK_ID_SCALER2,
	.qcc_addr = QCC_CHIPCTL_SCALER2_CONTROL,
	.callback_list  = LIST_HEAD_INIT(scaler2_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL2,
	}
};

static struct clk scaler3_clk = {
	.name	= "scaler3",
	.id = CLOCK_ID_SCALER3,
	.qcc_addr = QCC_CHIPCTL_SCALER3_CONTROL,
	.callback_list  = LIST_HEAD_INIT(scaler3_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL3,
	}
};

static struct clk qmm_clk = {
	.name	= "qmm",
	.id = CLOCK_ID_QMM,
	.qcc_addr = QCC_CHIPCTL_QMMCLK_CONTROL,
	.enable_mask = MOBI_CLK_CORE_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(qmm_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
		CLOCK_ID_SCALER0,
	}
};

static struct clk vin0_clk = {
	.name	= "vin0",
	.id = CLOCK_ID_VIN0,
	.qcc_addr = QCC_CHIPCTL_V0PIXCLK_CONTROL,
	.enable_mask = MOBI_CLK_VIDCTRL_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(vin0_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
		CLOCK_ID_PLL2,
		CLOCK_ID_SCALER2,
		CLOCK_ID_PLL3,
		CLOCK_ID_SCALER3,
		CLOCK_ID_PLL2HL,
		CLOCK_ID_PLL3HL,
	}
};

static struct clk vin1_clk = {
	.name	= "vin1",
	.id = CLOCK_ID_VIN1,
	.qcc_addr = QCC_CHIPCTL_V1PIXCLK_CONTROL,
	.enable_mask = MOBI_CLK_VIDCTRL_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(vin1_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
		CLOCK_ID_PLL2,
		CLOCK_ID_SCALER2,
		CLOCK_ID_PLL3,
		CLOCK_ID_SCALER3,
		CLOCK_ID_PLL2HL,
		CLOCK_ID_PLL3HL,
	}
};

static struct clk vout0_clk = {
	.name	= "vout0",
	.id = CLOCK_ID_VOUT0,
	.qcc_addr = QCC_CHIPCTL_V2PIXCLK_CONTROL,
	.enable_mask = MOBI_CLK_VIDCTRL_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(vout0_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
		CLOCK_ID_PLL2,
		CLOCK_ID_SCALER2,
		CLOCK_ID_PLL3,
		CLOCK_ID_SCALER3,
		CLOCK_ID_PLL2HL,
		CLOCK_ID_PLL3HL,
	}
};

static struct clk aud0_clk = {
	.name	= "aud0",
	.id = CLOCK_ID_AUD0,
	.qcc_addr = QCC_CHIPCTL_AUD0CLK_CONTROL,
	.enable_mask = MOBI_CLK_AUDCTRL_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(aud0_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
		CLOCK_ID_PLL2,
		CLOCK_ID_SCALER2,
		CLOCK_ID_PLL3,
		CLOCK_ID_SCALER3,
		CLOCK_ID_PLL2HL,
		CLOCK_ID_PLL3HL,
	}
};

static struct clk aud1_clk = {
	.name	= "aud1",
	.id = CLOCK_ID_AUD1,
	.qcc_addr = QCC_CHIPCTL_AUD1CLK_CONTROL,
	.enable_mask = MOBI_CLK_AUDCTRL_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(aud1_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
		CLOCK_ID_PLL2,
		CLOCK_ID_SCALER2,
		CLOCK_ID_PLL3,
		CLOCK_ID_SCALER3,
		CLOCK_ID_PLL2HL,
		CLOCK_ID_PLL3HL,
	}
};

static struct clk dbguart_clk = {
	.name	= "uartdbg",
	.id = CLOCK_ID_UARTDBG,
	.qcc_addr = QCC_CHIPCTL_UARTDBGCLK_CONTROL,
	.enable_mask = MOBI_CLK_HL6_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(dbguart_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
	}
};

static struct clk uart0_clk = {
	.name	= "uart0",
	.id = CLOCK_ID_UART0,
	.qcc_addr = QCC_CHIPCTL_UART0CLK_CONTROL,
	.enable_mask = MOBI_CLK_HL6_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(uart0_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
	}
};

static struct clk uart1_clk = {
	.name	= "uart1",
	.id = CLOCK_ID_UART1,
	.qcc_addr = QCC_CHIPCTL_UART1CLK_CONTROL,
	.enable_mask = MOBI_CLK_HL6_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(uart1_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
	}
};

static struct clk i2c0_clk = {
	.name	= "i2c0",
	.id = CLOCK_ID_I2C0,
	.qcc_addr = QCC_CHIPCTL_I2C0CLK_CONTROL,
	.enable_mask = MOBI_CLK_HL8_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(i2c0_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
	}
};

static struct clk i2c1_clk = {
	.name	= "i2c1",
	.id = CLOCK_ID_I2C1,
	.qcc_addr = QCC_CHIPCTL_I2C1CLK_CONTROL,
	.enable_mask = MOBI_CLK_HL8_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(i2c1_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
	}
};

static struct clk pll2hl_clk = {
	.name	= "pll2hl",
	.id = CLOCK_ID_PLL2HL,
	.qcc_addr = QCC_CHIPCTL_PLL2CLK_CONTROL,
	.enable_mask = MOBI_CLK_HL8_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(pll2hl_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
	}
};

static struct clk pll3hl_clk = {
	.name	= "pll3hl",
	.id = CLOCK_ID_PLL3HL,
	.qcc_addr = QCC_CHIPCTL_PLL3CLK_CONTROL,
	.enable_mask = MOBI_CLK_HL8_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(pll3hl_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
	}
};

static struct clk timer_clk = {
	.name	= "timer",
	.id = CLOCK_ID_TIMER,
	.qcc_addr = QCC_CHIPCTL_TIMERCLK_CONTROL,
	.enable_mask = MOBI_CLK_HL9_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(timer_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
	}
};

static struct clk sdmmc_clk = {
	.name	= "sdmmc",
	.id = CLOCK_ID_SDMMC,
	.qcc_addr = QCC_CHIPCTL_SDMMCCLK_CONTROL,
	.enable_mask = MOBI_CLK_HL9_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(sdmmc_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
	}
};

static struct clk ssi0_clk = {
	.name	= "ssi0",
	.id = CLOCK_ID_SSI0,
	.qcc_addr = QCC_CHIPCTL_SSI0CLK_CONTROL,
	.enable_mask = MOBI_CLK_HL9_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(ssi0_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
	}
};

static struct clk ssi1_clk = {
	.name	= "ssi1",
	.id = CLOCK_ID_SSI1,
	.qcc_addr = QCC_CHIPCTL_SSI1CLK_CONTROL,
	.enable_mask = MOBI_CLK_HL9_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(ssi1_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
	}
};

static struct clk bs_clk = {
	.name	= "bs",
	.id = CLOCK_ID_BS,
	.qcc_addr = QCC_CHIPCTL_BSCLK_CONTROL,
	.enable_mask = MOBI_CLK_HL9_ENABLE_MASK,
	.callback_list  = LIST_HEAD_INIT(bs_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
	}
};

static struct clk membus_clk = {
	.name	= "membus",
	.id = CLOCK_ID_MEMBUS,
	.qcc_addr = 0x0,
	.callback_list  = LIST_HEAD_INIT(membus_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL1,
		CLOCK_ID_SCALER1,
	}
};

static struct clk core_clk = {
	.name	= "core",
	.id = CLOCK_ID_CORE,
	.qcc_addr = 0x0,
	.callback_list  = LIST_HEAD_INIT(core_clk.callback_list),
	.dependencies = {
		CLOCK_ID_PLL0,
		CLOCK_ID_SCALER0,
		CLOCK_ID_ARM,
	}
};

#undef MOBI_CLOCK_TESTING
#ifdef MOBI_CLOCK_TESTING
static void pll_tst(enum clock_id clk_id)
{
	struct clk *clk = clock_get_by_id(clk_id);
	unsigned long freq;

	get_pll_freq(clk_id, &freq);
	printk("\n%s: Query of %s returned a freq = %lu\n", 
			__func__, clk->name, freq);

	printk("%s settings are\n"
			"\tpd      = %lu\n"
			"\treset   = %lu\n"
			"\tlock    = %lu\n"
			"\tintprog = %lu\n"
			"\tsout    = %lu(%d)\n"
			"\tsdiv    = %lu(%d)\n"
			"\tvcosel  = %lu\n"
			"\tclken   = %lu\n", 
			clk->name,
			CLOCK_READ_FIELD(PLL, PD, clk->shadow_reg),
			CLOCK_READ_FIELD(PLL, RESET, clk->shadow_reg),
			CLOCK_READ_FIELD(PLL, LOCK, clk->shadow_reg),
			CLOCK_READ_FIELD(PLL, INTPROG, clk->shadow_reg),
			CLOCK_READ_FIELD(PLL, SOUT, clk->shadow_reg),
			pll_sout_values[CLOCK_READ_FIELD(PLL, SOUT, clk->shadow_reg)],
			CLOCK_READ_FIELD(PLL, SDIV, clk->shadow_reg),
			pll_sdiv_values[CLOCK_READ_FIELD(PLL, SDIV, clk->shadow_reg)],
			CLOCK_READ_FIELD(PLL, VCOSEL, clk->shadow_reg),
			CLOCK_READ_FIELD(PLL, ENABLE, clk->shadow_reg));
	printk("\n");
}

static void scaler_tst(enum clock_id clk_id)
{
	struct clk *clk = clock_get_by_id(clk_id);
	unsigned long freq;

	get_scaler_freq(clk_id, &freq);
	printk("\n%s: Query of %s returned a freq = %lu\n", 
			__func__, clk->name, freq);

	printk("%s settings are\n"
			"\tdiv    = %lu\n"
			"\treset2 = %lu\n"
			"\tbypass = %lu\n"
			"\treset1 = %lu\n"
			"\tpre    = %lu(%d)\n",
			clk->name,
			CLOCK_READ_FIELD(SCALER, DIV, clk->shadow_reg),
			CLOCK_READ_FIELD(SCALER, RESET2, clk->shadow_reg),
			CLOCK_READ_FIELD(SCALER, BYPASS, clk->shadow_reg),
			CLOCK_READ_FIELD(SCALER, RESET1, clk->shadow_reg),
			CLOCK_READ_FIELD(SCALER, PRE, clk->shadow_reg),
			scaler_pre_values[CLOCK_READ_FIELD(SCALER, PRE, clk->shadow_reg)]);
	printk("\n");
}


void clk_internal_test(void)
{
	pll_tst(CLOCK_ID_PLL0);
	scaler_tst(CLOCK_ID_SCALER0);
}
EXPORT_SYMBOL(clk_internal_test);

#endif // MOBI_TESTING

#ifdef CONFIG_PROC_FS
static int clock_rate_proc_rd(char *buf, char **start, 
		off_t offset, int count, int *eof, void *data)
{   
	uint32_t clk_id = (uint32_t)data;
	int len = 0;

	len += sprintf(buf+len,"%lu\n", mobi_clock_get_rate(clk_id));
	*eof=1;

	return len;
}
#endif

static void clk_add_procfs(struct clk *clk)
{
#ifdef CONFIG_PROC_FS
	struct proc_dir_entry *pentry;
	char string[50]; 

	sprintf(string,"driver/clock/%s", clk->name);
	proc_mkdir(string, NULL);
	sprintf(string,"driver/clock/%s/rate", clk->name);
	pentry = create_proc_entry(string, 
			S_IRUSR | S_IRGRP | S_IROTH, 
			NULL); 
	if (pentry) { 
		pentry->read_proc = clock_rate_proc_rd;
		pentry->data = (void*) clk->id;
	}
#endif
}

static int clk_register(struct clk *clk)
{
	mutex_lock(&clocks_mutex);
	mutex_init(&clk->lock);
	list_add(&clk->node, &clocks);
	clk_add_procfs(clk);

#ifdef CONFIG_MERLIN_IS_FPGA
	/* lets provide some numbers in the shadow register so that we run
	*  the calculations.  done once here should minimize or remove most
	*  other FPGA dependancies
	 */
	clk->shadow_reg = 0x0; 

	switch (clk->id) {
		case CLOCK_ID_PLL0:
			clk->shadow_reg = 0x1cb71b0;
			break;
		case CLOCK_ID_PLL1:
		case CLOCK_ID_PLL2:
		case CLOCK_ID_PLL3: 
			clk->shadow_reg = 0x3; 
			break;
		case CLOCK_ID_SCALER0:
			clk->shadow_reg = 0x8006;
			break;
		case CLOCK_ID_SCALER1:
		case CLOCK_ID_SCALER2:
		case CLOCK_ID_SCALER3: 
			clk->shadow_reg = 0xa281; 
			break;
		case CLOCK_ID_QMM:
			clk->shadow_reg = 0x2001000; 
			break;
		case CLOCK_ID_VIN0:
		case CLOCK_ID_VIN1:
		case CLOCK_ID_VOUT0:
			break; 
		case CLOCK_ID_UARTDBG:
		case CLOCK_ID_UART0:
		case CLOCK_ID_UART1:
		case CLOCK_ID_I2C0:
		case CLOCK_ID_I2C1:
		case CLOCK_ID_PLL2HL:
		case CLOCK_ID_PLL3HL:
		case CLOCK_ID_TIMER:
		case CLOCK_ID_SDMMC:
		case CLOCK_ID_SSI0:
		case CLOCK_ID_SSI1:
		case CLOCK_ID_BS:
			break;
		default:
			break;
	} 
#endif

	mutex_unlock(&clocks_mutex);

	return 0;
}

/* this is unused sinc this code is built-in to the kernel */
static void clk_unregister(struct clk *clk)
{

	int i;
	struct clk_callback *cb;
	struct clk *dep_clk = NULL;
	struct list_head *lh;

#ifdef CONFIG_PROC_FS 
	char string[20]; 
	sprintf(string,"%s/rate", clk->name);
	remove_proc_entry(string, clock_proc_dir);
#endif
	mutex_lock(&clocks_mutex);

	/* can't have callbacks for a clock that is unregistered */
	mutex_lock(&clk->lock);
	/* delete the callback entries in the controlling clocks dependency list */
	for (i=0;i < CLOCK_ID_MAX;i++) {
		if (clk->dependencies[i] != 0) { // XXX don't really like this array...
			dep_clk = clock_get_by_id(clk->dependencies[i]);
			if (dep_clk != NULL) {
				mutex_lock(&dep_clk->lock);
				list_for_each(lh, &dep_clk->callback_list) {
					cb = list_entry(lh, struct clk_callback, node); 
					list_del(&cb->node);
				}
				mutex_unlock(&dep_clk->lock);
			}
		}
		else {
			break;
		}
	}
	/* and then delete the entry in the clock for the current driver */
	list_for_each(lh, &clk->callback_list) {
		cb = list_entry(lh, struct clk_callback, node); 
		list_del(&cb->node);
	}
	mutex_unlock(&clk->lock);

	list_del(&clk->node);
	mutex_unlock(&clocks_mutex);
}

/* creates a kernel command line variable so that we can be told
*  what the pll is being driven at if it is not the default
*  12MHz
 */
static int __init merlin_clk_get_system_pll_fin(char *str)
{
	unsigned long reg;
	char *pchar = NULL;
	__system_pll_fin = simple_strtoul(str, &pchar, 0);
	__system_pll_fin /= 2;

	/* even if clk_in is set on the kernel cmdline, we can look at the clksel
	*  register in the PLL to see if we should actually use clk_in or the usb
	*  osc.  this is useful for debug so we don't have to reconfig mboot 
	*/
	if (mobi_qcc_read(QCC_BID_CHIPCTL, pll0_clk.qcc_addr, &reg, 4) == 0) {
		if ((reg & MOBI_CLK_PLL_SELECT_MASK) == 0)
			__system_pll_fin = USB_OSC_RATE;
	}
	return(1);
}
__setup("clk_in=", merlin_clk_get_system_pll_fin);

void __early_clk_init(void)
{

	clk_register(&pll0_clk);
	clk_register(&scaler0_clk);
	clk_register(&apb_clk);
	clk_register(&dbguart_clk);
	clk_register(&timer_clk);
	
}

static int __init clk_init(void)
{

#ifdef CONFIG_PROC_FS
	if (clock_proc_dir == NULL) 
		/* create /proc/drivers/clock if possible */
		clock_proc_dir = proc_mkdir("driver/clock", NULL);

	if (clock_proc_dir == NULL) {
		clock_proc_dir = proc_mkdir("driver", NULL);
		if (clock_proc_dir != NULL) 
			proc_mkdir("driver/clock", NULL);
	}
#endif
	/* add entries to proc for the early registered clocks */
	clk_add_procfs(&pll0_clk);
	clk_add_procfs(&scaler0_clk);
	clk_add_procfs(&apb_clk);
	clk_add_procfs(&dbguart_clk);
	clk_add_procfs(&timer_clk);

	clk_register(&arm_clk);
	clk_register(&ahb_clk);
	clk_register(&membus_clk);

	clk_register(&pll1_clk);
	clk_register(&pll2_clk);
	clk_register(&pll3_clk);

	clk_register(&scaler1_clk);
	clk_register(&scaler2_clk);
	clk_register(&scaler3_clk);

	clk_register(&pll2hl_clk);
	clk_register(&pll3hl_clk);

	clk_register(&vin0_clk);
	clk_register(&vin1_clk);
	clk_register(&vout0_clk);
	clk_register(&aud0_clk);
	clk_register(&aud1_clk);

	clk_register(&uart0_clk);
	clk_register(&uart1_clk);
	clk_register(&i2c0_clk);
	clk_register(&i2c1_clk);
	clk_register(&sdmmc_clk);
	clk_register(&ssi0_clk);
	clk_register(&ssi1_clk);

	clk_register(&qmm_clk);
	clk_register(&bs_clk);
	clk_register(&core_clk);

	printk("CLOCK: Merlin input clock set to %luMHz\n", 
			__system_pll_fin/1000000);
	printk("CLOCK: ARM running at %luMHz\n", 
			mobi_clock_get_rate(CLOCK_ID_ARM)/1000000);
	printk("CLOCK: Core running at %luMHz\n", 
			mobi_clock_get_rate(CLOCK_ID_CORE)/1000000);
	printk("CLOCK: Memory running at %luMHz\n", 
			mobi_clock_get_rate(CLOCK_ID_MEMBUS)/1000000);

	return 0;
}

/// @cond
/* our clock api */
EXPORT_SYMBOL(mobi_clock_register_callback);
EXPORT_SYMBOL(mobi_clock_unregister_callback);
EXPORT_SYMBOL(mobi_clock_get_rate);
EXPORT_SYMBOL(mobi_clock_set_rate);
EXPORT_SYMBOL(__early_clk_init);

/* linux clk api */
EXPORT_SYMBOL(clk_put);
EXPORT_SYMBOL(clk_get_rate);
EXPORT_SYMBOL(clk_set_rate);
EXPORT_SYMBOL(clk_get);
EXPORT_SYMBOL(clk_enable);
EXPORT_SYMBOL(clk_disable);

core_initcall(clk_init);
/// @endcond
