/*HEADER_START*/
/*
 * linux/arch/arm/mach-falcon/reset.c
 *
 * Copyright (C) 2009 Maxim IC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*HEADER_STOP*/

/* enable doxygen */
/** \file
 * Driver for Reset block
 *
 * Many devices in the SOC have the ability to be put in reset.  This
 * can be used to conserve power when a device is not going to be used.
 * At bootup time, most devices are being held in reset and therfor
 * must be taken out of reset.  The normal sequence is during init/probe
 * a driver should called
 *
 * mobi_reset_disable(<device_id>)
 *
 * and at exit/shutdown call
 *
 * mobi_reset_enable(<device_id>)
 *
 */
#ifndef DOXYGEN_SKIP

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/list.h>
#include <linux/errno.h>
#include <linux/err.h>
#include <linux/string.h>
#include <linux/mutex.h>
#include <linux/proc_fs.h>
#include <linux/semaphore.h>
#include <linux/io.h>
#include <linux/delay.h>

#include <mach/platform.h>

#include "reset.h"

static struct proc_dir_entry *reset_proc_dir;

static struct reset_dev falcon_resets[] = {
	{ .name	= "chip", 		.id = RESET_ID_CHIP, 	},
	{ .name	= "codec", 		.id = RESET_ID_CODEC, 	},
	{ .name	= "pmc", 		.id = RESET_ID_PMC, 	},
	{ .name	= "ahb_pci", 	.id = RESET_ID_AHB_PCI, },
	{ .name	= "axi_video", 	.id = RESET_ID_AXI_VIDEO},
	{ .name	= "axi_mmu", 	.id = RESET_ID_AXI_MMU 	},
	{ .name	= "spi0", 		.id = RESET_ID_SPI0, 	},
	{ .name	= "spi1", 		.id = RESET_ID_SPI1, 	},
	{ .name	= "spi2", 		.id = RESET_ID_SPI2, 	},
	{ .name	= "i2c0", 		.id = RESET_ID_I2C0, 	},
	{ .name	= "i2c1", 		.id = RESET_ID_I2C1, 	},
	{ .name	= "pwm0", 		.id = RESET_ID_PWM0, 	},
	{ .name	= "pwm1", 		.id = RESET_ID_PWM1, 	},
	{ .name	= "pwm2", 		.id = RESET_ID_PWM2, 	},
	{ .name	= "gpio0", 		.id = RESET_ID_GPIO0, 	},
	{ .name	= "gpio1", 		.id = RESET_ID_GPIO1, 	},
	{ .name	= "gpio2", 		.id = RESET_ID_GPIO2, 	},
	{ .name	= "uartdbg", 	.id = RESET_ID_UARTDBG, },
	{ .name	= "uart0", 		.id = RESET_ID_UART0, 	},
	{ .name	= "uart1", 		.id = RESET_ID_UART1, 	},
	{ .name	= "wdt", 		.id = RESET_ID_WDT, 	},
	{ .name	= "timer1", 	.id = RESET_ID_TIMER1, 	},
	{ .name	= "timer2", 	.id = RESET_ID_TIMER2, 	},
	{ .name	= "timer3", 	.id = RESET_ID_TIMER3, 	},
	{ .name	= "timer4", 	.id = RESET_ID_TIMER4, 	},
	{ .name	= "timer5", 	.id = RESET_ID_TIMER5, 	},
	{ .name	= "sha", 		.id = RESET_ID_SHA, 	},
	{ .name	= "aes", 		.id = RESET_ID_AES, 	},
	{ .name	= "nand", 		.id = RESET_ID_NAND, 	},
	{ .name	= "sdmmc", 		.id = RESET_ID_SDMMC, 	},
	{ .name	= "qdsm", 		.id = RESET_ID_QDSM, 	},
	{ .name	= "eh2h_periph", 	.id = RESET_ID_EH2H_PERIPH, },
	{ .name	= "ictl", 		.id = RESET_ID_ICTL, 	},
	{ .name	= "dmac", 		.id = RESET_ID_DMAC, 	},
	{ .name	= "ahb_dma", 	.id = RESET_ID_DMA_MMU, },
	{ .name	= "sata_p0", 	.id = RESET_ID_SATA, 	},
	{ .name	= "sata_p0", 	.id = RESET_ID_SATA_P0, },
	{ .name	= "sata_p1", 	.id = RESET_ID_SATA_P1, },
	{ .name	= "gmac", 		.id = RESET_ID_GMAC, 	},
	{ .name	= "otgphy", 	.id = RESET_ID_OTGPHY, 	},
	{ .name	= "usbh", 		.id = RESET_ID_USBH, 	},
	{ .name	= "usbp", 		.id = RESET_ID_USBP, 	},
	{ .name	= "pcie_gpex", 	.id = RESET_ID_PCIE_GPEX},
	{ .name	= "pcie_pab", 	.id = RESET_ID_PCIE_PAB,},
	{ .name	= "pcie_link", 	.id = RESET_ID_PCIE_LINK},
	{ .name	= "pcie_csr", 	.id = RESET_ID_PCIE_CSR,},
	{ .name	= "pcie_amba", 	.id = RESET_ID_PCIE_AMBA},
	{ .name	= "pcie_phy", 	.id = RESET_ID_PCIE_PHY,},
};

#endif /* DOXYGEN_SKIP */

/**
 * \brief Default timer value if self-clearing is enabled
 */
#define QCC_DEFAULT_RESET_TIMER_VALUE 0xff

/**
 * \brief reset_control:
 * 	Assert or deassert reset for a device
 *
 * \param device_id	- id of device to take action on
 * \param action	- enable or disable
 *
 * \retval -ENODEV - if invalid device_id is provided
 * \retval Zero    - upon success
 *
 */
static uint32_t reset_control(uint32_t device_id, uint8_t action)
{
	int ret = 0;

	switch (device_id) {
	case RESET_ID_CHIP:
	case RESET_ID_CODEC:
	case RESET_ID_PMC:
	case RESET_ID_AHB_PCI:
	case RESET_ID_AXI_VIDEO:
	case RESET_ID_AXI_MMU:
	case RESET_ID_SPI0:
	case RESET_ID_SPI1:
	case RESET_ID_SPI2:
	case RESET_ID_I2C0:
	case RESET_ID_I2C1:
	case RESET_ID_PWM0:
	case RESET_ID_PWM1:
	case RESET_ID_PWM2:
	case RESET_ID_GPIO0:
	case RESET_ID_GPIO1:
	case RESET_ID_GPIO2:
	case RESET_ID_UARTDBG:
	case RESET_ID_UART0:
	case RESET_ID_UART1:
	case RESET_ID_WDT:
	case RESET_ID_TIMER1:
	case RESET_ID_TIMER2:
	case RESET_ID_TIMER3:
	case RESET_ID_TIMER4:
	case RESET_ID_TIMER5:
	case RESET_ID_SHA:
	case RESET_ID_AES:
	case RESET_ID_NAND:
	case RESET_ID_SDMMC:
	case RESET_ID_QDSM:
	case RESET_ID_EH2H_PERIPH:
	case RESET_ID_ICTL:
	case RESET_ID_DMAC:
	case RESET_ID_DMA_MMU:
	case RESET_ID_SATA:
	case RESET_ID_SATA_P0:
	case RESET_ID_SATA_P1:
	case RESET_ID_GMAC:
	case RESET_ID_PCIE_GPEX:
	case RESET_ID_PCIE_PAB:
	case RESET_ID_PCIE_LINK:
	case RESET_ID_PCIE_CSR:
	case RESET_ID_PCIE_AMBA:
	case RESET_ID_PCIE_PHY:
		if (device_id & IN_CTRL1_REG_MASK)
			/* clr the mask bit that indicates ctrl1 */
			ret = mobi_qcc_write(QCC_BID_CHIPCTL,
					(action == RESET_STATE_ENABLE ?
					 QCC_CHIPCTL_RESETCONTROLSET1_OFFSET :
					 QCC_CHIPCTL_RESETCONTROLCLR1_OFFSET),
					(device_id & ~(IN_CTRL1_REG_MASK)), 
					 QCC_ACCESS_LEN_4);
		else
			ret = mobi_qcc_write(QCC_BID_CHIPCTL,
					(action == RESET_STATE_ENABLE ?
					 QCC_CHIPCTL_RESETCONTROLSET0_OFFSET :
					 QCC_CHIPCTL_RESETCONTROLCLR0_OFFSET),
					device_id, 
					QCC_ACCESS_LEN_4);
		break;
	case RESET_ID_OTGPHY:
		/* clr the mask bit that indicates ctrl1 */
		if (action == RESET_STATE_ENABLE) {
			ret = mobi_qcc_write(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_RESETCONTROLSET1_OFFSET,
					(RESET_ID_USBH & ~(IN_CTRL1_REG_MASK)), 
					QCC_ACCESS_LEN_4);
			if (ret >= 0)
				ret = mobi_qcc_write(QCC_BID_CHIPCTL,
						QCC_CHIPCTL_RESETCONTROLSET1_OFFSET,
						(RESET_ID_USBP & ~(IN_CTRL1_REG_MASK)), 
						QCC_ACCESS_LEN_4);
			if (ret >= 0)
				ret = mobi_qcc_write(QCC_BID_CHIPCTL,
						QCC_CHIPCTL_RESETCONTROLSET1_OFFSET,
						(RESET_ID_OTGPHY & ~(IN_CTRL1_REG_MASK)), 
						QCC_ACCESS_LEN_4);
		} else {
			ret = mobi_qcc_write(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_RESETCONTROLCLR1_OFFSET,
					(RESET_ID_OTGPHY & ~(IN_CTRL1_REG_MASK)), 
					QCC_ACCESS_LEN_4);
			if (ret >= 0) {
				/* wait for OTG PHY clock to start toggling */
				udelay(400);
				ret = mobi_qcc_write(QCC_BID_CHIPCTL,
						QCC_CHIPCTL_RESETCONTROLCLR1_OFFSET,
						(RESET_ID_USBP & ~(IN_CTRL1_REG_MASK)), 
						QCC_ACCESS_LEN_4);
			}
			if (ret >= 0) {
				/* ??? 6 hclock cycles, how long? */
				udelay(50);
				ret = mobi_qcc_write(QCC_BID_CHIPCTL,
						QCC_CHIPCTL_RESETCONTROLCLR1_OFFSET,
						(RESET_ID_USBH & ~(IN_CTRL1_REG_MASK)), 
						QCC_ACCESS_LEN_4);
			}
		}
		break;
	default:
		return -ENODEV;
		break;
	}
	return ret;
}

/**
 * \brief mobi_reset_enable:
 *  	Put a device into reset
 *
 * \param device_id - id of device to be put into reset
 *
 * \retval -ENODEV  	- if invalid device id is provided
 * \retval Zero   	- on success
 *
 * \remark	- Is Exported
 *
 */
uint32_t mobi_reset_enable(uint32_t device_id)
{
	return reset_control(device_id, RESET_STATE_ENABLE);
}
EXPORT_SYMBOL(mobi_reset_enable);

/**
 * \brief mobi_reset_disable:
 *  	Put a device into reset
 *
 * \param device_id - id of device to be put into reset
 *
 * \retval -ENODEV  	- if invalid device id is provided
 * \retval Zero   	- on success
 *
 * \remark	- Is Exported
 *
 */
uint32_t mobi_reset_disable(uint32_t device_id)
{
	return reset_control(device_id, RESET_STATE_DISABLE);
}
EXPORT_SYMBOL(mobi_reset_disable);

/**
 * \brief mobi_reset_state:
 *  	Query the reset state of a device
 *
 * \param device_id - id of device to be put into reset
 *
 * \retval -ENODEV  	- if invalid device id is provided
 * \retval zero   	- if device is not in reset
 * \retval one   	- if device is in reset
 *
 * \remark	- Is Exported
 *
 */
uint32_t mobi_reset_state(uint32_t device_id)
{
	unsigned long reg_value = 0;
	int in_reset = 0;

	switch (device_id) {
	case RESET_ID_CHIP:
	case RESET_ID_CODEC:
	case RESET_ID_PMC:
	case RESET_ID_AHB_PCI:
	case RESET_ID_AXI_VIDEO:
	case RESET_ID_AXI_MMU:
	case RESET_ID_SPI0:
	case RESET_ID_SPI1:
	case RESET_ID_SPI2:
	case RESET_ID_I2C0:
	case RESET_ID_I2C1:
	case RESET_ID_PWM0:
	case RESET_ID_PWM1:
	case RESET_ID_PWM2:
	case RESET_ID_GPIO0:
	case RESET_ID_GPIO1:
	case RESET_ID_GPIO2:
	case RESET_ID_UARTDBG:
	case RESET_ID_UART0:
	case RESET_ID_UART1:
	case RESET_ID_WDT:
	case RESET_ID_TIMER1:
	case RESET_ID_TIMER2:
	case RESET_ID_TIMER3:
	case RESET_ID_TIMER4:
	case RESET_ID_TIMER5:
	case RESET_ID_SHA:
	case RESET_ID_AES:
	case RESET_ID_NAND:
	case RESET_ID_SDMMC:
	case RESET_ID_QDSM:
	case RESET_ID_EH2H_PERIPH:
	case RESET_ID_ICTL:
	case RESET_ID_DMAC:
	case RESET_ID_DMA_MMU:
	case RESET_ID_SATA:
	case RESET_ID_SATA_P0:
	case RESET_ID_SATA_P1:
	case RESET_ID_GMAC:
	case RESET_ID_OTGPHY:
	case RESET_ID_PCIE_GPEX:
	case RESET_ID_PCIE_PAB:
	case RESET_ID_PCIE_LINK:
	case RESET_ID_PCIE_CSR:
	case RESET_ID_PCIE_AMBA:
	case RESET_ID_PCIE_PHY:
		if (device_id & IN_CTRL1_REG_MASK)
			in_reset = mobi_qcc_read(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_RESETCONTROLSET1_OFFSET,
					&reg_value,
					4);
		else
			in_reset = mobi_qcc_read(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_RESETCONTROLSET0_OFFSET,
					&reg_value,
					4);

		if (in_reset == 0)
			if (reg_value & device_id)
				in_reset = 1;
		break;
	default:
		in_reset = -ENODEV;
		break;
	}

	return in_reset;
}
EXPORT_SYMBOL(mobi_reset_state);

#ifdef CONFIG_PROC_FS

/**
 * \brief reset_state_proc_rd:
 * 	Create a readable procfs entry for each register reset id.
 */
static int reset_state_proc_rd(char *buf, char **start,
		off_t offset, int count, int *eof, void *data)
{
	int i, len = 0;
	struct reset_dev *reset;
	unsigned long reset0_status_reg = 0;
	unsigned long reset1_status_reg = 0;

	mobi_qcc_read(QCC_BID_CHIPCTL,
			QCC_CHIPCTL_RESETCONTROLSET0_OFFSET, &reset0_status_reg, 4);
	mobi_qcc_read(QCC_BID_CHIPCTL,
			QCC_CHIPCTL_RESETCONTROLSET1_OFFSET, &reset1_status_reg, 4);
	len += sprintf(buf+len, "device        state\n");
	len += sprintf(buf+len, "-------------------\n");
	for (i = 0, reset = falcon_resets;
			i < ARRAY_SIZE(falcon_resets); i++, reset++) {

		len += sprintf(buf+len, "%-14s  ", reset->name);

		if (reset->id & IN_CTRL1_REG_MASK)
			len += sprintf(buf+len,
					"%d",
					(reset1_status_reg & reset->id ? 1 : 0));
		else
			len += sprintf(buf+len,
					"%d",
					(reset0_status_reg & reset->id ? 1 : 0));

		len += sprintf(buf+len, "\n");
	}
	len += sprintf(buf+len, "\n");
	return len;
}
#endif
/**
 * \brief mobi_reset_init:
 * 	Called at system startup.  Priority is core_initcall
 */
static int __init mobi_reset_init(void)
{

#ifdef CONFIG_PROC_FS
	if (reset_proc_dir == NULL)
		reset_proc_dir = proc_mkdir("driver/reset", NULL);

	if (reset_proc_dir == NULL) {
		reset_proc_dir = proc_mkdir("driver", NULL);
		if (reset_proc_dir != NULL)
			proc_mkdir("driver/reset", NULL);
	}

	if (reset_proc_dir != NULL)
		create_proc_read_entry("driver/reset/status",
				S_IRUSR | S_IRGRP | S_IROTH,
				NULL, reset_state_proc_rd, NULL);
#endif

	/*
	 *  XXX this may be removed in Falcon
	 *  make sure the reset timer is disable otherwise things will
	 * come out of reset automatically
	 mobi_qcc_write(QCC_BID_CHIPCTL,
	 QCC_CHIPCTL_SOC_RESETTIMER_EN,
	 0,
	 4);
	 */

	mobi_qcc_write(QCC_BID_CHIPCTL,
			QCC_CHIPCTL_RESETCONTROLSET0_OFFSET,
			SOC_RESET0_DISABLE_MASK,
			4);
	mobi_qcc_write(QCC_BID_CHIPCTL,
			QCC_CHIPCTL_RESETCONTROLSET1_OFFSET,
			SOC_RESET1_DISABLE_MASK,
			4);

	return 0;

}
core_initcall(mobi_reset_init);
