/*HEADER_START*/
/*
 * linux/arch/arm/mach-falcon/qcc.c
 *
 * Copyright (C) 2009 Maxim IC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*HEADER_STOP*/

#ifndef DOXYGEN_SKIP

#include <linux/init.h>
#include <linux/module.h>
#include <linux/mm.h>
#include <linux/sched.h>
#include <linux/hardirq.h>
#include <linux/types.h>
#include <linux/time.h>
#include <linux/proc_fs.h>
#include <linux/major.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/uaccess.h>

#include <mach/hardware.h>
#include <asm/mach-types.h>
#include <asm/setup.h>
#include <linux/io.h>
#include <asm/mach/arch.h>

#include <mach/platform.h>

/* qcc API */
#include <mach/mobi_qcc.h>

#include <linux/falcon/qcc_defs.h>
#include <linux/falcon/pci_bar0_offsets.h>

#endif

#define BID_TABLE_ENTRY_MASK ((uint64_t)0x00000000000000ffULL)
spinlock_t qcc_lock;
static void __iomem *qcc_base;
static struct bidtable_control bidtable_ctrl;

/*
 * lock an entry in the bidtable so it cannot be replaced
 * caller must be holding bidtable_lock
 */
static int32_t __attribute__((unused)) lock_bidtable_entry(uint8_t entry)
{
	if (hweight8(bidtable_ctrl.locked_entry_bitmask) <
			BIDTABLE_MAX_LOCKED_ENTRIES)
		bidtable_ctrl.locked_entry_bitmask |= (0x1 << entry);
	else
		return -1;

	return 0;
}

static void __attribute__((unused)) unlock_bidtable_entry(uint8_t entry)
{
	bidtable_ctrl.locked_entry_bitmask &= ~(0x1 << entry);
}

/*
 * if a given bid is active(in one of the bid table registers)
 * then then return which entry(0-3 is bidtable0, 4-6 is bidtable1)
 * otherwise return -1;
 *   this function and any corresponding read/writes should be called
 * while holding the bidtable_lock; otherwise, the bidtable entries
 * could change underneath you
 */
static int32_t find_bidtable_entry(uint8_t blockID)
{
	int i = 0;

	/* note, all blkIDs/bids are 8 bits */
	while (i < BIDTABLE_MAX_SUPPORTED_BIDS &&
			(((bidtable_ctrl.table >> (i * QCC_CHIPCTL_BIDTABLE0_BID0_SIZE))
			  & 0xFF) != blockID))
		i++;

	if (likely(i < BIDTABLE_MAX_SUPPORTED_BIDS))
		return i;

	return -1;
}

/*
 * for now, we just do a crude round robin replacement of blkID's.  later,
 * we may want to create a way to lock an entry in so it cannot be removed,
 *
 */
static int update_bidtable(uint8_t newblockID)
{
	int index = 0;
	int count = 0;

	while (count < BIDTABLE_MAX_SUPPORTED_BIDS) {
		if (bidtable_ctrl.locked_entry_bitmask &
				(0x1 << (bidtable_ctrl.update_index))) {
			count++;
			if (bidtable_ctrl.update_index + 1 == BIDTABLE_MAX_SUPPORTED_BIDS)
				bidtable_ctrl.update_index = 0;
			else
				bidtable_ctrl.update_index++;

		} else
			break;
	}
	if (count == BIDTABLE_MAX_SUPPORTED_BIDS)
		return -1;

	index = bidtable_ctrl.update_index;
	/* clr and set */
	bidtable_ctrl.table &=
		~(BID_TABLE_ENTRY_MASK << 
				(bidtable_ctrl.update_index * QCC_CHIPCTL_BIDTABLE0_BID0_SIZE));
	bidtable_ctrl.table |=
		((uint64_t)(newblockID) << 
		 (bidtable_ctrl.update_index * QCC_CHIPCTL_BIDTABLE0_BID0_SIZE));

	/* writeback */
	if (bidtable_ctrl.update_index < BIDTABLE_MAX_BIDS_IN_SINGLE_REGISTER)
		writel((bidtable_ctrl.table & QCC_CHIPCTL_BIDTABLE0_MASK),
				(qcc_base + QCC_CHIPCTL_BIDTABLE0_OFFSET));
	else
		writel(((bidtable_ctrl.table >> 
						QCC_CHIPCTL_BIDTABLE0_SIZE) & QCC_CHIPCTL_BIDTABLE1_MASK),
				qcc_base + QCC_CHIPCTL_BIDTABLE1_OFFSET);

	if (bidtable_ctrl.update_index + 1 == BIDTABLE_MAX_SUPPORTED_BIDS)
		bidtable_ctrl.update_index = 0;
	else
		bidtable_ctrl.update_index++;

	return index;
}

static void __attribute__((unused))
		replace_bidtable_entry(uint8_t entry, uint8_t newblockID)
{

	/* clr and set */
	bidtable_ctrl.table &=
		~(BID_TABLE_ENTRY_MASK << (entry * QCC_CHIPCTL_BIDTABLE0_BID0_SIZE));
	bidtable_ctrl.table |=
		((uint64_t)(newblockID) << (entry * QCC_CHIPCTL_BIDTABLE0_BID0_SIZE));

	/* update */
	if (entry < BIDTABLE_MAX_BIDS_IN_SINGLE_REGISTER)
		writel((bidtable_ctrl.table & QCC_CHIPCTL_BIDTABLE0_MASK),
				(qcc_base + QCC_CHIPCTL_BIDTABLE0_OFFSET));
	else
		writel(((bidtable_ctrl.table >> 
						QCC_CHIPCTL_BIDTABLE0_SIZE) & QCC_CHIPCTL_BIDTABLE1_MASK),
				qcc_base + QCC_CHIPCTL_BIDTABLE1_OFFSET);
}

int mobi_qcc_read(int bid, int addr, unsigned long *datap, int len)
{
	void __iomem *read_addr;
	int32_t bidtable_entry;

	if (bid > MAX_QCC_BID)
		return -EINVAL;

	if (bid == QCC_BID_CHIPCTL) {
		/* we don't have to go through the qcc chain
		 * to get to the chipctl registers
		 */
		read_addr = qcc_base + addr;

	} else {

		spin_lock(&qcc_lock);
		bidtable_entry = find_bidtable_entry(bid);
		if (bidtable_entry < 0)
			bidtable_entry = update_bidtable(bid);

		if (bidtable_entry < 0) {
			spin_unlock(&qcc_lock);
			return -EBUSY;
		}

		/* the maximum slots in a bidtable register is 4(bidtable0) */
		if (bidtable_entry < BIDTABLE_MAX_BIDS_IN_SINGLE_REGISTER) {
			read_addr = qcc_base +
				(QCC_BID0_OFFSET * (bidtable_entry+1)) + addr;
		} else {
			read_addr = qcc_base +
				(QCC_BID0_OFFSET * (bidtable_entry+1)) + addr;
		}
	}

	if (len == 4)
		*datap = readl(read_addr) & 0xFFFFFFFF;
	else if (len == 2)
		*datap = readw(read_addr) & 0xFFFF;
	else
		*datap = readb(read_addr) & 0xFF;

	if (bid != QCC_BID_CHIPCTL)
		spin_unlock(&qcc_lock);

	return 0;
}
EXPORT_SYMBOL(mobi_qcc_read);

int mobi_qcc_write(int bid, int addr, unsigned long data, int len)
{
	void __iomem *write_addr;
	int32_t bidtable_entry;

	if (bid > MAX_QCC_BID)
		return -EINVAL;

	if (bid == QCC_BID_CHIPCTL) {
		/* we don't have to go through the qcc chain
		 * to get to the chipctl registers
		 */
		write_addr = qcc_base + addr;

	} else {
		spin_lock(&qcc_lock);

		bidtable_entry = find_bidtable_entry(bid);
		if (bidtable_entry < 0)
			bidtable_entry = update_bidtable(bid);

		if (bidtable_entry < 0) {
			spin_unlock(&qcc_lock);
			return -EBUSY;
		}

		/* the maximum slots in a bidtable register is 4(bidtable0) */
		if (bidtable_entry < BIDTABLE_MAX_BIDS_IN_SINGLE_REGISTER) {
			write_addr = qcc_base +
				(QCC_BID0_OFFSET * (bidtable_entry+1)) + addr;
		} else {
			write_addr = qcc_base +
				(QCC_BID0_OFFSET * (bidtable_entry+1)) + addr;
		}
	}

	if (len == 4)
		writel((data & 0xFFFFFFFF), write_addr);
	else if (len == 2)
		writew((data & 0xFFFF), write_addr);
	else
		writeb((data & 0xFF), write_addr);

	if (bid != QCC_BID_CHIPCTL)
		spin_unlock(&qcc_lock);

	return 0;
}
EXPORT_SYMBOL(mobi_qcc_write);

#ifdef CONFIG_PROC_FS
#ifdef CONFIG_ENABLE_QCC_CHIPCTL_PROCFS_INTERFACE

static struct proc_dir_entry *qcc_proc_dir;

/* set these defaults to be the resetSet vectors in chipctl */
static int procfs_qccrd_bid = 8;
static int procfs_qccrd_addr = 0xb0;
static int procfs_qccrd_len = 4;

static int procfs_qccwr_bid = 8;
static int procfs_qccwr_addr = 0xb0;
static int procfs_qccwr_len = 4;
static unsigned long  procfs_qccwr_data;

/* advance the pointer to next non-empty character  in the input buffer */
static void proc_advance_ptr(char **ptr,
		uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	while (((**ptr == ' ') || (**ptr == '\t') || (**ptr == '\r')
				|| (**ptr == '\n')) && (*ptr-buf < count)) {
		(*ptr)++;
	}
}

/* returns pointer to next arg or error if not found */
static int __attribute__((unused)) proc_get_next_arg(char **ptr,
		uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	proc_advance_ptr(ptr, buffer, count);
	if (*ptr-buf >= count) {
		return -EINVAL;
	}
	return 0;
}

/* return the next arg found as unsigned long */
static int get_ularg(char **ptr, uint32_t buffer,
		unsigned long count, uint32_t *arg)
{
	char *buf = (char *)buffer;
	proc_advance_ptr(ptr, buffer, count);
	if (*ptr-buf >= count) {
		printk(KERN_ERR "Invalid argument string\n");
		return -1;

	} else {
		*arg = simple_strtoul(*ptr, ptr, 0);
		if (*arg < 0) {
			printk(KERN_ERR "Invalid number argument\n");
			return -1;
		}
	}
	return 0;
}

static int qcc_proc_get_parts(const char *buffer, unsigned long count,
		int *bid, int *addr, int *len, unsigned long *data, char iswrite)
{
	char *ptr = (char *)buffer;
	int error = 0;

	if (get_ularg(&ptr, (uint32_t)buffer, count, bid))
		goto err_help;

	if (*bid <= 0) {
		printk(KERN_ERR "qcc: Invalid bid\n");
		error = -EINVAL;
		goto err_help;
	}

	if (get_ularg(&ptr, (uint32_t)buffer, count, addr))
		goto err_help;

	if (*addr <= 0) {
		printk(KERN_ERR "qcc: Invalid addr\n");
		error = -EINVAL;
		goto err_help;
	}

	if (get_ularg(&ptr, (uint32_t)buffer, count, len))
		goto err_help;

	if (*len <= 0) {
		printk(KERN_ERR "qcc: Invalid len\n");
		error = -EINVAL;
		goto err_help;
	}

	if (iswrite) {
		if (get_ularg(&ptr,
					(uint32_t)buffer, count, (uint32_t *)data) < 0) {
			error = -EINVAL;
			goto err_help;
		}
	}
	/* finish off line otherwise count is off
	 * and we'll be back here again... */
	while (ptr-buffer < count)
		ptr++;

	return ptr - buffer;

err_help:
	while (ptr-buffer < count)
		ptr++;

	printk(KERN_ERR "qcc: syntax is 'echo \"<bid> <addr> <len> %s'\n",
			iswrite ? "<data>\"" : "");
	printk(KERN_ERR "qcc: where addr 0x<hex>\n");
	if (!iswrite)
		printk(KERN_ERR "qcc: cat readbid entry to see read data\n");

	return error;
}

static int qcc_proc_get_readaddr(struct file *file, const char *buffer,
		unsigned long count, void *data)
{
	qcc_proc_get_parts(buffer, count,
			&procfs_qccrd_bid, &procfs_qccrd_addr,
			&procfs_qccrd_len, NULL, 0);

	return count;
}

static int qcc_proc_show_readdata(char *buf, char **start, off_t offset,
		int count, int *eof, void *data)
{
	int len = 0, ret = 0;
	unsigned long qccdata = 0;
	ret = mobi_qcc_read(procfs_qccrd_bid, procfs_qccrd_addr,
			&qccdata, procfs_qccrd_len);
	if (ret == 0)
		len += sprintf(buf+len,
				"qcc_read: bid %d, addr 0x%x, data 0x%lx (%lu)\n",
				procfs_qccrd_bid, procfs_qccrd_addr, qccdata, qccdata);
	else
		printk(KERN_ERR "qcc: problem reading bid %d address 0x%x\n",
				procfs_qccrd_bid, procfs_qccrd_addr);

	*eof = 1;
	return len;
}

static int qcc_proc_writebid(struct file *file, const char *buffer,
		unsigned long count, void *data)
{
	int ret = 0;

	ret = (qcc_proc_get_parts(buffer, count,
				&procfs_qccwr_bid, &procfs_qccwr_addr,
				&procfs_qccwr_len, &procfs_qccwr_data, 1));

	if (ret > 0) {
		ret = mobi_qcc_write(procfs_qccwr_bid,
				procfs_qccwr_addr, procfs_qccwr_data, procfs_qccwr_len);

		if (ret < 0) {
			printk(KERN_ERR
					"qcc: problem writing data to bid %d address 0x%x\n",
					procfs_qccwr_bid, procfs_qccwr_addr);
		}
	}

	return count;
}

static int qcc_proc_writereg(struct file *file, const char *buffer,
		unsigned long count, void *data)
{
	char *ptr = (char *)buffer;
	uint32_t offset, writedata;
	int error = 0;

	get_ularg(&ptr, (uint32_t)buffer, count, &offset);
	if (offset <= 0) {
		printk(KERN_ERR "qcc: Invalid addr\n");
		error = -EINVAL;
		goto err;
	}

	get_ularg(&ptr, (uint32_t)buffer, count, &writedata);

	mobi_qcc_writereg(writedata, offset);
	printk(KERN_INFO "Write 0x%x(%d) to base 0x%p, offset 0x%x\n",
			writedata, writedata, qcc_base, offset);

err:
	/* flush */
	while (ptr-buffer < count)
		ptr++;

	return ptr - buffer;
}

static int qcc_proc_readreg(struct file *file, const char *buffer,
		unsigned long count, void *data)
{
	char *ptr = (char *)buffer;
	int qccdata = 0;
	uint32_t offset;
	int error;

	get_ularg(&ptr, (uint32_t)buffer, count, &offset);
	if (offset <= 0) {
		printk(KERN_ERR "qcc: Invalid addr\n");
		error = -EINVAL;
		goto err;
	}

	qccdata = mobi_qcc_readreg(offset);
	printk(KERN_INFO "Read at base 0x%p offset 0x%x is 0x%x(%d)\n",
			qcc_base, offset, qccdata, qccdata);

err:
	while (ptr-buffer < count)
		ptr++;

	return ptr - buffer;
}

static void qcc_procfs_init(void)
{
	struct proc_dir_entry *pentry;

	if (qcc_proc_dir == NULL)
		qcc_proc_dir = proc_mkdir("driver/qcc", NULL);

	if (qcc_proc_dir == NULL) {
		qcc_proc_dir = proc_mkdir("driver", NULL);
		if (qcc_proc_dir != NULL)
			proc_mkdir("driver/qcc", NULL);
		else
			return;
	}

	pentry = create_proc_entry("driver/qcc/readbid",
			S_IRUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->read_proc  = qcc_proc_show_readdata;
		pentry->write_proc = qcc_proc_get_readaddr;
	}

	pentry = create_proc_entry("driver/qcc/writebid",
			S_IRUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->write_proc = qcc_proc_writebid;
	}

	pentry = create_proc_entry("driver/qcc/readreg",
			S_IRUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->write_proc  = qcc_proc_readreg;
	}

	pentry = create_proc_entry("driver/qcc/writereg",
			S_IRUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->write_proc = qcc_proc_writereg;
	}

}
#endif
#endif

/*
 *  call this from the board file so we can access the qcc
 *  chain, which we need to set the timer clock which is
 *  needed to run the system timer
 */
void __init __early_qccsisc_init(void)
{
	qcc_base = __io_address(QCC_BASE);
	spin_lock_init(&qcc_lock);
}

/**
 * \brief mobi_qccsisc_init:
 * 	Called at system startup.  Priority is arch_initcall
 */
static int __init mobi_qccsisc_init(void)
{
#ifdef CONFIG_PROC_FS
#ifdef CONFIG_ENABLE_QCC_CHIPCTL_PROCFS_INTERFACE
	qcc_procfs_init();
#endif
#endif
	return 0;
}
core_initcall(mobi_qccsisc_init);
