/*HEADER_START*/
/*
 * linux/arch/arm/mach-falcon/ioctrl.c
 *
 * Copyright (C) Maxim IC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*HEADER_STOP*/

/* enable doxygen */
/** \file
 * Driver for Mobilygen I/O control Interface
 *
 * \verbatim
 * Accessing ioctrl module via sysfs
 *
 * 	The ioctrl module can be access either via the API or from sysfs.
 * The sysfs directory is: /sys/bus/platform/drivers/ioctrl/  In this
 * directory are the following entries which can be used to access the
 * module:
 * 	function
 * 	drivestrength
 * 	slewrate
 * 	gpio
 *
 * The syntax for each is as follows(assuming in ioctrl directory):
 * 	1) function
 * 	   read syntax: cat function
 * 	   output:
 * 	    	function    state
 *	    	-----------------
 *  		dbg_uart      0
 *  		i2c_cfg       0
 *  		v23_mosi      0
 *  		v23_mclk      0
 *  		v01_mosi      0
 *  		v01_mclk      0
 * 			spi_mosi      0
 *  		spi_mclk      0
 *  		spi_mss0      0
 *  		spi_master    0
 *
 *  	   write syntax: echo "<function_name> <state>" > function
 *  	   where <function_name> is one from the table above and
 *  	   state is either 0 or 1
 *
 *	2) drivestrength
 * 	   read syntax: cat drivestrength
 * 	   output:
 *  	   	group    drivestrength
 *	    	----------------------
 *  		serial        2
 *  		spi           2
 *  		v01spi        2
 *  		v23spi        2
 *  		video0        2
 *  		video1        2
 * 		video2        2
 *  		video3        2
 *  		host          2
 *  		ethernet      1
 *  	    audio         1
 *       sdmmc         1
 *
 *  	   write syntax: echo "<group_name> <strength>" > function
 *  	   where <group_name> is one from the table above and
 *  	   strength is a value define by ioctrl_drivestrength_t found
 *  	   in mobi_ioctrl.h
 *
 *	3) slewrate
 * 	   read syntax: cat slewrate
 * 	   output:
 *  	   	group    slewrate
 *	   ----------------------
 *  		serial        2
 *  		spi           1
 *  		v01spi        1
 *  		v23spi        1
 *  		video0        1
 *  		video1        1
 * 			video2        1
 *  		video3        1
 *  		host          1
 *  		ethernet      2
 *  	   	audio         2
 *       	sdmmc         2
 *
 *  	   write syntax: echo "<group_name> <slew>" > slewrate
 *  	   where <group_name> is one from the table above and
 *  	   slew is a value define by ioctrl_slewrate_t found
 *  	   in mobi_ioctrl.h
 *
 * 	4) gpio
 * 	   read syntax: cat gpio
 * 	   output:
 *
 *      	gpio   function    pullUp/Down
 *      	------------------------------
 *      	  0      N/A           U
 *      	  1      N/A           U
 *      	  2      N/A           U
 *      	  3      N/A           U
 *      	  4      N/A           U
 *      	  5      N/A           U
 *      	  6      N/A           U
 *      	  7      N/A           U
 *      	  8       0            U
 *      	  9       0            U
 *      	 10       0            U
 *      	 .
 *      	 .
 *      	 N
 *
 *  	   write syntax:
 *  	   	echo "function <gpio_num> <active>" > gpio
 *  	   	echo "pullup   <gpio_num>" > gpio
 *  	   	echo "pulldown <gpio_num>" > gpio
 *  	   	echo "tristate <gpio_num>" > gpio
 *
 *  	   the first word is the command, <gpio_num> is the
 *  	   gpio that you want to modify and active is 0 or 1,
 *  	   where 0 select the primary/alt function and 1 select
 *  	   GPIO function
 *  	   Note that function does not work for the first bank
 *  	   of gpio.  See the Serial IF document for details.
 *  	   pullup and pulldown are mutually exclusive; however
 *  	   both pullup and pulldown can be disabled(tristate)
 *
 * \endverbatim
 */

#ifndef DOXYGEN_SKIP

#include <linux/init.h>
#include <linux/version.h>
#include <linux/ctype.h>
#include <linux/platform_device.h>
#include <linux/bitops.h>

#include "ioctrl.h"
#include "mx64180_ioctrl_defs.h"

#define dprintk(x...)   printk(x)

#endif

#define NOT_IMPLEMENTED

/**
 * \brief mobi_ioctrl_set_drivestrength:
 * 	Set the drivestrength for a given group.
 *
 * \param grp: defines group to access
 * \param drivestrength: new drivestrength for group
 *
 * \retval -EINVAL - if group is invalid
 * \retval Negetive value - qcc access failure
 * \retval Zero - upon success
 *
 *
 */
int mobi_ioctrl_set_drivestrength(ioctrl_ds_group_t grp, 
		ioctrl_drivestrength_t drivestrength)
{
	uint32_t addr, mask, ds, ret = 0;
	unsigned long reg;

#define DRIVESTRENGTH_FIELD_SIZE 2
#define DRIVESTRENGTH_FIELD_MASK 0x3
#define DRIVESTRENGTH_GRP0_OFFSET 0
#define DRIVESTRENGTH_GRP1_OFFSET 16
#define DRIVESTRENGTH_GRP2_OFFSET 32
#define DRIVESTRENGTH_GRP3_OFFSET 48

	if (grp >= IOCTRL_DS_PWM0 && grp <= IOCTRL_DS_CLKIN) {
		addr = QCC_CHIPCTL_DRIVESTRENGTH0_OFFSET;
		mask = (DRIVESTRENGTH_FIELD_MASK << (grp - DRIVESTRENGTH_GRP0_OFFSET));
		ds = (drivestrength << (grp - DRIVESTRENGTH_GRP0_OFFSET));
	} else if (grp >= IOCTRL_DS_ETHTXCLK && grp <= IOCTRL_DS_RSVD0) {
		addr = QCC_CHIPCTL_DRIVESTRENGTH1_OFFSET;
		mask = (DRIVESTRENGTH_FIELD_MASK << (grp - DRIVESTRENGTH_GRP1_OFFSET));
		ds = (drivestrength << (grp - DRIVESTRENGTH_GRP2_OFFSET));
	} else if (grp >= IOCTRL_DS_VIDOUTINCLK0 && grp <= IOCTRL_DS_AUD1ISPDIF) {
		addr = QCC_CHIPCTL_DRIVESTRENGTH2_OFFSET;
		mask = (DRIVESTRENGTH_FIELD_MASK << (grp - DRIVESTRENGTH_GRP2_OFFSET));
		ds = (drivestrength << (grp - DRIVESTRENGTH_GRP1_OFFSET));
	} else if (grp >= IOCTRL_DS_VID0DATA && grp <= IOCTRL_DS_VIDINCLK3) {
		addr = QCC_CHIPCTL_DRIVESTRENGTH3_OFFSET;
		mask = (DRIVESTRENGTH_FIELD_MASK << (grp - DRIVESTRENGTH_GRP3_OFFSET));
		ds = (drivestrength << (grp - DRIVESTRENGTH_GRP2_OFFSET));
	} else {
		printk(KERN_ERR "%s: Invalid DriveStrength group\n", __func__);
		return -EINVAL;
	}

	if ((ret = mobi_qcc_read(QCC_BID_CHIPCTL,
					addr, &reg, QCC_ACCESS_LEN_4))) {
			printk(KERN_ERR "%s: QCC read error at 0x%4x\n",
					__func__, addr);
			return ret;
	}
	reg &= ~(mask);
	reg |= ds;
	if ((ret = mobi_qcc_write(QCC_BID_CHIPCTL,
					addr, reg, QCC_ACCESS_LEN_4))) {
			printk(KERN_ERR "%s: QCC write error at 0x%4x\n",
					__func__, addr);
			return ret;
	}

	return ret;

}
EXPORT_SYMBOL(mobi_ioctrl_set_drivestrength);

/* this function take a gpio number and determines the bank
 *  and the pin within that bank.
 */
static int gpio_info(uint32_t gpio, uint8_t *bank)
{
	int32_t gpio_size = 0, pin = gpio;
	int i;
	gpio_desc_t *gpiod = NULL;

	*bank = 0;
	for (i = 0; i < IOCTRL_NUM_GPIO_BANKS; i++) {
		gpiod = &gpio_desc[i];
		gpio_size += hweight32(gpiod->pu_mask);
		if (gpio < gpio_size) {
			return pin;
		} else {
			*bank += 1;
			pin = gpio - gpio_size;
		}
	}
	return -1;
}

/**
 * \brief mobi_ioctrl_set_gpio:
 * 	Used to set the function of a gpio and put a pullup/pulldown
 *  one a given gpio.
 *
 * \param gpio :  gpio number to change
 * \param ctrl :  the control being changed on the gpio
 * \param state:  value to set state to
 *
 * \retval -ENODEV - gpio does not exist
 * \retval -EINVAL - invalid ctrl or state
 * \retval other negetive values - qcc read/write failed
 * \retval Zero - upon success
 *
 * \remark
 *   Only one of pullup or pulldown can be enabled at the same time.
 * Enabling  one will disable the other.  The CONTROL_SELECT is used
 * to define how the gpio will be used.  In general, a state of 0 will
 * select the primary/alt function as seclected by ioctrl_set_func.  A
 * state of 1 will cause the pin to behave as a GPIO.  Note, this is
 * the general case and not every gpio will have a primary/alt function.
 * See the Serial IF document for full details.
 */
int mobi_ioctrl_gpio_pull_ctrl(uint32_t gpio, ioctrl_gpiocontrol_t ctrl, uint8_t state)
{
	unsigned long reg_pu, reg_pd;
	int32_t pin;
	uint32_t ret = 0, tmpret = 0;
	uint8_t bank;
	gpio_desc_t *gpiod = NULL;

	if (state != IOCTRL_GPIO_PULLUP_DISABLED ||
			state != IOCTRL_GPIO_PULLDOWN_ENABLED) {
		printk(KERN_ERR "%s: Invalid pull%s state: %d\n", 
				__func__, 
				(ctrl == IOCTRL_GPIO_CONTROL_PULLUP ? "up" : "down"),
				state);
		return -EINVAL;
	}
	/* get the pin and bank */
	pin = gpio_info(gpio, &bank);
	if (pin < 0)
		return -ENODEV;

	gpiod = &gpio_desc[bank];

	if (ctrl == IOCTRL_GPIO_CONTROL_PULLUP) {
		if ((ret = mobi_qcc_read(QCC_BID_CHIPCTL,
						gpiod->pu_addr, &reg_pu, QCC_ACCESS_LEN_4))) {
			printk(KERN_ERR "%s: QCC read error at 0x%4x\n",
					__func__, gpiod->pu_addr);
			return ret;
		}

		if (state == IOCTRL_GPIO_PULLUP_DISABLED) {
			reg_pu &= ~(0x1 << pin);
		} else {
			reg_pu |=  (0x1 << pin);
			reg_pd &= ~(0x1 << pin);
		}

		if ((tmpret = mobi_qcc_write(QCC_BID_CHIPCTL,
						gpiod->pu_addr, reg_pu, QCC_ACCESS_LEN_4))) {
			ret = tmpret;
			printk(KERN_ERR "%s: QCC write error at 0x%4x\n",
					__func__, gpiod->pu_addr);
		}
	} else {

		if ((ret = mobi_qcc_read(QCC_BID_CHIPCTL,
						gpiod->pd_addr, &reg_pd, QCC_ACCESS_LEN_4))) {
			printk(KERN_ERR "%s: QCC read error at 0x%4x\n",
					__func__, gpiod->pd_addr);
			return ret;
		}

		if (state == IOCTRL_GPIO_PULLDOWN_DISABLED) {
			reg_pd &= ~(0x1 << pin);
		} else {
			reg_pu &= ~(0x1 << pin);
			reg_pd |=  (0x1 << pin);
		}

		if ((tmpret = mobi_qcc_write(QCC_BID_CHIPCTL,
						gpiod->pd_addr, reg_pd, QCC_ACCESS_LEN_4))) {
			ret = tmpret;
			printk(KERN_ERR "%s: QCC write error at 0x%4x\n",
					__func__, gpiod->pd_addr);
		}
	}

	return ret;
}

/**
 * \brief mobi_ioctrl_gpio_pullup:
 * 	Used to control the pullup on a gpio pin
 *
 * \param gpio :  gpio number to change
 * \param enable :  enable or disable the pullup
 *
 * \retval -ENODEV - gpio does not exist
 * \retval -EINVAL - invalid enable
 * \retval other negetive values - qcc read/write failed
 * \retval Zero - upon success
 *
 * \remark
 *   Only one of pullup or pulldown can be enabled at the same time.
 * Enabling  one will disable the other. 
 */
int mobi_ioctrl_gpio_pullup(uint32_t gpio, uint8_t enable)
{
	return mobi_ioctrl_gpio_pull_ctrl(gpio, IOCTRL_GPIO_CONTROL_PULLUP, enable);
}
EXPORT_SYMBOL(mobi_ioctrl_gpio_pullup);

/**
 * \brief mobi_ioctrl_gpio_pulldown:
 * 	Used to control the pulldown on a gpio pin
 *
 * \param gpio :  gpio number to change
 * \param enable :  enable or disable the pullup
 *
 * \retval -ENODEV - gpio does not exist
 * \retval -EINVAL - invalid enable
 * \retval other negetive values - qcc read/write failed
 * \retval Zero - upon success
 *
 * \remark
 *   Only one of pullup or pulldown can be enabled at the same time.
 * Enabling  one will disable the other. 
 */
int mobi_ioctrl_gpio_pulldown(uint32_t gpio, uint8_t enable)
{
	return mobi_ioctrl_gpio_pull_ctrl(gpio, IOCTRL_GPIO_CONTROL_PULLDOWN, enable);
}
EXPORT_SYMBOL(mobi_ioctrl_gpio_pulldown);

int mobi_ioctrl_gpio_mode(ioctrl_gpio_mode_group_t grp, uint8_t mode)
{
	uint32_t mask;
	int32_t ret = 0;
	unsigned long reg;

	if (mode != IOCTRL_GPIO_MODE_FUNCTIONAL || 
			mode != IOCTRL_GPIO_MODE_GPIO) {
		printk(KERN_ERR "%s: Invalid gpio mode: %d\n", __func__, mode);
		return -EINVAL;
	}
	if ((ret = mobi_qcc_read(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_GPIOSELECT_OFFSET,
					&reg, QCC_ACCESS_LEN_4))) {
		printk(KERN_ERR "%s: QCC read error at 0x%4x\n",
				__func__, QCC_CHIPCTL_GPIOSELECT_OFFSET);
		return ret;
	}

	mask = (0x1 << grp);
	reg &= ~(mask);
	reg |= (mode << grp);

	if ((ret = mobi_qcc_write(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_GPIOSELECT_OFFSET,
					reg, QCC_ACCESS_LEN_4))) {
		printk(KERN_ERR "%s: QCC write error at 0x%4x\n",
				__func__, QCC_CHIPCTL_GPIOSELECT_OFFSET);
	}

	return ret;

}
EXPORT_SYMBOL(mobi_ioctrl_gpio_mode);

/**
 * \brief mobi_ioctrl_set_function:
 * 	Used to select between the primary and alternate function
 *  in the control/function register
 *
 * \param func   : select the function to be modified
 * \param active : selects the active signal for the given function
 *
 * \retval -EINVAL - invalid func or active value
 * \retval other negetive values - qcc read/write failed
 * \retval Zero - upon success
 *
 * \remark
 * 	This function is used to define what the active signal is
 *  on specific gpio pins when the gpio has been set to select
 *  PRIMARY mode.  See the Serial IF document for details
 */
int mobi_ioctrl_set_function(ioctrl_func_t func, ioctrl_func_active_t active)
{
	int ret = 0;
	unsigned long reg;

	if (func >= IOCTRL_FUNC_END) {
		printk(KERN_ERR
				"%s: Invalid funct %d\n", __func__, func);
		return -EINVAL;

		if (active > 1)
			printk(KERN_ERR
					"%s: Invalid active %d\n", __func__, active);
		return -EINVAL;
	}

	if ((ret = mobi_qcc_read(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_SERIALIOCONTROL_OFFSET,
					&reg, QCC_ACCESS_LEN_4))) {
		printk(KERN_ERR
				"%s: QCC read error at 0x%4x\n", __func__,
				QCC_CHIPCTL_SERIALIOCONTROL_OFFSET);
		return ret;
	}
	reg &= ~(0x1 << func);
	if (active == 1)
		reg |= (0x1 << func);

	if ((ret = mobi_qcc_write(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_SERIALIOCONTROL_OFFSET,
					reg, QCC_ACCESS_LEN_4))) {
		printk(KERN_ERR
				"%s: QCC write error at 0x%4x\n", __func__,
				QCC_CHIPCTL_SERIALIOCONTROL_OFFSET);
	}

	return ret;
}
EXPORT_SYMBOL(mobi_ioctrl_set_function);

/* support functions for the reading and writing to sysfs */
static void advance_ptr(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	while (((**ptr == ' ') || (**ptr == '\t') || (**ptr == '\r') ||
				(**ptr == '\n')) && (*ptr-buf < count)) {
		(*ptr)++;
	}
}

static int get_next_arg(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;

	advance_ptr(ptr, buffer, count);
	if (*ptr-buf >= count) {
		return -EINVAL;
	}
	return 0;
}

/* use this function after getting fixed number of input */
static int  __attribute__((unused))
check_for_arg(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;

	advance_ptr(ptr, buffer, count);
	if (*ptr-buf != count) {
		if (*ptr-buf >= count) {
			return -EINVAL;
		}
		return 1;
	}
	return 0;
}

/* get and return the next argument as an unsigned long */
static int get_ularg(char **ptr, uint32_t buffer,
		unsigned long count, uint32_t *arg)
{
	char *buf = (char *)buffer;
	advance_ptr(ptr, buffer, count);
	if (*ptr-buf >= count) {
		printk("Invalid argument string\n");
		return -1;
	} else {
		*arg = simple_strtoul(*ptr, ptr, 0);
		if (*arg < 0) {
			printk(KERN_ERR "invalid argument\n");
			return -1;
		}
	}
	return 0;
}

/* get length of str from start of ptr to next white space */
/* this restores the ptr back to the begining of the word */
static int wordlen(char **ptr, uint32_t buffer, unsigned long count)
{
	uint32_t start, len;
	char *buf = (char *)buffer;

	start = (uint32_t)(*ptr);
	while (isgraph(**ptr) && (*ptr-buf < count))
		(*ptr)++;
	len = ((uint32_t)(*ptr) - start);
	/* reset the value */
	*ptr = (char *)start;

	return len;
}

#define SHOW_DS 0
#define SHOW_SR 1
static ssize_t attr_show_group(char *buf, uint8_t slew)
{
#ifdef NOT_IMPLEMENTED
	printk(KERN_WARNING "%s has not been implemented\n", __func__);
	return 0;
#else
	uint16_t addr, padgrp = 0;
	int32_t len = 0, i = 0, j = 0;
	unsigned long reg;
	int32_t ret = 0;
	uint8_t ds, sr;

	if (slew)
		len += sprintf(buf+len,
				"\ngroup     slewrate\n-----------------------\n");
	else
		len += sprintf(buf+len,
				"\ngroup    drivestrength\n------------------------\n");

	/* i play a few games here so just to be clear(er)
	 *  there are 3 groups in each register, the number of groups
	 *  could change so will use the name array for size.  the outer
	 *  loop allows determine the address to read(it assumes current
	 *  and future DS addresses are consecutive.  the inner loop
	 *  prints out name and values. when the end of the name array is
	 *  hit, break out of the outer loop
	 */
	while (1) {
		addr = ((padgrp/3) << 2)+QCC_CHIPCTL_SERIALIO_DRIVESTRENGTH0;

		ret = mobi_qcc_read(QCC_BID_CHIPCTL, addr, &reg, QCC_ACCESS_LEN_4);
		if (ret) {
			printk(KERN_ERR "QCC read error at 0x%4x\n", addr);
		} else {
			j = 0;
			while (j < 3 && group_names[i] != NULL) {
				sr = IOCTRL_DRIVESTRENTH0_SR0_R(reg);
				ds = IOCTRL_DRIVESTRENTH0_DS0_R(reg);
				/* shift so we can just reuse the above macros */
				reg >>= (IOCTRL_DRIVESTRENTH0_SR0_SIZE +
						IOCTRL_DRIVESTRENTH0_DS0_SIZE);
				if (slew)
					len += sprintf(buf+len, "%-10s    %d - %s\n",
							group_names[i], sr, slewRateNames[sr]);
				else
					len += sprintf(buf+len, "%-10s    %d - %dma\n",
							group_names[i], ds, ds*2);
				j++;
				i++;
			}
		}
		if (j <= 2 || group_names[i] == NULL)
			break; /* out of while(1) */
		else
			padgrp += 3;
	}
	return len;
#endif
}

static ssize_t
attr_show_drivestrength(struct device_driver *dev, char *buf)
{
	return attr_show_group(buf, SHOW_DS);
}

static ssize_t
attr_store_group(const char *buf, size_t count)
{
#ifdef NOT_IMPLEMENTED
	printk(KERN_WARNING "%s has not been implemented\n", __func__);
	return 0;
#else
	char *ptr = (char *)buf;
	char cmd[20];
	int len, i = 0;
	uint32_t speed = 0;

	if (get_next_arg(&ptr, (uint32_t)buf, count) == 0) {
		len = wordlen(&ptr, (uint32_t) buf, count);
		if (len > 0) {
			strncpy(cmd, ptr, len);
			cmd[len] = '\0';
			ptr += len; /* goto end of word */
			while (group_names[i] != NULL) {
				if (strcmp(cmd, group_names[i]) == 0) {
					get_ularg(&ptr, (uint32_t)buf, count, &speed);
					/* call function to update register */
					mobi_ioctrl_set_drivestrength(i, speed);
					break;

				} else {
					i++;
				}
			}
		}
		if (len <= 0 || group_names[i] == NULL) {
			printk(KERN_ERR "Invalid command issued to ioctrl drivestrength\n");
			return count;/* return a positive, otherwise will try again ? */
		}
	} else {
		printk(KERN_ERR "Invalid command issued to ioctrl drivestrength\n");
		return count;
	}
	return count;
#endif
}

static ssize_t
attr_store_drivestrength(struct device_driver *dev,
		const char *buf, size_t count)
{
	return attr_store_group(buf, count);
}
DRIVER_ATTR(drivestrength, 0644, attr_show_drivestrength,
		attr_store_drivestrength);

static ssize_t
attr_show_function(struct device_driver *dev, char *buf)
{
	int len = 0, i = 0, ret = 0;
	unsigned long reg;

	if ((ret = mobi_qcc_read(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_SERIALIOCONTROL_OFFSET,
					&reg, QCC_ACCESS_LEN_4))) {
		printk(KERN_ERR "QCC read error at 0x%4x\n",
					QCC_CHIPCTL_SERIALIOCONTROL_OFFSET);
		return ret;
	}

	len += sprintf(buf+len, "\nfunction        state\n-------------------------\n");
	while (functions[i].name != NULL) {
		len += sprintf(buf+len, "%-10s    %lu(%s)\n",
				functions[i].name,
				((reg >> i) & 0x1),
				functions[i].state_names[((reg >> i) & 0x1)]);
		i++;
	}
	return len;
}

static ssize_t
attr_store_function(struct device_driver *dev, const char *buf, size_t count)
{
	char *ptr = (char *)buf;
	char cmd[20];
	int len, i = 0, state;

	if (get_next_arg(&ptr, (uint32_t)buf, count) == 0) {
		len = wordlen(&ptr, (uint32_t) buf, count);
		if (len > 0) {
			strncpy(cmd, ptr, len);
			cmd[len] = '\0';
			ptr += len; /* goto end of word */
			while (functions[i].name != NULL) {
				if (strcmp(cmd, functions[i].name) == 0) {
					get_ularg(&ptr, (uint32_t)buf, count, &state);
					/* call function to update register */
					mobi_ioctrl_set_function(i, state);
					break;
				} else {
					i++;
				}
			}
		}
		if (len <= 0 || functions[i].name == NULL) {
			printk(KERN_ERR "Invalid command issued to ioctrl function\n");
			return count;
		}
	} else {
		return count;
	}
	return count;
}
DRIVER_ATTR(function, 0644, attr_show_function, attr_store_function);

static ssize_t
attr_store_gpio(struct device_driver *dev, const char *buf, size_t count)
{
	char *ptr = (char *)buf;
	char cmd[20];
	int len, i = 0, gpio, state, ret = 0;

	if (get_next_arg(&ptr, (uint32_t)buf, count) == 0) {
		len = wordlen(&ptr, (uint32_t) buf, count);
		if (len > 0) {
			strncpy(cmd, ptr, len);
			cmd[len] = '\0';
			ptr += len; /* goto end of word */
			while (gpio_cmds[i] != NULL) {
				if (strcmp(cmd, gpio_cmds[i]) == 0) {
					get_ularg(&ptr, (uint32_t)buf, count, &gpio);
					break;
				} else {
					i++;
				}
			}
		}
		if (len <= 0 || gpio_cmds[i] == NULL) {
			printk(KERN_ERR "Invalid command issued to ioctrl gpio\n");
			printk(KERN_ERR "Syntax is: echo \"function <gpio> <0|1>\"\n");
			printk(KERN_ERR "       or  echo \"pull{up|down|none} <gpio>\"\n");
			printk(KERN_ERR "pullup/down are mutually exclusive, tristate\n");
			printk(KERN_ERR "disables both types\n");
			return count;
		}

		switch(i) {
		case 0:/* syntax "select gpio state" */
			/* grab the extra arg */
			get_ularg(&ptr, (uint32_t)buf, count, &state);
			ret = mobi_ioctrl_gpio_mode(gpio, state);
			break;
		case 1: /* pullup enable */
			ret = mobi_ioctrl_gpio_pullup(gpio, IOCTRL_GPIO_PULLUP_ENABLED);
			break;
		case 2: /* pulldown enable */
			ret = mobi_ioctrl_gpio_pulldown(gpio, IOCTRL_GPIO_PULLDOWN_ENABLED);
			break;
		case 3: /* pullup and down disable */
			ret = mobi_ioctrl_gpio_pullup(gpio, IOCTRL_GPIO_PULLDOWN_DISABLED);
			ret = mobi_ioctrl_gpio_pulldown(gpio, IOCTRL_GPIO_PULLUP_DISABLED);
			break;
		}
		if (ret == -ENODEV)
			printk(KERN_ERR "gpio %d does not exist\n", gpio);
	} else {
		return count;
	}

	return count;
}

static ssize_t
attr_show_gpio(struct device_driver *dev, char *buf)
{
	int i, j, len = 0;
	unsigned long pu_reg, pd_reg;
	int32_t pin, pu_size;
	uint32_t ret = 0, gpio = 0;
	uint8_t bank;
	gpio_desc_t *gpiod = NULL;

	len += sprintf(buf+len, "gpio        function        pullUp/Down\n");
	len += sprintf(buf+len, "---------------------------------------\n");

	/* the idea here, like other places in this file is try to limit
	 *  the code changes for chips that come later and may have more
	 *  gpio(banks and/or pins)
	 */
	for (i = 0; i < IOCTRL_NUM_GPIO_BANKS; i++) {
		pu_reg  = 0;
		/* get the pin and bank */
		pin = gpio_info(gpio, &bank);
		gpiod = &gpio_desc[bank];
#ifndef NOT_IMPLEMENTED
		if (gpiod->sel_addr > 0) {
			if ((ret = mobi_qcc_read(QCC_BID_CHIPCTL,
							QCC_CHIPCTL_GPIOSELECT_OFFSET,
							&sel_reg, 
							QCC_ACCESS_LEN_4))) {
				printk(KERN_ERR "%s: QCC read error at 0x%4x\n",
						__func__, gpiod->sel_addr);
				return ret;
			}
		}
#endif
		if (gpiod->pu_addr > 0) {
			if ((ret = mobi_qcc_read(QCC_BID_CHIPCTL,
							gpiod->pu_addr, &pu_reg, QCC_ACCESS_LEN_4))) {
				printk(KERN_ERR "%s: QCC read error at 0x%4x\n",
						__func__, gpiod->pu_addr);
				return ret;
			}
		}
		if (gpiod->pd_addr > 0) {
			if ((ret = mobi_qcc_read(QCC_BID_CHIPCTL,
							gpiod->pd_addr, &pd_reg, QCC_ACCESS_LEN_4))) {
				printk(KERN_ERR "%s: QCC read error at 0x%4x\n",
						__func__, gpiod->pd_addr);
				return ret;
			}
		}
		pu_size  = hweight32(gpiod->pu_mask);

		for (j = 0; j < pu_size; j++, gpio++) {
			len += sprintf(buf+len, "%3d", gpio);

#ifndef NOT_IMPLEMENTED
			/* mode is no longer an a pin by pin basis */
			/* gpi0 is not valid here, print blanks */
			if (j < sel_size) {
				if (((sel_reg >> j) & 0x1))
					len += sprintf(buf+len, "%18s", "gpio");
				else
					len += sprintf(buf+len, "%18s", "primary/alt");
				//len += sprintf(buf+len, "%8lu ", ((sel_reg >> j) & 0x1));
			} else {
				len += sprintf(buf+len, "%18s", "gpio (immutable)");
			}
#endif

			if (j < pu_size) {
				/* if either is enabled */
				if (((pu_reg >> j) & 0x1) | ((pd_reg >> j) & 0x1))
					len += sprintf(buf+len, "%12s\n",
							(((pu_reg >> j) & 0x1) ? "U" : "D"));
				else /* else both are disabled */
					len += sprintf(buf+len, "%12s\n", "Z");
			} else {
				len += sprintf(buf+len, "\n");
			}
		}
	}
	return len;
}
DRIVER_ATTR(gpio, 0644, attr_show_gpio, attr_store_gpio);

/**
 * \brief mobi_ioctrl_get_function:
 *	Returns the current status of the specified function.
 *
 * \param func : the function to get status for.
 *
 * \return If return value is 0 or 1, then this is the function status.
 * \return Otherwise a negative value is returned in case of error from QCC read.
 *
 */
int mobi_ioctrl_get_function(ioctrl_func_t func)
{
	int ret;
	unsigned long reg;
#ifndef NOT_IMPLEMENTED
	if (unlikely(ret = mobi_qcc_read(QCC_BID_CHIPCTL,
					QCC_SERIALIOCONTROL_OFFSET,
					&reg, QCC_ACCESS_LEN_4))) {
		printk(KERN_ERR "%s: QCC read error at 0x%4x, returned %d\n",
				__func__, QCC_SERIALIOCONTROL_OFFSET, ret);
		return ret;
	}
#endif
	return ((reg >> func) & 0x1);
}
EXPORT_SYMBOL(mobi_ioctrl_get_function);

static inline int get_bit_depth(unsigned long n)
{
	int d = 1;
	if (!n)
		return 0;
	while (n >>= 1)
		d++;
	return d;
}

/**
 * \brief mobi_ioctrl_get_gpio_bank_size:
 *	Returns the size of the specified GPIO bank.
 *
 * \param bank : the bank number, between 0 and IOCTRL_NUM_GPIO_BANKS.
 *
 * \return The size of the bank is returned,
 * \return -EINVAL if requested bank does not exist.
 *
 */
int mobi_ioctrl_get_gpio_bank_size(uint8_t bank)
{
	if (unlikely(bank >= IOCTRL_NUM_GPIO_BANKS))
		return -EINVAL;
	return get_bit_depth(gpio_desc[bank].pu_mask |
			gpio_desc[bank].pd_mask);
}
EXPORT_SYMBOL(mobi_ioctrl_get_gpio_bank_size);

/**
 * \brief mobi_ioctrl_get_gpio_bankindex:
 *	Returns the bank number and the GPIO index in this bank for a given GPIO.
 *
 * \param gpio : the GPIO.
 * \param bank : a pointer to the bank number to be returned.
 *
 * \return the GPIO index, if GPIO was found, or -EINVAL otherwise.
 *
 */
int mobi_ioctrl_get_gpio_bankindex(uint32_t gpio, uint8_t *bank)
{
	uint32_t cur = 0, prev = 0;
	for (*bank = 0; *bank < IOCTRL_NUM_GPIO_BANKS; (*bank)++) {
		prev = cur;
		cur += mobi_ioctrl_get_gpio_bank_size(*bank);
		if (gpio < cur)
			break;
	}
	if (*bank == IOCTRL_NUM_GPIO_BANKS)
		return -EINVAL;
	return gpio - prev;
}
EXPORT_SYMBOL(mobi_ioctrl_get_gpio_bankindex);

/**
 * \brief mobi_ioctrl_get_gpio:
 *	Returns the GPIO status: wether the GPIO or primary/alt function is active,
 * or if pullup/pulldown is active.
 *
 * \param gpio : the GPIO.
 * \param ctrl : the type of status to look for:
 * 					see ioctrl_gpiocontrol_t declaration.
 *
 * \return The actual status value requested,
 * 			or a negative value to indicate an error.
 *
 */
int mobi_ioctrl_get_gpio(uint32_t gpio, ioctrl_gpiocontrol_t ctrl)
{
	int ret, index;
	uint8_t bank;
	gpio_desc_t *gpiod;
	unsigned long reg;

	if (unlikely((index = mobi_ioctrl_get_gpio_bankindex(gpio, &bank)) < 0))
		return index;
	gpiod = &gpio_desc[bank];

#ifndef NOT_IMPLEMENTED
	if (ctrl == IOCTRL_GPIO_CONTROL_FUNCTION) {
		if (gpiod->sel_addr > 0) {
			if (unlikely(ret = mobi_qcc_read(QCC_BID_CHIPCTL,
							gpiod->sel_addr, &reg, QCC_ACCESS_LEN_4))) {
				printk(KERN_ERR "%s: QCC read error at 0x%4x, returned %d\n",
						__func__, gpiod->sel_addr, ret);
				return ret;
			}
		} else {
			/* Assume that no register to change function
			 * = immutable GPIO function. */
			return IOCTRL_GPIO_FUNCTION_GPIO;
		}
	}
	else 
#endif
		if (ctrl == IOCTRL_GPIO_CONTROL_PULLUP) {
		if (gpiod->pu_addr > 0) {
			if (unlikely(ret = mobi_qcc_read(QCC_BID_CHIPCTL,
							gpiod->pu_addr, &reg, QCC_ACCESS_LEN_4))) {
				printk(KERN_ERR "%s: QCC read error at 0x%4x, returned %d\n",
						__func__, gpiod->pu_addr, ret);
				return ret;
			}
		} else {
			return -EINVAL;
		}
	}
	else if (ctrl == IOCTRL_GPIO_CONTROL_PULLDOWN) {
		if (gpiod->pd_addr > 0) {
			if (unlikely(ret = mobi_qcc_read(QCC_BID_CHIPCTL,
							gpiod->pd_addr, &reg, QCC_ACCESS_LEN_4))) {
				printk(KERN_ERR "%s: QCC read error at 0x%4x, returned %d\n",
						__func__, gpiod->pd_addr, ret);
				return ret;
			}
		} else {
			return -EINVAL;
		}
	} else {
		return -EINVAL;
	}

	return ((reg >> index) & 0x1);
}
EXPORT_SYMBOL(mobi_ioctrl_get_gpio);

static const struct ioctrl_macro_s ioctrl_default_macros[] = { { .name = NULL } };
static const struct ioctrl_macro_s *ioctrl_macros;

/**
 * \brief mobi_ioctrl_set_macros:
 *	Sets the available macros.
 * See struct ioctrl_macro_s for how to define them.
 * But be careful: there is no locking done to modify this, so do it
 * only at init or when you are sure nobody is trying to apply macros.
 *
 * \param macros : the new macro array.
 *
 */
void mobi_ioctrl_set_macros(const struct ioctrl_macro_s *macros)
{
	if (macros)
		ioctrl_macros = macros;
}
EXPORT_SYMBOL(mobi_ioctrl_set_macros);

/**
 * \brief mobi_ioctrl_get_macro:
 *	Returns the macro struct associated with this macro name.
 *
 * \param name : the macro name to look for.
 *
 * \return The macro pointer if this macro name was found, NULL otherwise.
 *
 */
const struct ioctrl_macro_s *mobi_ioctrl_get_macro(const char *name)
{
	int i;
	for (i = 0; ioctrl_macros[i].name != NULL; i++) {
		if (!strcmp(ioctrl_macros[i].name, name))
			return &ioctrl_macros[i];
	}
	return NULL;
}
EXPORT_SYMBOL(mobi_ioctrl_get_macro);

/**
 * \brief mobi_ioctrl_apply_macro:
 *	Applies a macro given a macro struct (not the macro name for
 * performance issues). See mobi_ioctrl_get_macro() to get the macro struct
 * from the macro name.
 *
 * \param macro : the macro struct to apply.
 *
 * \return 0 if all macro parameters could be applied, a negative value otherwise.
 *
 */
int mobi_ioctrl_apply_macro(const struct ioctrl_macro_s *macro)
{
	int j, ret = 0;
	if (unlikely(!macro))
		return -EINVAL;
	for (j = 0; macro->func[j].func != IOCTRL_FUNC_END; j++) {
		if (unlikely(ret = mobi_ioctrl_set_function(macro->func[j].func,
						macro->func[j].active)))
			goto end;
	}
	for (j = 0; macro->gpio[j].gpio != -1; j++) {
#ifndef NOT_IMPLEMENTED
		if (unlikely(ret = mobi_ioctrl_gpio_mode(macro->gpio[j].gpio,
						macro->gpio[j].sel)))
			goto end;
#endif
		if (unlikely(ret = mobi_ioctrl_gpio_pullup(macro->gpio[j].gpio,
						macro->gpio[j].pu)))
			goto end;
		if (unlikely(ret = mobi_ioctrl_gpio_pulldown(macro->gpio[j].gpio,
						macro->gpio[j].pd)))
			goto end;
	}
end:
	if (unlikely(ret))
		printk(KERN_ERR "%s: failed to apply macro '%s', error is %d\n",
				__func__, macro->name, ret);
	return ret;
}
EXPORT_SYMBOL(mobi_ioctrl_apply_macro);

/**
 * \brief mobi_ioctrl_macro_is_active:
 *	Tests if the given macro is currently active, i.e. if
 * the current function status, GPIO status etc. are the same that the macro
 * would apply.
 *
 * \param macro : a pointer to the macro to test for.
 *
 * \return 0 if not active, 1 if active,
 * \return a negative value indicating an error otherwise.
 *
 */
int mobi_ioctrl_macro_is_active(const struct ioctrl_macro_s *macro)
{
	int j;
	if (unlikely(!macro))
		return -EINVAL;
	for (j = 0; macro->func[j].func != IOCTRL_FUNC_END; j++) {
		if (mobi_ioctrl_get_function(macro->func[j].func) != macro->func[j].active)
			return 0;
	}
	for (j = 0; macro->gpio[j].gpio != -1; j++) {
		if (mobi_ioctrl_get_gpio(macro->gpio[j].gpio,
					IOCTRL_GPIO_CONTROL_FUNCTION) != macro->gpio[j].sel)
			return 0;
		if (mobi_ioctrl_get_gpio(macro->gpio[j].gpio,
					IOCTRL_GPIO_CONTROL_PULLUP) != macro->gpio[j].pu)
			return 0;
		if (mobi_ioctrl_get_gpio(macro->gpio[j].gpio,
					IOCTRL_GPIO_CONTROL_PULLDOWN) != macro->gpio[j].pd)
			return 0;
	}
	return 1;
}
EXPORT_SYMBOL(mobi_ioctrl_macro_is_active);

static ssize_t attr_store_macro(struct device_driver *dev,
		const char *buf, size_t count)
{
	int i, len;
	char b[64];
	strncpy(b, buf, sizeof(b));
	len = strlen(b);
	while (isspace(b[len - 1])) {
		b[len - 1] = 0;
		if (--len == 0)
			break;
	}
	for (i = 0; ioctrl_macros[i].name != NULL; i++) {
		if (!strcmp(ioctrl_macros[i].name, b)) {
			mobi_ioctrl_apply_macro(&ioctrl_macros[i]);
			return count;
		}
	}
	printk(KERN_ERR "%s: invalid macro given; input was: %s\n", __func__, buf);
	return count;
}

ssize_t attr_show_macro(struct device_driver *dev, char *buf)
{
	int ret = 0, i, j;
	char b[32] = { 0 };
	ret += snprintf(buf + ret,
			PAGE_SIZE - ret, "Echo macro name in this file to apply macro.\n");
	ret += snprintf(buf + ret, PAGE_SIZE - ret, "Available macros:\n\n");
	for (i = 0; ioctrl_macros[i].name != NULL; i++) {
		ret += snprintf(buf + ret, PAGE_SIZE - ret,
				"%s: %s\n", ioctrl_macros[i].name,
				mobi_ioctrl_macro_is_active(&ioctrl_macros[i]) ? "ACTIVE" : "inactive");

		for (j = 0; ioctrl_macros[i].func[j].func != IOCTRL_FUNC_END; j++) {
			ret += snprintf(buf + ret, PAGE_SIZE - ret, " func=%s val=%d(cur=%d)\n",
					functions[ioctrl_macros[i].func[j].func].name,
					ioctrl_macros[i].func[j].active,
					mobi_ioctrl_get_function(ioctrl_macros[i].func[j].func));
		}
		for (j = 0; ioctrl_macros[i].gpio[j].gpio != -1; j++) {
			if (strlen(ioctrl_macros[i].gpio[j].name) > 11)
				strncpy(b, ioctrl_macros[i].gpio[j].name + 11, sizeof(b));
			ret += snprintf(buf + ret, PAGE_SIZE - ret,
					" gpio=%u|%s mode=%s(cur=%s) pull=%s(cur=%s)\n",
					ioctrl_macros[i].gpio[j].gpio, b,
					ioctrl_macros[i].gpio[j].sel ==
					IOCTRL_GPIO_FUNCTION_PRIMARY ? "FUNC" : "GPIO",
					mobi_ioctrl_get_gpio(ioctrl_macros[i].gpio[j].gpio, 
						IOCTRL_GPIO_CONTROL_FUNCTION) == IOCTRL_GPIO_FUNCTION_PRIMARY ? "FUNC" : "GPIO",
					ioctrl_macros[i].gpio[j].pu ? "U" :
					(ioctrl_macros[i].gpio[j].pd ? "D" : "N"),
					mobi_ioctrl_get_gpio(ioctrl_macros[i].gpio[j].gpio, IOCTRL_GPIO_CONTROL_PULLUP)
					? "U" : (mobi_ioctrl_get_gpio(ioctrl_macros[i].gpio[j].gpio,
							IOCTRL_GPIO_CONTROL_PULLDOWN) ? "D" : "N"));
		}
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "\n");
	}
	return ret;
}
DRIVER_ATTR(macro, 0644, attr_show_macro, attr_store_macro);

static int ioctrl_drv_probe(struct platform_device *pdev)
{
	struct falcon_ioctrl_data_t *pdata = pdev->dev.platform_data;
	int size, i, j, bank0_size;
	gpio_desc_t *gpiod = NULL;

	size = ARRAY_SIZE(pdata->function_defaults);
	for (i=0; i < size; i++) {
		mobi_ioctrl_set_function(i,
				pdata->function_defaults[i]);
	}

	size = ARRAY_SIZE(pdata->drivestrength_defaults);
	for (i = 0; i < size; i++) {
		mobi_ioctrl_set_drivestrength(i,
				pdata->drivestrength_defaults[i]);
	}

	/* this only applies to banks 1 and 2 but the gpios start from
	 *  0 in bank 0 so have to find the size of 0 based on the number
	 *  of pullups defined in bank0.  then we start the loop from
	 *  that count!
	 */
	gpiod = &gpio_desc[0];
	bank0_size = hweight32(gpiod->pu_mask);
	size = ARRAY_SIZE(pdata->gpiosel_defaults) + bank0_size;
	for (i = bank0_size, j = 0; i < size; i++, j++) {
		mobi_ioctrl_gpio_mode(i,
				pdata->gpiosel_defaults[j]);
	}

	size = ARRAY_SIZE(pdata->gpiopu_defaults);
	for (i = 0; i < size; i++) {
		mobi_ioctrl_gpio_pullup(i,
				pdata->gpiopu_defaults[i]);
	}

	return 0;
}

static int ioctrl_drv_remove(struct platform_device *pdev)
{
	return 0;
}

static void ioctrl_drv_shutdown(struct platform_device *pdev)
{
	return;
}

static struct platform_driver ioctrl_driver = {
	.driver = {
		.name   = PLATFORM_NAME_IOCTRL,
	},
	.probe          = ioctrl_drv_probe,
	.remove         = ioctrl_drv_remove,
	.shutdown       = ioctrl_drv_shutdown,
};

static int __init ioctrl_init(void)
{
	int ret = 0;

	ioctrl_macros = ioctrl_default_macros;

	ret = platform_driver_register(&ioctrl_driver);

	if (ret >= 0) {
		if (driver_create_file(&ioctrl_driver.driver, &driver_attr_function))
			printk(KERN_WARNING
					"Unable to create ioctrl function attribute\n");
		if (driver_create_file(&ioctrl_driver.driver,
					&driver_attr_drivestrength))
			printk(KERN_WARNING
					"Unable to create ioctrl drivestrength attribute\n");
		if (driver_create_file(&ioctrl_driver.driver, &driver_attr_gpio))
			printk(KERN_WARNING "Unable to create ioctrl gpio attribute\n");
		if (driver_create_file(&ioctrl_driver.driver, &driver_attr_macro))
			printk(KERN_WARNING "Unable to create ioctrl macro attribute\n");
	}
	return ret;
}

static void __exit ioctrl_exit(void)
{
	driver_remove_file(&ioctrl_driver.driver, &driver_attr_function);
	driver_remove_file(&ioctrl_driver.driver, &driver_attr_drivestrength);
	driver_remove_file(&ioctrl_driver.driver, &driver_attr_gpio);
	driver_remove_file(&ioctrl_driver.driver, &driver_attr_macro);

	platform_driver_unregister(&ioctrl_driver);
}
arch_initcall(ioctrl_init);
