/*HEADER_START*/
/*
 *  linux/arch/arm/mach-falcon/dma.c
 *
 *  Copyright (C) 2009 Maxim IC
 *  	initial version based on
 *  	linux/arch/arm/mach-imx/dma.c
 *
 *  imx DMA registration and IRQ dispatching
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  2004-03-03 Sascha Hauer <sascha@saschahauer.de>
 *             initial version heavily inspired by
 *             linux/arch/arm/mach-pxa/dma.c
 *
 *  2005-04-17 Pavel Pisa <pisa@cmp.felk.cvut.cz>
 *             Changed to support scatter gather DMA
 *             by taking Russell's code from RiscPC
 *
 *  2006-05-31 Pavel Pisa <pisa@cmp.felk.cvut.cz>
 *             Corrected error handling code.
 *
 */
/*HEADER_STOP*/

/** \file
 * Driver for Synopsys DW DMAC
 */

#ifndef DOXYGEN_SKIP

#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/errno.h>
#include <linux/proc_fs.h>
#include <linux/version.h>
#include <asm/cacheflush.h>

#include <asm/mach/arch.h>
#include <asm/mach/irq.h>
#include <asm/mach/map.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/delay.h>

#include <mach/dma.h>
#include <mach/mobi_reset.h>
#include <mach/mobi_qcc.h>

#define DRIVER_NAME "DW-DMAC "

#undef LOCAL_DW_DMAC_DEBUG_ENABLE

static int __attribute__((unused)) loglevel = -1;
#if defined(LOCAL_DW_DMAC_DEBUG_ENABLE) || defined(CONFIG_DW_DMAC_DEBUG)
#define DW_DMAC_DEBUG 1
#else
#define DW_DMAC_DEBUG 0
#endif

#if DW_DMAC_DEBUG
/*
 * many functions use the dma handle and I use the dmah variable name
 *  for debug, define a global so that dprintk will work in functions
 *  without dmah defined...
 */
int dmah = -1;
#define DBGPFX "DMAC:"

//#define DEBUG_SINGLE_DMAC_CHANNEL 1
#undef DEBUG_SINGLE_DMAC_CHANNEL

#ifndef DEBUG_SINGLE_DMAC_CHANNEL
#define dmac_debugch(n, fmt, args...)	\
	do { 						\
		if (loglevel >= n) { 	\
			printk(DBGPFX"%s(%d): " fmt, __func__, dmah, ##args); \
		} \
	} while(0)

#else

#define dmac_debugch(n, fmt, args...)	\
	do { 						\
		if ((loglevel >= n) && (dmah == DEBUG_SINGLE_DMAC_CHANNEL)) { \
			printk(DBGPFX"%s(%d): " fmt, __func__, dmah, ##args); \
		} \
	} while(0)

#endif
#else
#define dmac_debugch(n, fmt, args...)	do {} while(0)
#define dmac_debug(n, fmt, args...)	do {} while(0)
#endif

#define dprintk(x...)	printk(x)
#define dprintk0(fmt, x...)	do {} while(0)
#define dprintk1(fmt, x...)	dmac_debugch(1, fmt, ##x)
#define dprintk2(fmt, x...)	dmac_debugch(2, fmt, ##x)
#define dprintk3(fmt, x...)	dmac_debugch(3, fmt, ##x)
#define dprintk4(fmt, x...)	dmac_debugch(4, fmt, ##x)
#define dprintk5(fmt, x...)	dmac_debugch(5, fmt, ##x)

#define error(str, args...)  \
	printk(KERN_ERR DRIVER_NAME "ERROR: " str "\n", ##args)
#define warning(str, args...)  \
	printk(KERN_WARNING DRIVER_NAME "WARNING: " str "\n", ##args)
#define info(str, args...)  \
	printk(KERN_INFO DRIVER_NAME "INFO: " str "\n", ##args)

static void __iomem *dmac_base = NULL;
static void __iomem *dtcm_iobase = NULL;

#define DMAC_READL(offset, var) \
	var = 0; \
var |= (readl(dmac_base+(uint32_t)offset) & 0xffffffff)

#define DMAC_READD(offset, var) \
	var = (readl(dmac_base+(uint32_t)offset+0x4)); \
var <<= 32; \
var |= (readl(dmac_base+(uint32_t)offset))

/* write only lower 32 bits */
#define DMAC_WRITEL(data, offset) \
	writel((uint32_t)(data & 0xffffffff), dmac_base+(uint32_t)offset);

#define DMAC_WRITED(data, offset) \
	writel((uint32_t)(data >> 32), dmac_base+(uint32_t)offset+0x4); \
writel((uint32_t)(data), dmac_base+(uint32_t)offset)

/* return the offset for a specific register */
#define DMAC_CHREG_OFF(reg) \
	MOBI_DMAC_CH0_##reg##_OFFSET

/* returns the offset for a specific register for a given channel */
#define DMAC_CHREG_ADDR(ch, reg) \
	dma_ch_reg_base(ch)+DMAC_CHREG_OFF(reg)

spinlock_t dmac_lock;
spinlock_t codec_reset_mutex;

static struct mobi_dma_channel mobi_dma_channels[MAX_DMAC_CHANNELS];
static struct proc_dir_entry *dma_proc_dir=NULL;

void event_tasklet(unsigned long dmah);
DECLARE_TASKLET(dmach0_tasklet, event_tasklet, DMAC_CHANNEL_0);
DECLARE_TASKLET(dmach1_tasklet, event_tasklet, DMAC_CHANNEL_1);
DECLARE_TASKLET(dmach2_tasklet, event_tasklet, DMAC_CHANNEL_2);
DECLARE_TASKLET(dmach3_tasklet, event_tasklet, DMAC_CHANNEL_3);

/* the maximum size of a descritor list in DESC_RAM for a single channel */
#define MAX_DESC_RAM_LLI_SIZE	SZ_4K

#endif // DOXYGEN_SKIP

/* return the base offset for a given channel */
static uint32_t dma_ch_reg_base(int ch)
{
	switch (ch) {
		case 0: return CHANNEL0_OFFSET;
		case 1: return CHANNEL1_OFFSET;
		case 2: return CHANNEL2_OFFSET;
		case 3: return CHANNEL3_OFFSET;
		default: break;
	}
	return -1;
}

/* disable all interrupt for a given channel */
static void mask_channel_interrupts(mobi_dma_handle dma_ch)
{
	uint32_t ch_mask = (1<<(dma_ch+8) | 0x0);

	/* mask all the channel interrupts */
	DMAC_WRITEL(ch_mask, MOBI_DMAC_MASK_TFR_OFFSET);
	DMAC_WRITEL(ch_mask, MOBI_DMAC_MASK_BLOCK_OFFSET);
	DMAC_WRITEL(ch_mask, MOBI_DMAC_MASK_SRC_TRAN_OFFSET);
	DMAC_WRITEL(ch_mask, MOBI_DMAC_MASK_DST_TRAN_OFFSET);
	DMAC_WRITEL(ch_mask, MOBI_DMAC_MASK_ERR_OFFSET);
}

static void clear_channel_interrupts(mobi_dma_handle dma_ch)
{
	uint32_t ch_mask = (1<<dma_ch);
	/* clear all channel interrupts */
	DMAC_WRITEL(ch_mask, MOBI_DMAC_CLEAR_TFR_OFFSET);
	DMAC_WRITEL(ch_mask, MOBI_DMAC_CLEAR_BLOCK_OFFSET);
	DMAC_WRITEL(ch_mask, MOBI_DMAC_CLEAR_SRC_TRAN_OFFSET);
	DMAC_WRITEL(ch_mask, MOBI_DMAC_CLEAR_DST_TRAN_OFFSET);
	DMAC_WRITEL(ch_mask, MOBI_DMAC_CLEAR_ERR_OFFSET);
}

/* see page 166 of DMAC databook */
/* convert client setting to register value */
static int32_t msize_api2reg(uint32_t burst_size, uint8_t *regfield)
{
	switch(burst_size) {
		case  MOBI_DMA_CONFIG_BURST_SIZE_1  :
			*regfield = DMA_BURST_SIZE_1;
			break;
		case  MOBI_DMA_CONFIG_BURST_SIZE_4  :
			*regfield = DMA_BURST_SIZE_4;
			break;
		case  MOBI_DMA_CONFIG_BURST_SIZE_8  :
			*regfield = DMA_BURST_SIZE_8;
			break;
		case  MOBI_DMA_CONFIG_BURST_SIZE_16 :
			*regfield = DMA_BURST_SIZE_16;
			break;
		case  MOBI_DMA_CONFIG_BURST_SIZE_32 :
			*regfield = DMA_BURST_SIZE_32;
			break;
		case  MOBI_DMA_CONFIG_BURST_SIZE_64 :
			*regfield = DMA_BURST_SIZE_64;
			break;
		case  MOBI_DMA_CONFIG_BURST_SIZE_128:
			*regfield = DMA_BURST_SIZE_128;
			break;
		case  MOBI_DMA_CONFIG_BURST_SIZE_256:
			*regfield = DMA_BURST_SIZE_256;
			break;
		default:
			return -1;
	}
	return 0;
}

/* needs own function since overlaps values of config flags */
static int32_t msize_flag2reg(uint32_t burst_size, uint8_t *regfield)
{
	switch(burst_size) {
		case  MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_1  :
			*regfield = DMA_BURST_SIZE_1;
			break;
		case  MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_4  :
			*regfield = DMA_BURST_SIZE_4;
			break;
		case  MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_8  :
			*regfield = DMA_BURST_SIZE_8;
			break;
		case  MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_16 :
			*regfield = DMA_BURST_SIZE_16;
			break;
		case  MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_32 :
			*regfield = DMA_BURST_SIZE_32;
			break;
		case  MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_64 :
			*regfield = DMA_BURST_SIZE_64;
			break;
		case  MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_128:
			*regfield = DMA_BURST_SIZE_128;
			break;
		case  MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_256:
			*regfield = DMA_BURST_SIZE_256;
			break;
		default:
			return -1;
	}
	return 0;
}
/* encode value programmed in register to client setting */
static int32_t __attribute__((unused)) msize_reg2api(uint32_t burst_size)
{
	switch(burst_size) {
		case  DMA_BURST_SIZE_1  : return MOBI_DMA_CONFIG_BURST_SIZE_1 ;
		case  DMA_BURST_SIZE_4  : return MOBI_DMA_CONFIG_BURST_SIZE_4 ;
		case  DMA_BURST_SIZE_8  : return MOBI_DMA_CONFIG_BURST_SIZE_8 ;
		case  DMA_BURST_SIZE_16 : return MOBI_DMA_CONFIG_BURST_SIZE_16;
		case  DMA_BURST_SIZE_32 : return MOBI_DMA_CONFIG_BURST_SIZE_32;
		case  DMA_BURST_SIZE_64 : return MOBI_DMA_CONFIG_BURST_SIZE_64;
		case  DMA_BURST_SIZE_128: return MOBI_DMA_CONFIG_BURST_SIZE_128;
		case  DMA_BURST_SIZE_256: return MOBI_DMA_CONFIG_BURST_SIZE_256;
		default: return -1;
	}
	return -1;
}

/* register setting converted to real numeric value */
static int32_t msize_reg2num(uint32_t burst_size)
{
	switch(burst_size) {
		case  DMA_BURST_SIZE_1  : return 1 ;
		case  DMA_BURST_SIZE_4  : return 4 ;
		case  DMA_BURST_SIZE_8  : return 8 ;
		case  DMA_BURST_SIZE_16 : return 16;
		case  DMA_BURST_SIZE_32 : return 32;
		case  DMA_BURST_SIZE_64 : return 64;
		case  DMA_BURST_SIZE_128: return 128;
		case  DMA_BURST_SIZE_256: return 256;
		default: return -1;
	}
	return -1;
}

/*
 *  saves the value to be written to the register based on the
 *  client size setting
 */
static int32_t dma_set_msize(struct mobi_dma_channel *mobidma,
		uint32_t api_msize, uint32_t node)
{
	uint8_t regvalue;

	if (msize_api2reg(api_msize, &regvalue))
		return -EINVAL;

	if (node == DMA_NODE_IS_SRC)
		mobidma->ctl.smsize = regvalue;
	else
		mobidma->ctl.dmsize = regvalue;

	return 0;
}

/* see page 167 of DMAC databook */
/* convert client value to register value */
static int32_t trwidth_api2reg(uint32_t width, uint8_t *regfield)
{
	switch(width) {
		case  MOBI_DMA_CONFIG_TRANSFER_WIDTH_8 :
			*regfield = DMA_TRWIDTH_8;
			break;
		case  MOBI_DMA_CONFIG_TRANSFER_WIDTH_16:
			*regfield = DMA_TRWIDTH_16;
			break;
		case  MOBI_DMA_CONFIG_TRANSFER_WIDTH_32:
			*regfield = DMA_TRWIDTH_32;
			break;
		default:
			return -1;
	}
	return 0;
}

/* convert register value to client value */
static int32_t __attribute__((unused)) trwidth_reg2api(uint32_t width)
{
	switch(width) {
		case  DMA_TRWIDTH_8 : return MOBI_DMA_CONFIG_TRANSFER_WIDTH_8 ;
		case  DMA_TRWIDTH_16: return MOBI_DMA_CONFIG_TRANSFER_WIDTH_16;
		case  DMA_TRWIDTH_32: return MOBI_DMA_CONFIG_TRANSFER_WIDTH_32;
		default: return -1;
	}
	return -1;
}

/* get actuall numeric value from register value */
/* needed for calculations			 */
static uint32_t trwidth_reg2num(uint32_t width)
{
	switch(width) {
		case  DMA_TRWIDTH_8 : return 8;
		case  DMA_TRWIDTH_16: return 16;
		case  DMA_TRWIDTH_32: return 32;
		default: return -1;
	}
	return -1;
}

/*
 *  saves the value to be written to the register based on the
 *  client width setting
 */
static int32_t dma_set_trwidth(struct mobi_dma_channel *mobidma,
		uint32_t width, uint32_t node)
{
	uint8_t regvalue;

	if (trwidth_api2reg(width, &regvalue))
		return -EINVAL;

	if (node == DMA_NODE_IS_SRC)
		mobidma->ctl.strwidth = regvalue;
	else
		mobidma->ctl.dtrwidth = regvalue;

	return 0;
}

/* see page 167 of DMAC databook */
/* convert client value to register value */
static int32_t addradj_api2reg(uint32_t adj, uint8_t *regfield)
{
	switch(adj) {
		case  MOBI_DMA_CONFIG_ADDRADJ_INC :
			*regfield = DMA_ADDRADJ_INC;
			break;
		case  MOBI_DMA_CONFIG_ADDRADJ_DEC :
			*regfield = DMA_ADDRADJ_DEC;
			break;
		case  MOBI_DMA_CONFIG_ADDRADJ_NONE:
			*regfield = DMA_ADDRADJ_NONE;
			break;
		default:
			return -1;
	}
	return 0;
}

/* conversion for dma list node flags */
static int32_t addradj_flag2reg(uint32_t adj, uint8_t *regfield)
{
	switch(adj) {
		case  MOBI_DMA_LIST_NODE_FLAG_ADDRADJ_INC :
			*regfield = DMA_ADDRADJ_INC;
			break;
		case  MOBI_DMA_LIST_NODE_FLAG_ADDRADJ_DEC :
			*regfield = DMA_ADDRADJ_DEC;
			break;
		case  MOBI_DMA_LIST_NODE_FLAG_ADDRADJ_NONE:
			*regfield = DMA_ADDRADJ_NONE;
			break;
		default:
			return -1;
	}
	return 0;

}

/* convert register value to client value */
static int32_t __attribute__((unused)) addradj_reg2api(uint32_t adj)
{
	switch(adj) {
		case  DMA_ADDRADJ_INC : return MOBI_DMA_CONFIG_ADDRADJ_INC ;
		case  DMA_ADDRADJ_DEC : return MOBI_DMA_CONFIG_ADDRADJ_DEC;
		case  DMA_ADDRADJ_NONE: return MOBI_DMA_CONFIG_ADDRADJ_NONE;
		default: return -1;
	}
	return -1;
}
/*
 *  saves the value to be written to the register based on the
 *  client size setting
 */
static int32_t dma_set_addradj(struct mobi_dma_channel *mobidma,
		uint32_t adj, uint32_t node)
{
	uint8_t regvalue;

	if (addradj_api2reg(adj, &regvalue))
		return -EINVAL;

	if (node == DMA_NODE_IS_SRC)
		mobidma->ctl.sinc = regvalue;
	else
		mobidma->ctl.dinc = regvalue;

	return 0;
}

static int32_t dma_set_endianess(struct mobi_dma_channel *mobidma)
{
	uint32_t src_trwidth, dst_trwidth;
	uint32_t data_width  = mobidma->datawidth;

	dprintk3("src_endianess: %s, dst_endianess: %s\n",
			mobidma->src_endianess & MOBI_DMA_CONFIG_ENDIAN_BE ? "BE" : "LE",
			mobidma->dst_endianess & MOBI_DMA_CONFIG_ENDIAN_BE ? "BE" : "LE");

	/*
	 * We treat memory as a 32bits device for performance reasons ... but
	 * in fact the data are bytes inside. This is a problem to calculate
	 * the proper swapping
	 */
	if(mobidma->src_node_type==DMA_NODE_TYPE_MEM) {	
		dprintk4("SRC memory node, data width is 8 for endianess swap.\n");
		src_trwidth = 8;
	}
	else {
		src_trwidth = trwidth_reg2num(mobidma->ctl.strwidth);
	}
	if(mobidma->dst_node_type==DMA_NODE_TYPE_MEM) {	
		dprintk4("DST memory node, data width is 8 for endianess swap.\n");
		dst_trwidth = 8;
	}
	else {
		dst_trwidth = trwidth_reg2num(mobidma->ctl.dtrwidth);
	}
	dprintk4("src_trwidth %d, dst_trwidth %d, datawidth %d\n",
			src_trwidth, dst_trwidth, data_width);

	/*
	 *  no swapping if P2P or if both the same endianess or
	 *  if all widths are the same
	 */
	mobidma->endian_swap = DMA_ENDIANESS_SWAP_NONE;
	if (mobidma->transtype == DMA_TRANS_TYPE_P2P ||
			mobidma->src_endianess == mobidma->dst_endianess ||
			((src_trwidth == dst_trwidth) && (src_trwidth == data_width))) {
		dprintk3("mode 0x%x\n", mobidma->endian_swap);
		return 0;
	}
	/*
	 *  if we made to here, then something is big endian
	 *  will check if m2m when setting up the lli list and
	 *  set the bits in the correct address, otherwise, it's
	 *  dev and mem and we set the bits in the mem address
	 */
	if (src_trwidth == 32 && dst_trwidth == 32) {
		if (data_width == 8)
			mobidma->endian_swap = DMA_ENDIANESS_SWAP_MODE3;
		else if (data_width == 16)
			mobidma->endian_swap = DMA_ENDIANESS_SWAP_MODE2;
		else  // 32
			mobidma->endian_swap = DMA_ENDIANESS_SWAP_MODE0;
	}
	else if ((src_trwidth == 16 && dst_trwidth == 32) ||
			(src_trwidth == 32 && dst_trwidth == 16)) {
		if (data_width == 8)
			mobidma->endian_swap = DMA_ENDIANESS_SWAP_MODE1;
		if (data_width == 16)
			mobidma->endian_swap = DMA_ENDIANESS_SWAP_MODE0;
		else
			mobidma->endian_swap = DMA_ENDIANESS_SWAP_MODE2;
	}
	else if ((src_trwidth == 8 && dst_trwidth == 32) ||
			(src_trwidth == 32 && dst_trwidth == 8)) {
		if (data_width == 8)
			mobidma->endian_swap = DMA_ENDIANESS_SWAP_MODE0;
		if (data_width == 16)
			mobidma->endian_swap = DMA_ENDIANESS_SWAP_MODE2;
		else
			mobidma->endian_swap = DMA_ENDIANESS_SWAP_MODE3;
	}
	else if (((src_trwidth == 8  && dst_trwidth == 16) ||
				(src_trwidth == 16 && dst_trwidth == 8)) &&
			data_width == 8) {
		/* XXX uncertain */
		mobidma->endian_swap = DMA_ENDIANESS_SWAP_MODE2;
	}

	dprintk3("swap mode 0x%x, mmu endianess mask 0x%x\n",
			mobidma->endian_swap,
			mobidma->endian_swap << MMU_BRIDGE_ENDIAN_SWAP_SHIFT_L);

	return 0;
}
/*
 *  do some minimal checking before accessing a channel, make sure the
 *  channel is in the valid range and if so, verify that it has been
 *  allocated/requested
 */
static int32_t dma_validate_channel_access(mobi_dma_handle dmah, char *func)
{
	if (dmah < 0 || dmah >= MAX_DMAC_CHANNELS) {
		error("%s: invalid dma handle %d", func, dmah);
		return -ENODEV;
	}
	if (!mobi_dma_channels[dmah].name) {
		error("%s: channel %d has not been allocated",
				func, dmah);
		return -ENODEV;
	}
	return 0;
}

/*
 *  given the type(mem or per) of the src and dst and the
 *  type of flow control, figure out the ttfc register setting
 *  see page 167 of DMAC databook
 */
static int32_t ctl_ttfc_decode(uint32_t tt, uint32_t fc)
{

	if (tt == DMA_TRANS_TYPE_M2M && fc == DMA_FC_DMAC)
		return DMA_TTFC_M2M_DMAC;
	else if (tt == DMA_TRANS_TYPE_M2P && fc == DMA_FC_DMAC)
		return DMA_TTFC_M2P_DMAC;
	else if (tt == DMA_TRANS_TYPE_P2M && fc == DMA_FC_DMAC)
		return DMA_TTFC_P2M_DMAC;
	else if (tt == DMA_TRANS_TYPE_P2P && fc == DMA_FC_DMAC)
		return DMA_TTFC_P2P_DMAC;
	else if (tt == DMA_TRANS_TYPE_P2M && fc == DMA_FC_PER)
		return DMA_TTFC_P2M_PER;
	else if (tt == DMA_TRANS_TYPE_P2P && fc == DMA_FC_SRCP)
		return DMA_TTFC_P2P_SRCPER;
	else if (tt == DMA_TRANS_TYPE_M2P && fc == DMA_FC_PER)
		return DMA_TTFC_M2P_PER;
	else if (tt == DMA_TRANS_TYPE_P2P && fc == DMA_FC_DSTP)
		return DMA_TTFC_P2P_DSTPER;

	return -1;
}

static int32_t set_transfer_type(uint32_t src_node, uint32_t dst_node)
{
	if (src_node == DMA_NODE_TYPE_MEM && dst_node == DMA_NODE_TYPE_MEM)
		return DMA_TRANS_TYPE_M2M;
	if (src_node == DMA_NODE_TYPE_PER && dst_node == DMA_NODE_TYPE_MEM)
		return DMA_TRANS_TYPE_P2M;
	if (src_node == DMA_NODE_TYPE_MEM && dst_node == DMA_NODE_TYPE_PER)
		return DMA_TRANS_TYPE_M2P;
	if (src_node == DMA_NODE_TYPE_PER && dst_node == DMA_NODE_TYPE_PER)
		return DMA_TRANS_TYPE_P2P;

	return -1;
}

void reinit_interrupts(mobi_dma_handle dmah)
{
	struct mobi_dma_channel *mobidma = &mobi_dma_channels[dmah];

	/* re-init interrupts just to be sure we only get what we want */
	/* they are enabled at start transfer time */
	mask_channel_interrupts(dmah);
	clear_channel_interrupts(dmah);

	DMAC_WRITEL((1 << (dmah+8) | (1 << dmah)), MOBI_DMAC_MASK_TFR_OFFSET);
	DMAC_WRITEL((1 << (dmah+8) | (1 << dmah)), MOBI_DMAC_MASK_ERR_OFFSET);

#if 0
	DMAC_WRITEL((1 << (dmah+8) | (1 << dmah)), MOBI_DMAC_STATUS_BLOCK_OFFSET);
	DMAC_WRITEL((1 << (dmah+8) | (1 << dmah)), MOBI_DMAC_STATUS_SRC_TRAN_OFFSET);
	DMAC_WRITEL((1 << (dmah+8) | (1 << dmah)), MOBI_DMAC_STATUS_DST_TRAN_OFFSET);
#endif

	/* if sw handshaking enable those interrupts also */
	if (mobidma->cfg.shs_sel == DMA_SW_HANDSHAKING &&
			mobidma->src_node_type != DMA_NODE_TYPE_MEM) {
		DMAC_WRITEL((1 << (dmah+8) | (1 << dmah)),
				MOBI_DMAC_MASK_SRC_TRAN_OFFSET);
	}
	if (mobidma->cfg.dhs_sel == DMA_SW_HANDSHAKING &&
			mobidma->dst_node_type != DMA_NODE_TYPE_MEM) {
		DMAC_WRITEL((1 << (dmah+8) | (1 << dmah)),
				MOBI_DMAC_MASK_DST_TRAN_OFFSET);
	}
}

#ifdef CONFIG_PROC_FS
static int dma_state_proc_rd(char *buf, char **start,
		off_t offset, int count, int *eof, void *data)
{
	int len=0;
	uint64_t reg;
	mobi_dma_handle dmah;
	struct mobi_dma_channel *mobidma = NULL;

	for (dmah = 0; dmah < MAX_DMAC_CHANNELS; dmah++) {

		mobidma = &mobi_dma_channels[dmah];

		len+=sprintf(buf+len,"\nDMA Channel %d\n", dmah);
		len+=sprintf(buf+len,"--------------\n");
		len+=sprintf(buf+len,"Channel name:  ");
		if (strcmp(mobidma->name, "free") == 0) {
			len+=sprintf(buf+len,"not allocated...skipping\n");
			continue;
		}
		len+=sprintf(buf+len,"%s\n", mobidma->name);
		DMAC_READL(MOBI_DMAC_CHANNEL_OFFSET, reg);
		if (reg & (1 << dmah))
			len+=sprintf(buf+len,"Enabled: yes\n");
		else
			len+=sprintf(buf+len,"Enabled: no\n");

		len+=sprintf(buf+len,"Src Physical Address: 0x%x\n",
				mobidma->src_addr);
		len+=sprintf(buf+len,"Src DMA Address: 0x%x\n", mobidma->src_addr);
		len+=sprintf(buf+len,"Src width  : %d\n",
				trwidth_reg2num(mobidma->ctl.strwidth));
		len+=sprintf(buf+len,"Src msize  : %d\n",
				msize_reg2num(mobidma->ctl.smsize));
		len+=sprintf(buf+len,"Src node type  : %s\n",
				(mobidma->src_node_type == DMA_NODE_TYPE_MEM ?
				 "memory" : "peripheral"));
		len+=sprintf(buf+len,"Src handshake num : %d\n",
				mobidma->cfg.src_per);

		len+=sprintf(buf+len,"Dst Physical Address: 0x%x\n",
				mobidma->dst_addr);
		len+=sprintf(buf+len,"Dst DMA Address: 0x%x\n", mobidma->dst_addr);
		len+=sprintf(buf+len,"Dst width  : %d\n",
				trwidth_reg2num(mobidma->ctl.dtrwidth));
		len+=sprintf(buf+len,"Dst msize  : %d\n",
				msize_reg2num(mobidma->ctl.dmsize));
		len+=sprintf(buf+len,"Dst node type  : %s\n",
				(mobidma->dst_node_type == DMA_NODE_TYPE_MEM ?
				 "memory" : "peripheral"));
		len+=sprintf(buf+len,"Dst handshake num : %d\n",
				mobidma->cfg.dst_per);

		DMAC_READD(DMAC_CHREG_ADDR(dmah, CTL), reg);
		len+=sprintf(buf+len,"Control Register: 0x%llx\n", reg);
		DMAC_READD(DMAC_CHREG_ADDR(dmah, CFG), reg);
		len+=sprintf(buf+len,"Config  Register: 0x%llx\n", reg);

		len+=sprintf(buf+len, "\n\n");
	}
	len+=sprintf(buf+len, "\n");
	return len;
}

#if DW_DMAC_DEBUG
static void proc_advance_ptr(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	while(((**ptr==' ') || (**ptr=='\t') || (**ptr=='\r')
				|| (**ptr=='\n')) && (*ptr-buf<count)) {
		(*ptr)++;
	}
}

static int proc_get_next_arg(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	proc_advance_ptr(ptr, buffer, count);
	if (*ptr-buf >= count) {
		return -EINVAL;
	}
	return 0;
}

static int get_arg(char **ptr,
		uint32_t buffer, unsigned long count, uint32_t *arg)
{
	char *buf = (char *)buffer;
	proc_advance_ptr(ptr, buffer, count);
	if (*ptr-buf >= count) {
		printk("Invalid argument string\n");
		return -1;
	}
	else {
		*arg = simple_strtoul(*ptr, ptr, 0);
		if(*arg < 0) {
			error("invalid argument");
			return -1;
		}
	}
	return 0;
}

#define DEVICE_DBG_READ_REGISTER 	0x1
#define DEVICE_DBG_WRITE_REGISTER 	0x2
#define DEVICE_DBG_LOG_LEVEL	 	0x3
#define DEVICE_DBG_IO_BASE	 	0x4

static int mobidma_proc_wr_debug(struct file *file,
		const char *buffer, unsigned long count, void *data)
{
	char *ptr = (char *)buffer;
	uint32_t cmd = 0, arg1 = -1, arg2 = -1;
	uint32_t value;

	if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0) {
		cmd = simple_strtoul(ptr, &ptr, 0);
		if(cmd < 0) {
			error("Invalid debug command");
			goto err_help;
		}
	}
	else {
		error("Invalid cmd string");
		goto err_help;
	}

	//printk("debug: cmd: 0x%x, arg1: 0x%x(%d), arg2: 0x%x(%d)\n",
	//		cmd, arg1, arg1, arg2, arg2);
	switch(cmd) {
		case DEVICE_DBG_READ_REGISTER:
			get_arg(&ptr, (uint32_t)buffer, count, &arg1);
			value = readl(dmac_base + arg1);
			printk("Read at offset 0x%x is 0x%x(%d)\n",
					arg1, value, value);
			break;
		case DEVICE_DBG_WRITE_REGISTER:
			get_arg(&ptr, (uint32_t)buffer, count, &arg1);
			get_arg(&ptr, (uint32_t)buffer, count, &arg2);
			printk("Write 0x%x(%d) to offset 0x%x\n",
					arg2, arg2, arg1);
			writel(arg2, dmac_base + arg1);
			break;
		case DEVICE_DBG_LOG_LEVEL:
			get_arg(&ptr, (uint32_t)buffer, count, &loglevel);
			printk(KERN_INFO "DMAC setting debug level to %d\n",
					loglevel);
			break;
		case DEVICE_DBG_IO_BASE:
			printk(KERN_INFO "DMAC dmac_base = 0x%p\n", dmac_base);
			break;
		default:
			break;
	}
	while(ptr-buffer<count)
		ptr++;

	return ptr - buffer;

err_help:
#define PRINT_CMD(x)	#x
	printk("\nAvailable debug commands are(see code for details):\n");
	/* read everything we don't care about */
	while(ptr-buffer<count)
		ptr++;

	return -EINVAL;
}
#endif // DMAC_DEBUG
#endif // PROC_FS

/*
 *  this function looks the source address, determines if it is
 *  a device or memory and then sets some of the defines based
 *  on the identity of what is found
 *    things that need to be done in this manner are
 *    	- find the handshaking interface number
 *    	- is the source mem or devices
 *    	- will the source need to lock the master bus
 *    	- others?
 *
 *  WARNING:  A client is required to call config_src/dst before
 *  calling dma_setup_*.  The transfer width must be set for the transfer
 *  to proceed and we check that here to make sure config has been
 *  called
 */
static int32_t setup_src_defines(mobi_dma_handle dmah)
{
	struct mobi_dma_channel *mobidma = &mobi_dma_channels[dmah];

	uint32_t addr = mobidma->src_addr;
	int32_t ret = 0;

	mobidma->cfg.src_per = 0;

	/*
	 *  verify that config has been called on the source, fix those
	 *  defines that we think we can
	 */
	if (trwidth_reg2api(mobidma->ctl.strwidth) == -1) {
		error("Transaction width on channel %d "
				"src not set", dmah);
		return -EINVAL;
	}

	dprintk5("looking at address 0x%x\n", addr);
	/* now try to figure out what we are accessing */
	if (addr >= SDMMC_BASE && addr <= SDMMC_END) {
		mobidma->cfg.src_per = DMAC_HANDSHAKE_INTERFACE_1;
		mobidma->src_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.sms = DMAC_AHB_MASTER_1;
	}
	else if (addr >= AES_BASE && addr <= AES_END) {
		mobidma->cfg.src_per = DMAC_HANDSHAKE_INTERFACE_2;
		mobidma->src_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.sms = DMAC_AHB_MASTER_1;
	}
	else if (addr >= SHA_BASE && addr <= SHA_END) {
		mobidma->cfg.src_per = DMAC_HANDSHAKE_INTERFACE_3;
		mobidma->src_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.sms = DMAC_AHB_MASTER_1;
	}
	else if (addr >= FLASH_DATA_BASE && addr <= FLASH_DATA_END) {
		mobidma->src_node_type = DMA_NODE_TYPE_PERNOFC;
		mobidma->ctl.sms = DMAC_AHB_MASTER_1;
	}
	else if (addr >= SSI0_BASE && addr <= SSI0_END) {
		mobidma->cfg.src_per = DMAC_HANDSHAKE_INTERFACE_5;
		mobidma->src_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.sms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= SSI1_BASE && addr <= SSI1_END) {
		mobidma->cfg.src_per = DMAC_HANDSHAKE_INTERFACE_7;
		mobidma->src_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.sms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= SSI2_BASE && addr <= SSI2_END) {
		mobidma->cfg.src_per = DMAC_HANDSHAKE_INTERFACE_7;
		mobidma->src_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.sms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= I2C0_BASE && addr <= I2C0_END) {
		mobidma->cfg.src_per = DMAC_HANDSHAKE_INTERFACE_13;
		mobidma->src_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.sms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= I2C1_BASE && addr <= I2C1_END) {
		mobidma->cfg.src_per = DMAC_HANDSHAKE_INTERFACE_15;
		mobidma->src_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.sms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= UART0_BASE && addr <= UART0_END) {
		mobidma->cfg.src_per = DMAC_HANDSHAKE_INTERFACE_9;
		mobidma->src_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.sms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= UART1_BASE && addr <= UART1_END) {
		mobidma->cfg.src_per = DMAC_HANDSHAKE_INTERFACE_11;
		mobidma->src_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.sms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= UARTDBG_BASE && addr <= UARTDBG_END) {
		mobidma->cfg.src_per = DMAC_HANDSHAKE_INTERFACE_3;
		mobidma->src_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.sms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= QCC_BASE && addr <= QCC_END) {
		/* QCCSISC behave like the Aurora Flash controller:
		 *  no fc but can stall AHB */
		mobidma->src_node_type = DMA_NODE_TYPE_PERNOFC;
		mobidma->ctl.sms = DMAC_AHB_MASTER_1;
	}
	else  {
		/* if it's not a device it must be memory, should check that
		 *  this address is within our valid address range for physical
		 *  addresses
		 */
		mobidma->ctl.sms = DMAC_AHB_MASTER_3;
		mobidma->src_node_type = DMA_NODE_TYPE_MEM;
	}

	return ret;
}

/*
 *  this function looks the destination address, determines if it is
 *  a device or memory and then sets some of the defines based
 *  on the identity of what is found
 *    things that need to be done in this manner are
 *    	- find the handshaking interface number
 *    	- is the destination mem or devices
 *    	- will the destination need to lock the master bus
 *    	- others?
 *
 *  WARNING:  A client is required to call config_src/dst before
 *  calling dma_setup_*.  The transfer width must be set for the transfer
 *  to proceed and we check that here to make sure config has been
 *  called
 *    	
 */
static int32_t setup_dst_defines(mobi_dma_handle dmah)
{
	struct mobi_dma_channel *mobidma = &mobi_dma_channels[dmah];
	uint32_t addr = mobidma->dst_addr;
	int32_t ret = 0;

	mobidma->cfg.dst_per = 0;

	/*
	 *  verify that config has been called on the destination, fix those
	 *  defines that we think we can
	 */
	if (trwidth_reg2api(mobidma->ctl.dtrwidth) == -1) {
		error("Transaction width on channel %d "
				"destination not set", dmah);
		return -EINVAL;
	}
	dprintk5("looking at address 0x%x\n", addr);
	if (addr >= SDMMC_BASE && addr <= SDMMC_END) {
		mobidma->cfg.dst_per = DMAC_HANDSHAKE_INTERFACE_0;
		mobidma->dst_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.dms = DMAC_AHB_MASTER_1;
	}
	else if (addr >= AES_BASE && addr <= AES_END) {
		mobidma->cfg.dst_per = DMAC_HANDSHAKE_INTERFACE_2;
		mobidma->dst_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.dms = DMAC_AHB_MASTER_1;
	}
	else if (addr >= SHA_BASE && addr <= SHA_END) {
		mobidma->cfg.dst_per = DMAC_HANDSHAKE_INTERFACE_3;
		mobidma->dst_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.dms = DMAC_AHB_MASTER_1;
	}
	else if (addr >= FLASH_DATA_BASE && addr <= FLASH_DATA_END) {
		mobidma->dst_node_type = DMA_NODE_TYPE_PERNOFC;
		mobidma->ctl.dms = DMAC_AHB_MASTER_1;
	}
	else if (addr >= SSI0_BASE && addr <= SSI0_END) {
		mobidma->cfg.dst_per = DMAC_HANDSHAKE_INTERFACE_4;
		mobidma->dst_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.dms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= SSI1_BASE && addr <= SSI1_END) {
		mobidma->cfg.dst_per = DMAC_HANDSHAKE_INTERFACE_6;
		mobidma->dst_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.dms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= SSI2_BASE && addr <= SSI2_END) {
		mobidma->cfg.dst_per = DMAC_HANDSHAKE_INTERFACE_6;
		mobidma->dst_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.dms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= I2C0_BASE && addr <= I2C0_END) {
		mobidma->cfg.dst_per = DMAC_HANDSHAKE_INTERFACE_12;
		mobidma->dst_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.dms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= I2C1_BASE && addr <= I2C1_END) {
		mobidma->cfg.dst_per = DMAC_HANDSHAKE_INTERFACE_14;
		mobidma->dst_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.dms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= UART0_BASE && addr <= UART0_END) {
		mobidma->cfg.dst_per = DMAC_HANDSHAKE_INTERFACE_8;
		mobidma->dst_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.dms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= UART1_BASE && addr <= UART1_END) {
		mobidma->cfg.dst_per = DMAC_HANDSHAKE_INTERFACE_10;
		mobidma->dst_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.dms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= UARTDBG_BASE && addr <= UARTDBG_END) {
		mobidma->dst_node_type = DMA_NODE_TYPE_PER;
		mobidma->ctl.dms = DMAC_AHB_MASTER_2;
	}
	else if (addr >= QCC_BASE && addr <= QCC_END) {
		/* QCCSISC behave like the Aurora Flash controller: no fc but can stall AHB */
		mobidma->dst_node_type = DMA_NODE_TYPE_PERNOFC;
		mobidma->ctl.dms = DMAC_AHB_MASTER_1;
	}
	else  {
		/* if it's not a device it must be memory, should check that
		 *  this address is within our valid address range for physical
		 *  addresses
		 */
		mobidma->ctl.dms = DMAC_AHB_MASTER_3;
		mobidma->dst_node_type = DMA_NODE_TYPE_MEM;
	}

	return ret;
}

/**
 * \brief mobi_dma_get_max_blk_size:
 * 	Returns the maximum size for a single dma transaction
 *
 * \param dmah  : DMA handler
 * \param status: pointer to mobi_dma_status will will be
 * 	           updated with current status flags
 *
 * \retval -EINVAL - if invalid dma_handle provided
 * \retval -ENODEV - if dma_handle has not been allocated
 * \retval maximum size in bytes
 *
 * \remark
 * 	 In this case, a single DMA transaction, is defined from
 * 	the software's perspective.  The hardware will break these
 * 	transactions into much smaller ones on the bus.  Examples of
 * 	a single transaction would be a single buffer(less than
 * 	max_blk_size) sent to mobi_dma_single or a single node in a
 * 	mobi_dma_list or scatter/gather list. DMA request must be
 * 	checked that the size requested is not greater than the max
 * 	and if it is, the DMAC driver will have to break this into
 * 	multiple transactions.  The value returned from this function
 * 	can be used to check the size of your transfer or list nodes.
 *
 */
int32_t mobi_dma_get_max_blk_size(mobi_dma_handle dmah)
{
	int ret = 0;
	struct mobi_dma_channel *mobidma = NULL;

	if ((ret = dma_validate_channel_access(dmah, (char*)__FUNCTION__)) != 0)
		return ret;

	mobidma = &mobi_dma_channels[dmah];
	return((MAX_DMA_BLOCK_SIZE*trwidth_reg2num(mobidma->ctl.strwidth))/8);
}
EXPORT_SYMBOL(mobi_dma_get_max_blk_size);

/**
 * \brief mobi_dma_get_bytes:
 * 	Returns the number of bytes transfered so far.
 *
 * \param dmah  : DMA handler
 * \param status: pointer to mobi_dma_status will will be
 * 	           updated with current status flags
 *
 * \retval -EINVAL - if invalid dma_handle provided
 * \retval -ENODEV - if dma_handle has not been allocated
 * \retval number of bytes transfer
 *
 * \remark
 *   Will returned the number of bytes transfered at the
 * time this function is called.  If the transfer is still
 * running the current total will be returned or if transfer
 * has completed will return the total number of bytes
 * transfer.  function can be called at any time but is
 * best called after calling mobi_dma_disable.  calling this
 * function while a multiblock transfer is running or from
 * an event handler when doing multiple dma transfers
 * is not recommended due to the method that must be used
 * to calculate the running total(it could be slow for large
 * list)
 *
 */
int32_t mobi_dma_get_bytes(mobi_dma_handle dmah)
{
	struct mobi_dma_channel *mobidma = NULL;
	uint64_t ctl_reg;
	int32_t ret = 0, bytes_transfered = 0;
	void __iomem *plli = NULL;
	uint64_t bctl = 0;
	uint32_t block_ts, trwidth;
	int i, done, bytes;

	if ((ret = dma_validate_channel_access(dmah, (char*)__FUNCTION__)) != 0)
		return ret;

	mobidma = &mobi_dma_channels[dmah];
	DMAC_READD(DMAC_CHREG_ADDR(dmah, CTL), ctl_reg);

	if (mobidma->list_entries != 0 && mobidma->lli_iomap != NULL) {
		/* we only get one xfer complete interrupt at the end of
		 *  a multiblock transfer, but writeback is enabled so we
		 *  can read back the cnt from the lli list in dtcm and
		 *  extract the important parts and show some stats
		 */
		for (i=0;i<mobidma->list_entries;i++) {
			done = 0;
			plli = mobidma->lli_iomap + (i*LLI_BLOCK_DESCRIPTOR_SIZE);
			bctl =
				((uint64_t)readl((uint32_t)(plli+LLI_CTLH_OFFSET))<< 32
				 | readl((uint32_t)(plli+LLI_CTLL_OFFSET)));

			block_ts = MOBI_DMAC_CH0_CTL_BLOCK_TS_R(bctl);
			done = MOBI_DMAC_CH0_CTL_DONE_R(bctl);
			trwidth = trwidth_reg2num(MOBI_DMAC_CH0_CTL_SRC_TR_WIDTH_R(bctl));
			bytes = ((block_ts*trwidth)/8);
			dprintk5("block %d, done: %d, bytes: %d\n", i, done, bytes);
			if (done == 1)
				bytes_transfered += bytes;
		}
	}
	else {
		/* a single block transfer */
		DMAC_READD(DMAC_CHREG_ADDR(dmah, CTL), bctl);
		trwidth = trwidth_reg2num(MOBI_DMAC_CH0_CTL_SRC_TR_WIDTH_R(bctl));
		bytes_transfered =
			(int32_t)MOBI_DMAC_CH0_CTL_BLOCK_TS_R(ctl_reg)*(trwidth/8);
	}
	dprintk4("total bytes successfully transfered %d\n", bytes_transfered);
	return bytes_transfered;
}
EXPORT_SYMBOL(mobi_dma_get_bytes);

/**
 * \brief mobi_dma_get_status:
 * 	Returns the status of the channel
 *
 * \param dmah  : DMA handler
 *
 * \retval EINVAL - if invalid dma_handle provided
 * \retval ENODEV - if dma_handle has not been allocated
 * \retval mobi_dma_status  - current status of channel/transfer
 *
 *  \remark
 *  \verbatim
 *  if channel is valid return values are
 *   	MOBI_DMA_STATUS_XFER_RUNNING
 *   		DMA transfer is currently running
 *   	MOBI_DMA_STATUS_XFER_IDLE
 *   		The DMA controller has completed the transfer.
 *   		It is possible to retrieve an idle status before
 *   		receiving an interrupt.  this could happen in the
 *   		brief time that the driver is doing post-processing
 *   		after the xfer complete and before sending the
 *   		interrupt.
 *   	MOBI_DMA_STATUS_CHANNEL_ENABLED
 *   		DMA channel has been enabled via mobi_dma_enable
 *   	MOBI_DMA_STATUS_CHANNEL_DISABLED
 *   		DMA channel has been disabled via mobi_dma_disable
 *   	MOBI_DMA_STATUS_ERROR
 *   		An error has occurred during the tranfer.
 *  \endverbatim
 *
 */
mobi_dma_status mobi_dma_get_status(mobi_dma_handle dmah)
{
	struct mobi_dma_channel *mobidma = NULL;
	uint64_t ctl_reg = 0;
	uint32_t chen_reg = 0;
	int32_t status = 0;

	if ((status = dma_validate_channel_access(dmah, (char*)__FUNCTION__)) != 0)
		return status;

	mobidma = &mobi_dma_channels[dmah];

	DMAC_READL(MOBI_DMAC_CHANNEL_OFFSET, chen_reg);
	DMAC_READD(DMAC_CHREG_ADDR(dmah, CTL), ctl_reg);

	if (mobidma->enabled)
		status = MOBI_DMA_STATUS_CHANNEL_ENABLED;
	else
		status = MOBI_DMA_STATUS_CHANNEL_DISABLED;

	if (chen_reg & (1 << dmah))
		status |= MOBI_DMA_STATUS_XFER_RUNNING;
	else
		status |= MOBI_DMA_STATUS_XFER_IDLE;

	if (mobidma->error)
		status |= MOBI_DMA_STATUS_ERROR;

	return status;
}
EXPORT_SYMBOL(mobi_dma_get_status);

/*
 *  internal func used to set src flags:
 *  	MOBI_DMA_CONFIG_TRANSFER_WIDTH_*
 *  	MOBI_DMA_CONFIG_BURST_SIZE_*
 *  	MOBI_DMA_CONFIG_HANDSHAKE_*
 *  	MOBI_DMA_CONFIG_ADDRADJ*
 *  	MOBI_DMA_CONFIG_ENDIAN_*
 *  	MOBI_DMA_CONFIG_SG_ENABLE
		MOBI_DMA_CONFIG_RELOAD_START_ADDR_MASK
 *
 *  For details about these flags see mobi_dma_config
 *
 */
static int32_t dma_config_src(struct mobi_dma_channel *mobidma,
		uint32_t flags, struct mg_dma_config_data *cfg_data)
{

	int32_t ret = 0;
	uint32_t legal_flags =
		MOBI_DMA_CONFIG_TRANSFER_WIDTH_MASK |
		MOBI_DMA_CONFIG_ENDIAN_MASK			|
		MOBI_DMA_CONFIG_BURST_SIZE_MASK		|
		MOBI_DMA_CONFIG_HANDSHAKE_MASK		|
		MOBI_DMA_CONFIG_ADDRADJ_MASK		|
		MOBI_DMA_CONFIG_SG_ENABLE_MASK      |
		MOBI_DMA_CONFIG_RELOAD_START_ADDR_MASK;

#if DW_DMAC_DEBUG
	mobi_dma_handle dmah = mobidma->channel;
#endif


	dprintk3("mobidma->src_flags 0x%08x, setting flags = 0x%08x\n",
			mobidma->src_flags, flags);

	/* return error if a non-dst legal flag is set */
	if (flags & ~(legal_flags))
		return -EINVAL;

	if (flags & MOBI_DMA_CONFIG_TRANSFER_WIDTH_MASK) {
		if (dma_set_trwidth(mobidma,
					flags & MOBI_DMA_CONFIG_TRANSFER_WIDTH_MASK, DMA_NODE_IS_SRC))
			return -EINVAL;

		mobidma->src_flags &= ~MOBI_DMA_CONFIG_TRANSFER_WIDTH_MASK;
	}
	if (flags & MOBI_DMA_CONFIG_ENDIAN_MASK) {
		if (flags & MOBI_DMA_CONFIG_ENDIAN_LE)
			mobidma->src_endianess = MOBI_DMA_CONFIG_ENDIAN_LE;
		else if (flags & MOBI_DMA_CONFIG_ENDIAN_BE)
			mobidma->src_endianess = MOBI_DMA_CONFIG_ENDIAN_BE;
		else
			return -EINVAL;

		mobidma->src_flags &= ~MOBI_DMA_CONFIG_ENDIAN_MASK;
	}
	if (flags & MOBI_DMA_CONFIG_BURST_SIZE_MASK) {
		if (dma_set_msize(mobidma,
					flags & MOBI_DMA_CONFIG_BURST_SIZE_MASK, DMA_NODE_IS_SRC))
			return -EINVAL;

		mobidma->src_flags &= ~MOBI_DMA_CONFIG_BURST_SIZE_MASK;
	}
	if (flags & MOBI_DMA_CONFIG_HANDSHAKE_MASK) {
		if (flags & MOBI_DMA_CONFIG_HANDSHAKE_HW)
			mobidma->cfg.shs_sel = DMA_HW_HANDSHAKING;
		else if (flags & MOBI_DMA_CONFIG_HANDSHAKE_SW)
			mobidma->cfg.shs_sel = DMA_SW_HANDSHAKING;
		else
			return -EINVAL;
		mobidma->src_flags &= ~MOBI_DMA_CONFIG_HANDSHAKE_MASK;
	}
	if (flags & MOBI_DMA_CONFIG_ADDRADJ_MASK) {
		if (dma_set_addradj(mobidma,
					flags & MOBI_DMA_CONFIG_ADDRADJ_MASK, DMA_NODE_IS_SRC))
			return -EINVAL;

		mobidma->src_flags &= ~MOBI_DMA_CONFIG_ADDRADJ_MASK;
	}
	if (flags & MOBI_DMA_CONFIG_SG_ENABLE_MASK) {
		if (cfg_data == NULL) {
			return -EINVAL;
		}
		else {
			mobidma->src_sgparams.count = cfg_data->sg_params.count;
			mobidma->src_sgparams.interval = cfg_data->sg_params.interval;
		}
		mobidma->src_flags &= ~MOBI_DMA_CONFIG_SG_ENABLE_MASK;
	}
	if (flags & MOBI_DMA_CONFIG_RELOAD_START_ADDR_MASK) {
		mobidma->src_flags &= ~MOBI_DMA_CONFIG_RELOAD_START_ADDR_MASK;
	}

	/* now save them */
	mobidma->src_flags |= flags;
	dprintk3("new mobidma->src_flags 0x%08x\n", mobidma->src_flags);
	return ret;
}

/*
 *  internal func used to set dst flags:
 *  	MOBI_DMA_CONFIG_TRANSFER_WIDTH_*
 *  	MOBI_DMA_CONFIG_BURST_SIZE_*
 *  	MOBI_DMA_CONFIG_HANDSHAKE_*
 *  	MOBI_DMA_CONFIG_ADDRADJ*
 *  	MOBI_DMA_CONFIG_ENDIAN_*
 *  	MOBI_DMA_CONFIG_SG_ENABLE
 * 		MOBI_DMA_CONFIG_RELOAD_START_ADDR_MASK
 *
 *  For details about these flags see mobi_dma_config
 *
 */
static int32_t dma_config_dst(struct mobi_dma_channel *mobidma,
		uint32_t flags, struct mg_dma_config_data *cfg_data)
{

	int32_t ret = 0;
	uint32_t legal_flags =
		MOBI_DMA_CONFIG_TRANSFER_WIDTH_MASK	|
		MOBI_DMA_CONFIG_ENDIAN_MASK			|
		MOBI_DMA_CONFIG_BURST_SIZE_MASK		|
		MOBI_DMA_CONFIG_HANDSHAKE_MASK		|
		MOBI_DMA_CONFIG_ADDRADJ_MASK		|
		MOBI_DMA_CONFIG_SG_ENABLE_MASK      |
		MOBI_DMA_CONFIG_RELOAD_START_ADDR_MASK;

#if DW_DMAC_DEBUG
	mobi_dma_handle dmah = mobidma->channel;
#endif
	dprintk3("mobidma->dst_flags 0x%08x, setting flags = 0x%08x\n",
			mobidma->dst_flags, flags);

	/* return error if a non-dst legal flag is set */
	if (flags & ~(legal_flags))
		return -EINVAL;

	if (flags & MOBI_DMA_CONFIG_TRANSFER_WIDTH_MASK) {
		if (dma_set_trwidth(mobidma,
					flags & MOBI_DMA_CONFIG_TRANSFER_WIDTH_MASK, DMA_NODE_IS_DST))
			return -EINVAL;

		mobidma->dst_flags &= ~MOBI_DMA_CONFIG_TRANSFER_WIDTH_MASK;
	}
	if (flags & MOBI_DMA_CONFIG_ENDIAN_MASK) {
		if (flags & MOBI_DMA_CONFIG_ENDIAN_LE)
			mobidma->dst_endianess = MOBI_DMA_CONFIG_ENDIAN_LE;
		else if (flags & MOBI_DMA_CONFIG_ENDIAN_BE)
			mobidma->dst_endianess = MOBI_DMA_CONFIG_ENDIAN_BE;
		else
			return -EINVAL;

		mobidma->dst_flags &= ~MOBI_DMA_CONFIG_ENDIAN_MASK;
	}
	if (flags & MOBI_DMA_CONFIG_BURST_SIZE_MASK) {
		if (dma_set_msize(mobidma,
					flags & MOBI_DMA_CONFIG_BURST_SIZE_MASK, DMA_NODE_IS_DST))
			return -EINVAL;

		mobidma->dst_flags &= ~MOBI_DMA_CONFIG_BURST_SIZE_MASK;
	}
	if (flags & MOBI_DMA_CONFIG_HANDSHAKE_MASK) {
		if (flags & MOBI_DMA_CONFIG_HANDSHAKE_HW)
			mobidma->cfg.dhs_sel = DMA_HW_HANDSHAKING;
		else if (flags & MOBI_DMA_CONFIG_HANDSHAKE_SW)
			mobidma->cfg.dhs_sel = DMA_SW_HANDSHAKING;
		else
			return -EINVAL;

		mobidma->dst_flags &= ~MOBI_DMA_CONFIG_HANDSHAKE_MASK;
	}
	if (flags & MOBI_DMA_CONFIG_ADDRADJ_MASK) {
		if (dma_set_addradj(mobidma,
					flags & MOBI_DMA_CONFIG_ADDRADJ_MASK, DMA_NODE_IS_DST))
			return -EINVAL;

		mobidma->dst_flags &= ~MOBI_DMA_CONFIG_ADDRADJ_MASK;
	}
	if (flags & MOBI_DMA_CONFIG_SG_ENABLE_MASK) {
		if (cfg_data == NULL) {
			return -EINVAL;
		}
		else {
			mobidma->dst_sgparams.count = cfg_data->sg_params.count;
			mobidma->dst_sgparams.interval = cfg_data->sg_params.interval;
		}
		mobidma->dst_flags &= ~MOBI_DMA_CONFIG_SG_ENABLE_MASK;
	}
	if (flags & MOBI_DMA_CONFIG_RELOAD_START_ADDR_MASK) {
		mobidma->dst_flags &= ~MOBI_DMA_CONFIG_RELOAD_START_ADDR_MASK;
	}

	/* save all the flags that have been set */
	mobidma->dst_flags |= flags;
	dprintk3("new mobidma->dst_flags 0x%08x\n", mobidma->dst_flags);

	return ret;
}

/*
 *  internal func used to set xfer flags:
 *  	MOBI_DMA_CONFIG_DATA_WIDTH_*
 *
 *  For details about these flags see mobi_dma_config
 *
 */
static int32_t dma_config_xfer(struct mobi_dma_channel *mobidma,
		uint32_t flags, struct mg_dma_config_data *cfg_data)
{
	uint32_t legal_flags =
		MOBI_DMA_CONFIG_DATA_WIDTH_MASK |
		MOBI_DMA_CONFIG_MAX_BLK_SIZE_MASK;

#if DW_DMAC_DEBUG
	mobi_dma_handle dmah = mobidma->channel;
#endif
	dprintk3("flags = 0x%x\n", flags);

	if (flags & ~(legal_flags))
		return -EINVAL;

	if (flags & MOBI_DMA_CONFIG_DATA_WIDTH_MASK) {
		if (flags & MOBI_DMA_CONFIG_DATA_WIDTH_8)
			mobidma->datawidth = 8;
		else if (flags & MOBI_DMA_CONFIG_DATA_WIDTH_16)
			mobidma->datawidth = 16;
		else if (flags & MOBI_DMA_CONFIG_DATA_WIDTH_32)
			mobidma->datawidth = 32;
		else
			return -EINVAL;

		mobidma->xfer_flags &= ~MOBI_DMA_CONFIG_DATA_WIDTH_MASK;
	}
	if (flags & MOBI_DMA_CONFIG_MAX_BLK_SIZE_MASK) {
		if (cfg_data == NULL) {
			dprintk3("max_blk_size data is null\n");
			return -EINVAL;
		}
		else {
			dprintk3("requested max_blk_size %d, allowed max_blk_size %d\n",
				cfg_data->max_blk_size,
					((MAX_DMA_BLOCK_SIZE*trwidth_reg2num(mobidma->ctl.strwidth))/8));
			if (cfg_data->max_blk_size >
					((MAX_DMA_BLOCK_SIZE *
					  trwidth_reg2num(mobidma->ctl.strwidth))/8))
				return -EINVAL;

			else
				mobidma->max_blk_size = cfg_data->max_blk_size;
		}
		mobidma->xfer_flags &= ~MOBI_DMA_CONFIG_MAX_BLK_SIZE_MASK;
	}

	mobidma->xfer_flags |= flags;
	return(0);
}

/**
 * \brief mobi_dma_config
 * 	Configure the dma channel for transfer.  flags can apply to
 * three differerent "areas", src, dst and xfer.  apply flags to the
 * src and dst that explicitly define that interface.  For example, the
 * transfer_width of a device. xfer flags apply to the transfer as a
 * whole and is not necassarily something that specifically applies to
 * the source or destination.  In addition, some flags need some
 * addition parameters to be defined in order to apply the flag.  This
 * function is used to set the appropriate flags to configure the transfer.
 *
 * 	\param dmah  - DMA channel handle
 * 	\param what  - what are we configuring
 * 	\param flags - configuration flags
 * 	\param data  - void* pointer to data associated with a flag
 *
 * 	These flags apply when what is DMA_CONFIG_SRC or DMA_CONFIG_DST
 * 	    - MOBI_DMA_CONFIG_TRANSFER_WIDTH_*    - MUST be set
 * 	    	- 8, 16 or 32, for devices typically FIFO width
 * 	    - MOBI_DMA_CONFIG_BURST_SIZE_*	   - default of 8
 * 	    	- 1, 4, 8, 16, 32, burst transaction length. the
 * 	    	number of items of size 'WIDTH' to be used for each
 * 	    	burst transaction
 * 	    - MOBI_DMA_CONFIG_HANDSHAKE_*   	   - default is hw
 * 	    	- HW, SW, define the handshaking interface
 * 	    - MOBI_DMA_CONFIG_ADDRADJ	   	   - default increment	
 * 	    	- inc, dec, none, on devices, could be accessing a
 * 	    	FIFO or some type of buffer.  If access is not to
 * 	    	a fixed address, then need to know how to inc/dec
 * 	    	the device address.  Devices with FIFOs should set this
 * 	    	to NONE.
 * 	    - MOBI_DMA_CONFIG_ENDIAN_*
 * 	    	- set the endianess of a device, BE or LE.  Default
 * 	    	is LE. Cannont be set for mem->mem or dev->dev transfers
 * 	    	How the bytes will be swapped will depend on the width
 * 	    	of the src, dest and data width.
 *
 *
 * 	These flags apply when what is DMA_CONFIG_XFER
 * 	    - MOBI_DMA_CONFIG_SG_ENABLE
 * 	    	- this flags is used to define the params needed for
 * 	    	a scatter/gather transfer.  data should point to a
 * 	    	struct sg_params variable that has been initialized.  if
 * 	    	doing a single block transfer, this flag will enable s/g
 * 	    	for the transfer.  if doing a multiblock transfer, this
 * 	    	flag only defines the params.  to enable s/g for a entry
 * 	    	in a mobi_dma_list, set the appropriate src/dst_flag.
 * 	    - MOBI_DMA_CONFIG_DATA_WIDTH_*
 * 	    	- set the size of data element to be operated upon.
 * 	    	this value is used when determining how data will
 * 	    	be swapped when the src & dst are of different
 * 	    	endianess.  For example, a 32-bit src & dst with
 * 	    	different endianess should have a data width of
 * 	    	8 which would result in B3B2B1B0 <-> B0B1B2B3
 *
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval -EINVAL - if invalid flags has been set
 * \retval Zero    - upon success
 *
 * \remark
 *   these settings are persistant until the channel is released.  when
 *  using a mobi_dma_list values set here will be the default if no
 *  flags have been set in a mobi_dma_list nodes flag fields.  this
 *  function can be called multiple times with different flags and data.
 *
 */
int32_t mobi_dma_config(mobi_dma_handle dmah, uint8_t what,
		uint32_t flags, struct mg_dma_config_data *cfg_data)
{
	struct mobi_dma_channel *mobidma = NULL;
	int32_t ret = 0;

	if ((ret = dma_validate_channel_access(dmah, (char*)__FUNCTION__)) != 0)
		return ret;

	mobidma = &mobi_dma_channels[dmah];

	switch (what) {
		case DMA_CONFIG_SRC:
			ret = dma_config_src(mobidma, flags, cfg_data);
			break;
		case DMA_CONFIG_DST:
			ret = dma_config_dst(mobidma, flags, cfg_data);
			break;
		case DMA_CONFIG_XFER:
			ret = dma_config_xfer(mobidma, flags, cfg_data);
			break;
		default:
			ret = -EINVAL;
	}
	return(ret);
}
EXPORT_SYMBOL(mobi_dma_config);

#define DMA_ALIGNMENT_WORD	0x0
#define DMA_ALIGNMENT_SHORT	0x1
#define DMA_ALIGNMENT_BYTE	0x3
static uint8_t get_alignment_mask(struct mobi_dma_channel *mobidma)
{
	if (((mobidma->src_node_type == DMA_NODE_TYPE_MEM) &&
				(trwidth_reg2num(mobidma->ctl.dtrwidth) == 8))  ||
			((mobidma->dst_node_type == DMA_NODE_TYPE_MEM) &&
			 (trwidth_reg2num(mobidma->ctl.strwidth) == 8)) ||
			(mobidma->src_node_type == DMA_NODE_TYPE_MEM &&
			 mobidma->dst_node_type == DMA_NODE_TYPE_MEM)) {

		return(DMA_ALIGNMENT_BYTE);
	}
	else if (((mobidma->src_node_type == DMA_NODE_TYPE_MEM) &&
				(trwidth_reg2num(mobidma->ctl.dtrwidth) == 16)) ||
			((mobidma->dst_node_type == DMA_NODE_TYPE_MEM) &&
			 (trwidth_reg2num(mobidma->ctl.strwidth) == 16))) {

		return(DMA_ALIGNMENT_SHORT);
	}

	return(DMA_ALIGNMENT_WORD);
}

/*
 * if the list will fit in a 4K chunk of the DESC_RAM, then try to grab a
 *  region in the DESC_RAM.  If can't get a region or list won't fit in 4K,
 *  use RAM
 */
static int create_lli_mapping(mobi_dma_handle dmah,
		struct mobi_dma_channel *mobidma, int nents)
{

	mobidma->lli_size = nents*LLI_BLOCK_DESCRIPTOR_SIZE;
	mobidma->lli_phys = 0;


#ifndef CONFIG_DMAC_DESC_RAM_DISABLE

	if (mobidma->lli_size <= SZ_4K) {

		mobidma->lli_phys  = DESC_RAM_BASE+(MAX_DESC_RAM_LLI_SIZE*dmah);
		mobidma->lli_iomap = dtcm_iobase+(MAX_DESC_RAM_LLI_SIZE*dmah);

		dprintk3("Use DESC_RAM for lli, iomap: 0x%p, phys: 0x%x, size %d\n",
				mobidma->lli_iomap, mobidma->lli_phys,
				mobidma->lli_size);
		/* tells the dmac where the llp is, in the DESC_RAM on master1 */
		DMAC_WRITEL((MOBI_DMAC_CH0_LLP_LMS_W(DMAC_AHB_MASTER_1) |
					MOBI_DMAC_CH0_LLP_LOC_W(
						(((uint32_t)mobidma->lli_phys) >> 2))),
				DMAC_CHREG_ADDR(dmah, LLP));

		// skip this for now */
		//memset(mobidma->lli_iomap, 0x0, mobidma->lli_size);
	}
	else {
#endif

	mobidma->lli_iomap = kzalloc(mobidma->lli_size, GFP_ATOMIC);
	if (mobidma->lli_iomap == NULL)
		return -ENOMEM;

	mobidma->lli_phys = (uint32_t)mobidma->lli_iomap;
	dprintk1("Use kmalloc for lli, iomap: 0x%p, phys: 0x%x\n",
			mobidma->lli_iomap, mobidma->lli_phys);

	/* tells the dmac where the llp is */
	DMAC_WRITEL((MOBI_DMAC_CH0_LLP_LMS_W(DMAC_AHB_MASTER_3) |
				MOBI_DMAC_CH0_LLP_LOC_W(
					(((uint32_t)mobidma->lli_phys) >> 2))),
			DMAC_CHREG_ADDR(dmah, LLP));

#ifndef CONFIG_DMAC_DESC_RAM_DISABLE
	}
#endif

	return(0);
}

static void free_lli_mapping(struct mobi_dma_channel *mobidma)
{
	if (mobidma->lli_iomap != NULL) {
		mobidma->list_entries = 0;
		if (mobidma->lli_phys >= DESC_RAM_BASE && mobidma->lli_phys < DESC_RAM_END) {
			dprintk4("Free DESC_RAM for lli, iomap: 0x%p, phys: 0x%x, size %d\n",
					mobidma->lli_iomap, mobidma->lli_phys, mobidma->lli_size);
			mobidma->lli_iomap = 0;
		}
		else {
			dprintk1("Free RAM for lli, iomap: 0x%p\n", mobidma->lli_iomap);
			kfree(mobidma->lli_iomap);
			mobidma->lli_iomap = 0;
		}
	}
}

/*
 *  We need  to look at the all the entries in a dma list because not every
 *  entry will directly map to the dmac linked list.  We need to find these
 *  entries and determine how many entries the dmac's linked list will have.
 *  The following two criterea determine if a mobi_dma_list entry will be
 *  broken down into multiple dma lli entries.
 *    First, if a list entry is greater than the max dma size, then it
 *  must be broken into max dma sized chunks.
 *    Second, if doing memory access that is not 32 bits aligned, then
 *  we must create transactions that are either 32bit aligned or single byte
 *  transactions.  For example, 67 bytes to an 8 bit device would be 1 64
 *  byte dma and then 3 single bytes dmas.
 *
 */
int calc_num_list_entries(struct mobi_dma_list *pmlist,
		uint8_t alignment_mask, uint32_t max_blk_size)
{

	int32_t i, new_nents = 0;
	uint32_t max_size_nents = 0;

	for (i=0;pmlist != NULL;pmlist = pmlist->next,i++) {
		dprintk0("calc_num_list_entries(%d): align 0x%x, src 0x%x, dst 0x%x, size %d\n",
		  i, alignment_mask, pmlist->src_addr, pmlist->dst_addr, pmlist->size);

		/*
		 * this if statment checks if the data is aligned and if not then
		 * count just the number of entries needed for the unaligned data.
		 * it does not count how many nodes will be added for the remaining
		 * data
		 */
		if (pmlist->size & alignment_mask) {
			dprintk0("unaligned start: new_nents %d\n", new_nents);
			if (pmlist->size > 3) {
				if ((pmlist->size & alignment_mask) == 1)
					new_nents += 1;
				else if ((pmlist->size & alignment_mask) == 2)
					/* this only could be true for 16bit width */
					new_nents += (alignment_mask == 0x2 ? 1 : 2);
				else if ((pmlist->size & alignment_mask) == 3)
					new_nents += 3;
			}
			else {
				/* either 1-3 bytes, or 1 short */
				if (alignment_mask == DMA_ALIGNMENT_BYTE) /* byte align */
					new_nents += pmlist->size;
				else
					new_nents += 1;  /* short align */
			}
			dprintk0("unaligned end: new_nents %d\n", new_nents);
		}

		/*
		 * now that the check for unaligned data is done, lets seem
		 * how many entries we need for aligned data, first nodes
		 * greater than max block size
		 */
		if (pmlist->size > max_blk_size) {

			/* number of max sized entries in the block */
			max_size_nents = pmlist->size/max_blk_size;
			/*
			 *  if not a multiple of max size and larger than 3 bytes
			 *  then we add one more entry
			 */
			dprintk0("max_blocks: max_size_nents %d\n", max_size_nents);
			if ((pmlist->size - (max_blk_size*max_size_nents)) > 3)
				max_size_nents++;

			new_nents += max_size_nents;
			dprintk0("max_blocks: w/leftovers max_size_nents %d, new_nents %d\n",
					max_size_nents, new_nents);
		}
		else {
			/* if not greater than max then see if there is any aligned data
			 * remaining.  for example a 65 bytes transer would have one entry
			 * for the unalign byte(counted above) and one entry for the aligned
			 * data less than max size, so two entries
			 */
			dprintk0("else start: new_nents %d\n", new_nents);
			if ((alignment_mask == DMA_ALIGNMENT_WORD) ||
					(alignment_mask == DMA_ALIGNMENT_BYTE && pmlist->size > 3) ||
					(alignment_mask == DMA_ALIGNMENT_SHORT && pmlist->size > 2)) {

				new_nents++;
			}
			dprintk0("else end  : new_nents %d\n", new_nents);
		}
	}

	return(new_nents);
}

#define SG_INIT_SRC_GATHER 		0x01
#define SG_ENABLE_SRC_GATHER	0x02
#define SG_INIT_DST_SCATTER 	0x10
#define SG_ENABLE_DST_SCATTER 	0x20
static int config_sg_list_entry(struct mobi_dma_channel *mobidma,
		struct mobi_dma_list *pmlist, uint8_t *sg_config_flags)
{
	uint64_t sgreg = 0;

	if (pmlist->src_flags & MOBI_DMA_LIST_NODE_FLAG_SG_ENABLE_MASK) {
		*sg_config_flags |= SG_ENABLE_SRC_GATHER;
		if (*sg_config_flags & SG_INIT_SRC_GATHER) {
			if (mobidma->src_sgparams.interval <= 0 ||
					mobidma->src_sgparams.count <= 0) {
				error("Invalid source scatter/gatther params");
				return -EINVAL;
			}
			/* interval is 20 bits, count is 12 */
			sgreg = 0;
			sgreg =
				MOBI_DMAC_CH0_SGR_SGI_W(mobidma->src_sgparams.interval) |
				MOBI_DMAC_CH0_SGR_SGC_W(mobidma->src_sgparams.count);
			/* write count and interval to channels sgr */
			DMAC_WRITED(sgreg, DMAC_CHREG_ADDR(mobidma->channel, SGR));
			*sg_config_flags &= ~SG_INIT_SRC_GATHER;
			dprintk4("Setting src sg params: interval %u, count %u\n",
					(uint32_t)MOBI_DMAC_CH0_SGR_SGI_R(sgreg),
					(uint32_t)MOBI_DMAC_CH0_SGR_SGC_R(sgreg));
			/*
			 *  if the dst is set to default, lets program it to be the
			 *  same as the src in case we hit a reverse flag!
			 */
			if (mobidma->dst_sgparams.count == -1 &&
					mobidma->dst_sgparams.interval == -1) {

				mobidma->dst_sgparams.count = mobidma->src_sgparams.count;
				mobidma->dst_sgparams.interval =
					mobidma->src_sgparams.interval;
				sgreg = 0;
				sgreg =
					MOBI_DMAC_CH0_DSR_DSI_W(mobidma->dst_sgparams.interval) |
					MOBI_DMAC_CH0_DSR_DSC_W(mobidma->dst_sgparams.count);
				/* write count and interval to channels sgr */
				DMAC_WRITED(sgreg, DMAC_CHREG_ADDR(mobidma->channel, DSR));
				*sg_config_flags &= ~SG_INIT_DST_SCATTER;
				dprintk4("resettting uninitialized dst sg_params: "
						"interval %u, count %u\n",
						(uint32_t)MOBI_DMAC_CH0_DSR_DSI_R(sgreg),
						(uint32_t)MOBI_DMAC_CH0_DSR_DSC_R(sgreg));
			}
		}
	}
	else if (pmlist->dst_flags & MOBI_DMA_LIST_NODE_FLAG_SG_ENABLE_MASK) {
		*sg_config_flags |= SG_ENABLE_DST_SCATTER;
		if (*sg_config_flags & SG_INIT_DST_SCATTER) {
			if (mobidma->dst_sgparams.interval <= 0 ||
					mobidma->dst_sgparams.count <= 0) {
				error("Invalid destination scatter/gatther params");
				return -EINVAL;
			}
			/* write count and interval to channels dsr */
			sgreg = 0;
			sgreg =
				MOBI_DMAC_CH0_DSR_DSI_W(mobidma->dst_sgparams.interval) |
				MOBI_DMAC_CH0_DSR_DSC_W(mobidma->dst_sgparams.count);

			DMAC_WRITED(sgreg, DMAC_CHREG_ADDR(mobidma->channel, DSR));
			*sg_config_flags &= ~SG_INIT_DST_SCATTER;
			dprintk4("Setting dst sg params: interval %u, count %u\n",
					(uint32_t)MOBI_DMAC_CH0_DSR_DSI_R(sgreg),
					(uint32_t)MOBI_DMAC_CH0_DSR_DSC_R(sgreg));
			/*
			 *  if the dst is set to default, lets program it to be the
			 *  same as the src in case we hit a reverse flag!
			 */
			if (mobidma->src_sgparams.count == -1 &&
					mobidma->src_sgparams.interval == -1) {

				mobidma->src_sgparams.count = mobidma->dst_sgparams.count;
				mobidma->src_sgparams.interval =
					mobidma->dst_sgparams.interval;
				sgreg = 0;
				sgreg =
					MOBI_DMAC_CH0_SGR_SGI_W(mobidma->src_sgparams.interval)|
					MOBI_DMAC_CH0_SGR_SGC_W(mobidma->src_sgparams.count);
				/* write count and interval to channels sgr */
				DMAC_WRITED(sgreg, DMAC_CHREG_ADDR(mobidma->channel, DSR));
				*sg_config_flags &= ~SG_INIT_SRC_GATHER;
				dprintk4("resettting uninitialized src sg_params: "
						"interval %u, count %u\n",
						(uint32_t)MOBI_DMAC_CH0_SGR_SGI_R(sgreg),
						(uint32_t)MOBI_DMAC_CH0_SGR_SGC_R(sgreg));
			}
		}
	}

	/* if reverse flag set sgparams must be equal */
	if (pmlist->xfer_flags & MOBI_DMA_LIST_XFER_FLAG_REVERSE_DIR) {
		if (mobidma->src_sgparams.count != mobidma->dst_sgparams.count ||
				mobidma->src_sgparams.interval !=
				mobidma->dst_sgparams.interval) {

			error("Cannot enable scatter/gather and reverse_dir for a block "
					"when source and destination sgparams differ");
			return -EINVAL;
		}
	}

	return(0);
}

/* this function will take a mobi_dma_list that has been passes in by a
 *  client and create a lli list in the dtcm.  It will take care of any
 *  entries that are larger than the max block size and also will handle
 *  creating single byte dmas when the size is not word aligned in memory
 *  also will handle any special flags that are set with the list entry
 *  which is further contributing to this function become even bigger and
 *  uglier(although that just might have to be the way it is...)
 */
int32_t create_multiblock_list(mobi_dma_handle dmah,
		struct mobi_dma_list *mlist,
		uint32_t nents)
{

	struct mobi_dma_channel *mobidma = NULL;
	struct mobi_dma_list *pmlist = NULL;
	uint64_t lli_ctl_reg = 0, cfg_reg = 0;
	void __iomem *lli_head = NULL, *lli_tail = NULL;
	int32_t ttfc, xfertype, new_nents = 0;
	uint32_t lli_sar, lli_dar, lli_blockts, lli_src_trwidth, lli_dst_trwidth;
	uint32_t cur_lli_entry_size, prev_lli_entry_size = 0;
	int ret = 0, block_size, block_cnt;
	uint32_t entry_max_blk_size, max_blk_size = 0;
	uint8_t alignment_mask = 0, rev_flag_found = 0;
	uint8_t sinc, dinc, smsize, dmsize;
	uint8_t sg_config_flags = (SG_INIT_SRC_GATHER|SG_INIT_DST_SCATTER);
	uint32_t xfer_size = 0;

	mobidma = &mobi_dma_channels[dmah];

	if (mobidma->xfer_flags & MOBI_DMA_CONFIG_MAX_BLK_SIZE_MASK)
		max_blk_size = mobidma->max_blk_size;
	else
		max_blk_size =
			(MAX_DMA_BLOCK_SIZE*trwidth_reg2num(mobidma->ctl.strwidth))/8;

	/* does data need to be 8 or 16 bit aligned ? */
	alignment_mask = get_alignment_mask(mobidma);
	dprintk4("alignment_mask = %s(0x%x)\n",
			((alignment_mask == DMA_ALIGNMENT_WORD) ? "WORD" :
				(((alignment_mask == DMA_ALIGNMENT_BYTE) ? "BYTE" : "SHORT"))),
			alignment_mask);
	dprintk3("max_blk_size = %d(0x%x)\n", max_blk_size, max_blk_size);
	if (nents == 1)
		dprintk2("process %s single entry list of size %d(0x%x)\n",
				(mlist->size & alignment_mask) ? "unaligned" : "aligned",
				mlist->size, mlist->size);


	/* This is a big, somewhat messy function that is continuing to
	 *  grow, so what's going on here is 2 things.
	 *    First, if a list entry is greater than the max dma size, then it
	 *  must be broken into max dma sized chunks.
	 *    Second, if memory access that is not 32 bits is requested, then
	 *  we have to break that up in single transactions.  For example, 67 bytes
	 *  to an 8 bit device would be 1 64 byte dma and then 3 single bytes dmas.
	 *    Or the best of both worlds > max size dmas that are not aligned.  And
	 *  we have added the feature that will will allow us to do some operation
	 *  to a single node(or more) in the dma_list.  Basically, we have iterate
	 *  over the whole dma_list, processing flags and sometimes breaking things
	 *  up.  It's a time comsuming, messy proccess...
	 */
	new_nents = calc_num_list_entries(mlist, alignment_mask, max_blk_size);
	dprintk3("old nents = %d, new_nents = %d\n", nents, new_nents);

	/* get the memory and mapping for the DMAC linked list */
	if ((ret = create_lli_mapping(dmah, mobidma, new_nents)) != 0)
		return ret;

	lli_head = mobidma->lli_iomap;
	block_cnt = 0;
	for (pmlist = mlist; pmlist != NULL; pmlist = pmlist->next) {

		/* total in bytes this list entry wants to transfer */
		block_size = pmlist->size;
		xfer_size += block_size;

		/* if sg is enabled inside a list entry we MAY define a temp
		 *  max size to be the size of the sg count size in bytes. so
		 *  just re-init the loop max size for every new node
		 */
		entry_max_blk_size = max_blk_size;

		/*
		 *    inside the do loops below we work from the addresses that
		 *  are already in the lli entry, so we better init those address
		 *  everytime we get a new node from the mobi_dma_list
		 *    init the lli variables at the top of each loop so that it
		 *  has the correct address for non-memory nodes
		 */
		lli_sar = pmlist->src_addr;
		writel(lli_sar, (lli_head+LLI_SAR_OFFSET));

		lli_dar = pmlist->dst_addr;
		writel(lli_dar, (lli_head+LLI_DAR_OFFSET));

		lli_tail = lli_head;
		cur_lli_entry_size = 0;

		lli_src_trwidth = mobidma->ctl.strwidth;
		lli_dst_trwidth = mobidma->ctl.dtrwidth;

		/* we can do some changes on the fly, like address adjustment,
		 *  burst size, ???, so grab the value from the node in the
		 *  mobi_dma_list if it is defined.  this has precedence over the
		 *  value set by the config_* routines.  if the flag is not set then
		 *  use the value set from either the config routines or the default
		 *
		 *  NOTE:  if the REVERSE_DIR flag is set we will be swapping things
		 *  around below.  src_addr -> DAR, dst_addr -> SAR.  things like
		 *  sinc, dinc, msize, etc stay associated with given node but the
		 *  order is just switched in the ctl register.  however, if a node
		 *  flag is not set, we need to make sure the default value is taken
		 *  from the correct node.  For example, REV flag is set but no node
		 *  flags, that means the sinc flag will take the default value and
		 *  then be programmed into the dinc field in the clt reg.  however
		 *  since the node are being reversed the sinc must come from the
		 *  default dinc(yes, it is somewhat confusing)
		 */

		if (pmlist->src_flags & MOBI_DMA_LIST_NODE_FLAG_ADDRADJ_MASK) {
			if (addradj_flag2reg(pmlist->src_flags &
						MOBI_DMA_LIST_NODE_FLAG_ADDRADJ_MASK, &sinc)) {
				ret = -EINVAL;
				goto err_out;
			}
		}
		else {
			sinc = mobidma->ctl.sinc;
		}
		if (pmlist->dst_flags & MOBI_DMA_LIST_NODE_FLAG_ADDRADJ_MASK) {
			if (addradj_flag2reg(pmlist->dst_flags &
						MOBI_DMA_LIST_NODE_FLAG_ADDRADJ_MASK, &dinc)) {
				ret = -EINVAL;
				goto err_out;
			}
		}
		else {
			dinc = mobidma->ctl.dinc;
		}

		/* burst size */
		if (pmlist->src_flags & MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_MASK) {
			if (msize_flag2reg(pmlist->src_flags &
						MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_MASK, &smsize)) {
				ret = -EINVAL;
				goto err_out;
			}
		}
		else {
			smsize = mobidma->ctl.smsize;
		}
		if (pmlist->dst_flags & MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_MASK) {
			if (msize_flag2reg(pmlist->src_flags &
						MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_MASK, &dmsize)) {
				ret = -EINVAL;
				goto err_out;
			}
		}
		else {
			dmsize = mobidma->ctl.dmsize;
		}

		/* any entry in the mobi_dma_list can use the scatter/gather
		 *  mechnism; however, it will only be initialized once,  the
		 *  count and interval will be take from the first entry found
		 *  with the SG_ENABLE flag set.
		 */
		if (pmlist->src_flags & MOBI_DMA_LIST_NODE_FLAG_SG_ENABLE_MASK &&
				pmlist->dst_flags & MOBI_DMA_LIST_NODE_FLAG_SG_ENABLE_MASK) {
			error("Cannot enable scatter/gather on both the source"
					" and destination in a single block");
			ret = -EINVAL;
			goto err_out;
		}

		/* clear sg enable flags from last entry and then check if we need
		 *  to config sg for the current entry
		 */
		sg_config_flags &= ~(SG_ENABLE_SRC_GATHER | SG_ENABLE_DST_SCATTER);
		if (pmlist->src_flags & MOBI_DMA_LIST_NODE_FLAG_SG_ENABLE_MASK ||
				pmlist->dst_flags & MOBI_DMA_LIST_NODE_FLAG_SG_ENABLE_MASK) {

			if (config_sg_list_entry(mobidma, pmlist, &sg_config_flags))
				goto err_out;
		}

		/* this lets us know that the handshake interface must be set to
		 *  the same value in the config register at the end of this function
		 */
		if (rev_flag_found == 0 &&
				pmlist->xfer_flags & MOBI_DMA_LIST_XFER_FLAG_REVERSE_DIR)
			rev_flag_found = 1;

		dprintk4("\n");
		dprintk4("pre-processing pmlist entry 0x%p\n", pmlist);
		dprintk4("pmlist->src_addr 0x%x\n", pmlist->src_addr);
		dprintk4("pmlist->dst_addr 0x%x\n", pmlist->dst_addr);
		dprintk4("node: src_flags 0x%x, dst_flags 0x%x, xfer_flags 0x%x\n",
				pmlist->src_flags, pmlist->dst_flags, pmlist->xfer_flags);
		dprintk4("pmlist->size %d\n", pmlist->size);
		dprintk5("dma_config: src/dst flag settings\n");
		dprintk5("sinc 0x%x, dinc 0x%x\n", sinc, dinc);
		dprintk5("smsize 0x%x, dmsize 0x%x\n", smsize, dmsize);
		dprintk5("srcsg_enable %d, dstsg_enable %d\n",
				(sg_config_flags & SG_ENABLE_SRC_GATHER  ? 1 : 0),
				(sg_config_flags & SG_ENABLE_DST_SCATTER ? 1 : 0));
		dprintk4("\n");

		/* we need to adjust a few things as we process the list
		 *  1) size, when unaligned, cut the size short to an aligned size
		 *     then create 1-3 single byte dmas if trwidth is 8 or
		 *     create 1 2-byte dma if trwidth is 16
		 *  2) the transaction width will changed from 32 to 8 or 16
		 *  3) block_ts needs to be calculated correctly
		 */
		prev_lli_entry_size = 0;
		do {
			dprintk4("running block_size %d(0x%x)\n",
					block_size, block_size);

			prev_lli_entry_size = cur_lli_entry_size;
			if (block_size > entry_max_blk_size) {
				cur_lli_entry_size = entry_max_blk_size;
			}
			else if (block_size > 0x3) {
				/* at least a word so round off to word-align */
				cur_lli_entry_size = block_size & 0xfffffffc;
			}
			else {
				/* size if 1 for trwidth of 8, 2 for trwidth of 16 */
				cur_lli_entry_size = (alignment_mask == DMA_ALIGNMENT_BYTE ? 1 : 2);

				if (mobidma->src_node_type == DMA_NODE_TYPE_MEM)
					lli_src_trwidth =
						(alignment_mask == DMA_ALIGNMENT_BYTE ? DMA_TRWIDTH_8 :
						 DMA_TRWIDTH_16);

				if (mobidma->dst_node_type == DMA_NODE_TYPE_MEM)
					lli_dst_trwidth =
						(alignment_mask == DMA_ALIGNMENT_BYTE ? DMA_TRWIDTH_8 :
						 DMA_TRWIDTH_16);

			}

			/*
			 * we need to check for the special case where the address
			 * increments but each start address must be the same for
			 * every block.  this is for a device like the SD controller
			 * that has limited address space but if we dma to a fixed,
			 * non-incrementing address the dma won't burst.  so we use
			 * the RELOAD flag along with a lower max_blk_size that fits
			 * in the address space and we can then burst!
			 */
			lli_sar =  readl(lli_tail+LLI_SAR_OFFSET);
			if (!(mobidma->src_flags & MOBI_DMA_CONFIG_RELOAD_START_ADDR_MASK)) {
				switch (sinc) {
					case DMA_ADDRADJ_INC:
						lli_sar += prev_lli_entry_size;
						break;
					case DMA_ADDRADJ_DEC:
						lli_sar -= prev_lli_entry_size;
						break;
					default:
						/* no change to value read above */
						break;
				}
			}

			lli_dar =  readl(lli_tail+LLI_DAR_OFFSET);
			if (!(mobidma->dst_flags & MOBI_DMA_CONFIG_RELOAD_START_ADDR_MASK)) {
				switch (dinc) {
					case DMA_ADDRADJ_INC:
						lli_dar += prev_lli_entry_size;
						break;
					case DMA_ADDRADJ_DEC:
						lli_dar -= prev_lli_entry_size;
						break;
					default:
						/* no change to value read above */
						break;
				}
			}

			/* addresses are set, not set endian bits */
			lli_sar &= ~MMU_BRIDGE_ENDIAN_SWAP_MASK;
			lli_dar &= ~MMU_BRIDGE_ENDIAN_SWAP_MASK;
			if (mobidma->endian_swap != DMA_ENDIANESS_SWAP_NONE) {
				if (mobidma->src_node_type == DMA_NODE_TYPE_MEM)
					lli_sar |= mobidma->endian_swap <<
						MMU_BRIDGE_ENDIAN_SWAP_SHIFT_L;
				else
					lli_dar |= mobidma->endian_swap <<
						MMU_BRIDGE_ENDIAN_SWAP_SHIFT_L;
			}

			if (pmlist->xfer_flags & MOBI_DMA_LIST_XFER_FLAG_REVERSE_DIR) {
				/* first pass, src/dst are not swapped in the
				 *  mobi_dma_list  this will allow me to know the node type
				 *  without re-parsing the whole list.  with unaligned
				 *  access this would probably cause some kind of problem.
				 *  So now lets just swap the src and dst in the ctl reg and
				 *  then in the SAR/DAR locations in the lli.
				 */
				dprintk4("REVERSE_DIR xfer flag is set!\n");
				/* recompute the ttfc by switching nodes unless we are doing
				 * crypto,
				 */
				xfertype =
					set_transfer_type(mobidma->dst_node_type,
							mobidma->src_node_type);

				ttfc = ctl_ttfc_decode(xfertype, mobidma->flowctrl);

				lli_blockts = (8*(cur_lli_entry_size))/
					trwidth_reg2num(lli_dst_trwidth);
				lli_ctl_reg =
					MOBI_DMAC_CH0_CTL_BLOCK_TS_W((uint64_t)lli_blockts) |
					MOBI_DMAC_CH0_CTL_LLP_SRC_EN_W(1) |
					MOBI_DMAC_CH0_CTL_LLP_DST_EN_W(1) |
					MOBI_DMAC_CH0_CTL_SMS_W(mobidma->ctl.dms) |
					MOBI_DMAC_CH0_CTL_DMS_W(mobidma->ctl.sms) |
					MOBI_DMAC_CH0_CTL_TT_FC_W(ttfc) |
					MOBI_DMAC_CH0_CTL_SRC_GATHER_EN_W(
							(sg_config_flags & SG_ENABLE_DST_SCATTER) ? 1 : 0) |
					MOBI_DMAC_CH0_CTL_DST_SCATTER_EN_W(
							(sg_config_flags & SG_ENABLE_SRC_GATHER) ? 1 : 0) |
					MOBI_DMAC_CH0_CTL_SINC_W(dinc)	|
					MOBI_DMAC_CH0_CTL_DINC_W(sinc) 	|
					MOBI_DMAC_CH0_CTL_SRC_MSIZE_W(dmsize) |
					MOBI_DMAC_CH0_CTL_DST_MSIZE_W(smsize) |
					MOBI_DMAC_CH0_CTL_SRC_TR_WIDTH_W(lli_dst_trwidth)  |
					MOBI_DMAC_CH0_CTL_DST_TR_WIDTH_W(lli_src_trwidth)  |
					MOBI_DMAC_CH0_CTL_INT_EN_W(1);

				/* start updating linked list entry */
				writel(lli_dar, (lli_head+LLI_SAR_OFFSET));
				writel(lli_sar, (lli_head+LLI_DAR_OFFSET));
			}
			else {
				lli_blockts = (8*(cur_lli_entry_size))/
					trwidth_reg2num(lli_src_trwidth);
				lli_ctl_reg =
					MOBI_DMAC_CH0_CTL_BLOCK_TS_W((uint64_t)lli_blockts) |
					MOBI_DMAC_CH0_CTL_LLP_SRC_EN_W(1) |
					MOBI_DMAC_CH0_CTL_LLP_DST_EN_W(1) |
					MOBI_DMAC_CH0_CTL_SMS_W(mobidma->ctl.sms) |
					MOBI_DMAC_CH0_CTL_DMS_W(mobidma->ctl.dms) |
					MOBI_DMAC_CH0_CTL_TT_FC_W(mobidma->ctl.ttfc)|
					MOBI_DMAC_CH0_CTL_SRC_GATHER_EN_W(
							(sg_config_flags & SG_ENABLE_SRC_GATHER) ? 1 : 0) |
					MOBI_DMAC_CH0_CTL_DST_SCATTER_EN_W(
							(sg_config_flags & SG_ENABLE_DST_SCATTER) ? 1 : 0) |
					MOBI_DMAC_CH0_CTL_SINC_W(sinc)	|
					MOBI_DMAC_CH0_CTL_DINC_W(dinc) 	|
					MOBI_DMAC_CH0_CTL_SRC_MSIZE_W(smsize) |
					MOBI_DMAC_CH0_CTL_DST_MSIZE_W(dmsize) |
					MOBI_DMAC_CH0_CTL_SRC_TR_WIDTH_W(lli_src_trwidth)  |
					MOBI_DMAC_CH0_CTL_DST_TR_WIDTH_W(lli_dst_trwidth)  |
					MOBI_DMAC_CH0_CTL_INT_EN_W(1);

				/* start updating linked list entry */
				writel(lli_sar, (lli_head+LLI_SAR_OFFSET));
				writel(lli_dar, (lli_head+LLI_DAR_OFFSET));
			}

			if (block_cnt+1 == new_nents) {
				dprintk4("last entry, lli->next being set to null\n");
				/* then at the last descriptor and it's different */
				/* has be be row 1 or 5 on page 230 */
				writel(0x0, (lli_head+LLI_LLP_OFFSET));
				MOBI_DMAC_CH0_CTL_LLP_SRC_EN_CLR(lli_ctl_reg);
				MOBI_DMAC_CH0_CTL_LLP_DST_EN_CLR(lli_ctl_reg);
			}
			else {
				/* pointer to next linked list item in the list */
				writel((mobidma->lli_phys+
							((block_cnt+1)*LLI_BLOCK_DESCRIPTOR_SIZE)),
						lli_head+LLI_LLP_OFFSET);
			}

			writel((lli_ctl_reg & 0xffffffff), lli_head+LLI_CTLL_OFFSET);
			writel((lli_ctl_reg >> 32), lli_head+LLI_CTLH_OFFSET);

			dprintk4("block_cnt: %d, cur_lli_entry_size %d(0x%x)\n",
					block_cnt, cur_lli_entry_size, cur_lli_entry_size);
			dprintk4("src_trwidth %d, dst_trwidth %d, blockts %d\n",
					trwidth_reg2num(lli_src_trwidth),
					trwidth_reg2num(lli_dst_trwidth), lli_blockts);
			dprintk4("sinc %d, lli_sar now 0x%x\n", sinc, lli_sar);
			dprintk4("dinc %d, lli_dar now 0x%x\n", dinc, lli_dar);
			/*dprintk5("lli_head 0x%p, lli_tail 0x%p\n", lli_head, lli_tail); */

			lli_tail = lli_head;
			lli_head += LLI_BLOCK_DESCRIPTOR_SIZE;

			block_size -= cur_lli_entry_size;
			block_cnt++;

		} while (block_size != 0);
	}

#if DW_DMAC_DEBUG
	if (loglevel > 4) {
		int i;
		for (lli_head = mobidma->lli_iomap, i=1;
				i <= new_nents;
				lli_head += LLI_BLOCK_DESCRIPTOR_SIZE, i++) {

			dprintk("\nDMA block_id %d\n", i);
			dprintk("-----------------\n");
			dprintk("linked list entry: iomap 0x%p, phys 0x%x\n",
					lli_head,
					(mobidma->lli_phys+((i-1)*LLI_BLOCK_DESCRIPTOR_SIZE)));
			dprintk("\tsar  : 0x%x\n", readl(lli_head+LLI_SAR_OFFSET));
			dprintk("\tdar  : 0x%x\n", readl(lli_head+LLI_DAR_OFFSET));
			dprintk("\tllp  : 0x%x\n", readl(lli_head+LLI_LLP_OFFSET));
			dprintk("\tctl  : 0x%llx\n",
					((uint64_t)readl((uint32_t)(lli_head+LLI_CTLH_OFFSET))<< 32
					 | readl((uint32_t)(lli_head+LLI_CTLL_OFFSET))));
		}
		dprintk("\n");
	}
#endif

	dprintk3("total xfer_size is %d\n", xfer_size);
	mobidma->list_entries = new_nents;

	/* TODO except for the reverse flag, we could move the code below this
	 * line to another function, this is not list creation anyway so it doesn't
	 * belong in this function
	 */
	reinit_interrupts(dmah);

	lli_ctl_reg =
		MOBI_DMAC_CH0_CTL_LLP_SRC_EN_W(1) |
		MOBI_DMAC_CH0_CTL_LLP_DST_EN_W(1) |
		MOBI_DMAC_CH0_CTL_INT_EN_W(1);

	dprintk4("Writing to ctl register: 0x%llx\n", lli_ctl_reg);
	/*
	 *  I think the only thing that matters in this write is enabling the llp on
	 *  src and dst.  the block descriptors will contain the values that really
	 *  matter but we have to tell the DMAC to use the LLP
	 */
	DMAC_WRITED(lli_ctl_reg, DMAC_CHREG_ADDR(dmah, CTL));

	/*
	 *  there are some others we may want to look into
	 *  priority, FCMODE, FIFO_MODE, MAX_ABRST
	 *  note: even though we do a mem->mem we program the handshake interface
	 *  this is OK because it should be ignored when doing the mem->mem transfers
	 */
	cfg_reg =
		MOBI_DMAC_CH0_CFG_PROTCTL_W((uint64_t)1) /* recommended default */
		| MOBI_DMAC_CH0_CFG_HS_SEL_SRC_W(mobidma->cfg.shs_sel)
		| MOBI_DMAC_CH0_CFG_HS_SEL_DST_W(mobidma->cfg.dhs_sel);

	if (rev_flag_found == 0) {
		cfg_reg |= MOBI_DMAC_CH0_CFG_DEST_PER_W((uint64_t)mobidma->cfg.dst_per)|
			MOBI_DMAC_CH0_CFG_SRC_PER_W((uint64_t)mobidma->cfg.src_per);
	}
	else {
		/* NOTE going to take a chance here for now and assume one of
		 *  the nodes is memory, so duplicate the non-zero per in
		 *  reverse mode, this is because normally the handshaking
		 *  only matters in one direction, but if we do a reverse node
		 *  then we set the per in the opposite dir
		 */
		if (mobidma->dst_node_type == DMA_NODE_TYPE_MEM) {
			dprintk4("reverse flag detected, duplicate src_per\n");
			cfg_reg |=
				MOBI_DMAC_CH0_CFG_DEST_PER_W((uint64_t)mobidma->cfg.src_per) |
				MOBI_DMAC_CH0_CFG_SRC_PER_W((uint64_t)mobidma->cfg.src_per);
		}
		else {
			dprintk4("reverse flag detected, duplicate dst_per\n");
			cfg_reg |=
				MOBI_DMAC_CH0_CFG_DEST_PER_W((uint64_t)mobidma->cfg.dst_per) |
				MOBI_DMAC_CH0_CFG_SRC_PER_W((uint64_t)mobidma->cfg.dst_per);
		}
	}

	dprintk4("Writing to cfg register: 0x%llx\n", cfg_reg);
	DMAC_WRITED(cfg_reg, DMAC_CHREG_ADDR(dmah, CFG));
	/* in case where were are using DDR for the lli, small list are
	 * not being written fast enough so tranfers are hanging, so
	 * we need to flush the cache.  i wonder if there is a way to alloc
	 * memory and then mark it as non-cacheable?
	 */
	if (mobidma->lli_phys < DESC_RAM_BASE || mobidma->lli_phys > DESC_RAM_END)
		flush_cache_all();

err_out:
	if (ret)
		free_lli_mapping(mobidma);

	return ret;
}

/* do some basic alignment checking because
 *  - access to 32 and 16 bit devices must be the proper with
 */
static int check_basic_alignment(struct mobi_dma_channel *mobidma)
{

	/* check for invalid access to peripherals */
	if (mobidma->src_node_type == DMA_NODE_TYPE_PER &&
			mobidma->ctl.strwidth == DMA_TRWIDTH_32 &&
			mobidma->dma_length & 0x3) {
		error("Cannot access 32 bit wide "
				"source peripheral on non 32 bit boundry");
		return -EINVAL;
	}
	if (mobidma->dst_node_type == DMA_NODE_TYPE_PER &&
			mobidma->ctl.dtrwidth == DMA_TRWIDTH_32 &&
			mobidma->dma_length & 0x3) {
		error("Cannot access 32 bit wide "
				"destination peripheral on non 32 bit boundry");
		return -EINVAL;
	}
	if (mobidma->src_node_type == DMA_NODE_TYPE_PER &&
			mobidma->ctl.strwidth == DMA_TRWIDTH_16 &&
			mobidma->dma_length & 0x1) {
		error("Cannot access 16 bit wide "
				"source peripheral on non 16 bit boundry");
		return -EINVAL;
	}
	if (mobidma->dst_node_type == DMA_NODE_TYPE_PER &&
			mobidma->ctl.dtrwidth == DMA_TRWIDTH_16 &&
			mobidma->dma_length & 0x1) {
		error("Cannot access 16 bit wide "
				"destination peripheral on non 16 bit boundry");
		return -EINVAL;
	}

	return 0;
}

/**
 * \brief mobi_dma_setup_mlist:
 * 	setup DMA channel based on client provided mobi_dma_list
 *
 * \param dmah: DMA channel number
 * \param mlist: pointer to mobi_dma_list
 * \param nents: number of entries in mlist
 * \param dma_length: total length of the transfer request in bytes
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval -EINVAL - if invalid argument provided
 * \retval -ENOMEM - if internal memory allocation fails
 * \retval -EIO    - if internal resource request fails
 * \retval Zero   - upon success
 *
 * \remark
 * \verbatim
 *   This function allows a client to define a dma transfer
 * using the mobi_dma_list structure.  The structure has this
 * definition:
 * 	struct mobi_dma_list {
 *      	uint32_t src_addr;
 *              uint32_t dst_addr;
 *       	int32_t  size;	    // size in bytes for this list entry
 *	        uint32_t xfr_flags; // general flags applies to xfer of this block
 *	        uint32_t src_flags; // flags that apply to the src node
 *	        uint32_t dst_flags; // flags that apply to the dst node
 *		struct mobi_dma_list *next;
 *	};
 *
 *   This function should be used when accessing memory that is discontiguous,
 * or when trying to do multiple operations in a single transaction.  For
 * example, programming a device to perform an operation and then dma-ing
 * the data to the device.  This sequence can be programmined into a
 * mobi_dma_list and would be considered a single dma transfer.
 *   The client creates a linked list of type mobi_dma_list and passes
 * this list to the function.  There are no restrictions on the transfer
 * type; in other words, can do mem->mem, mem->per, per->mem or per->per.
 * The src and dst must remain the same device/target for the duration of
 * the transfer. See mobi_dma.h for explanation of the flags.
 * \endverbatim
 *
 */
int32_t mobi_dma_setup_mlist(mobi_dma_handle dmah,
		struct mobi_dma_list *mlist,
		uint32_t nents,
		uint32_t dma_length)
{
	struct mobi_dma_channel *mobidma = NULL;
	int32_t ret = 0;

	if ((ret = dma_validate_channel_access(dmah, (char*)__FUNCTION__)) != 0)
		return ret;

	mobidma = &mobi_dma_channels[dmah];

	dprintk3("entered\n");
	if (!mlist || !nents) {
		error("Channel %d empty mobi_dma_list", dmah);
		return -EINVAL;
	}
	if (!dma_length) {
		error("Channel %d has zero transfer length", dmah);
		return -EINVAL;
	}

	/* initial target addresses, physical */
	mobidma->src_addr = mlist->src_addr;
	mobidma->dst_addr = mlist->dst_addr;

	/*
	 * reset these to the default, will figure out which
	 *  one to really use later.  Need to reset because a channel may be
	 *  held open for read and writes and we need to reset the master
	 *  bus each time
	 */
	mobidma->ctl.sms = DMAC_AHB_MASTER_2;
	mobidma->ctl.dms = DMAC_AHB_MASTER_2;

	/* figure out some of the src details */
	if (setup_src_defines(dmah))
		return -EINVAL;

	/* figure out some of the dst details */
	if (setup_dst_defines(dmah))
		return -EINVAL;

	mobidma->mlist = mlist;
	mobidma->dma_length = dma_length;
	mobidma->blk_count = 0;

	/* check some basic device/memory alignment issues */
	if (check_basic_alignment(mobidma))
		return -EINVAL;

	/* had to wait to do this until we determine the type of both nodes */
	mobidma->transtype =
		set_transfer_type(mobidma->src_node_type, mobidma->dst_node_type);
	mobidma->ctl.ttfc = ctl_ttfc_decode(mobidma->transtype, mobidma->flowctrl);
	dma_set_endianess(mobidma);

	dprintk3("number of entries in initial mobi_list is %d\n", nents);

	return create_multiblock_list(dmah, mlist, nents);
}
EXPORT_SYMBOL(mobi_dma_setup_mlist);

/**
 * \brief mobi_dma_setup_sglist:
 * 	setup DMA channel using sglist to/from a device
 *
 * \param dmah: DMA channel number
 * \param sg: pointer to the scatter-gather list/vector
 * \param nents: scatter-gather list number of entries count
 * \param dma_length: total length of the transfer request in bytes
 * \param dev_addr: physical device port address
 * \param dmamode: DMA transfer mode, DMA_FROM_DEVICE or DMA_TO_DEVICE
 *
 * \retval -EINVAL - if invalid argument provided
 * \retval -ENOMEM - if internal memory allocation fails
 * \retval Zero   - upon success
 *
 * \remark
 *   This function can be used to setup a transfer between peripherals
 * and memory using a "classic" sglist.  The sglist can be created
 * using dma_map_sg and other related functions.
 *
 *
 */
int32_t mobi_dma_setup_sglist(mobi_dma_handle dmah,
		struct scatterlist *sglist,
		uint32_t nents,
		uint32_t dma_length,
		uint32_t dev_addr,
		enum dma_data_direction dmamode)
{
	int i;
	struct mobi_dma_channel *mobidma = NULL;
	struct scatterlist *sg;
	int32_t ret;
	struct mobi_dma_list *mlist = NULL;

	if ((ret = dma_validate_channel_access(dmah, (char*)__FUNCTION__)) != 0)
		return ret;

	mobidma = &mobi_dma_channels[dmah];

	if (!sglist || !nents) {
		error("Channel %d mobi_dma_setup_sglist empty sg list",
				dmah);
		return -EINVAL;
	}
	if (!sglist->length) {
		error("Channel %d mobi_dma_setup_sglist zero length sg entry",
				dmah);
		return -EINVAL;
	}
	if (!dma_length) {
		error("Channel %d: mobi_dma_setup_sglist zero transfer length",
				dmah);
		return -EINVAL;
	}

	dprintk3("nents %d, dma_length %d\n", nents, dma_length);
	mobidma->mlist = kmalloc(nents*sizeof(struct mobi_dma_list), GFP_ATOMIC);
	if (mobidma->mlist == NULL) {
		error("Channel %d Unable to allocate memory for sglist conversion",
				dmah);
		return -ENOMEM;
	}

	mlist = mobidma->mlist;
	mobidma->list_entries = nents;
	mobidma->dma_length = dma_length;
	mobidma->blk_count = 0;
	mobidma->ctl.sms = DMAC_AHB_MASTER_2;
	mobidma->ctl.dms = DMAC_AHB_MASTER_2;

	/* set the initial addresses to help configure nodes */
	if (dmamode == DMA_FROM_DEVICE) {
		/* dev -> mem */
		mobidma->src_addr = dev_addr;
		mobidma->dst_addr = sg_dma_address(sglist);
	}
	else {
		/* mem -> dev */
		mobidma->dst_addr = dev_addr;
		mobidma->src_addr = sg_dma_address(sglist);
	}

	/* check some basic device/memory alignment issues */
	if (check_basic_alignment(mobidma))
		return -EINVAL;

	/* figure out some of the src details */
	if (setup_src_defines(dmah))
		return -EINVAL;

	/* figure out some of the dst details */
	if (setup_dst_defines(dmah))
		return -EINVAL;

	/* have to wait to do this until we determine the type of both nodes */
	mobidma->transtype =
		set_transfer_type(mobidma->src_node_type, mobidma->dst_node_type);
	mobidma->ctl.ttfc = ctl_ttfc_decode(mobidma->transtype, mobidma->flowctrl);
	dma_set_endianess(mobidma);

	/*
	 * we could have a device where the address changes so we have to take
	 * care of that since only a fixed device address is passed in unless
	 * it is a device like SD that need to increment to burst but has
	 * small address space, in that case RELOAD_START_ADDR is set so we
	 * always start from the top of the fifo
	 */
	for (i=0, sg=sglist;i < nents;i++,sg++) {
		if (dmamode == DMA_FROM_DEVICE) {
			(&mlist[i])->dst_addr = sg_dma_address(sg);
			if (i > 0) {
				if (!(mobidma->src_flags &
							MOBI_DMA_CONFIG_RELOAD_START_ADDR_MASK)) {
					switch (mobidma->ctl.sinc) {
						case DMA_ADDRADJ_INC:
							(&mlist[i])->src_addr =
								(&mlist[i-1])->src_addr + (&mlist[i-1])->size;
							break;
						case DMA_ADDRADJ_DEC:
							(&mlist[i])->src_addr =
								(&mlist[i-1])->src_addr - (&mlist[i-1])->size;
							break;
						default:
							(&mlist[i])->src_addr = dev_addr;
							break;
					}
				}
				else
					(&mlist[i])->src_addr = dev_addr;
			}
			else {
				(&mlist[i])->src_addr = dev_addr;
			}
		}
		else {
			(&mlist[i])->src_addr = sg_dma_address(sg);
			if (i > 0) {
				if (!(mobidma->dst_flags &
							MOBI_DMA_CONFIG_RELOAD_START_ADDR_MASK)) {
					switch (mobidma->ctl.dinc) {
						case DMA_ADDRADJ_INC:
							(&mlist[i])->dst_addr =
								(&mlist[i-1])->dst_addr + (&mlist[i-1])->size;
							break;
						case DMA_ADDRADJ_DEC:
							(&mlist[i])->dst_addr =
								(&mlist[i-1])->dst_addr - (&mlist[i-1])->size;
							break;
						default:
							(&mlist[i])->dst_addr = dev_addr;
							break;
					}
				}
				else
					(&mlist[i])->dst_addr = dev_addr;
			}
			else {
				(&mlist[i])->dst_addr = dev_addr;
			}
		}
		(&mlist[i])->size          = sg_dma_len(sg);
		(&mlist[i])->xfer_flags    = MOBI_DMA_LIST_FLAG_NONE;
		(&mlist[i])->src_flags     = MOBI_DMA_LIST_FLAG_NONE;
		(&mlist[i])->dst_flags     = MOBI_DMA_LIST_FLAG_NONE;

		if (nents > 1 && i != nents-1)
			(&mlist[i])->next = &mlist[i+1];
		else
			(&mlist[i])->next = NULL;

		dprintk3("mlist[%d]: sar 0x%08x, dar 0x%08x, size %d\n",
				i, (&mlist[i])->src_addr, (&mlist[i])->dst_addr,
				(&mlist[i])->size);
	}

	ret = create_multiblock_list(dmah, mlist, nents);

	if (mobidma->mlist) {
		kfree(mobidma->mlist);
	}

	return ret;
}
EXPORT_SYMBOL(mobi_dma_setup_sglist);

int32_t need_mlist(struct mobi_dma_channel *mobidma, uint32_t dma_length)
{
	/*
	 *  OK, it's time to get ugly.  We can't do a large, non-word aligned
	 *  DMA to memory.  In other words, you can't DMA something like 34
	 *  bytes in a single DMA to memory.  However, we can do single DMA
	 *  transactions of a one or two bytes.  So the trick is, if the memory
	 *  is involved and the size is not aligned we created a dma list with
	 *  multiple entries.  For example, transfering 34 bytes from an 8-bit
	 *  device to memory results in 3 dma transaction, one 32 byte, and two
	 *  single bytes.  A 16-bit device would result in two dma transactions,
	 *  one 32 byte and one 16 byte
	 *
	 *  the second reason we might enter this if statement is if the requested
	 *  block size is larger than the max.  The max block size is defined by
	 *  the (source transfer width)*4092.  For the transfer to work under
	 *  these conditions, the block of memory will have to be contigious.
	 *
	 *  anyway, we will create an internal mobi_dma_list with one entry and
	 *  then call mobi_dma_setup_mlist.  This function handles non-aligned
	 *  buffers and buffers that are too large
	 *
	 */
	/*
	 *  look for unaligned size of larger than one
	 *  transaction and also checks if the dma is bigger than max, if so a
	 *  linked list must be created.
	 */
	/*
	 *  case 1: mem->dev 32->8, non-aligned, > 1 byte
	 *  case 2: dev->mem 8->32, non-aligned, > 1 byte
	 *  case 3: mem->dev 32->16, non-aligned, > 2 bytes
	 *  case 4: dev->mem 16->32, non-aligned, > 2 bytes
	 *  case 5 & 6: greater than max block size
	 *  case 7 : mem->mem, 32->32 non-aligned
	 */
	if (((dma_length & 0x3) && (dma_length != 1) &&
				(mobidma->src_node_type == DMA_NODE_TYPE_MEM) && (mobidma->ctl.dtrwidth == DMA_TRWIDTH_8)) ||
			((dma_length & 0x3) && (dma_length != 1) &&
			 (mobidma->dst_node_type == DMA_NODE_TYPE_MEM) && (mobidma->ctl.strwidth == DMA_TRWIDTH_8)) ||
			((dma_length & 0x2) && (dma_length != 2) &&
			 (mobidma->src_node_type == DMA_NODE_TYPE_MEM) && (mobidma->ctl.dtrwidth == DMA_TRWIDTH_16)) ||
			((dma_length & 0x2) && (dma_length != 2) &&
			 (mobidma->dst_node_type == DMA_NODE_TYPE_MEM) && (mobidma->ctl.strwidth == DMA_TRWIDTH_16)) ||
			((mobidma->src_node_type != DMA_NODE_TYPE_MEM) &&
			 ((dma_length*8)/trwidth_reg2num(mobidma->ctl.strwidth)) > MAX_DMA_BLOCK_SIZE) ||
			((mobidma->src_node_type == DMA_NODE_TYPE_MEM) &&
			 ((dma_length*8)/trwidth_reg2num(DMA_TRWIDTH_32)) > MAX_DMA_BLOCK_SIZE)  ||
			((dma_length & 0x3) &&
			 (mobidma->src_node_type == DMA_NODE_TYPE_MEM) && (mobidma->dst_node_type == DMA_NODE_TYPE_MEM))) {

#if DW_DMAC_DEBUG
		if (((mobidma->src_node_type != DMA_NODE_TYPE_MEM) &&
					((dma_length*8)/trwidth_reg2num(mobidma->ctl.strwidth)) > MAX_DMA_BLOCK_SIZE) ||
				((mobidma->src_node_type == DMA_NODE_TYPE_MEM) &&
				 ((dma_length*8)/trwidth_reg2num(DMA_TRWIDTH_32)) > MAX_DMA_BLOCK_SIZE)) {
			dprintk3("single dma exceeds max size: %d > %d\n",
					dma_length,
					(MAX_DMA_BLOCK_SIZE*
					 trwidth_reg2num(mobidma->ctl.strwidth))/8);
		}
		else {
			dprintk3("requesting unaligned access to memory!\n");
		}
#endif

		return(1);
	}
	return(0);
}

/* the dmac has it's own version of scatter/gather that some devices
 *  may find usefull if they want to read/write at fixed offsets
 */
int setup_dmac_sglist(mobi_dma_handle dmah, struct mobi_dma_channel *mobidma)
{
	uint64_t sgreg = 0;

	/* determine if sg is enabled */
	if (mobidma->dst_flags & MOBI_DMA_CONFIG_SG_ENABLE_MASK &&
			mobidma->src_flags & MOBI_DMA_CONFIG_SG_ENABLE_MASK) {
		error("Cannot enable scatter/gather on both "
				"source and destination");
		return -EINVAL;
	}
	if (mobidma->src_flags & MOBI_DMA_CONFIG_SG_ENABLE_MASK) {
		if (mobidma->src_sgparams.interval <= 0 ||
				mobidma->src_sgparams.count <= 0) {
			error("Invalid src scatter/gather params");
			return -EINVAL;
		}
		mobidma->ctl.gatheren = 1;
		sgreg =
			MOBI_DMAC_CH0_SGR_SGI_W(mobidma->src_sgparams.interval) |
			MOBI_DMAC_CH0_SGR_SGC_W(mobidma->src_sgparams.count);
		DMAC_WRITED(sgreg, DMAC_CHREG_ADDR(dmah, SGR));
		dprintk4("Setting src sg params: interval %u, count %u\n",
				(uint32_t)MOBI_DMAC_CH0_SGR_SGI_R(sgreg),
				(uint32_t)MOBI_DMAC_CH0_SGR_SGC_R(sgreg));
	}
	else if (mobidma->dst_flags & MOBI_DMA_CONFIG_SG_ENABLE_MASK) {
		if (mobidma->dst_sgparams.interval <= 0 ||
				mobidma->dst_sgparams.count <= 0) {
			error("Invalid dst scatter/gather params");
			return -EINVAL;
		}
		mobidma->ctl.scatteren = 1;
		sgreg =
			MOBI_DMAC_CH0_DSR_DSI_W(mobidma->dst_sgparams.interval) |
			MOBI_DMAC_CH0_DSR_DSC_W(mobidma->dst_sgparams.count);
		DMAC_WRITED(sgreg, DMAC_CHREG_ADDR(dmah, DSR));
		dprintk4("Setting dst sg params: interval %u, count %u\n",
				(uint32_t)MOBI_DMAC_CH0_DSR_DSI_R(sgreg),
				(uint32_t)MOBI_DMAC_CH0_DSR_DSC_R(sgreg));
	}
	return(0);
}

/**
 * \brief mobi_dma_setup_single:
 * 	  Setup DMA channel for a single block transfer.
 *
 * \param dmah: dma channel handle
 * \param dma_src_addr: the DMA/physical memory address of the src data block
 * \param dma_dst_addr: the DMA/physical memory address of the destination target
 * \param dma_length: length of the data block in bytes
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval -EINVAL - if invalid argument provided
 * \retval -ENOMEM - if internal memory allocation fails
 * \retval -EIO    - if internal resource request fails
 * \retval Zero    - upon success
 *
 * \remark
 *   A block is defined as either a tranfer of less than or equal
 * to MAX_DMA_BLOCK_SIZE or a single block of contiguous memory
 * greater than MAX_DMA_BLOCK_SIZE.  If doing a transfer of
 * greater than MAX_DMA_BLOCK_SIZE, the src and dst cannot be
 * the same type(mem->mem or per->per).
 *
 */
int32_t mobi_dma_setup_single(mobi_dma_handle dmah,
		uint32_t dma_src_addr, uint32_t dma_dst_addr,
		uint32_t dma_length)
{
	struct mobi_dma_channel *mobidma = NULL;
	uint64_t ctl_reg = 0, cfg_reg = 0;
	int32_t ret = 0;
	struct mobi_dma_list *entry = NULL;
	struct mobi_dma_list list_node;
	uint32_t src_trwidth, dst_trwidth;

	if ((ret = dma_validate_channel_access(dmah, (char*)__FUNCTION__)) != 0)
		return ret;

	mobidma = &mobi_dma_channels[dmah];

	if (!dma_src_addr || !dma_dst_addr) {
		error("Channel %d mobi_dma_setup_single null address",
				dmah);
		return -EINVAL;
	}

	if (!dma_length) {
		error("Channel %d mobi_dma_setup_single zero length",
				dmah);
		return -EINVAL;
	}

	mobidma->src_addr = dma_src_addr;
	mobidma->dst_addr = dma_dst_addr;
	mobidma->dma_length = dma_length;
	mobidma->list_entries = 0;

	/* reset these to the default, will figure out which
	 *  one to really use later.  Need to reset because a channel may be
	 *  held open for read and writes and we need to reset the master
	 *  bus each time
	 */
	mobidma->ctl.sms = DMAC_AHB_MASTER_2;
	mobidma->ctl.dms = DMAC_AHB_MASTER_2;

	reinit_interrupts(dmah);
	/* figure out some of the src details */
	if (setup_src_defines(dmah) != 0)
		return -EINVAL;
	/* figure out some of the dst details */
	if (setup_dst_defines(dmah) != 0)
		return -EINVAL;

	dprintk2("src_addr = 0x%x and is %s\n",
			mobidma->src_addr,
			(mobidma->src_node_type == DMA_NODE_TYPE_PER ? "per":"mem"));
	dprintk2("dst_addr = 0x%x and is %s\n",
			mobidma->dst_addr,
			(mobidma->dst_node_type == DMA_NODE_TYPE_PER ? "per":"mem"));
	dprintk2("dma_length = %d\n", mobidma->dma_length);

	/* check some basic device/memory alignment issues */
	if (check_basic_alignment(mobidma))
		return -EINVAL;

	if (need_mlist(mobidma, dma_length)) {
		/*
		 *  we will create a mobi_list with one entry and then
		 *  allow that function to modify/recreate the list for non
		 *  aligned transfers
		 */
		entry = &list_node;

		entry->src_addr = mobidma->src_addr;
		entry->dst_addr = mobidma->dst_addr;
		entry->size = dma_length;
		/* since this tranfsfer is being created from a single, all
		 *  the config flags have been applied and we should not be
		 *  setting and node dependant flags!
		 */
		entry->src_flags  = 0;
		entry->dst_flags  = 0;
		entry->xfer_flags = 0;
		entry->next = NULL;

		ret = mobi_dma_setup_mlist(dmah, entry, 1, dma_length);
		return ret;
	}

	/* have to wait to do this until we determine the type of both nodes */
	mobidma->transtype =
		set_transfer_type(mobidma->src_node_type, mobidma->dst_node_type);
	mobidma->ctl.ttfc =
		ctl_ttfc_decode(mobidma->transtype, mobidma->flowctrl);

	dma_set_endianess(mobidma);

	/* check if s/g flags are set and do setup if necessary */
	if ((ret = setup_dmac_sglist(dmah, mobidma)) < 0)
		return ret;

	/* if the lenth of the data is only 1 transaction, I can play tricks
	 *  to make it work
	 */
	if (mobidma->src_node_type == DMA_NODE_TYPE_MEM) {
		if(mobidma->ctl.dtrwidth == DMA_TRWIDTH_8 && mobidma->dma_length == 1)
			src_trwidth = DMA_TRWIDTH_8;
		else if (mobidma->ctl.dtrwidth == DMA_TRWIDTH_16 &&
				mobidma->dma_length == 2)
			src_trwidth = DMA_TRWIDTH_16;
		else
			src_trwidth = DMA_TRWIDTH_32;

		dprintk3("original ctl.strwidth = 0x%x\n", mobidma->ctl.strwidth);
	}
	else {
		src_trwidth = mobidma->ctl.strwidth;
	}
	if (mobidma->dst_node_type == DMA_NODE_TYPE_MEM) {
		if(mobidma->ctl.strwidth == DMA_TRWIDTH_8 && mobidma->dma_length == 1)
			dst_trwidth = DMA_TRWIDTH_8;
		else if (mobidma->ctl.strwidth == DMA_TRWIDTH_16 &&
				mobidma->dma_length == 2)
			dst_trwidth = DMA_TRWIDTH_16;
		else
			dst_trwidth = DMA_TRWIDTH_32;

		dprintk3("original ctl.dtrwidth = 0x%x\n", mobidma->ctl.dtrwidth);
	}
	else {
		dst_trwidth = mobidma->ctl.dtrwidth;
	}

	/* equation 5 in DMAC databook */
	mobidma->ctl.blockts = (8*mobidma->dma_length) /
		trwidth_reg2num(src_trwidth);

	dprintk3("transtype 0x%x, ttfc 0x%x, blockts 0x%x(%d)\n",
			mobidma->transtype, mobidma->ctl.ttfc,
			mobidma->ctl.blockts, mobidma->ctl.blockts);
	dprintk3("src_trwidth 0x%x, dst_trwith 0x%x, datawidth %d\n",
			src_trwidth, dst_trwidth, mobidma->datawidth);
	dprintk3("src msize 0x%x, dst msize 0x%x\n",
			mobidma->ctl.smsize, mobidma->ctl.dmsize);
	dprintk3("sinc 0x%x, dinc 0x%x\n",
			mobidma->ctl.sinc, mobidma->ctl.dinc);

	/* for a single block, we need to set sinc/dinc */
	ctl_reg =
		MOBI_DMAC_CH0_CTL_BLOCK_TS_W((uint64_t)mobidma->ctl.blockts)
		| MOBI_DMAC_CH0_CTL_LLP_SRC_EN_W(0)
		| MOBI_DMAC_CH0_CTL_LLP_DST_EN_W(0)
		| MOBI_DMAC_CH0_CTL_SMS_W(mobidma->ctl.sms)	
		| MOBI_DMAC_CH0_CTL_DMS_W(mobidma->ctl.dms)
		| MOBI_DMAC_CH0_CTL_TT_FC_W(mobidma->ctl.ttfc)
		| MOBI_DMAC_CH0_CTL_SRC_GATHER_EN_W(mobidma->ctl.gatheren)
		| MOBI_DMAC_CH0_CTL_DST_SCATTER_EN_W(mobidma->ctl.scatteren)
		| MOBI_DMAC_CH0_CTL_SINC_W(mobidma->ctl.sinc)
		| MOBI_DMAC_CH0_CTL_DINC_W(mobidma->ctl.dinc)
		| MOBI_DMAC_CH0_CTL_SRC_MSIZE_W(mobidma->ctl.smsize)
		| MOBI_DMAC_CH0_CTL_DST_MSIZE_W(mobidma->ctl.dmsize)
		| MOBI_DMAC_CH0_CTL_SRC_TR_WIDTH_W(src_trwidth)
		| MOBI_DMAC_CH0_CTL_DST_TR_WIDTH_W(dst_trwidth);

	DMAC_WRITED(ctl_reg, DMAC_CHREG_ADDR(dmah, CTL));

	/* there are some others we may want to look into
	 *  priority, bus/channel lock, FCMODE, FIFO_MODE, MAX_ABRST
	 */
	cfg_reg =
		MOBI_DMAC_CH0_CFG_DEST_PER_W((uint64_t)mobidma->cfg.dst_per)
		| MOBI_DMAC_CH0_CFG_SRC_PER_W((uint64_t)mobidma->cfg.src_per)
		| MOBI_DMAC_CH0_CFG_PROTCTL_W((uint64_t)1) /* recommended default */
		| MOBI_DMAC_CH0_CFG_HS_SEL_SRC_W(mobidma->cfg.shs_sel)
		| MOBI_DMAC_CH0_CFG_HS_SEL_DST_W(mobidma->cfg.dhs_sel);

	DMAC_WRITED(cfg_reg, DMAC_CHREG_ADDR(dmah, CFG));

	/* single block, so LLP is 0 */
	DMAC_WRITEL(0x0, DMAC_CHREG_ADDR(dmah, LLP));

	mobidma->src_addr &= ~MMU_BRIDGE_ENDIAN_SWAP_MASK;
	mobidma->dst_addr &= ~MMU_BRIDGE_ENDIAN_SWAP_MASK;

	/*
	 *  if we have endian issues set the bits in the address so mmu bridge
	 *  knows what to do
	 */
	if (mobidma->endian_swap != DMA_ENDIANESS_SWAP_NONE) {
		if (mobidma->src_node_type == DMA_NODE_TYPE_MEM) {
			mobidma->src_addr |=
				mobidma->endian_swap << MMU_BRIDGE_ENDIAN_SWAP_SHIFT_L;
		}
		else {
			mobidma->dst_addr |=
				mobidma->endian_swap << MMU_BRIDGE_ENDIAN_SWAP_SHIFT_L;
		}
	}
	dprintk3("write SAR = 0x%x, DAR = 0x%x\n",
			mobidma->src_addr, mobidma->dst_addr);

	DMAC_WRITEL(mobidma->src_addr, DMAC_CHREG_ADDR(dmah, SAR));
	DMAC_WRITEL(mobidma->dst_addr, DMAC_CHREG_ADDR(dmah, DAR));

	return 0;
}
EXPORT_SYMBOL(mobi_dma_setup_single);

/**
 * \brief mobi_dma_setup_handler:
 * 	setup DMA channel end notification handlers
 *
 * \param dmah: DMA channel number
 * \param handler: the pointer to the client's callback handler
 * \param data: user specified value to be passed back to the handlers
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval Zero    - upon success
 */
int mobi_dma_setup_handler(mobi_dma_handle dmah,
		void (*handler)(mobi_dma_handle, mobi_dma_event, void*),
		void *data)
{
	struct mobi_dma_channel *mobidma = NULL;
	unsigned long flags;
	int32_t ret;

	if ((ret = dma_validate_channel_access(dmah, (char*)__FUNCTION__)) != 0)
		return ret;

	mobidma = &mobi_dma_channels[dmah];

#ifdef CONFIG_PREEMPT_RT
	local_irq_save_nort(flags);
#else
	local_irq_save(flags);
#endif
	mobidma->handler = handler;
	mobidma->data = data;
#ifdef CONFIG_PREEMPT_RT
	local_irq_restore_nort(flags);
#else
	local_irq_restore(flags);
#endif
	return 0;
}
EXPORT_SYMBOL(mobi_dma_setup_handler);

/*
 * \brief update_ctl:
 *    This function is used to update one or more fields in a
 *  channel's control register.
 *
 *  \param dmah: DMA handle
 *  \param mask: bitmask for target fields,
 *  \param data: new value(s) to be writen to register, this must be bit
 *         aligned already!  Use the _W macros in the register header
 *         file to handle this
 */
static void update_ctl(mobi_dma_handle dmah, uint64_t mask, uint64_t data)
{
	uint64_t ctl_reg;

	DMAC_READD(DMAC_CHREG_ADDR(dmah, CTL), ctl_reg);
	ctl_reg &= ~(mask);
	ctl_reg |= data;
	//dprintk("%s:%d: ctl_reg = 0x%llx\n", __func__, dmah, ctl_reg);
	DMAC_WRITED(ctl_reg, DMAC_CHREG_ADDR(dmah, CTL));
}

/*
 *  \brief update_cfg:
 *    This function is used to update one or more fields in a
 *  channel's config register.
 *
 *  \param dmah: DMA handle
 *  \param mask: bitmask for target fields,
 *  \param data: new value(s) to be writen to register, this must be bit
 *         aligned already!  Use the _W macros in the register header
 *         file to handle this
 */
static void update_cfg(mobi_dma_handle dmah, uint64_t mask, uint64_t data)
{
	uint64_t cfg_reg;

	DMAC_READD(DMAC_CHREG_ADDR(dmah, CFG), cfg_reg);
	cfg_reg &= ~(mask);
	cfg_reg |= data;
	//dprintk("%s:%d: cfg_reg = 0x%llx\n", __func__, dmah, cfg_reg);
	DMAC_WRITED(cfg_reg, DMAC_CHREG_ADDR(dmah, CFG));
	DMAC_READD(DMAC_CHREG_ADDR(dmah, CFG), cfg_reg);
}

/**
 * \brief mobi_dma_swhs_sglreq:
 *   Starts a single transaction for the src/dst node when
 * using the sw handshaking interface.
 *
 * \param dmah: DMA handle
 * \param target:  DMA_NODE_IS_SRC or DMA_NODE_IS_DST
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval Zero   - upon success
 *
 * \remark
 *   This should be called when the client want to initiate a
 *   transfer that is not a multiple of the burst size.  See the
 *   DMAC databook for more details on using software handshake.
 *
 */
int32_t mobi_dma_swhs_sglreq(mobi_dma_handle dmah, uint32_t target)
{
	struct mobi_dma_channel *mobidma = NULL;
	int32_t ret;

	if ((ret = dma_validate_channel_access(dmah, (char*)__FUNCTION__)) != 0)
		return ret;

	mobidma = &mobi_dma_channels[dmah];

	if (target == DMA_NODE_IS_SRC) {
		DMAC_WRITEL(((1 << (dmah+8))|(1 << dmah)),
				MOBI_DMAC_REQ_SGL_SRC_OFFSET);
	}
	else {
		DMAC_WRITEL(((1 << (dmah+8))|(1 << dmah)),
				MOBI_DMAC_REQ_SGL_DST_OFFSET);
	}

	return 0;
}
EXPORT_SYMBOL(mobi_dma_swhs_sglreq);

/**
 * \brief mobi_dma_swhs_req:
 *   Starts a transaction for the src/dst node when using
 * 	the sw handshaking interface.
 *
 *
 * \param dmah: DMA handle
 * \param target:  DMA_NODE_IS_SRC or DMA_NODE_IS_DST
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval Zero   - upon success
 *
 * \remark
 *   This should be called when the client want to initiate a
 * transfer that is is equal or greater than the burst size.
 * It is not necessary to request a single transaction when
 * calling this function.  See the DMAC databook for more
 * details on using software handshake.
 */
int32_t mobi_dma_swhs_req(mobi_dma_handle dmah, uint32_t target)
{
	struct mobi_dma_channel *mobidma = NULL;
	int32_t ret;

	if ((ret = dma_validate_channel_access(dmah, (char*)__FUNCTION__)) != 0)
		return ret;

	mobidma = &mobi_dma_channels[dmah];

	if (target == DMA_NODE_IS_SRC) {
		DMAC_WRITEL(((1 << (dmah+8))|(1 << dmah)),
				MOBI_DMAC_REQ_SGL_SRC_OFFSET);
		DMAC_WRITEL(((1 << (dmah+8))|(1 << dmah)),
				MOBI_DMAC_REQ_SRC_OFFSET);
	}
	else {
		DMAC_WRITEL(((1 << (dmah+8))|(1 << dmah)),
				MOBI_DMAC_REQ_SGL_DST_OFFSET);
		DMAC_WRITEL(((1 << (dmah+8))|(1 << dmah)),
				MOBI_DMAC_REQ_DST_OFFSET);
	}
	return 0;
}
EXPORT_SYMBOL(mobi_dma_swhs_req);

/**
 * \brief mobi_dma_enable:
 *   starts DMA transfer
 *
 * \param dmah: DMA handle
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval -EBUSY  - cannot find an unlocked master bus
 *
 */
int32_t mobi_dma_enable(mobi_dma_handle dmah)
{
	struct mobi_dma_channel *mobidma = NULL;
	uint64_t ctl_reg = 0, ctl_mask = 0;
	uint64_t cfg_reg = 0, cfg_mask = 0;
	int32_t ret = 0;

	/*
	 * we can't allow any dma to start if the codec is being
	 * reset.  the suspend function will hold the lock until
	 * resume is called
	 */
	spin_lock(&codec_reset_mutex);

	if ((ret = dma_validate_channel_access(dmah, (char*)__FUNCTION__)) != 0) {
		spin_unlock(&codec_reset_mutex);
		return ret;
	}

	mobidma = &mobi_dma_channels[dmah];
	dprintk3("attempting to enable channel %d\n", dmah);

	/* request, free, enable, disable use common lock */
	spin_lock(&dmac_lock);

	mobidma->eventq.num = 0;
	mobidma->eventq.push_idx = 0;
	mobidma->eventq.pop_idx = 0;
	mobidma->error = 0;

	/* enable the interrupts, almost there! */
	ctl_reg |= MOBI_DMAC_CH0_CTL_INT_EN_W(1);
	ctl_mask |= MOBI_DMAC_CH0_CTL_INT_EN_MASK;

	/*
	 *  now do final update of cfg and ctl for interrupts and
	 *  any bus mastering stuff...
	 */
	update_cfg(dmah, cfg_mask, cfg_reg);
	update_ctl(dmah, ctl_mask, ctl_reg);

#if DW_DMAC_DEBUG
	DMAC_READD(DMAC_CHREG_ADDR(dmah, CFG), cfg_reg);
	DMAC_READD(DMAC_CHREG_ADDR(dmah, CTL), ctl_reg);
	dprintk2("enabling, cfg_reg = 0x%llx, ctl_reg = 0x%llx\n",
			cfg_reg, ctl_reg);
#endif

	/* everything ready, start DMA !! */
	DMAC_WRITEL(((1 << (dmah+8))|(1 << dmah)), MOBI_DMAC_CHANNEL_OFFSET);
	mobidma->enabled = 1;

	spin_unlock(&dmac_lock);
	spin_unlock(&codec_reset_mutex);

	return ret;
}
EXPORT_SYMBOL(mobi_dma_enable);

#define DMA_CHANNEL_SHUTDOWN 	0x1
#define DMA_CHANNEL_ABORT	 	0x2
int32_t dma_disable_channel(mobi_dma_handle dmah, uint8_t flags)
{
	uint64_t cfg_reg = 0, cfg_mask = 0;
	struct mobi_dma_channel *mobidma = NULL;
	int32_t counter;
	uint32_t chen_reg = 0;

	mobidma = &mobi_dma_channels[dmah];

	spin_lock(&dmac_lock);
	/* disable channel */
	DMAC_READL(MOBI_DMAC_CHANNEL_OFFSET, chen_reg);
	dprintk1("%s: chen_reg = 0x%x\n",
			(flags == DMA_CHANNEL_SHUTDOWN ? "shutdown" : "abort"),
			(chen_reg & (1 << dmah)));

	if ((mobidma->enabled == 0) && ((chen_reg & (1 << dmah)) == 0)) {
		dprintk1("channel already disabled, returning\n");
		goto out;
	}

	if (flags & DMA_CHANNEL_SHUTDOWN && mobidma->error == 0) {

		if (chen_reg & (1 << dmah)) {
			dprintk2("still enabled lets delay and check again before getting nasty\n");
			udelay(2000);
			DMAC_READL(MOBI_DMAC_CHANNEL_OFFSET, chen_reg);
		}
		if (chen_reg & (1 << dmah)) {
			/* then the dma transfer has not completed, lets delay and hope */
			dprintk2("channel still enabled, suspend and see if it finishes\n");
			cfg_reg = MOBI_DMAC_CH0_CFG_CH_SUSP_W(1);
			cfg_mask = MOBI_DMAC_CH0_CFG_CH_SUSP_MASK;
			update_cfg(dmah, cfg_mask, cfg_reg);

#define COUNTER_MAX 100000
			counter = 0;
			dprintk4("counting");
			do {
				DMAC_READD(DMAC_CHREG_ADDR(dmah, CFG), cfg_reg);
				counter++;
				if ((counter % 1000) == 0)
					dprintk(".");
			} while ((MOBI_DMAC_CH0_CFG_FIFO_EMPTY_R(cfg_reg) == 0) &&
					counter < COUNTER_MAX);
			dprintk("\n");

			if (counter == COUNTER_MAX)
				dprintk2("counter maxed, cfg_reg = 0x%llx\n", cfg_reg);

			if (MOBI_DMAC_CH0_CFG_FIFO_EMPTY_R(cfg_reg) == 0 &&
					counter == COUNTER_MAX) {
				warning("DMA FIFO for %s handle %d did not"
						" empty normally, aborting transfer",
						mobidma->name, dmah);
			}
			DMAC_WRITEL((1 << (dmah+8)| 0x0), MOBI_DMAC_CHANNEL_OFFSET);
			cfg_reg = MOBI_DMAC_CH0_CFG_CH_SUSP_W(0);
			cfg_mask = MOBI_DMAC_CH0_CFG_CH_SUSP_MASK;
			update_cfg(dmah, cfg_mask, cfg_reg);
		}
	}
	else {
		/* knock it to the floor */
		DMAC_WRITEL((1 << (dmah+8)| 0x0), MOBI_DMAC_CHANNEL_OFFSET);
		/* XXX this may not immediatly disabled the channel, should poll */
	}
	mobidma->enabled = 0;
	/* disable channel interrupts */
	update_ctl(dmah, MOBI_DMAC_CH0_CTL_INT_EN_MASK,
			MOBI_DMAC_CH0_CTL_INT_EN_W(0));

#if DW_DMAC_DEBUG
	{
		uint64_t ctl_reg = 0;
		DMAC_READD(DMAC_CHREG_ADDR(dmah, CFG), cfg_reg);
		DMAC_READD(DMAC_CHREG_ADDR(dmah, CTL), ctl_reg);
		dprintk2("cfg_reg = 0x%llx, ctl_reg = 0x%llx\n",
				cfg_reg, ctl_reg);
	}
#endif

out:
	spin_unlock(&dmac_lock);
	free_lli_mapping(mobidma);

	return 0;
}

/**
 * \brief mobi_dma_disable:
 *   Disable DMA channel
 *
 * \param dmah: DMA handle
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval Zero    - upon success
 *
 * \remark
 *   This function should be called after a transfer has completed.
 * If a transfer is still running and an error has not been
 * encountered an attempt is made to let the transfer complete and
 * disable the channel cleanly.  Note this does not release the channel
 * only disables it.
 *
 */
int32_t mobi_dma_disable(mobi_dma_handle dmah)
{
	int32_t ret;

	if ((ret = dma_validate_channel_access(dmah, (char*)__FUNCTION__)) != 0)
		return ret;
	return dma_disable_channel(dmah, DMA_CHANNEL_SHUTDOWN);
}
EXPORT_SYMBOL(mobi_dma_disable);

/**
 * \brief mobi_dma_abort:
 *   Abort transfer on DMA channel
 *
 * \param dmah: DMA handle
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval Zero    - upon success
 *
 * \remark
 * 		This function is used to immediately abort a running transfer.
 * 	No effort is made to shutdown cleanly and any data in transit will
 * 	be lost.
 *
 */
int32_t mobi_dma_abort(mobi_dma_handle dmah)
{
	int32_t ret;

	if ((ret = dma_validate_channel_access(dmah, (char*)__FUNCTION__)) != 0)
		return ret;
	return dma_disable_channel(dmah, DMA_CHANNEL_ABORT);
}
EXPORT_SYMBOL(mobi_dma_abort);

static void set_channel_defaults(struct mobi_dma_channel *mobidma)
{
	mobidma->channel = -1;
	mobidma->handler = NULL;
	mobidma->data = NULL;
	mobidma->mlist = NULL;
	mobidma->masterbus = 0;

	memset(&mobidma->ctl, 0x0, sizeof(mobidma->ctl));
	memset(&mobidma->cfg, 0x0, sizeof(mobidma->cfg));

	mobidma->ctl.sms = DMAC_AHB_MASTER_2;	/* master 1 is default */
	mobidma->ctl.dms = DMAC_AHB_MASTER_2;	/* master 1 is default */

	// we always know the know the length, DMAC is flow controller
	mobidma->flowctrl = DMA_FC_DMAC;

	/* the transaction width MUST be set by the client */
	mobidma->ctl.strwidth = -1;
	mobidma->ctl.dtrwidth = -1;

	mobidma->datawidth	  = 8;

	mobidma->src_flags  = 0x0;
	mobidma->dst_flags  = 0x0;
	mobidma->xfer_flags = 0x0;

	/* below this line we can define some defaults */
	mobidma->ctl.smsize = DMA_BURST_SIZE_8;
	mobidma->ctl.dmsize = DMA_BURST_SIZE_8;
	/* address increment type */
	mobidma->ctl.sinc = DMA_ADDRADJ_INC;
	mobidma->ctl.dinc = DMA_ADDRADJ_INC;

	/* HW handshaking, ignored for mem, can override */
	mobidma->cfg.dhs_sel = DMA_HW_HANDSHAKING;
	mobidma->cfg.shs_sel = DMA_HW_HANDSHAKING;

	/* arm is LE */
	mobidma->src_endianess = MOBI_DMA_CONFIG_ENDIAN_LE;
	mobidma->dst_endianess = MOBI_DMA_CONFIG_ENDIAN_LE;
	mobidma->endian_swap = DMA_ENDIANESS_SWAP_NONE;

	mobidma->src_sgparams.count = -1;
	mobidma->src_sgparams.interval = -1;
	mobidma->dst_sgparams.count = -1;
	mobidma->dst_sgparams.interval = -1;

	mobidma->cfg.src_per = 0;
	mobidma->cfg.dst_per = 0;
}

/**
 * \brief mobi_dma_request:
 *   Request/allocate a DMA channel.
 *
 * \param name: client's unique non-NULL identification
 *
 * \retval -EINVAL - if name is NULL
 * \retval -ENODEV - if a DMA channel is not available
 * \retval Zero    - upon success
 *
 */
mobi_dma_handle mobi_dma_request(const char *name, uint32_t options)
{
	mobi_dma_handle dma_ch = -1;
	struct mobi_dma_channel *mobidma = NULL;
	int i;

	/* basic sanity checks */
	if (!name)
		return -EINVAL;

	spin_lock(&dmac_lock);
	/* look for a NULL to indicate unused channel */
	for (i = 0; i < MAX_DMAC_CHANNELS; i++) {
		if (strcmp(mobi_dma_channels[i].name, "free") == 0) {
			dma_ch = i;
			mobidma = &mobi_dma_channels[dma_ch];
			break;
		}
	}
	if (mobidma == NULL) {
		error("No free DMA channel found");
		spin_unlock(&dmac_lock);
		return -ENODEV;
	}

	/* do some simple init, mask and clear all interrupts */
	/* but do not enable interrupts yet */
	mask_channel_interrupts(dma_ch);
	clear_channel_interrupts(dma_ch);

	set_channel_defaults(mobidma);

	strcpy(mobidma->name, name);
	mobidma->channel = dma_ch;
	dprintk2("found free dma channel %d for %s\n",
			dma_ch, mobidma->name);

	mobidma->options = options;

	spin_unlock(&dmac_lock);

	return dma_ch;
}
EXPORT_SYMBOL(mobi_dma_request);

/**
 * \brief mobi_dma_free:
 *   Release previously acquired channel
 *
 * \param dmah: DMA handle
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval -EPERM  - if channel has not been disabled
 * \retval Zero    - upon success
 *
 */
int mobi_dma_free(mobi_dma_handle dmah)
{
	struct mobi_dma_channel *mobidma = NULL;
	uint32_t chen_reg;
	int32_t ret;

	if ((ret = dma_validate_channel_access(dmah, (char*)__FUNCTION__)) != 0)
		return ret;

	mobidma = &mobi_dma_channels[dmah];

	spin_lock(&dmac_lock);

	DMAC_READL(MOBI_DMAC_CHANNEL_OFFSET, chen_reg);
	dprintk3("freeing channel %d used by %s, chen_reg = 0x%x\n",
			dmah, mobidma->name, (chen_reg & (1 << dmah)));

	if (chen_reg & (1 << dmah) || mobidma->enabled != 0) {
		error("Trying to free channel %d which has not been disabled",
				dmah);
		spin_unlock(&dmac_lock);
		return -EPERM;
	}

	set_channel_defaults(mobidma);
	/* this inidicates channel is free */
	strcpy(mobidma->name, "free");

	spin_unlock(&dmac_lock);

	return 0;
}
EXPORT_SYMBOL(mobi_dma_free);

int dmac_codec_suspend(void)
{
	uint32_t chen_reg;
	uint8_t n, channels_mask = 0;
	uint16_t timeout = 32768; /* ??? */
	/* hold this lock to prevent any new transfer from starting */
	spin_lock(&codec_reset_mutex);

	dprintk1("DMAC: Suspending DMA while resetting codec\n");

	for (n=0; n < MAX_DMAC_CHANNELS; n++)
		channels_mask |= ((channels_mask << n) | 0x1);
	/* now wait until all channels have completed, probably should
	 * have a timeout
	 */
	do {
		DMAC_READL(MOBI_DMAC_CHANNEL_OFFSET, chen_reg);
	} while ((chen_reg & channels_mask) && timeout--);

	if (chen_reg & channels_mask) {
		spin_unlock(&codec_reset_mutex);
		return -1;
	}
	else {
		return 0;
	}
}
EXPORT_SYMBOL(dmac_codec_suspend);

void dmac_codec_resume(void)
{
	dprintk1("DMAC: Resuming DMA after resetting codec\n");
	spin_unlock(&codec_reset_mutex);
}
EXPORT_SYMBOL(dmac_codec_resume);

static int irq_to_channel(uint32_t irq)
{
	switch (irq) {
		case MOBI_IRQ_DMAC0:
			return DMAC_CHANNEL_0;
		case MOBI_IRQ_DMAC1:
			return DMAC_CHANNEL_1;
		case MOBI_IRQ_DMAC2:
			return DMAC_CHANNEL_2;
		case MOBI_IRQ_DMAC3:
			return DMAC_CHANNEL_3;
		default:
			break;
	}
	return -1;
}

static int push_dma_event(struct dma_eventq *eventq, mobi_dma_event event)
{
	if (eventq->num == EVENTQ_SIZE)
		return -1;

	eventq->events[eventq->push_idx] = event;
	eventq->push_idx = (eventq->push_idx + 1) % EVENTQ_SIZE;
	eventq->num++;

	return 0;
}

static int __pop_dma_event(struct dma_eventq *eventq)
{
	mobi_dma_event event;

	if (eventq->num == 0)
		return -1;

	event = eventq->events[eventq->pop_idx];
	eventq->pop_idx = (eventq->pop_idx + 1) % EVENTQ_SIZE;
	eventq->num--;

	return event;
}

static int pop_dma_event(struct mobi_dma_channel *mobidma)
{
	mobi_dma_event event = 0;
	unsigned long flags;

	/* quickly disable intrrupts while we pop, this should avoid
	 *  a race with the interrupt handler pushing events
	 */
	local_save_flags(flags);
	local_irq_disable();
	event = __pop_dma_event(&mobidma->eventq);
	local_irq_restore(flags);

	return event;
}

void event_tasklet(unsigned long dmah)
{
	struct mobi_dma_channel *mobidma = &mobi_dma_channels[dmah];
	mobi_dma_event event = 0;

	while ((event = pop_dma_event(mobidma)) != -1) {
		if (mobidma->handler)
			mobidma->handler(mobidma->channel, event, mobidma->data);
	}
}

static irqreturn_t dma_irq_handler(int irq, void *dev_id)
{
	uint64_t status;
	struct mobi_dma_channel *mobidma;
	mobi_dma_handle dmah = irq_to_channel(irq);
	uint64_t ctl_reg;
	uint8_t i, do_task = 0;
	mobi_dma_event event_array[5];

	if (dmah < 0 || dmah > MAX_DMAC_CHANNELS)
		return IRQ_HANDLED;

	mobidma = &mobi_dma_channels[dmah];
	DMAC_READD(DMAC_CHREG_ADDR(dmah, CTL), ctl_reg);
	mobidma->error = 0;

	DMAC_READL(MOBI_DMAC_STATUS_ERR_OFFSET, status);
	if (status & (1 << dmah)) {
		dprintk1("ERROR status=0x%x\n", (uint32_t)status & 0xf);
		DMAC_WRITEL((1 << dmah), MOBI_DMAC_CLEAR_ERR_OFFSET);
		mobidma->error = 1;
		event_array[do_task++] = MOBI_DMA_EVENT_TRANSFER_ERROR;
	}

	DMAC_READL(MOBI_DMAC_STATUS_TFR_OFFSET, status);
	if (status & (1 << dmah)) {

#if DW_DMAC_DEBUG
		if (mobidma->list_entries) {

			void __iomem *plli = NULL;
			uint64_t bctl = 0;
			uint32_t block_ts, trwidth;
			int i, done, bytes, total_bytes = 0;

			dprintk1("multiblock XFER_COMPLETE\n");
			if (loglevel >= 5) {
				/* we only get one xfer complete interrupt at the end of
				 *  a multiblock transfer, but writeback is enabled so we
				 *  can read back the clt from the lli list in dtcm and
				 *  extract the important parts and show some stats
				 */
				for (i=0;i<mobidma->list_entries;i++) {
					done = 0;
					plli = mobidma->lli_iomap+
						(i*LLI_BLOCK_DESCRIPTOR_SIZE);
					bctl =
						((uint64_t)readl((uint32_t)(plli+LLI_CTLH_OFFSET))<< 32
						 | readl((uint32_t)(plli+LLI_CTLL_OFFSET)));

					block_ts = MOBI_DMAC_CH0_CTL_BLOCK_TS_R(bctl);
					done = MOBI_DMAC_CH0_CTL_DONE_R(bctl);
					trwidth =
						trwidth_reg2num(MOBI_DMAC_CH0_CTL_SRC_TR_WIDTH_R(bctl));
					bytes = ((block_ts*trwidth)/8);
					total_bytes += bytes;

					dprintk5("block %d: done %d block_ts 0x%x(%d), bytes %d, "
							"total_bytes %d\n",
							i, done, block_ts, block_ts, bytes, total_bytes);
				}
			}
		}
		else {
			dprintk1("single XFER_COMPLETE, block_ts 0x%x(%d), bytes %d\n",
					(int32_t)MOBI_DMAC_CH0_CTL_BLOCK_TS_R(ctl_reg),
					(int32_t)MOBI_DMAC_CH0_CTL_BLOCK_TS_R(ctl_reg),
					(((int32_t)MOBI_DMAC_CH0_CTL_BLOCK_TS_R(ctl_reg)*
					  trwidth_reg2num(
						  MOBI_DMAC_CH0_CTL_SRC_TR_WIDTH_R(ctl_reg)))/8));
		}
#endif
		DMAC_WRITEL((1 << dmah), MOBI_DMAC_CLEAR_TFR_OFFSET);
		event_array[do_task++] = MOBI_DMA_EVENT_TRANSFER_COMPLETE;
	}

	DMAC_READL(MOBI_DMAC_STATUS_BLOCK_OFFSET, status);
	if (status & (1 << dmah)) {
		dprintk1("BLOCK_COMPLETE, block_cnt %d\n", mobidma->blk_count);
#if DW_DMAC_DEBUG
		{
			void __iomem *plli =
				mobidma->lli_iomap+
				(mobidma->blk_count*LLI_BLOCK_DESCRIPTOR_SIZE);
			uint64_t bctl =
				((uint64_t)readl((uint32_t)(plli+LLI_CTLH_OFFSET))<< 32
				 | readl((uint32_t)(plli+LLI_CTLL_OFFSET)));
			uint32_t block_ts =
				MOBI_DMAC_CH0_CTL_BLOCK_TS_R(bctl);

			dprintk4("ctl readback: 0x%llx\n", bctl);
			dprintk4("block_ts 0x%x(%d), bytes %d\n",
					block_ts, block_ts,
					(block_ts*trwidth_reg2num(mobidma->ctl.strwidth))/8);
		}
#endif
		mobidma->blk_count++;
		DMAC_WRITEL((1 << dmah), MOBI_DMAC_CLEAR_BLOCK_OFFSET);
	}

	/* these two will be enabled when software handshaking is enabled */
	DMAC_READL(MOBI_DMAC_STATUS_SRC_TRAN_OFFSET, status);
	if (status & (1 << dmah)) {
		dprintk1("SRC_TRAN, block_ts 0x%x(%d)\n",
				(int32_t)MOBI_DMAC_CH0_CTL_BLOCK_TS_R(ctl_reg),
				(int32_t)MOBI_DMAC_CH0_CTL_BLOCK_TS_R(ctl_reg));

		DMAC_WRITEL((1 << dmah), MOBI_DMAC_CLEAR_SRC_TRAN_OFFSET);
		/* only send these back if doing dw handshaking */
		if (mobidma->cfg.shs_sel == DMA_SW_HANDSHAKING)
			event_array[do_task++] = MOBI_DMA_EVENT_SRC_TRANSACTION_COMPLETE;
	}

	DMAC_READL(MOBI_DMAC_STATUS_DST_TRAN_OFFSET, status);
	if (status & (1 << dmah)) {
		dprintk1("DST_TRAN, block_ts 0x%x(%d)\n",
				(int32_t)MOBI_DMAC_CH0_CTL_BLOCK_TS_R(ctl_reg),
				(int32_t)MOBI_DMAC_CH0_CTL_BLOCK_TS_R(ctl_reg));

		DMAC_WRITEL((1 << dmah), MOBI_DMAC_CLEAR_DST_TRAN_OFFSET);
		if (mobidma->cfg.dhs_sel == DMA_SW_HANDSHAKING)
			event_array[do_task++] = MOBI_DMA_EVENT_DST_TRANSACTION_COMPLETE;
	}

	if (do_task) {
		for (i=0; i < do_task; i++)
			push_dma_event(&mobidma->eventq, event_array[i]);

		switch (dmah) {
			case DMAC_CHANNEL_0:
				tasklet_hi_schedule(&dmach0_tasklet);
				break;
			case DMAC_CHANNEL_1:
				tasklet_hi_schedule(&dmach1_tasklet);
				break;
			case DMAC_CHANNEL_2:
				tasklet_hi_schedule(&dmach2_tasklet);
				break;
			case DMAC_CHANNEL_3:
				tasklet_hi_schedule(&dmach3_tasklet);
				break;
		}
	}

	return IRQ_HANDLED;
}

static char* irq_name[] = { "DMACH0", "DMACH1", "DMACH2", "DMACH3" };
static int __init mobi_dma_init(void)
{
	int ret;
	int i;
	uint64_t ch_mask, chwe_mask, ctrl_reg;
	uint32_t irq = 0;

#ifdef CONFIG_PROC_FS
#if DW_DMAC_DEBUG
	struct proc_dir_entry *pentry;
#endif
#endif

	/* if not set locally or by modparam, then init */
#if defined(LOCAL_DW_DMAC_DEBUG_ENABLE) || defined(CONFIG_DW_DMAC_DEBUG)
	if (loglevel == -1)
#if defined(CONFIG_DW_DMAC_DEBUG)
		loglevel = CONFIG_DW_DMAC_DEBUG_LEVEL;
#else
	loglevel = 0;
#endif
#endif

	if (loglevel != -1)
		printk(KERN_INFO "DMAC debug loglevel is %d\n", loglevel);

#ifndef CONFIG_DMAC_DESC_RAM_DISABLE
	dtcm_iobase = ioremap(DESC_RAM_BASE, DESC_RAM_SIZE);
	if (!dtcm_iobase) {
		error("Failed to ioremap DESC_RAM");
		return -EIO;
	}
#endif

	mobi_reset_disable(RESET_ID_DMAC);
	dmac_base = ioremap(DMAC_BASE, DMAC_SIZE);
	if (!dmac_base) {
		error("Failed to ioremap DMAC register space");
		if (dtcm_iobase != NULL)
			iounmap(dtcm_iobase);
		
		mobi_reset_enable(RESET_ID_DMAC);
		return -EIO;
	}

	spin_lock_init(&dmac_lock);
	spin_lock_init(&codec_reset_mutex);

#ifdef CONFIG_PROC_FS
	if (dma_proc_dir == NULL)
		dma_proc_dir = proc_mkdir("driver/dma", NULL);

	if (dma_proc_dir == NULL) {
		dma_proc_dir = proc_mkdir("driver", NULL);
		if (dma_proc_dir != NULL)
			proc_mkdir("driver/dma", NULL);
	}

	if (dma_proc_dir != NULL)
		create_proc_read_entry("driver/dma/status",
				S_IRUSR | S_IRGRP | S_IROTH,
				NULL, dma_state_proc_rd, NULL);

#if DW_DMAC_DEBUG
	pentry = create_proc_entry("driver/dma/debug",
			S_IRUSR | S_IRGRP | S_IROTH, NULL);
	if (pentry) {
		pentry->write_proc = mobidma_proc_wr_debug;
	}
#endif
#endif

	/* disable everything while we do some setup */
	DMAC_WRITED((uint64_t) 0x0, MOBI_DMAC_CFG_OFFSET);
	/* enable DMA module */
	DMAC_WRITEL(0x1, MOBI_DMAC_CFG_OFFSET);

	for (i = 0; i < MAX_DMAC_CHANNELS; i++) {
		switch (i) {
			case 0:  irq = MOBI_IRQ_DMAC0;
					 break;
			case 1:  irq = MOBI_IRQ_DMAC1;
					 break;
			case 2:  irq = MOBI_IRQ_DMAC2;
					 break;
			case 3:  irq = MOBI_IRQ_DMAC3;
					 break;
		}
		ret = request_irq(irq, dma_irq_handler, IRQF_DISABLED, irq_name[i], NULL);
		if (ret) {
			printk(KERN_CRIT "Could not register IRQ for DMA Channel i\n");
			continue;
		}
		spin_lock_init(&mobi_dma_channels[i].lock);
		strcpy(mobi_dma_channels[i].name, "free");
		/* make sure interrupts are disabled in channel control */
		DMAC_READD(DMAC_CHREG_ADDR(i, CTL), ctrl_reg);
		MOBI_DMAC_CH0_CTL_INT_EN_CLR(ctrl_reg);
		DMAC_WRITED(ctrl_reg, DMAC_CHREG_ADDR(i, CTL));
	}

	/* even if the DMAC does not control all the channels this code
	 * will be executed so we'll make sure everything is off now and
	 * if there is a dedicated channel the driver will handle everything
	 * else
	 */
	chwe_mask = MOBI_DMAC_CHANNEL0_WE_MASK |
		MOBI_DMAC_CHANNEL1_WE_MASK |
		MOBI_DMAC_CHANNEL2_WE_MASK |
		MOBI_DMAC_CHANNEL3_WE_MASK;

	ch_mask = MOBI_DMAC_CHANNEL0_ACCESS_MASK |
		MOBI_DMAC_CHANNEL1_ACCESS_MASK |
		MOBI_DMAC_CHANNEL2_ACCESS_MASK |
		MOBI_DMAC_CHANNEL3_ACCESS_MASK;

	/* mask all the channel interrupts */
	DMAC_WRITEL((chwe_mask|~(ch_mask)), MOBI_DMAC_MASK_TFR_OFFSET);
	DMAC_WRITEL((chwe_mask|~(ch_mask)), MOBI_DMAC_MASK_BLOCK_OFFSET);
	DMAC_WRITEL((chwe_mask|~(ch_mask)), MOBI_DMAC_MASK_SRC_TRAN_OFFSET);
	DMAC_WRITEL((chwe_mask|~(ch_mask)), MOBI_DMAC_MASK_DST_TRAN_OFFSET);
	DMAC_WRITEL((chwe_mask|~(ch_mask)), MOBI_DMAC_MASK_ERR_OFFSET);

	/* clear all channel interrupts */
	DMAC_WRITEL(ch_mask, MOBI_DMAC_CLEAR_TFR_OFFSET);
	DMAC_WRITEL(ch_mask, MOBI_DMAC_CLEAR_BLOCK_OFFSET);
	DMAC_WRITEL(ch_mask, MOBI_DMAC_CLEAR_SRC_TRAN_OFFSET);
	DMAC_WRITEL(ch_mask, MOBI_DMAC_CLEAR_DST_TRAN_OFFSET);
	DMAC_WRITEL(ch_mask, MOBI_DMAC_CLEAR_ERR_OFFSET);

	return ret;
}

arch_initcall(mobi_dma_init);
