/*
 *  linux/arch/arm/mach-falcon/clock.c
 *
 *  Copyright (C) 2006 Mobilygen Corp.
 *  Derived from mach-versatile/clock.c
 *
 *  Copyright (C) 2004 ARM Limited.
 *  Written by Deep Blue Solutions Limited.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
/** \file
 *
 * Driver for Mobilygen clock driver
 * \verbatim
 *
 *   This is a summary of the Mobilygen clock driver.  More details
 * can be found in the function documentation.  The driver provides
 * the following API calls.
 * 	mobi_clock_set_rate
 * 	  - this function is used to set the rate of a clock.  If
 * 	the new rate is 0, the clock will be disabled.   The driver
 * 	will attempt to program the clock to be as close to the requested
 * 	rate as possible without being greater than.  It is strongly
 * 	recommended to call mobi_clock_get_rate to get the new clock
 * 	rate.
 *
 * 	mobi_clock_get_rate
 * 	  - returns the current rate for a given clock
 *
 * The clk_* functions are only provided for compatibility
 * with existing linux code.
 *
 * \endverbatim
 *
 */

#ifndef DOXYGEN_SKIP

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/list.h>
#include <linux/errno.h>
#include <linux/err.h>
#include <linux/string.h>
#include <linux/clk.h>
#include <linux/mutex.h>
#include <linux/timex.h>
#include <linux/proc_fs.h>

#include <mach/platform.h>

#include "clock.h"

/* the default pll input clock is 48Mhz unless set on cmdline */
#define USB_OSC_RATE 48000000
unsigned long __system_pll_fin = USB_OSC_RATE;

static LIST_HEAD(clocks);
static DEFINE_MUTEX(clocks_mutex);
static struct proc_dir_entry *clock_proc_dir;

#undef FALCON_CLOCK_DEBUG

#ifdef FALCON_CLOCK_DEBUG
#define dprintk(x...)	printk(x)
#else
#define dprintk(x...)
#endif
#endif /* DOXYGEN_SKIP */

/*
 * Internal, looks up clk struct base on clk_id
 */
static struct clk *clock_get_by_id(clock_id_t clk_id)
{
	struct clk *p, *clk = NULL;

	mutex_lock(&clocks_mutex);
	list_for_each_entry(p, &clocks, node) {
		if (clk_id == p->id) {
			clk = p;
			break;
		}
	}
	mutex_unlock(&clocks_mutex);
	return clk;
}

static int32_t clk_ctrl_get(struct clk *clk, clk_ctrl_t *clk_ctrl)
{
	if (mobi_qcc_read(QCC_BID_CHIPCTL,
				clk->qcc_addr,
				(unsigned long *) &clk_ctrl->ctrl.reg,
				QCC_ACCESS_LEN_4)) {
		printk(KERN_ERR "Unable to read clock register, clk_id: %d\n",
				clk->id);
		return -1;
	}
	/* pll have two ctrl regs, ctrl1 is next address past ctrl0, ctrl0
	 * is the address stored in the qcc_addr field
	 */
	if (clk->id & CLOCK_CTRLREG_TYPE_PLL) {
		if (mobi_qcc_read(QCC_BID_CHIPCTL,
					clk->qcc_addr+4,
					(unsigned long *) &clk_ctrl->pll1.r,
					QCC_ACCESS_LEN_4)) {
			printk(KERN_ERR "Unable to read clock register, clk_id: %d\n",
					clk->id);
			return -1;
		}
	}

	return 0;
}

static int32_t clk_ctrl_set(struct clk *clk, clk_ctrl_t clk_ctrl)
{
	if (mobi_qcc_write(QCC_BID_CHIPCTL,
				clk->qcc_addr,
				clk_ctrl.ctrl.reg,
				QCC_ACCESS_LEN_4)) {
		printk(KERN_ERR "Unable to write clock register, clk_id: %d\n",
				clk->id);
		return -1;
	}
	/* pll have two ctrl regs, ctrl1 is next address past ctrl0, ctrl0
	 * is the address stored in the qcc_addr field
	 */
	if (clk->id & CLOCK_CTRLREG_TYPE_PLL) {
		if (mobi_qcc_write(QCC_BID_CHIPCTL,
					clk->qcc_addr+4,
					clk_ctrl.pll1.r,
					QCC_ACCESS_LEN_4)) {
			printk(KERN_ERR "Unable to write clock register, clk_id: %d\n",
					clk->id);
			return -1;
		}
	}
	return 0;
}

static int mobi_clock_enable(struct clk *clk)
{
	int32_t ret = 0;
	clk_ctrl_t clk_ctrl;

	/* can't enable everything so need to check */
	switch (clk->id) {
	case CLOCK_ID_PLL0:
	case CLOCK_ID_PLL1:
	case CLOCK_ID_PLL2:
	case CLOCK_ID_PLL3:
	case CLOCK_ID_PLL4:
	case CLOCK_ID_PLL5:
	case CLOCK_ID_PLL6:
	case CLOCK_ID_PLL7:
		clk_ctrl_get(clk, &clk_ctrl);
		clk_ctrl.ctrl.pll0.b.pll0config0_resetb = 1;
		clk_ctrl_set(clk, clk_ctrl);
		break;
	case CLOCK_ID_UARTDBG:
	case CLOCK_ID_UART0:
	case CLOCK_ID_UART1:
	case CLOCK_ID_I2C0:
	case CLOCK_ID_I2C1:
	case CLOCK_ID_TIMER:
	case CLOCK_ID_SDMMC:
		clk_ctrl_get(clk, &clk_ctrl);
		clk_ctrl.ctrl.hl.b.mmcgencontrol_clken = 1;
		clk_ctrl_set(clk, clk_ctrl);
		if (clk->gclk_id)
			mobi_gated_clock_enable(clk->gclk_id);
		break;
	default:
		printk(KERN_WARNING "Invalid clock, cannot disable"
				"clock_id %d\n", clk->id);
		return -1;
		break;
	}

	if (ret != 0)
		printk(KERN_ERR "ERROR - failed to disable clock_id %d\n",
				clk->id);

	return ret;
}


/*
 * just toggle the clk enable bit if it has one, must be holding
 * the clocks mutex when calling this
 */
static int mobi_clock_disable(struct clk *clk)
{
	int32_t ret = 0;
	clk_ctrl_t clk_ctrl;

	/* can't disable everything so need to check */
	switch (clk->id) {
	case CLOCK_ID_PLL0:
	case CLOCK_ID_PLL1:
	case CLOCK_ID_PLL2:
	case CLOCK_ID_PLL3:
	case CLOCK_ID_PLL4:
	case CLOCK_ID_PLL5:
	case CLOCK_ID_PLL6:
	case CLOCK_ID_PLL7:
		clk_ctrl_get(clk, &clk_ctrl);
		clk_ctrl.ctrl.pll0.b.pll0config0_resetb = 0;
		clk_ctrl_set(clk, clk_ctrl);
		break;
	case CLOCK_ID_UARTDBG:
	case CLOCK_ID_UART0:
	case CLOCK_ID_UART1:
	case CLOCK_ID_I2C0:
	case CLOCK_ID_I2C1:
	case CLOCK_ID_TIMER:
	case CLOCK_ID_SDMMC:
		clk_ctrl_get(clk, &clk_ctrl);
		clk_ctrl.ctrl.hl.b.mmcgencontrol_clken = 0;
		clk_ctrl_set(clk, clk_ctrl);
		if (clk->gclk_id)
			mobi_gated_clock_disable(clk->gclk_id);
		break;
	default:
		printk(KERN_WARNING "Invalid clock, cannot disable"
				"clock_id %d\n", clk->id);
		return -1;
		break;
	}

	if (ret != 0)
		printk(KERN_ERR "ERROR - failed to disable clock_id %d\n",
				clk->id);

	return ret;
}

#define PLL_SRC_SEL_EXT_CLKSEL  0
#define PLL_SRC_SEL_USB         1
#define PLL_SRC_SEL_VID0_INCLK  2
#define PLL_SRC_SEL_VID1_INCLK  3
static int32_t get_pll_fin(struct clk *clk, unsigned long *fin)
{
	int32_t ret = 0;
	clk_ctrl_t clk_ctrl;

	if (clk == NULL)
		return -EINVAL;

	if ((clk->id & CLOCK_CTRLREG_TYPE_PLL) == 0)
		return -EINVAL;

	mutex_lock(&clk->lock);
	if (clk_ctrl_get(clk, &clk_ctrl)) {
		ret = -EIO;
		goto out;
	}

	switch (clk_ctrl.ctrl.pll0.b.pll0config0_srcsel) {
	case PLL_SRC_SEL_EXT_CLKSEL:
		*fin = 27000000;
		break;
	case PLL_SRC_SEL_USB:
		*fin = 27000000;
		break;
	case PLL_SRC_SEL_VID0_INCLK:
		*fin = 27000000;
		break;
	case PLL_SRC_SEL_VID1_INCLK:
		*fin = 27000000;
		break;
	}

out:
	mutex_unlock(&clk->lock);

	return ret;
}

static int get_pll_freq(struct clk *clk, unsigned long *fout)
{
	unsigned long fin = 0;
	int ret = 0;
	clk_ctrl_t clk_ctrl;

	/* if fout upon return, clock is disabled */
	*fout = 0;
	
	if (clk == NULL)
		return -EINVAL;

	if ((clk->id & CLOCK_CTRLREG_TYPE_PLL) == 0)
		return -EINVAL;

	mutex_lock(&clk->lock);
	if (clk_ctrl_get(clk, &clk_ctrl)) {
		ret = -EIO;
		goto out;
	}

	if (clk_ctrl.ctrl.pll0.b.pll0config0_resetb == 0)
		goto out; /* clock is disabled */

	get_pll_fin(clk, &fin);
	if (fin <= 0)
		goto out;

	/* fout = (((fin/P)*M) / (2^S)) */
	*fout = ((fin/clk_ctrl.ctrl.pll0.b.pll0config0_p) *
			clk_ctrl.ctrl.pll0.b.pll0config0_m);

	/* 2^S */
	*fout /= ((2 << clk_ctrl.ctrl.pll0.b.pll0config0_s) >> 1);

out:
	mutex_unlock(&clk->lock);
	dprintk("pll id %d, fout %lu\n", clk_id, *fout);
	return ret;
}

static int get_pll_freq_by_id(clock_id_t clk_id, unsigned long *fout)
{
	return get_pll_freq(clock_get_by_id(clk_id), fout);
}

#define SRC_SEL_EXT_CLKSEL     	 0
#define SRC_SEL_DIV2EXT          1
#define SRC_SEL_PLL5             2
#define SRC_SEL_PLL0             3
#define SRC_SEL_PLL1             4
#define SRC_SEL_PLL2             5
#define SRC_SEL_PLL3             6
#define SRC_SEL_PLL4             7
#define SRC_SEL_PLL6             8
#define SRC_SEL_PLL7             9
static int32_t get_src_sel_freq(struct clk *clk, unsigned long *fin)
{
	int32_t ret = 0;
	clk_ctrl_t clk_ctrl;

	/* these two are always driven by PLL0 */
	if (clk->id == CLOCK_ID_ARM || clk->id == CLOCK_ID_AXI) {
		ret = get_pll_freq_by_id(CLOCK_ID_PLL0, fin);
	} else {
		mutex_lock(&clk->lock);
		clk_ctrl_get(clk, &clk_ctrl);
		mutex_unlock(&clk->lock);
		switch (clk_ctrl.ctrl.nr.b.qmmgencontrol_srcsel) {
		case SRC_SEL_EXT_CLKSEL:
			return 27000000; /* XXX to be completed */
			break;
		case SRC_SEL_DIV2EXT:
			return 27000000; /* XXX to be completed */
			break;
		case SRC_SEL_PLL0:
			ret = get_pll_freq_by_id(CLOCK_ID_PLL0, fin);
			break;
		case SRC_SEL_PLL1:
			ret = get_pll_freq_by_id(CLOCK_ID_PLL1, fin);
			break;
		case SRC_SEL_PLL2:
			ret = get_pll_freq_by_id(CLOCK_ID_PLL2, fin);
			break;
		case SRC_SEL_PLL3:
			ret = get_pll_freq_by_id(CLOCK_ID_PLL3, fin);
			break;
		case SRC_SEL_PLL4:
			ret = get_pll_freq_by_id(CLOCK_ID_PLL4, fin);
			break;
		case SRC_SEL_PLL5:
			ret = get_pll_freq_by_id(CLOCK_ID_PLL5, fin);
			break;
		case SRC_SEL_PLL6:
			ret = get_pll_freq_by_id(CLOCK_ID_PLL6, fin);
			break;
		case SRC_SEL_PLL7:
			ret = get_pll_freq_by_id(CLOCK_ID_PLL7, fin);
			break;
		}
	}
	return ret;
}

static int get_nr_freq(struct clk *clk, unsigned long *fout)
{
	unsigned long fin;
	int32_t ret = 0;
	clk_ctrl_t clk_ctrl;

	*fout = 0;

	if (clk == NULL)
		return -EINVAL;

	if ((clk->id & CLOCK_CTRLREG_TYPE_NR) == 0)
		return -EINVAL;

	mutex_lock(&clk->lock);
	if (clk_ctrl_get(clk, &clk_ctrl)) {
		ret = -EIO;
		goto out;
	}

	if (clk_ctrl.ctrl.nr.b.qmmgencontrol_clken == 0)
		goto out; /* clock is disabled */

	get_src_sel_freq(clk, &fin);
	if (fin <= 0)
		goto out;

	/* Fclko = Fin / (2*(((NR_FIELD_LIMIT-R)/N)+1)) */
	*fout = (fin/(2*(((NR_FIELD_LIMIT - clk_ctrl.ctrl.nr.b.qmmgencontrol_r) /
						clk_ctrl.ctrl.nr.b.qmmgencontrol_n) + 1)));
out:
	dprintk("%s(%d): N = %d, R = %d, fin = %lu, enabled = %d, fout = %lu\n",
			__func__, clk->id, N, R, fin, enabled, *fout);

	mutex_unlock(&clk->lock);
	return ret;
}

static int32_t get_hl_freq(struct clk *clk, unsigned long *fout)
{
	int ret = 0;
	unsigned long fin;
	clk_ctrl_t clk_ctrl;

	*fout = 0;

	if (clk == NULL)
		return -EINVAL;

	if ((clk->id & CLOCK_CTRLREG_TYPE_HL) == 0)
		return -EINVAL;

	mutex_lock(&clk->lock);
	if (clk_ctrl_get(clk, &clk_ctrl)) {
		ret = -EIO;
		goto out;
	}

	if (clk_ctrl.ctrl.hl.b.mmcgencontrol_clken == 0)
		goto out; /* clock is disabled */

	if (clk->gclk_id != GATED_CLOCK_ID_NONE) {
		if (mobi_gated_clock_state(clk->gclk_id) <= 0)
			goto out; /* clock is disabled */
	}

	get_src_sel_freq(clk, &fin);
	if (fin <= 0)
		goto out;

#ifdef CONFIG_FALCON_IS_FPGA
	/* XXX not sure about this ... ? */
	*fout = FALCON_SYSTEM_CLOCK;
#else
	/* Fclko = Fclki / (H+1) + (L+1) */
	*fout = fin / ((clk_ctrl.ctrl.hl.b.mmcgencontrol_high+1) +
			(clk_ctrl.ctrl.hl.b.mmcgencontrol_low+1));
#endif

out:
	dprintk("%s(%d): H = %d, L = %d\n\tfin = %lu, enabled = %d, fout = %lu\n",
			__func__, clk->id, H, L, fin, enabled, *fout);

	mutex_unlock(&clk->lock);
	return ret;
}

struct clk_calc_values {
	unsigned long calculated;
	int32_t M;
	uint8_t P;
	unsigned long difference;
};

/* determine the pll value to output a frequency
 * equation for pll output frequency is the following:
 * fout = (fin * intprog * sdiv)/sout
 */
static int32_t
set_pll_freq(struct clk *clk, unsigned long fout)
{
	int32_t S, M, P;
	int32_t S_start, S_limit, M_start, M_limit, P_start, P_limit;
	int32_t pow, best, ret = 0;
	struct clk_calc_values calc[8];
	unsigned long min_diff;
	clk_ctrl_t clk_ctrl;
	unsigned long fin;
	qcc_chipctl_pllstatus_t PLLStatus;

	/* yes, I know, constants bad, these refer the the limits of the S, M and P
	 * bits in the pll registers.  these are the defaults, taken from the HW
	 * designers test code.
	 */
	S_start = 0;
	S_limit = 5;
	M_start = 16;
	M_limit = 255;
	P_start = 1;
	P_limit = 63;

	if ((clk->id & CLOCK_CTRLREG_TYPE_PLL) == 0)
		return -EINVAL;

	mutex_lock(&clk->lock);
	/* fout == 0, means disable */
	if (fout == 0) {
		ret = mobi_clock_disable(clk);
		goto out;
	}

	if (clk_ctrl_get(clk, &clk_ctrl)) {
		ret = -EIO;
		goto out;
	}
	get_src_sel_freq(clk, &fin);
	if (fin <= 0)
		goto out;

	//memset(0x0, calc, sizeof(calc));

	/* PLL0    600MHz integer
	 * PLL1-4  600MHz fractional
	 * PLL5    1600MHz integer
	 * PLL6-7  1600MHz integer
	 */
	switch (clk->id) {
	 /* PLL1-4  600MHz fractional */
	case CLOCK_ID_PLL1:
	case CLOCK_ID_PLL2:
	case CLOCK_ID_PLL3:
	case CLOCK_ID_PLL4:
		S_limit = 4;
		break;
	case CLOCK_ID_PLL5:
	case CLOCK_ID_PLL6:
	case CLOCK_ID_PLL7:
		M_start = 64;
		break;
	}
		
	dprintk("%s: clk_id %d, requested freq %lu\n",
			__func__, pll->id, fout);

	/* fout = (((fin/P)*M) / (2^S)) */
	/* loop to do simple search for values */
	for (pow = S_start; pow <= S_limit; pow++) {
		/* S = 2^pow */
		S = ((2 << pow) >> 1);
		for (P = P_start; P < P_limit; P++) {
			for (M = M_start; M < M_limit; M++) {
				/*
				 * go thru the numbers until we find the first one greater than
				 *  the requested fout.  we want the freq to be <= to request
				 *  so deduct 1(unless already at 1) to get previous value
				 */
				calc[pow].calculated = ((__system_pll_fin / P) * M);
				calc[pow].calculated /= S;
				if (calc[pow].calculated > fout) {
					if (M != M_start) {
						calc[pow].calculated = ((__system_pll_fin / P) * (M-1));
						calc[pow].calculated /= S;
					}
					calc[pow].M = (M == M_start ? M : (M-1));;
					calc[pow].P = P;
					calc[pow].difference = fout - calc[pow].calculated;
					break;
				}
			}
		}
	}
	/* the index into the arrays is the register value of sdiv */
	for (S = 0, min_diff = calc[0].difference, best = 0;
			S <= S_limit; S++) {
		if (calc[S].difference < min_diff) {
			min_diff = calc[S].difference;
			best = S;
		}
	}

	clk_ctrl.ctrl.pll0.b.pll0config0_m = calc[best].M;
	clk_ctrl.ctrl.pll0.b.pll0config0_s = best;
	clk_ctrl.ctrl.pll0.b.pll0config0_p = calc[best].P;
	clk_ctrl.ctrl.pll0.b.pll0config0_load = 0;
	clk_ctrl.ctrl.pll0.b.pll0config0_constant = 0;
	if (clk_ctrl_set(clk, clk_ctrl)) {
		ret = -EIO;
		goto out;
	}
	clk_ctrl.ctrl.pll0.b.pll0config0_load = 1;
	if (clk_ctrl_set(clk, clk_ctrl)) {
		ret = -EIO;
		goto out;
	}
	clk_ctrl.ctrl.pll0.b.pll0config0_load = 0;
	if (clk_ctrl_set(clk, clk_ctrl)) {
		ret = -EIO;
		goto out;
	}

#ifndef CONFIG_FALCON_IS_FPGA
	/* poll for lock bit */
	do {
		if (mobi_qcc_read(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_PLLSTATUS_OFFSET,
					(unsinged long *) &PLLStatus.r,
					QCC_ACCESS_LEN_4)) {

				printk(KERN_ERR "Unable to read PLL status register\n");
				ret = -EIO;
				goto out;

			}
	} while (!PLLStatus.b.PLL0Locked);

#endif

out:
	if (ret < 0)
		printk(KERN_ERR "Unable to reprogram PLL rate\n");

	mutex_unlock(&clk->lock);

	return ret;
}

/*
 *  Find the values for N and N that will give us the closest
 *  freqency to the request rate without going over
 *  HL and NR dividers will adjust freq locally, but the input
 *  freq must be determined upstream
 */
static void nr_calculate_div(unsigned long fout,
		unsigned long fin, int *ndiv, int *rdiv)
{

	int N = 0, R = 1;
	unsigned long estimate = 0, min_diff = 0;

	/* Fclko = Fin / (2*(((NR_FIELD_LIMIT-R)/N)+1)) */
	/* maybe not the most efficient but should work */
	dprintk("nr_calc: fin = %lu, fout requested = %lu\n", fin, fout);
	if (fout == fin) {
		/* best we can do is fin/2 */
		*ndiv = 1;
		*rdiv = NR_FIELD_LIMIT;
	} else {
		for (N = 1; N < NR_FIELD_LIMIT; N++) {
			*ndiv = N;
			for (R = NR_FIELD_LIMIT-1; R > 0; R--) {
				estimate = fin/(2*(((NR_FIELD_LIMIT-R)/N)+1));
				if (estimate == fout) {
					goto done;
				} else if (estimate < fout) {
					if (min_diff > (fout - estimate))
						min_diff = fout - estimate;
					break;
				}
			}
		}
	}

done:
	dprintk("%s: N = %d, R = %d, fout given = %lu\n",
			__func__, *ndiv , *rdiv , estimate);
}

/*
 * handle the control for nr driven clocks
 */
static int32_t set_nr_freq(struct clk *clk, unsigned long fout)
{
	uint32_t ndiv, rdiv;
	int32_t ret = 0;
	unsigned long fin;
	clk_ctrl_t clk_ctrl;

	dprintk("%s: clk_id %d, requested freq %lu\n",
			__func__, clk->id, fout);

	if (clk == NULL)
		return -EINVAL;

	if ((clk->id & CLOCK_CTRLREG_TYPE_NR) == 0)
		return -EINVAL;

	mutex_lock(&clk->lock);
	/* fout == 0, means disable */
	if (fout == 0) {
		ret = mobi_clock_disable(clk);
		goto out;
	}

	if (clk_ctrl_get(clk, &clk_ctrl)) {
		ret = -EIO;
		goto out;
	}
	get_src_sel_freq(clk, &fin);
	if (fin <= 0)
		goto out;

	nr_calculate_div(fout, fin, &ndiv, &rdiv);

	clk_ctrl.ctrl.nr.b.qmmgencontrol_load = 0;
	clk_ctrl.ctrl.nr.b.qmmgencontrol_r = rdiv;
	clk_ctrl.ctrl.nr.b.qmmgencontrol_n = ndiv;
	if (clk_ctrl_set(clk, clk_ctrl)) {
		ret = -EIO;
		goto out;
	}
	clk_ctrl.ctrl.nr.b.qmmgencontrol_load = 1;
	if (clk_ctrl_set(clk, clk_ctrl)) {
		ret = -EIO;
		goto out;
	}
	clk_ctrl.ctrl.nr.b.qmmgencontrol_load = 0;
	if (clk_ctrl_set(clk, clk_ctrl)) {
		ret = -EIO;
		goto out;
	}

out:
#ifdef FALCON_CLOCK_DEBUG
	{
		unsigned long foutd;
		get_nr_freq(clk, &foutd);
		dprintk("%s: ret = %d, fin = %lu, fout req = %lu, fout got = %lu\n",
				__func__, ret, fin, fout, foutd);
	}
#endif

	mutex_unlock(&clk->lock);
	return ret;
}

/* this function handles the control for clocks driven by H/L dividers */
static int32_t set_hl_freq(struct clk *clk, unsigned long fout)
{
	int32_t ret = 0;
	uint32_t hdiv, ldiv;
	unsigned long fin = 0;
	uint32_t cycles;
	clk_ctrl_t clk_ctrl;

	if (clk == NULL)
		return -EINVAL;

	if ((clk->id & CLOCK_CTRLREG_TYPE_HL) == 0)
		return -EINVAL;

	mutex_lock(&clk->lock);
	/* fout == 0, means disable */
	if (fout == 0) {
		ret = mobi_clock_disable(clk);
		goto out;
	}

	if (clk_ctrl_get(clk, &clk_ctrl)) {
		ret = -EIO;
		goto out;
	}

	get_src_sel_freq(clk, &fin);
	if (fin <= 0)
		goto out;

	/*
	 * from Merlin clock spec, section HL Clock Divider
	 * To generate a clock, clko, from another clock clki where
	 * Fclko = Fclki / (H+1) + (L+1)
	 * initiall try equal duty cycle, so H = L which gives
	 * H = (Fclki - 2*Fclko) / 2 * Fclko
	 * H = (Fclki / 2* Fclko) - 1
	 */

	cycles = (fin + (fout >> 1)) / fout;
	hdiv = (cycles >> 1) - 1;
	ldiv = cycles - (hdiv) - 2;
	/* we don't want to be over, so increase one of the divisors */
	if (fin / ((hdiv + 1) + (ldiv + 1)) > fout)
		(hdiv)++;

	clk_ctrl.ctrl.hl.b.mmcgencontrol_load = 0;
	clk_ctrl.ctrl.hl.b.mmcgencontrol_low = ldiv;
	clk_ctrl.ctrl.hl.b.mmcgencontrol_high = hdiv;
	if (clk_ctrl_set(clk, clk_ctrl)) {
		ret = -EIO;
		goto out;
	}
	clk_ctrl.ctrl.hl.b.mmcgencontrol_load = 1;
	if (clk_ctrl_set(clk, clk_ctrl)) {
		ret = -EIO;
		goto out;
	}
	clk_ctrl.ctrl.hl.b.mmcgencontrol_load = 0;
	if (clk_ctrl_set(clk, clk_ctrl)) {
		ret = -EIO;
		goto out;
	}

out:
#ifdef FALCON_CLOCK_DEBUG
	{
		unsigned long foutd;
		get_hl_freq(clk, &foutd);
		dprintk("%s: ret = %d, fin = %lu, fout req = %lu, fout got = %lu\n",
				__func__, ret, fin, fout, foutd);
	}
#endif

	mutex_unlock(&clk->lock);
	return ret;
}

/**
 * \brief mobi_clock_get_rate:
 * 	Get the current output frequency of a clock
 *
 * \param clk_id  : id of the clock that an app is interested in.
 *
 * \retval zero   - if unable to determine frequency or invalid clk_id is given
 * \retval "freq" - a number greater than zero that is the
 * 		    frequency of the clock.
 *
 * \remark
 * 	Just to be clear, a return value of 0 is an indication
 * that something when wrong when trying to determing the
 * frequency of the clock.
 */
unsigned long mobi_clock_get_rate(clock_id_t clk_id)
{
	struct clk *clk = NULL;
	unsigned long freq = 0;
	int ret = 0;

	clk = clock_get_by_id(clk_id);
	if (clk == NULL)
		return 0;

	if (clk->id & CLOCK_CTRLREG_TYPE_PLL)
		ret = get_pll_freq(clk, &freq);
	else if (clk->id & CLOCK_CTRLREG_TYPE_NR)
		ret = get_nr_freq(clk, &freq);
	else if (clk->id & CLOCK_CTRLREG_TYPE_HL)
		ret = get_hl_freq(clk, &freq);
	else {
		switch (clk_id) {
		case CLOCK_ID_ARM:
			ret = get_pll_freq_by_id(CLOCK_ID_PLL0, &freq);
			break;
		case CLOCK_ID_AHB:
			ret = get_pll_freq_by_id(CLOCK_ID_PLL0, &freq);
			freq = freq >> 1;
			break;
		case CLOCK_ID_APB:
			ret = get_pll_freq_by_id(CLOCK_ID_PLL0, &freq);
			freq = freq >> 3;
			break;
			/*
		case CLOCK_ID_MEM:
			ret = get_scaler_freq(CLOCK_ID_SCALER1, &freq);
			freq>>=2;
			break;
		case CLOCK_ID_CORE:
			ret = get_pll_freq(CLOCK_ID_PLL0, &freq);
			freq>>=2;
			break;
			*/
		default:
			ret = 0;
		}
	}

	if (ret != 0)
		printk(KERN_ERR "%s:  Error when retrieving rate for clock %d: %d\n",
				__func__, clk->id, ret);

	return freq;
}
EXPORT_SYMBOL(mobi_clock_get_rate);

/**
 * \brief mobi_clock_set_rate
 * 	Set the frequency of a clock.
 *
 * \param clk_id  : id of the clock that an app is interested in.
 * \param freq    : new frequency for the the clock
 *
 * \retval -EINVAL - if invalid clk_id is given
 * \retval -EIO	   - if an error was encounted programming new freqency
 * \return Zero	   - upon success
 *
 * \remark
 * 	Passing in a frequency of '0' will disable the clock.
 */
int32_t mobi_clock_set_rate(clock_id_t clk_id, unsigned long freq)
{
	struct clk *clk = NULL;
	int ret = 0;

	clk = clock_get_by_id(clk_id);
	if (clk == NULL)
		return 0;

	if (clk->id & CLOCK_CTRLREG_TYPE_PLL) {
		ret = set_pll_freq(clk, freq);
	} else if (clk->id & CLOCK_CTRLREG_TYPE_NR) {
		if (clk->id == CLOCK_ID_ARM) {
			printk(KERN_ERR "ARM clock can not be changed\n");
			ret = -EINVAL;
		} else {
			ret = set_nr_freq(clk, freq);
		}
	} else if (clk->id & CLOCK_CTRLREG_TYPE_HL) {
		ret = set_hl_freq(clk, freq);
	} else if (clk->id & CLOCK_CTRLREG_TYPE_NONE) {
		switch (clk_id) {
		case CLOCK_ID_AHB:
			printk(KERN_ERR "AHB clock cannot be changed\n");
			ret = -EINVAL;
			break;
		case CLOCK_ID_APB:
			printk(KERN_ERR "APB clock cannot be changed\n");
			ret = -EINVAL;
			break;
			/*
		case CLOCK_ID_MEM:
			ret = get_scaler_freq(CLOCK_ID_SCALER1, &freq);
			freq>>=2;
			break;
		case CLOCK_ID_CORE:
			ret = get_pll_freq(CLOCK_ID_PLL0, &freq);
			freq>>=2;
			break;
			*/
		default:
			ret = -EINVAL;
		}
	} else {
		ret = -EINVAL;
	}

	/*
	 * if we get an error back it's likely
	 * we had some i/o error with the qcc
	 */
	if (ret != 0)
		ret = -EIO;

	return ret;
}
EXPORT_SYMBOL(mobi_clock_set_rate);

/**
 * \brief mobi_gated_clock_enable
 * 	 Enable a gated clock
 *
 * \param clk_id  : id of the gated clock to enable
 *
 * \retval -EINVAL  - if invalid gclk_id is given
 * \retval -EIO	   - if an error was encounted programming new state
 * \return Zero	   - upon success
 *
 */
int32_t mobi_gated_clock_enable(gated_clock_id_t gclk_id)
{
	int ret = 0;

	if (gclk_id & GATED_CLOCK_ID_VALID_MASK) {
		if (mobi_qcc_write(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_CLOCKENABLESET_OFFSET,
					gclk_id,
					QCC_ACCESS_LEN_4)) {
			printk(KERN_ERR "Failed to enable gated clk_id: %d\n",
					gclk_id);
			ret = -EIO;
		}
	} else {
		ret = -EINVAL;
	}

	return ret;
}

/**
 * \brief mobi_gated_clock_disable
 * 	 Disable a gated clock
 *
 * \param clk_id  : id of the gated clock to disable
 *
 * \retval -EINVAL  - if invalid gclk_id is given
 * \retval -EIO	   - if an error was encounted programming new state
 * \return Zero	   - upon success
 *
 */
int32_t mobi_gated_clock_disable(gated_clock_id_t gclk_id)
{
	int ret = 0;

	if (gclk_id & GATED_CLOCK_ID_VALID_MASK) {
		if (mobi_qcc_write(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_CLOCKENABLECLR_OFFSET,
					gclk_id,
					QCC_ACCESS_LEN_4)) {
			printk(KERN_ERR "Failed to enable gated clk_id: %d\n",
					gclk_id);
			ret = -EIO;
		}
	} else {
		ret = -EINVAL;
	}

	return ret;
}

/**
 * \brief mobi_gated_clock_state
 * 	 State a gated clock
 *
 * \param clk_id  : id of the gated clock to query
 *
 * \retval -EINVAL  - if invalid gclk_id is given
 * \retval -EIO	   - if an error was encounted querying
 * \return One	   - if clock is enabled
 * \return Zero	   - if clock is disabled
 *
 */
int32_t mobi_gated_clock_state(gated_clock_id_t gclk_id)
{
	unsigned long enable_state = 0;

	if (gclk_id & GATED_CLOCK_ID_VALID_MASK) {
		if (mobi_qcc_read(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_CLOCKENABLESET_OFFSET,
					&enable_state,
					QCC_ACCESS_LEN_4)) {
			printk(KERN_ERR "Failed to enable gated clk_id: %d\n",
					gclk_id);
			return -EIO;
		}
	} else {
		return -EINVAL;
	}

	return ((enable_state & gclk_id) ? 1 : 0);
}

/*
 * 	linux/include/linux/clk.h seems to define an API for a couple
 *  of clock functions, clk_get, clk_enable, clk_disable, clk_get_rate,
 *  clk_put.  It's not clear really how much these have to be supported?
 *  lots of other clock files do nothing, but we'll implement them since
 *  it is not to difficult
 */
void clk_put(struct clk *clk)
{
	module_put(clk->owner);
}
EXPORT_SYMBOL(clk_put);

unsigned long clk_get_rate(struct clk *clk)
{
	return mobi_clock_get_rate(clk->id);
}
EXPORT_SYMBOL(clk_get_rate);

int clk_set_rate(struct clk *clk, unsigned long rate)
{
	return mobi_clock_set_rate(clk->id, rate);
}
EXPORT_SYMBOL(clk_set_rate);

int clk_enable(struct clk *clk)
{
	int ret = 0;
	mutex_lock(&clk->lock);
	ret = mobi_clock_enable(clk);
	mutex_unlock(&clk->lock);
	return ret;
}
EXPORT_SYMBOL(clk_enable);

void clk_disable(struct clk *clk)
{
	mutex_lock(&clk->lock);
	mobi_clock_disable(clk);
	mutex_unlock(&clk->lock);
	return;
}
EXPORT_SYMBOL(clk_disable);

struct clk *clk_get(struct device *dev, const char *id)
{
	struct clk *p, *clk = ERR_PTR(-ENOENT);

	mutex_lock(&clocks_mutex);
	list_for_each_entry(p, &clocks, node) {
		if (strcmp(id, p->name) == 0 && try_module_get(p->owner)) {
			clk = p;
			break;
		}
	}
	mutex_unlock(&clocks_mutex);

	return clk;
}
EXPORT_SYMBOL(clk_get);

struct clk *mobi_clk_get(clock_id_t clk_id)
{
	struct clk *p, *clk = ERR_PTR(-ENOENT);

	p = clock_get_by_id(clk_id);
	if (p != NULL)
		clk = p;

	return clk;
}
EXPORT_SYMBOL(mobi_clk_get);

#define QCC_CHIPCTL_NONE_OFFSET 0x0
#define DEFINE_CLOCK(nm, i, qa, gid)       \
	static struct clk nm##_clk = {         \
		.name           = CLOCK_ID_##i##_NAME, \
		.id             = CLOCK_ID_##i,               \
		.qcc_addr       = QCC_CHIPCTL_##qa##_OFFSET,              \
		.gclk_id        = GATED_CLOCK_ID_##gid,             \
	}

DEFINE_CLOCK(arm, ARM, ARMGENCONTROL, NONE);
DEFINE_CLOCK(ahb, AHB, NONE, NONE);
DEFINE_CLOCK(apb, APB, NONE, APB_SS);
DEFINE_CLOCK(axi, AXI, AXIGENCONTROL, AXI_SS);
DEFINE_CLOCK(pll0, PLL0, PLL0CONFIG0, NONE);
DEFINE_CLOCK(pll1, PLL1, PLL1CONFIG0, NONE);
DEFINE_CLOCK(pll2, PLL2, PLL2CONFIG0, NONE);
DEFINE_CLOCK(pll3, PLL3, PLL3CONFIG0, NONE);
DEFINE_CLOCK(pll4, PLL4, PLL4CONFIG0, NONE);
DEFINE_CLOCK(pll5, PLL5, PLL5CONFIG0, NONE);
DEFINE_CLOCK(pll6, PLL6, PLL6CONFIG0, NONE);
DEFINE_CLOCK(pll7, PLL7, PLL7CONFIG0, NONE);
DEFINE_CLOCK(qmm,  QMM, QMMGENCONTROL, NONE);
DEFINE_CLOCK(uartdbg, UARTDBG, DBGUARTGENCONTROL, NONE);
DEFINE_CLOCK(uart0, UART0, UART0GENCONTROL, NONE);
DEFINE_CLOCK(uart1, UART1, UART1GENCONTROL, NONE);
DEFINE_CLOCK(i2c0, I2C0, I2C0GENCONTROL, NONE);
DEFINE_CLOCK(i2c1, I2C1, I2C1GENCONTROL, NONE);
DEFINE_CLOCK(timer, TIMER, TIMERGENCONTROL, NONE);
DEFINE_CLOCK(sdmmc, SDMMC, MMCGENCONTROL, AHB_SDMMC);
DEFINE_CLOCK(mem, MEM, MEMGENCONTROL, NONE);
DEFINE_CLOCK(core, CORE, NONE, NONE);

#undef MOBI_CLOCK_TESTING
#ifdef MOBI_CLOCK_TESTING
/* XXX not updated for Falcon */
static void pll_tst(enum clock_id clk_id)
{
	struct clk *clk = clock_get_by_id(clk_id);
	unsigned long freq;

	get_pll_freq(clk_id, &freq);
	printk("\n%s: Query of %s returned a freq = %lu\n",
			__func__, clk->name, freq);

	printk("%s settings are\n"
			"\tpd      = %lu\n"
			"\treset   = %lu\n"
			"\tlock    = %lu\n"
			"\tintprog = %lu\n"
			"\tsout    = %lu(%d)\n"
			"\tsdiv    = %lu(%d)\n"
			"\tvcosel  = %lu\n"
			"\tclken   = %lu\n",
			clk->name,
			CLOCK_READ_FIELD(PLL, PD, clk->shadow_reg),
			CLOCK_READ_FIELD(PLL, RESET, clk->shadow_reg),
			CLOCK_READ_FIELD(PLL, LOCK, clk->shadow_reg),
			CLOCK_READ_FIELD(PLL, INTPROG, clk->shadow_reg),
			CLOCK_READ_FIELD(PLL, SOUT, clk->shadow_reg),
			pll_sout_values[CLOCK_READ_FIELD(PLL, SOUT, clk->shadow_reg)],
			CLOCK_READ_FIELD(PLL, SDIV, clk->shadow_reg),
			pll_sdiv_values[CLOCK_READ_FIELD(PLL, SDIV, clk->shadow_reg)],
			CLOCK_READ_FIELD(PLL, VCOSEL, clk->shadow_reg),
			CLOCK_READ_FIELD(PLL, ENABLE, clk->shadow_reg));
	printk("\n");
}

void clk_internal_test(void)
{
	pll_tst(CLOCK_ID_PLL0);
}
EXPORT_SYMBOL(clk_internal_test);

#endif // MOBI_TESTING

#ifdef CONFIG_PROC_FS
static int clock_rate_proc_rd(char *buf, char **start,
		off_t offset, int count, int *eof, void *data)
{
	uint32_t clk_id = (uint32_t)data;
	int len = 0;

	len += sprintf(buf+len, "%lu\n", mobi_clock_get_rate(clk_id));
	*eof = 1;

	return len;
}
#endif

static void clk_add_procfs(struct clk *clk)
{
#ifdef CONFIG_PROC_FS
	struct proc_dir_entry *pentry;
	char string[50];

	sprintf(string, "driver/clock/%s", clk->name);
	proc_mkdir(string, NULL);
	sprintf(string, "driver/clock/%s/rate", clk->name);
	pentry = create_proc_entry(string,
			S_IRUSR | S_IRGRP | S_IROTH,
			NULL);
	if (pentry) {
		pentry->read_proc = clock_rate_proc_rd;
		pentry->data = (void *) clk->id;
	}
#endif
}

static int clk_register(struct clk *clk)
{
	mutex_lock(&clocks_mutex);
	mutex_init(&clk->lock);
	list_add(&clk->node, &clocks);
	clk_add_procfs(clk);

#ifdef CONFIG_MERLIN_IS_FPGA
	/* lets provide some numbers in the shadow register so that we run
	 *  the calculations.  done once here should minimize or remove most
	 *  other FPGA dependancies
	 */
	/* XXX for falcon, I'm going to reads of control register instead
	 * of shadow, the amount of reads is minimal. so this fixed value
	 * for fpga should be put in the ctrl read function
	 */
	clk->shadow_reg = 0x0;

	switch (clk->id) {
		case CLOCK_ID_PLL0:
		clk->cltrl.pll.r = 0x1cb71b0;
		clk->pll1_ctrl = 0x0; /* XXX unknown */
		break;
	case CLOCK_ID_PLL1:
	case CLOCK_ID_PLL2:
	case CLOCK_ID_PLL3:
	case CLOCK_ID_PLL4:
	case CLOCK_ID_PLL5:
	case CLOCK_ID_PLL6:
	case CLOCK_ID_PLL7:
		clk->cltrl.pll.r = 0x3;
		clk->pll1_ctrl = 0x0; /* XXX unknown */
		break;
	case CLOCK_ID_QMM:
		clk->cltrl.nr.r = 0x2001000;
		break;
	case CLOCK_ID_UARTDBG:
	case CLOCK_ID_UART0:
	case CLOCK_ID_UART1:
	case CLOCK_ID_I2C0:
	case CLOCK_ID_I2C1:
	case CLOCK_ID_TIMER:
	case CLOCK_ID_SDMMC:
	case CLOCK_ID_SSI0:
	case CLOCK_ID_SSI1:
		break;
	default:
		break;
	}
#endif

	mutex_unlock(&clocks_mutex);

	return 0;
}

/* creates a kernel command line variable so that we can be told
 *  what the pll is being driven at if it is not the default
 *  12MHz
 */
static int __init falcon_clk_get_system_pll_fin(char *str)
{
	char *pchar = NULL;
	__system_pll_fin = simple_strtoul(str, &pchar, 0);
	__system_pll_fin /= 2;

	/*
	 * unsigned long reg;
	 * even if clk_in is set on the kernel cmdline, we can look at the clksel
	 *  register in the PLL to see if we should actually use clk_in or the usb
	 *  osc.  this is useful for debug so we don't have to reconfig mboot
	 */
	/*
	if (mobi_qcc_read(QCC_BID_CHIPCTL, pll0_clk.qcc_addr, &reg, 4) == 0) {
		if ((reg & MOBI_CLK_PLL_SELECT_MASK) == 0)
			__system_pll_fin = USB_OSC_RATE;
	}
	*/
	return 1;
}
__setup("clk_in=", falcon_clk_get_system_pll_fin);

void __early_clk_init(void)
{
	clk_register(&pll0_clk);
	clk_register(&apb_clk);
	clk_register(&uartdbg_clk);
	clk_register(&timer_clk);
}
EXPORT_SYMBOL(__early_clk_init);

static int __init clk_init(void)
{

#ifdef CONFIG_PROC_FS
	if (clock_proc_dir == NULL)
		/* create /proc/drivers/clock if possible */
		clock_proc_dir = proc_mkdir("driver/clock", NULL);

	if (clock_proc_dir == NULL) {
		clock_proc_dir = proc_mkdir("driver", NULL);
		if (clock_proc_dir != NULL)
			proc_mkdir("driver/clock", NULL);
	}
#endif
	/* add entries to proc for the early registered clocks */
	clk_add_procfs(&pll0_clk);
	clk_add_procfs(&apb_clk);
	clk_add_procfs(&uartdbg_clk);
	clk_add_procfs(&timer_clk);

	clk_register(&arm_clk);
	clk_register(&ahb_clk);
	clk_register(&axi_clk);
	clk_register(&mem_clk);

	clk_register(&pll1_clk);
	clk_register(&pll2_clk);
	clk_register(&pll3_clk);
	clk_register(&pll4_clk);
	clk_register(&pll5_clk);
	clk_register(&pll6_clk);
	clk_register(&pll7_clk);

	clk_register(&uart0_clk);
	clk_register(&uart1_clk);
	clk_register(&i2c0_clk);
	clk_register(&i2c1_clk);
	clk_register(&sdmmc_clk);

	clk_register(&qmm_clk);
	clk_register(&core_clk);

	printk("CLOCK: Falcon input clock set to %luMHz\n",
			__system_pll_fin/1000000);
	printk("CLOCK: ARM running at %luMHz\n",
			mobi_clock_get_rate(CLOCK_ID_ARM)/1000000);
	/*
	printk("CLOCK: Core running at %luMHz\n",
			mobi_clock_get_rate(CLOCK_ID_CORE)/1000000);
	printk("CLOCK: Memory running at %luMHz\n",
			mobi_clock_get_rate(CLOCK_ID_MEM)/1000000);
	*/

	return 0;
}
core_initcall(clk_init);
