/*
 * table_array.c
 * $Id: table_array.c,v 1.2 2003/05/14 00:31:47 m4 Exp $
 */

#include <net-snmp/net-snmp-config.h>

#if HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#if ! defined(NDEBUG) && ! defined(NETSNMP_USE_ASSERT)
# define NETSNMP_TMP_NDEBUG
# define NDEBUG
#endif
#include <assert.h>
#if defined(NETSNMP_TMP_NDEBUG)
# undef NDEBUG
# undef NETSNMP_TMP_NDEBUG
#endif

#include <search.h>

#include <net-snmp/net-snmp-includes.h>
#include <net-snmp/agent/net-snmp-agent-includes.h>

#include <net-snmp/agent/table.h>
#include <net-snmp/agent/table_array.h>

#if HAVE_DMALLOC_H
#include <dmalloc.h>
#endif

/*
 * snmp.h:#define SNMP_MSG_INTERNAL_SET_BEGIN        -1 
 * snmp.h:#define SNMP_MSG_INTERNAL_SET_RESERVE1     0 
 * snmp.h:#define SNMP_MSG_INTERNAL_SET_RESERVE2     1 
 * snmp.h:#define SNMP_MSG_INTERNAL_SET_ACTION       2 
 * snmp.h:#define SNMP_MSG_INTERNAL_SET_COMMIT       3 
 * snmp.h:#define SNMP_MSG_INTERNAL_SET_FREE         4 
 * snmp.h:#define SNMP_MSG_INTERNAL_SET_UNDO         5 
 */

static const char *mode_name[] = {
    "Reserve 1",
    "Reserve 2",
    "Action",
    "Commit",
    "Free",
    "Undo"
};

/*
 * PRIVATE structure for holding important info for each table.
 */
typedef struct table_array_data_s {

   /** registration info for the table */
    netsnmp_table_registration_info *tblreg_info;

   /** container for the table rows */
    oid_array       array;

    /*
     * mutex_type                lock;
     */

   /** container for new/changing rows */
    oid_array       changing;

   /** do we want to group rows with the same index
    * together when calling callbacks? */
    int             group_rows;

   /** callbacks for this table */
    netsnmp_table_array_callbacks *cb;

} table_array_data;

/** @defgroup table_array table_array: Helps you implement a table when data can be stored locally. The data is stored in a sorted array, using a binary search for lookups.
 *  @ingroup table
 *
 *  The table_array handler is used (automatically) in conjuntion
 *  with the @link table table@endlink handler. It is primarily
 *  intended to be used with the mib2c configuration file
 *  mib2c.array-user.conf.
 *
 *  The code generated by mib2c is useful when you have control of
 *  the data for each row. If you cannot control when rows are added
 *  and deleted (or at least be notified of changes to row data),
 *  then this handler is probably not for you.
 *
 *  This handler makes use of callbacks (function pointers) to
 *  handle various tasks. Code is generated for each callback,
 *  but will need to be reviewed and flushed out by the user.
 *
 *  NOTE NOTE NOTE: Once place where mib2c is somewhat lacking
 *  is with regards to tables with external indices. If your
 *  table makes use of one or more external indices, please
 *  review the generated code very carefully for comments
 *  regarding external indices.
 *
 *  NOTE NOTE NOTE: This helper, the API and callbacks are still
 *  being tested and may change.
 *
 *  The generated code will define a structure for storage of table
 *  related data. This structure must be used, as it contains the index
 *  OID for the row, which is used for keeping the array sorted. You can
 *  add addition fields or data to the structure for your own use.
 *
 *  The generated code will also have code to handle SNMP-SET processing.
 *  If your table does not support any SET operations, simply comment
 *  out the #define <PREFIX>_SET_HANDLING (where <PREFIX> is your
 *  table name) in the header file.
 *
 *  SET processing modifies the row in-place. The duplicate_row
 *  callback will be called to save a copy of the original row.
 *  In the event of a failure before the commite phase, the
 *  row_copy callback will be called to restore the original row
 *  from the copy.
 *
 *  Code will be generated to handle row creation. This code may be
 *  disabled by commenting out the #define <PREFIX>_ROW_CREATION
 *  in the header file.
 *
 *  If your table contains a RowStatus object, by default the
 *  code will not allow object in an active row to be modified.
 *  To allow active rows to be modified, remove the comment block
 *  around the #define <PREFIX>_CAN_MODIFY_ACTIVE_ROW in the header
 *  file.
 *
 *  Code will be generated to maintain a secondary index for all
 *  rows, stored in a binary tree. This is very useful for finding
 *  rows by a key other than the OID index. By default, the functions
 *  for maintaining this tree will be based on a character string.
 *  NOTE: this will likely be made into a more generic mechanism,
 *  using new callback methods, in the near future.
 *
 *  The generated code contains many TODO comments. Make sure you
 *  check each one to see if it applies to your code. Examples include
 *  checking indices for syntax (ranges, etc), initializing default
 *  values in newly created rows, checking for row activation and
 *  deactivation requirements, etc.
 *
 * @{
 */

/**********************************************************************
 **********************************************************************
 *                                                                    *
 *                                                                    *
 * PUBLIC Registration functions                                      *
 *                                                                    *
 *                                                                    *
 **********************************************************************
 **********************************************************************/
/** register specified callbacks for the specified table/oid. If the
    group_rows parameter is set, the row related callbacks will be
    called once for each unique row index. Otherwise, each callback
    will be called only once, for all objects.
*/
int
netsnmp_register_table_array(netsnmp_handler_registration *reginfo,
                             netsnmp_table_registration_info *tabreg,
                             netsnmp_table_array_callbacks *cb,
                             int group_rows)
{
    table_array_data *tad = SNMP_MALLOC_TYPEDEF(table_array_data);
    tad->tblreg_info = tabreg;  /* we need it too, but it really is not ours */

    /*
     * check for required callbacks
     */
    if ((cb->can_set &&
         ((NULL==cb->duplicate_row) || (NULL==cb->delete_row) ||
         (NULL==cb->row_copy)) ) ||
        (cb->tree && (NULL==cb->row_compare))) {
        snmp_log(LOG_ERR, "table_array registration with incomplete "
                 "callback structure.\n");
        return SNMPERR_GENERR;
    }
    
    tad->array = netsnmp_initialize_oid_array(sizeof(void *));
    /*
     * netsnmp_mutex_init(&tad->lock); 
     */
    tad->changing = netsnmp_initialize_oid_array(sizeof(void *));
    tad->cb = cb;

    reginfo->handler->myvoid = tad;

    return netsnmp_register_table(reginfo, tabreg);
}

/** find the handler for the table_array helper. */
netsnmp_mib_handler *
netsnmp_find_table_array_handler(netsnmp_handler_registration *reginfo)
{
    netsnmp_mib_handler *mh = reginfo->handler;
    while (mh) {
        if (mh->access_method == netsnmp_table_array_helper_handler)
            break;
        mh = mh->next;
    }

    return mh;
}

/** find the context data used by the table_array helper */
oid_array      *
netsnmp_extract_array_context(netsnmp_request_info *request)
{
    return netsnmp_request_get_list_data(request, TABLE_ARRAY_NAME);
}

/** search for the specified index. This routine searches for
    modified rows first (i.e. during SET request processing).
*/
const netsnmp_oid_array_header *
netsnmp_table_array_get_by_index(netsnmp_handler_registration *reginfo,
                                 netsnmp_oid_array_header *hdr)
{
    table_array_data *tad;
    netsnmp_oid_array_header *rtn = NULL;
    netsnmp_mib_handler *mh;

    if (reginfo == NULL)
        return NULL;

    mh = netsnmp_find_table_array_handler(reginfo);
    if (mh == NULL)
        return NULL;

    tad = (table_array_data *) mh->myvoid;
    if (tad == NULL || tad->array == NULL)
        return NULL;

    /*
     * netsnmp_mutex_lock(&tad->lock);
     */

    if (tad->changing && netsnmp_get_oid_data_count(tad->changing)) {
        rtn = netsnmp_get_oid_data(tad->changing, hdr, 1);
        if (!rtn)
            rtn = netsnmp_get_oid_data(tad->array, hdr, 1);
    } else {
        rtn = netsnmp_get_oid_data(tad->array, hdr, 1);
    }

    /*
     * netsnmp_mutex_unlock(&tad->lock);
     */

    return rtn;
}

/** returns all rows in the data set with the same prefix. This
    is usefull when a table contains multiple indices.
*/
const netsnmp_oid_array_header **
netsnmp_table_array_get_subset(netsnmp_handler_registration *reginfo,
                               netsnmp_oid_array_header *hdr, int *len)
{
    table_array_data *tad;
    const netsnmp_oid_array_header **rtn = NULL;
    netsnmp_mib_handler *mh;

    if (reginfo == NULL)
        return NULL;

    mh = netsnmp_find_table_array_handler(reginfo);
    if (mh == NULL)
        return NULL;

    tad = (table_array_data *) mh->myvoid;
    if (!tad->array)
        return NULL;

    /*
     * netsnmp_mutex_lock(&tad->lock);
     */

    /*
     * I really really don't want to merge two tables. so just freak
     * out if this is called during a set. 
     */
    assert(!tad->changing
           || netsnmp_get_oid_data_count(tad->changing) == 0);

    rtn =
        (const netsnmp_oid_array_header **)
        netsnmp_get_oid_data_subset(tad->array, hdr, len);

    /*
     * netsnmp_mutex_unlock(&tad->lock);
     */

    return rtn;
}

/** this function is called to validate RowStatus transitions. */
int
netsnmp_table_array_check_row_status(netsnmp_table_array_callbacks *cb,
                                     netsnmp_oid_array_header *ctx_new,
                                     netsnmp_oid_array_header *ctx_old,
                                     netsnmp_array_group *ag,
                                     int *rs_new, int *rs_old)
{
    /*
     * xxx-rks: revisit row delete scenario
     */
    if (ctx_new) {
        /*
         * either a new row, or change to old row
         */
        /*
         * is it set to active?
         */
        if (RS_IS_GOING_ACTIVE(*rs_new)) {
            /*
             * is it ready to be active?
             */
            if ((NULL==cb->can_activate) ||
                cb->can_activate(ctx_old, ctx_new, ag))
                *rs_new = RS_ACTIVE;
            else
                return SNMP_ERR_INCONSISTENTVALUE;
        } else {
            /*
             * not going active
             */
            if (ctx_old) {
                /*
                 * change
                 */
                if (RS_IS_ACTIVE(*rs_old)) {
                    /*
                     * check pre-reqs for deactivation
                     */
                    if (cb->can_deactivate &&
                        !cb->can_deactivate(ctx_old, ctx_new, ag)) {
                        return SNMP_ERR_INCONSISTENTVALUE;
                    }
                }
            } else {
                /*
                 * new row
                 */
            }

            if (*rs_new != RS_DESTROY) {
                if ((NULL==cb->can_activate) ||
                    cb->can_activate(ctx_old, ctx_new, ag))
                    *rs_new = RS_NOTINSERVICE;
                else
                    *rs_new = RS_NOTREADY;
            }
        }
    } else {
        /*
         * check pre-reqs for delete row
         */
        if (cb->can_delete && !cb->can_delete(ctx_old, ctx_new, ag)) {
            return SNMP_ERR_INCONSISTENTVALUE;
        }
    }

    return SNMP_ERR_NOERROR;
}

/** @} */

#ifndef DOXYGEN_SHOULD_SKIP_THIS
/**********************************************************************
 **********************************************************************
 **********************************************************************
 **********************************************************************
 *                                                                    *
 *                                                                    *
 *                                                                    *
 *                                                                    *
 * EVERYTHING BELOW THIS IS PRIVATE IMPLEMENTATION DETAILS.           *
 *                                                                    *
 *                                                                    *
 *                                                                    *
 *                                                                    *
 **********************************************************************
 **********************************************************************
 **********************************************************************
 **********************************************************************/

/**********************************************************************
 **********************************************************************
 *                                                                    *
 *                                                                    *
 * Structures, Utility/convenience functions                          *
 *                                                                    *
 *                                                                    *
 **********************************************************************
 **********************************************************************/
/*
 * context info for SET requests
 */
typedef struct set_context_s {
    netsnmp_agent_request_info *agtreq_info;
    table_array_data *tad;
    int             status;
} set_context;

static void
release_netsnmp_array_group(netsnmp_oid_array_header *g, void *v)
{
    netsnmp_array_group_item *tmp;
    netsnmp_array_group *group = (netsnmp_array_group *) g;

    while (group->list) {
        tmp = group->list;
        group->list = tmp->next;
        free(tmp);
    }

    free(group);
}

static void
release_netsnmp_array_groups(void *vp)
{
    oid_array       a = (oid_array) vp;
    netsnmp_for_each_oid_data(a, release_netsnmp_array_group, NULL, 0);
}

inline netsnmp_oid_array_header *
find_next_row(netsnmp_table_request_info *tblreq_info,
              table_array_data * tad)
{
    netsnmp_oid_array_header *row = NULL;
    netsnmp_oid_array_header f_index;

    /*
     * below our minimum column?
     */
    if (tblreq_info->colnum < tad->tblreg_info->min_column) {
        tblreq_info->colnum = tad->tblreg_info->min_column;
        row = netsnmp_get_oid_data(tad->array, NULL, 0);
    } else {
        f_index.idx = tblreq_info->index_oid;
        f_index.idx_len = tblreq_info->index_oid_len;

        row = netsnmp_get_oid_data(tad->array, &f_index, 0);

        /*
         * we don't have a row, but we might be at the end of a
         * column, so try the next one.
         */
        if (!row) {
            ++tblreq_info->colnum;
            if (tad->tblreg_info->valid_columns) {
                tblreq_info->colnum = netsnmp_closest_column
                    (tblreq_info->colnum, tad->tblreg_info->valid_columns);
            } else if (tblreq_info->colnum > tad->tblreg_info->max_column)
                tblreq_info->colnum = 0;

            if (tblreq_info->colnum != 0)
                row = netsnmp_get_oid_data(tad->array, NULL, 0);
        }
    }

    return row;
}

inline void
build_new_oid(netsnmp_handler_registration *reginfo,
              netsnmp_table_request_info *tblreq_info,
              netsnmp_oid_array_header *row, netsnmp_request_info *current)
{
    oid             coloid[MAX_OID_LEN];
    int             coloid_len;

    coloid_len = reginfo->rootoid_len + 2;
    memcpy(coloid, reginfo->rootoid, reginfo->rootoid_len * sizeof(oid));

    /** table.entry */
    coloid[reginfo->rootoid_len] = 1;

    /** table.entry.column */
    coloid[reginfo->rootoid_len + 1] = tblreq_info->colnum;

    /** table.entry.column.index */
    memcpy(&coloid[reginfo->rootoid_len + 2], row->idx,
           row->idx_len * sizeof(oid));

    snmp_set_var_objid(current->requestvb, coloid,
                       reginfo->rootoid_len + 2 + row->idx_len);
}

/**********************************************************************
 **********************************************************************
 *                                                                    *
 *                                                                    *
 * GET procession functions                                           *
 *                                                                    *
 *                                                                    *
 **********************************************************************
 **********************************************************************/
inline int
process_get_requests(netsnmp_handler_registration *reginfo,
                     netsnmp_agent_request_info *agtreq_info,
                     netsnmp_request_info *requests,
                     table_array_data * tad)
{
    int             rc = SNMP_ERR_NOERROR;
    netsnmp_request_info *current;
    netsnmp_oid_array_header *row = NULL;
    netsnmp_table_request_info *tblreq_info;
    netsnmp_variable_list *var;

    /*
     * Loop through each of the requests, and
     * try to find the appropriate row from the oid_array.
     */
    for (current = requests; current; current = current->next) {

        var = current->requestvb;
        DEBUGMSGTL(("table_array:get",
                    "  process_get_request oid:"));
        DEBUGMSGOID(("table_array:get", var->name,
                     var->name_length));
        DEBUGMSG(("table_array:get", "\n"));

        /*
         * skip anything that doesn't need processing.
         */
        if (current->processed != 0) {
            DEBUGMSGTL(("table_array:get", "already processed\n"));
            continue;
        }

        /*
         * Get pointer to the table information for this request. This
         * information was saved by table_helper_handler. When
         * debugging, we double check a few assumptions. For example,
         * the table_helper_handler should enforce column boundaries.
         */
        tblreq_info = netsnmp_extract_table_info(current);
        assert(tblreq_info->colnum <= tad->tblreg_info->max_column);

        if ((agtreq_info->mode == MODE_GETNEXT) ||
            (agtreq_info->mode == MODE_GETBULK)) {
            /*
             * find the row
             */
            row = find_next_row(tblreq_info, tad);
            if (!row) {
                /*
                 * no results found.
                 *
                 * xxx-rks: how do we skip this entry for the next handler,
                 * but still allow it a chance to hit another handler?
                 */
                DEBUGMSGTL(("table_array:get", "no row found\n"));
                continue;
            }

            /*
             * * if data was found, make sure it has the column we want
             */
/* xxx-rks: add suport for sparse tables */

            /*
             * build new oid
             */
            build_new_oid(reginfo, tblreq_info, row, current);

        } /** GETNEXT/GETBULK */
        else {
            netsnmp_oid_array_header f_index;
            f_index.idx = tblreq_info->index_oid;
            f_index.idx_len = tblreq_info->index_oid_len;

            row = netsnmp_get_oid_data(tad->array, &f_index, 1);
            if (!row) {
                DEBUGMSGTL(("table_array:get", "no row found\n"));
                netsnmp_set_request_error(agtreq_info, current,
                                          SNMP_ERR_NOSUCHNAME);
                continue;
            }
        } /** GET */

        /*
         * get the data
         */
        rc = tad->cb->get_value(current, row, tblreq_info);

    } /** for ( ... requests ... ) */

    return rc;
}

/**********************************************************************
 **********************************************************************
 *                                                                    *
 *                                                                    *
 * SET procession functions                                           *
 *                                                                    *
 *                                                                    *
 **********************************************************************
 **********************************************************************/
inline void
group_requests(netsnmp_agent_request_info *agtreq_info,
               netsnmp_request_info *requests,
               oid_array netsnmp_array_group_tbl, table_array_data * tad)
{
    netsnmp_table_request_info *tblreq_info;
    netsnmp_variable_list *var;
    netsnmp_oid_array_header *row, *tmp, g_index;
    netsnmp_request_info *current;
    netsnmp_array_group *g;
    netsnmp_array_group_item *i;

    for (current = requests; current; current = current->next) {

        var = current->requestvb;
        /*
         * don't log OID, helper:table already did it 
         */
#if 0
        DEBUGMSGTL(("table_array:group", "  oid:"));
        DEBUGMSGOID(("table_array:group", var->name,
                     var->name_length));
        DEBUGMSG(("table_array:group", "\n"));
#endif
        /*
         * skip anything that doesn't need processing.
         */
        if (current->processed != 0) {
            DEBUGMSGTL(("table_array:group",
                        "already processed\n"));
            continue;
        }

        /*
         * 3.2.1 Setup and paranoia
         * *
         * * Get pointer to the table information for this request. This
         * * information was saved by table_helper_handler. When
         * * debugging, we double check a few assumptions. For example,
         * * the table_helper_handler should enforce column boundaries.
         */
        row = NULL;
        tblreq_info = netsnmp_extract_table_info(current);
        assert(tblreq_info->colnum <= tad->tblreg_info->max_column);

        /*
         * search for index
         */
        g_index.idx = tblreq_info->index_oid;
        g_index.idx_len = tblreq_info->index_oid_len;
        tmp = netsnmp_get_oid_data(netsnmp_array_group_tbl, &g_index, 1);
        if (tmp) {
            DEBUGMSGTL(("table_array:group",
                        "    existing group:"));
            DEBUGMSGOID(("table_array:group", g_index.idx,
                         g_index.idx_len));
            DEBUGMSG(("table_array:group", "\n"));
            g = (netsnmp_array_group *) tmp;
            i = SNMP_MALLOC_TYPEDEF(netsnmp_array_group_item);
            i->ri = current;
            i->tri = tblreq_info;
            i->next = g->list;
            g->list = i;
            continue;
        }

        DEBUGMSGTL(("table_array:group", "    new group"));
        DEBUGMSGOID(("table_array:group", g_index.idx,
                     g_index.idx_len));
        DEBUGMSG(("table_array:group", "\n"));
        g = SNMP_MALLOC_TYPEDEF(netsnmp_array_group);
        i = SNMP_MALLOC_TYPEDEF(netsnmp_array_group_item);
        g->list = i;
        g->table = tad->array;
        i->ri = current;
        i->tri = tblreq_info;

        /*
         * search for row. all changes are made to the original row,
         * later, we'll make a copy in old_row before we start processing.
         */
        row = g->new_row = netsnmp_get_oid_data(tad->array, &g_index, 1);
        if (!g->new_row) {
            if (!tad->cb->create_row) {
                netsnmp_set_request_error(agtreq_info, current,
                                          SNMP_ERR_NOSUCHNAME);
                free(g);
                free(i);
                continue;
            }
            /** use old_row temporarily */
            row = g->old_row = tad->cb->create_row(&g_index);
            if (!row) {
                netsnmp_set_request_error(agtreq_info, current,
                                          SNMP_ERR_GENERR);
                free(g);
                free(i);
                continue;
            }
        }

        g->index.idx = row->idx;
        g->index.idx_len = row->idx_len;

        netsnmp_add_oid_data(netsnmp_array_group_tbl, g);

    } /** for( current ... ) */
}

static void
process_set_group(netsnmp_oid_array_header *o, void *c)
{
    /* xxx-rks: should we continue processing after an error?? */
    set_context    *context = (set_context *) c;
    netsnmp_array_group *ag = (netsnmp_array_group *) o;

    switch (context->agtreq_info->mode) {

    case MODE_SET_RESERVE1:/** -> SET_RESERVE2 || SET_FREE */

        /*
         * if no row, copy new row from old_row
         */
        if (!ag->new_row) {
            ag->new_row = ag->old_row;
            ag->old_row = NULL;
        }
        else {
            /*
             * save a copy of row in old_row (undo info)
             */
            ag->old_row = context->tad->cb->duplicate_row(ag->new_row);
            if (!ag->old_row) {
                netsnmp_set_mode_request_error(MODE_SET_BEGIN,
                                               ag->list->ri,
                                               SNMP_ERR_RESOURCEUNAVAILABLE);
                return;
            }
        }
        
        if (context->tad->cb->set_reserve1)
            context->tad->cb->set_reserve1(ag);
        break;

    case MODE_SET_RESERVE2:/** -> SET_ACTION || SET_FREE */
        if (context->tad->cb->set_reserve2)
            context->tad->cb->set_reserve2(ag);
        break;

    case MODE_SET_ACTION:/** -> SET_COMMIT || SET_UNDO */
        /*
         * if we have and old_row, this row existed before.
         * if we have no old_row, this is a new row.
         * if table has secondary index, update it
         */
        if (ag->old_row) {
            /** if no row, it has been deleted */
            if (!ag->new_row) {
                /** remove deleted row */
                netsnmp_remove_oid_data(ag->table, ag->old_row, NULL);
                if (context->tad->cb->tree)
                    tdelete(ag->old_row,context->tad->cb->tree,
                            context->tad->cb->row_compare);
            }
        } else {
            /** insert new row */
            netsnmp_add_oid_data(ag->table, ag->new_row);
            if (context->tad->cb->tree)
                tsearch(ag->new_row,context->tad->cb->tree,
                        context->tad->cb->row_compare);
        }

        netsnmp_add_oid_data(context->tad->changing, ag->new_row);
        if (context->tad->cb->set_action)
            context->tad->cb->set_action(ag);
        break;

    case MODE_SET_COMMIT:/** FINAL CHANCE ON SUCCESS */
        if (context->tad->cb->set_commit)
            context->tad->cb->set_commit(ag);

        /** no more use for old_row, so free it */
        if (ag->old_row) {
            context->tad->cb->delete_row(ag->old_row);
            ag->old_row = NULL;
        }
        netsnmp_remove_oid_data(context->tad->changing, ag->new_row, NULL);


#if 0
        /* XXX-rks: finish row cooperative notifications
         * if the table has requested it, send cooperative notifications
         * for row operations.
         */
        if (context->tad->notifications) {
            if (ag->old_row) {
                if (!ag->new_row)
                    netsnmp_monitor_notify(EVENT_ROW_DEL);
                else
                    netsnmp_monitor_notify(EVENT_ROW_MOD);
            }
            else
                netsnmp_monitor_notify(EVENT_ROW_ADD);
        }
#endif
        break;

    case MODE_SET_FREE:/** FINAL CHANCE ON FAILURE */
        if (context->tad->cb->set_free)
            context->tad->cb->set_free(ag);

        /** no more use for old_row, so free it */
        if (ag->old_row) {
            context->tad->cb->delete_row(ag->old_row);
            ag->old_row = NULL;
        }
        break;

    case MODE_SET_UNDO:/** FINAL CHANCE ON FAILURE */
        if (ag->old_row) {
            /*
             * if we have old_row, this row existed before.
             */
            if (!ag->new_row) {
                /*
                 * old_row but no new_row means a deleted row.
                 * insert old_row
                 */
                netsnmp_add_oid_data(ag->table, ag->old_row);
                if (context->tad->cb->tree)
                    tsearch(ag->old_row,context->tad->cb->tree,
                            context->tad->cb->row_compare);
            }
        } else {
            /*
             * if we have no old_row, this was a new row.
             */
            assert(ag->new_row != NULL);
            netsnmp_remove_oid_data(ag->table, ag->new_row, NULL);
            if (context->tad->cb->tree)
                tdelete(ag->new_row,context->tad->cb->tree,
                        context->tad->cb->row_compare);
        }
        netsnmp_remove_oid_data(context->tad->changing, ag->new_row, NULL);

        /** status already set - don't change it now */
        if (context->tad->cb->set_undo)
            context->tad->cb->set_undo(ag);

        /** no more use for old_row, so free it */
        if (ag->old_row) {
            if (ag->new_row) {
                /*
                 * copy old_row to new_row (restore values)
                 */
                context->tad->cb->row_copy(ag->new_row, ag->old_row);
            }
            context->tad->cb->delete_row(ag->old_row);
            ag->old_row = NULL;
        }
        else {
            context->tad->cb->delete_row(ag->new_row);
            ag->new_row = NULL;
        }
        break;

    default:
        snmp_log(LOG_ERR, "unknown mode processing SET for "
                 "netsnmp_table_array_helper_handler\n");
        /**context->status = SNMP_ERR_GENERR*/ ;
        break;
    }
}

inline int
process_set_requests(netsnmp_agent_request_info *agtreq_info,
                     netsnmp_request_info *requests,
                     table_array_data * tad, char *handler_name)
{
    set_context     context;
    oid_array       netsnmp_array_group_tbl;

    /*
     * create and save structure for set info
     */
    netsnmp_array_group_tbl = (oid_array) netsnmp_agent_get_list_data
        (agtreq_info, handler_name);
    if (netsnmp_array_group_tbl == NULL) {
        netsnmp_data_list *tmp;
        netsnmp_array_group_tbl =
            netsnmp_initialize_oid_array(sizeof(void *));

        DEBUGMSGTL(("table_array", "Grouping requests by oid\n"));

        tmp = netsnmp_create_data_list(handler_name,
                                       netsnmp_array_group_tbl,
                                       release_netsnmp_array_groups);
        netsnmp_agent_add_list_data(agtreq_info, tmp);
        /*
         * group requests.
         */
        group_requests(agtreq_info, requests, netsnmp_array_group_tbl,
                       tad);
    }

    /*
     * process each group one at a time
     */
    context.agtreq_info = agtreq_info;
    context.tad = tad;
    context.status = SNMP_ERR_NOERROR;
    netsnmp_for_each_oid_data(netsnmp_array_group_tbl, process_set_group,
                              &context, 0);

    return context.status;
}


/**********************************************************************
 **********************************************************************
 *                                                                    *
 *                                                                    *
 * netsnmp_table_array_helper_handler()                               *
 *                                                                    *
 *                                                                    *
 **********************************************************************
 **********************************************************************/
int
netsnmp_table_array_helper_handler(netsnmp_mib_handler *handler,
                                   netsnmp_handler_registration *reginfo,
                                   netsnmp_agent_request_info *agtreq_info,
                                   netsnmp_request_info *requests)
{

    /*
     * First off, get our pointer from the handler. This
     * lets us get to the table registration information we
     * saved in get_table_array_handler(), as well as the
     * oid_array where the actual table data is stored.
     */
    int             rc = SNMP_ERR_NOERROR;
    table_array_data *tad = (table_array_data *) handler->myvoid;

    if (agtreq_info->mode < 0 || agtreq_info->mode > 5) {
        DEBUGMSGTL(("table_array", "Mode %d, Got request:\n",
                    agtreq_info->mode));
    } else {
        DEBUGMSGTL(("table_array", "Mode %s, Got request:\n",
                    mode_name[agtreq_info->mode]));
    }

    /*
     * 3.1.1
     *
     * This handler will be called 1 time for any type of GET
     * request, but will be called multiple times for SET
     * requests. We don't need to find each row for every
     * pass of the SET processing, so we'll cache results.
     */
    if (MODE_IS_SET(agtreq_info->mode)) {
        /*
         * netsnmp_mutex_lock(&tad->lock);
         */
        rc = process_set_requests(agtreq_info, requests,
                                  tad, handler->handler_name);
        /*
         * netsnmp_mutex_unlock(&tad->lock);
         */
    } else
        rc = process_get_requests(reginfo, agtreq_info, requests, tad);

    /*
     * Now we should have row pointers for each request. Call the
     * next handler to process the row.
     *
     * rc = netsnmp_call_next_handler(handler, reginfo, agtreq_info, requests);
     */

    return rc;
}
#endif /** DOXYGEN_SHOULD_SKIP_THIS */
