/*
 * Common routines for IPsec SA maintenance routines.
 *
 * Copyright (C) 1996, 1997  John Ioannidis.
 * Copyright (C) 1998, 1999, 2000, 2001  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: ipsec_sa.c,v 1.5 2003/05/23 21:30:10 swahl Exp $
 *
 * This is the file formerly known as "ipsec_xform.h"
 *
 */

#include <linux/config.h>
#include <linux/version.h>
#include <linux/kernel.h> /* printk() */

#include "ipsec_param.h"

#ifdef MALLOC_SLAB
# include <linux/slab.h> /* kmalloc() */
#else /* MALLOC_SLAB */
# include <linux/malloc.h> /* kmalloc() */
#endif /* MALLOC_SLAB */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/interrupt.h> /* mark_bh */

#include <linux/netdevice.h>   /* struct device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */
#include <linux/skbuff.h>
#include <linux/random.h>	/* get_random_bytes() */
#include <freeswan.h>
#ifdef SPINLOCK
#ifdef SPINLOCK_23
#include <linux/spinlock.h> /* *lock* */
#else /* SPINLOCK_23 */
#include <asm/spinlock.h> /* *lock* */
#endif /* SPINLOCK_23 */
#endif /* SPINLOCK */
#ifdef NET_21
#include <asm/uaccess.h>
#include <linux/in6.h>
#endif
#include <asm/checksum.h>
#include <net/ip.h>

#include "radij.h"

#include "ipsec_stats.h"
#include "ipsec_life.h"
#include "ipsec_sa.h"
#include "ipsec_xform.h"

#include "ipsec_encap.h"
#include "ipsec_radij.h"
#include "ipsec_netlink.h"
#include "ipsec_xform.h"
#include "ipsec_ipe4.h"
#include "ipsec_ah.h"
#include "ipsec_esp.h"

#include <pfkeyv2.h>
#include <pfkey.h>

#include "ipsec_proto.h"


#ifdef CONFIG_IPSEC_DEBUG
int debug_xform = 0;
#endif /* CONFIG_IPSEC_DEBUG */

#define SENDERR(_x) do { error = -(_x); goto errlab; } while (0)

struct ipsec_sa *ipsec_sadb_hash[SADB_HASHMOD];
#ifdef SPINLOCK
spinlock_t tdb_lock = SPIN_LOCK_UNLOCKED;
#else /* SPINLOCK */
spinlock_t tdb_lock;
#endif /* SPINLOCK */

int
ipsec_sadb_init(void)
{
	int i;

	for(i = 1; i < SADB_HASHMOD; i++) {
		ipsec_sadb_hash[i] = NULL;
	}
	return 0;
}

struct ipsec_sa *
ipsec_sa_getbyid(struct sa_id *said)
{
	int hashval;
	struct ipsec_sa *ips;
        char sa[SATOA_BUF];
	size_t sa_len;

	if(!said) {
		KLIPS_PRINT(debug_xform,
			    "klips_error:gettdb: "
			    "null pointer passed in!\n");
		return NULL;
	}

	sa_len = satoa(*said, 0, sa, SATOA_BUF);

	hashval = (said->spi+said->dst.s_addr+said->proto) % SADB_HASHMOD;
	
	KLIPS_PRINT(debug_xform,
		    "klips_debug:gettdb: "
		    "linked entry in tdb table for hash=%d of SA:%s requested.\n",
		    hashval,
		    sa_len ? sa : " (error)");

	if(!(ips = ipsec_sadb_hash[hashval])) {
		KLIPS_PRINT(debug_xform,
			    "klips_debug:gettdb: "
			    "no entries in tdb table for hash=%d of SA:%s.\n",
			    hashval,
			    sa_len ? sa : " (error)");
		return NULL;
	}

	for (; ips; ips = ips->ips_hnext) {
		if ((ips->ips_said.spi == said->spi) &&
		    (ips->ips_said.dst.s_addr == said->dst.s_addr) &&
		    (ips->ips_said.proto == said->proto)) {
			return ips;
		}
	}
	
	KLIPS_PRINT(debug_xform,
		    "klips_debug:gettdb: "
		    "no entry in linked list for hash=%d of SA:%s.\n",
		    hashval,
		    sa_len ? sa : " (error)");
	return NULL;
}

/*
  The tdb table better *NOT* be locked before it is handed in, or SMP locks will happen
*/
int
ipsec_sa_put(struct ipsec_sa *ips)
{
	int error = 0;
	unsigned int hashval;

	if(!ips) {
		KLIPS_PRINT(debug_xform,
			    "klips_error:puttdb: "
			    "null pointer passed in!\n");
		return -ENODATA;
	}
	hashval = ((ips->ips_said.spi + ips->ips_said.dst.s_addr + ips->ips_said.proto) % SADB_HASHMOD);

#ifndef NO_BRECIS_HW
	/* if (ips->ips_said.proto == IPPROTO_ESP) */
	{
		/*
		** Set up brecis hardware SA from the freeswan sa structure's
		** contents.
		*/

		int msp_sec_info[2] ;
#if 0
		int i ;
#endif
		msp_sec_enquire(msp_sec_info) ;

		ips->crypt_desc.ioctl.type = MSP_SEC_DES ;
		ips->crypt_desc.ioctl.mode = 0 ;
		ips->crypt_desc.next = 0 ;
		if (ips->ips_iv != 0)
			memcpy(&ips->crypt_desc.ioctl.ivhigh,
			       ips->ips_iv, 8) ;
		
		ips->auth_desc.ioctl.type = MSP_SEC_HMAC2 ;
		ips->auth_desc.next = &ips->auth_desc2 ;
		ips->auth_desc2.next = 0 ;

		if(ips->ips_flags & EMT_INBOUND)
		{
			ips->brecis_sa.flags =
				SAFLG_MODE_ESP_IN
				| SAFLG_CPI
				| SAFLG_CV
				| SAFLG_CBC_DECRYPT ;
		}
		else
		{
			ips->brecis_sa.flags =
				SAFLG_MODE_ESP_OUT
				| SAFLG_SI
				| SAFLG_CRI
				| SAFLG_CV
				| SAFLG_CBC_ENCRYPT ;
		}
		
#ifdef CONFIG_BRECIS_SEC_V2	/* AES only on v2 hw */
		if (ips->ips_encalg == ESP_AES)
		{
			switch(ips->ips_key_bits_e)
			{
			case 32:
			case 256:
				ips->brecis_sa.flags |= SAFLG_AES_256 ;
				break ;
			case 24:
			case 192:
				ips->brecis_sa.flags |= SAFLG_AES_192 ;
				break ;
			default:
				ips->brecis_sa.flags |= SAFLG_AES_128 ;
				break ;
			}
			
			memcpy(&ips->brecis_sa.crypt_keys[0],
			       ips->ips_key_e,
			       ips->ips_key_bits_e/8) ;

			if ( (ips->ips_flags & EMT_INBOUND) )
			{
				ips->brecis_sa.flags |= SAFLG_AES_DECRYPT ;
				msp_sec2_set_aes_decrypt_key(&ips->brecis_sa,
							     0,0,0) ;
			}
			
		}
		else
#endif /* CONFIG_BRECIS_SEC_V2 */
		if (ips->ips_encalg == ESP_3DES)
		{
			/* 3DES */
			if(ips->ips_flags & EMT_INBOUND)
			{
				ips->brecis_sa.flags |=
					SAFLG_3DES
					| SAFLG_DES_K1_DECRYPT
					| SAFLG_DES_K3_DECRYPT ;

				ips->crypt_desc.ioctl.mode =
					DMC_K1_DEC | DMC_K2_ENC | DMC_K3_DEC
					| DMC_3DES | DMC_MOD_ECB | DMC_MOD_DEC
					| DMC_MOD_CBC ;

				if(ips->ips_key_e)
				{
					memcpy(&ips->brecis_sa.crypt_keys[0],
					       &((struct des_eks *)(ips->ips_key_e))[2].ks[17],
					       8) ;
					memcpy(&ips->brecis_sa.crypt_keys[2],
					       &((struct des_eks *)(ips->ips_key_e))[1].ks[17],
					       8) ;
					memcpy(&ips->brecis_sa.crypt_keys[4],
					       &((struct des_eks *)(ips->ips_key_e))[0].ks[17],
					       8) ;
					memcpy(&ips->crypt_desc.ioctl.desc_key1high,
					       &ips->brecis_sa.crypt_keys[0],
					       24) ;
				}
			}
			else
			{
				ips->brecis_sa.flags |=
					SAFLG_3DES
					| SAFLG_DES_K2_DECRYPT ;

				ips->crypt_desc.ioctl.mode =
					DMC_K1_ENC | DMC_K2_DEC | DMC_K3_ENC
					| DMC_3DES | DMC_MOD_ECB | DMC_MOD_ENC
					| DMC_MOD_CBC ;

				if(ips->ips_key_e)
				{
					memcpy(&ips->brecis_sa.crypt_keys[0],
					       &((struct des_eks *)(ips->ips_key_e))[0].ks[17],
					       8) ;
					memcpy(&ips->brecis_sa.crypt_keys[2],
					       &((struct des_eks *)(ips->ips_key_e))[1].ks[17],
					       8) ;
					memcpy(&ips->brecis_sa.crypt_keys[4],
					       &((struct des_eks *)(ips->ips_key_e))[2].ks[17],
					       8) ;
					memcpy(&ips->crypt_desc.ioctl.desc_key1high,
					       &ips->brecis_sa.crypt_keys[0],
					       24) ;
				}
				if (ips->ips_iv)
					memcpy(&ips->crypt_desc.ioctl.ivhigh,
					       ips->ips_iv, 8) ;
			}
		}
		else if (ips->ips_encalg == ESP_DES)
		{
			/* DES */
			if(ips->ips_flags & EMT_INBOUND)
			{
				ips->brecis_sa.flags |=
					SAFLG_DES
					| SAFLG_DES_K1_DECRYPT;

				ips->crypt_desc.ioctl.mode =
					DMC_K1_DEC
					| DMC_MOD_ECB | DMC_MOD_DEC
					| DMC_MOD_CBC ;

				if(ips->ips_key_e)
				{
					memcpy(&ips->brecis_sa.crypt_keys[0],
					       &((struct des_eks *)(ips->ips_key_e))[0].ks[17],
					       8) ;
					memcpy(&ips->crypt_desc.ioctl.desc_key1high,
					       &ips->brecis_sa.crypt_keys[0],
					       8) ;
				}
			}
			else
			{
				ips->brecis_sa.flags |=
					SAFLG_DES;

				ips->crypt_desc.ioctl.mode =
					DMC_K1_ENC
					| DMC_MOD_ECB | DMC_MOD_ENC
					| DMC_MOD_CBC ;

				if(ips->ips_key_e)
				{
					memcpy(&ips->brecis_sa.crypt_keys[0],
					       &((struct des_eks *)(ips->ips_key_e))[0].ks[17],
					       8) ;
					memcpy(&ips->crypt_desc.ioctl.desc_key1high,
					       &ips->brecis_sa.crypt_keys[0],
					       8) ;
				}
				if (ips->ips_iv)
					memcpy(&ips->crypt_desc.ioctl.ivhigh,
					       ips->ips_iv, 8) ;
			}
		}
		ips->brecis_sa.esp_spi = ips->ips_said.spi ;
		ips->brecis_sa.esp_sequence = 1 ;
		if (ips->ips_authalg == AH_MD5)
		{
			ips->brecis_sa.flags |= SAFLG_MD5_96 ;
			ips->auth_desc.ioctl.mode = HMC_MD5 ;
		}
		else if (ips->ips_authalg == AH_SHA)
		{
			ips->brecis_sa.flags |= SAFLG_SHA1_96 ;
			ips->auth_desc.ioctl.mode = HMC_SHA1 ;
		}
		else
		{
			ips->brecis_sa.flags |= SAFLG_HASHNULL ;
			ips->auth_desc.ioctl.type = 0 ;
		}
		ips->auth_desc.ioctl.srcaddr = (void *)&ips->brecis_sa.hash_chain_a[0];
		ips->auth_desc.ioctl.dstaddr = (void *)&ips->brecis_sa.hash_chain_b[0];

		ips->brecis_sa.hash_init_len[0] = 0 ;
		ips->brecis_sa.hash_init_len[1] = 0x200 ;
		if (ips->ips_authalg == AH_MD5)
		{
			memcpy(ips->brecis_sa.hash_chain_a,
			       ((struct md5_ctx*)(ips->ips_key_a))->ictx.state,
			       16) ;
			memcpy(ips->brecis_sa.hash_chain_b,
			       ((struct md5_ctx*)(ips->ips_key_a))->octx.state,
			       16) ;
		}
		else if (ips->ips_authalg == AH_SHA)
		{
			memcpy(ips->brecis_sa.hash_chain_a,
			       ((struct sha1_ctx*)(ips->ips_key_a))->ictx.state,
			       20) ;
			memcpy(ips->brecis_sa.hash_chain_b,
			       ((struct sha1_ctx*)(ips->ips_key_a))->octx.state,
			       20) ;
		}

		if(    ( (msp_sec_info[0] & SECENQ_HW2) == 0 )
		    || ( (msp_sec_info[1] & SECENQ_API2) == 0 ) )
		{
			/* mark the sa so we will use the other API */
			ips->brecis_sa.flags = 0 ;
		}

#if 0
		printk("BRECIS SA STRUCTURE\n  brecis_sa @ %p:\n", &ips->brecis_sa) ;
		for(i = 0 ; i < sizeof(ips->brecis_sa) ; i++)
		{
			if ( i % 16 == 0 )
				printk("    %04x:", i) ;
			printk(" %02x", ((unsigned char *)&ips->brecis_sa)[i]) ;
			if ( i % 16 == 15 )
				printk("\n") ;
		}
		if ( i % 16 != 0 )
			printk("\n") ;
				
		printk("  crypt_desc @ %p:\n", &ips->crypt_desc) ;
		for(i = 0 ; i < sizeof(ips->crypt_desc) ; i++)
		{
			if ( i % 16 == 0 )
				printk("    %04x:", i) ;
			printk(" %02x", ((unsigned char *)&ips->crypt_desc)[i]) ;
			if ( i % 16 == 15 )
				printk("\n") ;
		}
		if ( i % 16 != 0 )
			printk("\n") ;
				
		printk("  auth_desc @ %p:\n", &ips->auth_desc) ;
		for(i = 0 ; i < sizeof(ips->auth_desc) ; i++)
		{
			if ( i % 16 == 0 )
				printk("    %04x:", i) ;
			printk(" %02x", ((unsigned char *)&ips->brecis_sa)[i]) ;
			if ( i % 16 == 15 )
				printk("\n") ;
		}
		if ( i % 16 != 0 )
			printk("\n") ;
		printk("\n") ;

#endif
	}
		
		
	
#endif
	spin_lock_bh(&tdb_lock);
	
	ips->ips_hnext = ipsec_sadb_hash[hashval];
	ipsec_sadb_hash[hashval] = ips;
	
	spin_unlock_bh(&tdb_lock);

	return error;
}

/*
  The tdb table better be locked before it is handed in, or races might happen
*/
int
ipsec_sa_del(struct ipsec_sa *ips)
{
	unsigned int hashval;
	struct ipsec_sa *tdbtp;
        char sa[SATOA_BUF];
	size_t sa_len;

	if(!ips) {
		KLIPS_PRINT(debug_xform,
			    "klips_error:deltdb: "
			    "null pointer passed in!\n");
		return -ENODATA;
	}
	
	sa_len = satoa(ips->ips_said, 0, sa, SATOA_BUF);
	if(ips->ips_inext || ips->ips_onext) {
		KLIPS_PRINT(debug_xform,
			    "klips_error:deltdb: "
			    "SA:%s still linked!\n",
			    sa_len ? sa : " (error)");
		return -EMLINK;
	}
	
	hashval = ((ips->ips_said.spi + ips->ips_said.dst.s_addr + ips->ips_said.proto) % SADB_HASHMOD);
	
	KLIPS_PRINT(debug_xform,
		    "klips_debug:deltdb: "
		    "deleting SA:%s, hashval=%d.\n",
		    sa_len ? sa : " (error)",
		    hashval);
	if(!ipsec_sadb_hash[hashval]) {
		KLIPS_PRINT(debug_xform,
			    "klips_debug:deltdb: "
			    "no entries in tdb table for hash=%d of SA:%s.\n",
			    hashval,
			    sa_len ? sa : " (error)");
		return -ENOENT;
	}
	
	if (ips == ipsec_sadb_hash[hashval]) {
		ipsec_sadb_hash[hashval] = ipsec_sadb_hash[hashval]->ips_hnext;
		ips->ips_hnext = NULL;
		KLIPS_PRINT(debug_xform,
			    "klips_debug:deltdb: "
			    "successfully deleted first tdb in chain.\n");
		return 0;
	} else {
		for (tdbtp = ipsec_sadb_hash[hashval];
		     tdbtp;
		     tdbtp = tdbtp->ips_hnext) {
			if (tdbtp->ips_hnext == ips) {
				tdbtp->ips_hnext = ips->ips_hnext;
				ips->ips_hnext = NULL;
				KLIPS_PRINT(debug_xform,
					    "klips_debug:deltdb: "
					    "successfully deleted link in tdb chain.\n");
				return 0;
			}
		}
	}
	
	KLIPS_PRINT(debug_xform,
		    "klips_debug:deltdb: "
		    "no entries in linked list for hash=%d of SA:%s.\n",
		    hashval,
		    sa_len ? sa : " (error)");
	return -ENOENT;
}

/*
  The tdb table better be locked before it is handed in, or races might happen
*/
int
ipsec_sa_delchain(struct ipsec_sa *ips)
{
	struct ipsec_sa *tdbdel;
	int error = 0;
        char sa[SATOA_BUF];
	size_t sa_len;

	if(!ips) {
		KLIPS_PRINT(debug_xform,
			    "klips_error:deltdbchain: "
			    "null pointer passed in!\n");
		return -ENODATA;
	}

	sa_len = satoa(ips->ips_said, 0, sa, SATOA_BUF);
	KLIPS_PRINT(debug_xform,
		    "klips_debug:deltdbchain: "
		    "passed SA:%s\n",
		    sa_len ? sa : " (error)");
	while(ips->ips_onext) {
		ips = ips->ips_onext;
	}

	while(ips) {
		/* XXX send a pfkey message up to advise of deleted TDB */
		sa_len = satoa(ips->ips_said, 0, sa, SATOA_BUF);
		KLIPS_PRINT(debug_xform,
			    "klips_debug:deltdbchain: "
			    "unlinking and delting SA:%s",
			    sa_len ? sa : " (error)");
		tdbdel = ips;
		ips = ips->ips_inext;
		if(ips) {
			sa_len = satoa(ips->ips_said, 0, sa, SATOA_BUF);
			KLIPS_PRINT(debug_xform,
				    ", inext=%s",
				    sa_len ? sa : " (error)");
			tdbdel->ips_inext = NULL;
			ips->ips_onext = NULL;
		}
		KLIPS_PRINT(debug_xform,
			    ".\n");
		if((error = ipsec_sa_del(tdbdel))) {
			KLIPS_PRINT(debug_xform,
				    "klips_debug:deltdbchain: "
				    "deltdb returned error %d.\n", -error);
			return error;
		}
		if((error = ipsec_sa_wipe(tdbdel))) {
			KLIPS_PRINT(debug_xform,
				    "klips_debug:deltdbchain: "
				    "ipsec_tdbwipe returned error %d.\n", -error);
			return error;
		}
	}
	return error;
}

int 
ipsec_sadb_cleanup(__u8 proto)
{
	int i;
	int error = 0;
	struct ipsec_sa *ips, **ipsprev, *tdbdel;
        char sa[SATOA_BUF];
	size_t sa_len;

	KLIPS_PRINT(debug_xform,
		    "klips_debug:ipsec_tdbcleanup: "
		    "cleaning up proto=%d.\n",
		    proto);

	spin_lock_bh(&tdb_lock);

	for (i = 0; i < SADB_HASHMOD; i++) {
		ipsprev = &(ipsec_sadb_hash[i]);
		ips = ipsec_sadb_hash[i];
		for(; ips;) {
			sa_len = satoa(ips->ips_said, 0, sa, SATOA_BUF);
			KLIPS_PRINT(debug_xform,
				    "klips_debug:ipsec_tdbcleanup: "
				    "checking SA:%s, hash=%d",
				    sa_len ? sa : " (error)",
				    i);
			tdbdel = ips;
			ips = tdbdel->ips_hnext;
			if(ips) {
				sa_len = satoa(ips->ips_said, 0, sa, SATOA_BUF);
				KLIPS_PRINT(debug_xform,
					    ", hnext=%s",
					    sa_len ? sa : " (error)");
			}
			if(*ipsprev) {
				sa_len = satoa((*ipsprev)->ips_said, 0, sa, SATOA_BUF);
				KLIPS_PRINT(debug_xform,
					    ", *ipsprev=%s",
					    sa_len ? sa : " (error)");
				if((*ipsprev)->ips_hnext) {
					sa_len = satoa((*ipsprev)->ips_hnext->ips_said, 0, sa, SATOA_BUF);
					KLIPS_PRINT(debug_xform,
						    ", *ipsprev->ips_hnext=%s",
						    sa_len ? sa : " (error)");
				}
			}
			KLIPS_PRINT(debug_xform,
				    ".\n");
			if(!proto || (proto == tdbdel->ips_said.proto)) {
				sa_len = satoa(tdbdel->ips_said, 0, sa, SATOA_BUF);
				KLIPS_PRINT(debug_xform,
					    "klips_debug:ipsec_tdbcleanup: "
					    "deleting SA chain:%s.\n",
					    sa_len ? sa : " (error)");
				if((error = ipsec_sa_delchain(tdbdel))) {
					SENDERR(-error);
				}
				ipsprev = &(ipsec_sadb_hash[i]);
				ips = ipsec_sadb_hash[i];

				KLIPS_PRINT(debug_xform,
					    "klips_debug:ipsec_tdbcleanup: "
					    "deleted SA chain:%s",
					    sa_len ? sa : " (error)");
				if(ips) {
					sa_len = satoa(ips->ips_said, 0, sa, SATOA_BUF);
					KLIPS_PRINT(debug_xform,
						    ", tdbh[%d]=%s",
						    i,
						    sa_len ? sa : " (error)");
				}
				if(*ipsprev) {
					sa_len = satoa((*ipsprev)->ips_said, 0, sa, SATOA_BUF);
					KLIPS_PRINT(debug_xform,
						    ", *ipsprev=%s",
						    sa_len ? sa : " (error)");
					if((*ipsprev)->ips_hnext) {
						sa_len = satoa((*ipsprev)->ips_hnext->ips_said, 0, sa, SATOA_BUF);
						KLIPS_PRINT(debug_xform,
							    ", *ipsprev->ips_hnext=%s",
							    sa_len ? sa : " (error)");
					}
				}
				KLIPS_PRINT(debug_xform,
					    ".\n");
			} else {
				ipsprev = &tdbdel;
			}
		}
	}
 errlab:

	spin_unlock_bh(&tdb_lock);

	return(error);
}

int
ipsec_sa_wipe(struct ipsec_sa *ips)
{
	if(!ips) {
		return -ENODATA;
	}

	if(ips->ips_addr_s) {
		memset((caddr_t)(ips->ips_addr_s), 0, ips->ips_addr_s_size);
		kfree(ips->ips_addr_s);
	}
	ips->ips_addr_s = NULL;

	if(ips->ips_addr_d) {
		memset((caddr_t)(ips->ips_addr_d), 0, ips->ips_addr_d_size);
		kfree(ips->ips_addr_d);
	}
	ips->ips_addr_d = NULL;

	if(ips->ips_addr_p) {
		memset((caddr_t)(ips->ips_addr_p), 0, ips->ips_addr_p_size);
		kfree(ips->ips_addr_p);
	}
	ips->ips_addr_p = NULL;

#ifdef CONFIG_IPSEC_NAT_TRAVERSAL
	if(ips->ips_natt_oa) {
		memset((caddr_t)(ips->ips_natt_oa), 0, ips->ips_natt_oa_size);
		kfree(ips->ips_natt_oa);
	}
	ips->ips_natt_oa = NULL;
#endif

	if(ips->ips_key_a) {
		memset((caddr_t)(ips->ips_key_a), 0, ips->ips_key_a_size);
		kfree(ips->ips_key_a);
	}
	ips->ips_key_a = NULL;

	if(ips->ips_key_e) {
		memset((caddr_t)(ips->ips_key_e), 0, ips->ips_key_e_size);
		kfree(ips->ips_key_e);
	}
	ips->ips_key_e = NULL;

	if(ips->ips_iv) {
		memset((caddr_t)(ips->ips_iv), 0, ips->ips_iv_size);
		kfree(ips->ips_iv);
	}
	ips->ips_iv = NULL;

	if(ips->ips_ident_s.data) {
		memset((caddr_t)(ips->ips_ident_s.data),
                       0,
		       ips->ips_ident_s.len * IPSEC_PFKEYv2_ALIGN - sizeof(struct sadb_ident));
		kfree(ips->ips_ident_s.data);
        }
	ips->ips_ident_s.data = NULL;
	
	if(ips->ips_ident_d.data) {
		memset((caddr_t)(ips->ips_ident_d.data),
                       0,
		       ips->ips_ident_d.len * IPSEC_PFKEYv2_ALIGN - sizeof(struct sadb_ident));
		kfree(ips->ips_ident_d.data);
        }
	ips->ips_ident_d.data = NULL;

	memset((caddr_t)ips, 0, sizeof(*ips));
	kfree(ips);
	ips = NULL;

	return 0;
}

