/*
 * RCSID $Id: ipsec_hw_sha1.c,v 1.4 2003/03/31 23:11:00 swahl Exp $
 */

/*
 * Perform SHA1 using Brecis hardware, using similar API
 * to ipsec_sha1.c, the software implementation.
 */


#include <asm/byteorder.h>
#include <linux/string.h>

#include "ipsec_sha1.h"
#include <brecis/msp_sec.h>

#ifdef __KERNEL__
#include <brecis/msp_sec_kern.h>

static struct msp_sec_context sec_ctx ;
#endif

#ifndef __KERNEL__
static int secfd = -1 ;
#endif

/* Hash a single 512-bit block. */

void
SHA1Transform(__u32 state[5], __u8 buffer[64], __u32 numblocks)
{
#ifndef __KERNEL__
	struct sec_ioctl sioc ;

	if (numblocks == 0)
		return ;

	sioc.type = MSP_SEC_HASH ;
	sioc.options = 0 ;
	sioc.cbufcount = 0 ;
	sioc.buflen = 64 * numblocks ;
	sioc.srcaddr = buffer ;
	sioc.dstaddr = (void *) state ;

	sioc.chainvar[0] = state[0] ;
	sioc.chainvar[1] = state[1] ;
	sioc.chainvar[2] = state[2] ;
	sioc.chainvar[3] = state[3] ;
	sioc.chainvar[4] = state[4] ;

	sioc.mode = HMC_SHA1 | HMC_CRV ;

	ioctl(secfd, MSP_SEC_CTL, &sioc) ;
#else
	struct sec_desc desc ;

	if (numblocks == 0)
		return ;

	desc.next = 0 ;

	desc.ioctl.type = MSP_SEC_HASH ;
	desc.ioctl.options = 0 ;
	desc.ioctl.buflen = 64 * numblocks ;
	desc.ioctl.srcaddr = buffer ;
	desc.ioctl.dstaddr = (void *) state ;

	desc.ioctl.chainvar[0] = state[0] ;
	desc.ioctl.chainvar[1] = state[1] ;
	desc.ioctl.chainvar[2] = state[2] ;
	desc.ioctl.chainvar[3] = state[3] ;
	desc.ioctl.chainvar[4] = state[4] ;

	desc.ioctl.mode = HMC_SHA1 | HMC_CRV ;

	msp_sec_ioctl_kernel(&sec_ctx, MSP_SEC_CTL, &desc) ;
#endif
}

/* SHA1Init - Initialize new context */

void SHA1Init(SHA1_CTX* context)
{
#ifndef __KERNEL__
	if (secfd == -1)
		secfd = open(SECDEV,0) ;
#endif

	/* SHA1 initialization constants */
	context->state[0] = 0x67452301;
	context->state[1] = 0xEFCDAB89;
	context->state[2] = 0x98BADCFE;
	context->state[3] = 0x10325476;
	context->state[4] = 0xC3D2E1F0;
	context->count[0] = context->count[1] = 0;
}


/* Run your data through this. */

void SHA1Update(SHA1_CTX* context, unsigned char* data, __u32 len)
{
__u32 i, j;

    j = context->count[0];
    if ((context->count[0] += len << 3) < j)
	context->count[1]++;
    context->count[1] += (len>>29);
    j = (j >> 3) & 63;
    if ((j + len) > 63) {
        memcpy(&context->buffer[j], data, (i = 64-j));
        SHA1Transform(context->state, context->buffer,1);
	//        for ( ; i + 63 < len; i += 64) {
	//            SHA1Transform(context->state, &data[i]);
	//        }
	SHA1Transform(context->state, &data[i], (len - i) / 64) ;
	// i += 64 * ((len - i) / 64) ;
	i += (len - i) & ~63 ;
        j = 0;
    }
    else i = 0;
    memcpy(&context->buffer[j], &data[i], len - i);
}


/* Add padding and return the message digest. */

void SHA1Final(unsigned char digest[20], SHA1_CTX* context)
{
__u32 i, j;
unsigned char finalcount[8];

    for (i = 0; i < 8; i++) {
        finalcount[i] = (unsigned char)((context->count[(i >= 4 ? 0 : 1)]
         >> ((3-(i & 3)) * 8) ) & 255);  /* Endian independent */
    }
    SHA1Update(context, (unsigned char *)"\200", 1);
    while ((context->count[0] & 504) != 448) {
        SHA1Update(context, (unsigned char *)"\0", 1);
    }
    SHA1Update(context, finalcount, 8);  /* Should cause a SHA1Transform() */
    for (i = 0; i < 20; i++) {
        digest[i] = (unsigned char)
         ((context->state[i>>2] >> ((3-(i & 3)) * 8) ) & 255);
    }
    /* Wipe variables */
    i = j = 0;
    memset(context->buffer, 0, 64);
    memset(context->state, 0, 20);
    memset(context->count, 0, 8);
    memset(&finalcount, 0, 8);
#ifdef SHA1HANDSOFF  /* make SHA1Transform overwrite its own static vars */
    SHA1Transform(context->state, context->buffer, 1);
#endif
}


