/*
 * access.c - Access register.
 *
 * Mark D. Rustad, 1/14/2003
 *
 * Copyright 2003 BRECIS Communications, Corp.
 */

/*
 * access <addr>[#<count> | =<value>] ...
 *
 * Reads or writes registers. A single register, or a group may be printed
 * out. Single registers can be modified. Multiple sets of reads and/or
 * writes can be done with a single command invocation.
 *
 * Examples:
 *	access bc000000
 *		Prints the contents of the single register at bc000000
 *
 *	access bc000000#4
 *		Prints the contents of 4 registers starting a bc000000
 *
 *	access bc000000=0
 *		Writes 0 into the register at bc000000
 */

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

int	main(int argc, char *argv[])
{
	char	**args = argv;
	char	*ptr;

	if (argc < 2) {
		fprintf(stderr, "%s: No argument given\n", argv[0]);
		printf("Usage: %s <addr> [<value>]\n", argv[0]);
		exit(1);
	}

	while (--argc > 0) {
		unsigned long	*addr;
		long	count;
		int	i;
		unsigned long	value;

		++args;
		errno = 0;
		addr = (unsigned long *)strtoul(*args, &ptr, 16);
		if (errno) {
			fprintf(stderr, "%s: Conversion error in \"%s\"\n",
				argv[0], *args);
			exit(2);
		}
		count = 1;
		switch (*ptr) {
		case '#':
			++ptr;
			count = strtol(ptr, &ptr, 10);
			if (errno) {
				fprintf(stderr, "%s: Conversion error in "
					"\"%s\"\n", argv[0], *args);
				exit(2);
			}
			if (*ptr != 0) {
				fprintf(stderr, "%s: Syntax error in \"%s\"\n",
					argv[0], *args);
				exit(3);
			}
		case 0:
			for (i = 0; i < count; ++i) {
				printf("%p = %08lx\n", addr, *addr);
				++addr;
			}
			break;

		case '=':
			++ptr;
			value = strtoul(ptr, &ptr, 16);
			if (errno) {
				fprintf(stderr, "%s: Conversion error in "
					"\"%s\"\n", argv[0], *args);
				exit(2);
			}
			if (*ptr != 0) {
				fprintf(stderr, "%s: Syntax error in \"%s\"\n",
					argv[0], *args);
				exit(3);
			}
			printf("%p <- %08lx\n", addr, value);
			*addr = value;
			break;

		default:
			fprintf(stderr, "%s: Syntax error in \"%s\"\n",
				argv[0], *args);
			exit(3);
		}
	}

	return 0;
}

